"""Configuration dataclasses for the optimisation engine."""

from __future__ import annotations

from collections.abc import Mapping, Sequence
from dataclasses import dataclass, field
from pathlib import Path
from typing import Any, Literal


@dataclass(slots=True)
class GridPolicyOptions:
    """Configuration for how simulations align with experiments."""

    policy: Literal["exp_to_sim", "fixed_user"] = "exp_to_sim"
    values: Sequence[float] | None = None


@dataclass(slots=True)
class JacobianOptions:
    """Jacobian evaluation controls."""

    enabled: bool = False
    perturbation: float = 1e-6
    parallel: bool = True


@dataclass(slots=True)
class CleanupOptions:
    """Cleanup behaviour for working directories."""

    remove_previous: bool = False
    mode: Literal["none", "retain_best", "all"] = "none"


@dataclass(slots=True)
class RunnerOptions:
    """Execution backend configuration."""

    jobs: int = 1
    command: Sequence[str] | None = None
    env: Mapping[str, str] | None = None


@dataclass(slots=True)
class StorageOptions:
    """Storage and logging configuration."""

    mode: Literal["disk", "tmp"] = "disk"
    root: str | Path | None = None
    log_file: str | Path | None = None


@dataclass(slots=True)
class MonitorOptions:
    """Monitoring client configuration."""

    enabled: bool = True
    socket: str | Path | None = None
    label: str | None = None


@dataclass(slots=True)
class OptimizerOptions:
    """Optimizer adapter selection and advanced settings."""

    name: Literal["least_squares", "minimize"] = "least_squares"
    settings: Mapping[str, Any] = field(default_factory=dict)
    reparametrize: bool = True


@dataclass(slots=True)
class EngineOptions:
    """Aggregate container for optimisation engine configuration."""

    jacobian: JacobianOptions = field(default_factory=JacobianOptions)
    cleanup: CleanupOptions = field(default_factory=CleanupOptions)
    runner: RunnerOptions = field(default_factory=RunnerOptions)
    storage: StorageOptions = field(default_factory=StorageOptions)
    monitor: MonitorOptions = field(default_factory=MonitorOptions)
    optimizer: OptimizerOptions = field(default_factory=OptimizerOptions)


__all__ = [
    "CleanupOptions",
    "EngineOptions",
    "GridPolicyOptions",
    "JacobianOptions",
    "MonitorOptions",
    "OptimizerOptions",
    "RunnerOptions",
    "StorageOptions",
]
