"""CLI helpers for installing and running the monitoring service."""

from __future__ import annotations

import argparse
import logging
import shutil
import subprocess
import sys
from pathlib import Path

from .monitoring_service import MonitoringService
from .paths import default_registry_path, default_socket_path

UNIT_TEMPLATE = (
    "[Unit]\n"
    "Description=pyfebiopt monitoring web service\n"
    "After=network.target\n"
    "\n"
    "[Service]\n"
    "Type=simple\n"
    "ExecStart={python} -m pyfebiopt.monitoring.service run "
    "--registry {registry} --socket {socket} --host {host} --port {port}\n"
    "Restart=on-failure\n"
    "RestartSec=2\n"
    "\n"
    "[Install]\n"
    "WantedBy=default.target\n"
)


def ensure_dependencies() -> None:
    """Ensure the optional monitoring stack is available."""
    try:
        import fastapi  # noqa: F401
        import psutil  # noqa: F401
        import uvicorn  # noqa: F401
    except ImportError as exc:  # pragma: no cover - optional dependency check
        raise RuntimeError(
            "pyfebiopt-monitor requires the 'monitor' extra. Install with "
            "pip install pyfebiopt[monitor]."
        ) from exc


def run_service(
    *,
    registry: Path | None = None,
    socket: Path | None = None,
    host: str | None = None,
    port: int | None = None,
) -> int:
    """Start the monitoring web service with the given overrides.

    Returns:
        int: Process exit code (0 on normal exit).
    """
    ensure_dependencies()
    import uvicorn

    from .webapp import create_app

    service = MonitoringService(
        registry_path=registry or default_registry_path(),
        event_socket=socket or default_socket_path(),
    )
    service.start()
    app = create_app(service)
    try:
        uvicorn.run(app, host=host or "127.0.0.1", port=int(port or 8765))
    finally:
        service.stop()
    return 0


def install_service(
    *,
    user: bool = True,
    force: bool = False,
    host: str = "127.0.0.1",
    port: int = 8765,
) -> None:
    """Install the systemd unit for the monitoring service."""
    ensure_dependencies()
    unit_content = UNIT_TEMPLATE.format(
        python=sys.executable,
        registry=default_registry_path(),
        socket=default_socket_path(),
        host=host,
        port=port,
    )
    if user:
        unit_dir = Path.home() / ".config/systemd/user"
    else:
        unit_dir = Path("/etc/systemd/system")
    unit_dir.mkdir(parents=True, exist_ok=True)
    unit_path = unit_dir / "pyfebiopt-monitor.service"
    if unit_path.exists() and not force:
        raise FileExistsError(f"Unit already exists at {unit_path}. Use --force to overwrite.")
    unit_path.write_text(unit_content, encoding="utf-8")
    _systemctl(["daemon-reload"], user=user)
    _systemctl(["enable", "pyfebiopt-monitor.service"], user=user)
    _systemctl(["start", "pyfebiopt-monitor.service"], user=user)
    print(f"Systemd unit installed at {unit_path}")


def uninstall_service(*, user: bool = True) -> None:
    """Disable and remove the monitoring systemd unit."""
    unit_path = (
        Path.home() / ".config/systemd/user/pyfebiopt-monitor.service"
        if user
        else Path("/etc/systemd/system/pyfebiopt-monitor.service")
    )
    _systemctl(["disable", "pyfebiopt-monitor.service"], user=user, check=False)
    _systemctl(["stop", "pyfebiopt-monitor.service"], user=user, check=False)
    if unit_path.exists():
        unit_path.unlink()
    _systemctl(["daemon-reload"], user=user, check=False)
    print("pyfebiopt-monitor service uninstalled")


def _systemctl(args: list[str], *, user: bool, check: bool = True) -> None:
    """Wrapper around systemctl invocation used by the install scripts."""
    executable = shutil.which("systemctl")
    if executable is None:
        raise RuntimeError("systemctl executable not found in PATH")
    cmd = [executable]
    if user:
        cmd.append("--user")
    cmd.extend(args)
    result = subprocess.run(cmd, check=check)
    if check and result.returncode != 0:
        raise RuntimeError(f"systemctl {' '.join(args)} failed with code {result.returncode}")


def parse_args(argv: list[str] | None = None) -> argparse.Namespace:
    """Parse the CLI arguments for the monitoring helper.

    Returns:
        argparse.Namespace: Parsed arguments.
    """
    parser = argparse.ArgumentParser(description="pyFEBiOpt monitoring service helper")
    subparsers = parser.add_subparsers(dest="command", required=True)

    subparsers.add_parser("run", help="Run the monitoring web service with default settings")

    install_parser = subparsers.add_parser("install", help="Install and enable the systemd unit")
    install_parser.add_argument(
        "--system", action="store_true", help="Install system-wide (requires root)"
    )
    install_parser.add_argument("--force", action="store_true", help="Overwrite existing unit")
    install_parser.add_argument(
        "--host",
        type=str,
        default="127.0.0.1",
        help="Host to bind the web UI (default: 127.0.0.1)",
    )
    install_parser.add_argument(
        "--port", type=int, default=8765, help="Port to bind the web UI (default: 8765)"
    )

    run_parser = subparsers.choices["run"]
    run_parser.add_argument("--registry", type=Path, default=None, help="Registry path override")
    run_parser.add_argument("--socket", type=Path, default=None, help="Event socket override")
    run_parser.add_argument(
        "--host", type=str, default=None, help="Host override for the web UI"
    )
    run_parser.add_argument(
        "--port", type=int, default=None, help="Port override for the web UI"
    )

    uninstall_parser = subparsers.add_parser(
        "uninstall", help="Disable and remove the systemd unit"
    )
    uninstall_parser.add_argument(
        "--system", action="store_true", help="Remove system-wide unit"
    )

    return parser.parse_args(argv)


def main(argv: list[str] | None = None) -> int:
    """Dispatch commands from the monitoring helper CLI.

    Returns:
        int: Process exit code.
    """
    args = parse_args(argv)
    logging.basicConfig(level=logging.INFO, format="%(asctime)s [%(levelname)s] %(message)s")
    if args.command == "run":
        return run_service(
            registry=args.registry,
            socket=args.socket,
            host=args.host,
            port=args.port,
        )
    if args.command == "install":
        install_service(
            user=not args.system,
            force=args.force,
            host=args.host,
            port=args.port,
        )
        return 0
    if args.command == "uninstall":
        uninstall_service(user=not args.system)
        return 0
    return 0


if __name__ == "__main__":  # pragma: no cover
    raise SystemExit(main())
