"""Controller that wires together registry, event listener, and system stats."""

from __future__ import annotations

import threading
from collections.abc import Iterable
from pathlib import Path

from .events import EventSocketListener
from .paths import default_registry_path, default_socket_path
from .registry import RunRegistry
from .state import StorageInventory
from .system_stats import SystemStatsCollector


class MonitoringService:
    """Own the monitor registry, event socket listener, and system stats."""

    def __init__(
        self,
        *,
        registry_path: Path | None = None,
        event_socket: Path | None = None,
        storage_roots: Iterable[Path] | None = None,
        poll_interval: float = 5.0,
        stats_collector: SystemStatsCollector | None = None,
    ) -> None:
        """Initialize registry, listener config, and stats collector.

        Args:
            registry_path: Optional path for the run registry JSON.
            event_socket: Optional UNIX socket path for event ingestion.
            storage_roots: Optional storage roots to inventory for jobs.
            poll_interval: Seconds between inventory refreshes.
            stats_collector: Custom collector; defaults to ``SystemStatsCollector``.
        """
        self.registry = RunRegistry(registry_path or default_registry_path())
        self.event_socket = Path(event_socket or default_socket_path())
        self.inventory = (
            StorageInventory(storage_roots, poll_interval=poll_interval)
            if storage_roots
            else None
        )
        self.stats = stats_collector or SystemStatsCollector()
        self._listener: EventSocketListener | None = None
        self._lock = threading.Lock()

    def start(self) -> None:
        """Start the event listener if configured and not already running."""
        with self._lock:
            if self._listener is not None:
                return
            listener = EventSocketListener(self.event_socket, self.registry)
            listener.start()
            self._listener = listener

    def stop(self) -> None:
        """Stop the event listener if it is running."""
        with self._lock:
            listener = self._listener
            self._listener = None
        if listener is not None:
            listener.stop()

    @property
    def running(self) -> bool:
        """Return True when the event listener is active."""
        return self._listener is not None


__all__ = ["MonitoringService"]
