"""Emit optimization lifecycle events to a monitoring service."""

from __future__ import annotations

import time
import uuid
from collections.abc import Mapping
from dataclasses import dataclass
from pathlib import Path
from typing import Any

from .events import EventEmitter, NullEventEmitter, create_event_emitter
from .paths import default_socket_path


def _slug(value: str) -> str:
    """Sanitize labels for use in run identifiers.

    Returns:
        str: Lowered, alphanumeric/underscore/hyphen-only string (never empty).
    """
    return "".join(ch for ch in value if ch.isalnum() or ch in ("-", "_")).strip("_") or "run"


def _sanitize_series(
    series: Mapping[str, Mapping[str, Any]] | None,
) -> dict[str, dict[str, list[float]]]:
    """Normalize optional series data so each field is numeric.

    Returns:
        dict[str, dict[str, list[float]]]: Cleaned copy with numeric lists
        (empty when input is None).
    """
    cleaned: dict[str, dict[str, list[float]]] = {}
    if not series:
        return cleaned
    for key, data in series.items():
        entry: dict[str, list[float]] = {}
        if not isinstance(data, Mapping):
            continue
        for field in ("x", "y_exp", "y_sim"):
            values = data.get(field)
            if values is None:
                continue
            try:
                numeric = [float(v) for v in values]
            except Exception:
                continue
            entry[field] = numeric
        if entry:
            cleaned[str(key)] = entry
    return cleaned


def generate_run_id(label: str | None = None) -> str:
    """Produce a fresh run identifier using the provided label.

    Returns:
        str: Run identifier combining the label slug and a random suffix.
    """
    prefix = _slug(label or "run")
    suffix = uuid.uuid4().hex[:12]
    return f"{prefix}-{suffix}"


@dataclass
class MonitorConfig:
    """Hold options for the optimization monitoring client."""

    socket_path: Path | None = None
    run_id: str | None = None
    label: str | None = None


class OptimizationMonitorClient:
    """Emit optimization lifecycle events to the monitor service."""

    def __init__(self, config: MonitorConfig | None = None):
        """Initialize the emitter and run metadata."""
        if config is None:
            config = MonitorConfig()
        socket_path = config.socket_path or default_socket_path()
        emitter = create_event_emitter(socket_path)
        self._emitter: EventEmitter = emitter
        self.run_id = config.run_id or generate_run_id(config.label)
        self.label = config.label or self.run_id
        self._started = False

    def run_started(
        self,
        *,
        parameters: Mapping[str, Any] | None = None,
        cases: list[Mapping[str, Any]] | None = None,
        optimizer: Mapping[str, Any] | None = None,
        meta: Mapping[str, Any] | None = None,
    ) -> None:
        """Emit the run started event."""
        payload: dict[str, Any] = {
            "label": self.label,
            "parameters": dict(parameters or {}),
            "meta": dict(meta or {}),
        }
        if cases:
            payload["cases"] = list(cases)
        if optimizer:
            payload["optimizer"] = dict(optimizer)
        self._emit("run_started", payload)
        self._started = True

    def record_iteration(
        self,
        *,
        index: int,
        cost: float,
        theta: Mapping[str, float],
        metrics: Mapping[str, Any] | None = None,
        series: Mapping[str, Mapping[str, Any]] | None = None,
    ) -> None:
        """Emit iteration progress with metrics and series data."""
        payload: dict[str, Any] = {
            "index": int(index),
            "cost": float(cost),
            "theta": {name: float(val) for name, val in theta.items()},
            "metrics": dict(metrics or {}),
            "timestamp": time.time(),
        }
        series_payload = _sanitize_series(series)
        if series_payload:
            payload["series"] = series_payload
        self._emit("iteration", payload)

    def run_completed(
        self,
        *,
        summary: Mapping[str, Any] | None = None,
        exit_code: int | None = None,
    ) -> None:
        """Announce the completed run with optional summary."""
        payload: dict[str, Any] = {}
        if summary:
            payload["summary"] = dict(summary)
        if exit_code is not None:
            payload["exit_code"] = exit_code
        self._emit("run_completed", payload)

    def run_failed(self, *, reason: str, exit_code: int | None = None) -> None:
        """Notify the monitor of a failed run."""
        payload: dict[str, Any] = {"reason": reason}
        if exit_code is not None:
            payload["exit_code"] = exit_code
        self._emit("run_failed", payload)

    def emit_meta(self, meta: Mapping[str, Any]) -> None:
        """Send additional metadata to the monitor."""
        self._emit("meta", dict(meta))

    def _emit(self, event: str, payload: dict[str, Any]) -> None:
        """Forward the event payload through the emitter."""
        try:
            self._emitter.emit(self.run_id, event, payload)
        except Exception:
            if isinstance(self._emitter, NullEventEmitter):
                return
            raise


__all__ = ["MonitorConfig", "OptimizationMonitorClient", "generate_run_id"]
