# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['cert_manager']

package_data = \
{'': ['*']}

install_requires = \
['requests', 'toml>=0.9,<0.10']

setup_kwargs = {
    'name': 'cert-manager',
    'version': '2.0.0',
    'description': 'Python interface to the Sectigo Certificate Manager REST API',
    'long_description': '# python-cert_manager\n\nThis library provides a [Python][1] interface to the [Sectigo][2] Certificate Manager REST API.  python-cert_manager is open sourced under the [BSD 3-Clause license](LICENSE.txt).\n\n![checks](https://github.com/broadinstitute/python-cert_manager/workflows/checks/badge.svg?branch=main)\n[![codecov](https://codecov.io/gh/broadinstitute/python-cert_manager/branch/main/graph/badge.svg)](https://codecov.io/gh/broadinstitute/python-cert_manager)\n\n## Basics\n\n`cert_manager` runs on [Python][1] >= 3.6\n\n## Features\n\nThere are many API endpoints under Certificate Manager, and this library currently supports a subset of those endpoints.  The current list of written and tested endpoint classes includes:\n\n* Organization (/organization)\n* Person (/person)\n* SSL (/ssl)\n\nOther endpoints we hope to add in the near future:\n\n* Client Administrator (/admin)\n* Code Signing Certificates (/csod)\n* Custom Fields (/customField)\n* Domain Control Validation (/dcv)\n* Device Certificates (/device)\n* Discovery (/discovery)\n* Domain (/domain)\n* SMIME (/smime)\n\n## Installing\n\nYou can use pip to install cert_manager:\n\n```sh\npip install cert_manager\n```\n\n## Examples\n\nThis is a simple example that just shows initializing the `Client` object and using it to query the `Organization` and `SSL` endpoints:\n\n```python\nfrom cert_manager import Organization\nfrom cert_manager import Client\nfrom cert_manager import SSL\n\nclient = Client(\n    base_url="https://cert-manager.com/api",\n    login_uri="SomeOrg",\n    username="your_username",\n    password="your_password",\n)\n\norg = Organization(client=client)\nssl = SSL(client=client)\n\nprint(ssl.types)\nprint(org.all())\n```\n\nThe most common process you would do, however, is enroll and then collect a certificate you want to order from the Certificate Manager:\n\n```python\nfrom time import sleep\n\nfrom cert_manager import Organization\nfrom cert_manager import Client\nfrom cert_manager import SSL\n\nclient = Client(\n    base_url="https://cert-manager.com/api",\n    login_uri="SomeOrg",\n    username="your_username",\n    password="your_password",\n)\n\n# We need to enroll the certificate under an organization, so we will need to query the API for that\norg = Organization(client=client)\n# We need the SSL module to enroll the certificate\nssl = SSL(client=client)\n\ncert_org = org.find(dept_name="MyDept")\n\nresult = ssl.enroll(cert_type_name="InCommon SSL (SHA-2)", csr="host.csr", term=365, org_id=cert_org[0]["id"])\n\n# This is just for demonstration purposes.\n# Doing a wait loop like this to poll for the certificate is not the best way to go about this.\nwhile(True):\n    # Collect the certificate from Sectigo\n    try:\n        cert_pem = ssl.collect(cert_id=result["sslId"], cert_format="x509CO")\n        print(cert_pem)\n        break\n    except Pending:\n        print("Certificate is still pending...sleeping for 60s")\n        sleep(60)\n        continue\n    except Exception:\n        # For some unexpected exception, exit\n        break\n```\n\n## Contributing\n\nPull requests to add functionality and fix bugs are always welcome.  Please check the CONTRIBUTING.md for specifics on contributions.\n\n### Testing\n\nWe try to have a high level of test coverage on the code.  Therefore, when adding anything to the repo, tests should be written to test a new feature or to test a bug fix so that there won\'t be a regression.  This library is setup to be pretty simple to build a working development environment using [Docker][4].  Therefore, it is suggested that you have [Docker][4] installed where you clone this repository to make development easier.\n\nTo start a development environment, you should be able to just run the `dev.sh` script.  This script will use the `Dockerfile` in this repository to build a [Docker][4] container with all the dependencies for development installed using [Pipenv][3].\n\n```sh\n./dev.sh\n```\n\nThe first time you run the script, it should build the [Docker][4] image and then drop you into the container\'s shell.  The directory where you cloned this repository should be volume mounted in to `/usr/src`, which should also be the current working directory.  From there, you can make changes as you see fit.  Tests can be run from the `/usr/src` directory by simply typing `green` as [green][5] has been setup to with the correct parameters.\n\n## Changelog\n\nTo generate the `CHANGELOG.md`, you will need [Docker][4] and a GitHub personal access token.  We currently use [github-changelog-generator](https://github.com/github-changelog-generator/github-changelog-generator) for this purpose.  The following should generate the file using information from GitHub:\n\n```sh\ndocker run -it --rm \\\n    -e CHANGELOG_GITHUB_TOKEN=\'yourtokenhere\' \\\n    -v "$(pwd)":/working \\\n    -w /working \\\n    ferrarimarco/github-changelog-generator --verbose\n```\n\nTo generate the log for an upcoming release that has not yet been tagged, you can run a command to include the upcoming release version.  For example, `2.0.0`:\n\n```sh\ndocker run -it --rm \\\n    -e CHANGELOG_GITHUB_TOKEN=\'yourtokenhere\' \\\n    -v "$(pwd)":/working \\\n    -w /working \\\n    ferrarimarco/github-changelog-generator --verbose --future-release 2.0.0 --unreleased\n```\n\nAs a note, this repository uses the default labels for formatting the `CHANGELOG.md`.  Label information can be found here: [Advanced-change-log-generation-examples](https://github.com/github-changelog-generator/github-changelog-generator/wiki/Advanced-change-log-generation-examples#section-options)\n\n## Releases\n\nReleases to the codebase are typically done using the [bump2version][6] tool.  This tool takes care of updating the version in all necessary files, updating its own configuration, and making a GitHub commit and tag.  We typically do version bumps as part of a PR, so you don\'t want to have [bump2version][6] tag the version at the same time it does the commit as commit hashes may change.  Therefore, to bump the version a patch level, one would run the command:\n\n```sh\nbump2version --verbose --no-tag patch\n```\n\nOnce the PR is merged, you can then checkout the new `main` branch and tag it using the new version number that is now in `.bumpversion.cfg`:\n\n```sh\ngit checkout main\ngit pull --rebase\ngit tag 1.0.0 -m \'Bump version: 0.1.0 → 1.0.0\'\ngit push --tags\n```\n\n[1]: https://www.python.org/ "Python"\n[2]: https://sectigo.com/ "Sectigo"\n[3]: https://pipenv.readthedocs.io/en/latest/ "Pipenv"\n[4]: https://www.docker.com/ "Docker"\n[5]: https://github.com/CleanCut/green "green"\n[6]: https://pypi.org/project/bump2version/ "bump2version"\n',
    'author': 'Andrew Teixeira',
    'author_email': 'teixeira@broadinstitute.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/broadinstitute/python-cert_manager.git',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
