def make_sneath_dict() -> dict[tuple[str, str], int]:
    """
    :return: A dictionary representative of TABLE 2 from 'Relations between Chemical Structure and Biological Activity
             in Peptides' by P. H. A. Sneath. Keys are pairs of strings. Cf. scientific paper for more details on how
             those scores are derived.
    """
    sneath_dict: dict[tuple[str, str], int] = {
        ('L', 'L'): 0,
        ('L', 'I'): 5,
        ('I', 'L'): 5,
        ('I', 'I'): 0,
        ('L', 'V'): 9,
        ('V', 'L'): 9,
        ('I', 'V'): 7,
        ('V', 'I'): 7,
        ('V', 'V'): 0,
        ('L', 'G'): 24,
        ('G', 'L'): 24,
        ('I', 'G'): 25,
        ('G', 'I'): 25,
        ('V', 'G'): 19,
        ('G', 'V'): 19,
        ('G', 'G'): 0,
        ('L', 'A'): 15,
        ('A', 'L'): 15,
        ('I', 'A'): 17,
        ('A', 'I'): 17,
        ('V', 'A'): 12,
        ('A', 'V'): 12,
        ('G', 'A'): 9,
        ('A', 'G'): 9,
        ('A', 'A'): 0,
        ('L', 'P'): 23,
        ('P', 'L'): 23,
        ('I', 'P'): 24,
        ('P', 'I'): 24,
        ('V', 'P'): 20,
        ('P', 'V'): 20,
        ('G', 'P'): 17,
        ('P', 'G'): 17,
        ('A', 'P'): 16,
        ('P', 'A'): 16,
        ('P', 'P'): 0,
        ('L', 'Q'): 22,
        ('Q', 'L'): 22,
        ('I', 'Q'): 24,
        ('Q', 'I'): 24,
        ('V', 'Q'): 25,
        ('Q', 'V'): 25,
        ('G', 'Q'): 32,
        ('Q', 'G'): 32,
        ('A', 'Q'): 26,
        ('Q', 'A'): 26,
        ('P', 'Q'): 33,
        ('Q', 'P'): 33,
        ('Q', 'Q'): 0,
        ('L', 'N'): 20,
        ('N', 'L'): 20,
        ('I', 'N'): 23,
        ('N', 'I'): 23,
        ('V', 'N'): 23,
        ('N', 'V'): 23,
        ('G', 'N'): 26,
        ('N', 'G'): 26,
        ('A', 'N'): 25,
        ('N', 'A'): 25,
        ('P', 'N'): 31,
        ('N', 'P'): 31,
        ('Q', 'N'): 10,
        ('N', 'Q'): 10,
        ('N', 'N'): 0,
        ('L', 'M'): 20,
        ('M', 'L'): 20,
        ('I', 'M'): 22,
        ('M', 'I'): 22,
        ('V', 'M'): 23,
        ('M', 'V'): 23,
        ('G', 'M'): 34,
        ('M', 'G'): 34,
        ('A', 'M'): 25,
        ('M', 'A'): 25,
        ('P', 'M'): 31,
        ('M', 'P'): 31,
        ('Q', 'M'): 13,
        ('M', 'Q'): 13,
        ('N', 'M'): 21,
        ('M', 'N'): 21,
        ('M', 'M'): 0,
        ('L', 'T'): 23,
        ('T', 'L'): 23,
        ('I', 'T'): 21,
        ('T', 'I'): 21,
        ('V', 'T'): 17,
        ('T', 'V'): 17,
        ('G', 'T'): 20,
        ('T', 'G'): 20,
        ('A', 'T'): 20,
        ('T', 'A'): 20,
        ('P', 'T'): 25,
        ('T', 'P'): 25,
        ('Q', 'T'): 24,
        ('T', 'Q'): 24,
        ('N', 'T'): 19,
        ('T', 'N'): 19,
        ('M', 'T'): 25,
        ('T', 'M'): 25,
        ('T', 'T'): 0,
        ('L', 'S'): 23,
        ('S', 'L'): 23,
        ('I', 'S'): 25,
        ('S', 'I'): 25,
        ('V', 'S'): 20,
        ('S', 'V'): 20,
        ('G', 'S'): 19,
        ('S', 'G'): 19,
        ('A', 'S'): 16,
        ('S', 'A'): 16,
        ('P', 'S'): 24,
        ('S', 'P'): 24,
        ('Q', 'S'): 21,
        ('S', 'Q'): 21,
        ('N', 'S'): 15,
        ('S', 'N'): 15,
        ('M', 'S'): 22,
        ('S', 'M'): 22,
        ('T', 'S'): 12,
        ('S', 'T'): 12,
        ('S', 'S'): 0,
        ('L', 'C'): 24,
        ('C', 'L'): 24,
        ('I', 'C'): 26,
        ('C', 'I'): 26,
        ('V', 'C'): 21,
        ('C', 'V'): 21,
        ('G', 'C'): 21,
        ('C', 'G'): 21,
        ('A', 'C'): 13,
        ('C', 'A'): 13,
        ('P', 'C'): 25,
        ('C', 'P'): 25,
        ('Q', 'C'): 22,
        ('C', 'Q'): 22,
        ('N', 'C'): 19,
        ('C', 'N'): 19,
        ('M', 'C'): 17,
        ('C', 'M'): 17,
        ('T', 'C'): 19,
        ('C', 'T'): 19,
        ('S', 'C'): 13,
        ('C', 'S'): 13,
        ('C', 'C'): 0,
        ('L', 'E'): 30,
        ('E', 'L'): 30,
        ('I', 'E'): 31,
        ('E', 'I'): 31,
        ('V', 'E'): 31,
        ('E', 'V'): 31,
        ('G', 'E'): 37,
        ('E', 'G'): 37,
        ('A', 'E'): 34,
        ('E', 'A'): 34,
        ('P', 'E'): 43,
        ('E', 'P'): 43,
        ('Q', 'E'): 14,
        ('E', 'Q'): 14,
        ('N', 'E'): 19,
        ('E', 'N'): 19,
        ('M', 'E'): 26,
        ('E', 'M'): 26,
        ('T', 'E'): 34,
        ('E', 'T'): 34,
        ('S', 'E'): 29,
        ('E', 'S'): 29,
        ('C', 'E'): 33,
        ('E', 'C'): 33,
        ('E', 'E'): 0,
        ('L', 'D'): 25,
        ('D', 'L'): 25,
        ('I', 'D'): 28,
        ('D', 'I'): 28,
        ('V', 'D'): 28,
        ('D', 'V'): 28,
        ('G', 'D'): 33,
        ('D', 'G'): 33,
        ('A', 'D'): 30,
        ('D', 'A'): 30,
        ('P', 'D'): 40,
        ('D', 'P'): 40,
        ('Q', 'D'): 22,
        ('D', 'Q'): 22,
        ('N', 'D'): 14,
        ('D', 'N'): 14,
        ('M', 'D'): 31,
        ('D', 'M'): 31,
        ('T', 'D'): 29,
        ('D', 'T'): 29,
        ('S', 'D'): 25,
        ('D', 'S'): 25,
        ('C', 'D'): 28,
        ('D', 'C'): 28,
        ('E', 'D'): 7,
        ('D', 'E'): 7,
        ('D', 'D'): 0,
        ('L', 'K'): 23,
        ('K', 'L'): 23,
        ('I', 'K'): 24,
        ('K', 'I'): 24,
        ('V', 'K'): 26,
        ('K', 'V'): 26,
        ('G', 'K'): 31,
        ('K', 'G'): 31,
        ('A', 'K'): 26,
        ('K', 'A'): 26,
        ('P', 'K'): 31,
        ('K', 'P'): 31,
        ('Q', 'K'): 21,
        ('K', 'Q'): 21,
        ('N', 'K'): 27,
        ('K', 'N'): 27,
        ('M', 'K'): 24,
        ('K', 'M'): 24,
        ('T', 'K'): 34,
        ('K', 'T'): 34,
        ('S', 'K'): 31,
        ('K', 'S'): 31,
        ('C', 'K'): 32,
        ('K', 'C'): 32,
        ('E', 'K'): 26,
        ('K', 'E'): 26,
        ('D', 'K'): 34,
        ('K', 'D'): 34,
        ('K', 'K'): 0,
        ('L', 'R'): 33,
        ('R', 'L'): 33,
        ('I', 'R'): 34,
        ('R', 'I'): 34,
        ('V', 'R'): 36,
        ('R', 'V'): 36,
        ('G', 'R'): 43,
        ('R', 'G'): 43,
        ('A', 'R'): 37,
        ('R', 'A'): 37,
        ('P', 'R'): 43,
        ('R', 'P'): 43,
        ('Q', 'R'): 23,
        ('R', 'Q'): 23,
        ('N', 'R'): 31,
        ('R', 'N'): 31,
        ('M', 'R'): 28,
        ('R', 'M'): 28,
        ('T', 'R'): 38,
        ('R', 'T'): 38,
        ('S', 'R'): 37,
        ('R', 'S'): 37,
        ('C', 'R'): 36,
        ('R', 'C'): 36,
        ('E', 'R'): 31,
        ('R', 'E'): 31,
        ('D', 'R'): 39,
        ('R', 'D'): 39,
        ('K', 'R'): 14,
        ('R', 'K'): 14,
        ('R', 'R'): 0,
        ('L', 'Y'): 30,
        ('Y', 'L'): 30,
        ('I', 'Y'): 34,
        ('Y', 'I'): 34,
        ('V', 'Y'): 36,
        ('Y', 'V'): 36,
        ('G', 'Y'): 36,
        ('Y', 'G'): 36,
        ('A', 'Y'): 34,
        ('Y', 'A'): 34,
        ('P', 'Y'): 37,
        ('Y', 'P'): 37,
        ('Q', 'Y'): 29,
        ('Y', 'Q'): 29,
        ('N', 'Y'): 28,
        ('Y', 'N'): 28,
        ('M', 'Y'): 32,
        ('Y', 'M'): 32,
        ('T', 'Y'): 32,
        ('Y', 'T'): 32,
        ('S', 'Y'): 29,
        ('Y', 'S'): 29,
        ('C', 'Y'): 34,
        ('Y', 'C'): 34,
        ('E', 'Y'): 34,
        ('Y', 'E'): 34,
        ('D', 'Y'): 34,
        ('Y', 'D'): 34,
        ('K', 'Y'): 34,
        ('Y', 'K'): 34,
        ('R', 'Y'): 36,
        ('Y', 'R'): 36,
        ('Y', 'Y'): 0,
        ('L', 'F'): 19,
        ('F', 'L'): 19,
        ('I', 'F'): 22,
        ('F', 'I'): 22,
        ('V', 'F'): 26,
        ('F', 'V'): 26,
        ('G', 'F'): 29,
        ('F', 'G'): 29,
        ('A', 'F'): 26,
        ('F', 'A'): 26,
        ('P', 'F'): 27,
        ('F', 'P'): 27,
        ('Q', 'F'): 24,
        ('F', 'Q'): 24,
        ('N', 'F'): 24,
        ('F', 'N'): 24,
        ('M', 'F'): 24,
        ('F', 'M'): 24,
        ('T', 'F'): 28,
        ('F', 'T'): 28,
        ('S', 'F'): 25,
        ('F', 'S'): 25,
        ('C', 'F'): 29,
        ('F', 'C'): 29,
        ('E', 'F'): 35,
        ('F', 'E'): 35,
        ('D', 'F'): 35,
        ('F', 'D'): 35,
        ('K', 'F'): 28,
        ('F', 'K'): 28,
        ('R', 'F'): 34,
        ('F', 'R'): 34,
        ('Y', 'F'): 13,
        ('F', 'Y'): 13,
        ('F', 'F'): 0,
        ('L', 'W'): 30,
        ('W', 'L'): 30,
        ('I', 'W'): 34,
        ('W', 'I'): 34,
        ('V', 'W'): 37,
        ('W', 'V'): 37,
        ('G', 'W'): 39,
        ('W', 'G'): 39,
        ('A', 'W'): 36,
        ('W', 'A'): 36,
        ('P', 'W'): 37,
        ('W', 'P'): 37,
        ('Q', 'W'): 31,
        ('W', 'Q'): 31,
        ('N', 'W'): 32,
        ('W', 'N'): 32,
        ('M', 'W'): 31,
        ('W', 'M'): 31,
        ('T', 'W'): 38,
        ('W', 'T'): 38,
        ('S', 'W'): 35,
        ('W', 'S'): 35,
        ('C', 'W'): 37,
        ('W', 'C'): 37,
        ('E', 'W'): 43,
        ('W', 'E'): 43,
        ('D', 'W'): 45,
        ('W', 'D'): 45,
        ('K', 'W'): 34,
        ('W', 'K'): 34,
        ('R', 'W'): 36,
        ('W', 'R'): 36,
        ('Y', 'W'): 21,
        ('W', 'Y'): 21,
        ('F', 'W'): 13,
        ('W', 'F'): 13,
        ('W', 'W'): 0,
        ('L', 'H'): 25,
        ('H', 'L'): 25,
        ('I', 'H'): 28,
        ('H', 'I'): 28,
        ('V', 'H'): 31,
        ('H', 'V'): 31,
        ('G', 'H'): 34,
        ('H', 'G'): 34,
        ('A', 'H'): 29,
        ('H', 'A'): 29,
        ('P', 'H'): 36,
        ('H', 'P'): 36,
        ('Q', 'H'): 27,
        ('H', 'Q'): 27,
        ('N', 'H'): 24,
        ('H', 'N'): 24,
        ('M', 'H'): 30,
        ('H', 'M'): 30,
        ('T', 'H'): 34,
        ('H', 'T'): 34,
        ('S', 'H'): 28,
        ('H', 'S'): 28,
        ('C', 'H'): 31,
        ('H', 'C'): 31,
        ('E', 'H'): 27,
        ('H', 'E'): 27,
        ('D', 'H'): 35,
        ('H', 'D'): 35,
        ('K', 'H'): 27,
        ('H', 'K'): 27,
        ('R', 'H'): 31,
        ('H', 'R'): 31,
        ('Y', 'H'): 23,
        ('H', 'Y'): 23,
        ('F', 'H'): 18,
        ('H', 'F'): 18,
        ('W', 'H'): 25,
        ('H', 'W'): 25,
        ('H', 'H'): 0,
    }

    return sneath_dict


def make_miyata_dict() -> dict[tuple[str, str], float]:
    """
    :return: A dictionary representative of TABLE 1 from 'Two Types of Amino Acid Substitutions in Protein Evolution'
             by Takashi Miyata, Sanzo Miyazawa, and Teruo Yasunaga. Keys are pairs of strings. Cf. scientific paper for
             more details on how those scores are derived.
    """
    miyata_dict: dict[tuple[str, str], float] = {
        ('C', 'C'): 0,
        ('P', 'C'): 1.33,
        ('C', 'P'): 1.33,
        ('A', 'C'): 1.39,
        ('C', 'A'): 1.39,
        ('G', 'C'): 2.22,
        ('C', 'G'): 2.22,
        ('S', 'C'): 2.84,
        ('C', 'S'): 2.84,
        ('T', 'C'): 1.45,
        ('C', 'T'): 1.45,
        ('Q', 'C'): 2.48,
        ('C', 'Q'): 2.48,
        ('E', 'C'): 3.26,
        ('C', 'E'): 3.26,
        ('N', 'C'): 2.83,
        ('C', 'N'): 2.83,
        ('D', 'C'): 3.48,
        ('C', 'D'): 3.48,
        ('H', 'C'): 2.56,
        ('C', 'H'): 2.56,
        ('K', 'C'): 3.27,
        ('C', 'K'): 3.27,
        ('R', 'C'): 3.06,
        ('C', 'R'): 3.06,
        ('V', 'C'): 0.86,
        ('C', 'V'): 0.86,
        ('L', 'C'): 1.65,
        ('C', 'L'): 1.65,
        ('I', 'C'): 1.63,
        ('C', 'I'): 1.63,
        ('M', 'C'): 1.46,
        ('C', 'M'): 1.46,
        ('F', 'C'): 2.24,
        ('C', 'F'): 2.24,
        ('Y', 'C'): 2.38,
        ('C', 'Y'): 2.38,
        ('W', 'C'): 3.34,
        ('C', 'W'): 3.34,
        ('P', 'P'): 0,
        ('A', 'P'): 0.06,
        ('P', 'A'): 0.06,
        ('G', 'P'): 0.97,
        ('P', 'G'): 0.97,
        ('S', 'P'): 0.56,
        ('P', 'S'): 0.56,
        ('T', 'P'): 0.87,
        ('P', 'T'): 0.87,
        ('Q', 'P'): 1.92,
        ('P', 'Q'): 1.92,
        ('E', 'P'): 2.48,
        ('P', 'E'): 2.48,
        ('N', 'P'): 1.8,
        ('P', 'N'): 1.8,
        ('D', 'P'): 2.4,
        ('P', 'D'): 2.4,
        ('H', 'P'): 2.15,
        ('P', 'H'): 2.15,
        ('K', 'P'): 2.94,
        ('P', 'K'): 2.94,
        ('R', 'P'): 2.9,
        ('P', 'R'): 2.9,
        ('V', 'P'): 1.79,
        ('P', 'V'): 1.79,
        ('L', 'P'): 2.7,
        ('P', 'L'): 2.7,
        ('I', 'P'): 2.62,
        ('P', 'I'): 2.62,
        ('M', 'P'): 2.36,
        ('P', 'M'): 2.36,
        ('F', 'P'): 3.17,
        ('P', 'F'): 3.17,
        ('Y', 'P'): 3.12,
        ('P', 'Y'): 3.12,
        ('W', 'P'): 4.17,
        ('P', 'W'): 4.17,
        ('A', 'A'): 0,
        ('G', 'A'): 0.91,
        ('A', 'G'): 0.91,
        ('S', 'A'): 0.51,
        ('A', 'S'): 0.51,
        ('T', 'A'): 0.9,
        ('A', 'T'): 0.9,
        ('Q', 'A'): 1.92,
        ('A', 'Q'): 1.92,
        ('E', 'A'): 2.46,
        ('A', 'E'): 2.46,
        ('N', 'A'): 1.78,
        ('A', 'N'): 1.78,
        ('D', 'A'): 2.37,
        ('A', 'D'): 2.37,
        ('H', 'A'): 2.17,
        ('A', 'H'): 2.17,
        ('K', 'A'): 2.96,
        ('A', 'K'): 2.96,
        ('R', 'A'): 2.92,
        ('A', 'R'): 2.92,
        ('V', 'A'): 1.85,
        ('A', 'V'): 1.85,
        ('L', 'A'): 2.76,
        ('A', 'L'): 2.76,
        ('I', 'A'): 2.69,
        ('A', 'I'): 2.69,
        ('M', 'A'): 2.42,
        ('A', 'M'): 2.42,
        ('F', 'A'): 3.23,
        ('A', 'F'): 3.23,
        ('Y', 'A'): 3.18,
        ('A', 'Y'): 3.18,
        ('W', 'A'): 4.23,
        ('A', 'W'): 4.23,
        ('G', 'G'): 0,
        ('S', 'G'): 0.85,
        ('G', 'S'): 0.85,
        ('T', 'G'): 1.7,
        ('G', 'T'): 1.7,
        ('Q', 'G'): 2.48,
        ('G', 'Q'): 2.48,
        ('E', 'G'): 2.78,
        ('G', 'E'): 2.78,
        ('N', 'G'): 1.96,
        ('G', 'N'): 1.96,
        ('D', 'G'): 2.37,
        ('G', 'D'): 2.37,
        ('H', 'G'): 2.78,
        ('G', 'H'): 2.78,
        ('K', 'G'): 3.54,
        ('G', 'K'): 3.54,
        ('R', 'G'): 3.58,
        ('G', 'R'): 3.58,
        ('V', 'G'): 2.76,
        ('G', 'V'): 2.76,
        ('L', 'G'): 3.67,
        ('G', 'L'): 3.67,
        ('I', 'G'): 3.6,
        ('G', 'I'): 3.6,
        ('M', 'G'): 3.34,
        ('G', 'M'): 3.34,
        ('F', 'G'): 4.14,
        ('G', 'F'): 4.14,
        ('Y', 'G'): 4.08,
        ('G', 'Y'): 4.08,
        ('W', 'G'): 5.13,
        ('G', 'W'): 5.13,
        ('S', 'S'): 0,
        ('T', 'S'): 0.89,
        ('S', 'T'): 0.89,
        ('Q', 'S'): 1.65,
        ('S', 'Q'): 1.65,
        ('E', 'S'): 2.06,
        ('S', 'E'): 2.06,
        ('N', 'S'): 1.31,
        ('S', 'N'): 1.31,
        ('D', 'S'): 1.87,
        ('S', 'D'): 1.87,
        ('H', 'S'): 1.94,
        ('S', 'H'): 1.94,
        ('K', 'S'): 2.71,
        ('S', 'K'): 2.71,
        ('R', 'S'): 2.74,
        ('S', 'R'): 2.74,
        ('V', 'S'): 2.15,
        ('S', 'V'): 2.15,
        ('L', 'S'): 3.04,
        ('S', 'L'): 3.04,
        ('I', 'S'): 2.95,
        ('S', 'I'): 2.95,
        ('M', 'S'): 2.67,
        ('S', 'M'): 2.67,
        ('F', 'S'): 3.45,
        ('S', 'F'): 3.45,
        ('Y', 'S'): 3.33,
        ('S', 'Y'): 3.33,
        ('W', 'S'): 4.38,
        ('S', 'W'): 4.38,
        ('T', 'T'): 0,
        ('Q', 'T'): 1.12,
        ('T', 'Q'): 1.12,
        ('E', 'T'): 1.83,
        ('T', 'E'): 1.83,
        ('N', 'T'): 1.4,
        ('T', 'N'): 1.4,
        ('D', 'T'): 2.05,
        ('T', 'D'): 2.05,
        ('H', 'T'): 1.32,
        ('T', 'H'): 1.32,
        ('K', 'T'): 2.1,
        ('T', 'K'): 2.1,
        ('R', 'T'): 2.03,
        ('T', 'R'): 2.03,
        ('V', 'T'): 1.42,
        ('T', 'V'): 1.42,
        ('L', 'T'): 2.25,
        ('T', 'L'): 2.25,
        ('I', 'T'): 2.14,
        ('T', 'I'): 2.14,
        ('M', 'T'): 1.86,
        ('T', 'M'): 1.86,
        ('F', 'T'): 2.6,
        ('T', 'F'): 2.6,
        ('Y', 'T'): 2.45,
        ('T', 'Y'): 2.45,
        ('W', 'T'): 3.5,
        ('T', 'W'): 3.5,
        ('Q', 'Q'): 0,
        ('E', 'Q'): 0.84,
        ('Q', 'E'): 0.84,
        ('N', 'Q'): 0.99,
        ('Q', 'N'): 0.99,
        ('D', 'Q'): 1.47,
        ('Q', 'D'): 1.47,
        ('H', 'Q'): 0.32,
        ('Q', 'H'): 0.32,
        ('K', 'Q'): 1.06,
        ('Q', 'K'): 1.06,
        ('R', 'Q'): 1.13,
        ('Q', 'R'): 1.13,
        ('V', 'Q'): 2.13,
        ('Q', 'V'): 2.13,
        ('L', 'Q'): 2.7,
        ('Q', 'L'): 2.7,
        ('I', 'Q'): 2.57,
        ('Q', 'I'): 2.57,
        ('M', 'Q'): 2.3,
        ('Q', 'M'): 2.3,
        ('F', 'Q'): 2.81,
        ('Q', 'F'): 2.81,
        ('Y', 'Q'): 2.48,
        ('Q', 'Y'): 2.48,
        ('W', 'Q'): 3.42,
        ('Q', 'W'): 3.42,
        ('E', 'E'): 0,
        ('N', 'E'): 0.85,
        ('E', 'N'): 0.85,
        ('D', 'E'): 0.9,
        ('E', 'D'): 0.9,
        ('H', 'E'): 0.96,
        ('E', 'H'): 0.96,
        ('K', 'E'): 1.14,
        ('E', 'K'): 1.14,
        ('R', 'E'): 1.45,
        ('E', 'R'): 1.45,
        ('V', 'E'): 2.97,
        ('E', 'V'): 2.97,
        ('L', 'E'): 3.53,
        ('E', 'L'): 3.53,
        ('I', 'E'): 3.39,
        ('E', 'I'): 3.39,
        ('M', 'E'): 3.13,
        ('E', 'M'): 3.13,
        ('F', 'E'): 3.59,
        ('E', 'F'): 3.59,
        ('Y', 'E'): 3.22,
        ('E', 'Y'): 3.22,
        ('W', 'E'): 4.08,
        ('E', 'W'): 4.08,
        ('N', 'N'): 0,
        ('D', 'N'): 0.65,
        ('N', 'D'): 0.65,
        ('H', 'N'): 1.29,
        ('N', 'H'): 1.29,
        ('K', 'N'): 1.84,
        ('N', 'K'): 1.84,
        ('R', 'N'): 2.04,
        ('N', 'R'): 2.04,
        ('V', 'N'): 2.76,
        ('N', 'V'): 2.76,
        ('L', 'N'): 3.49,
        ('N', 'L'): 3.49,
        ('I', 'N'): 3.37,
        ('N', 'I'): 3.37,
        ('M', 'N'): 3.08,
        ('N', 'M'): 3.08,
        ('F', 'N'): 3.7,
        ('N', 'F'): 3.7,
        ('Y', 'N'): 3.42,
        ('N', 'Y'): 3.42,
        ('W', 'N'): 4.39,
        ('N', 'W'): 4.39,
        ('D', 'D'): 0,
        ('H', 'D'): 1.72,
        ('D', 'H'): 1.72,
        ('K', 'D'): 2.05,
        ('D', 'K'): 2.05,
        ('R', 'D'): 2.34,
        ('D', 'R'): 2.34,
        ('V', 'D'): 3.4,
        ('D', 'V'): 3.4,
        ('L', 'D'): 4.1,
        ('D', 'L'): 4.1,
        ('I', 'D'): 3.98,
        ('D', 'I'): 3.98,
        ('M', 'D'): 3.69,
        ('D', 'M'): 3.69,
        ('F', 'D'): 4.27,
        ('D', 'F'): 4.27,
        ('Y', 'D'): 3.95,
        ('D', 'Y'): 3.95,
        ('W', 'D'): 4.88,
        ('D', 'W'): 4.88,
        ('H', 'H'): 0,
        ('K', 'H'): 0.79,
        ('H', 'K'): 0.79,
        ('R', 'H'): 0.82,
        ('H', 'R'): 0.82,
        ('V', 'H'): 2.11,
        ('H', 'V'): 2.11,
        ('L', 'H'): 2.59,
        ('H', 'L'): 2.59,
        ('I', 'H'): 2.45,
        ('H', 'I'): 2.45,
        ('M', 'H'): 2.19,
        ('H', 'M'): 2.19,
        ('F', 'H'): 2.63,
        ('H', 'F'): 2.63,
        ('Y', 'H'): 2.27,
        ('H', 'Y'): 2.27,
        ('W', 'H'): 3.16,
        ('H', 'W'): 3.16,
        ('K', 'K'): 0,
        ('R', 'K'): 0.4,
        ('K', 'R'): 0.4,
        ('V', 'K'): 2.7,
        ('K', 'V'): 2.7,
        ('L', 'K'): 2.98,
        ('K', 'L'): 2.98,
        ('I', 'K'): 2.84,
        ('K', 'I'): 2.84,
        ('M', 'K'): 2.63,
        ('K', 'M'): 2.63,
        ('F', 'K'): 2.85,
        ('K', 'F'): 2.85,
        ('Y', 'K'): 2.42,
        ('K', 'Y'): 2.42,
        ('W', 'K'): 3.11,
        ('K', 'W'): 3.11,
        ('R', 'R'): 0,
        ('V', 'R'): 2.43,
        ('R', 'V'): 2.43,
        ('L', 'R'): 2.62,
        ('R', 'L'): 2.62,
        ('I', 'R'): 2.49,
        ('R', 'I'): 2.49,
        ('M', 'R'): 2.29,
        ('R', 'M'): 2.29,
        ('F', 'R'): 2.47,
        ('R', 'F'): 2.47,
        ('Y', 'R'): 2.02,
        ('R', 'Y'): 2.02,
        ('W', 'R'): 2.72,
        ('R', 'W'): 2.72,
        ('V', 'V'): 0,
        ('L', 'V'): 0.91,
        ('V', 'L'): 0.91,
        ('I', 'V'): 0.85,
        ('V', 'I'): 0.85,
        ('M', 'V'): 0.62,
        ('V', 'M'): 0.62,
        ('F', 'V'): 1.43,
        ('V', 'F'): 1.43,
        ('Y', 'V'): 1.52,
        ('V', 'Y'): 1.52,
        ('W', 'V'): 2.51,
        ('V', 'W'): 2.51,
        ('L', 'L'): 0,
        ('I', 'L'): 0.14,
        ('L', 'I'): 0.14,
        ('M', 'L'): 0.41,
        ('L', 'M'): 0.41,
        ('F', 'L'): 0.63,
        ('L', 'F'): 0.63,
        ('Y', 'L'): 0.94,
        ('L', 'Y'): 0.94,
        ('W', 'L'): 1.73,
        ('L', 'W'): 1.73,
        ('I', 'I'): 0,
        ('M', 'I'): 0.29,
        ('I', 'M'): 0.29,
        ('F', 'I'): 0.61,
        ('I', 'F'): 0.61,
        ('Y', 'I'): 0.86,
        ('I', 'Y'): 0.86,
        ('W', 'I'): 1.72,
        ('I', 'W'): 1.72,
        ('M', 'M'): 0,
        ('F', 'M'): 0.82,
        ('M', 'F'): 0.82,
        ('Y', 'M'): 0.93,
        ('M', 'Y'): 0.93,
        ('W', 'M'): 1.89,
        ('M', 'W'): 1.89,
        ('F', 'F'): 0,
        ('Y', 'F'): 0.48,
        ('F', 'Y'): 0.48,
        ('W', 'F'): 1.11,
        ('F', 'W'): 1.11,
        ('Y', 'Y'): 0,
        ('W', 'Y'): 1.06,
        ('Y', 'W'): 1.06,
        ('W', 'W'): 0,
    }

    return miyata_dict


def make_symmetric_epstein_dict() -> dict[tuple[str, str], float]:
    """
    :return: A symmetrised* version of the dictionary representative of TABLE 2 from 'Non-randomness of Amino-acid
             Changes in the Evolution of Homologous Proteins' by Charles J. Epstein. Keys are pairs of strings. Cf.
             scientific paper for more details on how those scores are derived.
             * cf. paper for formula used to infer the symmetrised version (mean).
    """
    epstein_dict: dict[tuple[str, str], float] = {
        ("F", "M"): 0.075,
        ("M", "F"): 0.075,
        ("F", "L"): 0.115,
        ("L", "F"): 0.115,
        ("F", "I"): 0.115,
        ("I", "F"): 0.115,
        ("F", "V"): 0.15,
        ("V", "F"): 0.15,
        ("F", "P"): 0.15,
        ("P", "F"): 0.15,
        ("F", "Y"): 0.205,
        ("Y", "F"): 0.205,
        ("F", "W"): 0.23,
        ("W", "F"): 0.23,
        ("F", "C"): 0.25,
        ("C", "F"): 0.25,
        ("F", "A"): 0.465,
        ("A", "F"): 0.465,
        ("F", "G"): 0.57,
        ("G", "F"): 0.57,
        ("F", "S"): 0.81,
        ("S", "F"): 0.81,
        ("F", "T"): 0.81,
        ("T", "F"): 0.81,
        ("F", "H"): 0.8,
        ("H", "F"): 0.8,
        ("F", "E"): 1.0,
        ("E", "F"): 1.0,
        ("F", "Q"): 1.0,
        ("Q", "F"): 1.0,
        ("F", "D"): 1.0,
        ("D", "F"): 1.0,
        ("F", "N"): 1.0,
        ("N", "F"): 1.0,
        ("F", "K"): 1.0,
        ("K", "F"): 1.0,
        ("F", "R"): 1.0,
        ("R", "F"): 1.0,
        ("M", "L"): 0.04,
        ("L", "M"): 0.04,
        ("M", "I"): 0.04,
        ("I", "M"): 0.04,
        ("M", "V"): 0.1,
        ("V", "M"): 0.1,
        ("M", "P"): 0.1,
        ("P", "M"): 0.1,
        ("M", "Y"): 0.235,
        ("Y", "M"): 0.235,
        ("M", "W"): 0.28,
        ("W", "M"): 0.28,
        ("M", "C"): 0.215,
        ("C", "M"): 0.215,
        ("M", "A"): 0.43,
        ("A", "M"): 0.43,
        ("M", "G"): 0.49,
        ("G", "M"): 0.49,
        ("M", "S"): 0.8,
        ("S", "M"): 0.8,
        ("M", "T"): 0.8,
        ("T", "M"): 0.8,
        ("M", "H"): 0.8,
        ("H", "M"): 0.8,
        ("M", "E"): 1.0,
        ("E", "M"): 1.0,
        ("M", "Q"): 1.0,
        ("Q", "M"): 1.0,
        ("M", "D"): 1.0,
        ("D", "M"): 1.0,
        ("M", "N"): 1.0,
        ("N", "M"): 1.0,
        ("M", "K"): 1.0,
        ("K", "M"): 1.0,
        ("M", "R"): 1.0,
        ("R", "M"): 1.0,
        ("L", "I"): 0.0,
        ("I", "L"): 0.0,
        ("L", "V"): 0.04,
        ("V", "L"): 0.04,
        ("L", "P"): 0.04,
        ("P", "L"): 0.04,
        ("L", "Y"): 0.25,
        ("Y", "L"): 0.25,
        ("L", "W"): 0.305,
        ("W", "L"): 0.305,
        ("L", "C"): 0.205,
        ("C", "L"): 0.205,
        ("L", "A"): 0.43,
        ("A", "L"): 0.43,
        ("L", "G"): 0.525,
        ("G", "L"): 0.525,
        ("L", "S"): 0.8,
        ("S", "L"): 0.8,
        ("L", "T"): 0.8,
        ("T", "L"): 0.8,
        ("L", "H"): 0.905,
        ("H", "L"): 0.905,
        ("L", "E"): 1.0,
        ("E", "L"): 1.0,
        ("L", "Q"): 1.0,
        ("Q", "L"): 1.0,
        ("L", "D"): 1.0,
        ("D", "L"): 1.0,
        ("L", "N"): 1.0,
        ("N", "L"): 1.0,
        ("L", "K"): 1.0,
        ("K", "L"): 1.0,
        ("L", "R"): 1.005,
        ("R", "L"): 1.005,
        ("I", "V"): 0.04,
        ("V", "I"): 0.04,
        ("I", "P"): 0.04,
        ("P", "I"): 0.04,
        ("I", "Y"): 0.25,
        ("Y", "I"): 0.25,
        ("I", "W"): 0.305,
        ("W", "I"): 0.305,
        ("I", "C"): 0.205,
        ("C", "I"): 0.205,
        ("I", "A"): 0.43,
        ("A", "I"): 0.43,
        ("I", "G"): 0.525,
        ("G", "I"): 0.525,
        ("I", "S"): 0.8,
        ("S", "I"): 0.8,
        ("I", "T"): 0.8,
        ("T", "I"): 0.8,
        ("I", "H"): 0.905,
        ("H", "I"): 0.905,
        ("I", "E"): 1.0,
        ("E", "I"): 1.0,
        ("I", "Q"): 1.0,
        ("Q", "I"): 1.0,
        ("I", "D"): 1.0,
        ("D", "I"): 1.0,
        ("I", "N"): 1.0,
        ("N", "I"): 1.0,
        ("I", "K"): 1.0,
        ("K", "I"): 1.0,
        ("I", "R"): 1.005,
        ("R", "I"): 1.005,
        ("V", "P"): 0.0,
        ("P", "V"): 0.0,
        ("V", "Y"): 0.28,
        ("Y", "V"): 0.28,
        ("V", "W"): 0.335,
        ("W", "V"): 0.335,
        ("V", "C"): 0.2,
        ("C", "V"): 0.2,
        ("V", "A"): 0.405,
        ("A", "V"): 0.405,
        ("V", "G"): 0.51,
        ("G", "V"): 0.51,
        ("V", "S"): 0.8,
        ("S", "V"): 0.8,
        ("V", "T"): 0.8,
        ("T", "V"): 0.8,
        ("V", "H"): 0.805,
        ("H", "V"): 0.805,
        ("V", "E"): 1.0,
        ("E", "V"): 1.0,
        ("V", "Q"): 1.0,
        ("Q", "V"): 1.0,
        ("V", "D"): 1.0,
        ("D", "V"): 1.0,
        ("V", "N"): 1.0,
        ("N", "V"): 1.0,
        ("V", "K"): 1.0,
        ("K", "V"): 1.0,
        ("V", "R"): 1.015,
        ("R", "V"): 1.015,
        ("P", "Y"): 0.28,
        ("Y", "P"): 0.28,
        ("P", "W"): 0.335,
        ("W", "P"): 0.335,
        ("P", "C"): 0.2,
        ("C", "P"): 0.2,
        ("P", "A"): 0.405,
        ("A", "P"): 0.405,
        ("P", "G"): 0.51,
        ("G", "P"): 0.51,
        ("P", "S"): 0.8,
        ("S", "P"): 0.8,
        ("P", "T"): 0.8,
        ("T", "P"): 0.8,
        ("P", "H"): 0.805,
        ("H", "P"): 0.805,
        ("P", "E"): 1.0,
        ("E", "P"): 1.0,
        ("P", "Q"): 1.0,
        ("Q", "P"): 1.0,
        ("P", "D"): 1.0,
        ("D", "P"): 1.0,
        ("P", "N"): 1.0,
        ("N", "P"): 1.0,
        ("P", "K"): 1.0,
        ("K", "P"): 1.0,
        ("P", "R"): 1.015,
        ("R", "P"): 1.015,
        ("Y", "W"): 0.075,
        ("W", "Y"): 0.075,
        ("Y", "C"): 0.19,
        ("C", "Y"): 0.19,
        ("Y", "A"): 0.335,
        ("A", "Y"): 0.335,
        ("Y", "G"): 0.43,
        ("G", "Y"): 0.43,
        ("Y", "S"): 0.62,
        ("S", "Y"): 0.62,
        ("Y", "T"): 0.61,
        ("T", "Y"): 0.61,
        ("Y", "H"): 0.6,
        ("H", "Y"): 0.6,
        ("Y", "E"): 0.8,
        ("E", "Y"): 0.8,
        ("Y", "Q"): 0.8,
        ("Q", "Y"): 0.8,
        ("Y", "D"): 0.81,
        ("D", "Y"): 0.81,
        ("Y", "N"): 0.81,
        ("N", "Y"): 0.81,
        ("Y", "K"): 0.8,
        ("K", "Y"): 0.8,
        ("Y", "R"): 0.8,
        ("R", "Y"): 0.8,
        ("W", "C"): 0.265,
        ("C", "W"): 0.265,
        ("W", "A"): 0.395,
        ("A", "W"): 0.395,
        ("W", "G"): 0.485,
        ("G", "W"): 0.485,
        ("W", "S"): 0.63,
        ("S", "W"): 0.63,
        ("W", "T"): 0.63,
        ("T", "W"): 0.63,
        ("W", "H"): 0.61,
        ("H", "W"): 0.61,
        ("W", "E"): 0.81,
        ("E", "W"): 0.81,
        ("W", "Q"): 0.81,
        ("Q", "W"): 0.81,
        ("W", "D"): 0.81,
        ("D", "W"): 0.81,
        ("W", "N"): 0.81,
        ("N", "W"): 0.81,
        ("W", "K"): 0.81,
        ("K", "W"): 0.81,
        ("W", "R"): 0.8,
        ("R", "W"): 0.8,
        ("C", "A"): 0.235,
        ("A", "C"): 0.235,
        ("C", "G"): 0.325,
        ("G", "C"): 0.325,
        ("C", "S"): 0.6,
        ("S", "C"): 0.6,
        ("C", "T"): 0.6,
        ("T", "C"): 0.6,
        ("C", "H"): 0.615,
        ("H", "C"): 0.615,
        ("C", "E"): 0.805,
        ("E", "C"): 0.805,
        ("C", "Q"): 0.805,
        ("Q", "C"): 0.805,
        ("C", "D"): 0.8,
        ("D", "C"): 0.8,
        ("C", "N"): 0.8,
        ("N", "C"): 0.8,
        ("C", "K"): 0.805,
        ("K", "C"): 0.805,
        ("C", "R"): 0.815,
        ("R", "C"): 0.815,
        ("A", "G"): 0.1,
        ("G", "A"): 0.1,
        ("A", "S"): 0.4,
        ("S", "A"): 0.4,
        ("A", "T"): 0.405,
        ("T", "A"): 0.405,
        ("A", "H"): 0.445,
        ("H", "A"): 0.445,
        ("A", "E"): 0.62,
        ("E", "A"): 0.62,
        ("A", "Q"): 0.62,
        ("Q", "A"): 0.62,
        ("A", "D"): 0.615,
        ("D", "A"): 0.615,
        ("A", "N"): 0.615,
        ("N", "A"): 0.615,
        ("A", "K"): 0.62,
        ("K", "A"): 0.62,
        ("A", "R"): 0.645,
        ("R", "A"): 0.645,
        ("G", "S"): 0.31,
        ("S", "G"): 0.31,
        ("G", "T"): 0.325,
        ("T", "G"): 0.325,
        ("G", "H"): 0.38,
        ("H", "G"): 0.38,
        ("G", "E"): 0.54,
        ("E", "G"): 0.54,
        ("G", "Q"): 0.54,
        ("Q", "G"): 0.54,
        ("G", "D"): 0.525,
        ("D", "G"): 0.525,
        ("G", "N"): 0.525,
        ("N", "G"): 0.525,
        ("G", "K"): 0.54,
        ("K", "G"): 0.54,
        ("G", "R"): 0.57,
        ("R", "G"): 0.57,
        ("S", "T"): 0.03,
        ("T", "S"): 0.03,
        ("S", "H"): 0.1,
        ("H", "S"): 0.1,
        ("S", "E"): 0.215,
        ("E", "S"): 0.215,
        ("S", "Q"): 0.215,
        ("Q", "S"): 0.215,
        ("S", "D"): 0.205,
        ("D", "S"): 0.205,
        ("S", "N"): 0.205,
        ("N", "S"): 0.205,
        ("S", "K"): 0.215,
        ("K", "S"): 0.215,
        ("S", "R"): 0.24,
        ("R", "S"): 0.24,
        ("T", "H"): 0.08,
        ("H", "T"): 0.08,
        ("T", "E"): 0.21,
        ("E", "T"): 0.21,
        ("T", "Q"): 0.21,
        ("Q", "T"): 0.21,
        ("T", "D"): 0.2,
        ("D", "T"): 0.2,
        ("T", "N"): 0.2,
        ("N", "T"): 0.2,
        ("T", "K"): 0.21,
        ("K", "T"): 0.21,
        ("T", "R"): 0.22,
        ("R", "T"): 0.22,
        ("H", "E"): 0.2,
        ("E", "H"): 0.2,
        ("H", "Q"): 0.2,
        ("Q", "H"): 0.2,
        ("H", "D"): 0.21,
        ("D", "H"): 0.21,
        ("H", "N"): 0.21,
        ("N", "H"): 0.21,
        ("H", "K"): 0.2,
        ("K", "H"): 0.2,
        ("H", "R"): 0.2,
        ("R", "H"): 0.2,
        ("E", "Q"): 0.0,
        ("Q", "E"): 0.0,
        ("E", "D"): 0.03,
        ("D", "E"): 0.03,
        ("E", "N"): 0.03,
        ("N", "E"): 0.03,
        ("E", "K"): 0.0,
        ("K", "E"): 0.0,
        ("E", "R"): 0.05,
        ("R", "E"): 0.05,
        ("Q", "D"): 0.03,
        ("D", "Q"): 0.03,
        ("Q", "N"): 0.03,
        ("N", "Q"): 0.03,
        ("Q", "K"): 0.0,
        ("K", "Q"): 0.0,
        ("Q", "R"): 0.05,
        ("R", "Q"): 0.05,
        ("D", "N"): 0.0,
        ("N", "D"): 0.0,
        ("D", "K"): 0.03,
        ("K", "D"): 0.03,
        ("D", "R"): 0.08,
        ("R", "D"): 0.08,
        ("N", "K"): 0.03,
        ("K", "N"): 0.03,
        ("N", "R"): 0.08,
        ("R", "N"): 0.08,
        ("K", "R"): 0.05,
        ("R", "K"): 0.05,
        ("F", "F"): 0,
        ("M", "M"): 0,
        ("L", "L"): 0,
        ("I", "I"): 0,
        ("V", "V"): 0,
        ("P", "P"): 0,
        ("Y", "Y"): 0,
        ("W", "W"): 0,
        ("C", "C"): 0,
        ("A", "A"): 0,
        ("G", "G"): 0,
        ("S", "S"): 0,
        ("T", "T"): 0,
        ("H", "H"): 0,
        ("E", "E"): 0,
        ("Q", "Q"): 0,
        ("D", "D"): 0,
        ("N", "N"): 0,
        ("K", "K"): 0,
        ("R", "R"): 0,
    }


    return epstein_dict


def make_asymmetric_epstein_dict() -> dict[tuple[str, str], float]:
    """
    :return: A dictionary representative of TABLE 2 from 'Non-randomness of Amino-acid Changes in the Evolution of
             Homologous Proteins' by Charles J. Epstein. Keys are pairs of strings. Cf. scientific paper for
             more details on how those scores are derived.
    """
    epstein_dict: dict[tuple[str, str], float] = {
        ("F", "M"): 0.05,
        ("M", "F"): 0.1,
        ("F", "L"): 0.08,
        ("L", "F"): 0.15,
        ("F", "I"): 0.08,
        ("I", "F"): 0.15,
        ("F", "V"): 0.1,
        ("V", "F"): 0.2,
        ("F", "P"): 0.1,
        ("P", "F"): 0.2,
        ("F", "Y"): 0.21,
        ("Y", "F"): 0.2,
        ("F", "W"): 0.25,
        ("W", "F"): 0.21,
        ("F", "C"): 0.22,
        ("C", "F"): 0.28,
        ("F", "A"): 0.43,
        ("A", "F"): 0.5,
        ("F", "G"): 0.53,
        ("G", "F"): 0.61,
        ("F", "S"): 0.81,
        ("S", "F"): 0.81,
        ("F", "T"): 0.81,
        ("T", "F"): 0.81,
        ("F", "H"): 0.8,
        ("H", "F"): 0.8,
        ("F", "E"): 1.0,
        ("E", "F"): 1.0,
        ("F", "Q"): 1.0,
        ("Q", "F"): 1.0,
        ("F", "D"): 1.0,
        ("D", "F"): 1.0,
        ("F", "N"): 1.0,
        ("N", "F"): 1.0,
        ("F", "K"): 1.0,
        ("K", "F"): 1.0,
        ("F", "R"): 1.0,
        ("R", "F"): 1.0,
        ("M", "L"): 0.03,
        ("L", "M"): 0.05,
        ("M", "I"): 0.03,
        ("I", "M"): 0.05,
        ("M", "V"): 0.1,
        ("V", "M"): 0.1,
        ("M", "P"): 0.1,
        ("P", "M"): 0.1,
        ("M", "Y"): 0.25,
        ("Y", "M"): 0.22,
        ("M", "W"): 0.32,
        ("W", "M"): 0.24,
        ("M", "C"): 0.21,
        ("C", "M"): 0.22,
        ("M", "A"): 0.41,
        ("A", "M"): 0.45,
        ("M", "G"): 0.42,
        ("G", "M"): 0.56,
        ("M", "S"): 0.8,
        ("S", "M"): 0.8,
        ("M", "T"): 0.8,
        ("T", "M"): 0.8,
        ("M", "H"): 0.8,
        ("H", "M"): 0.8,
        ("M", "E"): 1.0,
        ("E", "M"): 1.0,
        ("M", "Q"): 1.0,
        ("Q", "M"): 1.0,
        ("M", "D"): 1.0,
        ("D", "M"): 1.0,
        ("M", "N"): 1.0,
        ("N", "M"): 1.0,
        ("M", "K"): 1.0,
        ("K", "M"): 1.0,
        ("M", "R"): 1.0,
        ("R", "M"): 1.0,
        ("L", "I"): 0.0,
        ("I", "L"): 0.0,
        ("L", "V"): 0.03,
        ("V", "L"): 0.05,
        ("L", "P"): 0.03,
        ("P", "L"): 0.05,
        ("L", "Y"): 0.28,
        ("Y", "L"): 0.22,
        ("L", "W"): 0.36,
        ("W", "L"): 0.25,
        ("L", "C"): 0.2,
        ("C", "L"): 0.21,
        ("L", "A"): 0.43,
        ("A", "L"): 0.43,
        ("L", "G"): 0.51,
        ("G", "L"): 0.54,
        ("L", "S"): 0.8,
        ("S", "L"): 0.8,
        ("L", "T"): 0.8,
        ("T", "L"): 0.8,
        ("L", "H"): 0.81,
        ("H", "L"): 1.0,
        ("L", "E"): 1.0,
        ("E", "L"): 1.0,
        ("L", "Q"): 1.0,
        ("Q", "L"): 1.0,
        ("L", "D"): 1.0,
        ("D", "L"): 1.0,
        ("L", "N"): 1.0,
        ("N", "L"): 1.0,
        ("L", "K"): 1.0,
        ("K", "L"): 1.0,
        ("L", "R"): 1.01,
        ("R", "L"): 1.0,
        ("I", "V"): 0.03,
        ("V", "I"): 0.05,
        ("I", "P"): 0.03,
        ("P", "I"): 0.05,
        ("I", "Y"): 0.28,
        ("Y", "I"): 0.22,
        ("I", "W"): 0.36,
        ("W", "I"): 0.25,
        ("I", "C"): 0.2,
        ("C", "I"): 0.21,
        ("I", "A"): 0.43,
        ("A", "I"): 0.43,
        ("I", "G"): 0.51,
        ("G", "I"): 0.54,
        ("I", "S"): 0.8,
        ("S", "I"): 0.8,
        ("I", "T"): 0.8,
        ("T", "I"): 0.8,
        ("I", "H"): 0.81,
        ("H", "I"): 1.0,
        ("I", "E"): 1.0,
        ("E", "I"): 1.0,
        ("I", "Q"): 1.0,
        ("Q", "I"): 1.0,
        ("I", "D"): 1.0,
        ("D", "I"): 1.0,
        ("I", "N"): 1.0,
        ("N", "I"): 1.0,
        ("I", "K"): 1.0,
        ("K", "I"): 1.0,
        ("I", "R"): 1.01,
        ("R", "I"): 1.0,
        ("V", "P"): 0.0,
        ("P", "V"): 0.0,
        ("V", "Y"): 0.32,
        ("Y", "V"): 0.24,
        ("V", "W"): 0.4,
        ("W", "V"): 0.27,
        ("V", "C"): 0.2,
        ("C", "V"): 0.2,
        ("V", "A"): 0.4,
        ("A", "V"): 0.41,
        ("V", "G"): 0.5,
        ("G", "V"): 0.52,
        ("V", "S"): 0.8,
        ("S", "V"): 0.8,
        ("V", "T"): 0.8,
        ("T", "V"): 0.8,
        ("V", "H"): 0.81,
        ("H", "V"): 0.8,
        ("V", "E"): 1.0,
        ("E", "V"): 1.0,
        ("V", "Q"): 1.0,
        ("Q", "V"): 1.0,
        ("V", "D"): 1.0,
        ("D", "V"): 1.0,
        ("V", "N"): 1.0,
        ("N", "V"): 1.0,
        ("V", "K"): 1.0,
        ("K", "V"): 1.0,
        ("V", "R"): 1.02,
        ("R", "V"): 1.01,
        ("P", "Y"): 0.32,
        ("Y", "P"): 0.24,
        ("P", "W"): 0.4,
        ("W", "P"): 0.27,
        ("P", "C"): 0.2,
        ("C", "P"): 0.2,
        ("P", "A"): 0.4,
        ("A", "P"): 0.41,
        ("P", "G"): 0.5,
        ("G", "P"): 0.52,
        ("P", "S"): 0.8,
        ("S", "P"): 0.8,
        ("P", "T"): 0.8,
        ("T", "P"): 0.8,
        ("P", "H"): 0.81,
        ("H", "P"): 0.8,
        ("P", "E"): 1.0,
        ("E", "P"): 1.0,
        ("P", "Q"): 1.0,
        ("Q", "P"): 1.0,
        ("P", "D"): 1.0,
        ("D", "P"): 1.0,
        ("P", "N"): 1.0,
        ("N", "P"): 1.0,
        ("P", "K"): 1.0,
        ("K", "P"): 1.0,
        ("P", "R"): 1.02,
        ("R", "P"): 1.01,
        ("Y", "W"): 0.1,
        ("W", "Y"): 0.05,
        ("Y", "C"): 0.13,
        ("C", "Y"): 0.25,
        ("Y", "A"): 0.27,
        ("A", "Y"): 0.4,
        ("Y", "G"): 0.36,
        ("G", "Y"): 0.5,
        ("Y", "S"): 0.62,
        ("S", "Y"): 0.62,
        ("Y", "T"): 0.61,
        ("T", "Y"): 0.61,
        ("Y", "H"): 0.6,
        ("H", "Y"): 0.6,
        ("Y", "E"): 0.8,
        ("E", "Y"): 0.8,
        ("Y", "Q"): 0.8,
        ("Q", "Y"): 0.8,
        ("Y", "D"): 0.81,
        ("D", "Y"): 0.81,
        ("Y", "N"): 0.81,
        ("N", "Y"): 0.81,
        ("Y", "K"): 0.8,
        ("K", "Y"): 0.8,
        ("Y", "R"): 0.8,
        ("R", "Y"): 0.8,
        ("W", "C"): 0.18,
        ("C", "W"): 0.35,
        ("W", "A"): 0.3,
        ("A", "W"): 0.49,
        ("W", "G"): 0.39,
        ("G", "W"): 0.58,
        ("W", "S"): 0.63,
        ("S", "W"): 0.63,
        ("W", "T"): 0.63,
        ("T", "W"): 0.63,
        ("W", "H"): 0.61,
        ("H", "W"): 0.61,
        ("W", "E"): 0.81,
        ("E", "W"): 0.81,
        ("W", "Q"): 0.81,
        ("Q", "W"): 0.81,
        ("W", "D"): 0.81,
        ("D", "W"): 0.81,
        ("W", "N"): 0.81,
        ("N", "W"): 0.81,
        ("W", "K"): 0.81,
        ("K", "W"): 0.81,
        ("W", "R"): 0.8,
        ("R", "W"): 0.8,
        ("C", "A"): 0.25,
        ("A", "C"): 0.22,
        ("C", "G"): 0.31,
        ("G", "C"): 0.34,
        ("C", "S"): 0.6,
        ("S", "C"): 0.6,
        ("C", "T"): 0.6,
        ("T", "C"): 0.6,
        ("C", "H"): 0.62,
        ("H", "C"): 0.61,
        ("C", "E"): 0.81,
        ("E", "C"): 0.8,
        ("C", "Q"): 0.81,
        ("Q", "C"): 0.8,
        ("C", "D"): 0.8,
        ("D", "C"): 0.8,
        ("C", "N"): 0.8,
        ("N", "C"): 0.8,
        ("C", "K"): 0.81,
        ("K", "C"): 0.8,
        ("C", "R"): 0.82,
        ("R", "C"): 0.81,
        ("A", "G"): 0.1,
        ("G", "A"): 0.1,
        ("A", "S"): 0.4,
        ("S", "A"): 0.4,
        ("A", "T"): 0.41,
        ("T", "A"): 0.4,
        ("A", "H"): 0.47,
        ("H", "A"): 0.42,
        ("A", "E"): 0.63,
        ("E", "A"): 0.61,
        ("A", "Q"): 0.63,
        ("Q", "A"): 0.61,
        ("A", "D"): 0.62,
        ("D", "A"): 0.61,
        ("A", "N"): 0.62,
        ("N", "A"): 0.61,
        ("A", "K"): 0.63,
        ("K", "A"): 0.61,
        ("A", "R"): 0.67,
        ("R", "A"): 0.62,
        ("G", "S"): 0.32,
        ("S", "G"): 0.3,
        ("G", "T"): 0.34,
        ("T", "G"): 0.31,
        ("G", "H"): 0.42,
        ("H", "G"): 0.34,
        ("G", "E"): 0.56,
        ("E", "G"): 0.52,
        ("G", "Q"): 0.56,
        ("Q", "G"): 0.52,
        ("G", "D"): 0.54,
        ("D", "G"): 0.51,
        ("G", "N"): 0.54,
        ("N", "G"): 0.51,
        ("G", "K"): 0.56,
        ("K", "G"): 0.52,
        ("G", "R"): 0.61,
        ("R", "G"): 0.53,
        ("S", "T"): 0.03,
        ("T", "S"): 0.03,
        ("S", "H"): 0.1,
        ("H", "S"): 0.1,
        ("S", "E"): 0.21,
        ("E", "S"): 0.22,
        ("S", "Q"): 0.21,
        ("Q", "S"): 0.22,
        ("S", "D"): 0.2,
        ("D", "S"): 0.21,
        ("S", "N"): 0.2,
        ("N", "S"): 0.21,
        ("S", "K"): 0.21,
        ("K", "S"): 0.22,
        ("S", "R"): 0.24,
        ("R", "S"): 0.24,
        ("T", "H"): 0.08,
        ("H", "T"): 0.08,
        ("T", "E"): 0.21,
        ("E", "T"): 0.21,
        ("T", "Q"): 0.21,
        ("Q", "T"): 0.21,
        ("T", "D"): 0.2,
        ("D", "T"): 0.2,
        ("T", "N"): 0.2,
        ("N", "T"): 0.2,
        ("T", "K"): 0.21,
        ("K", "T"): 0.21,
        ("T", "R"): 0.22,
        ("R", "T"): 0.22,
        ("H", "E"): 0.2,
        ("E", "H"): 0.2,
        ("H", "Q"): 0.2,
        ("Q", "H"): 0.2,
        ("H", "D"): 0.21,
        ("D", "H"): 0.21,
        ("H", "N"): 0.21,
        ("N", "H"): 0.21,
        ("H", "K"): 0.2,
        ("K", "H"): 0.2,
        ("H", "R"): 0.2,
        ("R", "H"): 0.2,
        ("E", "Q"): 0.0,
        ("Q", "E"): 0.0,
        ("E", "D"): 0.03,
        ("D", "E"): 0.03,
        ("E", "N"): 0.03,
        ("N", "E"): 0.03,
        ("E", "K"): 0.0,
        ("K", "E"): 0.0,
        ("E", "R"): 0.05,
        ("R", "E"): 0.05,
        ("Q", "D"): 0.03,
        ("D", "Q"): 0.03,
        ("Q", "N"): 0.03,
        ("N", "Q"): 0.03,
        ("Q", "K"): 0.0,
        ("K", "Q"): 0.0,
        ("Q", "R"): 0.05,
        ("R", "Q"): 0.05,
        ("D", "N"): 0.0,
        ("N", "D"): 0.0,
        ("D", "K"): 0.03,
        ("K", "D"): 0.03,
        ("D", "R"): 0.08,
        ("R", "D"): 0.08,
        ("N", "K"): 0.03,
        ("K", "N"): 0.03,
        ("N", "R"): 0.08,
        ("R", "N"): 0.08,
        ("K", "R"): 0.05,
        ("R", "K"): 0.05,
        ("F", "F"): 0,
        ("M", "M"): 0,
        ("L", "L"): 0,
        ("I", "I"): 0,
        ("V", "V"): 0,
        ("P", "P"): 0,
        ("Y", "Y"): 0,
        ("W", "W"): 0,
        ("C", "C"): 0,
        ("A", "A"): 0,
        ("G", "G"): 0,
        ("S", "S"): 0,
        ("T", "T"): 0,
        ("H", "H"): 0,
        ("E", "E"): 0,
        ("Q", "Q"): 0,
        ("D", "D"): 0,
        ("N", "N"): 0,
        ("K", "K"): 0,
        ("R", "R"): 0,
    }

    return epstein_dict


def make_symmetric_experimental_exchangeability_dict() -> dict[tuple[str, str], float]:
    """
    :return: A dictionary representative of TABLE 2 & 3 from 'The Exchangeability of Amino Acids in Proteins' by Lev Y.
             Yampolsky and Arlin Stoltzfus. Keys are pairs of strings. Cf. scientific paper for more details on how
             those scores are derived. Cf. paper for formula used to infer the symmetrised version (weighted mean).
             It does not have any missing values like the asymmetrical verison does, cf. article:
             'The symmetric matrix EXSij has no missing values because exchange data are available for every possible
              unordered pair of amino acids.'
    """
    symmetric_exp_ex_dict: dict[tuple[str, str], float] = {
        ('C', 'S'): 357.5192307692308,
        ('S', 'C'): 357.5192307692308,
        ('C', 'T'): 313.34285714285716,
        ('T', 'C'): 313.34285714285716,
        ('C', 'P'): 309.0,
        ('P', 'C'): 309.0,
        ('C', 'A'): 386.0,
        ('A', 'C'): 386.0,
        ('C', 'G'): 269.8636363636364,
        ('G', 'C'): 269.8636363636364,
        ('C', 'N'): 229.53571428571428,
        ('N', 'C'): 229.53571428571428,
        ('C', 'D'): 279.3225806451613,
        ('D', 'C'): 279.3225806451613,
        ('C', 'E'): 318.4375,
        ('E', 'C'): 318.4375,
        ('C', 'Q'): 383.0,
        ('Q', 'C'): 383.0,
        ('C', 'H'): 306.6666666666667,
        ('H', 'C'): 306.6666666666667,
        ('C', 'R'): 239.72727272727272,
        ('R', 'C'): 239.72727272727272,
        ('C', 'K'): 315.7096774193548,
        ('K', 'C'): 315.7096774193548,
        ('C', 'M'): 317.3333333333333,
        ('M', 'C'): 317.3333333333333,
        ('C', 'I'): 354.3333333333333,
        ('I', 'C'): 354.3333333333333,
        ('C', 'L'): 363.85483870967744,
        ('L', 'C'): 363.85483870967744,
        ('C', 'V'): 370.0408163265306,
        ('V', 'C'): 370.0408163265306,
        ('C', 'F'): 223.72413793103448,
        ('F', 'C'): 223.72413793103448,
        ('C', 'Y'): 191.02631578947367,
        ('Y', 'C'): 191.02631578947367,
        ('C', 'W'): 137.30769230769232,
        ('W', 'C'): 137.30769230769232,
        ('S', 'T'): 421.7536231884058,
        ('T', 'S'): 421.7536231884058,
        ('S', 'P'): 309.5,
        ('P', 'S'): 309.5,
        ('S', 'A'): 434.1240310077519,
        ('A', 'S'): 434.1240310077519,
        ('S', 'G'): 353.9428571428571,
        ('G', 'S'): 353.9428571428571,
        ('S', 'N'): 360.9574468085106,
        ('N', 'S'): 360.9574468085106,
        ('S', 'D'): 280.13157894736844,
        ('D', 'S'): 280.13157894736844,
        ('S', 'E'): 348.5135135135135,
        ('E', 'S'): 348.5135135135135,
        ('S', 'Q'): 396.30263157894734,
        ('Q', 'S'): 396.30263157894734,
        ('S', 'H'): 356.29411764705884,
        ('H', 'S'): 356.29411764705884,
        ('S', 'R'): 315.46666666666664,
        ('R', 'S'): 315.46666666666664,
        ('S', 'K'): 324.20512820512823,
        ('K', 'S'): 324.20512820512823,
        ('S', 'M'): 345.8888888888889,
        ('M', 'S'): 345.8888888888889,
        ('S', 'I'): 204.77966101694915,
        ('I', 'S'): 204.77966101694915,
        ('S', 'L'): 242.05172413793105,
        ('L', 'S'): 242.05172413793105,
        ('S', 'V'): 328.962962962963,
        ('V', 'S'): 328.962962962963,
        ('S', 'F'): 273.3333333333333,
        ('F', 'S'): 273.3333333333333,
        ('S', 'Y'): 247.98591549295776,
        ('Y', 'S'): 247.98591549295776,
        ('S', 'W'): 110.54545454545455,
        ('W', 'S'): 110.54545454545455,
        ('T', 'P'): 198.16,
        ('P', 'T'): 198.16,
        ('T', 'A'): 383.05263157894734,
        ('A', 'T'): 383.05263157894734,
        ('T', 'G'): 295.135593220339,
        ('G', 'T'): 295.135593220339,
        ('T', 'N'): 285.8484848484849,
        ('N', 'T'): 285.8484848484849,
        ('T', 'D'): 217.5,
        ('D', 'T'): 217.5,
        ('T', 'E'): 224.0,
        ('E', 'T'): 224.0,
        ('T', 'Q'): 314.625,
        ('Q', 'T'): 314.625,
        ('T', 'H'): 239.16666666666666,
        ('H', 'T'): 239.16666666666666,
        ('T', 'R'): 281.6923076923077,
        ('R', 'T'): 281.6923076923077,
        ('T', 'K'): 339.8095238095238,
        ('K', 'T'): 339.8095238095238,
        ('T', 'M'): 192.875,
        ('M', 'T'): 192.875,
        ('T', 'I'): 195.1875,
        ('I', 'T'): 195.1875,
        ('T', 'L'): 240.08333333333334,
        ('L', 'T'): 240.08333333333334,
        ('T', 'V'): 377.75,
        ('V', 'T'): 377.75,
        ('T', 'F'): 271.22222222222223,
        ('F', 'T'): 271.22222222222223,
        ('T', 'Y'): 260.0,
        ('Y', 'T'): 260.0,
        ('T', 'W'): 57.833333333333336,
        ('W', 'T'): 57.833333333333336,
        ('P', 'A'): 324.8534482758621,
        ('A', 'P'): 324.8534482758621,
        ('P', 'G'): 244.21052631578948,
        ('G', 'P'): 244.21052631578948,
        ('P', 'N'): 285.69444444444446,
        ('N', 'P'): 285.69444444444446,
        ('P', 'D'): 225.1,
        ('D', 'P'): 225.1,
        ('P', 'E'): 269.9622641509434,
        ('E', 'P'): 269.9622641509434,
        ('P', 'Q'): 274.1111111111111,
        ('Q', 'P'): 274.1111111111111,
        ('P', 'H'): 313.9347826086956,
        ('H', 'P'): 313.9347826086956,
        ('P', 'R'): 198.81012658227849,
        ('R', 'P'): 198.81012658227849,
        ('P', 'K'): 242.07272727272726,
        ('K', 'P'): 242.07272727272726,
        ('P', 'M'): 128.0,
        ('M', 'P'): 128.0,
        ('P', 'I'): 153.2325581395349,
        ('I', 'P'): 153.2325581395349,
        ('P', 'L'): 186.63716814159292,
        ('L', 'P'): 186.63716814159292,
        ('P', 'V'): 221.75471698113208,
        ('V', 'P'): 221.75471698113208,
        ('P', 'F'): 246.97297297297297,
        ('F', 'P'): 246.97297297297297,
        ('P', 'Y'): 251.77777777777777,
        ('Y', 'P'): 251.77777777777777,
        ('P', 'W'): 185.5,
        ('W', 'P'): 185.5,
        ('A', 'G'): 378.171974522293,
        ('G', 'A'): 378.171974522293,
        ('A', 'N'): 403.3333333333333,
        ('N', 'A'): 403.3333333333333,
        ('A', 'D'): 275.85714285714283,
        ('D', 'A'): 275.85714285714283,
        ('A', 'E'): 396.45299145299145,
        ('E', 'A'): 396.45299145299145,
        ('A', 'Q'): 401.1958762886598,
        ('Q', 'A'): 401.1958762886598,
        ('A', 'H'): 346.08,
        ('H', 'A'): 346.08,
        ('A', 'R'): 367.16,
        ('R', 'A'): 367.16,
        ('A', 'K'): 412.5,
        ('K', 'A'): 412.5,
        ('A', 'M'): 386.53846153846155,
        ('M', 'A'): 386.53846153846155,
        ('A', 'I'): 322.4782608695652,
        ('I', 'A'): 322.4782608695652,
        ('A', 'L'): 330.55102040816325,
        ('L', 'A'): 330.55102040816325,
        ('A', 'V'): 370.57894736842104,
        ('V', 'A'): 370.57894736842104,
        ('A', 'F'): 284.11842105263156,
        ('F', 'A'): 284.11842105263156,
        ('A', 'Y'): 323.7,
        ('Y', 'A'): 323.7,
        ('A', 'W'): 97.0,
        ('W', 'A'): 97.0,
        ('G', 'N'): 346.75555555555553,
        ('N', 'G'): 346.75555555555553,
        ('G', 'D'): 241.4814814814815,
        ('D', 'G'): 241.4814814814815,
        ('G', 'E'): 302.55882352941177,
        ('E', 'G'): 302.55882352941177,
        ('G', 'Q'): 332.4634146341463,
        ('Q', 'G'): 332.4634146341463,
        ('G', 'H'): 274.8360655737705,
        ('H', 'G'): 274.8360655737705,
        ('G', 'R'): 208.41666666666666,
        ('R', 'G'): 208.41666666666666,
        ('G', 'K'): 358.2857142857143,
        ('K', 'G'): 358.2857142857143,
        ('G', 'M'): 210.75862068965517,
        ('M', 'G'): 210.75862068965517,
        ('G', 'I'): 147.03703703703704,
        ('I', 'G'): 147.03703703703704,
        ('G', 'L'): 197.94915254237287,
        ('L', 'G'): 197.94915254237287,
        ('G', 'V'): 245.36936936936937,
        ('V', 'G'): 245.36936936936937,
        ('G', 'F'): 148.52631578947367,
        ('F', 'G'): 148.52631578947367,
        ('G', 'Y'): 245.2,
        ('Y', 'G'): 245.2,
        ('G', 'W'): 167.72,
        ('W', 'G'): 167.72,
        ('N', 'D'): 203.8,
        ('D', 'N'): 203.8,
        ('N', 'E'): 257.15625,
        ('E', 'N'): 257.15625,
        ('N', 'Q'): 301.75,
        ('Q', 'N'): 301.75,
        ('N', 'H'): 243.0392156862745,
        ('H', 'N'): 243.0392156862745,
        ('N', 'R'): 235.6764705882353,
        ('R', 'N'): 235.6764705882353,
        ('N', 'K'): 286.9310344827586,
        ('K', 'N'): 286.9310344827586,
        ('N', 'M'): 348.0,
        ('M', 'N'): 348.0,
        ('N', 'I'): 178.0,
        ('I', 'N'): 178.0,
        ('N', 'L'): 217.22222222222223,
        ('L', 'N'): 217.22222222222223,
        ('N', 'V'): 194.53846153846155,
        ('V', 'N'): 194.53846153846155,
        ('N', 'F'): 200.75,
        ('F', 'N'): 200.75,
        ('N', 'Y'): 221.82608695652175,
        ('Y', 'N'): 221.82608695652175,
        ('N', 'W'): 120.0,
        ('W', 'N'): 120.0,
        ('D', 'E'): 385.69117647058823,
        ('E', 'D'): 385.69117647058823,
        ('D', 'Q'): 257.2647058823529,
        ('Q', 'D'): 257.2647058823529,
        ('D', 'H'): 274.74074074074076,
        ('H', 'D'): 274.74074074074076,
        ('D', 'R'): 238.16216216216216,
        ('R', 'D'): 238.16216216216216,
        ('D', 'K'): 241.81578947368422,
        ('K', 'D'): 241.81578947368422,
        ('D', 'M'): 300.125,
        ('M', 'D'): 300.125,
        ('D', 'I'): 178.11111111111111,
        ('I', 'D'): 178.11111111111111,
        ('D', 'L'): 215.33333333333334,
        ('L', 'D'): 215.33333333333334,
        ('D', 'V'): 191.19444444444446,
        ('V', 'D'): 191.19444444444446,
        ('D', 'F'): 217.11111111111111,
        ('F', 'D'): 217.11111111111111,
        ('D', 'Y'): 199.45901639344262,
        ('Y', 'D'): 199.45901639344262,
        ('D', 'W'): 103.0,
        ('W', 'D'): 103.0,
        ('E', 'Q'): 397.7368421052632,
        ('Q', 'E'): 397.7368421052632,
        ('E', 'H'): 360.175,
        ('H', 'E'): 360.175,
        ('E', 'R'): 264.01666666666665,
        ('R', 'E'): 264.01666666666665,
        ('E', 'K'): 295.7042253521127,
        ('K', 'E'): 295.7042253521127,
        ('E', 'M'): 271.8888888888889,
        ('M', 'E'): 271.8888888888889,
        ('E', 'I'): 216.97368421052633,
        ('I', 'E'): 216.97368421052633,
        ('E', 'L'): 240.9784946236559,
        ('L', 'E'): 240.9784946236559,
        ('E', 'V'): 223.8,
        ('V', 'E'): 223.8,
        ('E', 'F'): 246.21052631578948,
        ('F', 'E'): 246.21052631578948,
        ('E', 'Y'): 290.6666666666667,
        ('Y', 'E'): 290.6666666666667,
        ('E', 'W'): 95.0,
        ('W', 'E'): 95.0,
        ('Q', 'H'): 367.5,
        ('H', 'Q'): 367.5,
        ('Q', 'R'): 332.13157894736844,
        ('R', 'Q'): 332.13157894736844,
        ('Q', 'K'): 401.7439024390244,
        ('K', 'Q'): 401.7439024390244,
        ('Q', 'M'): 418.44444444444446,
        ('M', 'Q'): 418.44444444444446,
        ('Q', 'I'): 227.31578947368422,
        ('I', 'Q'): 227.31578947368422,
        ('Q', 'L'): 306.89473684210526,
        ('L', 'Q'): 306.89473684210526,
        ('Q', 'V'): 311.6666666666667,
        ('V', 'Q'): 311.6666666666667,
        ('Q', 'F'): 340.27906976744185,
        ('F', 'Q'): 340.27906976744185,
        ('Q', 'Y'): 363.6666666666667,
        ('Y', 'Q'): 363.6666666666667,
        ('Q', 'W'): 105.54545454545455,
        ('W', 'Q'): 105.54545454545455,
        ('H', 'R'): 266.8,
        ('R', 'H'): 266.8,
        ('H', 'K'): 352.0,
        ('K', 'H'): 352.0,
        ('H', 'M'): 289.77272727272725,
        ('M', 'H'): 289.77272727272725,
        ('H', 'I'): 218.2,
        ('I', 'H'): 218.2,
        ('H', 'L'): 305.609756097561,
        ('L', 'H'): 305.609756097561,
        ('H', 'V'): 253.22222222222223,
        ('V', 'H'): 253.22222222222223,
        ('H', 'F'): 287.2068965517241,
        ('F', 'H'): 287.2068965517241,
        ('H', 'Y'): 221.90697674418604,
        ('Y', 'H'): 221.90697674418604,
        ('H', 'W'): 147.9090909090909,
        ('W', 'H'): 147.9090909090909,
        ('R', 'K'): 380.26506024096386,
        ('K', 'R'): 380.26506024096386,
        ('R', 'M'): 109.55555555555556,
        ('M', 'R'): 109.55555555555556,
        ('R', 'I'): 126.22222222222223,
        ('I', 'R'): 126.22222222222223,
        ('R', 'L'): 205.09016393442624,
        ('L', 'R'): 205.09016393442624,
        ('R', 'V'): 189.88888888888889,
        ('V', 'R'): 189.88888888888889,
        ('R', 'F'): 184.6888888888889,
        ('F', 'R'): 184.6888888888889,
        ('R', 'Y'): 297.5,
        ('Y', 'R'): 297.5,
        ('R', 'W'): 93.0,
        ('W', 'R'): 93.0,
        ('K', 'M'): 270.0,
        ('M', 'K'): 270.0,
        ('K', 'I'): 222.5686274509804,
        ('I', 'K'): 222.5686274509804,
        ('K', 'L'): 216.79545454545453,
        ('L', 'K'): 216.79545454545453,
        ('K', 'V'): 236.03846153846155,
        ('V', 'K'): 236.03846153846155,
        ('K', 'F'): 263.75,
        ('F', 'K'): 263.75,
        ('K', 'Y'): 284.2682926829268,
        ('Y', 'K'): 284.2682926829268,
        ('K', 'W'): 142.30769230769232,
        ('W', 'K'): 142.30769230769232,
        ('M', 'I'): 356.7,
        ('I', 'M'): 356.7,
        ('M', 'L'): 446.6363636363636,
        ('L', 'M'): 446.6363636363636,
        ('M', 'V'): 458.0,
        ('V', 'M'): 458.0,
        ('M', 'F'): 321.1764705882353,
        ('F', 'M'): 321.1764705882353,
        ('M', 'Y'): 313.6,
        ('Y', 'M'): 313.6,
        ('M', 'W'): 458.6666666666667,
        ('W', 'M'): 458.6666666666667,
        ('I', 'L'): 389.1012658227848,
        ('L', 'I'): 389.1012658227848,
        ('I', 'V'): 509.5531914893617,
        ('V', 'I'): 509.5531914893617,
        ('I', 'F'): 299.5483870967742,
        ('F', 'I'): 299.5483870967742,
        ('I', 'Y'): 323.0,
        ('Y', 'I'): 323.0,
        ('I', 'W'): 73.0,
        ('W', 'I'): 73.0,
        ('L', 'V'): 311.967032967033,
        ('V', 'L'): 311.967032967033,
        ('L', 'F'): 325.5652173913044,
        ('F', 'L'): 325.5652173913044,
        ('L', 'Y'): 309.8888888888889,
        ('Y', 'L'): 309.8888888888889,
        ('L', 'W'): 163.0,
        ('W', 'L'): 163.0,
        ('V', 'F'): 221.2153846153846,
        ('F', 'V'): 221.2153846153846,
        ('V', 'Y'): 209.0,
        ('Y', 'V'): 209.0,
        ('V', 'W'): 250.8,
        ('W', 'V'): 250.8,
        ('F', 'Y'): 348.6923076923077,
        ('Y', 'F'): 348.6923076923077,
        ('F', 'W'): 311.2,
        ('W', 'F'): 311.2,
        ('Y', 'W'): 286.5,
        ('W', 'Y'): 286.5,
        ('C', 'C'): 0,
        ('S', 'S'): 0,
        ('T', 'T'): 0,
        ('P', 'P'): 0,
        ('A', 'A'): 0,
        ('G', 'G'): 0,
        ('N', 'N'): 0,
        ('D', 'D'): 0,
        ('E', 'E'): 0,
        ('Q', 'Q'): 0,
        ('H', 'H'): 0,
        ('R', 'R'): 0,
        ('K', 'K'): 0,
        ('M', 'M'): 0,
        ('I', 'I'): 0,
        ('L', 'L'): 0,
        ('V', 'V'): 0,
        ('F', 'F'): 0,
        ('Y', 'Y'): 0,
        ('W', 'W'): 0,
    }

    return symmetric_exp_ex_dict


def make_asymmetric_experimental_exchangeability_dict() -> dict[tuple[str, str], int | None]:
    """
    :return: A dictionary representative of TABLE 3 from 'The Exchangeability of Amino Acids in Proteins' by Lev Y.
             Yampolsky and Arlin Stoltzfus. Keys are pairs of strings. First string of the tuple is the source and
             second string of the tuple is the destination. Cf. scientific paper for more details on how those
             scores are derived.
             Caution: it has missing values which is why the type of the output is either float or None.
    """
    asymmetric_exp_ex_dict: dict[tuple[str, str], int | None] = {
        ('C', 'S'): 258,
        ('S', 'C'): 373,
        ('C', 'T'): 121,
        ('T', 'C'): 325,
        ('C', 'P'): 201,
        ('P', 'C'): 345,
        ('C', 'A'): 334,
        ('A', 'C'): 393,
        ('C', 'G'): 288,
        ('G', 'C'): 267,
        ('C', 'N'): 109,
        ('N', 'C'): 234,
        ('C', 'D'): 109,
        ('D', 'C'): 285,
        ('C', 'E'): 270,
        ('E', 'C'): 332,
        ('C', 'Q'): 383,
        ('Q', 'C'): 383,
        ('C', 'H'): 258,
        ('H', 'C'): 331,
        ('C', 'R'): 306,
        ('R', 'C'): 225,
        ('C', 'K'): 252,
        ('K', 'C'): 331,
        ('C', 'M'): 169,
        ('M', 'C'): 347,
        ('C', 'I'): 109,
        ('I', 'C'): 362,
        ('C', 'L'): 347,
        ('L', 'C'): 366,
        ('C', 'V'): 89,
        ('V', 'C'): 382,
        ('C', 'F'): 349,
        ('F', 'C'): 176,
        ('C', 'Y'): 349,
        ('Y', 'C'): 142,
        ('C', 'W'): 139,
        ('W', 'C'): 137,
        ('S', 'T'): 481,
        ('T', 'S'): 408,
        ('S', 'P'): 249,
        ('P', 'S'): 392,
        ('S', 'A'): 490,
        ('A', 'S'): 384,
        ('S', 'G'): 418,
        ('G', 'S'): 304,
        ('S', 'N'): 390,
        ('N', 'S'): 355,
        ('S', 'D'): 314,
        ('D', 'S'): 275,
        ('S', 'E'): 343,
        ('E', 'S'): 355,
        ('S', 'Q'): 352,
        ('Q', 'S'): 443,
        ('S', 'H'): 353,
        ('H', 'S'): 365,
        ('S', 'R'): 363,
        ('R', 'S'): 270,
        ('S', 'K'): 275,
        ('K', 'S'): 376,
        ('S', 'M'): 321,
        ('M', 'S'): 353,
        ('S', 'I'): 270,
        ('I', 'S'): 196,
        ('S', 'L'): 295,
        ('L', 'S'): 212,
        ('S', 'V'): 358,
        ('V', 'S'): 326,
        ('S', 'F'): 334,
        ('F', 'S'): 152,
        ('S', 'Y'): 294,
        ('Y', 'S'): 173,
        ('S', 'W'): 160,
        ('W', 'S'): 92,
        ('T', 'P'): 164,
        ('P', 'T'): 286,
        ('T', 'A'): 402,
        ('A', 'T'): 312,
        ('T', 'G'): 332,
        ('G', 'T'): 187,
        ('T', 'N'): 240,
        ('N', 'T'): 329,
        ('T', 'D'): 190,
        ('D', 'T'): 245,
        ('T', 'E'): 212,
        ('E', 'T'): 292,
        ('T', 'Q'): 308,
        ('Q', 'T'): 361,
        ('T', 'H'): 246,
        ('H', 'T'): 205,
        ('T', 'R'): 299,
        ('R', 'T'): 199,
        ('T', 'K'): 256,
        ('K', 'T'): 476,
        ('T', 'M'): 152,
        ('M', 'T'): 261,
        ('T', 'I'): 198,
        ('I', 'T'): 193,
        ('T', 'L'): 271,
        ('L', 'T'): 165,
        ('T', 'V'): 362,
        ('V', 'T'): 398,
        ('T', 'F'): 273,
        ('F', 'T'): 257,
        ('T', 'Y'): 260,
        ('Y', 'T'): None,
        ('T', 'W'): 66,
        ('W', 'T'): 17,
        ('P', 'A'): 454,
        ('A', 'P'): 243,
        ('P', 'G'): 404,
        ('G', 'P'): 140,
        ('P', 'N'): 352,
        ('N', 'P'): 275,
        ('P', 'D'): 254,
        ('D', 'P'): 220,
        ('P', 'E'): 346,
        ('E', 'P'): 216,
        ('P', 'Q'): 384,
        ('Q', 'P'): 212,
        ('P', 'H'): 369,
        ('H', 'P'): 220,
        ('P', 'R'): 254,
        ('R', 'P'): 145,
        ('P', 'K'): 231,
        ('K', 'P'): 252,
        ('P', 'M'): 257,
        ('M', 'P'): 85,
        ('P', 'I'): 204,
        ('I', 'P'): 145,
        ('P', 'L'): 258,
        ('L', 'P'): 146,
        ('P', 'V'): 421,
        ('V', 'P'): 201,
        ('P', 'F'): 339,
        ('F', 'P'): 112,
        ('P', 'Y'): 298,
        ('Y', 'P'): 194,
        ('P', 'W'): 305,
        ('W', 'P'): 66,
        ('A', 'G'): 387,
        ('G', 'A'): 369,
        ('A', 'N'): 430,
        ('N', 'A'): 400,
        ('A', 'D'): 193,
        ('D', 'A'): 293,
        ('A', 'E'): 275,
        ('E', 'A'): 520,
        ('A', 'Q'): 320,
        ('Q', 'A'): 499,
        ('A', 'H'): 301,
        ('H', 'A'): 462,
        ('A', 'R'): 295,
        ('R', 'A'): 459,
        ('A', 'K'): 225,
        ('K', 'A'): 600,
        ('A', 'M'): 549,
        ('M', 'A'): 357,
        ('A', 'I'): 245,
        ('I', 'A'): 326,
        ('A', 'L'): 313,
        ('L', 'A'): 343,
        ('A', 'V'): 319,
        ('V', 'A'): 389,
        ('A', 'F'): 305,
        ('F', 'A'): 236,
        ('A', 'Y'): 286,
        ('Y', 'A'): 402,
        ('A', 'W'): 165,
        ('W', 'A'): 63,
        ('G', 'N'): 210,
        ('N', 'G'): 391,
        ('G', 'D'): 188,
        ('D', 'G'): 264,
        ('G', 'E'): 206,
        ('E', 'G'): 407,
        ('G', 'Q'): 272,
        ('Q', 'G'): 406,
        ('G', 'H'): 235,
        ('H', 'G'): 370,
        ('G', 'R'): 178,
        ('R', 'G'): 251,
        ('G', 'K'): 219,
        ('K', 'G'): 492,
        ('G', 'M'): 197,
        ('M', 'G'): 218,
        ('G', 'I'): 110,
        ('I', 'G'): 160,
        ('G', 'L'): 193,
        ('L', 'G'): 201,
        ('G', 'V'): 208,
        ('V', 'G'): 269,
        ('G', 'F'): 168,
        ('F', 'G'): 94,
        ('G', 'Y'): 188,
        ('Y', 'G'): 357,
        ('G', 'W'): 173,
        ('W', 'G'): 162,
        ('N', 'D'): 208,
        ('D', 'N'): 201,
        ('N', 'E'): 257,
        ('E', 'N'): 258,
        ('N', 'Q'): 298,
        ('Q', 'N'): 338,
        ('N', 'H'): 248,
        ('H', 'N'): 225,
        ('N', 'R'): 252,
        ('R', 'N'): 67,
        ('N', 'K'): 183,
        ('K', 'N'): 457,
        ('N', 'M'): 236,
        ('M', 'N'): 544,
        ('N', 'I'): 184,
        ('I', 'N'): 172,
        ('N', 'L'): 233,
        ('L', 'N'): 162,
        ('N', 'V'): 233,
        ('V', 'N'): 108,
        ('N', 'F'): 210,
        ('F', 'N'): 136,
        ('N', 'Y'): 251,
        ('Y', 'N'): 129,
        ('N', 'W'): 120,
        ('W', 'N'): None,
        ('D', 'E'): 344,
        ('E', 'D'): 533,
        ('D', 'Q'): 263,
        ('Q', 'D'): 68,
        ('D', 'H'): 298,
        ('H', 'D'): 141,
        ('D', 'R'): 252,
        ('R', 'D'): 124,
        ('D', 'K'): 208,
        ('K', 'D'): 465,
        ('D', 'M'): 245,
        ('M', 'D'): 392,
        ('D', 'I'): 299,
        ('I', 'D'): 27,
        ('D', 'L'): 236,
        ('L', 'D'): 112,
        ('D', 'V'): 175,
        ('V', 'D'): 228,
        ('D', 'F'): 233,
        ('F', 'D'): 90,
        ('D', 'Y'): 227,
        ('Y', 'D'): 87,
        ('D', 'W'): 103,
        ('W', 'D'): None,
        ('E', 'Q'): 341,
        ('Q', 'E'): 439,
        ('E', 'H'): 380,
        ('H', 'E'): 319,
        ('E', 'R'): 279,
        ('R', 'E'): 250,
        ('E', 'K'): 323,
        ('K', 'E'): 272,
        ('E', 'M'): 219,
        ('M', 'E'): 287,
        ('E', 'I'): 450,
        ('I', 'E'): 197,
        ('E', 'L'): 321,
        ('L', 'E'): 199,
        ('E', 'V'): 351,
        ('V', 'E'): 192,
        ('E', 'F'): 342,
        ('F', 'E'): 62,
        ('E', 'Y'): 348,
        ('Y', 'E'): 176,
        ('E', 'W'): 145,
        ('W', 'E'): 65,
        ('Q', 'H'): 396,
        ('H', 'Q'): 301,
        ('Q', 'R'): 366,
        ('R', 'Q'): 288,
        ('Q', 'K'): 354,
        ('K', 'Q'): 441,
        ('Q', 'M'): 504,
        ('M', 'Q'): 394,
        ('Q', 'I'): 467,
        ('I', 'Q'): 191,
        ('Q', 'L'): 391,
        ('L', 'Q'): 250,
        ('Q', 'V'): 603,
        ('V', 'Q'): 280,
        ('Q', 'F'): 383,
        ('F', 'Q'): 216,
        ('Q', 'Y'): 361,
        ('Y', 'Q'): 369,
        ('Q', 'W'): 159,
        ('W', 'Q'): 61,
        ('H', 'R'): 275,
        ('R', 'H'): 263,
        ('H', 'K'): 332,
        ('K', 'H'): 362,
        ('H', 'M'): 315,
        ('M', 'H'): 278,
        ('H', 'I'): 205,
        ('I', 'H'): 221,
        ('H', 'L'): 364,
        ('L', 'H'): 288,
        ('H', 'V'): 255,
        ('V', 'H'): 253,
        ('H', 'F'): 328,
        ('F', 'H'): 237,
        ('H', 'Y'): 260,
        ('Y', 'H'): 197,
        ('H', 'W'): 72,
        ('W', 'H'): 239,
        ('R', 'K'): 306,
        ('K', 'R'): 440,
        ('R', 'M'): 68,
        ('M', 'R'): 112,
        ('R', 'I'): 139,
        ('I', 'R'): 124,
        ('R', 'L'): 242,
        ('L', 'R'): 185,
        ('R', 'V'): 189,
        ('V', 'R'): 190,
        ('R', 'F'): 213,
        ('F', 'R'): 122,
        ('R', 'Y'): 272,
        ('Y', 'R'): 340,
        ('R', 'W'): 63,
        ('W', 'R'): 103,
        ('K', 'M'): 414,
        ('M', 'K'): 135,
        ('K', 'I'): 491,
        ('I', 'K'): 121,
        ('K', 'L'): 301,
        ('L', 'K'): 171,
        ('K', 'V'): 487,
        ('V', 'K'): 197,
        ('K', 'F'): 360,
        ('F', 'K'): 85,
        ('K', 'Y'): 343,
        ('Y', 'K'): 171,
        ('K', 'W'): 218,
        ('W', 'K'): 54,
        ('M', 'I'): 612,
        ('I', 'M'): 279,
        ('M', 'L'): 513,
        ('L', 'M'): 367,
        ('M', 'V'): 354,
        ('V', 'M'): 562,
        ('M', 'F'): 330,
        ('F', 'M'): 255,
        ('M', 'Y'): 308,
        ('Y', 'M'): 392,
        ('M', 'W'): 633,
        ('W', 'M'): 110,
        ('I', 'L'): 417,
        ('L', 'I'): 301,
        ('I', 'V'): 494,
        ('V', 'I'): 537,
        ('I', 'F'): 331,
        ('F', 'I'): 181,
        ('I', 'Y'): 323,
        ('Y', 'I'): None,
        ('I', 'W'): 73,
        ('W', 'I'): None,
        ('L', 'V'): 275,
        ('V', 'L'): 333,
        ('L', 'F'): 336,
        ('F', 'L'): 296,
        ('L', 'Y'): 295,
        ('Y', 'L'): 362,
        ('L', 'W'): 152,
        ('W', 'L'): 177,
        ('V', 'F'): 207,
        ('F', 'V'): 291,
        ('V', 'Y'): 209,
        ('Y', 'V'): None,
        ('V', 'W'): 286,
        ('W', 'V'): 110,
        ('F', 'Y'): 332,
        ('Y', 'F'): 360,
        ('F', 'W'): 232,
        ('W', 'F'): 364,
        ('Y', 'W'): 303,
        ('W', 'Y'): 281,
        ('C', 'C'): 0,
        ('S', 'S'): 0,
        ('T', 'T'): 0,
        ('P', 'P'): 0,
        ('A', 'A'): 0,
        ('G', 'G'): 0,
        ('N', 'N'): 0,
        ('D', 'D'): 0,
        ('E', 'E'): 0,
        ('Q', 'Q'): 0,
        ('H', 'H'): 0,
        ('R', 'R'): 0,
        ('K', 'K'): 0,
        ('M', 'M'): 0,
        ('I', 'I'): 0,
        ('L', 'L'): 0,
        ('V', 'V'): 0,
        ('F', 'F'): 0,
        ('Y', 'Y'): 0,
        ('W', 'W'): 0,
    }

    return asymmetric_exp_ex_dict


def make_grantham_dict() -> dict[tuple[str, str], int]:
    """
    :return: A dictionary representative of TABLE 2 from 'Amino Acid Difference Formula to Help Explain Protein
             Evolution' by R. Grantham. Keys are pairs of strings. Cf. scientific paper for more details on how
             those scores are derived.
    """
    grantham_dict: dict[tuple[str, str], int] = {
        ('A', 'A'): 0,
        ('A', 'C'): 195,
        ('A', 'D'): 126,
        ('A', 'E'): 107,
        ('A', 'F'): 113,
        ('A', 'G'): 60,
        ('A', 'H'): 86,
        ('A', 'I'): 94,
        ('A', 'K'): 106,
        ('A', 'L'): 96,
        ('A', 'M'): 84,
        ('A', 'N'): 111,
        ('A', 'P'): 27,
        ('A', 'Q'): 91,
        ('A', 'R'): 112,
        ('A', 'S'): 99,
        ('A', 'T'): 58,
        ('A', 'V'): 64,
        ('A', 'W'): 148,
        ('A', 'Y'): 112,
        ('C', 'A'): 195,
        ('C', 'C'): 0,
        ('C', 'D'): 154,
        ('C', 'E'): 170,
        ('C', 'F'): 205,
        ('C', 'G'): 159,
        ('C', 'H'): 174,
        ('C', 'I'): 198,
        ('C', 'K'): 202,
        ('C', 'L'): 198,
        ('C', 'M'): 196,
        ('C', 'N'): 139,
        ('C', 'P'): 169,
        ('C', 'Q'): 154,
        ('C', 'R'): 180,
        ('C', 'S'): 112,
        ('C', 'T'): 149,
        ('C', 'V'): 192,
        ('C', 'W'): 215,
        ('C', 'Y'): 194,
        ('D', 'A'): 126,
        ('D', 'C'): 154,
        ('D', 'D'): 0,
        ('D', 'E'): 45,
        ('D', 'F'): 177,
        ('D', 'G'): 94,
        ('D', 'H'): 81,
        ('D', 'I'): 168,
        ('D', 'K'): 101,
        ('D', 'L'): 172,
        ('D', 'M'): 160,
        ('D', 'N'): 23,
        ('D', 'P'): 108,
        ('D', 'Q'): 61,
        ('D', 'R'): 96,
        ('D', 'S'): 65,
        ('D', 'T'): 85,
        ('D', 'V'): 152,
        ('D', 'W'): 181,
        ('D', 'Y'): 160,
        ('E', 'A'): 107,
        ('E', 'C'): 170,
        ('E', 'D'): 45,
        ('E', 'E'): 0,
        ('E', 'F'): 140,
        ('E', 'G'): 98,
        ('E', 'H'): 40,
        ('E', 'I'): 134,
        ('E', 'K'): 56,
        ('E', 'L'): 138,
        ('E', 'M'): 126,
        ('E', 'N'): 42,
        ('E', 'P'): 93,
        ('E', 'Q'): 29,
        ('E', 'R'): 54,
        ('E', 'S'): 80,
        ('E', 'T'): 65,
        ('E', 'V'): 121,
        ('E', 'W'): 152,
        ('E', 'Y'): 122,
        ('F', 'A'): 113,
        ('F', 'C'): 205,
        ('F', 'D'): 177,
        ('F', 'E'): 140,
        ('F', 'F'): 0,
        ('F', 'G'): 153,
        ('F', 'H'): 100,
        ('F', 'I'): 21,
        ('F', 'K'): 102,
        ('F', 'L'): 22,
        ('F', 'M'): 28,
        ('F', 'N'): 158,
        ('F', 'P'): 114,
        ('F', 'Q'): 116,
        ('F', 'R'): 97,
        ('F', 'S'): 155,
        ('F', 'T'): 103,
        ('F', 'V'): 50,
        ('F', 'W'): 40,
        ('F', 'Y'): 22,
        ('G', 'A'): 60,
        ('G', 'C'): 159,
        ('G', 'D'): 94,
        ('G', 'E'): 98,
        ('G', 'F'): 153,
        ('G', 'G'): 0,
        ('G', 'H'): 98,
        ('G', 'I'): 135,
        ('G', 'K'): 127,
        ('G', 'L'): 138,
        ('G', 'M'): 127,
        ('G', 'N'): 80,
        ('G', 'P'): 42,
        ('G', 'Q'): 87,
        ('G', 'R'): 125,
        ('G', 'S'): 56,
        ('G', 'T'): 59,
        ('G', 'V'): 109,
        ('G', 'W'): 184,
        ('G', 'Y'): 147,
        ('H', 'A'): 86,
        ('H', 'C'): 174,
        ('H', 'D'): 81,
        ('H', 'E'): 40,
        ('H', 'F'): 100,
        ('H', 'G'): 98,
        ('H', 'H'): 0,
        ('H', 'I'): 94,
        ('H', 'K'): 32,
        ('H', 'L'): 99,
        ('H', 'M'): 87,
        ('H', 'N'): 68,
        ('H', 'P'): 77,
        ('H', 'Q'): 24,
        ('H', 'R'): 29,
        ('H', 'S'): 89,
        ('H', 'T'): 47,
        ('H', 'V'): 84,
        ('H', 'W'): 115,
        ('H', 'Y'): 83,
        ('I', 'A'): 94,
        ('I', 'C'): 198,
        ('I', 'D'): 168,
        ('I', 'E'): 134,
        ('I', 'F'): 21,
        ('I', 'G'): 135,
        ('I', 'H'): 94,
        ('I', 'I'): 0,
        ('I', 'K'): 102,
        ('I', 'L'): 5,
        ('I', 'M'): 10,
        ('I', 'N'): 149,
        ('I', 'P'): 95,
        ('I', 'Q'): 109,
        ('I', 'R'): 97,
        ('I', 'S'): 142,
        ('I', 'T'): 89,
        ('I', 'V'): 29,
        ('I', 'W'): 61,
        ('I', 'Y'): 33,
        ('K', 'A'): 106,
        ('K', 'C'): 202,
        ('K', 'D'): 101,
        ('K', 'E'): 56,
        ('K', 'F'): 102,
        ('K', 'G'): 127,
        ('K', 'H'): 32,
        ('K', 'I'): 102,
        ('K', 'K'): 0,
        ('K', 'L'): 107,
        ('K', 'M'): 95,
        ('K', 'N'): 94,
        ('K', 'P'): 103,
        ('K', 'Q'): 53,
        ('K', 'R'): 26,
        ('K', 'S'): 121,
        ('K', 'T'): 78,
        ('K', 'V'): 97,
        ('K', 'W'): 110,
        ('K', 'Y'): 85,
        ('L', 'A'): 96,
        ('L', 'C'): 198,
        ('L', 'D'): 172,
        ('L', 'E'): 138,
        ('L', 'F'): 22,
        ('L', 'G'): 138,
        ('L', 'H'): 99,
        ('L', 'I'): 5,
        ('L', 'K'): 107,
        ('L', 'L'): 0,
        ('L', 'M'): 15,
        ('L', 'N'): 153,
        ('L', 'P'): 98,
        ('L', 'Q'): 113,
        ('L', 'R'): 102,
        ('L', 'S'): 145,
        ('L', 'T'): 92,
        ('L', 'V'): 32,
        ('L', 'W'): 61,
        ('L', 'Y'): 36,
        ('M', 'A'): 84,
        ('M', 'C'): 196,
        ('M', 'D'): 160,
        ('M', 'E'): 126,
        ('M', 'F'): 28,
        ('M', 'G'): 127,
        ('M', 'H'): 87,
        ('M', 'I'): 10,
        ('M', 'K'): 95,
        ('M', 'L'): 15,
        ('M', 'M'): 0,
        ('M', 'N'): 142,
        ('M', 'P'): 87,
        ('M', 'Q'): 101,
        ('M', 'R'): 91,
        ('M', 'S'): 135,
        ('M', 'T'): 81,
        ('M', 'V'): 21,
        ('M', 'W'): 67,
        ('M', 'Y'): 36,
        ('N', 'A'): 111,
        ('N', 'C'): 139,
        ('N', 'D'): 23,
        ('N', 'E'): 42,
        ('N', 'F'): 158,
        ('N', 'G'): 80,
        ('N', 'H'): 68,
        ('N', 'I'): 149,
        ('N', 'K'): 94,
        ('N', 'L'): 153,
        ('N', 'M'): 142,
        ('N', 'N'): 0,
        ('N', 'P'): 91,
        ('N', 'Q'): 46,
        ('N', 'R'): 86,
        ('N', 'S'): 46,
        ('N', 'T'): 65,
        ('N', 'V'): 133,
        ('N', 'W'): 174,
        ('N', 'Y'): 143,
        ('P', 'A'): 27,
        ('P', 'C'): 169,
        ('P', 'D'): 108,
        ('P', 'E'): 93,
        ('P', 'F'): 114,
        ('P', 'G'): 42,
        ('P', 'H'): 77,
        ('P', 'I'): 95,
        ('P', 'K'): 103,
        ('P', 'L'): 98,
        ('P', 'M'): 87,
        ('P', 'N'): 91,
        ('P', 'P'): 0,
        ('P', 'Q'): 76,
        ('P', 'R'): 103,
        ('P', 'S'): 74,
        ('P', 'T'): 38,
        ('P', 'V'): 68,
        ('P', 'W'): 147,
        ('P', 'Y'): 110,
        ('Q', 'A'): 91,
        ('Q', 'C'): 154,
        ('Q', 'D'): 61,
        ('Q', 'E'): 29,
        ('Q', 'F'): 116,
        ('Q', 'G'): 87,
        ('Q', 'H'): 24,
        ('Q', 'I'): 109,
        ('Q', 'K'): 53,
        ('Q', 'L'): 113,
        ('Q', 'M'): 101,
        ('Q', 'N'): 46,
        ('Q', 'P'): 76,
        ('Q', 'Q'): 0,
        ('Q', 'R'): 43,
        ('Q', 'S'): 68,
        ('Q', 'T'): 42,
        ('Q', 'V'): 96,
        ('Q', 'W'): 130,
        ('Q', 'Y'): 99,
        ('R', 'A'): 112,
        ('R', 'C'): 180,
        ('R', 'D'): 96,
        ('R', 'E'): 54,
        ('R', 'F'): 97,
        ('R', 'G'): 125,
        ('R', 'H'): 29,
        ('R', 'I'): 97,
        ('R', 'K'): 26,
        ('R', 'L'): 102,
        ('R', 'M'): 91,
        ('R', 'N'): 86,
        ('R', 'P'): 103,
        ('R', 'Q'): 43,
        ('R', 'R'): 0,
        ('R', 'S'): 110,
        ('R', 'T'): 71,
        ('R', 'V'): 96,
        ('R', 'W'): 101,
        ('R', 'Y'): 77,
        ('S', 'A'): 99,
        ('S', 'C'): 112,
        ('S', 'D'): 65,
        ('S', 'E'): 80,
        ('S', 'F'): 155,
        ('S', 'G'): 56,
        ('S', 'H'): 89,
        ('S', 'I'): 142,
        ('S', 'K'): 121,
        ('S', 'L'): 145,
        ('S', 'M'): 135,
        ('S', 'N'): 46,
        ('S', 'P'): 74,
        ('S', 'Q'): 68,
        ('S', 'R'): 110,
        ('S', 'S'): 0,
        ('S', 'T'): 58,
        ('S', 'V'): 124,
        ('S', 'W'): 177,
        ('S', 'Y'): 144,
        ('T', 'A'): 58,
        ('T', 'C'): 149,
        ('T', 'D'): 85,
        ('T', 'E'): 65,
        ('T', 'F'): 103,
        ('T', 'G'): 59,
        ('T', 'H'): 47,
        ('T', 'I'): 89,
        ('T', 'K'): 78,
        ('T', 'L'): 92,
        ('T', 'M'): 81,
        ('T', 'N'): 65,
        ('T', 'P'): 38,
        ('T', 'Q'): 42,
        ('T', 'R'): 71,
        ('T', 'S'): 58,
        ('T', 'T'): 0,
        ('T', 'V'): 69,
        ('T', 'W'): 128,
        ('T', 'Y'): 92,
        ('V', 'A'): 64,
        ('V', 'C'): 192,
        ('V', 'D'): 152,
        ('V', 'E'): 121,
        ('V', 'F'): 50,
        ('V', 'G'): 109,
        ('V', 'H'): 84,
        ('V', 'I'): 29,
        ('V', 'K'): 97,
        ('V', 'L'): 32,
        ('V', 'M'): 21,
        ('V', 'N'): 133,
        ('V', 'P'): 68,
        ('V', 'Q'): 96,
        ('V', 'R'): 96,
        ('V', 'S'): 124,
        ('V', 'T'): 69,
        ('V', 'V'): 0,
        ('V', 'W'): 88,
        ('V', 'Y'): 55,
        ('W', 'A'): 148,
        ('W', 'C'): 215,
        ('W', 'D'): 181,
        ('W', 'E'): 152,
        ('W', 'F'): 40,
        ('W', 'G'): 184,
        ('W', 'H'): 115,
        ('W', 'I'): 61,
        ('W', 'K'): 110,
        ('W', 'L'): 61,
        ('W', 'M'): 67,
        ('W', 'N'): 174,
        ('W', 'P'): 147,
        ('W', 'Q'): 130,
        ('W', 'R'): 101,
        ('W', 'S'): 177,
        ('W', 'T'): 128,
        ('W', 'V'): 88,
        ('W', 'W'): 0,
        ('W', 'Y'): 37,
        ('Y', 'A'): 112,
        ('Y', 'C'): 194,
        ('Y', 'D'): 160,
        ('Y', 'E'): 122,
        ('Y', 'F'): 22,
        ('Y', 'G'): 147,
        ('Y', 'H'): 83,
        ('Y', 'I'): 33,
        ('Y', 'K'): 85,
        ('Y', 'L'): 36,
        ('Y', 'M'): 36,
        ('Y', 'N'): 143,
        ('Y', 'P'): 110,
        ('Y', 'Q'): 99,
        ('Y', 'R'): 77,
        ('Y', 'S'): 144,
        ('Y', 'T'): 92,
        ('Y', 'V'): 55,
        ('Y', 'W'): 37,
        ('Y', 'Y'): 0,
    }

    return grantham_dict


def make_symmetric_koshi_golstein_dicts() -> tuple[
    dict[tuple[str, str], float],
    dict[tuple[str, str], float],
    dict[tuple[str, str], float],
]:
    """
    :return: All symmetrised dictionaries representative of FIGURE 2 from 'Context-dependent optimal substituion
             matrices' by Koshi and Golstein. Keys are pairs of strings. Cf. scientific paper for more details on how
             those scores are derived.
    """
    symmetric_koshi_goldstein_all_residues_dict: dict[tuple[str, str], float] = {
        ("-", "-"): 69.2,
        ("-", "A"): 5.25,
        ("-", "R"): 6.65,
        ("-", "N"): 3.75,
        ("-", "D"): 4.3,
        ("-", "C"): 8.1,
        ("-", "Q"): 6.7,
        ("-", "E"): 4.65,
        ("-", "G"): 4.95,
        ("-", "H"): 5.6,
        ("-", "I"): 4.05,
        ("-", "L"): 4.2,
        ("-", "K"): 2.95,
        ("-", "M"): 5.55,
        ("-", "F"): 3.75,
        ("-", "P"): 7.05,
        ("-", "S"): 6.9,
        ("-", "T"): 5.15,
        ("-", "W"): 3.5,
        ("-", "Y"): 4.35,
        ("-", "V"): 5.75,
        ("A", "-"): 5.25,
        ("A", "A"): 61.9,
        ("A", "R"): 1.05,
        ("A", "N"): 1.2,
        ("A", "D"): 1.65,
        ("A", "C"): 4.1,
        ("A", "Q"): 1.7,
        ("A", "E"): 2.4,
        ("A", "G"): 3.1,
        ("A", "H"): 0.6,
        ("A", "I"): 0.7,
        ("A", "L"): 1.25,
        ("A", "K"): 2.1,
        ("A", "M"): 1.25,
        ("A", "F"): 0.25,
        ("A", "P"): 2.7,
        ("A", "S"): 5.5,
        ("A", "T"): 3.15,
        ("A", "W"): 0.15,
        ("A", "Y"): 0.5,
        ("A", "V"): 3.15,
        ("R", "-"): 6.65,
        ("R", "A"): 1.05,
        ("R", "R"): 67.2,
        ("R", "N"): 1.45,
        ("R", "D"): 0.6,
        ("R", "C"): 0.1,
        ("R", "Q"): 2.85,
        ("R", "E"): 0.85,
        ("R", "G"): 0.5,
        ("R", "H"): 2.0,
        ("R", "I"): 0.45,
        ("R", "L"): 0.85,
        ("R", "K"): 6.9,
        ("R", "M"): 0.55,
        ("R", "F"): 0.2,
        ("R", "P"): 0.8,
        ("R", "S"): 1.65,
        ("R", "T"): 1.15,
        ("R", "W"): 0.35,
        ("R", "Y"): 0.5,
        ("R", "V"): 0.45,
        ("N", "-"): 3.75,
        ("N", "A"): 1.2,
        ("N", "R"): 1.45,
        ("N", "N"): 60.9,
        ("N", "D"): 5.75,
        ("N", "C"): 0.3,
        ("N", "Q"): 1.7,
        ("N", "E"): 1.7,
        ("N", "G"): 1.95,
        ("N", "H"): 2.2,
        ("N", "I"): 0.45,
        ("N", "L"): 0.5,
        ("N", "K"): 2.9,
        ("N", "M"): 0.45,
        ("N", "F"): 0.25,
        ("N", "P"): 0.5,
        ("N", "S"): 4.3,
        ("N", "T"): 2.1,
        ("N", "W"): 0.25,
        ("N", "Y"): 0.8,
        ("N", "V"): 0.7,
        ("D", "-"): 4.3,
        ("D", "A"): 1.65,
        ("D", "R"): 0.6,
        ("D", "N"): 5.75,
        ("D", "D"): 65.0,
        ("D", "C"): 1.0,
        ("D", "Q"): 1.45,
        ("D", "E"): 6.35,
        ("D", "G"): 1.75,
        ("D", "H"): 0.7,
        ("D", "I"): 0.25,
        ("D", "L"): 0.25,
        ("D", "K"): 1.65,
        ("D", "M"): 0.8,
        ("D", "F"): 0.2,
        ("D", "P"): 0.85,
        ("D", "S"): 2.3,
        ("D", "T"): 1.05,
        ("D", "W"): 0.0,
        ("D", "Y"): 0.45,
        ("D", "V"): 0.2,
        ("C", "-"): 8.1,
        ("C", "A"): 4.1,
        ("C", "R"): 0.1,
        ("C", "N"): 0.3,
        ("C", "D"): 1.0,
        ("C", "C"): 64.4,
        ("C", "Q"): 0.1,
        ("C", "E"): 0.75,
        ("C", "G"): 0.85,
        ("C", "H"): 0.4,
        ("C", "I"): 1.1,
        ("C", "L"): 0.75,
        ("C", "K"): 0.05,
        ("C", "M"): 0.45,
        ("C", "F"): 1.0,
        ("C", "P"): 0.0,
        ("C", "S"): 1.75,
        ("C", "T"): 0.7,
        ("C", "W"): 0.45,
        ("C", "Y"): 0.45,
        ("C", "V"): 1.0,
        ("Q", "-"): 6.7,
        ("Q", "A"): 1.7,
        ("Q", "R"): 2.85,
        ("Q", "N"): 1.7,
        ("Q", "D"): 1.45,
        ("Q", "C"): 0.1,
        ("Q", "Q"): 58.4,
        ("Q", "E"): 4.6,
        ("Q", "G"): 0.75,
        ("Q", "H"): 2.1,
        ("Q", "I"): 0.3,
        ("Q", "L"): 1.5,
        ("Q", "K"): 3.8,
        ("Q", "M"): 0.9,
        ("Q", "F"): 0.35,
        ("Q", "P"): 0.9,
        ("Q", "S"): 1.5,
        ("Q", "T"): 1.5,
        ("Q", "W"): 0.3,
        ("Q", "Y"): 0.35,
        ("Q", "V"): 0.55,
        ("E", "-"): 4.65,
        ("E", "A"): 2.4,
        ("E", "R"): 0.85,
        ("E", "N"): 1.7,
        ("E", "D"): 6.35,
        ("E", "C"): 0.75,
        ("E", "Q"): 4.6,
        ("E", "E"): 62.1,
        ("E", "G"): 1.15,
        ("E", "H"): 1.15,
        ("E", "I"): 0.4,
        ("E", "L"): 0.7,
        ("E", "K"): 4.15,
        ("E", "M"): 0.55,
        ("E", "F"): 0.15,
        ("E", "P"): 1.1,
        ("E", "S"): 1.85,
        ("E", "T"): 1.25,
        ("E", "W"): 0.1,
        ("E", "Y"): 0.3,
        ("E", "V"): 0.8,
        ("G", "-"): 4.95,
        ("G", "A"): 3.1,
        ("G", "R"): 0.5,
        ("G", "N"): 1.95,
        ("G", "D"): 1.75,
        ("G", "C"): 0.85,
        ("G", "Q"): 0.75,
        ("G", "E"): 1.15,
        ("G", "G"): 77.8,
        ("G", "H"): 0.55,
        ("G", "I"): 0.1,
        ("G", "L"): 0.15,
        ("G", "K"): 1.05,
        ("G", "M"): 0.35,
        ("G", "F"): 0.25,
        ("G", "P"): 0.5,
        ("G", "S"): 2.6,
        ("G", "T"): 1.0,
        ("G", "W"): 0.3,
        ("G", "Y"): 0.25,
        ("G", "V"): 0.35,
        ("H", "-"): 5.6,
        ("H", "A"): 0.6,
        ("H", "R"): 2.0,
        ("H", "N"): 2.2,
        ("H", "D"): 0.7,
        ("H", "C"): 0.4,
        ("H", "Q"): 2.1,
        ("H", "E"): 1.15,
        ("H", "G"): 0.55,
        ("H", "H"): 65.4,
        ("H", "I"): 0.3,
        ("H", "L"): 0.95,
        ("H", "K"): 1.35,
        ("H", "M"): 0.5,
        ("H", "F"): 0.95,
        ("H", "P"): 0.7,
        ("H", "S"): 1.25,
        ("H", "T"): 0.55,
        ("H", "W"): 0.25,
        ("H", "Y"): 2.3,
        ("H", "V"): 0.2,
        ("I", "-"): 4.05,
        ("I", "A"): 0.7,
        ("I", "R"): 0.45,
        ("I", "N"): 0.45,
        ("I", "D"): 0.25,
        ("I", "C"): 1.1,
        ("I", "Q"): 0.3,
        ("I", "E"): 0.4,
        ("I", "G"): 0.1,
        ("I", "H"): 0.3,
        ("I", "I"): 64.8,
        ("I", "L"): 6.2,
        ("I", "K"): 0.4,
        ("I", "M"): 3.75,
        ("I", "F"): 1.55,
        ("I", "P"): 0.4,
        ("I", "S"): 0.35,
        ("I", "T"): 1.8,
        ("I", "W"): 0.35,
        ("I", "Y"): 0.6,
        ("I", "V"): 9.85,
        ("L", "-"): 4.2,
        ("L", "A"): 1.25,
        ("L", "R"): 0.85,
        ("L", "N"): 0.5,
        ("L", "D"): 0.25,
        ("L", "C"): 0.75,
        ("L", "Q"): 1.5,
        ("L", "E"): 0.7,
        ("L", "G"): 0.15,
        ("L", "H"): 0.95,
        ("L", "I"): 6.2,
        ("L", "L"): 71.7,
        ("L", "K"): 1.1,
        ("L", "M"): 6.2,
        ("L", "F"): 3.25,
        ("L", "P"): 0.6,
        ("L", "S"): 0.35,
        ("L", "T"): 1.0,
        ("L", "W"): 0.6,
        ("L", "Y"): 1.4,
        ("L", "V"): 3.65,
        ("K", "-"): 2.95,
        ("K", "A"): 2.1,
        ("K", "R"): 6.9,
        ("K", "N"): 2.9,
        ("K", "D"): 1.65,
        ("K", "C"): 0.05,
        ("K", "Q"): 3.8,
        ("K", "E"): 4.15,
        ("K", "G"): 1.05,
        ("K", "H"): 1.35,
        ("K", "I"): 0.4,
        ("K", "L"): 1.1,
        ("K", "K"): 58.7,
        ("K", "M"): 0.65,
        ("K", "F"): 0.3,
        ("K", "P"): 1.05,
        ("K", "S"): 1.95,
        ("K", "T"): 2.3,
        ("K", "W"): 0.35,
        ("K", "Y"): 0.45,
        ("K", "V"): 0.9,
        ("M", "-"): 5.55,
        ("M", "A"): 1.25,
        ("M", "R"): 0.55,
        ("M", "N"): 0.45,
        ("M", "D"): 0.8,
        ("M", "C"): 0.45,
        ("M", "Q"): 0.9,
        ("M", "E"): 0.55,
        ("M", "G"): 0.35,
        ("M", "H"): 0.5,
        ("M", "I"): 3.75,
        ("M", "L"): 6.2,
        ("M", "K"): 0.65,
        ("M", "M"): 59.4,
        ("M", "F"): 1.55,
        ("M", "P"): 0.35,
        ("M", "S"): 0.45,
        ("M", "T"): 1.25,
        ("M", "W"): 0.5,
        ("M", "Y"): 0.3,
        ("M", "V"): 2.1,
        ("F", "-"): 3.75,
        ("F", "A"): 0.25,
        ("F", "R"): 0.2,
        ("F", "N"): 0.25,
        ("F", "D"): 0.2,
        ("F", "C"): 1.0,
        ("F", "Q"): 0.35,
        ("F", "E"): 0.15,
        ("F", "G"): 0.25,
        ("F", "H"): 0.95,
        ("F", "I"): 1.55,
        ("F", "L"): 3.25,
        ("F", "K"): 0.3,
        ("F", "M"): 1.55,
        ("F", "F"): 73.8,
        ("F", "P"): 0.2,
        ("F", "S"): 0.7,
        ("F", "T"): 0.35,
        ("F", "W"): 1.85,
        ("F", "Y"): 6.75,
        ("F", "V"): 1.35,
        ("P", "-"): 7.05,
        ("P", "A"): 2.7,
        ("P", "R"): 0.8,
        ("P", "N"): 0.5,
        ("P", "D"): 0.85,
        ("P", "C"): 0.0,
        ("P", "Q"): 0.9,
        ("P", "E"): 1.1,
        ("P", "G"): 0.5,
        ("P", "H"): 0.7,
        ("P", "I"): 0.4,
        ("P", "L"): 0.6,
        ("P", "K"): 1.05,
        ("P", "M"): 0.35,
        ("P", "F"): 0.2,
        ("P", "P"): 71.2,
        ("P", "S"): 2.3,
        ("P", "T"): 1.25,
        ("P", "W"): 0.0,
        ("P", "Y"): 0.3,
        ("P", "V"): 0.55,
        ("S", "-"): 6.9,
        ("S", "A"): 5.5,
        ("S", "R"): 1.65,
        ("S", "N"): 4.3,
        ("S", "D"): 2.3,
        ("S", "C"): 1.75,
        ("S", "Q"): 1.5,
        ("S", "E"): 1.85,
        ("S", "G"): 2.6,
        ("S", "H"): 1.25,
        ("S", "I"): 0.35,
        ("S", "L"): 0.35,
        ("S", "K"): 1.95,
        ("S", "M"): 0.45,
        ("S", "F"): 0.7,
        ("S", "P"): 2.3,
        ("S", "S"): 56.8,
        ("S", "T"): 6.5,
        ("S", "W"): 0.8,
        ("S", "Y"): 0.55,
        ("S", "V"): 0.85,
        ("T", "-"): 5.15,
        ("T", "A"): 3.15,
        ("T", "R"): 1.15,
        ("T", "N"): 2.1,
        ("T", "D"): 1.05,
        ("T", "C"): 0.7,
        ("T", "Q"): 1.5,
        ("T", "E"): 1.25,
        ("T", "G"): 1.0,
        ("T", "H"): 0.55,
        ("T", "I"): 1.8,
        ("T", "L"): 1.0,
        ("T", "K"): 2.3,
        ("T", "M"): 1.25,
        ("T", "F"): 0.35,
        ("T", "P"): 1.25,
        ("T", "S"): 6.5,
        ("T", "T"): 61.8,
        ("T", "W"): 0.15,
        ("T", "Y"): 0.65,
        ("T", "V"): 2.85,
        ("W", "-"): 3.5,
        ("W", "A"): 0.15,
        ("W", "R"): 0.35,
        ("W", "N"): 0.25,
        ("W", "D"): 0.0,
        ("W", "C"): 0.45,
        ("W", "Q"): 0.3,
        ("W", "E"): 0.1,
        ("W", "G"): 0.3,
        ("W", "H"): 0.25,
        ("W", "I"): 0.35,
        ("W", "L"): 0.6,
        ("W", "K"): 0.35,
        ("W", "M"): 0.5,
        ("W", "F"): 1.85,
        ("W", "P"): 0.0,
        ("W", "S"): 0.8,
        ("W", "T"): 0.15,
        ("W", "W"): 82.3,
        ("W", "Y"): 1.6,
        ("W", "V"): 0.45,
        ("Y", "-"): 4.35,
        ("Y", "A"): 0.5,
        ("Y", "R"): 0.5,
        ("Y", "N"): 0.8,
        ("Y", "D"): 0.45,
        ("Y", "C"): 0.45,
        ("Y", "Q"): 0.35,
        ("Y", "E"): 0.3,
        ("Y", "G"): 0.25,
        ("Y", "H"): 2.3,
        ("Y", "I"): 0.6,
        ("Y", "L"): 1.4,
        ("Y", "K"): 0.45,
        ("Y", "M"): 0.3,
        ("Y", "F"): 6.75,
        ("Y", "P"): 0.3,
        ("Y", "S"): 0.55,
        ("Y", "T"): 0.65,
        ("Y", "W"): 1.6,
        ("Y", "Y"): 73.2,
        ("Y", "V"): 0.6,
        ("V", "-"): 5.75,
        ("V", "A"): 3.15,
        ("V", "R"): 0.45,
        ("V", "N"): 0.7,
        ("V", "D"): 0.2,
        ("V", "C"): 1.0,
        ("V", "Q"): 0.55,
        ("V", "E"): 0.8,
        ("V", "G"): 0.35,
        ("V", "H"): 0.2,
        ("V", "I"): 9.85,
        ("V", "L"): 3.65,
        ("V", "K"): 0.9,
        ("V", "M"): 2.1,
        ("V", "F"): 1.35,
        ("V", "P"): 0.55,
        ("V", "S"): 0.85,
        ("V", "T"): 2.85,
        ("V", "W"): 0.45,
        ("V", "Y"): 0.6,
        ("V", "V"): 64.0,
    }
    symmetric_koshi_goldstein_exposed_residues_dict: dict[tuple[str, str], float] = {
        ("-", "-"): 60.2,
        ("-", "A"): 12.4,
        ("-", "R"): 6.55,
        ("-", "N"): 4.3,
        ("-", "D"): 4.25,
        ("-", "C"): 10.45,
        ("-", "Q"): 8.0,
        ("-", "E"): 4.5,
        ("-", "G"): 6.8,
        ("-", "H"): 8.15,
        ("-", "I"): 9.9,
        ("-", "L"): 8.65,
        ("-", "K"): 2.75,
        ("-", "M"): 8.3,
        ("-", "F"): 8.2,
        ("-", "P"): 7.95,
        ("-", "S"): 6.5,
        ("-", "T"): 7.9,
        ("-", "W"): 7.9,
        ("-", "Y"): 6.7,
        ("-", "V"): 10.1,
        ("A", "-"): 12.4,
        ("A", "A"): 51.3,
        ("A", "R"): 0.95,
        ("A", "N"): 1.2,
        ("A", "D"): 1.95,
        ("A", "C"): 2.6,
        ("A", "Q"): 2.15,
        ("A", "E"): 2.75,
        ("A", "G"): 2.8,
        ("A", "H"): 0.65,
        ("A", "I"): 0.75,
        ("A", "L"): 1.4,
        ("A", "K"): 2.5,
        ("A", "M"): 1.75,
        ("A", "F"): 0.4,
        ("A", "P"): 3.5,
        ("A", "S"): 5.8,
        ("A", "T"): 3.4,
        ("A", "W"): 0.2,
        ("A", "Y"): 0.4,
        ("A", "V"): 2.85,
        ("R", "-"): 6.55,
        ("R", "A"): 0.95,
        ("R", "R"): 62.9,
        ("R", "N"): 1.55,
        ("R", "D"): 0.7,
        ("R", "C"): 0.05,
        ("R", "Q"): 3.3,
        ("R", "E"): 0.9,
        ("R", "G"): 0.55,
        ("R", "H"): 2.45,
        ("R", "I"): 0.45,
        ("R", "L"): 1.15,
        ("R", "K"): 7.6,
        ("R", "M"): 0.6,
        ("R", "F"): 0.45,
        ("R", "P"): 0.95,
        ("R", "S"): 1.75,
        ("R", "T"): 1.25,
        ("R", "W"): 0.35,
        ("R", "Y"): 0.4,
        ("R", "V"): 0.6,
        ("N", "-"): 4.3,
        ("N", "A"): 1.2,
        ("N", "R"): 1.55,
        ("N", "N"): 52.6,
        ("N", "D"): 7.2,
        ("N", "C"): 0.35,
        ("N", "Q"): 2.05,
        ("N", "E"): 2.0,
        ("N", "G"): 2.5,
        ("N", "H"): 3.35,
        ("N", "I"): 0.3,
        ("N", "L"): 0.7,
        ("N", "K"): 3.55,
        ("N", "M"): 0.5,
        ("N", "F"): 0.45,
        ("N", "P"): 0.35,
        ("N", "S"): 5.3,
        ("N", "T"): 2.6,
        ("N", "W"): 0.2,
        ("N", "Y"): 0.8,
        ("N", "V"): 0.7,
        ("D", "-"): 4.25,
        ("D", "A"): 1.95,
        ("D", "R"): 0.7,
        ("D", "N"): 7.2,
        ("D", "D"): 59.2,
        ("D", "C"): 0.8,
        ("D", "Q"): 1.7,
        ("D", "E"): 7.5,
        ("D", "G"): 2.3,
        ("D", "H"): 0.65,
        ("D", "I"): 0.25,
        ("D", "L"): 0.25,
        ("D", "K"): 1.95,
        ("D", "M"): 0.1,
        ("D", "F"): 0.15,
        ("D", "P"): 1.1,
        ("D", "S"): 2.8,
        ("D", "T"): 1.2,
        ("D", "W"): 0.0,
        ("D", "Y"): 0.3,
        ("D", "V"): 0.3,
        ("C", "-"): 10.45,
        ("C", "A"): 2.6,
        ("C", "R"): 0.05,
        ("C", "N"): 0.35,
        ("C", "D"): 0.8,
        ("C", "C"): 64.7,
        ("C", "Q"): 0.05,
        ("C", "E"): 0.55,
        ("C", "G"): 0.7,
        ("C", "H"): 0.3,
        ("C", "I"): 0.7,
        ("C", "L"): 0.6,
        ("C", "K"): 0.1,
        ("C", "M"): 0.95,
        ("C", "F"): 1.35,
        ("C", "P"): 0.0,
        ("C", "S"): 1.3,
        ("C", "T"): 0.55,
        ("C", "W"): 0.0,
        ("C", "Y"): 0.3,
        ("C", "V"): 0.45,
        ("Q", "-"): 8.0,
        ("Q", "A"): 2.15,
        ("Q", "R"): 3.3,
        ("Q", "N"): 2.05,
        ("Q", "D"): 1.7,
        ("Q", "C"): 0.05,
        ("Q", "Q"): 51.0,
        ("Q", "E"): 5.2,
        ("Q", "G"): 0.8,
        ("Q", "H"): 1.95,
        ("Q", "I"): 0.35,
        ("Q", "L"): 1.9,
        ("Q", "K"): 4.25,
        ("Q", "M"): 0.95,
        ("Q", "F"): 0.25,
        ("Q", "P"): 0.9,
        ("Q", "S"): 1.7,
        ("Q", "T"): 1.95,
        ("Q", "W"): 0.05,
        ("Q", "Y"): 0.35,
        ("Q", "V"): 1.1,
        ("E", "-"): 4.5,
        ("E", "A"): 2.75,
        ("E", "R"): 0.9,
        ("E", "N"): 2.0,
        ("E", "D"): 7.5,
        ("E", "C"): 0.55,
        ("E", "Q"): 5.2,
        ("E", "E"): 58.3,
        ("E", "G"): 1.35,
        ("E", "H"): 1.4,
        ("E", "I"): 0.45,
        ("E", "L"): 0.7,
        ("E", "K"): 4.55,
        ("E", "M"): 1.2,
        ("E", "F"): 0.25,
        ("E", "P"): 1.35,
        ("E", "S"): 2.2,
        ("E", "T"): 1.5,
        ("E", "W"): 0.1,
        ("E", "Y"): 0.35,
        ("E", "V"): 0.95,
        ("G", "-"): 6.8,
        ("G", "A"): 2.8,
        ("G", "R"): 0.55,
        ("G", "N"): 2.5,
        ("G", "D"): 2.3,
        ("G", "C"): 0.7,
        ("G", "Q"): 0.8,
        ("G", "E"): 1.35,
        ("G", "G"): 71.0,
        ("G", "H"): 0.55,
        ("G", "I"): 0.0,
        ("G", "L"): 0.2,
        ("G", "K"): 1.2,
        ("G", "M"): 0.7,
        ("G", "F"): 0.3,
        ("G", "P"): 0.6,
        ("G", "S"): 3.25,
        ("G", "T"): 1.15,
        ("G", "W"): 0.1,
        ("G", "Y"): 0.2,
        ("G", "V"): 0.5,
        ("H", "-"): 8.15,
        ("H", "A"): 0.65,
        ("H", "R"): 2.45,
        ("H", "N"): 3.35,
        ("H", "D"): 0.65,
        ("H", "C"): 0.3,
        ("H", "Q"): 1.95,
        ("H", "E"): 1.4,
        ("H", "G"): 0.55,
        ("H", "H"): 54.8,
        ("H", "I"): 0.35,
        ("H", "L"): 0.85,
        ("H", "K"): 1.3,
        ("H", "M"): 0.3,
        ("H", "F"): 1.2,
        ("H", "P"): 0.9,
        ("H", "S"): 1.7,
        ("H", "T"): 0.35,
        ("H", "W"): 0.0,
        ("H", "Y"): 2.95,
        ("H", "V"): 1.5,
        ("I", "-"): 9.9,
        ("I", "A"): 0.75,
        ("I", "R"): 0.45,
        ("I", "N"): 0.3,
        ("I", "D"): 0.25,
        ("I", "C"): 0.7,
        ("I", "Q"): 0.35,
        ("I", "E"): 0.45,
        ("I", "G"): 0.0,
        ("I", "H"): 0.35,
        ("I", "I"): 54.4,
        ("I", "L"): 5.7,
        ("I", "K"): 0.55,
        ("I", "M"): 4.3,
        ("I", "F"): 1.6,
        ("I", "P"): 0.5,
        ("I", "S"): 0.05,
        ("I", "T"): 2.45,
        ("I", "W"): 0.45,
        ("I", "Y"): 1.1,
        ("I", "V"): 10.85,
        ("L", "-"): 8.65,
        ("L", "A"): 1.4,
        ("L", "R"): 1.15,
        ("L", "N"): 0.7,
        ("L", "D"): 0.25,
        ("L", "C"): 0.6,
        ("L", "Q"): 1.9,
        ("L", "E"): 0.7,
        ("L", "G"): 0.2,
        ("L", "H"): 0.85,
        ("L", "I"): 5.7,
        ("L", "L"): 64.4,
        ("L", "K"): 1.1,
        ("L", "M"): 5.6,
        ("L", "F"): 2.8,
        ("L", "P"): 0.75,
        ("L", "S"): 0.4,
        ("L", "T"): 1.45,
        ("L", "W"): 1.25,
        ("L", "Y"): 1.85,
        ("L", "V"): 3.15,
        ("K", "-"): 2.75,
        ("K", "A"): 2.5,
        ("K", "R"): 7.6,
        ("K", "N"): 3.55,
        ("K", "D"): 1.95,
        ("K", "C"): 0.1,
        ("K", "Q"): 4.25,
        ("K", "E"): 4.55,
        ("K", "G"): 1.2,
        ("K", "H"): 1.3,
        ("K", "I"): 0.55,
        ("K", "L"): 1.1,
        ("K", "K"): 56.0,
        ("K", "M"): 1.1,
        ("K", "F"): 0.55,
        ("K", "P"): 1.1,
        ("K", "S"): 2.4,
        ("K", "T"): 2.6,
        ("K", "W"): 0.3,
        ("K", "Y"): 0.45,
        ("K", "V"): 1.05,
        ("M", "-"): 8.3,
        ("M", "A"): 1.75,
        ("M", "R"): 0.6,
        ("M", "N"): 0.5,
        ("M", "D"): 0.1,
        ("M", "C"): 0.95,
        ("M", "Q"): 0.95,
        ("M", "E"): 1.2,
        ("M", "G"): 0.7,
        ("M", "H"): 0.3,
        ("M", "I"): 4.3,
        ("M", "L"): 5.6,
        ("M", "K"): 1.1,
        ("M", "M"): 49.9,
        ("M", "F"): 1.35,
        ("M", "P"): 0.2,
        ("M", "S"): 0.85,
        ("M", "T"): 1.7,
        ("M", "W"): 0.5,
        ("M", "Y"): 0.6,
        ("M", "V"): 2.2,
        ("F", "-"): 8.2,
        ("F", "A"): 0.4,
        ("F", "R"): 0.45,
        ("F", "N"): 0.45,
        ("F", "D"): 0.15,
        ("F", "C"): 1.35,
        ("F", "Q"): 0.25,
        ("F", "E"): 0.25,
        ("F", "G"): 0.3,
        ("F", "H"): 1.2,
        ("F", "I"): 1.6,
        ("F", "L"): 2.8,
        ("F", "K"): 0.55,
        ("F", "M"): 1.35,
        ("F", "F"): 66.6,
        ("F", "P"): 0.25,
        ("F", "S"): 0.55,
        ("F", "T"): 0.35,
        ("F", "W"): 1.8,
        ("F", "Y"): 6.85,
        ("F", "V"): 1.4,
        ("P", "-"): 7.95,
        ("P", "A"): 3.5,
        ("P", "R"): 0.95,
        ("P", "N"): 0.35,
        ("P", "D"): 1.1,
        ("P", "C"): 0.0,
        ("P", "Q"): 0.9,
        ("P", "E"): 1.35,
        ("P", "G"): 0.6,
        ("P", "H"): 0.9,
        ("P", "I"): 0.5,
        ("P", "L"): 0.75,
        ("P", "K"): 1.1,
        ("P", "M"): 0.2,
        ("P", "F"): 0.25,
        ("P", "P"): 66.0,
        ("P", "S"): 2.9,
        ("P", "T"): 1.3,
        ("P", "W"): 0.25,
        ("P", "Y"): 0.25,
        ("P", "V"): 0.6,
        ("S", "-"): 6.5,
        ("S", "A"): 5.8,
        ("S", "R"): 1.75,
        ("S", "N"): 5.3,
        ("S", "D"): 2.8,
        ("S", "C"): 1.3,
        ("S", "Q"): 1.7,
        ("S", "E"): 2.2,
        ("S", "G"): 3.25,
        ("S", "H"): 1.7,
        ("S", "I"): 0.05,
        ("S", "L"): 0.4,
        ("S", "K"): 2.4,
        ("S", "M"): 0.85,
        ("S", "F"): 0.55,
        ("S", "P"): 2.9,
        ("S", "S"): 50.1,
        ("S", "T"): 7.75,
        ("S", "W"): 0.0,
        ("S", "Y"): 0.6,
        ("S", "V"): 1.0,
        ("T", "-"): 7.9,
        ("T", "A"): 3.4,
        ("T", "R"): 1.25,
        ("T", "N"): 2.6,
        ("T", "D"): 1.2,
        ("T", "C"): 0.55,
        ("T", "Q"): 1.95,
        ("T", "E"): 1.5,
        ("T", "G"): 1.15,
        ("T", "H"): 0.35,
        ("T", "I"): 2.45,
        ("T", "L"): 1.45,
        ("T", "K"): 2.6,
        ("T", "M"): 1.7,
        ("T", "F"): 0.35,
        ("T", "P"): 1.3,
        ("T", "S"): 7.75,
        ("T", "T"): 53.0,
        ("T", "W"): 0.2,
        ("T", "Y"): 0.55,
        ("T", "V"): 2.6,
        ("W", "-"): 7.9,
        ("W", "A"): 0.2,
        ("W", "R"): 0.35,
        ("W", "N"): 0.2,
        ("W", "D"): 0.0,
        ("W", "C"): 0.0,
        ("W", "Q"): 0.05,
        ("W", "E"): 0.1,
        ("W", "G"): 0.1,
        ("W", "H"): 0.0,
        ("W", "I"): 0.45,
        ("W", "L"): 1.25,
        ("W", "K"): 0.3,
        ("W", "M"): 0.5,
        ("W", "F"): 1.8,
        ("W", "P"): 0.25,
        ("W", "S"): 0.0,
        ("W", "T"): 0.2,
        ("W", "W"): 75.4,
        ("W", "Y"): 2.6,
        ("W", "V"): 0.2,
        ("Y", "-"): 6.7,
        ("Y", "A"): 0.4,
        ("Y", "R"): 0.4,
        ("Y", "N"): 0.8,
        ("Y", "D"): 0.3,
        ("Y", "C"): 0.3,
        ("Y", "Q"): 0.35,
        ("Y", "E"): 0.35,
        ("Y", "G"): 0.2,
        ("Y", "H"): 2.95,
        ("Y", "I"): 1.1,
        ("Y", "L"): 1.85,
        ("Y", "K"): 0.45,
        ("Y", "M"): 0.6,
        ("Y", "F"): 6.85,
        ("Y", "P"): 0.25,
        ("Y", "S"): 0.6,
        ("Y", "T"): 0.55,
        ("Y", "W"): 2.6,
        ("Y", "Y"): 66.8,
        ("Y", "V"): 0.95,
        ("V", "-"): 10.1,
        ("V", "A"): 2.85,
        ("V", "R"): 0.6,
        ("V", "N"): 0.7,
        ("V", "D"): 0.3,
        ("V", "C"): 0.45,
        ("V", "Q"): 1.1,
        ("V", "E"): 0.95,
        ("V", "G"): 0.5,
        ("V", "H"): 1.5,
        ("V", "I"): 10.85,
        ("V", "L"): 3.15,
        ("V", "K"): 1.05,
        ("V", "M"): 2.2,
        ("V", "F"): 1.4,
        ("V", "P"): 0.6,
        ("V", "S"): 1.0,
        ("V", "T"): 2.6,
        ("V", "W"): 0.2,
        ("V", "Y"): 0.95,
        ("V", "V"): 54.8,
    }
    symmetric_koshi_goldstein_buried_residues_dict: dict[tuple[str, str], float] = {
        ("-", "-"): 79.2,
        ("-", "A"): 0.6,
        ("-", "R"): 4.9,
        ("-", "N"): 3.7,
        ("-", "D"): 5.8,
        ("-", "C"): 3.45,
        ("-", "Q"): 3.65,
        ("-", "E"): 5.95,
        ("-", "G"): 2.15,
        ("-", "H"): 0.6,
        ("-", "I"): 1.1,
        ("-", "L"): 0.65,
        ("-", "K"): 12.15,
        ("-", "M"): 1.3,
        ("-", "F"): 1.15,
        ("-", "P"): 4.7,
        ("-", "S"): 5.3,
        ("-", "T"): 2.1,
        ("-", "W"): 1.15,
        ("-", "Y"): 1.2,
        ("-", "V"): 0.85,
        ("A", "-"): 0.6,
        ("A", "A"): 76.3,
        ("A", "R"): 0.65,
        ("A", "N"): 1.25,
        ("A", "D"): 1.55,
        ("A", "C"): 4.3,
        ("A", "Q"): 1.5,
        ("A", "E"): 2.5,
        ("A", "G"): 3.2,
        ("A", "H"): 0.4,
        ("A", "I"): 0.65,
        ("A", "L"): 1.05,
        ("A", "K"): 1.1,
        ("A", "M"): 0.85,
        ("A", "F"): 0.25,
        ("A", "P"): 2.85,
        ("A", "S"): 5.75,
        ("A", "T"): 2.75,
        ("A", "W"): 0.0,
        ("A", "Y"): 0.4,
        ("A", "V"): 3.1,
        ("R", "-"): 4.9,
        ("R", "A"): 0.65,
        ("R", "R"): 72.9,
        ("R", "N"): 1.0,
        ("R", "D"): 0.35,
        ("R", "C"): 0.25,
        ("R", "Q"): 2.05,
        ("R", "E"): 1.1,
        ("R", "G"): 0.6,
        ("R", "H"): 1.25,
        ("R", "I"): 0.7,
        ("R", "L"): 0.85,
        ("R", "K"): 4.55,
        ("R", "M"): 0.2,
        ("R", "F"): 0.2,
        ("R", "P"): 0.5,
        ("R", "S"): 1.4,
        ("R", "T"): 1.2,
        ("R", "W"): 0.35,
        ("R", "Y"): 0.4,
        ("R", "V"): 0.15,
        ("N", "-"): 3.7,
        ("N", "A"): 1.25,
        ("N", "R"): 1.0,
        ("N", "N"): 65.7,
        ("N", "D"): 3.05,
        ("N", "C"): 0.35,
        ("N", "Q"): 1.3,
        ("N", "E"): 0.75,
        ("N", "G"): 1.55,
        ("N", "H"): 1.25,
        ("N", "I"): 0.75,
        ("N", "L"): 0.95,
        ("N", "K"): 1.6,
        ("N", "M"): 0.35,
        ("N", "F"): 0.1,
        ("N", "P"): 0.75,
        ("N", "S"): 3.15,
        ("N", "T"): 1.7,
        ("N", "W"): 0.05,
        ("N", "Y"): 0.8,
        ("N", "V"): 0.8,
        ("D", "-"): 5.8,
        ("D", "A"): 1.55,
        ("D", "R"): 0.35,
        ("D", "N"): 3.05,
        ("D", "D"): 66.9,
        ("D", "C"): 0.95,
        ("D", "Q"): 0.75,
        ("D", "E"): 3.35,
        ("D", "G"): 1.25,
        ("D", "H"): 0.65,
        ("D", "I"): 0.5,
        ("D", "L"): 0.5,
        ("D", "K"): 0.7,
        ("D", "M"): 0.1,
        ("D", "F"): 0.15,
        ("D", "P"): 0.6,
        ("D", "S"): 1.75,
        ("D", "T"): 0.95,
        ("D", "W"): 0.1,
        ("D", "Y"): 0.6,
        ("D", "V"): 0.35,
        ("C", "-"): 3.45,
        ("C", "A"): 4.3,
        ("C", "R"): 0.25,
        ("C", "N"): 0.35,
        ("C", "D"): 0.95,
        ("C", "C"): 71.6,
        ("C", "Q"): 0.25,
        ("C", "E"): 0.5,
        ("C", "G"): 0.85,
        ("C", "H"): 0.35,
        ("C", "I"): 1.1,
        ("C", "L"): 1.25,
        ("C", "K"): 0.0,
        ("C", "M"): 0.6,
        ("C", "F"): 0.8,
        ("C", "P"): 0.05,
        ("C", "S"): 1.8,
        ("C", "T"): 0.8,
        ("C", "W"): 0.25,
        ("C", "Y"): 0.35,
        ("C", "V"): 2.0,
        ("Q", "-"): 3.65,
        ("Q", "A"): 1.5,
        ("Q", "R"): 2.05,
        ("Q", "N"): 1.3,
        ("Q", "D"): 0.75,
        ("Q", "C"): 0.25,
        ("Q", "Q"): 67.3,
        ("Q", "E"): 2.9,
        ("Q", "G"): 0.6,
        ("Q", "H"): 1.65,
        ("Q", "I"): 0.35,
        ("Q", "L"): 2.25,
        ("Q", "K"): 2.1,
        ("Q", "M"): 1.45,
        ("Q", "F"): 0.6,
        ("Q", "P"): 1.05,
        ("Q", "S"): 1.1,
        ("Q", "T"): 0.95,
        ("Q", "W"): 0.5,
        ("Q", "Y"): 0.3,
        ("Q", "V"): 0.3,
        ("E", "-"): 5.95,
        ("E", "A"): 2.5,
        ("E", "R"): 1.1,
        ("E", "N"): 0.75,
        ("E", "D"): 3.35,
        ("E", "C"): 0.5,
        ("E", "Q"): 2.9,
        ("E", "E"): 62.5,
        ("E", "G"): 1.1,
        ("E", "H"): 0.5,
        ("E", "I"): 0.15,
        ("E", "L"): 1.55,
        ("E", "K"): 1.7,
        ("E", "M"): 0.3,
        ("E", "F"): 0.0,
        ("E", "P"): 0.65,
        ("E", "S"): 1.55,
        ("E", "T"): 0.95,
        ("E", "W"): 0.0,
        ("E", "Y"): 0.8,
        ("E", "V"): 1.55,
        ("G", "-"): 2.15,
        ("G", "A"): 3.2,
        ("G", "R"): 0.6,
        ("G", "N"): 1.55,
        ("G", "D"): 1.25,
        ("G", "C"): 0.85,
        ("G", "Q"): 0.6,
        ("G", "E"): 1.1,
        ("G", "G"): 85.7,
        ("G", "H"): 0.1,
        ("G", "I"): 0.05,
        ("G", "L"): 0.25,
        ("G", "K"): 1.1,
        ("G", "M"): 0.25,
        ("G", "F"): 0.25,
        ("G", "P"): 0.55,
        ("G", "S"): 1.95,
        ("G", "T"): 0.65,
        ("G", "W"): 0.2,
        ("G", "Y"): 0.05,
        ("G", "V"): 0.25,
        ("H", "-"): 0.6,
        ("H", "A"): 0.4,
        ("H", "R"): 1.25,
        ("H", "N"): 1.25,
        ("H", "D"): 0.65,
        ("H", "C"): 0.35,
        ("H", "Q"): 1.65,
        ("H", "E"): 0.5,
        ("H", "G"): 0.1,
        ("H", "H"): 84.1,
        ("H", "I"): 0.35,
        ("H", "L"): 0.6,
        ("H", "K"): 1.05,
        ("H", "M"): 0.5,
        ("H", "F"): 1.25,
        ("H", "P"): 0.55,
        ("H", "S"): 0.65,
        ("H", "T"): 0.4,
        ("H", "W"): 0.35,
        ("H", "Y"): 2.05,
        ("H", "V"): 0.15,
        ("I", "-"): 1.1,
        ("I", "A"): 0.65,
        ("I", "R"): 0.7,
        ("I", "N"): 0.75,
        ("I", "D"): 0.5,
        ("I", "C"): 1.1,
        ("I", "Q"): 0.35,
        ("I", "E"): 0.15,
        ("I", "G"): 0.05,
        ("I", "H"): 0.35,
        ("I", "I"): 73.9,
        ("I", "L"): 6.35,
        ("I", "K"): 1.0,
        ("I", "M"): 3.65,
        ("I", "F"): 1.5,
        ("I", "P"): 0.6,
        ("I", "S"): 0.4,
        ("I", "T"): 1.95,
        ("I", "W"): 0.45,
        ("I", "Y"): 0.55,
        ("I", "V"): 9.95,
        ("L", "-"): 0.65,
        ("L", "A"): 1.05,
        ("L", "R"): 0.85,
        ("L", "N"): 0.95,
        ("L", "D"): 0.5,
        ("L", "C"): 1.25,
        ("L", "Q"): 2.25,
        ("L", "E"): 1.55,
        ("L", "G"): 0.25,
        ("L", "H"): 0.6,
        ("L", "I"): 6.35,
        ("L", "L"): 78.6,
        ("L", "K"): 2.95,
        ("L", "M"): 7.6,
        ("L", "F"): 3.55,
        ("L", "P"): 0.8,
        ("L", "S"): 0.25,
        ("L", "T"): 1.05,
        ("L", "W"): 0.5,
        ("L", "Y"): 1.25,
        ("L", "V"): 3.6,
        ("K", "-"): 12.15,
        ("K", "A"): 1.1,
        ("K", "R"): 4.55,
        ("K", "N"): 1.6,
        ("K", "D"): 0.7,
        ("K", "C"): 0.0,
        ("K", "Q"): 2.1,
        ("K", "E"): 1.7,
        ("K", "G"): 1.1,
        ("K", "H"): 1.05,
        ("K", "I"): 1.0,
        ("K", "L"): 2.95,
        ("K", "K"): 46.8,
        ("K", "M"): 0.7,
        ("K", "F"): 0.35,
        ("K", "P"): 0.35,
        ("K", "S"): 0.9,
        ("K", "T"): 1.85,
        ("K", "W"): 0.45,
        ("K", "Y"): 0.8,
        ("K", "V"): 2.3,
        ("M", "-"): 1.3,
        ("M", "A"): 0.85,
        ("M", "R"): 0.2,
        ("M", "N"): 0.35,
        ("M", "D"): 0.1,
        ("M", "C"): 0.6,
        ("M", "Q"): 1.45,
        ("M", "E"): 0.3,
        ("M", "G"): 0.25,
        ("M", "H"): 0.5,
        ("M", "I"): 3.65,
        ("M", "L"): 7.6,
        ("M", "K"): 0.7,
        ("M", "M"): 68.7,
        ("M", "F"): 1.6,
        ("M", "P"): 0.3,
        ("M", "S"): 0.25,
        ("M", "T"): 1.05,
        ("M", "W"): 0.55,
        ("M", "Y"): 0.3,
        ("M", "V"): 2.6,
        ("F", "-"): 1.15,
        ("F", "A"): 0.25,
        ("F", "R"): 0.2,
        ("F", "N"): 0.1,
        ("F", "D"): 0.15,
        ("F", "C"): 0.8,
        ("F", "Q"): 0.6,
        ("F", "E"): 0.0,
        ("F", "G"): 0.25,
        ("F", "H"): 1.25,
        ("F", "I"): 1.5,
        ("F", "L"): 3.55,
        ("F", "K"): 0.35,
        ("F", "M"): 1.6,
        ("F", "F"): 80.2,
        ("F", "P"): 0.25,
        ("F", "S"): 0.8,
        ("F", "T"): 0.3,
        ("F", "W"): 1.5,
        ("F", "Y"): 7.15,
        ("F", "V"): 1.3,
        ("P", "-"): 4.7,
        ("P", "A"): 2.85,
        ("P", "R"): 0.5,
        ("P", "N"): 0.75,
        ("P", "D"): 0.6,
        ("P", "C"): 0.05,
        ("P", "Q"): 1.05,
        ("P", "E"): 0.65,
        ("P", "G"): 0.55,
        ("P", "H"): 0.55,
        ("P", "I"): 0.6,
        ("P", "L"): 0.8,
        ("P", "K"): 0.35,
        ("P", "M"): 0.3,
        ("P", "F"): 0.25,
        ("P", "P"): 73.3,
        ("P", "S"): 2.05,
        ("P", "T"): 1.05,
        ("P", "W"): 0.0,
        ("P", "Y"): 0.1,
        ("P", "V"): 0.65,
        ("S", "-"): 5.3,
        ("S", "A"): 5.75,
        ("S", "R"): 1.4,
        ("S", "N"): 3.15,
        ("S", "D"): 1.75,
        ("S", "C"): 1.8,
        ("S", "Q"): 1.1,
        ("S", "E"): 1.55,
        ("S", "G"): 1.95,
        ("S", "H"): 0.65,
        ("S", "I"): 0.4,
        ("S", "L"): 0.25,
        ("S", "K"): 0.9,
        ("S", "M"): 0.25,
        ("S", "F"): 0.8,
        ("S", "P"): 2.05,
        ("S", "S"): 62.2,
        ("S", "T"): 5.2,
        ("S", "W"): 0.3,
        ("S", "Y"): 0.55,
        ("S", "V"): 0.85,
        ("T", "-"): 2.1,
        ("T", "A"): 2.75,
        ("T", "R"): 1.2,
        ("T", "N"): 1.7,
        ("T", "D"): 0.95,
        ("T", "C"): 0.8,
        ("T", "Q"): 0.95,
        ("T", "E"): 0.95,
        ("T", "G"): 0.65,
        ("T", "H"): 0.4,
        ("T", "I"): 1.95,
        ("T", "L"): 1.05,
        ("T", "K"): 1.85,
        ("T", "M"): 1.05,
        ("T", "F"): 0.3,
        ("T", "P"): 1.05,
        ("T", "S"): 5.2,
        ("T", "T"): 68.2,
        ("T", "W"): 0.1,
        ("T", "Y"): 0.45,
        ("T", "V"): 4.0,
        ("W", "-"): 1.15,
        ("W", "A"): 0.0,
        ("W", "R"): 0.35,
        ("W", "N"): 0.05,
        ("W", "D"): 0.1,
        ("W", "C"): 0.25,
        ("W", "Q"): 0.5,
        ("W", "E"): 0.0,
        ("W", "G"): 0.2,
        ("W", "H"): 0.35,
        ("W", "I"): 0.45,
        ("W", "L"): 0.5,
        ("W", "K"): 0.45,
        ("W", "M"): 0.55,
        ("W", "F"): 1.5,
        ("W", "P"): 0.0,
        ("W", "S"): 0.3,
        ("W", "T"): 0.1,
        ("W", "W"): 89.0,
        ("W", "Y"): 1.2,
        ("W", "V"): 0.5,
        ("Y", "-"): 1.2,
        ("Y", "A"): 0.4,
        ("Y", "R"): 0.4,
        ("Y", "N"): 0.8,
        ("Y", "D"): 0.6,
        ("Y", "C"): 0.35,
        ("Y", "Q"): 0.3,
        ("Y", "E"): 0.8,
        ("Y", "G"): 0.05,
        ("Y", "H"): 2.05,
        ("Y", "I"): 0.55,
        ("Y", "L"): 1.25,
        ("Y", "K"): 0.8,
        ("Y", "M"): 0.3,
        ("Y", "F"): 7.15,
        ("Y", "P"): 0.1,
        ("Y", "S"): 0.55,
        ("Y", "T"): 0.45,
        ("Y", "W"): 1.2,
        ("Y", "Y"): 78.0,
        ("Y", "V"): 0.35,
        ("V", "-"): 0.85,
        ("V", "A"): 3.1,
        ("V", "R"): 0.15,
        ("V", "N"): 0.8,
        ("V", "D"): 0.35,
        ("V", "C"): 2.0,
        ("V", "Q"): 0.3,
        ("V", "E"): 1.55,
        ("V", "G"): 0.25,
        ("V", "H"): 0.15,
        ("V", "I"): 9.95,
        ("V", "L"): 3.6,
        ("V", "K"): 2.3,
        ("V", "M"): 2.6,
        ("V", "F"): 1.3,
        ("V", "P"): 0.65,
        ("V", "S"): 0.85,
        ("V", "T"): 4.0,
        ("V", "W"): 0.5,
        ("V", "Y"): 0.35,
        ("V", "V"): 73.0,
    }

    return (
        symmetric_koshi_goldstein_all_residues_dict,
        symmetric_koshi_goldstein_exposed_residues_dict,
        symmetric_koshi_goldstein_buried_residues_dict,
    )


def make_asymmetric_koshi_golstein_dicts() -> tuple[
    dict[tuple[str, str], float],
    dict[tuple[str, str], float],
    dict[tuple[str, str], float],
]:
    """
    :return: All dictionaries representative of FIGURE 2 from 'Context-dependent optimal substituion matrices' by
             Koshi and Golstein. Keys are pairs of strings. Cf. scientific paper for more details on how
             those scores are derived.
    """
    asymmetric_koshi_goldstein_all_residues_dict: dict[tuple[str, str], float] = {
        ("-", "-"): 69.2,
        ("-", "A"): 1.6,
        ("-", "R"): 2.0,
        ("-", "N"): 1.1,
        ("-", "D"): 1.3,
        ("-", "C"): 2.4,
        ("-", "Q"): 2.0,
        ("-", "E"): 1.4,
        ("-", "G"): 1.5,
        ("-", "H"): 1.7,
        ("-", "I"): 1.2,
        ("-", "L"): 1.3,
        ("-", "K"): 0.9,
        ("-", "M"): 1.7,
        ("-", "F"): 1.1,
        ("-", "P"): 2.1,
        ("-", "S"): 2.1,
        ("-", "T"): 1.5,
        ("-", "W"): 1.0,
        ("-", "Y"): 1.3,
        ("-", "V"): 1.7,
        ("A", "-"): 8.9,
        ("A", "A"): 61.9,
        ("A", "R"): 0.8,
        ("A", "N"): 0.6,
        ("A", "D"): 1.3,
        ("A", "C"): 1.6,
        ("A", "Q"): 1.0,
        ("A", "E"): 1.6,
        ("A", "G"): 3.7,
        ("A", "H"): 0.3,
        ("A", "I"): 0.6,
        ("A", "L"): 1.6,
        ("A", "K"): 1.3,
        ("A", "M"): 0.5,
        ("A", "F"): 0.2,
        ("A", "P"): 1.9,
        ("A", "S"): 5.9,
        ("A", "T"): 3.2,
        ("A", "W"): 0.2,
        ("A", "Y"): 0.4,
        ("A", "V"): 2.6,
        ("R", "-"): 11.3,
        ("R", "A"): 1.3,
        ("R", "R"): 67.2,
        ("R", "N"): 1.4,
        ("R", "D"): 0.7,
        ("R", "C"): 0.2,
        ("R", "Q"): 2.4,
        ("R", "E"): 0.7,
        ("R", "G"): 0.6,
        ("R", "H"): 1.3,
        ("R", "I"): 0.6,
        ("R", "L"): 1.0,
        ("R", "K"): 5.7,
        ("R", "M"): 0.3,
        ("R", "F"): 0.2,
        ("R", "P"): 0.7,
        ("R", "S"): 1.9,
        ("R", "T"): 1.4,
        ("R", "W"): 0.3,
        ("R", "Y"): 0.5,
        ("R", "V"): 0.4,
        ("N", "-"): 6.4,
        ("N", "A"): 1.8,
        ("N", "R"): 1.5,
        ("N", "N"): 60.9,
        ("N", "D"): 5.4,
        ("N", "C"): 0.3,
        ("N", "Q"): 1.8,
        ("N", "E"): 2.0,
        ("N", "G"): 2.5,
        ("N", "H"): 1.6,
        ("N", "I"): 0.5,
        ("N", "L"): 0.8,
        ("N", "K"): 3.5,
        ("N", "M"): 0.7,
        ("N", "F"): 0.1,
        ("N", "P"): 0.4,
        ("N", "S"): 5.6,
        ("N", "T"): 2.0,
        ("N", "W"): 0.4,
        ("N", "Y"): 1.0,
        ("N", "V"): 1.0,
        ("D", "-"): 7.3,
        ("D", "A"): 2.0,
        ("D", "R"): 0.5,
        ("D", "N"): 6.1,
        ("D", "D"): 65.0,
        ("D", "C"): 0.4,
        ("D", "Q"): 1.2,
        ("D", "E"): 6.9,
        ("D", "G"): 1.9,
        ("D", "H"): 0.3,
        ("D", "I"): 0.4,
        ("D", "L"): 0.4,
        ("D", "K"): 1.9,
        ("D", "M"): 0.3,
        ("D", "F"): 0.2,
        ("D", "P"): 1.1,
        ("D", "S"): 2.4,
        ("D", "T"): 1.0,
        ("D", "W"): 0.0,
        ("D", "Y"): 0.5,
        ("D", "V"): 0.2,
        ("C", "-"): 13.8,
        ("C", "A"): 6.6,
        ("C", "R"): 0.0,
        ("C", "N"): 0.3,
        ("C", "D"): 1.6,
        ("C", "C"): 64.4,
        ("C", "Q"): 0.0,
        ("C", "E"): 1.2,
        ("C", "G"): 1.3,
        ("C", "H"): 0.5,
        ("C", "I"): 1.8,
        ("C", "L"): 1.2,
        ("C", "K"): 0.0,
        ("C", "M"): 0.5,
        ("C", "F"): 1.6,
        ("C", "P"): 0.0,
        ("C", "S"): 2.6,
        ("C", "T"): 1.0,
        ("C", "W"): 0.2,
        ("C", "Y"): 0.2,
        ("C", "V"): 1.2,
        ("Q", "-"): 11.4,
        ("Q", "A"): 2.4,
        ("Q", "R"): 3.3,
        ("Q", "N"): 1.6,
        ("Q", "D"): 1.7,
        ("Q", "C"): 0.2,
        ("Q", "Q"): 58.4,
        ("Q", "E"): 4.9,
        ("Q", "G"): 1.0,
        ("Q", "H"): 1.8,
        ("Q", "I"): 0.5,
        ("Q", "L"): 2.0,
        ("Q", "K"): 3.5,
        ("Q", "M"): 0.8,
        ("Q", "F"): 0.3,
        ("Q", "P"): 1.0,
        ("Q", "S"): 1.9,
        ("Q", "T"): 1.9,
        ("Q", "W"): 0.5,
        ("Q", "Y"): 0.5,
        ("Q", "V"): 0.6,
        ("E", "-"): 7.9,
        ("E", "A"): 3.2,
        ("E", "R"): 1.0,
        ("E", "N"): 1.4,
        ("E", "D"): 5.8,
        ("E", "C"): 0.3,
        ("E", "Q"): 4.3,
        ("E", "E"): 62.1,
        ("E", "G"): 1.2,
        ("E", "H"): 0.9,
        ("E", "I"): 0.4,
        ("E", "L"): 1.1,
        ("E", "K"): 4.0,
        ("E", "M"): 0.2,
        ("E", "F"): 0.1,
        ("E", "P"): 1.2,
        ("E", "S"): 2.2,
        ("E", "T"): 1.3,
        ("E", "W"): 0.2,
        ("E", "Y"): 0.1,
        ("E", "V"): 1.2,
        ("G", "-"): 8.4,
        ("G", "A"): 2.5,
        ("G", "R"): 0.4,
        ("G", "N"): 1.4,
        ("G", "D"): 1.6,
        ("G", "C"): 0.4,
        ("G", "Q"): 0.5,
        ("G", "E"): 1.1,
        ("G", "G"): 77.8,
        ("G", "H"): 0.3,
        ("G", "I"): 0.0,
        ("G", "L"): 0.2,
        ("G", "K"): 1.0,
        ("G", "M"): 0.1,
        ("G", "F"): 0.2,
        ("G", "P"): 0.3,
        ("G", "S"): 1.9,
        ("G", "T"): 1.1,
        ("G", "W"): 0.1,
        ("G", "Y"): 0.2,
        ("G", "V"): 0.4,
        ("H", "-"): 9.5,
        ("H", "A"): 0.9,
        ("H", "R"): 2.7,
        ("H", "N"): 2.8,
        ("H", "D"): 1.1,
        ("H", "C"): 0.3,
        ("H", "Q"): 2.4,
        ("H", "E"): 1.4,
        ("H", "G"): 0.8,
        ("H", "H"): 65.4,
        ("H", "I"): 0.5,
        ("H", "L"): 1.5,
        ("H", "K"): 1.5,
        ("H", "M"): 0.8,
        ("H", "F"): 1.5,
        ("H", "P"): 1.0,
        ("H", "S"): 1.9,
        ("H", "T"): 0.5,
        ("H", "W"): 0.4,
        ("H", "Y"): 2.9,
        ("H", "V"): 0.3,
        ("I", "-"): 6.9,
        ("I", "A"): 0.8,
        ("I", "R"): 0.3,
        ("I", "N"): 0.4,
        ("I", "D"): 0.1,
        ("I", "C"): 0.4,
        ("I", "Q"): 0.1,
        ("I", "E"): 0.4,
        ("I", "G"): 0.2,
        ("I", "H"): 0.1,
        ("I", "I"): 64.8,
        ("I", "L"): 7.7,
        ("I", "K"): 0.4,
        ("I", "M"): 2.0,
        ("I", "F"): 1.7,
        ("I", "P"): 0.3,
        ("I", "S"): 0.3,
        ("I", "T"): 1.6,
        ("I", "W"): 0.3,
        ("I", "Y"): 0.3,
        ("I", "V"): 10.9,
        ("L", "-"): 7.1,
        ("L", "A"): 0.9,
        ("L", "R"): 0.7,
        ("L", "N"): 0.2,
        ("L", "D"): 0.1,
        ("L", "C"): 0.3,
        ("L", "Q"): 1.0,
        ("L", "E"): 0.3,
        ("L", "G"): 0.1,
        ("L", "H"): 0.4,
        ("L", "I"): 4.7,
        ("L", "L"): 71.7,
        ("L", "K"): 0.8,
        ("L", "M"): 3.2,
        ("L", "F"): 2.8,
        ("L", "P"): 0.3,
        ("L", "S"): 0.2,
        ("L", "T"): 0.6,
        ("L", "W"): 0.2,
        ("L", "Y"): 0.8,
        ("L", "V"): 3.7,
        ("K", "-"): 5.0,
        ("K", "A"): 2.9,
        ("K", "R"): 8.1,
        ("K", "N"): 2.3,
        ("K", "D"): 1.4,
        ("K", "C"): 0.1,
        ("K", "Q"): 4.1,
        ("K", "E"): 4.3,
        ("K", "G"): 1.1,
        ("K", "H"): 1.2,
        ("K", "I"): 0.4,
        ("K", "L"): 1.4,
        ("K", "K"): 58.7,
        ("K", "M"): 0.5,
        ("K", "F"): 0.5,
        ("K", "P"): 1.2,
        ("K", "S"): 2.5,
        ("K", "T"): 2.6,
        ("K", "W"): 0.1,
        ("K", "Y"): 0.3,
        ("K", "V"): 1.4,
        ("M", "-"): 9.4,
        ("M", "A"): 2.0,
        ("M", "R"): 0.8,
        ("M", "N"): 0.2,
        ("M", "D"): 1.3,
        ("M", "C"): 0.4,
        ("M", "Q"): 1.0,
        ("M", "E"): 0.9,
        ("M", "G"): 0.6,
        ("M", "H"): 0.2,
        ("M", "I"): 5.5,
        ("M", "L"): 9.2,
        ("M", "K"): 0.8,
        ("M", "M"): 59.4,
        ("M", "F"): 2.2,
        ("M", "P"): 0.1,
        ("M", "S"): 0.6,
        ("M", "T"): 1.7,
        ("M", "W"): 0.2,
        ("M", "Y"): 0.2,
        ("M", "V"): 3.4,
        ("F", "-"): 6.4,
        ("F", "A"): 0.3,
        ("F", "R"): 0.2,
        ("F", "N"): 0.4,
        ("F", "D"): 0.2,
        ("F", "C"): 0.4,
        ("F", "Q"): 0.4,
        ("F", "E"): 0.2,
        ("F", "G"): 0.3,
        ("F", "H"): 0.4,
        ("F", "I"): 1.4,
        ("F", "L"): 3.7,
        ("F", "K"): 0.1,
        ("F", "M"): 0.9,
        ("F", "F"): 73.8,
        ("F", "P"): 0.3,
        ("F", "S"): 0.9,
        ("F", "T"): 0.6,
        ("F", "W"): 1.4,
        ("F", "Y"): 6.4,
        ("F", "V"): 1.5,
        ("P", "-"): 12.0,
        ("P", "A"): 3.5,
        ("P", "R"): 0.9,
        ("P", "N"): 0.6,
        ("P", "D"): 0.6,
        ("P", "C"): 0.0,
        ("P", "Q"): 0.8,
        ("P", "E"): 1.0,
        ("P", "G"): 0.7,
        ("P", "H"): 0.4,
        ("P", "I"): 0.5,
        ("P", "L"): 0.9,
        ("P", "K"): 0.9,
        ("P", "M"): 0.6,
        ("P", "F"): 0.1,
        ("P", "P"): 71.2,
        ("P", "S"): 2.8,
        ("P", "T"): 1.4,
        ("P", "W"): 0.0,
        ("P", "Y"): 0.5,
        ("P", "V"): 0.7,
        ("S", "-"): 11.7,
        ("S", "A"): 5.1,
        ("S", "R"): 1.4,
        ("S", "N"): 3.0,
        ("S", "D"): 2.2,
        ("S", "C"): 0.9,
        ("S", "Q"): 1.1,
        ("S", "E"): 1.5,
        ("S", "G"): 3.3,
        ("S", "H"): 0.6,
        ("S", "I"): 0.4,
        ("S", "L"): 0.5,
        ("S", "K"): 1.4,
        ("S", "M"): 0.3,
        ("S", "F"): 0.5,
        ("S", "P"): 1.8,
        ("S", "S"): 56.8,
        ("S", "T"): 6.1,
        ("S", "W"): 0.3,
        ("S", "Y"): 0.5,
        ("S", "V"): 0.8,
        ("T", "-"): 8.8,
        ("T", "A"): 3.1,
        ("T", "R"): 0.9,
        ("T", "N"): 2.2,
        ("T", "D"): 1.1,
        ("T", "C"): 0.4,
        ("T", "Q"): 1.1,
        ("T", "E"): 1.2,
        ("T", "G"): 0.9,
        ("T", "H"): 0.6,
        ("T", "I"): 2.0,
        ("T", "L"): 1.4,
        ("T", "K"): 2.0,
        ("T", "M"): 0.8,
        ("T", "F"): 0.1,
        ("T", "P"): 1.1,
        ("T", "S"): 6.9,
        ("T", "T"): 61.8,
        ("T", "W"): 0.1,
        ("T", "Y"): 0.4,
        ("T", "V"): 3.2,
        ("W", "-"): 6.0,
        ("W", "A"): 0.1,
        ("W", "R"): 0.4,
        ("W", "N"): 0.1,
        ("W", "D"): 0.0,
        ("W", "C"): 0.7,
        ("W", "Q"): 0.1,
        ("W", "E"): 0.0,
        ("W", "G"): 0.5,
        ("W", "H"): 0.1,
        ("W", "I"): 0.4,
        ("W", "L"): 1.0,
        ("W", "K"): 0.6,
        ("W", "M"): 0.8,
        ("W", "F"): 2.3,
        ("W", "P"): 0.0,
        ("W", "S"): 1.3,
        ("W", "T"): 0.2,
        ("W", "W"): 82.3,
        ("W", "Y"): 2.5,
        ("W", "V"): 0.7,
        ("Y", "-"): 7.4,
        ("Y", "A"): 0.6,
        ("Y", "R"): 0.5,
        ("Y", "N"): 0.6,
        ("Y", "D"): 0.4,
        ("Y", "C"): 0.7,
        ("Y", "Q"): 0.2,
        ("Y", "E"): 0.5,
        ("Y", "G"): 0.3,
        ("Y", "H"): 1.7,
        ("Y", "I"): 0.9,
        ("Y", "L"): 2.0,
        ("Y", "K"): 0.6,
        ("Y", "M"): 0.4,
        ("Y", "F"): 7.1,
        ("Y", "P"): 0.1,
        ("Y", "S"): 0.6,
        ("Y", "T"): 0.9,
        ("Y", "W"): 0.7,
        ("Y", "Y"): 73.2,
        ("Y", "V"): 0.7,
        ("V", "-"): 9.8,
        ("V", "A"): 3.7,
        ("V", "R"): 0.5,
        ("V", "N"): 0.4,
        ("V", "D"): 0.2,
        ("V", "C"): 0.8,
        ("V", "Q"): 0.5,
        ("V", "E"): 0.4,
        ("V", "G"): 0.3,
        ("V", "H"): 0.1,
        ("V", "I"): 8.8,
        ("V", "L"): 3.6,
        ("V", "K"): 0.4,
        ("V", "M"): 0.8,
        ("V", "F"): 1.2,
        ("V", "P"): 0.4,
        ("V", "S"): 0.9,
        ("V", "T"): 2.5,
        ("V", "W"): 0.2,
        ("V", "Y"): 0.5,
        ("V", "V"): 64.0,
    }
    asymmetric_koshi_goldstein_exposed_residues_dict: dict[tuple[str, str], float] = {
        ("-", "-"): 60.2,
        ("-", "A"): 3.3,
        ("-", "R"): 1.7,
        ("-", "N"): 1.1,
        ("-", "D"): 1.1,
        ("-", "C"): 2.8,
        ("-", "Q"): 2.1,
        ("-", "E"): 1.2,
        ("-", "G"): 1.8,
        ("-", "H"): 2.2,
        ("-", "I"): 2.6,
        ("-", "L"): 2.3,
        ("-", "K"): 0.7,
        ("-", "M"): 2.2,
        ("-", "F"): 2.2,
        ("-", "P"): 2.1,
        ("-", "S"): 1.7,
        ("-", "T"): 2.1,
        ("-", "W"): 2.1,
        ("-", "Y"): 1.8,
        ("-", "V"): 2.7,
        ("A", "-"): 21.5,
        ("A", "A"): 51.3,
        ("A", "R"): 0.7,
        ("A", "N"): 0.6,
        ("A", "D"): 1.1,
        ("A", "C"): 1.0,
        ("A", "Q"): 1.2,
        ("A", "E"): 1.2,
        ("A", "G"): 3.0,
        ("A", "H"): 0.4,
        ("A", "I"): 1.2,
        ("A", "L"): 2.1,
        ("A", "K"): 1.2,
        ("A", "M"): 0.7,
        ("A", "F"): 0.2,
        ("A", "P"): 2.1,
        ("A", "S"): 4.2,
        ("A", "T"): 2.8,
        ("A", "W"): 0.3,
        ("A", "Y"): 0.2,
        ("A", "V"): 3.2,
        ("R", "-"): 11.4,
        ("R", "A"): 1.2,
        ("R", "R"): 62.9,
        ("R", "N"): 1.6,
        ("R", "D"): 0.7,
        ("R", "C"): 0.1,
        ("R", "Q"): 2.9,
        ("R", "E"): 0.9,
        ("R", "G"): 0.7,
        ("R", "H"): 1.6,
        ("R", "I"): 0.6,
        ("R", "L"): 1.1,
        ("R", "K"): 7.3,
        ("R", "M"): 0.5,
        ("R", "F"): 0.4,
        ("R", "P"): 0.6,
        ("R", "S"): 2.2,
        ("R", "T"): 1.6,
        ("R", "W"): 0.6,
        ("R", "Y"): 0.4,
        ("R", "V"): 0.8,
        ("N", "-"): 7.5,
        ("N", "A"): 1.8,
        ("N", "R"): 1.5,
        ("N", "N"): 52.6,
        ("N", "D"): 6.4,
        ("N", "C"): 0.4,
        ("N", "Q"): 2.3,
        ("N", "E"): 2.7,
        ("N", "G"): 2.8,
        ("N", "H"): 1.7,
        ("N", "I"): 0.4,
        ("N", "L"): 1.0,
        ("N", "K"): 4.8,
        ("N", "M"): 0.8,
        ("N", "F"): 0.4,
        ("N", "P"): 0.1,
        ("N", "S"): 7.9,
        ("N", "T"): 2.7,
        ("N", "W"): 0.3,
        ("N", "Y"): 1.2,
        ("N", "V"): 0.9,
        ("D", "-"): 7.4,
        ("D", "A"): 2.8,
        ("D", "R"): 0.7,
        ("D", "N"): 8.0,
        ("D", "D"): 59.2,
        ("D", "C"): 0.3,
        ("D", "Q"): 1.7,
        ("D", "E"): 8.3,
        ("D", "G"): 2.3,
        ("D", "H"): 0.3,
        ("D", "I"): 0.4,
        ("D", "L"): 0.4,
        ("D", "K"): 2.3,
        ("D", "M"): 0.2,
        ("D", "F"): 0.1,
        ("D", "P"): 1.4,
        ("D", "S"): 2.6,
        ("D", "T"): 1.2,
        ("D", "W"): 0.0,
        ("D", "Y"): 0.4,
        ("D", "V"): 0.2,
        ("C", "-"): 18.1,
        ("C", "A"): 4.2,
        ("C", "R"): 0.0,
        ("C", "N"): 0.3,
        ("C", "D"): 1.3,
        ("C", "C"): 64.7,
        ("C", "Q"): 0.0,
        ("C", "E"): 0.9,
        ("C", "G"): 1.1,
        ("C", "H"): 0.5,
        ("C", "I"): 1.1,
        ("C", "L"): 1.0,
        ("C", "K"): 0.0,
        ("C", "M"): 1.5,
        ("C", "F"): 2.2,
        ("C", "P"): 0.0,
        ("C", "S"): 1.7,
        ("C", "T"): 0.9,
        ("C", "W"): 0.0,
        ("C", "Y"): 0.1,
        ("C", "V"): 0.3,
        ("Q", "-"): 13.9,
        ("Q", "A"): 3.1,
        ("Q", "R"): 3.7,
        ("Q", "N"): 1.8,
        ("Q", "D"): 1.7,
        ("Q", "C"): 0.1,
        ("Q", "Q"): 51.0,
        ("Q", "E"): 5.9,
        ("Q", "G"): 1.0,
        ("Q", "H"): 1.7,
        ("Q", "I"): 0.5,
        ("Q", "L"): 3.0,
        ("Q", "K"): 3.9,
        ("Q", "M"): 0.8,
        ("Q", "F"): 0.1,
        ("Q", "P"): 1.0,
        ("Q", "S"): 2.3,
        ("Q", "T"): 2.6,
        ("Q", "W"): 0.1,
        ("Q", "Y"): 0.6,
        ("Q", "V"): 1.4,
        ("E", "-"): 7.8,
        ("E", "A"): 4.3,
        ("E", "R"): 0.9,
        ("E", "N"): 1.3,
        ("E", "D"): 6.7,
        ("E", "C"): 0.2,
        ("E", "Q"): 4.5,
        ("E", "E"): 58.3,
        ("E", "G"): 1.1,
        ("E", "H"): 1.1,
        ("E", "I"): 0.6,
        ("E", "L"): 1.0,
        ("E", "K"): 4.2,
        ("E", "M"): 0.5,
        ("E", "F"): 0.3,
        ("E", "P"): 1.4,
        ("E", "S"): 2.5,
        ("E", "T"): 1.8,
        ("E", "W"): 0.2,
        ("E", "Y"): 0.1,
        ("E", "V"): 1.3,
        ("G", "-"): 11.8,
        ("G", "A"): 2.6,
        ("G", "R"): 0.4,
        ("G", "N"): 2.2,
        ("G", "D"): 2.3,
        ("G", "C"): 0.3,
        ("G", "Q"): 0.6,
        ("G", "E"): 1.6,
        ("G", "G"): 71.0,
        ("G", "H"): 0.3,
        ("G", "I"): 0.0,
        ("G", "L"): 0.1,
        ("G", "K"): 1.2,
        ("G", "M"): 0.3,
        ("G", "F"): 0.5,
        ("G", "P"): 0.3,
        ("G", "S"): 2.2,
        ("G", "T"): 1.7,
        ("G", "W"): 0.2,
        ("G", "Y"): 0.2,
        ("G", "V"): 0.4,
        ("H", "-"): 14.1,
        ("H", "A"): 0.9,
        ("H", "R"): 3.3,
        ("H", "N"): 5.0,
        ("H", "D"): 1.0,
        ("H", "C"): 0.1,
        ("H", "Q"): 2.2,
        ("H", "E"): 1.7,
        ("H", "G"): 0.8,
        ("H", "H"): 54.8,
        ("H", "I"): 0.6,
        ("H", "L"): 1.1,
        ("H", "K"): 1.3,
        ("H", "M"): 0.5,
        ("H", "F"): 1.9,
        ("H", "P"): 1.4,
        ("H", "S"): 2.6,
        ("H", "T"): 0.1,
        ("H", "W"): 0.0,
        ("H", "Y"): 4.4,
        ("H", "V"): 2.4,
        ("I", "-"): 17.2,
        ("I", "A"): 0.3,
        ("I", "R"): 0.3,
        ("I", "N"): 0.2,
        ("I", "D"): 0.1,
        ("I", "C"): 0.3,
        ("I", "Q"): 0.2,
        ("I", "E"): 0.3,
        ("I", "G"): 0.0,
        ("I", "H"): 0.1,
        ("I", "I"): 54.4,
        ("I", "L"): 7.6,
        ("I", "K"): 0.7,
        ("I", "M"): 2.0,
        ("I", "F"): 1.5,
        ("I", "P"): 0.6,
        ("I", "S"): 0.0,
        ("I", "T"): 2.1,
        ("I", "W"): 0.3,
        ("I", "Y"): 0.6,
        ("I", "V"): 11.3,
        ("L", "-"): 15.0,
        ("L", "A"): 0.7,
        ("L", "R"): 1.2,
        ("L", "N"): 0.4,
        ("L", "D"): 0.1,
        ("L", "C"): 0.2,
        ("L", "Q"): 0.8,
        ("L", "E"): 0.4,
        ("L", "G"): 0.3,
        ("L", "H"): 0.6,
        ("L", "I"): 3.8,
        ("L", "L"): 64.4,
        ("L", "K"): 0.8,
        ("L", "M"): 3.5,
        ("L", "F"): 2.8,
        ("L", "P"): 0.5,
        ("L", "S"): 0.3,
        ("L", "T"): 0.6,
        ("L", "W"): 0.5,
        ("L", "Y"): 0.7,
        ("L", "V"): 2.6,
        ("K", "-"): 4.8,
        ("K", "A"): 3.8,
        ("K", "R"): 7.9,
        ("K", "N"): 2.3,
        ("K", "D"): 1.6,
        ("K", "C"): 0.2,
        ("K", "Q"): 4.6,
        ("K", "E"): 4.9,
        ("K", "G"): 1.2,
        ("K", "H"): 1.3,
        ("K", "I"): 0.4,
        ("K", "L"): 1.4,
        ("K", "K"): 56.0,
        ("K", "M"): 0.4,
        ("K", "F"): 0.4,
        ("K", "P"): 0.9,
        ("K", "S"): 3.0,
        ("K", "T"): 3.2,
        ("K", "W"): 0.1,
        ("K", "Y"): 0.3,
        ("K", "V"): 1.4,
        ("M", "-"): 14.4,
        ("M", "A"): 2.8,
        ("M", "R"): 0.7,
        ("M", "N"): 0.2,
        ("M", "D"): 0.0,
        ("M", "C"): 0.4,
        ("M", "Q"): 1.1,
        ("M", "E"): 1.9,
        ("M", "G"): 1.1,
        ("M", "H"): 0.1,
        ("M", "I"): 6.6,
        ("M", "L"): 7.7,
        ("M", "K"): 1.8,
        ("M", "M"): 49.9,
        ("M", "F"): 2.1,
        ("M", "P"): 0.1,
        ("M", "S"): 1.4,
        ("M", "T"): 2.7,
        ("M", "W"): 0.8,
        ("M", "Y"): 0.8,
        ("M", "V"): 3.5,
        ("F", "-"): 14.2,
        ("F", "A"): 0.6,
        ("F", "R"): 0.5,
        ("F", "N"): 0.5,
        ("F", "D"): 0.2,
        ("F", "C"): 0.5,
        ("F", "Q"): 0.4,
        ("F", "E"): 0.2,
        ("F", "G"): 0.1,
        ("F", "H"): 0.5,
        ("F", "I"): 1.7,
        ("F", "L"): 2.8,
        ("F", "K"): 0.7,
        ("F", "M"): 0.6,
        ("F", "F"): 66.6,
        ("F", "P"): 0.4,
        ("F", "S"): 0.8,
        ("F", "T"): 0.5,
        ("F", "W"): 1.1,
        ("F", "Y"): 5.9,
        ("F", "V"): 1.2,
        ("P", "-"): 13.8,
        ("P", "A"): 4.9,
        ("P", "R"): 1.3,
        ("P", "N"): 0.6,
        ("P", "D"): 0.8,
        ("P", "C"): 0.0,
        ("P", "Q"): 0.8,
        ("P", "E"): 1.3,
        ("P", "G"): 0.9,
        ("P", "H"): 0.4,
        ("P", "I"): 0.4,
        ("P", "L"): 1.0,
        ("P", "K"): 1.3,
        ("P", "M"): 0.3,
        ("P", "F"): 0.1,
        ("P", "P"): 66.0,
        ("P", "S"): 3.5,
        ("P", "T"): 1.3,
        ("P", "W"): 0.1,
        ("P", "Y"): 0.4,
        ("P", "V"): 0.9,
        ("S", "-"): 11.3,
        ("S", "A"): 7.4,
        ("S", "R"): 1.3,
        ("S", "N"): 2.7,
        ("S", "D"): 3.0,
        ("S", "C"): 0.9,
        ("S", "Q"): 1.1,
        ("S", "E"): 1.9,
        ("S", "G"): 4.3,
        ("S", "H"): 0.8,
        ("S", "I"): 0.1,
        ("S", "L"): 0.5,
        ("S", "K"): 1.8,
        ("S", "M"): 0.3,
        ("S", "F"): 0.3,
        ("S", "P"): 2.3,
        ("S", "S"): 50.1,
        ("S", "T"): 8.1,
        ("S", "W"): 0.0,
        ("S", "Y"): 0.5,
        ("S", "V"): 1.3,
        ("T", "-"): 13.7,
        ("T", "A"): 4.0,
        ("T", "R"): 0.9,
        ("T", "N"): 2.5,
        ("T", "D"): 1.2,
        ("T", "C"): 0.2,
        ("T", "Q"): 1.3,
        ("T", "E"): 1.2,
        ("T", "G"): 0.6,
        ("T", "H"): 0.6,
        ("T", "I"): 2.8,
        ("T", "L"): 2.3,
        ("T", "K"): 2.0,
        ("T", "M"): 0.7,
        ("T", "F"): 0.2,
        ("T", "P"): 1.3,
        ("T", "S"): 7.4,
        ("T", "T"): 53.0,
        ("T", "W"): 0.3,
        ("T", "Y"): 0.4,
        ("T", "V"): 3.6,
        ("W", "-"): 13.7,
        ("W", "A"): 0.1,
        ("W", "R"): 0.1,
        ("W", "N"): 0.1,
        ("W", "D"): 0.0,
        ("W", "C"): 0.0,
        ("W", "Q"): 0.0,
        ("W", "E"): 0.0,
        ("W", "G"): 0.0,
        ("W", "H"): 0.0,
        ("W", "I"): 0.6,
        ("W", "L"): 2.0,
        ("W", "K"): 0.5,
        ("W", "M"): 0.2,
        ("W", "F"): 2.5,
        ("W", "P"): 0.4,
        ("W", "S"): 0.0,
        ("W", "T"): 0.1,
        ("W", "W"): 75.4,
        ("W", "Y"): 4.2,
        ("W", "V"): 0.1,
        ("Y", "-"): 11.6,
        ("Y", "A"): 0.6,
        ("Y", "R"): 0.4,
        ("Y", "N"): 0.4,
        ("Y", "D"): 0.2,
        ("Y", "C"): 0.5,
        ("Y", "Q"): 0.1,
        ("Y", "E"): 0.6,
        ("Y", "G"): 0.2,
        ("Y", "H"): 1.5,
        ("Y", "I"): 1.6,
        ("Y", "L"): 3.0,
        ("Y", "K"): 0.6,
        ("Y", "M"): 0.4,
        ("Y", "F"): 7.8,
        ("Y", "P"): 0.1,
        ("Y", "S"): 0.7,
        ("Y", "T"): 0.7,
        ("Y", "W"): 1.0,
        ("Y", "Y"): 66.8,
        ("Y", "V"): 1.2,
        ("V", "-"): 17.5,
        ("V", "A"): 2.5,
        ("V", "R"): 0.4,
        ("V", "N"): 0.5,
        ("V", "D"): 0.4,
        ("V", "C"): 0.6,
        ("V", "Q"): 0.8,
        ("V", "E"): 0.6,
        ("V", "G"): 0.6,
        ("V", "H"): 0.6,
        ("V", "I"): 10.4,
        ("V", "L"): 3.7,
        ("V", "K"): 0.7,
        ("V", "M"): 0.9,
        ("V", "F"): 1.6,
        ("V", "P"): 0.3,
        ("V", "S"): 0.7,
        ("V", "T"): 1.6,
        ("V", "W"): 0.3,
        ("V", "Y"): 0.7,
        ("V", "V"): 54.8,
    }
    asymmetric_koshi_goldstein_buried_residues_dict: dict[tuple[str, str], float] = {
        ("-", "-"): 79.2,
        ("-", "A"): 0.2,
        ("-", "R"): 1.6,
        ("-", "N"): 1.2,
        ("-", "D"): 1.9,
        ("-", "C"): 1.2,
        ("-", "Q"): 1.2,
        ("-", "E"): 2.0,
        ("-", "G"): 0.7,
        ("-", "H"): 0.2,
        ("-", "I"): 0.4,
        ("-", "L"): 0.2,
        ("-", "K"): 4.0,
        ("-", "M"): 0.4,
        ("-", "F"): 0.4,
        ("-", "P"): 1.6,
        ("-", "S"): 1.8,
        ("-", "T"): 0.7,
        ("-", "W"): 0.4,
        ("-", "Y"): 0.4,
        ("-", "V"): 0.3,
        ("A", "-"): 1.0,
        ("A", "A"): 76.3,
        ("A", "R"): 0.6,
        ("A", "N"): 0.5,
        ("A", "D"): 0.6,
        ("A", "C"): 1.7,
        ("A", "Q"): 0.6,
        ("A", "E"): 1.0,
        ("A", "G"): 2.8,
        ("A", "H"): 0.2,
        ("A", "I"): 0.9,
        ("A", "L"): 1.5,
        ("A", "K"): 0.4,
        ("A", "M"): 0.5,
        ("A", "F"): 0.3,
        ("A", "P"): 1.1,
        ("A", "S"): 4.4,
        ("A", "T"): 1.7,
        ("A", "W"): 0.0,
        ("A", "Y"): 0.4,
        ("A", "V"): 3.7,
        ("R", "-"): 8.2,
        ("R", "A"): 0.7,
        ("R", "R"): 72.9,
        ("R", "N"): 0.9,
        ("R", "D"): 0.1,
        ("R", "C"): 0.4,
        ("R", "Q"): 2.4,
        ("R", "E"): 0.4,
        ("R", "G"): 0.9,
        ("R", "H"): 1.7,
        ("R", "I"): 1.1,
        ("R", "L"): 1.4,
        ("R", "K"): 3.7,
        ("R", "M"): 0.2,
        ("R", "F"): 0.3,
        ("R", "P"): 0.6,
        ("R", "S"): 1.7,
        ("R", "T"): 1.9,
        ("R", "W"): 0.1,
        ("R", "Y"): 0.4,
        ("R", "V"): 0.1,
        ("N", "-"): 6.2,
        ("N", "A"): 2.0,
        ("N", "R"): 1.1,
        ("N", "N"): 65.7,
        ("N", "D"): 3.1,
        ("N", "C"): 0.6,
        ("N", "Q"): 1.3,
        ("N", "E"): 0.9,
        ("N", "G"): 2.5,
        ("N", "H"): 1.6,
        ("N", "I"): 1.2,
        ("N", "L"): 1.5,
        ("N", "K"): 1.6,
        ("N", "M"): 0.4,
        ("N", "F"): 0.1,
        ("N", "P"): 0.5,
        ("N", "S"): 4.8,
        ("N", "T"): 2.6,
        ("N", "W"): 0.1,
        ("N", "Y"): 1.0,
        ("N", "V"): 1.3,
        ("D", "-"): 9.7,
        ("D", "A"): 2.5,
        ("D", "R"): 0.6,
        ("D", "N"): 3.0,
        ("D", "D"): 66.9,
        ("D", "C"): 0.4,
        ("D", "Q"): 0.6,
        ("D", "E"): 4.2,
        ("D", "G"): 2.0,
        ("D", "H"): 0.7,
        ("D", "I"): 0.8,
        ("D", "L"): 0.8,
        ("D", "K"): 0.9,
        ("D", "M"): 0.0,
        ("D", "F"): 0.2,
        ("D", "P"): 1.0,
        ("D", "S"): 2.7,
        ("D", "T"): 1.5,
        ("D", "W"): 0.2,
        ("D", "Y"): 1.0,
        ("D", "V"): 0.5,
        ("C", "-"): 5.7,
        ("C", "A"): 6.9,
        ("C", "R"): 0.1,
        ("C", "N"): 0.1,
        ("C", "D"): 1.5,
        ("C", "C"): 71.6,
        ("C", "Q"): 0.1,
        ("C", "E"): 0.6,
        ("C", "G"): 1.4,
        ("C", "H"): 0.3,
        ("C", "I"): 1.8,
        ("C", "L"): 2.0,
        ("C", "K"): 0.0,
        ("C", "M"): 0.5,
        ("C", "F"): 1.3,
        ("C", "P"): 0.0,
        ("C", "S"): 2.0,
        ("C", "T"): 0.7,
        ("C", "W"): 0.1,
        ("C", "Y"): 0.1,
        ("C", "V"): 3.2,
        ("Q", "-"): 6.1,
        ("Q", "A"): 2.4,
        ("Q", "R"): 1.7,
        ("Q", "N"): 1.3,
        ("Q", "D"): 0.9,
        ("Q", "C"): 0.4,
        ("Q", "Q"): 67.3,
        ("Q", "E"): 2.0,
        ("Q", "G"): 0.8,
        ("Q", "H"): 1.9,
        ("Q", "I"): 0.6,
        ("Q", "L"): 3.6,
        ("Q", "K"): 2.9,
        ("Q", "M"): 2.3,
        ("Q", "F"): 1.0,
        ("Q", "P"): 0.9,
        ("Q", "S"): 1.3,
        ("Q", "T"): 1.1,
        ("Q", "W"): 0.8,
        ("Q", "Y"): 0.4,
        ("Q", "V"): 0.5,
        ("E", "-"): 9.9,
        ("E", "A"): 4.0,
        ("E", "R"): 1.8,
        ("E", "N"): 0.6,
        ("E", "D"): 2.5,
        ("E", "C"): 0.4,
        ("E", "Q"): 3.8,
        ("E", "E"): 62.5,
        ("E", "G"): 1.8,
        ("E", "H"): 0.5,
        ("E", "I"): 0.1,
        ("E", "L"): 2.5,
        ("E", "K"): 1.6,
        ("E", "M"): 0.3,
        ("E", "F"): 0.0,
        ("E", "P"): 0.5,
        ("E", "S"): 2.5,
        ("E", "T"): 1.0,
        ("E", "W"): 0.0,
        ("E", "Y"): 1.3,
        ("E", "V"): 2.5,
        ("G", "-"): 3.6,
        ("G", "A"): 3.6,
        ("G", "R"): 0.3,
        ("G", "N"): 0.6,
        ("G", "D"): 0.5,
        ("G", "C"): 0.3,
        ("G", "Q"): 0.4,
        ("G", "E"): 0.4,
        ("G", "G"): 85.7,
        ("G", "H"): 0.2,
        ("G", "I"): 0.1,
        ("G", "L"): 0.3,
        ("G", "K"): 0.5,
        ("G", "M"): 0.1,
        ("G", "F"): 0.1,
        ("G", "P"): 0.2,
        ("G", "S"): 1.7,
        ("G", "T"): 0.7,
        ("G", "W"): 0.2,
        ("G", "Y"): 0.1,
        ("G", "V"): 0.4,
        ("H", "-"): 1.0,
        ("H", "A"): 0.6,
        ("H", "R"): 0.8,
        ("H", "N"): 0.9,
        ("H", "D"): 0.6,
        ("H", "C"): 0.4,
        ("H", "Q"): 1.4,
        ("H", "E"): 0.5,
        ("H", "G"): 0.0,
        ("H", "H"): 84.1,
        ("H", "I"): 0.6,
        ("H", "L"): 1.0,
        ("H", "K"): 0.6,
        ("H", "M"): 0.8,
        ("H", "F"): 2.0,
        ("H", "P"): 0.3,
        ("H", "S"): 0.5,
        ("H", "T"): 0.5,
        ("H", "W"): 0.6,
        ("H", "Y"): 2.6,
        ("H", "V"): 0.2,
        ("I", "-"): 1.8,
        ("I", "A"): 0.4,
        ("I", "R"): 0.3,
        ("I", "N"): 0.3,
        ("I", "D"): 0.2,
        ("I", "C"): 0.4,
        ("I", "Q"): 0.1,
        ("I", "E"): 0.2,
        ("I", "G"): 0.0,
        ("I", "H"): 0.1,
        ("I", "I"): 73.9,
        ("I", "L"): 7.7,
        ("I", "K"): 0.4,
        ("I", "M"): 2.2,
        ("I", "F"): 1.5,
        ("I", "P"): 0.2,
        ("I", "S"): 0.3,
        ("I", "T"): 0.8,
        ("I", "W"): 0.2,
        ("I", "Y"): 0.2,
        ("I", "V"): 8.8,
        ("L", "-"): 1.1,
        ("L", "A"): 0.6,
        ("L", "R"): 0.3,
        ("L", "N"): 0.4,
        ("L", "D"): 0.2,
        ("L", "C"): 0.5,
        ("L", "Q"): 0.9,
        ("L", "E"): 0.6,
        ("L", "G"): 0.2,
        ("L", "H"): 0.2,
        ("L", "I"): 5.0,
        ("L", "L"): 78.6,
        ("L", "K"): 1.1,
        ("L", "M"): 3.0,
        ("L", "F"): 2.4,
        ("L", "P"): 0.3,
        ("L", "S"): 0.1,
        ("L", "T"): 0.4,
        ("L", "W"): 0.2,
        ("L", "Y"): 0.5,
        ("L", "V"): 3.4,
        ("K", "-"): 20.3,
        ("K", "A"): 1.8,
        ("K", "R"): 5.4,
        ("K", "N"): 1.6,
        ("K", "D"): 0.5,
        ("K", "C"): 0.0,
        ("K", "Q"): 1.3,
        ("K", "E"): 1.8,
        ("K", "G"): 1.7,
        ("K", "H"): 1.5,
        ("K", "I"): 1.6,
        ("K", "L"): 4.8,
        ("K", "K"): 46.8,
        ("K", "M"): 1.1,
        ("K", "F"): 0.6,
        ("K", "P"): 0.4,
        ("K", "S"): 1.1,
        ("K", "T"): 2.8,
        ("K", "W"): 0.2,
        ("K", "Y"): 1.1,
        ("K", "V"): 3.7,
        ("M", "-"): 2.2,
        ("M", "A"): 1.2,
        ("M", "R"): 0.2,
        ("M", "N"): 0.3,
        ("M", "D"): 0.2,
        ("M", "C"): 0.7,
        ("M", "Q"): 0.6,
        ("M", "E"): 0.3,
        ("M", "G"): 0.4,
        ("M", "H"): 0.2,
        ("M", "I"): 5.1,
        ("M", "L"): 12.2,
        ("M", "K"): 0.3,
        ("M", "M"): 68.7,
        ("M", "F"): 2.2,
        ("M", "P"): 0.1,
        ("M", "S"): 0.1,
        ("M", "T"): 0.4,
        ("M", "W"): 0.4,
        ("M", "Y"): 0.1,
        ("M", "V"): 4.2,
        ("F", "-"): 1.9,
        ("F", "A"): 0.2,
        ("F", "R"): 0.1,
        ("F", "N"): 0.1,
        ("F", "D"): 0.1,
        ("F", "C"): 0.3,
        ("F", "Q"): 0.2,
        ("F", "E"): 0.0,
        ("F", "G"): 0.4,
        ("F", "H"): 0.5,
        ("F", "I"): 1.5,
        ("F", "L"): 4.7,
        ("F", "K"): 0.1,
        ("F", "M"): 1.0,
        ("F", "F"): 80.2,
        ("F", "P"): 0.1,
        ("F", "S"): 0.5,
        ("F", "T"): 0.4,
        ("F", "W"): 1.6,
        ("F", "Y"): 4.6,
        ("F", "V"): 1.4,
        ("P", "-"): 7.8,
        ("P", "A"): 4.6,
        ("P", "R"): 0.4,
        ("P", "N"): 1.0,
        ("P", "D"): 0.2,
        ("P", "C"): 0.1,
        ("P", "Q"): 1.2,
        ("P", "E"): 0.8,
        ("P", "G"): 0.9,
        ("P", "H"): 0.8,
        ("P", "I"): 1.0,
        ("P", "L"): 1.3,
        ("P", "K"): 0.3,
        ("P", "M"): 0.5,
        ("P", "F"): 0.4,
        ("P", "P"): 73.3,
        ("P", "S"): 2.9,
        ("P", "T"): 1.5,
        ("P", "W"): 0.0,
        ("P", "Y"): 0.0,
        ("P", "V"): 1.0,
        ("S", "-"): 8.8,
        ("S", "A"): 7.1,
        ("S", "R"): 1.1,
        ("S", "N"): 1.5,
        ("S", "D"): 0.8,
        ("S", "C"): 1.6,
        ("S", "Q"): 0.9,
        ("S", "E"): 0.6,
        ("S", "G"): 2.2,
        ("S", "H"): 0.8,
        ("S", "I"): 0.5,
        ("S", "L"): 0.4,
        ("S", "K"): 0.7,
        ("S", "M"): 0.4,
        ("S", "F"): 1.1,
        ("S", "P"): 1.2,
        ("S", "S"): 62.2,
        ("S", "T"): 5.9,
        ("S", "W"): 0.1,
        ("S", "Y"): 0.9,
        ("S", "V"): 1.2,
        ("T", "-"): 3.5,
        ("T", "A"): 3.8,
        ("T", "R"): 0.5,
        ("T", "N"): 0.8,
        ("T", "D"): 0.4,
        ("T", "C"): 0.9,
        ("T", "Q"): 0.8,
        ("T", "E"): 0.9,
        ("T", "G"): 0.6,
        ("T", "H"): 0.3,
        ("T", "I"): 3.1,
        ("T", "L"): 1.7,
        ("T", "K"): 0.9,
        ("T", "M"): 1.7,
        ("T", "F"): 0.2,
        ("T", "P"): 0.6,
        ("T", "S"): 4.5,
        ("T", "T"): 68.2,
        ("T", "W"): 0.0,
        ("T", "Y"): 0.4,
        ("T", "V"): 6.4,
        ("W", "-"): 1.9,
        ("W", "A"): 0.0,
        ("W", "R"): 0.6,
        ("W", "N"): 0.0,
        ("W", "D"): 0.0,
        ("W", "C"): 0.4,
        ("W", "Q"): 0.2,
        ("W", "E"): 0.0,
        ("W", "G"): 0.2,
        ("W", "H"): 0.1,
        ("W", "I"): 0.7,
        ("W", "L"): 0.8,
        ("W", "K"): 0.7,
        ("W", "M"): 0.7,
        ("W", "F"): 1.4,
        ("W", "P"): 0.0,
        ("W", "S"): 0.5,
        ("W", "T"): 0.2,
        ("W", "W"): 89.0,
        ("W", "Y"): 1.7,
        ("W", "V"): 0.8,
        ("Y", "-"): 2.0,
        ("Y", "A"): 0.4,
        ("Y", "R"): 0.4,
        ("Y", "N"): 0.6,
        ("Y", "D"): 0.2,
        ("Y", "C"): 0.6,
        ("Y", "Q"): 0.2,
        ("Y", "E"): 0.3,
        ("Y", "G"): 0.0,
        ("Y", "H"): 1.5,
        ("Y", "I"): 0.9,
        ("Y", "L"): 2.0,
        ("Y", "K"): 0.5,
        ("Y", "M"): 0.5,
        ("Y", "F"): 9.7,
        ("Y", "P"): 0.2,
        ("Y", "S"): 0.2,
        ("Y", "T"): 0.5,
        ("Y", "W"): 0.7,
        ("Y", "Y"): 78.0,
        ("Y", "V"): 0.6,
        ("V", "-"): 1.4,
        ("V", "A"): 2.5,
        ("V", "R"): 0.2,
        ("V", "N"): 0.3,
        ("V", "D"): 0.2,
        ("V", "C"): 0.8,
        ("V", "Q"): 0.1,
        ("V", "E"): 0.6,
        ("V", "G"): 0.1,
        ("V", "H"): 0.1,
        ("V", "I"): 11.1,
        ("V", "L"): 3.8,
        ("V", "K"): 0.9,
        ("V", "M"): 1.0,
        ("V", "F"): 1.2,
        ("V", "P"): 0.3,
        ("V", "S"): 0.5,
        ("V", "T"): 1.6,
        ("V", "W"): 0.2,
        ("V", "Y"): 0.1,
        ("V", "V"): 73.0,
    }

    return (
        asymmetric_koshi_goldstein_all_residues_dict,
        asymmetric_koshi_goldstein_exposed_residues_dict,
        asymmetric_koshi_goldstein_buried_residues_dict,
    )


SNEATH_DICT: dict[tuple[str, str], int]
MIYATA_DICT: dict[tuple[str, str], float]
SYMMETRIC_EPSTEIN_DICT: dict[tuple[str, str], float]
ASYMMETRIC_EPSTEIN_DICT: dict[tuple[str, str], float]
SYMMETRIC_EXPERIMENTAL_EXCHANGEABILITY_DICT: dict[tuple[str, str], float]
ASYMMETRIC_EXPERIMENTAL_EXCHANGEABILITY_DICT: dict[tuple[str, str], int | None]
GRANTHAM_DICT: dict[tuple[str, str], int]
SYMMETRIC_KOSHI_GOLDSTEIN_ALL_RESIDUES_DICT: dict[tuple[str, str], float]
ASYMMETRIC_KOSHI_GOLDSTEIN_ALL_RESIDUES_DICT: dict[tuple[str, str], float]
SYMMETRIC_KOSHI_GOLDSTEIN_EXPOSED_RESIDUES_DICT: dict[tuple[str, str], float]
ASYMMETRIC_KOSHI_GOLDSTEIN_EXPOSED_RESIDUES_DICT: dict[tuple[str, str], float]
SYMMETRIC_KOSHI_GOLDSTEIN_BURIED_RESIDUES_DICT: dict[tuple[str, str], float]
ASYMMETRIC_KOSHI_GOLDSTEIN_BURIED_RESIDUES_DICT: dict[tuple[str, str], float]


SNEATH_DICT                                  = make_sneath_dict()
MIYATA_DICT                                  = make_miyata_dict()
SYMMETRIC_EPSTEIN_DICT                       = make_symmetric_epstein_dict()
ASYMMETRIC_EPSTEIN_DICT                      = make_asymmetric_epstein_dict()
SYMMETRIC_EXPERIMENTAL_EXCHANGEABILITY_DICT  = make_symmetric_experimental_exchangeability_dict()
ASYMMETRIC_EXPERIMENTAL_EXCHANGEABILITY_DICT = make_asymmetric_experimental_exchangeability_dict()
GRANTHAM_DICT                                = make_grantham_dict()
# Koshi-Goldstein dicts
(
    SYMMETRIC_KOSHI_GOLDSTEIN_ALL_RESIDUES_DICT,
    SYMMETRIC_KOSHI_GOLDSTEIN_EXPOSED_RESIDUES_DICT,
    SYMMETRIC_KOSHI_GOLDSTEIN_BURIED_RESIDUES_DICT,
) = make_symmetric_koshi_golstein_dicts()
(
    ASYMMETRIC_KOSHI_GOLDSTEIN_ALL_RESIDUES_DICT,
    ASYMMETRIC_KOSHI_GOLDSTEIN_EXPOSED_RESIDUES_DICT,
    ASYMMETRIC_KOSHI_GOLDSTEIN_BURIED_RESIDUES_DICT,
) = make_asymmetric_koshi_golstein_dicts()
