# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['nautobot_circuit_maintenance',
 'nautobot_circuit_maintenance.api',
 'nautobot_circuit_maintenance.graphql',
 'nautobot_circuit_maintenance.handle_notifications',
 'nautobot_circuit_maintenance.migrations',
 'nautobot_circuit_maintenance.tests']

package_data = \
{'': ['*'],
 'nautobot_circuit_maintenance': ['fixtures/*',
                                  'templates/nautobot_circuit_maintenance/*']}

install_requires = \
['bs4>=0.0.1,<0.0.2',
 'circuit-maintenance-parser>=2.0.0,<3.0.0',
 'google-api-python-client>=2.9.0,<3.0.0',
 'google-auth-httplib2>=0.1.0,<0.2.0',
 'google-auth-oauthlib>=0.4.4,<0.5.0',
 'google-oauth>=1.0.0,<2.0.0',
 'nautobot',
 'pydantic>=1.8.1,<2.0.0']

extras_require = \
{'metrics': ['nautobot-capacity-metrics']}

setup_kwargs = {
    'name': 'nautobot-circuit-maintenance',
    'version': '0.2.5',
    'description': 'Nautobot plugin to automatically handle Circuit Maintenances Notifications',
    'long_description': '# Nautobot Circuit Maintenance plugin\n\nA plugin for [Nautobot](https://github.com/nautobot/nautobot) to easily handle Circuit Maintenances related to Nautobot Circuits.\n\n`nautobot-circuit-maintenance` lets you handle maintenances for your Circuits based on notifications received by email by leveraging on [circuit-maintenance-parser](https://github.com/networktocode/circuit-maintenance-parser), a notifications parser library for multiple network service providers that exposes structured data following a recommendation defined in this [draft NANOG BCOP](https://github.com/jda/maintnote-std/blob/master/standard.md).\n\n## Installation\n\nThe plugin is available as a Python package in pypi and can be installed with pip\n\n```shell\npip install nautobot-circuit-maintenance\n```\n\n> The plugin is compatible with Nautobot 1.0.0b4 and higher\n\nTo ensure Circuit Maintenance is automatically re-installed during future upgrades, create a file named `local_requirements.txt` (if not already existing) in the Nautobot root directory (alongside `requirements.txt`) and list the `nautobot-circuit-maintenance` package:\n\n```no-highlight\n# echo nautobot-circuit-maintenance >> local_requirements.txt\n```\n\nOnce installed, the plugin needs to be enabled in your `configuration.py`.\n\n```python\n# In your configuration.py\nPLUGINS = ["nautobot_circuit_maintenance"]\n```\n\n```py\nPLUGINS_CONFIG = {\n    "nautobot_circuit_maintenance": {\n        "raw_notifications": {\n            "initial_days_since": 100,\n            "raw_notification_size": 500,\n        },\n        "notification_sources": [\n            {\n              ...\n            }\n        ]\n    }\n}\n```\n\nIn the `raw_notifications` section, you can define:\n\n- `initial_days_since`: define how many days back the plugin will check for `RawNotification`s for each\n  `NotificationSource`, in order to limit the number of notifications to be processed on the first run of the plugin.\n  In subsequent runs, the last notification date will be used as the reference to limit. If not defined, it defaults to **365 days**.\n- `raw_notification_size`: define how many bytes from a notification will be stored in the database to not store too big objects. If not defined, it defaults to **1000** bytes.\n\nThe `notification_sources` have custom definition depending on the `Source` type, and are defined in the [Usage](#Usage) section.\n\n## Usage\n\n### 1. Define source emails per Provider\n\nIn the Nautobot UI, under **Circuits -> Providers**, for each Provider that we would like to track via the Circuit Maintenance plugin, we **must** configure at least one email source address (or a comma-separated list of addresses) in the **`Custom Fields -> Emails for Circuit Maintenance plugin** field.\n\nThese are the source email addresses that the plugin will detect and will use to classify each notification for each specific provider.\n\nAlso, by default, the Provider **slug** is used to match the provider parser from the `circuit-maintenance-parser` library, but if a custom mapping is desired (i.e. CentruryLink to Lumen), you can define this custom mapping in the **`Custom Fields -> Provider Parser for Circuit Maintenance plugin** field.\n\n### 2. Configure Notification Sources\n\nNotification Sources are defined in two steps:\n\n#### 2.1 Define Notification Sources in `nautobot_config.py`\n\nIn `nautobot_config.py`, in the `PLUGINS_CONFIG`, under the `nautobot_circuit_maintenance` key, we should define the Notification Sources that will be able later in the UI, where you will be able to **validate** if the authentication credentials provided are working fine or not.\n\nThere are two mandatory attributes (other keys are dependent on the integration type, and will be documented below):\n\n- `name`: Name to identify the Source and will be available in the UI.\n- `url`: URL to reach the Notification Source (i.e. `imap://imap.gmail.com:993`)\n\nThere is also one optional attribute:\n\n- `attach_all_providers`: Flag that enables auto linking of newly created `Providers` to this Notification Source.\n\n> Currently, only IMAP and HTTPS (accounts.google.com) integrations are supported as URL scheme\n\n##### 2.1.1 IMAP\n\nThere are 2 extra required attributes:\n\n- `account`: Identifier (i.e. email address) to use to authenticate.\n- `secret`: Password to IMAP authentication.\n\n> Gmail example: [How to setup Gmail with App Passwords](https://support.google.com/accounts/answer/185833)\n\nThere is also one optional attribute:\n\n- `source_header`: Specify a particular email header to use to identify the source of a particular notification and assign it to the appropriate provider. If unset, `From` will be used, but if your emails are not received directly from the provider but instead pass through a mailing list or alias, you might need to set this to a different value such as `X-Original-Sender` instead.\n\n```py\nPLUGINS_CONFIG = {\n    "nautobot_circuit_maintenance": {\n        "notification_sources": [\n            {\n                "name": "my custom name",\n                "account": os.getenv("CM_NS_1_ACCOUNT", ""),\n                "secret": os.getenv("CM_NS_1_SECRET", ""),\n                "url": os.getenv("CM_NS_1_URL", ""),\n                "source_header": os.getenv("CM_NS_1_SOURCE_HEADER", "From"),  # optional\n                "attach_all_providers": True,  # optional\n            }\n        ]\n    }\n}\n```\n\n##### 2.1.2 Gmail API integrations\n\nThere are 2 extra required attributes:\n\n- `account`: Identifier (i.e. email address) to access via OAuth or to impersonate as service account.\n- `credentials_file`: JSON file containing all the necessary data to identify the API integration (see below).\n\nThere are also three optional attributes:\n\n- `source_header`: Specify a particular email header to use to identify the source of a particular notification and assign it to the appropriate provider. If unset, `From` will be used, but if your emails are not received directly from the provider but instead pass through a mailing list or alias, you might need to set this to a different value such as `X-Original-Sender` instead.\n- `limit_emails_with_not_header_from`: List of emails used to restrict the emails retrieved when NOT using the `source_header` "From" and we can\'t use the `Provider` original emails to filter.\n- `extra_scopes`: Specify a list of additional Google OAuth2 scopes to request access to in addition to GMail API access.\n\n```py\nPLUGINS_CONFIG = {\n    "nautobot_circuit_maintenance": {\n        "notification_sources": [\n            {\n                "name": "my custom name",\n                "account": os.getenv("CM_NS_1_ACCOUNT", ""),\n                "credentials_file": os.getenv("CM_NS_1_CREDENTIALS_FILE", ""),\n                "url": os.getenv("CM_NS_1_URL", ""),\n                "source_header": os.getenv("CM_NS_1_SOURCE_HEADER", "From"),          # optional\n                "limit_emails_with_not_header_from": ["email@example.com"],                  # optional\n                "extra_scopes": ["https://www.googleapis.com/auth/calendar.events"],  # optional\n            }\n        ]\n    }\n}\n```\n\nTo enable Gmail API access, there are some common steps for both Service Account and OAuth authentication:\n\n1. Create a **New Project** in [Google Cloud Console](https://console.cloud.google.com/).\n2. Under **APIs and Services**, enable **Gmail API** for the selected project.\n\n###### 2.1.2.1 Service Account\n\nTo create a [Service Account](https://support.google.com/a/answer/7378726?hl=en) integration:\n\n3. Still under **APIs and Services**, in **Credentials**, create a new **Service Account** and save the credentials file generated to be used when configuring Nautobot Sources.\n4. With Admin rights, edit the newly created Service Account and expand the **Show Domain-Wide Delegation** section. Enable Google Workspace Domain-wide Delegation and save the changes. Copy the Client ID shown.\n5. With Super Admin rights, open the [Google Workspace admin console](https://admin.google.com). Navigate to **Security**, **API controls**, and select the **Manage Domain Wide Delegation** at the bottom of the page.\n6. Add a new API client and paste in the Client ID copied earlier. In the **OAuth scopes** field add the scopes `https://www.googleapis.com/auth/gmail.readonly` and `https://mail.google.com/`. Save the new client configuration by clicking _Authorize_.\n\n###### 2.1.2.2 OAuth\n\nTo create a [OAuth 2.0](https://developers.google.com/identity/protocols/oauth2/web-server) integration:\n\n3. Still under **APIs and Services**, in **Credentials**, create a new **OAuth client ID** selecting the **Web application** application type.\n4. Under **Authorized redirect URIs** add the location where your Nautobot server is listening plus `plugins/circuit-maintenance/source/google_oauth2callback/`. For instance: `http://localhost:8080/plugins/circuit-maintenance/source/google_oauth2callback/`\n5. Save the credentials file generated to be used when configuring Nautobot Sources.\n\n> For OAuth integration, it\'s recommendable that, at least the first time, you run a manual **Validate** of the Notification Source to complete the OAuth authentication workflow, identifying your Google credentials.\n\n> Typically the `url` setting to configure in your `nautobot_config.py` for use with OAuth integration will be `"https://accounts.google.com/o/oauth2/auth"`.\n\n#### 2.2 Add `Providers` to the Notification Sources\n\nIn the Circuit Maintenance plugin UI section, there is a **Notification Sources** button (yellow) where you can configure the Notification Sources to track new circuit maintenance notifications from specific providers.\n\nBecause the Notification Sources are defined by the configuration, you can only view and edit `providers`, but not `add` or `delete` new Notification Sources via UI or API.\n\n> Note that for emails from a given Provider to be processed, you must _both_ define a source email address(es) for that Provider (Usage section 1, above) _and_ add that provider to a specific Notification Source as described in this section.\n\n### 3. Run Handle Notifications Job\n\nThere is an asynchronous task defined as a **Nautobot Job**, **Handle Circuit Maintenance Notifications** that will connect to the emails sources defined under the Notification Sources section (step above), and will fetch new notifications received since the last notification was fetched.\nEach notification will be parsed using the [circuit-maintenance-parser](https://github.com/networktocode/circuit-maintenance-parser) library, and if a valid parsing is executed, a new **Circuit Maintenance** will be created, or if it was already created, it will updated with the new data.\n\nSo, for each email notification received, several objects will be created:\n\n#### 3.1 Notification\n\nEach notification received will create a related object, containing the raw data received, and linking to the corresponding **parsed notification** in case the [circuit-maintenance-parser](https://github.com/networktocode/circuit-maintenance-parser) was able to parse it correctly.\n\n#### 3.2 Parsed Notification\n\nWhen a notification was successfully parsed, it will create a **parsed notification** object, that will contain the structured output from the parser library , following the recommendation defined in [draft NANOG BCOP](https://github.com/jda/maintnote-std/blob/master/standard.md), and a link to the related **Circuit Maintenance** object created.\n\n#### 3.3 Circuit Maintenance\n\nThe **Circuit Maintenance** is where all the necessary information related to a Circuit maintenance is tracked, and reuses most of the data model defined in the parser library.\n\nAttributes:\n\n- Name: name or identifier of the maintenance.\n- Description: description of the maintenance.\n- Status: current state of the maintenance.\n- Start time: defined start time of the maintenance work.\n- End time: defined end time of the maintenance work.\n- Ack: boolean to show if the maintenance has been properly handled by the operator.\n- Circuits: list of circuits and its specific impact linked to this maintenance.\n- Notes: list of internal notes linked to this maintenance.\n- Notifications: list of all the parsed notifications that have been processed for this maintenance.\n\n<p align="center">\n<img src="https://raw.githubusercontent.com/nautobot/nautobot-plugin-circuit-maintenance/develop/docs/images/circuit_maintenance.png" width="800" class="center">\n</p>\n\n### Metrics\n\nLeveraging the `nautobot-capacity-metrics` plugin, the `nautobot-circuit-maintenance` plugin can expose application metrics at `/api/plugins/capacity-metrics/app-metrics` if desired.\n\nCurrent exposed metric is the `circuit operational status` which shows the operational status for each `Circuit`(attached to a `CircuitTermination`) depending on related Circuit Maintenances (1: Operational, 2: Under active maintenance):\n\n```\n# HELP circuit_maintenance_status Circuit operational status\n# TYPE circuit_maintenance_status gauge\ncircuit_maintenance_status{circuit="1111111",circuit_type="Transit",provider="ntt",site="Barcelona"} 2.0\ncircuit_maintenance_status{circuit="2222222",circuit_type="Peering",provider="colt",site="Girona"} 1.0\n```\n\nMetric generation is **disabled** by default, to **enable** them, add a `enable: True` in the `nautobot_circuit_maintenance.metrics` dict. (Of course you must also install the `nautobot_capacity_metrics` plugin and ensure that it is included in `PLUGINS` as a prerequisite to enabling this feature.)\n\nBy default, each circuit has attached some labels and values (cid, provider, type and site), but these labels can be defined in the Plugin configuration by adding an optional dictionary (under "metrics" -> "labels_attached") with the label name and the attributes within the Circuit object. (Note: in case of a value that can be multiple values, such as `terminations`, the first defined one will be used)\n\n```\nPLUGINS_CONFIG = {\n    "nautobot_circuit_maintenance": {\n        ...\n        "metrics": {\n            "enable": True,\n            "labels_attached": {\n                "circuit": "circuit.cid",\n                "provider": "circuit.provider.name",\n                "circuit_type": "circuit.type.name",\n                "site": "site.name",\n            }\n        },\n    }\n}\n```\n\n### Rest API\n\nThe plugin includes 6 API endpoints to manage its related objects, complete info in the Swagger section.\n\n- Circuit Maintenance: `/api/plugins\u200b/circuit-maintenance\u200b/maintenance`\n- Circuit Impact: `/api/plugins\u200b/circuit-maintenance\u200b/circuitimpact`\n- Note: `/api/plugins\u200b/circuit-maintenance\u200b/note`\n\n### GraphQL API\n\nCircuit Maintenance and Circuit Impact objects are available for GraphQL queries.\n\n## Contributing\n\nPull requests are welcomed and automatically built and tested against multiple version of Python and multiple version of Nautobot through TravisCI.\n\nThe project is packaged with a light development environment based on `docker-compose` to help with the local development of the project and to run the tests within TravisCI.\n\nThe project is following Network to Code software development guideline and is leveraging:\n\n- Black, Pylint, Bandit and pydocstyle for Python linting and formatting.\n- Django unit test to ensure the plugin is working properly.\n\n### CLI Helper Commands\n\nThe project is coming with a CLI helper based on [invoke](http://www.pyinvoke.org/) to help setup the development environment. The commands are listed below in 3 categories `dev environment`, `utility` and `testing`.\n\nEach command can be executed with `invoke <command>`. All commands support the arguments `--nautobot-ver` and `--python-ver` if you want to manually define the version of Python and Nautobot to use. Each command also has its own help `invoke <command> --help`\n\n#### Local dev environment\n\n```no-highlight\n  build            Build all docker images.\n  debug            Start Nautobot and its dependencies in debug mode.\n  destroy          Destroy all containers and volumes.\n  restart          Restart Nautobot and its dependencies.\n  start            Start Nautobot and its dependencies in detached mode.\n  stop             Stop Nautobot and its dependencies.\n```\n\n#### Utility\n\n```no-highlight\n  cli              Launch a bash shell inside the running Nautobot container.\n  create-user      Create a new user in django (default: admin), will prompt for password.\n  makemigrations   Run Make Migration in Django.\n  nbshell          Launch a nbshell session.\n```\n\n#### Testing\n\n```no-highlight\n  bandit           Run bandit to validate basic static code security analysis.\n  black            Run black to check that Python files adhere to its style standards.\n  flake8           This will run flake8 for the specified name and Python version.\n  pydocstyle       Run pydocstyle to validate docstring formatting adheres to NTC defined standards.\n  pylint           Run pylint code analysis.\n  tests            Run all tests for this plugin.\n  unittest         Run Django unit tests for the plugin.\n```\n\n## Questions\n\nFor any questions or comments, please check the [FAQ](FAQ.md) first and feel free to swing by the [Network to Code slack channel](https://networktocode.slack.com/) (channel #nautobot).\nSign up [here](http://slack.networktocode.com/)\n',
    'author': 'Network to Code, LLC',
    'author_email': 'opensource@networktocode.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/nautobot/nautobot-plugin-circuit-maintenance',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
