"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deserialize = exports.serialize = void 0;
const yaml = require("yaml");
const yaml_types = require("yaml/types");
/**
 * Serializes the given data structure into valid YAML.
 *
 * @param obj the data structure to serialize
 * @returns a string containing the YAML representation of {@param obj}
 */
function serialize(obj) {
    const oldFold = yaml_types.strOptions.fold.lineWidth;
    try {
        yaml_types.strOptions.fold.lineWidth = 0;
        return yaml.stringify(obj, { schema: 'yaml-1.1' });
    }
    finally {
        yaml_types.strOptions.fold.lineWidth = oldFold;
    }
}
exports.serialize = serialize;
/**
 * Deserialize the YAML into the appropriate data structure.
 *
 * @param str the string containing YAML
 * @returns the data structure the YAML represents
 *   (most often in case of CloudFormation, an object)
 */
function deserialize(str) {
    return parseYamlStrWithCfnTags(str);
}
exports.deserialize = deserialize;
function makeTagForCfnIntrinsic(intrinsicName, addFnPrefix = true, resolveFun) {
    return {
        identify(value) { return typeof value === 'string'; },
        tag: `!${intrinsicName}`,
        resolve: resolveFun || ((_doc, cstNode) => {
            const ret = {};
            ret[addFnPrefix ? `Fn::${intrinsicName}` : intrinsicName] =
                // the +1 is to account for the ! the short form begins with
                parseYamlStrWithCfnTags(cstNode.toString().substring(intrinsicName.length + 1));
            return ret;
        }),
    };
}
const shortForms = [
    'Base64', 'Cidr', 'FindInMap', 'GetAZs', 'ImportValue', 'Join', 'Sub',
    'Select', 'Split', 'Transform', 'And', 'Equals', 'If', 'Not', 'Or',
].map(name => makeTagForCfnIntrinsic(name)).concat(makeTagForCfnIntrinsic('Ref', false), makeTagForCfnIntrinsic('Condition', false), makeTagForCfnIntrinsic('GetAtt', true, (_doc, cstNode) => {
    const parsedArguments = parseYamlStrWithCfnTags(cstNode.toString().substring('!GetAtt'.length));
    let value;
    if (typeof parsedArguments === 'string') {
        // if the arguments to !GetAtt are a string,
        // the part before the first '.' is the logical ID,
        // and the rest is the attribute name
        // (which can contain '.')
        const firstDot = parsedArguments.indexOf('.');
        if (firstDot === -1) {
            throw new Error(`Short-form Fn::GetAtt must contain a '.' in its string argument, got: '${parsedArguments}'`);
        }
        value = [
            parsedArguments.substring(0, firstDot),
            parsedArguments.substring(firstDot + 1),
        ];
    }
    else {
        // this is the form where the arguments to Fn::GetAtt are already an array -
        // in this case, nothing more to do
        value = parsedArguments;
    }
    return { 'Fn::GetAtt': value };
}));
function parseYamlStrWithCfnTags(text) {
    return yaml.parse(text, {
        customTags: shortForms,
        schema: 'yaml-1.1',
    });
}
//# sourceMappingURL=data:application/json;base64,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