"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FactName = exports.Fact = void 0;
/**
 * A database of regional information.
 */
class Fact {
    constructor() {
        throw new Error('Use the static methods of Fact instead!');
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     *          may not be an exhaustive list of all available AWS regions.
     */
    static get regions() {
        return Object.keys(this.database);
    }
    /**
     * Retrieves a fact from this Fact database.
     *
     * @param region the name of the region (e.g: `us-east-1`)
     * @param name   the name of the fact being looked up (see the `FactName` class for details)
     *
     * @returns the fact value if it is known, and `undefined` otherwise.
     */
    static find(region, name) {
        const regionFacts = this.database[region];
        return regionFacts && regionFacts[name];
    }
    /**
     * Retrieve a fact from the Fact database. (retrieval will fail if the specified region or
     * fact name does not exist.)
     *
     * @param region the name of the region (e.g: `us-east-1`)
     * @param name the name of the fact being looked up (see the `FactName` class for details)
     */
    static requireFact(region, name) {
        const foundFact = this.find(region, name);
        if (!foundFact) {
            throw new Error(`No fact ${name} could be found for region: ${region} and name: ${name}`);
        }
        return foundFact;
    }
    /**
     * Registers a new fact in this Fact database.
     *
     * @param fact           the new fact to be registered.
     * @param allowReplacing whether new facts can replace existing facts or not.
     */
    static register(fact, allowReplacing = false) {
        const regionFacts = this.database[fact.region] || (this.database[fact.region] = {});
        if (fact.name in regionFacts && regionFacts[fact.name] !== fact.value && !allowReplacing) {
            throw new Error(`Region ${fact.region} already has a fact ${fact.name}, with value ${regionFacts[fact.name]}`);
        }
        if (fact.value !== undefined) {
            regionFacts[fact.name] = fact.value;
        }
    }
    /**
     * Removes a fact from the database.
     * @param region the region for which the fact is to be removed.
     * @param name   the name of the fact to remove.
     * @param value  the value that should be removed (removal will fail if the value is specified, but does not match the
     *               current stored value).
     */
    static unregister(region, name, value) {
        const regionFacts = this.database[region] || {};
        if (name in regionFacts && value && regionFacts[name] !== value) {
            throw new Error(`Attempted to remove ${name} from ${region} with value ${value}, but the fact's value is ${regionFacts[name]}`);
        }
        delete regionFacts[name];
    }
}
exports.Fact = Fact;
Fact.database = {};
/**
 * All standardized fact names.
 */
class FactName {
    /**
     * The name of the regional service principal for a given service.
     *
     * @param service the service name, either simple (e.g: `s3`, `codedeploy`) or qualified (e.g: `s3.amazonaws.com`).
     *                The `.amazonaws.com` and `.amazonaws.com.cn` domains are stripped from service names, so they are
     *                canonicalized in that respect.
     */
    static servicePrincipal(service) {
        return `service-principal:${service.replace(/\.amazonaws\.com(\.cn)?$/, '')}`;
    }
}
exports.FactName = FactName;
/**
 * The name of the partition for a region (e.g: 'aws', 'aws-cn', ...)
 */
FactName.PARTITION = 'partition';
/**
 * The domain suffix for a region (e.g: 'amazonaws.com`)
 */
FactName.DOMAIN_SUFFIX = 'domainSuffix';
/**
 * Whether the AWS::CDK::Metadata CloudFormation Resource is available in-region or not. The value is a boolean
 * modelled as `YES` or `NO`.
 */
FactName.CDK_METADATA_RESOURCE_AVAILABLE = 'cdk:metadata-resource:available';
/**
 * The endpoint used for hosting S3 static websites
 */
FactName.S3_STATIC_WEBSITE_ENDPOINT = 's3-static-website:endpoint';
/**
 * The endpoint used for aliasing S3 static websites in Route 53
 */
FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID = 's3-static-website:route-53-hosted-zone-id';
/**
 * The prefix for VPC Endpoint Service names,
 * cn.com.amazonaws.vpce for China regions,
 * com.amazonaws.vpce otherwise.
 */
FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'vpcEndpointServiceNamePrefix';
/**
 * The account for ELBv2 in this region
 */
FactName.ELBV2_ACCOUNT = 'elbv2Account';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the
 * AWS Deep Learning Containers images in a given region.
 */
FactName.DLC_REPOSITORY_ACCOUNT = 'dlcRepositoryAccount';
//# sourceMappingURL=data:application/json;base64,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