"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkPipeline = void 0;
const path = require("path");
const codepipeline = require("../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const actions_1 = require("./actions");
const construct_internals_1 = require("./private/construct-internals");
const stage_1 = require("./stage");
/**
 * A Pipeline to deploy CDK apps
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 */
class CdkPipeline extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._stages = [];
        this._outputArtifacts = {};
        if (!core_1.App.isApp(this.node.root)) {
            throw new Error('CdkPipeline must be created under an App');
        }
        this._cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        const pipelineStack = core_1.Stack.of(this);
        if (props.codePipeline) {
            if (props.pipelineName) {
                throw new Error('Cannot set \'pipelineName\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            this._pipeline = props.codePipeline;
        }
        else {
            this._pipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: props.pipelineName,
                restartExecutionOnUpdate: true,
            });
        }
        if (props.sourceAction && !props.synthAction) {
            // Because of ordering limitations, you can: bring your own Source, bring your own
            // Both, or bring your own Nothing. You cannot bring your own Build (which because of the
            // current CodePipeline API must go BEFORE what we're adding) and then having us add a
            // Source after it. That doesn't make any sense.
            throw new Error('When passing a \'sourceAction\' you must also pass a \'synthAction\' (or a \'codePipeline\' that already has both)');
        }
        if (!props.sourceAction && (!props.codePipeline || props.codePipeline.stages.length < 1)) {
            throw new Error('You must pass a \'sourceAction\' (or a \'codePipeline\' that already has a Source stage)');
        }
        if (props.sourceAction) {
            this._pipeline.addStage({
                stageName: 'Source',
                actions: [props.sourceAction],
            });
        }
        if (props.synthAction) {
            this._pipeline.addStage({
                stageName: 'Build',
                actions: [props.synthAction],
            });
        }
        this._pipeline.addStage({
            stageName: 'UpdatePipeline',
            actions: [new actions_1.UpdatePipelineAction(this, 'UpdatePipeline', {
                    cloudAssemblyInput: this._cloudAssemblyArtifact,
                    pipelineStackName: pipelineStack.stackName,
                    cdkCliVersion: props.cdkCliVersion,
                    projectName: maybeSuffix(props.pipelineName, '-selfupdate'),
                })],
        });
        this._assets = new AssetPublishing(this, 'Assets', {
            cloudAssemblyInput: this._cloudAssemblyArtifact,
            cdkCliVersion: props.cdkCliVersion,
            pipeline: this._pipeline,
            projectName: maybeSuffix(props.pipelineName, '-publish'),
        });
        core_1.Aspects.of(this).add({ visit: () => this._assets.removeAssetsStageIfEmpty() });
    }
    /**
     * The underlying CodePipeline object
     *
     * You can use this to add more Stages to the pipeline, or Actions
     * to Stages.
     */
    get codePipeline() {
        return this._pipeline;
    }
    /**
     * Access one of the pipeline's stages by stage name
     *
     * You can use this to add more Actions to a stage.
     */
    stage(stageName) {
        return this._pipeline.stage(stageName);
    }
    /**
     * Add pipeline stage that will deploy the given application stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplicationStage(appStage, options = {}) {
        const stage = this.addStage(appStage.stageName);
        stage.addApplication(appStage, options);
        return stage;
    }
    /**
     * Add a new, empty stage to the pipeline
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     */
    addStage(stageName) {
        const pipelineStage = this._pipeline.addStage({
            stageName,
        });
        const stage = new stage_1.CdkStage(this, stageName, {
            cloudAssemblyArtifact: this._cloudAssemblyArtifact,
            pipelineStage,
            stageName,
            host: {
                publishAsset: this._assets.addPublishAssetAction.bind(this._assets),
                stackOutputArtifact: (artifactId) => this._outputArtifacts[artifactId],
            },
        });
        this._stages.push(stage);
        return stage;
    }
    /**
     * Get the StackOutput object that holds this CfnOutput's value in this pipeline
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     */
    stackOutput(cfnOutput) {
        const stack = core_1.Stack.of(cfnOutput);
        if (!this._outputArtifacts[stack.artifactId]) {
            // We should have stored the ArtifactPath in the map, but its Artifact
            // property isn't publicly readable...
            this._outputArtifacts[stack.artifactId] = new codepipeline.Artifact(`Artifact_${stack.artifactId}_Outputs`);
        }
        return new stage_1.StackOutput(this._outputArtifacts[stack.artifactId].atPath('outputs.json'), cfnOutput.logicalId);
    }
    /**
     * Validate that we don't have any stacks violating dependency order in the pipeline
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     */
    validate() {
        const ret = new Array();
        ret.push(...this.validateDeployOrder());
        ret.push(...this.validateRequestedOutputs());
        return ret;
    }
    /**
     * Return all StackDeployActions in an ordered list
     */
    get stackActions() {
        return flatMap(this._pipeline.stages, s => s.actions.filter(isDeployAction));
    }
    *validateDeployOrder() {
        const stackActions = this.stackActions;
        for (const stackAction of stackActions) {
            // For every dependency, it must be executed in an action before this one is prepared.
            for (const depId of stackAction.dependencyStackArtifactIds) {
                const depAction = stackActions.find(s => s.stackArtifactId === depId);
                if (depAction === undefined) {
                    core_1.Annotations.of(this).addWarning(`Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depId}', but that dependency is not deployed through the pipeline!`);
                }
                else if (!(depAction.executeRunOrder < stackAction.prepareRunOrder)) {
                    yield `Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depAction.stackName}', but is deployed before it in the pipeline!`;
                }
            }
        }
    }
    *validateRequestedOutputs() {
        const artifactIds = this.stackActions.map(s => s.stackArtifactId);
        for (const artifactId of Object.keys(this._outputArtifacts)) {
            if (!artifactIds.includes(artifactId)) {
                yield `Trying to use outputs for Stack '${artifactId}', but Stack is not deployed in this pipeline. Add it to the pipeline.`;
            }
        }
    }
}
exports.CdkPipeline = CdkPipeline;
function isDeployAction(a) {
    return a instanceof actions_1.DeployCdkStackAction;
}
function flatMap(xs, f) {
    return Array.prototype.concat([], ...xs.map(f));
}
/**
 * Add appropriate publishing actions to the asset publishing stage
 */
class AssetPublishing extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.publishers = {};
        this.assetRoles = {};
        this._fileAssetCtr = 1;
        this._dockerAssetCtr = 1;
        this.myCxAsmRoot = path.resolve(construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(this)).outdir);
        // We MUST add the Stage immediately here, otherwise it will be in the wrong place
        // in the pipeline!
        this.stage = this.props.pipeline.addStage({ stageName: 'Assets' });
        this.pipeline = this.props.pipeline;
    }
    /**
     * Make sure there is an action in the stage to publish the given asset
     *
     * Assets are grouped by asset ID (which represent individual assets) so all assets
     * are published in parallel. For each assets, all destinations are published sequentially
     * so that we can reuse expensive operations between them (mostly: building a Docker image).
     */
    addPublishAssetAction(command) {
        // FIXME: this is silly, we need the relative path here but no easy way to get it
        const relativePath = path.relative(this.myCxAsmRoot, command.assetManifestPath);
        // The path cannot be outside the asm root. I don't really understand how this could ever
        // come to pass, but apparently it has (see https://github.com/aws/aws-cdk/issues/9766).
        // Add a sanity check here so we can catch it more quickly next time.
        if (relativePath.startsWith(`..${path.sep}`)) {
            throw new Error(`The asset manifest (${command.assetManifestPath}) cannot be outside the Cloud Assembly directory (${this.myCxAsmRoot}). Please report this error at https://github.com/aws/aws-cdk/issues to help us debug why this is happening.`);
        }
        // Late-binding here (rather than in the constructor) to prevent creating the role in cases where no asset actions are created.
        if (!this.assetRoles[command.assetType]) {
            this.generateAssetRole(command.assetType);
        }
        let action = this.publishers[command.assetId];
        if (!action) {
            // The asset ID would be a logical candidate for the construct path and project names, but if the asset
            // changes it leads to recreation of a number of Role/Policy/Project resources which is slower than
            // necessary. Number sequentially instead.
            //
            // FIXME: The ultimate best solution is probably to generate a single Project per asset type
            // and reuse that for all assets.
            const id = command.assetType === actions_1.AssetType.FILE ? `FileAsset${this._fileAssetCtr++}` : `DockerAsset${this._dockerAssetCtr++}`;
            // NOTE: It's important that asset changes don't force a pipeline self-mutation.
            // This can cause an infinite loop of updates (see https://github.com/aws/aws-cdk/issues/9080).
            // For that reason, we use the id as the actionName below, rather than the asset hash.
            action = this.publishers[command.assetId] = new actions_1.PublishAssetsAction(this, id, {
                actionName: id,
                cloudAssemblyInput: this.props.cloudAssemblyInput,
                cdkCliVersion: this.props.cdkCliVersion,
                assetType: command.assetType,
                role: this.assetRoles[command.assetType],
            });
            this.stage.addAction(action);
        }
        action.addPublishCommand(relativePath, command.assetSelector);
    }
    /**
     * Remove the Assets stage if it turns out we didn't add any Assets to publish
     */
    removeAssetsStageIfEmpty() {
        if (Object.keys(this.publishers).length === 0) {
            // Hacks to get access to innards of Pipeline
            // Modify 'stages' array in-place to remove Assets stage if empty
            const stages = this.props.pipeline._stages;
            const ix = stages.indexOf(this.stage);
            if (ix > -1) {
                stages.splice(ix, 1);
            }
        }
    }
    /**
     * This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
     * roles into one, and re-using across all assets, saves significant size of the final synthesized output.
     * Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
     * Generates one role per asset type to separate file and Docker/image-based permissions.
     */
    generateAssetRole(assetType) {
        if (this.assetRoles[assetType]) {
            return this.assetRoles[assetType];
        }
        const rolePrefix = assetType === actions_1.AssetType.DOCKER_IMAGE ? 'Docker' : 'File';
        const assetRole = new iam.Role(this, `${rolePrefix}Role`, {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codebuild.amazonaws.com'), new iam.AccountPrincipal(core_1.Stack.of(this).account)),
        });
        // Logging permissions
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: '/aws/codebuild/*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: [logGroupArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        }));
        // CodeBuild report groups
        const codeBuildArn = core_1.Stack.of(this).formatArn({
            service: 'codebuild',
            resource: 'report-group',
            resourceName: '*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'codebuild:CreateReportGroup',
                'codebuild:CreateReport',
                'codebuild:UpdateReport',
                'codebuild:BatchPutTestCases',
            ],
            resources: [codeBuildArn],
        }));
        // CodeBuild start/stop
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        // Publishing role access
        const rolePattern = assetType === actions_1.AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        // Artifact access
        this.pipeline.artifactBucket.grantRead(assetRole);
        this.assetRoles[assetType] = assetRole.withoutPolicyUpdates();
        return this.assetRoles[assetType];
    }
}
function maybeSuffix(x, suffix) {
    if (x === undefined) {
        return undefined;
    }
    return `${x}${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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