"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Provider = void 0;
const path = require("path");
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const consts = require("./runtime/consts");
const util_1 = require("./util");
const waiter_state_machine_1 = require("./waiter-state-machine");
const RUNTIME_HANDLER_PATH = path.join(__dirname, 'runtime');
const FRAMEWORK_HANDLER_TIMEOUT = core_1.Duration.minutes(15); // keep it simple for now
/**
 * Defines an AWS CloudFormation custom resource provider.
 */
class Provider extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.isCompleteHandler && (props.queryInterval || props.totalTimeout)) {
            throw new Error('"queryInterval" and "totalTimeout" can only be configured if "isCompleteHandler" is specified. '
                + 'Otherwise, they have no meaning');
        }
        this.onEventHandler = props.onEventHandler;
        this.isCompleteHandler = props.isCompleteHandler;
        this.logRetention = props.logRetention;
        const onEventFunction = this.createFunction(consts.FRAMEWORK_ON_EVENT_HANDLER_NAME);
        if (this.isCompleteHandler) {
            const isCompleteFunction = this.createFunction(consts.FRAMEWORK_IS_COMPLETE_HANDLER_NAME);
            const timeoutFunction = this.createFunction(consts.FRAMEWORK_ON_TIMEOUT_HANDLER_NAME);
            const retry = util_1.calculateRetryPolicy(props);
            const waiterStateMachine = new waiter_state_machine_1.WaiterStateMachine(this, 'waiter-state-machine', {
                isCompleteHandler: isCompleteFunction,
                timeoutHandler: timeoutFunction,
                backoffRate: retry.backoffRate,
                interval: retry.interval,
                maxAttempts: retry.maxAttempts,
            });
            // the on-event entrypoint is going to start the execution of the waiter
            onEventFunction.addEnvironment(consts.WAITER_STATE_MACHINE_ARN_ENV, waiterStateMachine.stateMachineArn);
            waiterStateMachine.grantStartExecution(onEventFunction);
        }
        this.entrypoint = onEventFunction;
        this.serviceToken = this.entrypoint.functionArn;
    }
    /**
     * Called by `CustomResource` which uses this provider.
     * @deprecated use `provider.serviceToken` instead
     */
    bind(_) {
        return {
            serviceToken: this.entrypoint.functionArn,
        };
    }
    createFunction(entrypoint) {
        const fn = new lambda.Function(this, `framework-${entrypoint}`, {
            code: lambda.Code.fromAsset(RUNTIME_HANDLER_PATH),
            description: `AWS CDK resource provider framework - ${entrypoint} (${this.node.path})`.slice(0, 256),
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: `framework.${entrypoint}`,
            timeout: FRAMEWORK_HANDLER_TIMEOUT,
            logRetention: this.logRetention,
        });
        fn.addEnvironment(consts.USER_ON_EVENT_FUNCTION_ARN_ENV, this.onEventHandler.functionArn);
        this.onEventHandler.grantInvoke(fn);
        if (this.isCompleteHandler) {
            fn.addEnvironment(consts.USER_IS_COMPLETE_FUNCTION_ARN_ENV, this.isCompleteHandler.functionArn);
            this.isCompleteHandler.grantInvoke(fn);
        }
        return fn;
    }
}
exports.Provider = Provider;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJvdmlkZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJwcm92aWRlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2QkFBNkI7QUFFN0IsOENBQThDLENBQUMsc0RBQXNEO0FBRXJHLHdDQUFvRCxDQUFDLGdEQUFnRDtBQUNyRywyQ0FBMkM7QUFDM0MsaUNBQThDO0FBQzlDLGlFQUE0RDtBQUM1RCxNQUFNLG9CQUFvQixHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0FBQzdELE1BQU0seUJBQXlCLEdBQUcsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLHlCQUF5QjtBQXdEakY7O0dBRUc7QUFDSCxNQUFhLFFBQVMsU0FBUSxnQkFBUztJQWtCbkMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFvQjtRQUMxRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLElBQUksQ0FBQyxLQUFLLENBQUMsaUJBQWlCLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxJQUFJLEtBQUssQ0FBQyxZQUFZLENBQUMsRUFBRTtZQUN6RSxNQUFNLElBQUksS0FBSyxDQUFDLGlHQUFpRztrQkFDM0csaUNBQWlDLENBQUMsQ0FBQztTQUM1QztRQUNELElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQztRQUMzQyxJQUFJLENBQUMsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDO1FBQ2pELElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUN2QyxNQUFNLGVBQWUsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLE1BQU0sQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQ3BGLElBQUksSUFBSSxDQUFDLGlCQUFpQixFQUFFO1lBQ3hCLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsa0NBQWtDLENBQUMsQ0FBQztZQUMxRixNQUFNLGVBQWUsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO1lBQ3RGLE1BQU0sS0FBSyxHQUFHLDJCQUFvQixDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQzFDLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSx5Q0FBa0IsQ0FBQyxJQUFJLEVBQUUsc0JBQXNCLEVBQUU7Z0JBQzVFLGlCQUFpQixFQUFFLGtCQUFrQjtnQkFDckMsY0FBYyxFQUFFLGVBQWU7Z0JBQy9CLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztnQkFDOUIsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO2dCQUN4QixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7YUFDakMsQ0FBQyxDQUFDO1lBQ0gsd0VBQXdFO1lBQ3hFLGVBQWUsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLDRCQUE0QixFQUFFLGtCQUFrQixDQUFDLGVBQWUsQ0FBQyxDQUFDO1lBQ3hHLGtCQUFrQixDQUFDLG1CQUFtQixDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQzNEO1FBQ0QsSUFBSSxDQUFDLFVBQVUsR0FBRyxlQUFlLENBQUM7UUFDbEMsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQztJQUNwRCxDQUFDO0lBQ0Q7OztPQUdHO0lBQ0ksSUFBSSxDQUFDLENBQVk7UUFDcEIsT0FBTztZQUNILFlBQVksRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLFdBQVc7U0FDNUMsQ0FBQztJQUNOLENBQUM7SUFDTyxjQUFjLENBQUMsVUFBa0I7UUFDckMsTUFBTSxFQUFFLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxhQUFhLFVBQVUsRUFBRSxFQUFFO1lBQzVELElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxvQkFBb0IsQ0FBQztZQUNqRCxXQUFXLEVBQUUseUNBQXlDLFVBQVUsS0FBSyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDO1lBQ3BHLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7WUFDbkMsT0FBTyxFQUFFLGFBQWEsVUFBVSxFQUFFO1lBQ2xDLE9BQU8sRUFBRSx5QkFBeUI7WUFDbEMsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1NBQ2xDLENBQUMsQ0FBQztRQUNILEVBQUUsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLDhCQUE4QixFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDMUYsSUFBSSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDcEMsSUFBSSxJQUFJLENBQUMsaUJBQWlCLEVBQUU7WUFDeEIsRUFBRSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsaUNBQWlDLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFdBQVcsQ0FBQyxDQUFDO1lBQ2hHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxXQUFXLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDMUM7UUFDRCxPQUFPLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDSjtBQXhFRCw0QkF3RUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgY2ZuIGZyb20gXCIuLi8uLi8uLi9hd3MtY2xvdWRmb3JtYXRpb25cIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZGZvcm1hdGlvbidcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tIFwiLi4vLi4vLi4vYXdzLWxhbWJkYVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSdcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSBcIi4uLy4uLy4uL2F3cy1sb2dzXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtbG9ncydcbmltcG9ydCB7IENvbnN0cnVjdCwgRHVyYXRpb24gfSBmcm9tIFwiLi4vLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCAqIGFzIGNvbnN0cyBmcm9tICcuL3J1bnRpbWUvY29uc3RzJztcbmltcG9ydCB7IGNhbGN1bGF0ZVJldHJ5UG9saWN5IH0gZnJvbSAnLi91dGlsJztcbmltcG9ydCB7IFdhaXRlclN0YXRlTWFjaGluZSB9IGZyb20gJy4vd2FpdGVyLXN0YXRlLW1hY2hpbmUnO1xuY29uc3QgUlVOVElNRV9IQU5ETEVSX1BBVEggPSBwYXRoLmpvaW4oX19kaXJuYW1lLCAncnVudGltZScpO1xuY29uc3QgRlJBTUVXT1JLX0hBTkRMRVJfVElNRU9VVCA9IER1cmF0aW9uLm1pbnV0ZXMoMTUpOyAvLyBrZWVwIGl0IHNpbXBsZSBmb3Igbm93XG4vKipcbiAqIEluaXRpYWxpemF0aW9uIHByb3BlcnRpZXMgZm9yIHRoZSBgUHJvdmlkZXJgIGNvbnN0cnVjdC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQcm92aWRlclByb3BzIHtcbiAgICAvKipcbiAgICAgKiBUaGUgQVdTIExhbWJkYSBmdW5jdGlvbiB0byBpbnZva2UgZm9yIGFsbCByZXNvdXJjZSBsaWZlY3ljbGUgb3BlcmF0aW9uc1xuICAgICAqIChDUkVBVEUvVVBEQVRFL0RFTEVURSkuXG4gICAgICpcbiAgICAgKiBUaGlzIGZ1bmN0aW9uIGlzIHJlc3BvbnNpYmxlIHRvIGJlZ2luIHRoZSByZXF1ZXN0ZWQgcmVzb3VyY2Ugb3BlcmF0aW9uXG4gICAgICogKENSRUFURS9VUERBVEUvREVMRVRFKSBhbmQgcmV0dXJuIGFueSBhZGRpdGlvbmFsIHByb3BlcnRpZXMgdG8gYWRkIHRvIHRoZVxuICAgICAqIGV2ZW50LCB3aGljaCB3aWxsIGxhdGVyIGJlIHBhc3NlZCB0byBgaXNDb21wbGV0ZWAuIFRoZSBgUGh5c2ljYWxSZXNvdXJjZUlkYFxuICAgICAqIHByb3BlcnR5IG11c3QgYmUgaW5jbHVkZWQgaW4gdGhlIHJlc3BvbnNlLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG9uRXZlbnRIYW5kbGVyOiBsYW1iZGEuSUZ1bmN0aW9uO1xuICAgIC8qKlxuICAgICAqIFRoZSBBV1MgTGFtYmRhIGZ1bmN0aW9uIHRvIGludm9rZSBpbiBvcmRlciB0byBkZXRlcm1pbmUgaWYgdGhlIG9wZXJhdGlvbiBpc1xuICAgICAqIGNvbXBsZXRlLlxuICAgICAqXG4gICAgICogVGhpcyBmdW5jdGlvbiB3aWxsIGJlIGNhbGxlZCBpbW1lZGlhdGVseSBhZnRlciBgb25FdmVudGAgYW5kIHRoZW5cbiAgICAgKiBwZXJpb2RpY2FsbHkgYmFzZWQgb24gdGhlIGNvbmZpZ3VyZWQgcXVlcnkgaW50ZXJ2YWwgYXMgbG9uZyBhcyBpdCByZXR1cm5zXG4gICAgICogYGZhbHNlYC4gSWYgdGhlIGZ1bmN0aW9uIHN0aWxsIHJldHVybnMgYGZhbHNlYCBhbmQgdGhlIGFsbG90ZWQgdGltZW91dCBoYXNcbiAgICAgKiBwYXNzZWQsIHRoZSBvcGVyYXRpb24gd2lsbCBmYWlsLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBwcm92aWRlciBpcyBzeW5jaHJvbm91cy4gVGhpcyBtZWFucyB0aGF0IHRoZSBgb25FdmVudGAgaGFuZGxlclxuICAgICAqIGlzIGV4cGVjdGVkIHRvIGZpbmlzaCBhbGwgbGlmZWN5Y2xlIG9wZXJhdGlvbnMgd2l0aGluIHRoZSBpbml0aWFsIGludm9jYXRpb24uXG4gICAgICovXG4gICAgcmVhZG9ubHkgaXNDb21wbGV0ZUhhbmRsZXI/OiBsYW1iZGEuSUZ1bmN0aW9uO1xuICAgIC8qKlxuICAgICAqIFRpbWUgYmV0d2VlbiBjYWxscyB0byB0aGUgYGlzQ29tcGxldGVgIGhhbmRsZXIgd2hpY2ggZGV0ZXJtaW5lcyBpZiB0aGVcbiAgICAgKiByZXNvdXJjZSBoYXMgYmVlbiBzdGFiaWxpemVkLlxuICAgICAqXG4gICAgICogVGhlIGZpcnN0IGBpc0NvbXBsZXRlYCB3aWxsIGJlIGNhbGxlZCBpbW1lZGlhdGVseSBhZnRlciBgaGFuZGxlcmAgYW5kIHRoZW5cbiAgICAgKiBldmVyeSBgcXVlcnlJbnRlcnZhbGAgc2Vjb25kcywgYW5kIHVudGlsIGB0aW1lb3V0YCBoYXMgYmVlbiByZWFjaGVkIG9yIHVudGlsXG4gICAgICogYGlzQ29tcGxldGVgIHJldHVybnMgYHRydWVgLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgRHVyYXRpb24uc2Vjb25kcyg1KVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHF1ZXJ5SW50ZXJ2YWw/OiBEdXJhdGlvbjtcbiAgICAvKipcbiAgICAgKiBUb3RhbCB0aW1lb3V0IGZvciB0aGUgZW50aXJlIG9wZXJhdGlvbi5cbiAgICAgKlxuICAgICAqIFRoZSBtYXhpbXVtIHRpbWVvdXQgaXMgMiBob3VycyAoeWVzLCBpdCBjYW4gZXhjZWVkIHRoZSBBV1MgTGFtYmRhIDE1IG1pbnV0ZXMpXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5taW51dGVzKDMwKVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRvdGFsVGltZW91dD86IER1cmF0aW9uO1xuICAgIC8qKlxuICAgICAqIFRoZSBudW1iZXIgb2YgZGF5cyBmcmFtZXdvcmsgbG9nIGV2ZW50cyBhcmUga2VwdCBpbiBDbG91ZFdhdGNoIExvZ3MuIFdoZW5cbiAgICAgKiB1cGRhdGluZyB0aGlzIHByb3BlcnR5LCB1bnNldHRpbmcgaXQgZG9lc24ndCByZW1vdmUgdGhlIGxvZyByZXRlbnRpb24gcG9saWN5LlxuICAgICAqIFRvIHJlbW92ZSB0aGUgcmV0ZW50aW9uIHBvbGljeSwgc2V0IHRoZSB2YWx1ZSB0byBgSU5GSU5JVEVgLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgbG9ncy5SZXRlbnRpb25EYXlzLklORklOSVRFXG4gICAgICovXG4gICAgcmVhZG9ubHkgbG9nUmV0ZW50aW9uPzogbG9ncy5SZXRlbnRpb25EYXlzO1xufVxuLyoqXG4gKiBEZWZpbmVzIGFuIEFXUyBDbG91ZEZvcm1hdGlvbiBjdXN0b20gcmVzb3VyY2UgcHJvdmlkZXIuXG4gKi9cbmV4cG9ydCBjbGFzcyBQcm92aWRlciBleHRlbmRzIENvbnN0cnVjdCBpbXBsZW1lbnRzIGNmbi5JQ3VzdG9tUmVzb3VyY2VQcm92aWRlciB7XG4gICAgLyoqXG4gICAgICogVGhlIHVzZXItZGVmaW5lZCBBV1MgTGFtYmRhIGZ1bmN0aW9uIHdoaWNoIGlzIGludm9rZWQgZm9yIGFsbCByZXNvdXJjZVxuICAgICAqIGxpZmVjeWNsZSBvcGVyYXRpb25zIChDUkVBVEUvVVBEQVRFL0RFTEVURSkuXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IG9uRXZlbnRIYW5kbGVyOiBsYW1iZGEuSUZ1bmN0aW9uO1xuICAgIC8qKlxuICAgICAqIFRoZSB1c2VyLWRlZmluZWQgQVdTIExhbWJkYSBmdW5jdGlvbiB3aGljaCBpcyBpbnZva2VkIGFzeW5jaHJvbm91c2x5IGluXG4gICAgICogb3JkZXIgdG8gZGV0ZXJtaW5lIGlmIHRoZSBvcGVyYXRpb24gaXMgY29tcGxldGUuXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGlzQ29tcGxldGVIYW5kbGVyPzogbGFtYmRhLklGdW5jdGlvbjtcbiAgICAvKipcbiAgICAgKiBUaGUgc2VydmljZSB0b2tlbiB0byB1c2UgaW4gb3JkZXIgdG8gZGVmaW5lIGN1c3RvbSByZXNvdXJjZXMgdGhhdCBhcmVcbiAgICAgKiBiYWNrZWQgYnkgdGhpcyBwcm92aWRlci5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgc2VydmljZVRva2VuOiBzdHJpbmc7XG4gICAgcHJpdmF0ZSByZWFkb25seSBlbnRyeXBvaW50OiBsYW1iZGEuRnVuY3Rpb247XG4gICAgcHJpdmF0ZSByZWFkb25seSBsb2dSZXRlbnRpb24/OiBsb2dzLlJldGVudGlvbkRheXM7XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFByb3ZpZGVyUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgaWYgKCFwcm9wcy5pc0NvbXBsZXRlSGFuZGxlciAmJiAocHJvcHMucXVlcnlJbnRlcnZhbCB8fCBwcm9wcy50b3RhbFRpbWVvdXQpKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1wicXVlcnlJbnRlcnZhbFwiIGFuZCBcInRvdGFsVGltZW91dFwiIGNhbiBvbmx5IGJlIGNvbmZpZ3VyZWQgaWYgXCJpc0NvbXBsZXRlSGFuZGxlclwiIGlzIHNwZWNpZmllZC4gJ1xuICAgICAgICAgICAgICAgICsgJ090aGVyd2lzZSwgdGhleSBoYXZlIG5vIG1lYW5pbmcnKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLm9uRXZlbnRIYW5kbGVyID0gcHJvcHMub25FdmVudEhhbmRsZXI7XG4gICAgICAgIHRoaXMuaXNDb21wbGV0ZUhhbmRsZXIgPSBwcm9wcy5pc0NvbXBsZXRlSGFuZGxlcjtcbiAgICAgICAgdGhpcy5sb2dSZXRlbnRpb24gPSBwcm9wcy5sb2dSZXRlbnRpb247XG4gICAgICAgIGNvbnN0IG9uRXZlbnRGdW5jdGlvbiA9IHRoaXMuY3JlYXRlRnVuY3Rpb24oY29uc3RzLkZSQU1FV09SS19PTl9FVkVOVF9IQU5ETEVSX05BTUUpO1xuICAgICAgICBpZiAodGhpcy5pc0NvbXBsZXRlSGFuZGxlcikge1xuICAgICAgICAgICAgY29uc3QgaXNDb21wbGV0ZUZ1bmN0aW9uID0gdGhpcy5jcmVhdGVGdW5jdGlvbihjb25zdHMuRlJBTUVXT1JLX0lTX0NPTVBMRVRFX0hBTkRMRVJfTkFNRSk7XG4gICAgICAgICAgICBjb25zdCB0aW1lb3V0RnVuY3Rpb24gPSB0aGlzLmNyZWF0ZUZ1bmN0aW9uKGNvbnN0cy5GUkFNRVdPUktfT05fVElNRU9VVF9IQU5ETEVSX05BTUUpO1xuICAgICAgICAgICAgY29uc3QgcmV0cnkgPSBjYWxjdWxhdGVSZXRyeVBvbGljeShwcm9wcyk7XG4gICAgICAgICAgICBjb25zdCB3YWl0ZXJTdGF0ZU1hY2hpbmUgPSBuZXcgV2FpdGVyU3RhdGVNYWNoaW5lKHRoaXMsICd3YWl0ZXItc3RhdGUtbWFjaGluZScsIHtcbiAgICAgICAgICAgICAgICBpc0NvbXBsZXRlSGFuZGxlcjogaXNDb21wbGV0ZUZ1bmN0aW9uLFxuICAgICAgICAgICAgICAgIHRpbWVvdXRIYW5kbGVyOiB0aW1lb3V0RnVuY3Rpb24sXG4gICAgICAgICAgICAgICAgYmFja29mZlJhdGU6IHJldHJ5LmJhY2tvZmZSYXRlLFxuICAgICAgICAgICAgICAgIGludGVydmFsOiByZXRyeS5pbnRlcnZhbCxcbiAgICAgICAgICAgICAgICBtYXhBdHRlbXB0czogcmV0cnkubWF4QXR0ZW1wdHMsXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIC8vIHRoZSBvbi1ldmVudCBlbnRyeXBvaW50IGlzIGdvaW5nIHRvIHN0YXJ0IHRoZSBleGVjdXRpb24gb2YgdGhlIHdhaXRlclxuICAgICAgICAgICAgb25FdmVudEZ1bmN0aW9uLmFkZEVudmlyb25tZW50KGNvbnN0cy5XQUlURVJfU1RBVEVfTUFDSElORV9BUk5fRU5WLCB3YWl0ZXJTdGF0ZU1hY2hpbmUuc3RhdGVNYWNoaW5lQXJuKTtcbiAgICAgICAgICAgIHdhaXRlclN0YXRlTWFjaGluZS5ncmFudFN0YXJ0RXhlY3V0aW9uKG9uRXZlbnRGdW5jdGlvbik7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5lbnRyeXBvaW50ID0gb25FdmVudEZ1bmN0aW9uO1xuICAgICAgICB0aGlzLnNlcnZpY2VUb2tlbiA9IHRoaXMuZW50cnlwb2ludC5mdW5jdGlvbkFybjtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ2FsbGVkIGJ5IGBDdXN0b21SZXNvdXJjZWAgd2hpY2ggdXNlcyB0aGlzIHByb3ZpZGVyLlxuICAgICAqIEBkZXByZWNhdGVkIHVzZSBgcHJvdmlkZXIuc2VydmljZVRva2VuYCBpbnN0ZWFkXG4gICAgICovXG4gICAgcHVibGljIGJpbmQoXzogQ29uc3RydWN0KTogY2ZuLkN1c3RvbVJlc291cmNlUHJvdmlkZXJDb25maWcge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgc2VydmljZVRva2VuOiB0aGlzLmVudHJ5cG9pbnQuZnVuY3Rpb25Bcm4sXG4gICAgICAgIH07XG4gICAgfVxuICAgIHByaXZhdGUgY3JlYXRlRnVuY3Rpb24oZW50cnlwb2ludDogc3RyaW5nKSB7XG4gICAgICAgIGNvbnN0IGZuID0gbmV3IGxhbWJkYS5GdW5jdGlvbih0aGlzLCBgZnJhbWV3b3JrLSR7ZW50cnlwb2ludH1gLCB7XG4gICAgICAgICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoUlVOVElNRV9IQU5ETEVSX1BBVEgpLFxuICAgICAgICAgICAgZGVzY3JpcHRpb246IGBBV1MgQ0RLIHJlc291cmNlIHByb3ZpZGVyIGZyYW1ld29yayAtICR7ZW50cnlwb2ludH0gKCR7dGhpcy5ub2RlLnBhdGh9KWAuc2xpY2UoMCwgMjU2KSxcbiAgICAgICAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xMF9YLFxuICAgICAgICAgICAgaGFuZGxlcjogYGZyYW1ld29yay4ke2VudHJ5cG9pbnR9YCxcbiAgICAgICAgICAgIHRpbWVvdXQ6IEZSQU1FV09SS19IQU5ETEVSX1RJTUVPVVQsXG4gICAgICAgICAgICBsb2dSZXRlbnRpb246IHRoaXMubG9nUmV0ZW50aW9uLFxuICAgICAgICB9KTtcbiAgICAgICAgZm4uYWRkRW52aXJvbm1lbnQoY29uc3RzLlVTRVJfT05fRVZFTlRfRlVOQ1RJT05fQVJOX0VOViwgdGhpcy5vbkV2ZW50SGFuZGxlci5mdW5jdGlvbkFybik7XG4gICAgICAgIHRoaXMub25FdmVudEhhbmRsZXIuZ3JhbnRJbnZva2UoZm4pO1xuICAgICAgICBpZiAodGhpcy5pc0NvbXBsZXRlSGFuZGxlcikge1xuICAgICAgICAgICAgZm4uYWRkRW52aXJvbm1lbnQoY29uc3RzLlVTRVJfSVNfQ09NUExFVEVfRlVOQ1RJT05fQVJOX0VOViwgdGhpcy5pc0NvbXBsZXRlSGFuZGxlci5mdW5jdGlvbkFybik7XG4gICAgICAgICAgICB0aGlzLmlzQ29tcGxldGVIYW5kbGVyLmdyYW50SW52b2tlKGZuKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gZm47XG4gICAgfVxufVxuIl19