"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCustomResource = exports.AwsCustomResourcePolicy = exports.PhysicalResourceId = exports.PhysicalResourceIdReference = void 0;
const fs = require("fs");
const path = require("path");
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const runtime_1 = require("./runtime");
// don't use "require" since the typescript compiler emits errors since this
// file is not listed in tsconfig.json.
const metadata = JSON.parse(fs.readFileSync(path.join(__dirname, 'sdk-api-metadata.json'), 'utf-8'));
const awsSdkMetadata = metadata;
/**
 * Reference to the physical resource id that can be passed to the AWS operation as a parameter.
 */
class PhysicalResourceIdReference {
    /**
     * toJSON serialization to replace `PhysicalResourceIdReference` with a magic string.
     */
    toJSON() {
        return runtime_1.PHYSICAL_RESOURCE_ID_REFERENCE;
    }
}
exports.PhysicalResourceIdReference = PhysicalResourceIdReference;
/**
 * Physical ID of the custom resource.
 */
class PhysicalResourceId {
    /**
     * @param responsePath Path to a response data element to be used as the physical id.
     * @param id Literal string to be used as the physical id.
     */
    constructor(responsePath, id) {
        this.responsePath = responsePath;
        this.id = id;
    }
    /**
     * Extract the physical resource id from the path (dot notation) to the data in the API call response.
     */
    static fromResponse(responsePath) {
        return new PhysicalResourceId(responsePath, undefined);
    }
    /**
     * Explicit physical resource id.
     */
    static of(id) {
        return new PhysicalResourceId(undefined, id);
    }
}
exports.PhysicalResourceId = PhysicalResourceId;
/**
 * The IAM Policy that will be applied to the different calls.
 */
class AwsCustomResourcePolicy {
    /**
     * @param statements statements for explicit policy.
     * @param resources resources for auto-generated from SDK calls.
     */
    constructor(statements, resources) {
        this.statements = statements;
        this.resources = resources;
    }
    /**
     * Explicit IAM Policy Statements.
     *
     * @param statements the statements to propagate to the SDK calls.
     */
    static fromStatements(statements) {
        return new AwsCustomResourcePolicy(statements, undefined);
    }
    /**
     * Generate IAM Policy Statements from the configured SDK calls.
     *
     * Each SDK call with be translated to an IAM Policy Statement in the form of: `call.service:call.action` (e.g `s3:PutObject`).
     *
     * @param options options for the policy generation
     */
    static fromSdkCalls(options) {
        return new AwsCustomResourcePolicy([], options.resources);
    }
}
exports.AwsCustomResourcePolicy = AwsCustomResourcePolicy;
/**
 * Use this constant to configure access to any resource.
 */
AwsCustomResourcePolicy.ANY_RESOURCE = ['*'];
/**
 * Defines a custom resource that is materialized using specific AWS API calls.
 *
 * Use this to bridge any gap that might exist in the CloudFormation Coverage.
 * You can specify exactly which calls are invoked for the 'CREATE', 'UPDATE' and 'DELETE' life cycle events.
 *
 */
class AwsCustomResource extends cdk.Construct {
    // 'props' cannot be optional, even though all its properties are optional.
    // this is because at least one sdk call must be provided.
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        if (!props.onCreate && !props.onUpdate && !props.onDelete) {
            throw new Error('At least `onCreate`, `onUpdate` or `onDelete` must be specified.');
        }
        for (const call of [props.onCreate, props.onUpdate]) {
            if (call && !call.physicalResourceId) {
                throw new Error('`physicalResourceId` must be specified for onCreate and onUpdate calls.');
            }
        }
        for (const call of [props.onCreate, props.onUpdate, props.onDelete]) {
            if ((_a = call === null || call === void 0 ? void 0 : call.physicalResourceId) === null || _a === void 0 ? void 0 : _a.responsePath) {
                AwsCustomResource.breakIgnoreErrorsCircuit([call], 'PhysicalResourceId.fromResponse');
            }
        }
        if ((_b = props.onCreate) === null || _b === void 0 ? void 0 : _b.parameters) {
            const flattenedOnCreateParams = runtime_1.flatten(JSON.parse(JSON.stringify(props.onCreate.parameters)));
            for (const param in flattenedOnCreateParams) {
                if (flattenedOnCreateParams[param] === runtime_1.PHYSICAL_RESOURCE_ID_REFERENCE) {
                    throw new Error('`PhysicalResourceIdReference` must not be specified in `onCreate` parameters.');
                }
            }
        }
        this.props = props;
        const provider = new lambda.SingletonFunction(this, 'Provider', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'runtime')),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            uuid: '679f53fa-c002-430c-b0da-5b7982bd2287',
            lambdaPurpose: 'AWS',
            timeout: props.timeout || cdk.Duration.minutes(2),
            role: props.role,
            logRetention: props.logRetention,
            functionName: props.functionName,
        });
        this.grantPrincipal = provider.grantPrincipal;
        // Create the policy statements for the custom resource function role, or use the user-provided ones
        const statements = [];
        if (props.policy.statements.length !== 0) {
            // Use custom statements provided by the user
            for (const statement of props.policy.statements) {
                statements.push(statement);
            }
        }
        else {
            // Derive statements from AWS SDK calls
            for (const call of [props.onCreate, props.onUpdate, props.onDelete]) {
                if (call) {
                    const statement = new iam.PolicyStatement({
                        actions: [awsSdkToIamAction(call.service, call.action)],
                        resources: props.policy.resources,
                    });
                    statements.push(statement);
                }
            }
        }
        const policy = new iam.Policy(this, 'CustomResourcePolicy', {
            statements: statements,
        });
        if (provider.role !== undefined) {
            policy.attachToRole(provider.role);
        }
        const create = props.onCreate || props.onUpdate;
        this.customResource = new cdk.CustomResource(this, 'Resource', {
            resourceType: props.resourceType || 'Custom::AWS',
            serviceToken: provider.functionArn,
            pascalCaseProperties: true,
            properties: {
                create: create && encodeBooleans(create),
                update: props.onUpdate && encodeBooleans(props.onUpdate),
                delete: props.onDelete && encodeBooleans(props.onDelete),
                installLatestAwsSdk: (_c = props.installLatestAwsSdk) !== null && _c !== void 0 ? _c : true,
            },
        });
        // If the policy was deleted first, then the function might lose permissions to delete the custom resource
        // This is here so that the policy doesn't get removed before onDelete is called
        this.customResource.node.addDependency(policy);
    }
    static breakIgnoreErrorsCircuit(sdkCalls, caller) {
        for (const call of sdkCalls) {
            if (call === null || call === void 0 ? void 0 : call.ignoreErrorCodesMatching) {
                throw new Error(`\`${caller}\`` + ' cannot be called along with `ignoreErrorCodesMatching`.');
            }
        }
    }
    /**
     * Returns response data for the AWS SDK call.
     *
     * Example for S3 / listBucket : 'Buckets.0.Name'
     *
     * Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getDataString` for string attributes.
     *
     * Note that you cannot use this method if `ignoreErrorCodesMatching`
     * is configured for any of the SDK calls. This is because in such a case,
     * the response data might not exist, and will cause a CloudFormation deploy time error.
     *
     * @param dataPath the path to the data
     */
    getResponseFieldReference(dataPath) {
        AwsCustomResource.breakIgnoreErrorsCircuit([this.props.onCreate, this.props.onUpdate], 'getData');
        return this.customResource.getAtt(dataPath);
    }
    /**
     * Returns response data for the AWS SDK call as string.
     *
     * Example for S3 / listBucket : 'Buckets.0.Name'
     *
     * Note that you cannot use this method if `ignoreErrorCodesMatching`
     * is configured for any of the SDK calls. This is because in such a case,
     * the response data might not exist, and will cause a CloudFormation deploy time error.
     *
     * @param dataPath the path to the data
     */
    getResponseField(dataPath) {
        AwsCustomResource.breakIgnoreErrorsCircuit([this.props.onCreate, this.props.onUpdate], 'getDataString');
        return this.customResource.getAttString(dataPath);
    }
}
exports.AwsCustomResource = AwsCustomResource;
/**
 * Transform SDK service/action to IAM action using metadata from aws-sdk module.
 * Example: CloudWatchLogs with putRetentionPolicy => logs:PutRetentionPolicy
 *
 * TODO: is this mapping correct for all services?
 */
function awsSdkToIamAction(service, action) {
    const srv = service.toLowerCase();
    const iamService = (awsSdkMetadata[srv] && awsSdkMetadata[srv].prefix) || srv;
    const iamAction = action.charAt(0).toUpperCase() + action.slice(1);
    return `${iamService}:${iamAction}`;
}
/**
 * Encodes booleans as special strings
 */
function encodeBooleans(object) {
    return JSON.parse(JSON.stringify(object), (_k, v) => {
        switch (v) {
            case true:
                return 'TRUE:BOOLEAN';
            case false:
                return 'FALSE:BOOLEAN';
            default:
                return v;
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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