"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RemoveTag = exports.Tags = exports.Tag = void 0;
// import * as cxapi from '@aws-cdk/cx-api';
const aspect_1 = require("./aspect");
const tag_manager_1 = require("./tag-manager");
const annotations_1 = require("./annotations");
/**
 * The common functionality for Tag and Remove Tag Aspects
 */
class TagBase {
    constructor(key, props = {}) {
        this.key = key;
        this.props = props;
    }
    visit(construct) {
        if (tag_manager_1.TagManager.isTaggable(construct)) {
            this.applyTag(construct);
        }
    }
}
/**
 * The Tag Aspect will handle adding a tag to this node and cascading tags to children
 */
class Tag extends TagBase {
    constructor(key, value, props = {}) {
        super(key, props);
        this.defaultPriority = 100;
        if (value === undefined) {
            throw new Error('Tag must have a value');
        }
        this.value = value;
    }
    /**
     * DEPRECATED: add tags to the node of a construct and all its the taggable children
     *
     * @deprecated use `Tags.of(scope).add()`
     */
    static add(scope, key, value, props = {}) {
        annotations_1.Annotations.of(scope).addDeprecation('@aws-cdk/core.Tag.add(scope,k,v)', 'Use "Tags.of(scope).add(k,v)" instead');
        Tags.of(scope).add(key, value, props);
    }
    /**
     * DEPRECATED: remove tags to the node of a construct and all its the taggable children
     *
     * @deprecated use `Tags.of(scope).remove()`
     */
    static remove(scope, key, props = {}) {
        annotations_1.Annotations.of(scope).addDeprecation('@aws-cdk/core.Tag.remove(scope,k,v)', 'Use "Tags.of(scope).remove(k,v)" instead');
        Tags.of(scope).remove(key, props);
    }
    applyTag(resource) {
        if (resource.tags.applyTagAspectHere(this.props.includeResourceTypes, this.props.excludeResourceTypes)) {
            resource.tags.setTag(this.key, this.value, this.props.priority !== undefined ? this.props.priority : this.defaultPriority, this.props.applyToLaunchedInstances !== false);
        }
    }
}
exports.Tag = Tag;
/**
 * Manages AWS tags for all resources within a construct scope.
 */
class Tags {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Returns the tags API for this scope.
     * @param scope The scope
     */
    static of(scope) {
        return new Tags(scope);
    }
    /**
     * add tags to the node of a construct and all its the taggable children
     */
    add(key, value, props = {}) {
        aspect_1.Aspects.of(this.scope).add(new Tag(key, value, props));
    }
    /**
     * remove tags to the node of a construct and all its the taggable children
     */
    remove(key, props = {}) {
        aspect_1.Aspects.of(this.scope).add(new RemoveTag(key, props));
    }
}
exports.Tags = Tags;
/**
 * The RemoveTag Aspect will handle removing tags from this node and children
 */
class RemoveTag extends TagBase {
    constructor(key, props = {}) {
        super(key, props);
        this.defaultPriority = 200;
    }
    applyTag(resource) {
        if (resource.tags.applyTagAspectHere(this.props.includeResourceTypes, this.props.excludeResourceTypes)) {
            resource.tags.removeTag(this.key, this.props.priority !== undefined ? this.props.priority : this.defaultPriority);
        }
    }
}
exports.RemoveTag = RemoveTag;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFnLWFzcGVjdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRhZy1hc3BlY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNENBQTRDO0FBQzVDLHFDQUE0QztBQUU1QywrQ0FBc0Q7QUFDdEQsK0NBQTRDO0FBZ0Q1Qzs7R0FFRztBQUNILE1BQWUsT0FBTztJQU1sQixZQUFZLEdBQVcsRUFBRSxRQUFrQixFQUFFO1FBQ3pDLElBQUksQ0FBQyxHQUFHLEdBQUcsR0FBRyxDQUFDO1FBQ2YsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUM7SUFDdkIsQ0FBQztJQUNNLEtBQUssQ0FBQyxTQUFxQjtRQUM5QixJQUFJLHdCQUFVLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxFQUFFO1lBQ2xDLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDNUI7SUFDTCxDQUFDO0NBRUo7QUFDRDs7R0FFRztBQUNILE1BQWEsR0FBSSxTQUFRLE9BQU87SUF3QjVCLFlBQVksR0FBVyxFQUFFLEtBQWEsRUFBRSxRQUFrQixFQUFFO1FBQ3hELEtBQUssQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFGTCxvQkFBZSxHQUFHLEdBQUcsQ0FBQztRQUduQyxJQUFJLEtBQUssS0FBSyxTQUFTLEVBQUU7WUFDckIsTUFBTSxJQUFJLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1NBQzVDO1FBQ0QsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUM7SUFDdkIsQ0FBQztJQTdCRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLEdBQUcsQ0FBQyxLQUFnQixFQUFFLEdBQVcsRUFBRSxLQUFhLEVBQUUsUUFBa0IsRUFBRTtRQUNoRix5QkFBVyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsa0NBQWtDLEVBQUUsdUNBQXVDLENBQUMsQ0FBQztRQUNsSCxJQUFJLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxLQUFnQixFQUFFLEdBQVcsRUFBRSxRQUFrQixFQUFFO1FBQ3BFLHlCQUFXLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxxQ0FBcUMsRUFBRSwwQ0FBMEMsQ0FBQyxDQUFDO1FBQ3hILElBQUksQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBYVMsUUFBUSxDQUFDLFFBQW1CO1FBQ2xDLElBQUksUUFBUSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLG9CQUFvQixFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsb0JBQW9CLENBQUMsRUFBRTtZQUNwRyxRQUFRLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGVBQWUsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLHdCQUF3QixLQUFLLEtBQUssQ0FBQyxDQUFDO1NBQzdLO0lBQ0wsQ0FBQztDQUNKO0FBcENELGtCQW9DQztBQUNEOztHQUVHO0FBQ0gsTUFBYSxJQUFJO0lBUWIsWUFBcUMsS0FBaUI7UUFBakIsVUFBSyxHQUFMLEtBQUssQ0FBWTtJQUFJLENBQUM7SUFQM0Q7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLEVBQUUsQ0FBQyxLQUFpQjtRQUM5QixPQUFPLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNCLENBQUM7SUFFRDs7T0FFRztJQUNJLEdBQUcsQ0FBQyxHQUFXLEVBQUUsS0FBYSxFQUFFLFFBQWtCLEVBQUU7UUFDdkQsZ0JBQU8sQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDM0QsQ0FBQztJQUNEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEdBQVcsRUFBRSxRQUFrQixFQUFFO1FBQzNDLGdCQUFPLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSxTQUFTLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDMUQsQ0FBQztDQUNKO0FBckJELG9CQXFCQztBQUNEOztHQUVHO0FBQ0gsTUFBYSxTQUFVLFNBQVEsT0FBTztJQUVsQyxZQUFZLEdBQVcsRUFBRSxRQUFrQixFQUFFO1FBQ3pDLEtBQUssQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFGTCxvQkFBZSxHQUFHLEdBQUcsQ0FBQztJQUd2QyxDQUFDO0lBQ1MsUUFBUSxDQUFDLFFBQW1CO1FBQ2xDLElBQUksUUFBUSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLG9CQUFvQixFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsb0JBQW9CLENBQUMsRUFBRTtZQUNwRyxRQUFRLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQztTQUNySDtJQUNMLENBQUM7Q0FDSjtBQVZELDhCQVVDIiwic291cmNlc0NvbnRlbnQiOlsiLy8gaW1wb3J0ICogYXMgY3hhcGkgZnJvbSAnQGF3cy1jZGsvY3gtYXBpJztcbmltcG9ydCB7IElBc3BlY3QsIEFzcGVjdHMgfSBmcm9tICcuL2FzcGVjdCc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIElDb25zdHJ1Y3QgfSBmcm9tICcuL2NvbnN0cnVjdC1jb21wYXQnO1xuaW1wb3J0IHsgSVRhZ2dhYmxlLCBUYWdNYW5hZ2VyIH0gZnJvbSAnLi90YWctbWFuYWdlcic7XG5pbXBvcnQgeyBBbm5vdGF0aW9ucyB9IGZyb20gJy4vYW5ub3RhdGlvbnMnO1xuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhIHRhZ1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFRhZ1Byb3BzIHtcbiAgICAvKipcbiAgICAgKiBXaGV0aGVyIHRoZSB0YWcgc2hvdWxkIGJlIGFwcGxpZWQgdG8gaW5zdGFuY2VzIGluIGFuIEF1dG9TY2FsaW5nR3JvdXBcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IHRydWVcbiAgICAgKi9cbiAgICByZWFkb25seSBhcHBseVRvTGF1bmNoZWRJbnN0YW5jZXM/OiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIEFuIGFycmF5IG9mIFJlc291cmNlIFR5cGVzIHRoYXQgd2lsbCBub3QgcmVjZWl2ZSB0aGlzIHRhZ1xuICAgICAqXG4gICAgICogQW4gZW1wdHkgYXJyYXkgd2lsbCBhbGxvdyB0aGlzIHRhZyB0byBiZSBhcHBsaWVkIHRvIGFsbCByZXNvdXJjZXMuIEFcbiAgICAgKiBub24tZW1wdHkgYXJyYXkgd2lsbCBhcHBseSB0aGlzIHRhZyBvbmx5IGlmIHRoZSBSZXNvdXJjZSB0eXBlIGlzIG5vdCBpblxuICAgICAqIHRoaXMgYXJyYXkuXG4gICAgICogQGRlZmF1bHQgW11cbiAgICAgKi9cbiAgICByZWFkb25seSBleGNsdWRlUmVzb3VyY2VUeXBlcz86IHN0cmluZ1tdO1xuICAgIC8qKlxuICAgICAqIEFuIGFycmF5IG9mIFJlc291cmNlIFR5cGVzIHRoYXQgd2lsbCByZWNlaXZlIHRoaXMgdGFnXG4gICAgICpcbiAgICAgKiBBbiBlbXB0eSBhcnJheSB3aWxsIG1hdGNoIGFueSBSZXNvdXJjZS4gQSBub24tZW1wdHkgYXJyYXkgd2lsbCBhcHBseSB0aGlzXG4gICAgICogdGFnIG9ubHkgdG8gUmVzb3VyY2UgdHlwZXMgdGhhdCBhcmUgaW5jbHVkZWQgaW4gdGhpcyBhcnJheS5cbiAgICAgKiBAZGVmYXVsdCBbXVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGluY2x1ZGVSZXNvdXJjZVR5cGVzPzogc3RyaW5nW107XG4gICAgLyoqXG4gICAgICogUHJpb3JpdHkgb2YgdGhlIHRhZyBvcGVyYXRpb25cbiAgICAgKlxuICAgICAqIEhpZ2hlciBvciBlcXVhbCBwcmlvcml0eSB0YWdzIHdpbGwgdGFrZSBwcmVjZWRlbmNlLlxuICAgICAqXG4gICAgICogU2V0dGluZyBwcmlvcml0eSB3aWxsIGVuYWJsZSB0aGUgdXNlciB0byBjb250cm9sIHRhZ3Mgd2hlbiB0aGV5IG5lZWQgdG8gbm90XG4gICAgICogZm9sbG93IHRoZSBkZWZhdWx0IHByZWNlZGVuY2UgcGF0dGVybiBvZiBsYXN0IGFwcGxpZWQgYW5kIGNsb3Nlc3QgdG8gdGhlXG4gICAgICogY29uc3RydWN0IGluIHRoZSB0cmVlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHRcbiAgICAgKlxuICAgICAqIERlZmF1bHQgcHJpb3JpdGllczpcbiAgICAgKlxuICAgICAqIC0gMTAwIGZvciB7QGxpbmsgU2V0VGFnfVxuICAgICAqIC0gMjAwIGZvciB7QGxpbmsgUmVtb3ZlVGFnfVxuICAgICAqIC0gNTAgZm9yIHRhZ3MgYWRkZWQgZGlyZWN0bHkgdG8gQ2xvdWRGb3JtYXRpb24gcmVzb3VyY2VzXG4gICAgICpcbiAgICAgKi9cbiAgICByZWFkb25seSBwcmlvcml0eT86IG51bWJlcjtcbn1cbi8qKlxuICogVGhlIGNvbW1vbiBmdW5jdGlvbmFsaXR5IGZvciBUYWcgYW5kIFJlbW92ZSBUYWcgQXNwZWN0c1xuICovXG5hYnN0cmFjdCBjbGFzcyBUYWdCYXNlIGltcGxlbWVudHMgSUFzcGVjdCB7XG4gICAgLyoqXG4gICAgICogVGhlIHN0cmluZyBrZXkgZm9yIHRoZSB0YWdcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkga2V5OiBzdHJpbmc7XG4gICAgcHJvdGVjdGVkIHJlYWRvbmx5IHByb3BzOiBUYWdQcm9wcztcbiAgICBjb25zdHJ1Y3RvcihrZXk6IHN0cmluZywgcHJvcHM6IFRhZ1Byb3BzID0ge30pIHtcbiAgICAgICAgdGhpcy5rZXkgPSBrZXk7XG4gICAgICAgIHRoaXMucHJvcHMgPSBwcm9wcztcbiAgICB9XG4gICAgcHVibGljIHZpc2l0KGNvbnN0cnVjdDogSUNvbnN0cnVjdCk6IHZvaWQge1xuICAgICAgICBpZiAoVGFnTWFuYWdlci5pc1RhZ2dhYmxlKGNvbnN0cnVjdCkpIHtcbiAgICAgICAgICAgIHRoaXMuYXBwbHlUYWcoY29uc3RydWN0KTtcbiAgICAgICAgfVxuICAgIH1cbiAgICBwcm90ZWN0ZWQgYWJzdHJhY3QgYXBwbHlUYWcocmVzb3VyY2U6IElUYWdnYWJsZSk6IHZvaWQ7XG59XG4vKipcbiAqIFRoZSBUYWcgQXNwZWN0IHdpbGwgaGFuZGxlIGFkZGluZyBhIHRhZyB0byB0aGlzIG5vZGUgYW5kIGNhc2NhZGluZyB0YWdzIHRvIGNoaWxkcmVuXG4gKi9cbmV4cG9ydCBjbGFzcyBUYWcgZXh0ZW5kcyBUYWdCYXNlIHtcbiAgICAvKipcbiAgICAgKiBERVBSRUNBVEVEOiBhZGQgdGFncyB0byB0aGUgbm9kZSBvZiBhIGNvbnN0cnVjdCBhbmQgYWxsIGl0cyB0aGUgdGFnZ2FibGUgY2hpbGRyZW5cbiAgICAgKlxuICAgICAqIEBkZXByZWNhdGVkIHVzZSBgVGFncy5vZihzY29wZSkuYWRkKClgXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBhZGQoc2NvcGU6IENvbnN0cnVjdCwga2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcsIHByb3BzOiBUYWdQcm9wcyA9IHt9KSB7XG4gICAgICAgIEFubm90YXRpb25zLm9mKHNjb3BlKS5hZGREZXByZWNhdGlvbignQGF3cy1jZGsvY29yZS5UYWcuYWRkKHNjb3BlLGssdiknLCAnVXNlIFwiVGFncy5vZihzY29wZSkuYWRkKGssdilcIiBpbnN0ZWFkJyk7XG4gICAgICAgIFRhZ3Mub2Yoc2NvcGUpLmFkZChrZXksIHZhbHVlLCBwcm9wcyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIERFUFJFQ0FURUQ6IHJlbW92ZSB0YWdzIHRvIHRoZSBub2RlIG9mIGEgY29uc3RydWN0IGFuZCBhbGwgaXRzIHRoZSB0YWdnYWJsZSBjaGlsZHJlblxuICAgICAqXG4gICAgICogQGRlcHJlY2F0ZWQgdXNlIGBUYWdzLm9mKHNjb3BlKS5yZW1vdmUoKWBcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHJlbW92ZShzY29wZTogQ29uc3RydWN0LCBrZXk6IHN0cmluZywgcHJvcHM6IFRhZ1Byb3BzID0ge30pIHtcbiAgICAgICAgQW5ub3RhdGlvbnMub2Yoc2NvcGUpLmFkZERlcHJlY2F0aW9uKCdAYXdzLWNkay9jb3JlLlRhZy5yZW1vdmUoc2NvcGUsayx2KScsICdVc2UgXCJUYWdzLm9mKHNjb3BlKS5yZW1vdmUoayx2KVwiIGluc3RlYWQnKTtcbiAgICAgICAgVGFncy5vZihzY29wZSkucmVtb3ZlKGtleSwgcHJvcHMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgc3RyaW5nIHZhbHVlIG9mIHRoZSB0YWdcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgdmFsdWU6IHN0cmluZztcbiAgICBwcml2YXRlIHJlYWRvbmx5IGRlZmF1bHRQcmlvcml0eSA9IDEwMDtcbiAgICBjb25zdHJ1Y3RvcihrZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZywgcHJvcHM6IFRhZ1Byb3BzID0ge30pIHtcbiAgICAgICAgc3VwZXIoa2V5LCBwcm9wcyk7XG4gICAgICAgIGlmICh2YWx1ZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1RhZyBtdXN0IGhhdmUgYSB2YWx1ZScpO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMudmFsdWUgPSB2YWx1ZTtcbiAgICB9XG4gICAgcHJvdGVjdGVkIGFwcGx5VGFnKHJlc291cmNlOiBJVGFnZ2FibGUpIHtcbiAgICAgICAgaWYgKHJlc291cmNlLnRhZ3MuYXBwbHlUYWdBc3BlY3RIZXJlKHRoaXMucHJvcHMuaW5jbHVkZVJlc291cmNlVHlwZXMsIHRoaXMucHJvcHMuZXhjbHVkZVJlc291cmNlVHlwZXMpKSB7XG4gICAgICAgICAgICByZXNvdXJjZS50YWdzLnNldFRhZyh0aGlzLmtleSwgdGhpcy52YWx1ZSwgdGhpcy5wcm9wcy5wcmlvcml0eSAhPT0gdW5kZWZpbmVkID8gdGhpcy5wcm9wcy5wcmlvcml0eSA6IHRoaXMuZGVmYXVsdFByaW9yaXR5LCB0aGlzLnByb3BzLmFwcGx5VG9MYXVuY2hlZEluc3RhbmNlcyAhPT0gZmFsc2UpO1xuICAgICAgICB9XG4gICAgfVxufVxuLyoqXG4gKiBNYW5hZ2VzIEFXUyB0YWdzIGZvciBhbGwgcmVzb3VyY2VzIHdpdGhpbiBhIGNvbnN0cnVjdCBzY29wZS5cbiAqL1xuZXhwb3J0IGNsYXNzIFRhZ3Mge1xuICAgIC8qKlxuICAgICAqIFJldHVybnMgdGhlIHRhZ3MgQVBJIGZvciB0aGlzIHNjb3BlLlxuICAgICAqIEBwYXJhbSBzY29wZSBUaGUgc2NvcGVcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIG9mKHNjb3BlOiBJQ29uc3RydWN0KTogVGFncyB7XG4gICAgICAgIHJldHVybiBuZXcgVGFncyhzY29wZSk7XG4gICAgfVxuICAgIHByaXZhdGUgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBzY29wZTogSUNvbnN0cnVjdCkgeyB9XG4gICAgLyoqXG4gICAgICogYWRkIHRhZ3MgdG8gdGhlIG5vZGUgb2YgYSBjb25zdHJ1Y3QgYW5kIGFsbCBpdHMgdGhlIHRhZ2dhYmxlIGNoaWxkcmVuXG4gICAgICovXG4gICAgcHVibGljIGFkZChrZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZywgcHJvcHM6IFRhZ1Byb3BzID0ge30pIHtcbiAgICAgICAgQXNwZWN0cy5vZih0aGlzLnNjb3BlKS5hZGQobmV3IFRhZyhrZXksIHZhbHVlLCBwcm9wcykpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiByZW1vdmUgdGFncyB0byB0aGUgbm9kZSBvZiBhIGNvbnN0cnVjdCBhbmQgYWxsIGl0cyB0aGUgdGFnZ2FibGUgY2hpbGRyZW5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVtb3ZlKGtleTogc3RyaW5nLCBwcm9wczogVGFnUHJvcHMgPSB7fSkge1xuICAgICAgICBBc3BlY3RzLm9mKHRoaXMuc2NvcGUpLmFkZChuZXcgUmVtb3ZlVGFnKGtleSwgcHJvcHMpKTtcbiAgICB9XG59XG4vKipcbiAqIFRoZSBSZW1vdmVUYWcgQXNwZWN0IHdpbGwgaGFuZGxlIHJlbW92aW5nIHRhZ3MgZnJvbSB0aGlzIG5vZGUgYW5kIGNoaWxkcmVuXG4gKi9cbmV4cG9ydCBjbGFzcyBSZW1vdmVUYWcgZXh0ZW5kcyBUYWdCYXNlIHtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGRlZmF1bHRQcmlvcml0eSA9IDIwMDtcbiAgICBjb25zdHJ1Y3RvcihrZXk6IHN0cmluZywgcHJvcHM6IFRhZ1Byb3BzID0ge30pIHtcbiAgICAgICAgc3VwZXIoa2V5LCBwcm9wcyk7XG4gICAgfVxuICAgIHByb3RlY3RlZCBhcHBseVRhZyhyZXNvdXJjZTogSVRhZ2dhYmxlKTogdm9pZCB7XG4gICAgICAgIGlmIChyZXNvdXJjZS50YWdzLmFwcGx5VGFnQXNwZWN0SGVyZSh0aGlzLnByb3BzLmluY2x1ZGVSZXNvdXJjZVR5cGVzLCB0aGlzLnByb3BzLmV4Y2x1ZGVSZXNvdXJjZVR5cGVzKSkge1xuICAgICAgICAgICAgcmVzb3VyY2UudGFncy5yZW1vdmVUYWcodGhpcy5rZXksIHRoaXMucHJvcHMucHJpb3JpdHkgIT09IHVuZGVmaW5lZCA/IHRoaXMucHJvcHMucHJpb3JpdHkgOiB0aGlzLmRlZmF1bHRQcmlvcml0eSk7XG4gICAgICAgIH1cbiAgICB9XG59XG4iXX0=