"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.synthesize = void 0;
const cxapi = require("../../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const annotations_1 = require("../annotations");
const aspect_1 = require("../aspect");
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const prepare_app_1 = require("./prepare-app");
const tree_metadata_1 = require("./tree-metadata");
function synthesize(root, options = {}) {
    // we start by calling "synth" on all nested assemblies (which will take care of all their children)
    synthNestedAssemblies(root, options);
    invokeAspects(root);
    // This is mostly here for legacy purposes as the framework itself does not use prepare anymore.
    prepareTree(root);
    // resolve references
    prepare_app_1.prepareApp(root);
    // give all children an opportunity to validate now that we've finished prepare
    if (!options.skipValidation) {
        validateTree(root);
    }
    // in unit tests, we support creating free-standing stacks, so we create the
    // assembly builder here.
    const builder = stage_1.Stage.isStage(root)
        ? root._assemblyBuilder
        : new cxapi.CloudAssemblyBuilder(options.outdir);
    // next, we invoke "onSynthesize" on all of our children. this will allow
    // stacks to add themselves to the synthesized cloud assembly.
    synthesizeTree(root, builder);
    return builder.buildAssembly({
        runtimeInfo: options.runtimeInfo,
    });
}
exports.synthesize = synthesize;
/**
 * Find Assemblies inside the construct and call 'synth' on them
 *
 * (They will in turn recurse again)
 */
function synthNestedAssemblies(root, options) {
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            child.synth(options);
        }
        else {
            synthNestedAssemblies(child, options);
        }
    }
}
/**
 * Invoke aspects on the given construct tree.
 *
 * Aspects are not propagated across Assembly boundaries. The same Aspect will not be invoked
 * twice for the same construct.
 */
function invokeAspects(root) {
    const invokedByPath = {};
    let nestedAspectWarning = false;
    recurse(root, []);
    function recurse(construct, inheritedAspects) {
        const node = construct.node;
        const aspects = aspect_1.Aspects.of(construct);
        const allAspectsHere = [...inheritedAspects !== null && inheritedAspects !== void 0 ? inheritedAspects : [], ...aspects.aspects];
        const nodeAspectsCount = aspects.aspects.length;
        for (const aspect of allAspectsHere) {
            let invoked = invokedByPath[node.path];
            if (!invoked) {
                invoked = invokedByPath[node.path] = [];
            }
            if (invoked.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            // if an aspect was added to the node while invoking another aspect it will not be invoked, emit a warning
            // the `nestedAspectWarning` flag is used to prevent the warning from being emitted for every child
            if (!nestedAspectWarning && nodeAspectsCount !== aspects.aspects.length) {
                annotations_1.Annotations.of(construct).addWarning('We detected an Aspect was added via another Aspect, and will not be applied');
                nestedAspectWarning = true;
            }
            // mark as invoked for this node
            invoked.push(aspect);
        }
        for (const child of construct.node.children) {
            if (!stage_1.Stage.isStage(child)) {
                recurse(child, allAspectsHere);
            }
        }
    }
}
/**
 * Prepare all constructs in the given construct tree in post-order.
 *
 * Stop at Assembly boundaries.
 */
function prepareTree(root) {
    visit(root, 'post', construct => construct.onPrepare());
}
/**
 * Synthesize children in post-order into the given builder
 *
 * Stop at Assembly boundaries.
 */
function synthesizeTree(root, builder) {
    visit(root, 'post', construct => {
        const session = {
            outdir: builder.outdir,
            assembly: builder,
        };
        if (construct instanceof stack_1.Stack) {
            construct.synthesizer.synthesize(session);
        }
        else if (construct instanceof tree_metadata_1.TreeMetadata) {
            construct._synthesizeTree(session);
        }
        // this will soon be deprecated and removed in 2.x
        // see https://github.com/aws/aws-cdk-rfcs/issues/192
        construct.onSynthesize(session);
    });
}
/**
 * Validate all constructs in the given construct tree
 */
function validateTree(root) {
    const errors = new Array();
    visit(root, 'pre', construct => {
        for (const message of construct.onValidate()) {
            errors.push({ message, source: construct });
        }
    });
    if (errors.length > 0) {
        const errorList = errors.map(e => `[${e.source.node.path}] ${e.message}`).join('\n  ');
        throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
    }
}
/**
 * Visit the given construct tree in either pre or post order, stopping at Assemblies
 */
function visit(root, order, cb) {
    if (order === 'pre') {
        cb(root);
    }
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            continue;
        }
        visit(child, order, cb);
    }
    if (order === 'post') {
        cb(root);
    }
}
//# sourceMappingURL=data:application/json;base64,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