"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetStaging = void 0;
const crypto = require("crypto");
const os = require("os");
const path = require("path");
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const fs = require("fs-extra");
const assets_1 = require("./assets");
const construct_compat_1 = require("./construct-compat");
const fs_1 = require("./fs");
const stack_1 = require("./stack");
const stage_1 = require("./stage");
/**
 * Stages a file or directory from a location on the file system into a staging
 * directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 */
class AssetStaging extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.sourcePath = props.sourcePath;
        this.fingerprintOptions = props;
        const outdir = (_a = stage_1.Stage.of(this)) === null || _a === void 0 ? void 0 : _a.outdir;
        if (!outdir) {
            throw new Error('unable to determine cloud assembly output directory. Assets must be defined indirectly within a "Stage" or an "App" scope');
        }
        // Determine the hash type based on the props as props.assetHashType is
        // optional from a caller perspective.
        const hashType = determineHashType(props.assetHashType, props.assetHash);
        if (props.bundling) {
            // Check if we actually have to bundle for this stack
            const bundlingStacks = (_b = this.node.tryGetContext(cxapi.BUNDLING_STACKS)) !== null && _b !== void 0 ? _b : ['*'];
            const runBundling = bundlingStacks.includes(stack_1.Stack.of(this).stackName) || bundlingStacks.includes('*');
            if (runBundling) {
                // Determine the source hash in advance of bundling if the asset hash type
                // is SOURCE so that the bundler can opt to re-use its previous output.
                const sourceHash = hashType === assets_1.AssetHashType.SOURCE
                    ? this.calculateHash(hashType, props.assetHash, props.bundling)
                    : undefined;
                this.bundleDir = this.bundle(props.bundling, outdir, sourceHash);
                this.assetHash = sourceHash !== null && sourceHash !== void 0 ? sourceHash : this.calculateHash(hashType, props.assetHash, props.bundling);
                this.relativePath = renderAssetFilename(this.assetHash);
                this.stagedPath = this.relativePath;
            }
            else { // Bundling is skipped
                this.assetHash = props.assetHashType === assets_1.AssetHashType.BUNDLE
                    ? this.calculateHash(assets_1.AssetHashType.CUSTOM, this.node.path) // Use node path as dummy hash because we're not bundling
                    : this.calculateHash(hashType, props.assetHash);
                this.stagedPath = this.sourcePath;
            }
        }
        else {
            this.assetHash = this.calculateHash(hashType, props.assetHash);
            const stagingDisabled = this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
            if (stagingDisabled) {
                this.stagedPath = this.sourcePath;
            }
            else {
                this.relativePath = renderAssetFilename(this.assetHash, path.extname(this.sourcePath));
                this.stagedPath = this.relativePath;
            }
        }
        this.sourceHash = this.assetHash;
        this.stageAsset(outdir);
    }
    stageAsset(outdir) {
        // Staging is disabled
        if (!this.relativePath) {
            return;
        }
        const targetPath = path.join(outdir, this.relativePath);
        // Staging the bundling asset.
        if (this.bundleDir) {
            const isAlreadyStaged = fs.existsSync(targetPath);
            if (isAlreadyStaged && path.resolve(this.bundleDir) !== path.resolve(targetPath)) {
                // When an identical asset is already staged and the bundler used an
                // intermediate bundling directory, we remove the extra directory.
                fs.removeSync(this.bundleDir);
            }
            else if (!isAlreadyStaged) {
                fs.renameSync(this.bundleDir, targetPath);
            }
            return;
        }
        // Already staged
        if (fs.existsSync(targetPath)) {
            return;
        }
        // Copy file/directory to staging directory
        const stat = fs.statSync(this.sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(this.sourcePath, targetPath);
        }
        else if (stat.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(this.sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${this.sourcePath}`);
        }
    }
    /**
     * Bundles an asset and provides the emitted asset's directory in return.
     *
     * @param options Bundling options
     * @param outdir Parent directory to create the bundle output directory in
     * @param sourceHash The asset source hash if known in advance. If this field
     * is provided, the bundler may opt to skip bundling, providing any already-
     * emitted bundle. If this field is not provided, the bundler uses an
     * intermediate directory in outdir.
     * @returns The fully resolved bundle output directory.
     */
    bundle(options, outdir, sourceHash) {
        var _a, _b, _c;
        let bundleDir;
        if (sourceHash) {
            // When an asset hash is known in advance of bundling, the bundler outputs
            // directly to the assembly output directory.
            bundleDir = path.resolve(path.join(outdir, renderAssetFilename(sourceHash)));
            if (fs.existsSync(bundleDir)) {
                // Pre-existing bundle directory. The bundle has already been generated
                // once before, so we'll give the caller nothing.
                return bundleDir;
            }
            fs.ensureDirSync(bundleDir);
        }
        else {
            // When the asset hash isn't known in advance, bundler outputs to an
            // intermediate directory.
            // Create temp directory for bundling inside the temp staging directory
            bundleDir = path.resolve(fs.mkdtempSync(path.join(outdir, 'bundling-temp-')));
            // Chmod the bundleDir to full access.
            fs.chmodSync(bundleDir, 0o777);
        }
        let user;
        if (options.user) {
            user = options.user;
        }
        else { // Default to current user
            const userInfo = os.userInfo();
            user = userInfo.uid !== -1 // uid is -1 on Windows
                ? `${userInfo.uid}:${userInfo.gid}`
                : '1000:1000';
        }
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: AssetStaging.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: AssetStaging.BUNDLING_OUTPUT_DIR,
            },
            ...(_a = options.volumes) !== null && _a !== void 0 ? _a : [],
        ];
        let localBundling;
        try {
            process.stderr.write(`Bundling asset ${this.node.path}...\n`);
            localBundling = (_b = options.local) === null || _b === void 0 ? void 0 : _b.tryBundle(bundleDir, options);
            if (!localBundling) {
                options.image._run({
                    command: options.command,
                    user,
                    volumes,
                    environment: options.environment,
                    workingDirectory: (_c = options.workingDirectory) !== null && _c !== void 0 ? _c : AssetStaging.BUNDLING_INPUT_DIR,
                });
            }
        }
        catch (err) {
            // When bundling fails, keep the bundle output for diagnosability, but
            // rename it out of the way so that the next run doesn't assume it has a
            // valid bundleDir.
            const bundleErrorDir = bundleDir + '-error';
            if (fs.existsSync(bundleErrorDir)) {
                // Remove the last bundleErrorDir.
                fs.removeSync(bundleErrorDir);
            }
            fs.renameSync(bundleDir, bundleErrorDir);
            throw new Error(`Failed to bundle asset ${this.node.path}, bundle output is located at ${bundleErrorDir}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            const outputDir = localBundling ? bundleDir : AssetStaging.BUNDLING_OUTPUT_DIR;
            throw new Error(`Bundling did not produce any output. Check that content is written to ${outputDir}.`);
        }
        return bundleDir;
    }
    calculateHash(hashType, assetHash, bundling) {
        if (hashType === assets_1.AssetHashType.CUSTOM && !assetHash) {
            throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
        }
        // When bundling a CUSTOM or SOURCE asset hash type, we want the hash to include
        // the bundling configuration. We handle CUSTOM and bundled SOURCE hash types
        // as a special case to preserve existing user asset hashes in all other cases.
        if (hashType == assets_1.AssetHashType.CUSTOM || (hashType == assets_1.AssetHashType.SOURCE && bundling)) {
            const hash = crypto.createHash('sha256');
            // if asset hash is provided by user, use it, otherwise fingerprint the source.
            hash.update(assetHash !== null && assetHash !== void 0 ? assetHash : fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions));
            // If we're bundling an asset, include the bundling configuration in the hash
            if (bundling) {
                hash.update(JSON.stringify(bundling));
            }
            return hash.digest('hex');
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
                if (!this.bundleDir) {
                    throw new Error('Cannot use `AssetHashType.BUNDLE` when `bundling` is not specified.');
                }
                return fs_1.FileSystem.fingerprint(this.bundleDir, this.fingerprintOptions);
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
/**
 * The directory inside the bundling container into which the asset sources will be mounted.
 * @experimental
 */
AssetStaging.BUNDLING_INPUT_DIR = '/asset-input';
/**
 * The directory inside the bundling container into which the bundled output should be written.
 * @experimental
 */
AssetStaging.BUNDLING_OUTPUT_DIR = '/asset-output';
function renderAssetFilename(assetHash, extension = '') {
    return `asset.${assetHash}${extension}`;
}
/**
 * Determines the hash type from user-given prop values.
 *
 * @param assetHashType Asset hash type construct prop
 * @param assetHash Asset hash given in the construct props
 */
function determineHashType(assetHashType, assetHash) {
    if (assetHash) {
        if (assetHashType && assetHashType !== assets_1.AssetHashType.CUSTOM) {
            throw new Error(`Cannot specify \`${assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
        }
        return assets_1.AssetHashType.CUSTOM;
    }
    else if (assetHashType) {
        return assetHashType;
    }
    else {
        return assets_1.AssetHashType.SOURCE;
    }
}
//# sourceMappingURL=data:application/json;base64,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