"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Arn = void 0;
const cfn_fn_1 = require("./cfn-fn");
const token_1 = require("./token");
const util_1 = require("./util");
class Arn {
    /**
     * Creates an ARN from components.
     *
     * If `partition`, `region` or `account` are not specified, the stack's
     * partition, region and account will be used.
     *
     * If any component is the empty string, an empty string will be inserted
     * into the generated ARN at the location that component corresponds to.
     *
     * The ARN will be formatted as follows:
     *
     *   arn:{partition}:{service}:{region}:{account}:{resource}{sep}{resource-name}
     *
     * The required ARN pieces that are omitted will be taken from the stack that
     * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
     * can be 'undefined'.
     */
    static format(components, stack) {
        const partition = components.partition !== undefined ? components.partition : stack.partition;
        const region = components.region !== undefined ? components.region : stack.region;
        const account = components.account !== undefined ? components.account : stack.account;
        const sep = components.sep !== undefined ? components.sep : '/';
        const values = ['arn', ':', partition, ':', components.service, ':', region, ':', account, ':', components.resource];
        if (sep !== '/' && sep !== ':' && sep !== '') {
            throw new Error('resourcePathSep may only be ":", "/" or an empty string');
        }
        if (components.resourceName != null) {
            values.push(sep);
            values.push(components.resourceName);
        }
        return values.join('');
    }
    /**
     * Given an ARN, parses it and returns components.
     *
     * If the ARN is a concrete string, it will be parsed and validated. The
     * separator (`sep`) will be set to '/' if the 6th component includes a '/',
     * in which case, `resource` will be set to the value before the '/' and
     * `resourceName` will be the rest. In case there is no '/', `resource` will
     * be set to the 6th components and `resourceName` will be set to the rest
     * of the string.
     *
     * If the ARN includes tokens (or is a token), the ARN cannot be validated,
     * since we don't have the actual value yet at the time of this function
     * call. You will have to know the separator and the type of ARN. The
     * resulting `ArnComponents` object will contain tokens for the
     * subexpressions of the ARN, not string literals. In this case this
     * function cannot properly parse the complete final resourceName (path) out
     * of ARNs that use '/' to both separate the 'resource' from the
     * 'resourceName' AND to subdivide the resourceName further. For example, in
     * S3 ARNs:
     *
     *    arn:aws:s3:::my_corporate_bucket/path/to/exampleobject.png
     *
     * After parsing the resourceName will not contain
     * 'path/to/exampleobject.png' but simply 'path'. This is a limitation
     * because there is no slicing functionality in CloudFormation templates.
     *
     * @param arn The ARN to parse
     * @param sepIfToken The separator used to separate resource from resourceName
     * @param hasName Whether there is a name component in the ARN at all. For
     * example, SNS Topics ARNs have the 'resource' component contain the topic
     * name, and no 'resourceName' component.
     *
     * @returns an ArnComponents object which allows access to the various
     * components of the ARN.
     *
     * @returns an ArnComponents object which allows access to the various
     *      components of the ARN.
     */
    static parse(arn, sepIfToken = '/', hasName = true) {
        const components = arn.split(':');
        const looksLikeArn = arn.startsWith('arn:') && components.length >= 6 && components.length <= 7;
        if (token_1.Token.isUnresolved(arn) && !looksLikeArn) {
            return parseToken(arn, sepIfToken, hasName);
        }
        // If the ARN merely contains Tokens, but otherwise *looks* mostly like an ARN,
        // it's a string of the form 'arn:${partition}:service:${region}:${account}:abc/xyz'.
        // Parse fields out to the best of our ability.
        // Tokens won't contain ":", so this won't break them.
        if (components.length < 6) {
            throw new Error('ARNs must have at least 6 components: ' + arn);
        }
        const [arnPrefix, partition, service, region, account, sixth, ...rest] = components;
        if (arnPrefix !== 'arn') {
            throw new Error('ARNs must start with "arn:": ' + arn);
        }
        if (!service) {
            throw new Error('The `service` component (3rd component) is required: ' + arn);
        }
        if (!sixth) {
            throw new Error('The `resource` component (6th component) is required: ' + arn);
        }
        let resource;
        let resourceName;
        let sep;
        let sepIndex = sixth.indexOf('/');
        if (sepIndex !== -1) {
            sep = '/';
        }
        else if (rest.length > 0) {
            sep = ':';
            sepIndex = -1;
        }
        if (sepIndex !== -1) {
            resource = sixth.substr(0, sepIndex);
            resourceName = sixth.substr(sepIndex + 1);
        }
        else {
            resource = sixth;
        }
        if (rest.length > 0) {
            if (!resourceName) {
                resourceName = '';
            }
            else {
                resourceName += ':';
            }
            resourceName += rest.join(':');
        }
        // "|| undefined" will cause empty strings to be treated as "undefined".
        // Optional ARN attributes (e.g. region, account) should return as empty string
        // if they are provided as such.
        return util_1.filterUndefined({
            service: service || undefined,
            resource: resource || undefined,
            partition: partition || undefined,
            region,
            account,
            resourceName,
            sep,
        });
    }
    constructor() { }
}
exports.Arn = Arn;
/**
 * Given a Token evaluating to ARN, parses it and returns components.
 *
 * The ARN cannot be validated, since we don't have the actual value yet
 * at the time of this function call. You will have to know the separator
 * and the type of ARN.
 *
 * The resulting `ArnComponents` object will contain tokens for the
 * subexpressions of the ARN, not string literals.
 *
 * WARNING: this function cannot properly parse the complete final
 * resourceName (path) out of ARNs that use '/' to both separate the
 * 'resource' from the 'resourceName' AND to subdivide the resourceName
 * further. For example, in S3 ARNs:
 *
 *    arn:aws:s3:::my_corporate_bucket/path/to/exampleobject.png
 *
 * After parsing the resourceName will not contain 'path/to/exampleobject.png'
 * but simply 'path'. This is a limitation because there is no slicing
 * functionality in CloudFormation templates.
 *
 * @param arnToken The input token that contains an ARN
 * @param sep The separator used to separate resource from resourceName
 * @param hasName Whether there is a name component in the ARN at all.
 * For example, SNS Topics ARNs have the 'resource' component contain the
 * topic name, and no 'resourceName' component.
 * @returns an ArnComponents object which allows access to the various
 * components of the ARN.
 */
function parseToken(arnToken, sep = '/', hasName = true) {
    // Arn ARN looks like:
    // arn:partition:service:region:account-id:resource
    // arn:partition:service:region:account-id:resourcetype/resource
    // arn:partition:service:region:account-id:resourcetype:resource
    // We need the 'hasName' argument because {Fn::Select}ing a nonexistent field
    // throws an error.
    const components = cfn_fn_1.Fn.split(':', arnToken);
    const partition = cfn_fn_1.Fn.select(1, components).toString();
    const service = cfn_fn_1.Fn.select(2, components).toString();
    const region = cfn_fn_1.Fn.select(3, components).toString();
    const account = cfn_fn_1.Fn.select(4, components).toString();
    if (sep === ':') {
        const resource = cfn_fn_1.Fn.select(5, components).toString();
        const resourceName = hasName ? cfn_fn_1.Fn.select(6, components).toString() : undefined;
        return { partition, service, region, account, resource, resourceName, sep };
    }
    else {
        const lastComponents = cfn_fn_1.Fn.split(sep, cfn_fn_1.Fn.select(5, components));
        const resource = cfn_fn_1.Fn.select(0, lastComponents).toString();
        const resourceName = hasName ? cfn_fn_1.Fn.select(1, lastComponents).toString() : undefined;
        return { partition, service, region, account, resource, resourceName, sep };
    }
}
//# sourceMappingURL=data:application/json;base64,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