"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Annotations = void 0;
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const DEPRECATIONS_SYMBOL = Symbol.for('@aws-cdk/core.deprecations');
/**
 * Includes API for attaching annotations such as warning messages to constructs.
 */
class Annotations {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Returns the annotations API for a construct scope.
     * @param scope The scope
     */
    static of(scope) {
        return new Annotations(scope);
    }
    /**
     * Adds a warning metadata entry to this construct.
     *
     * The CLI will display the warning when an app is synthesized, or fail if run
     * in --strict mode.
     *
     * @param message The warning message.
     */
    addWarning(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.WARN, message);
    }
    /**
     * Adds an info metadata entry to this construct.
     *
     * The CLI will display the info message when apps are synthesized.
     *
     * @param message The info message.
     */
    addInfo(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.INFO, message);
    }
    /**
     * Adds an { "error": <message> } metadata entry to this construct.
     * The toolkit will fail synthesis when errors are reported.
     * @param message The error message.
     */
    addError(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.ERROR, message);
    }
    /**
     * Adds a deprecation warning for a specific API.
     *
     * Deprecations will be added only once per construct as a warning and will be
     * deduplicated based on the `api`.
     *
     * If the environment variable `CDK_BLOCK_DEPRECATIONS` is set, this method
     * will throw an error instead with the deprecation message.
     *
     * @param api The API being deprecated in the format `module.Class.property`
     * (e.g. `@aws-cdk/core.Construct.node`).
     * @param message The deprecation message to display, with information about
     * alternatives.
     */
    addDeprecation(api, message) {
        const text = `The API ${api} is deprecated: ${message}. This API will be removed in the next major release`;
        // throw if CDK_BLOCK_DEPRECATIONS is set
        if (process.env.CDK_BLOCK_DEPRECATIONS) {
            throw new Error(`${this.scope.node.path}: ${text}`);
        }
        // de-dup based on api key
        const set = this.deprecationsReported;
        if (set.has(api)) {
            return;
        }
        this.addWarning(text);
        set.add(api);
    }
    /**
     * Adds a message metadata entry to the construct node, to be displayed by the CDK CLI.
     * @param level The message level
     * @param message The message itself
     */
    addMessage(level, message) {
        this.scope.node.addMetadata(level, message);
    }
    /**
     * Returns the set of deprecations reported on this construct.
     */
    get deprecationsReported() {
        let set = this.scope[DEPRECATIONS_SYMBOL];
        if (!set) {
            set = new Set();
            Object.defineProperty(this.scope, DEPRECATIONS_SYMBOL, { value: set });
        }
        return set;
    }
}
exports.Annotations = Annotations;
//# sourceMappingURL=data:application/json;base64,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