"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnInclude = void 0;
const core = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cfn_parse = require("../../core/lib/cfn-parse"); // Automatically re-written from '@aws-cdk/core/lib/cfn-parse'
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
/**
 * Construct to import an existing CloudFormation template file into a CDK application.
 * All resources defined in the template file can be retrieved by calling the {@link getResource} method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 */
class CfnInclude extends core.CfnElement {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        this.parameters = {};
        this.mappings = {};
        this.rules = {};
        this.hooks = {};
        this.outputs = {};
        this.nestedStacks = {};
        this.parametersToReplace = props.parameters || {};
        // read the template into a JS object
        this.template = futils.readYamlSync(props.templateFile);
        this.preserveLogicalIds = (_a = props.preserveLogicalIds) !== null && _a !== void 0 ? _a : true;
        // check if all user specified parameter values exist in the template
        for (const logicalId of Object.keys(this.parametersToReplace)) {
            if (!(logicalId in (this.template.Parameters || {}))) {
                throw new Error(`Parameter with logical ID '${logicalId}' was not found in the template`);
            }
        }
        // instantiate the Mappings
        this.mappingsScope = new core.Construct(this, '$Mappings');
        for (const mappingName of Object.keys(this.template.Mappings || {})) {
            this.createMapping(mappingName);
        }
        // instantiate all parameters
        for (const logicalId of Object.keys(this.template.Parameters || {})) {
            this.createParameter(logicalId);
        }
        // instantiate the conditions
        this.conditionsScope = new core.Construct(this, '$Conditions');
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.getOrCreateCondition(conditionName);
        }
        // instantiate the rules
        this.rulesScope = new core.Construct(this, '$Rules');
        for (const ruleName of Object.keys(this.template.Rules || {})) {
            this.createRule(ruleName);
        }
        this.nestedStacksToInclude = props.loadNestedStacks || {};
        // instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
        // verify that all nestedStacks have been instantiated
        for (const nestedStackId of Object.keys(props.loadNestedStacks || {})) {
            if (!(nestedStackId in this.resources)) {
                throw new Error(`Nested Stack with logical ID '${nestedStackId}' was not found in the template`);
            }
        }
        // instantiate the Hooks
        this.hooksScope = new core.Construct(this, '$Hooks');
        for (const hookName of Object.keys(this.template.Hooks || {})) {
            this.createHook(hookName);
        }
        const outputScope = new core.Construct(this, '$Ouputs');
        for (const logicalId of Object.keys(this.template.Outputs || {})) {
            this.createOutput(logicalId, outputScope);
        }
    }
    /**
     * Returns the low-level CfnResource from the template with the given logical ID.
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *     // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *     const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *     // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnCondition object from the 'Conditions'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnParameter object from the 'Parameters'
     * section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Parameter with the given name is not present in the template,
     * throws an exception.
     *
     * @param parameterName the name of the parameter to retrieve
     */
    getParameter(parameterName) {
        const ret = this.parameters[parameterName];
        if (!ret) {
            throw new Error(`Parameter with name '${parameterName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnMapping object from the 'Mappings' section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Mapping with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param mappingName the name of the Mapping in the template to retrieve
     */
    getMapping(mappingName) {
        const ret = this.mappings[mappingName];
        if (!ret) {
            throw new Error(`Mapping with name '${mappingName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnOutput object from the 'Outputs'
     * section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If an Output with the given name is not present in the template,
     * throws an exception.
     *
     * @param logicalId the name of the output to retrieve
     */
    getOutput(logicalId) {
        const ret = this.outputs[logicalId];
        if (!ret) {
            throw new Error(`Output with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnRule object from the 'Rules'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Rule with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param ruleName the name of the Rule in the CloudFormation template
     */
    getRule(ruleName) {
        const ret = this.rules[ruleName];
        if (!ret) {
            throw new Error(`Rule with name '${ruleName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnHook object from the 'Hooks'
     * section of the included CloudFormation template with the given logical ID.
     * Any modifications performed on the returned object will be reflected in the resulting CDK template.
     *
     * If a Hook with the given logical ID is not present in the template,
     * an exception will be thrown.
     *
     * @param hookLogicalId the logical ID of the Hook in the included CloudFormation template's 'Hooks' section
     */
    getHook(hookLogicalId) {
        const ret = this.hooks[hookLogicalId];
        if (!ret) {
            throw new Error(`Hook with logical ID '${hookLogicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns a loaded NestedStack with name logicalId.
     * For a nested stack to be returned by this method,
     * it must be specified either in the {@link CfnIncludeProps.loadNestedStacks} property,
     * or through the {@link loadNestedStack} method.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template
     */
    getNestedStack(logicalId) {
        if (!this.nestedStacks[logicalId]) {
            if (!this.template.Resources[logicalId]) {
                throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
            }
            else if (this.template.Resources[logicalId].Type !== 'AWS::CloudFormation::Stack') {
                throw new Error(`Resource with logical ID '${logicalId}' is not a CloudFormation Stack`);
            }
            else {
                throw new Error(`Nested Stack '${logicalId}' was not included in the parent template. ` +
                    'To retrieve an included nested stack, it must be specified either in the `loadNestedStacks` property, or through the `loadNestedStack` method');
            }
        }
        return this.nestedStacks[logicalId];
    }
    /**
     * Includes a template for a child stack inside of this parent template.
     * A child with this logical ID must exist in the template,
     * and be of type AWS::CloudFormation::Stack.
     * This is equivalent to specifying the value in the {@link CfnIncludeProps.loadNestedStacks}
     * property on object construction.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template
     * @param nestedStackProps the properties of the included child Stack
     * @returns the same {@link IncludedNestedStack} object that {@link getNestedStack} returns for this logical ID
     */
    loadNestedStack(logicalId, nestedStackProps) {
        if (logicalId in this.nestedStacks) {
            throw new Error(`Nested Stack '${logicalId}' was already included in its parent template`);
        }
        const cfnStack = this.resources[logicalId];
        if (!cfnStack) {
            throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
        }
        if (cfnStack instanceof core.CfnStack) {
            // delete the old CfnStack child - one will be created by the NestedStack object
            this.node.tryRemoveChild(logicalId);
            // remove the previously created CfnStack resource from the resources map
            delete this.resources[logicalId];
            // createNestedStack() (called by getOrCreateResource()) expects this to be filled
            this.nestedStacksToInclude[logicalId] = nestedStackProps;
            this.getOrCreateResource(logicalId);
            return this.nestedStacks[logicalId];
        }
        else {
            throw new Error(`Nested Stack with logical ID '${logicalId}' is not an AWS::CloudFormation::Stack resource`);
        }
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            const self = this;
            const finder = {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            };
            const cfnParser = new cfn_parse.CfnParser({
                finder,
                parameters: this.parametersToReplace,
            });
            switch (section) {
                case 'Conditions':
                case 'Mappings':
                case 'Resources':
                case 'Parameters':
                case 'Rules':
                case 'Hooks':
                case 'Outputs':
                    // these are rendered as a side effect of instantiating the L1s
                    break;
                default:
                    ret[section] = cfnParser.parseValue(this.template[section]);
            }
        }
        return ret;
    }
    createMapping(mappingName) {
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findCondition() { throw new Error('Referring to Conditions in Mapping definitions is not allowed'); },
                findMapping() { throw new Error('Referring to other Mappings in Mapping definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Mapping definitions is not allowed'); },
                findResource() { throw new Error('Using GetAtt expressions in Mapping definitions is not allowed'); },
            },
            parameters: {},
        });
        const cfnMapping = new core.CfnMapping(this.mappingsScope, mappingName, {
            mapping: cfnParser.parseValue(this.template.Mappings[mappingName]),
        });
        this.mappings[mappingName] = cfnMapping;
        this.overrideLogicalIdIfNeeded(cfnMapping, mappingName);
    }
    createParameter(logicalId) {
        if (logicalId in this.parametersToReplace) {
            return;
        }
        const expression = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt expressions in Parameter definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Parameter definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Parameter definitions is not allowed'); },
                findMapping() { throw new Error('Referring to Mappings in Parameter definitions is not allowed'); },
            },
            parameters: {},
        }).parseValue(this.template.Parameters[logicalId]);
        const cfnParameter = new core.CfnParameter(this, logicalId, {
            type: expression.Type,
            default: expression.Default,
            allowedPattern: expression.AllowedPattern,
            allowedValues: expression.AllowedValues,
            constraintDescription: expression.ConstraintDescription,
            description: expression.Description,
            maxLength: expression.MaxLength,
            maxValue: expression.MaxValue,
            minLength: expression.MinLength,
            minValue: expression.MinValue,
            noEcho: expression.NoEcho,
        });
        this.overrideLogicalIdIfNeeded(cfnParameter, logicalId);
        this.parameters[logicalId] = cfnParameter;
    }
    createRule(ruleName) {
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findRefTarget(refTarget) {
                    // only parameters can be referenced in Rules
                    return self.parameters[refTarget];
                },
                findResource() { throw new Error('Using GetAtt expressions in Rule definitions is not allowed'); },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
            context: cfn_parse.CfnParsingContext.RULES,
        });
        const ruleProperties = cfnParser.parseValue(this.template.Rules[ruleName]);
        const rule = new core.CfnRule(this.rulesScope, ruleName, {
            ruleCondition: ruleProperties.RuleCondition,
            assertions: ruleProperties.Assertions,
        });
        this.rules[ruleName] = rule;
        this.overrideLogicalIdIfNeeded(rule, ruleName);
    }
    createHook(hookName) {
        var _a;
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        });
        const hookAttributes = this.template.Hooks[hookName];
        let hook;
        switch (hookAttributes.Type) {
            case 'AWS::CodeDeploy::BlueGreen':
                hook = core.CfnCodeDeployBlueGreenHook._fromCloudFormation(this.hooksScope, hookName, hookAttributes, {
                    parser: cfnParser,
                });
                break;
            default: {
                const hookProperties = (_a = cfnParser.parseValue(hookAttributes.Properties)) !== null && _a !== void 0 ? _a : {};
                hook = new core.CfnHook(this.hooksScope, hookName, {
                    type: hookAttributes.Type,
                    properties: hookProperties,
                });
            }
        }
        this.hooks[hookName] = hook;
        this.overrideLogicalIdIfNeeded(hook, hookName);
    }
    createOutput(logicalId, scope) {
        const self = this;
        const outputAttributes = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        }).parseValue(this.template.Outputs[logicalId]);
        const cfnOutput = new core.CfnOutput(scope, logicalId, {
            value: outputAttributes.Value,
            description: outputAttributes.Description,
            exportName: outputAttributes.Export ? outputAttributes.Export.Name : undefined,
            condition: (() => {
                if (!outputAttributes.Condition) {
                    return undefined;
                }
                else if (this.conditions[outputAttributes.Condition]) {
                    return self.getCondition(outputAttributes.Condition);
                }
                throw new Error(`Output with name '${logicalId}' refers to a Condition with name ` +
                    `'${outputAttributes.Condition}' which was not found in this template`);
            })(),
        });
        this.overrideLogicalIdIfNeeded(cfnOutput, logicalId);
        this.outputs[logicalId] = cfnOutput;
    }
    getOrCreateCondition(conditionName) {
        if (conditionName in this.conditions) {
            return this.conditions[conditionName];
        }
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt in Condition definitions is not allowed'); },
                findRefTarget(elementName) {
                    // only Parameters can be referenced in the 'Conditions' section
                    return self.parameters[elementName];
                },
                findCondition(cName) {
                    return cName in (self.template.Conditions || {})
                        ? self.getOrCreateCondition(cName)
                        : undefined;
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            context: cfn_parse.CfnParsingContext.CONDITIONS,
            parameters: this.parametersToReplace,
        });
        const cfnCondition = new core.CfnCondition(this.conditionsScope, conditionName, {
            expression: cfnParser.parseValue(this.template.Conditions[conditionName]),
        });
        this.overrideLogicalIdIfNeeded(cfnCondition, conditionName);
        this.conditions[conditionName] = cfnCondition;
        return cfnCondition;
    }
    getOrCreateResource(logicalId) {
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const resourceAttributes = this.template.Resources[logicalId];
        // fail early for resource attributes we don't support yet
        const knownAttributes = [
            'Type', 'Properties', 'Condition', 'DependsOn', 'Metadata', 'Version',
            'CreationPolicy', 'UpdatePolicy', 'DeletionPolicy', 'UpdateReplacePolicy',
        ];
        for (const attribute of Object.keys(resourceAttributes)) {
            if (!knownAttributes.includes(attribute)) {
                throw new Error(`The '${attribute}' resource attribute is not supported by cloudformation-include yet. ` +
                    'Either remove it from the template, or use the CdkInclude class from the core package instead.');
            }
        }
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findMapping(mappingName) {
                return self.mappings[mappingName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId);
            },
            findRefTarget(elementName) {
                if (elementName in self.parameters) {
                    return self.parameters[elementName];
                }
                return this.findResource(elementName);
            },
        };
        const cfnParser = new cfn_parse.CfnParser({
            finder,
            parameters: this.parametersToReplace,
        });
        let l1Instance;
        if (this.nestedStacksToInclude[logicalId]) {
            l1Instance = this.createNestedStack(logicalId, cfnParser);
        }
        else {
            const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
            // The AWS::CloudFormation::CustomResource type corresponds to the CfnCustomResource class.
            // Unfortunately, it's quite useless; it only has a single property, ServiceToken.
            // For that reason, even the CustomResource class from @core doesn't use it!
            // So, special-case the handling of this one resource type
            if (l1ClassFqn && resourceAttributes.Type !== 'AWS::CloudFormation::CustomResource') {
                const options = {
                    parser: cfnParser,
                };
                const [moduleName, ...className] = l1ClassFqn.split('.');
                const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
                const jsClassFromModule = module[className.join('.')];
                l1Instance = jsClassFromModule._fromCloudFormation(this, logicalId, resourceAttributes, options);
            }
            else {
                l1Instance = new core.CfnResource(this, logicalId, {
                    type: resourceAttributes.Type,
                    properties: cfnParser.parseValue(resourceAttributes.Properties),
                });
                cfnParser.handleAttributes(l1Instance, resourceAttributes, logicalId);
            }
        }
        this.overrideLogicalIdIfNeeded(l1Instance, logicalId);
        this.resources[logicalId] = l1Instance;
        return l1Instance;
    }
    createNestedStack(nestedStackId, cfnParser) {
        const templateResources = this.template.Resources || {};
        const nestedStackAttributes = templateResources[nestedStackId] || {};
        if (nestedStackAttributes.Type !== 'AWS::CloudFormation::Stack') {
            throw new Error(`Nested Stack with logical ID '${nestedStackId}' is not an AWS::CloudFormation::Stack resource`);
        }
        if (nestedStackAttributes.CreationPolicy) {
            throw new Error('CreationPolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        if (nestedStackAttributes.UpdatePolicy) {
            throw new Error('UpdatePolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        const nestedStackProps = cfnParser.parseValue(nestedStackAttributes.Properties);
        const nestedStack = new core.NestedStack(this, nestedStackId, {
            parameters: this.parametersForNestedStack(nestedStackProps.Parameters, nestedStackId),
            notificationArns: nestedStackProps.NotificationArns,
            timeout: nestedStackProps.Timeout,
        });
        const template = new CfnInclude(nestedStack, nestedStackId, this.nestedStacksToInclude[nestedStackId]);
        this.nestedStacks[nestedStackId] = { stack: nestedStack, includedTemplate: template };
        // we know this is never undefined for nested stacks
        const nestedStackResource = nestedStack.nestedStackResource;
        cfnParser.handleAttributes(nestedStackResource, nestedStackAttributes, nestedStackId);
        return nestedStackResource;
    }
    parametersForNestedStack(parameters, nestedStackId) {
        var _a;
        if (parameters == null) {
            return undefined;
        }
        const parametersToReplace = (_a = this.nestedStacksToInclude[nestedStackId].parameters) !== null && _a !== void 0 ? _a : {};
        const ret = {};
        for (const paramName of Object.keys(parameters)) {
            if (!(paramName in parametersToReplace)) {
                ret[paramName] = parameters[paramName];
            }
        }
        return ret;
    }
    overrideLogicalIdIfNeeded(element, id) {
        if (this.preserveLogicalIds) {
            element.overrideLogicalId(id);
        }
    }
}
exports.CfnInclude = CfnInclude;
//# sourceMappingURL=data:application/json;base64,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