"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Canary = exports.Runtime = exports.Test = void 0;
const crypto = require("crypto");
const aws_cloudwatch_1 = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const synthetics_generated_1 = require("./synthetics.generated");
/**
 * Specify a test that the canary should run
 */
class Test {
    /**
     * Construct a Test property
     *
     * @param code The code that the canary should run
     * @param handler The handler of the canary
     */
    constructor(code, handler) {
        this.code = code;
        this.handler = handler;
    }
    /**
     * Specify a custom test with your own code
     *
     * @returns `Test` associated with the specified Code object
     * @param options The configuration options
     */
    static custom(options) {
        Test.validateHandler(options.handler);
        return new Test(options.code, options.handler);
    }
    /**
     * Verifies that the given handler ends in '.handler'. Returns the handler if successful and
     * throws an error if not.
     *
     * @param handler - the handler given by the user
     */
    static validateHandler(handler) {
        if (!handler.endsWith('.handler')) {
            throw new Error(`Canary Handler must end in '.handler' (${handler})`);
        }
        if (handler.length > 21) {
            throw new Error(`Canary Handler must be less than 21 characters (${handler})`);
        }
    }
}
exports.Test = Test;
/**
 * Runtime options for a canary
 */
class Runtime {
    /**
     * @param name The name of the runtime version
     */
    constructor(name) {
        this.name = name;
    }
}
exports.Runtime = Runtime;
/**
 * `syn-1.0` includes the following:
 *
 * - Synthetics library 1.0
 * - Synthetics handler code 1.0
 * - Lambda runtime Node.js 10.x
 * - Puppeteer-core version 1.14.0
 * - The Chromium version that matches Puppeteer-core 1.14.0
 */
Runtime.SYNTHETICS_1_0 = new Runtime('syn-1.0');
/**
 * Define a new Canary
 */
class Canary extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        if (props.canaryName && !cdk.Token.isUnresolved(props.canaryName)) {
            validateName(props.canaryName);
        }
        super(scope, id, {
            physicalName: props.canaryName || cdk.Lazy.stringValue({
                produce: () => this.generateUniqueName(),
            }),
        });
        this.artifactsBucket = (_b = (_a = props.artifactsBucketLocation) === null || _a === void 0 ? void 0 : _a.bucket) !== null && _b !== void 0 ? _b : new s3.Bucket(this, 'ArtifactsBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
        });
        this.role = (_c = props.role) !== null && _c !== void 0 ? _c : this.createDefaultRole((_d = props.artifactsBucketLocation) === null || _d === void 0 ? void 0 : _d.prefix);
        const resource = new synthetics_generated_1.CfnCanary(this, 'Resource', {
            artifactS3Location: this.artifactsBucket.s3UrlForObject((_e = props.artifactsBucketLocation) === null || _e === void 0 ? void 0 : _e.prefix),
            executionRoleArn: this.role.roleArn,
            startCanaryAfterCreation: (_f = props.startAfterCreation) !== null && _f !== void 0 ? _f : true,
            runtimeVersion: (_h = (_g = props.runtime) === null || _g === void 0 ? void 0 : _g.name) !== null && _h !== void 0 ? _h : Runtime.SYNTHETICS_1_0.name,
            name: this.physicalName,
            schedule: this.createSchedule(props),
            failureRetentionPeriod: (_j = props.failureRetentionPeriod) === null || _j === void 0 ? void 0 : _j.toDays(),
            successRetentionPeriod: (_k = props.successRetentionPeriod) === null || _k === void 0 ? void 0 : _k.toDays(),
            code: this.createCode(props),
        });
        this.canaryId = resource.attrId;
        this.canaryState = resource.attrState;
        this.canaryName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Measure the Duration of a single canary run, in seconds.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricDuration(options) {
        return this.metric('Duration', options);
    }
    /**
     * Measure the percentage of successful canary runs.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricSuccessPercent(options) {
        return this.metric('SuccessPercent', options);
    }
    /**
     * Measure the number of failed canary runs over a given time period.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricFailed(options) {
        return this.metric('Failed', options);
    }
    /**
     * @param metricName - the name of the metric
     * @param options - configuration options for the metric
     *
     * @returns a CloudWatch metric associated with the canary.
     * @default avg over 5 minutes
     */
    metric(metricName, options) {
        return new aws_cloudwatch_1.Metric({
            metricName,
            namespace: 'CloudWatchSynthetics',
            dimensions: { CanaryName: this.canaryName },
            statistic: 'avg',
            ...options,
        }).attachTo(this);
    }
    /**
     * Returns a default role for the canary
     */
    createDefaultRole(prefix) {
        // Created role will need these policies to run the Canary.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-synthetics-canary.html#cfn-synthetics-canary-executionrolearn
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:ListAllMyBuckets'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.arnForObjects(`${prefix ? prefix + '/*' : '*'}`)],
                    actions: ['s3:PutObject', 's3:GetBucketLocation'],
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['cloudwatch:PutMetricData'],
                    conditions: { StringEquals: { 'cloudwatch:namespace': 'CloudWatchSynthetics' } },
                }),
                new iam.PolicyStatement({
                    resources: ['arn:aws:logs:::*'],
                    actions: ['logs:CreateLogStream', 'logs:CreateLogGroup', 'logs:PutLogEvents'],
                }),
            ],
        });
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                canaryPolicy: policy,
            },
        });
    }
    /**
     * Returns the code object taken in by the canary resource.
     */
    createCode(props) {
        var _a, _b, _c;
        const codeConfig = {
            handler: props.test.handler,
            ...props.test.code.bind(this, props.test.handler),
        };
        return {
            handler: codeConfig.handler,
            script: codeConfig.inlineCode,
            s3Bucket: (_a = codeConfig.s3Location) === null || _a === void 0 ? void 0 : _a.bucketName,
            s3Key: (_b = codeConfig.s3Location) === null || _b === void 0 ? void 0 : _b.objectKey,
            s3ObjectVersion: (_c = codeConfig.s3Location) === null || _c === void 0 ? void 0 : _c.objectVersion,
        };
    }
    /**
     * Returns a canary schedule object
     */
    createSchedule(props) {
        var _a, _b, _c, _d;
        return {
            durationInSeconds: String(`${(_b = (_a = props.timeToLive) === null || _a === void 0 ? void 0 : _a.toSeconds()) !== null && _b !== void 0 ? _b : 0}`),
            expression: (_d = (_c = props.schedule) === null || _c === void 0 ? void 0 : _c.expressionString) !== null && _d !== void 0 ? _d : 'rate(5 minutes)',
        };
    }
    /**
     * Creates a unique name for the canary. The generated name is the physical ID of the canary.
     */
    generateUniqueName() {
        const name = this.node.uniqueId.toLowerCase().replace(' ', '-');
        if (name.length <= 21) {
            return name;
        }
        else {
            return name.substring(0, 15) + nameHash(name);
        }
    }
}
exports.Canary = Canary;
/**
 * Take a hash of the given name.
 *
 * @param name the name to be hashed
 */
function nameHash(name) {
    const md5 = crypto.createHash('sha256').update(name).digest('hex');
    return md5.slice(0, 6);
}
const nameRegex = /^[0-9a-z_\-]+$/;
/**
 * Verifies that the name fits the regex expression: ^[0-9a-z_\-]+$.
 *
 * @param name - the given name of the canary
 */
function validateName(name) {
    if (name.length > 21) {
        throw new Error(`Canary name is too large, must be between 1 and 21 characters, but is ${name.length} (got "${name}")`);
    }
    if (!nameRegex.test(name)) {
        throw new Error(`Canary name must be lowercase, numbers, hyphens, or underscores (got "${name}")`);
    }
}
//# sourceMappingURL=data:application/json;base64,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