"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateTrainingJob = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
const base_types_1 = require("./base-types");
/**
 * Class representing the SageMaker Create Training Job task.
 *
 * @experimental
 */
class SageMakerCreateTrainingJob extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        /**
         * Allows specify security group connections for instances of this fleet.
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateTrainingJob.SUPPORTED_INTEGRATION_PATTERNS);
        // set the default resource config if not defined.
        this.resourceConfig = props.resourceConfig || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
            volumeSize: core_1.Size.gibibytes(10),
        };
        // set the stopping condition if not defined
        this.stoppingCondition = props.stoppingCondition || {
            maxRuntime: core_1.Duration.hours(1),
        };
        // check that either algorithm name or image is defined
        if (!props.algorithmSpecification.algorithmName && !props.algorithmSpecification.trainingImage) {
            throw new Error('Must define either an algorithm name or training image URI in the algorithm specification');
        }
        // set the input mode to 'File' if not defined
        this.algorithmSpecification = props.algorithmSpecification.trainingInputMode
            ? props.algorithmSpecification
            : { ...props.algorithmSpecification, trainingInputMode: base_types_1.InputMode.FILE };
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.inputDataConfig = props.inputDataConfig.map((config) => {
            if (!config.dataSource.s3DataSource.s3DataType) {
                return {
                    ...config,
                    dataSource: { s3DataSource: { ...config.dataSource.s3DataSource, s3DataType: base_types_1.S3DataType.S3_PREFIX } },
                };
            }
            else {
                return config;
            }
        });
        // add the security groups to the connections object
        if (props.vpcConfig) {
            this.vpc = props.vpcConfig.vpc;
            this.subnets = props.vpcConfig.subnets ? this.vpc.selectSubnets(props.vpcConfig.subnets).subnetIds : this.vpc.selectSubnets().subnetIds;
        }
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     */
    get role() {
        if (this._role === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._role;
    }
    get grantPrincipal() {
        if (this._grantPrincipal === undefined) {
            throw new Error('Principal not available yet--use the object in a Task first');
        }
        return this._grantPrincipal;
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createTrainingJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    renderParameters() {
        return {
            TrainingJobName: this.props.trainingJobName,
            RoleArn: this._role.roleArn,
            ...this.renderAlgorithmSpecification(this.algorithmSpecification),
            ...this.renderInputDataConfig(this.inputDataConfig),
            ...this.renderOutputDataConfig(this.props.outputDataConfig),
            ...this.renderResourceConfig(this.resourceConfig),
            ...this.renderStoppingCondition(this.stoppingCondition),
            ...this.renderHyperparameters(this.props.hyperparameters),
            ...this.renderTags(this.props.tags),
            ...this.renderVpcConfig(this.props.vpcConfig),
        };
    }
    renderAlgorithmSpecification(spec) {
        return {
            AlgorithmSpecification: {
                TrainingInputMode: spec.trainingInputMode,
                ...(spec.trainingImage ? { TrainingImage: spec.trainingImage.bind(this).imageUri } : {}),
                ...(spec.algorithmName ? { AlgorithmName: spec.algorithmName } : {}),
                ...(spec.metricDefinitions
                    ? { MetricDefinitions: spec.metricDefinitions.map((metric) => ({ Name: metric.name, Regex: metric.regex })) }
                    : {}),
            },
        };
    }
    renderInputDataConfig(config) {
        return {
            InputDataConfig: config.map((channel) => ({
                ChannelName: channel.channelName,
                DataSource: {
                    S3DataSource: {
                        S3Uri: channel.dataSource.s3DataSource.s3Location.bind(this, { forReading: true }).uri,
                        S3DataType: channel.dataSource.s3DataSource.s3DataType,
                        ...(channel.dataSource.s3DataSource.s3DataDistributionType
                            ? { S3DataDistributionType: channel.dataSource.s3DataSource.s3DataDistributionType }
                            : {}),
                        ...(channel.dataSource.s3DataSource.attributeNames ? { AttributeNames: channel.dataSource.s3DataSource.attributeNames } : {}),
                    },
                },
                ...(channel.compressionType ? { CompressionType: channel.compressionType } : {}),
                ...(channel.contentType ? { ContentType: channel.contentType } : {}),
                ...(channel.inputMode ? { InputMode: channel.inputMode } : {}),
                ...(channel.recordWrapperType ? { RecordWrapperType: channel.recordWrapperType } : {}),
            })),
        };
    }
    renderOutputDataConfig(config) {
        return {
            OutputDataConfig: {
                S3OutputPath: config.s3OutputLocation.bind(this, { forWriting: true }).uri,
                ...(config.encryptionKey ? { KmsKeyId: config.encryptionKey.keyArn } : {}),
            },
        };
    }
    renderResourceConfig(config) {
        return {
            ResourceConfig: {
                InstanceCount: config.instanceCount,
                InstanceType: 'ml.' + config.instanceType,
                VolumeSizeInGB: config.volumeSize.toGibibytes(),
                ...(config.volumeEncryptionKey ? { VolumeKmsKeyId: config.volumeEncryptionKey.keyArn } : {}),
            },
        };
    }
    renderStoppingCondition(config) {
        return {
            StoppingCondition: {
                MaxRuntimeInSeconds: config.maxRuntime && config.maxRuntime.toSeconds(),
            },
        };
    }
    renderHyperparameters(params) {
        return params ? { HyperParameters: params } : {};
    }
    renderTags(tags) {
        return tags ? { Tags: Object.keys(tags).map((key) => ({ Key: key, Value: tags[key] })) } : {};
    }
    renderVpcConfig(config) {
        return config
            ? {
                VpcConfig: {
                    SecurityGroupIds: core_1.Lazy.listValue({ produce: () => this.securityGroups.map((sg) => sg.securityGroupId) }),
                    Subnets: this.subnets,
                },
            }
            : {};
    }
    makePolicyStatements() {
        // set the sagemaker role or create new one
        this._grantPrincipal = this._role =
            this.props.role ||
                new iam.Role(this, 'SagemakerRole', {
                    assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
                    inlinePolicies: {
                        CreateTrainingJob: new iam.PolicyDocument({
                            statements: [
                                new iam.PolicyStatement({
                                    actions: [
                                        'cloudwatch:PutMetricData',
                                        'logs:CreateLogStream',
                                        'logs:PutLogEvents',
                                        'logs:CreateLogGroup',
                                        'logs:DescribeLogStreams',
                                        'ecr:GetAuthorizationToken',
                                        ...(this.props.vpcConfig
                                            ? [
                                                'ec2:CreateNetworkInterface',
                                                'ec2:CreateNetworkInterfacePermission',
                                                'ec2:DeleteNetworkInterface',
                                                'ec2:DeleteNetworkInterfacePermission',
                                                'ec2:DescribeNetworkInterfaces',
                                                'ec2:DescribeVpcs',
                                                'ec2:DescribeDhcpOptions',
                                                'ec2:DescribeSubnets',
                                                'ec2:DescribeSecurityGroups',
                                            ]
                                            : []),
                                    ],
                                    resources: ['*'],
                                }),
                            ],
                        }),
                    },
                });
        if (this.props.outputDataConfig.encryptionKey) {
            this.props.outputDataConfig.encryptionKey.grantEncrypt(this._role);
        }
        if (this.props.resourceConfig && this.props.resourceConfig.volumeEncryptionKey) {
            this.props.resourceConfig.volumeEncryptionKey.grant(this._role, 'kms:CreateGrant');
        }
        // create a security group if not defined
        if (this.vpc && this.securityGroup === undefined) {
            this.securityGroup = new ec2.SecurityGroup(this, 'TrainJobSecurityGroup', {
                vpc: this.vpc,
            });
            this.connections.addSecurityGroup(this.securityGroup);
            this.securityGroups.push(this.securityGroup);
        }
        const stack = core_1.Stack.of(this);
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTrainingJob', 'sagemaker:DescribeTrainingJob', 'sagemaker:StopTrainingJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'training-job',
                        // If the job name comes from input, we cannot target the policy to a particular ARN prefix reliably...
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.trainingJobName) ? '*' : `${this.props.trainingJobName}*`,
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this._role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTrainingJobsRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.SageMakerCreateTrainingJob = SageMakerCreateTrainingJob;
SageMakerCreateTrainingJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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