"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InvocationType = exports.RunLambdaTask = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * Invoke a Lambda function as a Task
 *
 * OUTPUT: the output of this task is either the return value of Lambda's
 * Invoke call, or whatever the Lambda Function posted back using
 * `SendTaskSuccess/SendTaskFailure` in `waitForTaskToken` mode.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 * @deprecated Use `LambdaInvoke`
 */
class RunLambdaTask {
    constructor(lambdaFunction, props = {}) {
        this.lambdaFunction = lambdaFunction;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call Lambda.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is missing in payload (pass JsonPath.taskToken somewhere in payload)');
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('lambda', 'invoke', this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    resources: [this.lambdaFunction.functionArn],
                    actions: ['lambda:InvokeFunction'],
                })],
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: { LambdaFunctionArn: this.lambdaFunction.functionArn },
            parameters: {
                FunctionName: this.lambdaFunction.functionName,
                Payload: this.props.payload ? this.props.payload.value : sfn.TaskInput.fromDataAt('$').value,
                InvocationType: this.props.invocationType,
                ClientContext: this.props.clientContext,
                Qualifier: this.props.qualifier,
            },
        };
    }
}
exports.RunLambdaTask = RunLambdaTask;
/**
 * Invocation type of a Lambda
 */
var InvocationType;
(function (InvocationType) {
    /**
     * Invoke synchronously
     *
     * The API response includes the function response and additional data.
     */
    InvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Invoke asynchronously
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    InvocationType["EVENT"] = "Event";
    /**
     * TValidate parameter values and verify that the user or role has permission to invoke the function.
     */
    InvocationType["DRY_RUN"] = "DryRun";
})(InvocationType = exports.InvocationType || (exports.InvocationType = {}));
//# sourceMappingURL=data:application/json;base64,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