"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrTerminateCluster = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * A Step Functions Task to terminate an EMR Cluster.
 *
 * @experimental
 */
class EmrTerminateCluster extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrTerminateCluster.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.createPolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('elasticmapreduce', 'terminateCluster', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                ClusterId: this.props.clusterId,
            }),
        };
    }
    /**
     * This generates the PolicyStatements required by the Task to call TerminateCluster.
     */
    createPolicyStatements() {
        const stack = core_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: [
                    'elasticmapreduce:DescribeCluster',
                    'elasticmapreduce:TerminateJobFlows',
                ],
                resources: [`arn:aws:elasticmapreduce:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:cluster/*`],
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRTerminateJobFlowsRule',
                    })],
            }));
        }
        return policyStatements;
    }
}
exports.EmrTerminateCluster = EmrTerminateCluster;
EmrTerminateCluster.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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