"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpsRedirect = void 0;
const crypto = require("crypto");
const aws_certificatemanager_1 = require("../../aws-certificatemanager"); // Automatically re-written from '@aws-cdk/aws-certificatemanager'
const aws_cloudfront_1 = require("../../aws-cloudfront"); // Automatically re-written from '@aws-cdk/aws-cloudfront'
const aws_route53_1 = require("../../aws-route53"); // Automatically re-written from '@aws-cdk/aws-route53'
const aws_route53_targets_1 = require("../../aws-route53-targets"); // Automatically re-written from '@aws-cdk/aws-route53-targets'
const aws_s3_1 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Allows creating a domainA -> domainB redirect using CloudFront and S3.
 * You can specify multiple domains to be redirected.
 */
class HttpsRedirect extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const domainNames = (_a = props.recordNames) !== null && _a !== void 0 ? _a : [props.zone.zoneName];
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).parseArn(props.certificate.certificateArn).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error(`The certificate must be in the us-east-1 region and the certificate you provided is in ${certificateRegion}.`);
            }
        }
        const redirectCertArn = props.certificate ? props.certificate.certificateArn : new aws_certificatemanager_1.DnsValidatedCertificate(this, 'RedirectCertificate', {
            domainName: domainNames[0],
            subjectAlternativeNames: domainNames,
            hostedZone: props.zone,
            region: 'us-east-1',
        }).certificateArn;
        const redirectBucket = new aws_s3_1.Bucket(this, 'RedirectBucket', {
            websiteRedirect: {
                hostName: props.targetDomain,
                protocol: aws_s3_1.RedirectProtocol.HTTPS,
            },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        const redirectDist = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'RedirectDistribution', {
            defaultRootObject: '',
            originConfigs: [{
                    behaviors: [{ isDefaultBehavior: true }],
                    customOriginSource: {
                        domainName: redirectBucket.bucketWebsiteDomainName,
                        originProtocolPolicy: aws_cloudfront_1.OriginProtocolPolicy.HTTP_ONLY,
                    },
                }],
            aliasConfiguration: {
                acmCertRef: redirectCertArn,
                names: domainNames,
            },
            comment: `Redirect to ${props.targetDomain} from ${domainNames.join(', ')}`,
            priceClass: aws_cloudfront_1.PriceClass.PRICE_CLASS_ALL,
            viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        });
        domainNames.forEach((domainName) => {
            const hash = crypto.createHash('md5').update(domainName).digest('hex').substr(0, 6);
            const aliasProps = {
                recordName: domainName,
                zone: props.zone,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(redirectDist)),
            };
            new aws_route53_1.ARecord(this, `RedirectAliasRecord${hash}`, aliasProps);
            new aws_route53_1.AaaaRecord(this, `RedirectAliasRecordSix${hash}`, aliasProps);
        });
    }
}
exports.HttpsRedirect = HttpsRedirect;
//# sourceMappingURL=data:application/json;base64,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