"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.defaultDeletionProtection = exports.applyRemovalPolicy = exports.engineDescription = exports.setupS3ImportExport = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Validates the S3 import/export props and returns the import/export roles, if any.
 * If `combineRoles` is true, will reuse the import role for export (or vice versa) if possible.
 *
 * Notably, `combineRoles` is (by default) set to true for instances, but false for clusters.
 * This is because the `combineRoles` functionality is most applicable to instances and didn't exist
 * for the initial clusters implementation. To maintain backwards compatibility, it is set to false for clusters.
 */
function setupS3ImportExport(scope, props, combineRoles) {
    let s3ImportRole = props.s3ImportRole;
    let s3ExportRole = props.s3ExportRole;
    if (props.s3ImportBuckets && props.s3ImportBuckets.length > 0) {
        if (props.s3ImportRole) {
            throw new Error('Only one of s3ImportRole or s3ImportBuckets must be specified, not both.');
        }
        s3ImportRole = (combineRoles && s3ExportRole) ? s3ExportRole : new iam.Role(scope, 'S3ImportRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const bucket of props.s3ImportBuckets) {
            bucket.grantRead(s3ImportRole);
        }
    }
    if (props.s3ExportBuckets && props.s3ExportBuckets.length > 0) {
        if (props.s3ExportRole) {
            throw new Error('Only one of s3ExportRole or s3ExportBuckets must be specified, not both.');
        }
        s3ExportRole = (combineRoles && s3ImportRole) ? s3ImportRole : new iam.Role(scope, 'S3ExportRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const bucket of props.s3ExportBuckets) {
            bucket.grantReadWrite(s3ExportRole);
        }
    }
    return { s3ImportRole, s3ExportRole };
}
exports.setupS3ImportExport = setupS3ImportExport;
function engineDescription(engine) {
    var _a;
    return engine.engineType + (((_a = engine.engineVersion) === null || _a === void 0 ? void 0 : _a.fullVersion) ? `-${engine.engineVersion.fullVersion}` : '');
}
exports.engineDescription = engineDescription;
function applyRemovalPolicy(cfnDatabase, removalPolicy) {
    if (!removalPolicy) {
        // the default DeletionPolicy is 'Snapshot', which is fine,
        // but we should also make it 'Snapshot' for UpdateReplace policy
        cfnDatabase.cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.SNAPSHOT;
    }
    else {
        // just apply whatever removal policy the customer explicitly provided
        cfnDatabase.applyRemovalPolicy(removalPolicy);
    }
}
exports.applyRemovalPolicy = applyRemovalPolicy;
/**
 * By default, deletion protection is disabled.
 * Enable if explicitly provided or if the RemovalPolicy has been set to RETAIN
 */
function defaultDeletionProtection(deletionProtection, removalPolicy) {
    return deletionProtection !== undefined
        ? deletionProtection
        : (removalPolicy === core_1.RemovalPolicy.RETAIN ? true : undefined);
}
exports.defaultDeletionProtection = defaultDeletionProtection;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXRpbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInV0aWwudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsd0NBQXdDLENBQUMsbURBQW1EO0FBRTVGLHdDQUF5RixDQUFDLGdEQUFnRDtBQVMxSTs7Ozs7OztHQU9HO0FBQ0gsU0FBZ0IsbUJBQW1CLENBQUMsS0FBZ0IsRUFBRSxLQUFrQyxFQUFFLFlBQXNCO0lBSTVHLElBQUksWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUM7SUFDdEMsSUFBSSxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztJQUN0QyxJQUFJLEtBQUssQ0FBQyxlQUFlLElBQUksS0FBSyxDQUFDLGVBQWUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1FBQzNELElBQUksS0FBSyxDQUFDLFlBQVksRUFBRTtZQUNwQixNQUFNLElBQUksS0FBSyxDQUFDLDBFQUEwRSxDQUFDLENBQUM7U0FDL0Y7UUFDRCxZQUFZLEdBQUcsQ0FBQyxZQUFZLElBQUksWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7WUFDL0YsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDO1NBQzNELENBQUMsQ0FBQztRQUNILEtBQUssTUFBTSxNQUFNLElBQUksS0FBSyxDQUFDLGVBQWUsRUFBRTtZQUN4QyxNQUFNLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQyxDQUFDO1NBQ2xDO0tBQ0o7SUFDRCxJQUFJLEtBQUssQ0FBQyxlQUFlLElBQUksS0FBSyxDQUFDLGVBQWUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1FBQzNELElBQUksS0FBSyxDQUFDLFlBQVksRUFBRTtZQUNwQixNQUFNLElBQUksS0FBSyxDQUFDLDBFQUEwRSxDQUFDLENBQUM7U0FDL0Y7UUFDRCxZQUFZLEdBQUcsQ0FBQyxZQUFZLElBQUksWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7WUFDL0YsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDO1NBQzNELENBQUMsQ0FBQztRQUNILEtBQUssTUFBTSxNQUFNLElBQUksS0FBSyxDQUFDLGVBQWUsRUFBRTtZQUN4QyxNQUFNLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxDQUFDO1NBQ3ZDO0tBQ0o7SUFDRCxPQUFPLEVBQUUsWUFBWSxFQUFFLFlBQVksRUFBRSxDQUFDO0FBQzFDLENBQUM7QUE3QkQsa0RBNkJDO0FBQ0QsU0FBZ0IsaUJBQWlCLENBQUMsTUFBdUI7O0lBQ3JELE9BQU8sTUFBTSxDQUFDLFVBQVUsR0FBRyxDQUFDLE9BQUEsTUFBTSxDQUFDLGFBQWEsMENBQUUsV0FBVyxFQUFDLENBQUMsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxhQUFhLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO0FBQ2pILENBQUM7QUFGRCw4Q0FFQztBQUNELFNBQWdCLGtCQUFrQixDQUFDLFdBQXdCLEVBQUUsYUFBNkI7SUFDdEYsSUFBSSxDQUFDLGFBQWEsRUFBRTtRQUNoQiwyREFBMkQ7UUFDM0QsaUVBQWlFO1FBQ2pFLFdBQVcsQ0FBQyxVQUFVLENBQUMsbUJBQW1CLEdBQUcsd0JBQWlCLENBQUMsUUFBUSxDQUFDO0tBQzNFO1NBQ0k7UUFDRCxzRUFBc0U7UUFDdEUsV0FBVyxDQUFDLGtCQUFrQixDQUFDLGFBQWEsQ0FBQyxDQUFDO0tBQ2pEO0FBQ0wsQ0FBQztBQVZELGdEQVVDO0FBQ0Q7OztHQUdHO0FBQ0gsU0FBZ0IseUJBQXlCLENBQUMsa0JBQTRCLEVBQUUsYUFBNkI7SUFDakcsT0FBTyxrQkFBa0IsS0FBSyxTQUFTO1FBQ25DLENBQUMsQ0FBQyxrQkFBa0I7UUFDcEIsQ0FBQyxDQUFDLENBQUMsYUFBYSxLQUFLLG9CQUFhLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0FBQ3RFLENBQUM7QUFKRCw4REFJQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCAqIGFzIHMzIGZyb20gXCIuLi8uLi8uLi9hd3MtczNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMydcbmltcG9ydCB7IENvbnN0cnVjdCwgQ2ZuRGVsZXRpb25Qb2xpY3ksIENmblJlc291cmNlLCBSZW1vdmFsUG9saWN5IH0gZnJvbSBcIi4uLy4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBJSW5zdGFuY2VFbmdpbmUgfSBmcm9tICcuLi9pbnN0YW5jZS1lbmdpbmUnO1xuLyoqIENvbW1vbiBiYXNlIG9mIGBEYXRhYmFzZUluc3RhbmNlUHJvcHNgIGFuZCBgRGF0YWJhc2VDbHVzdGVyQmFzZVByb3BzYCB0aGF0IGhhcyBvbmx5IHRoZSBTMyBwcm9wcyAqL1xuZXhwb3J0IGludGVyZmFjZSBEYXRhYmFzZVMzSW1wb3J0RXhwb3J0UHJvcHMge1xuICAgIHJlYWRvbmx5IHMzSW1wb3J0Um9sZT86IGlhbS5JUm9sZTtcbiAgICByZWFkb25seSBzM0ltcG9ydEJ1Y2tldHM/OiBzMy5JQnVja2V0W107XG4gICAgcmVhZG9ubHkgczNFeHBvcnRSb2xlPzogaWFtLklSb2xlO1xuICAgIHJlYWRvbmx5IHMzRXhwb3J0QnVja2V0cz86IHMzLklCdWNrZXRbXTtcbn1cbi8qKlxuICogVmFsaWRhdGVzIHRoZSBTMyBpbXBvcnQvZXhwb3J0IHByb3BzIGFuZCByZXR1cm5zIHRoZSBpbXBvcnQvZXhwb3J0IHJvbGVzLCBpZiBhbnkuXG4gKiBJZiBgY29tYmluZVJvbGVzYCBpcyB0cnVlLCB3aWxsIHJldXNlIHRoZSBpbXBvcnQgcm9sZSBmb3IgZXhwb3J0IChvciB2aWNlIHZlcnNhKSBpZiBwb3NzaWJsZS5cbiAqXG4gKiBOb3RhYmx5LCBgY29tYmluZVJvbGVzYCBpcyAoYnkgZGVmYXVsdCkgc2V0IHRvIHRydWUgZm9yIGluc3RhbmNlcywgYnV0IGZhbHNlIGZvciBjbHVzdGVycy5cbiAqIFRoaXMgaXMgYmVjYXVzZSB0aGUgYGNvbWJpbmVSb2xlc2AgZnVuY3Rpb25hbGl0eSBpcyBtb3N0IGFwcGxpY2FibGUgdG8gaW5zdGFuY2VzIGFuZCBkaWRuJ3QgZXhpc3RcbiAqIGZvciB0aGUgaW5pdGlhbCBjbHVzdGVycyBpbXBsZW1lbnRhdGlvbi4gVG8gbWFpbnRhaW4gYmFja3dhcmRzIGNvbXBhdGliaWxpdHksIGl0IGlzIHNldCB0byBmYWxzZSBmb3IgY2x1c3RlcnMuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBzZXR1cFMzSW1wb3J0RXhwb3J0KHNjb3BlOiBDb25zdHJ1Y3QsIHByb3BzOiBEYXRhYmFzZVMzSW1wb3J0RXhwb3J0UHJvcHMsIGNvbWJpbmVSb2xlcz86IGJvb2xlYW4pOiB7XG4gICAgczNJbXBvcnRSb2xlPzogaWFtLklSb2xlO1xuICAgIHMzRXhwb3J0Um9sZT86IGlhbS5JUm9sZTtcbn0ge1xuICAgIGxldCBzM0ltcG9ydFJvbGUgPSBwcm9wcy5zM0ltcG9ydFJvbGU7XG4gICAgbGV0IHMzRXhwb3J0Um9sZSA9IHByb3BzLnMzRXhwb3J0Um9sZTtcbiAgICBpZiAocHJvcHMuczNJbXBvcnRCdWNrZXRzICYmIHByb3BzLnMzSW1wb3J0QnVja2V0cy5sZW5ndGggPiAwKSB7XG4gICAgICAgIGlmIChwcm9wcy5zM0ltcG9ydFJvbGUpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignT25seSBvbmUgb2YgczNJbXBvcnRSb2xlIG9yIHMzSW1wb3J0QnVja2V0cyBtdXN0IGJlIHNwZWNpZmllZCwgbm90IGJvdGguJyk7XG4gICAgICAgIH1cbiAgICAgICAgczNJbXBvcnRSb2xlID0gKGNvbWJpbmVSb2xlcyAmJiBzM0V4cG9ydFJvbGUpID8gczNFeHBvcnRSb2xlIDogbmV3IGlhbS5Sb2xlKHNjb3BlLCAnUzNJbXBvcnRSb2xlJywge1xuICAgICAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ3Jkcy5hbWF6b25hd3MuY29tJyksXG4gICAgICAgIH0pO1xuICAgICAgICBmb3IgKGNvbnN0IGJ1Y2tldCBvZiBwcm9wcy5zM0ltcG9ydEJ1Y2tldHMpIHtcbiAgICAgICAgICAgIGJ1Y2tldC5ncmFudFJlYWQoczNJbXBvcnRSb2xlKTtcbiAgICAgICAgfVxuICAgIH1cbiAgICBpZiAocHJvcHMuczNFeHBvcnRCdWNrZXRzICYmIHByb3BzLnMzRXhwb3J0QnVja2V0cy5sZW5ndGggPiAwKSB7XG4gICAgICAgIGlmIChwcm9wcy5zM0V4cG9ydFJvbGUpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignT25seSBvbmUgb2YgczNFeHBvcnRSb2xlIG9yIHMzRXhwb3J0QnVja2V0cyBtdXN0IGJlIHNwZWNpZmllZCwgbm90IGJvdGguJyk7XG4gICAgICAgIH1cbiAgICAgICAgczNFeHBvcnRSb2xlID0gKGNvbWJpbmVSb2xlcyAmJiBzM0ltcG9ydFJvbGUpID8gczNJbXBvcnRSb2xlIDogbmV3IGlhbS5Sb2xlKHNjb3BlLCAnUzNFeHBvcnRSb2xlJywge1xuICAgICAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ3Jkcy5hbWF6b25hd3MuY29tJyksXG4gICAgICAgIH0pO1xuICAgICAgICBmb3IgKGNvbnN0IGJ1Y2tldCBvZiBwcm9wcy5zM0V4cG9ydEJ1Y2tldHMpIHtcbiAgICAgICAgICAgIGJ1Y2tldC5ncmFudFJlYWRXcml0ZShzM0V4cG9ydFJvbGUpO1xuICAgICAgICB9XG4gICAgfVxuICAgIHJldHVybiB7IHMzSW1wb3J0Um9sZSwgczNFeHBvcnRSb2xlIH07XG59XG5leHBvcnQgZnVuY3Rpb24gZW5naW5lRGVzY3JpcHRpb24oZW5naW5lOiBJSW5zdGFuY2VFbmdpbmUpIHtcbiAgICByZXR1cm4gZW5naW5lLmVuZ2luZVR5cGUgKyAoZW5naW5lLmVuZ2luZVZlcnNpb24/LmZ1bGxWZXJzaW9uID8gYC0ke2VuZ2luZS5lbmdpbmVWZXJzaW9uLmZ1bGxWZXJzaW9ufWAgOiAnJyk7XG59XG5leHBvcnQgZnVuY3Rpb24gYXBwbHlSZW1vdmFsUG9saWN5KGNmbkRhdGFiYXNlOiBDZm5SZXNvdXJjZSwgcmVtb3ZhbFBvbGljeT86IFJlbW92YWxQb2xpY3kpOiB2b2lkIHtcbiAgICBpZiAoIXJlbW92YWxQb2xpY3kpIHtcbiAgICAgICAgLy8gdGhlIGRlZmF1bHQgRGVsZXRpb25Qb2xpY3kgaXMgJ1NuYXBzaG90Jywgd2hpY2ggaXMgZmluZSxcbiAgICAgICAgLy8gYnV0IHdlIHNob3VsZCBhbHNvIG1ha2UgaXQgJ1NuYXBzaG90JyBmb3IgVXBkYXRlUmVwbGFjZSBwb2xpY3lcbiAgICAgICAgY2ZuRGF0YWJhc2UuY2ZuT3B0aW9ucy51cGRhdGVSZXBsYWNlUG9saWN5ID0gQ2ZuRGVsZXRpb25Qb2xpY3kuU05BUFNIT1Q7XG4gICAgfVxuICAgIGVsc2Uge1xuICAgICAgICAvLyBqdXN0IGFwcGx5IHdoYXRldmVyIHJlbW92YWwgcG9saWN5IHRoZSBjdXN0b21lciBleHBsaWNpdGx5IHByb3ZpZGVkXG4gICAgICAgIGNmbkRhdGFiYXNlLmFwcGx5UmVtb3ZhbFBvbGljeShyZW1vdmFsUG9saWN5KTtcbiAgICB9XG59XG4vKipcbiAqIEJ5IGRlZmF1bHQsIGRlbGV0aW9uIHByb3RlY3Rpb24gaXMgZGlzYWJsZWQuXG4gKiBFbmFibGUgaWYgZXhwbGljaXRseSBwcm92aWRlZCBvciBpZiB0aGUgUmVtb3ZhbFBvbGljeSBoYXMgYmVlbiBzZXQgdG8gUkVUQUlOXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBkZWZhdWx0RGVsZXRpb25Qcm90ZWN0aW9uKGRlbGV0aW9uUHJvdGVjdGlvbj86IGJvb2xlYW4sIHJlbW92YWxQb2xpY3k/OiBSZW1vdmFsUG9saWN5KTogYm9vbGVhbiB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIGRlbGV0aW9uUHJvdGVjdGlvbiAhPT0gdW5kZWZpbmVkXG4gICAgICAgID8gZGVsZXRpb25Qcm90ZWN0aW9uXG4gICAgICAgIDogKHJlbW92YWxQb2xpY3kgPT09IFJlbW92YWxQb2xpY3kuUkVUQUlOID8gdHJ1ZSA6IHVuZGVmaW5lZCk7XG59XG4iXX0=