"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OptionGroup = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rds_generated_1 = require("./rds.generated");
/**
 * An option group
 */
class OptionGroup extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * The connections object for the options.
         */
        this.optionConnections = {};
        this.configurations = [];
        const majorEngineVersion = (_a = props.engine.engineVersion) === null || _a === void 0 ? void 0 : _a.majorVersion;
        if (!majorEngineVersion) {
            throw new Error("OptionGroup cannot be used with an engine that doesn't specify a version");
        }
        props.configurations.forEach(config => this.addConfiguration(config));
        const optionGroup = new rds_generated_1.CfnOptionGroup(this, 'Resource', {
            engineName: props.engine.engineType,
            majorEngineVersion,
            optionGroupDescription: props.description || `Option group for ${props.engine.engineType} ${majorEngineVersion}`,
            optionConfigurations: core_1.Lazy.anyValue({ produce: () => this.renderConfigurations(this.configurations) }),
        });
        this.optionGroupName = optionGroup.ref;
    }
    /**
     * Import an existing option group.
     */
    static fromOptionGroupName(scope, id, optionGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.optionGroupName = optionGroupName;
            }
            addConfiguration(_) { return false; }
        }
        return new Import(scope, id);
    }
    addConfiguration(configuration) {
        this.configurations.push(configuration);
        if (configuration.port) {
            if (!configuration.vpc) {
                throw new Error('`port` and `vpc` must be specified together.');
            }
            const securityGroups = configuration.securityGroups && configuration.securityGroups.length > 0
                ? configuration.securityGroups
                : [new ec2.SecurityGroup(this, `SecurityGroup${configuration.name}`, {
                        description: `Security group for ${configuration.name} option`,
                        vpc: configuration.vpc,
                    })];
            this.optionConnections[configuration.name] = new ec2.Connections({
                securityGroups: securityGroups,
                defaultPort: ec2.Port.tcp(configuration.port),
            });
        }
        return true;
    }
    /**
     * Renders the option configurations specifications.
     */
    renderConfigurations(configurations) {
        const configs = [];
        for (const config of configurations) {
            const securityGroups = config.vpc
                ? this.optionConnections[config.name].securityGroups.map(sg => sg.securityGroupId)
                : undefined;
            configs.push({
                optionName: config.name,
                optionSettings: config.settings && Object.entries(config.settings).map(([name, value]) => ({ name, value })),
                optionVersion: config.version,
                port: config.port,
                vpcSecurityGroupMemberships: securityGroups,
            });
        }
        return configs;
    }
}
exports.OptionGroup = OptionGroup;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3B0aW9uLWdyb3VwLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsib3B0aW9uLWdyb3VwLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHFDQUFxQyxDQUFDLG1EQUFtRDtBQUN6RixxQ0FBa0UsQ0FBQyxnREFBZ0Q7QUFFbkgsbURBQWlEO0FBa0ZqRDs7R0FFRztBQUNILE1BQWEsV0FBWSxTQUFRLGVBQVE7SUFzQnJDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBdUI7O1FBQzdELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFSckI7O1dBRUc7UUFDYSxzQkFBaUIsR0FFN0IsRUFBRSxDQUFDO1FBQ1UsbUJBQWMsR0FBMEIsRUFBRSxDQUFDO1FBR3hELE1BQU0sa0JBQWtCLFNBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhLDBDQUFFLFlBQVksQ0FBQztRQUNwRSxJQUFJLENBQUMsa0JBQWtCLEVBQUU7WUFDckIsTUFBTSxJQUFJLEtBQUssQ0FBQywwRUFBMEUsQ0FBQyxDQUFDO1NBQy9GO1FBQ0QsS0FBSyxDQUFDLGNBQWMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztRQUN0RSxNQUFNLFdBQVcsR0FBRyxJQUFJLDhCQUFjLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNyRCxVQUFVLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxVQUFVO1lBQ25DLGtCQUFrQjtZQUNsQixzQkFBc0IsRUFBRSxLQUFLLENBQUMsV0FBVyxJQUFJLG9CQUFvQixLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVUsSUFBSSxrQkFBa0IsRUFBRTtZQUNoSCxvQkFBb0IsRUFBRSxXQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLEVBQUUsQ0FBQztTQUN6RyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsZUFBZSxHQUFHLFdBQVcsQ0FBQyxHQUFHLENBQUM7SUFDM0MsQ0FBQztJQW5DRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxlQUF1QjtRQUNuRixNQUFNLE1BQU8sU0FBUSxlQUFRO1lBQTdCOztnQkFDb0Isb0JBQWUsR0FBRyxlQUFlLENBQUM7WUFFdEQsQ0FBQztZQURVLGdCQUFnQixDQUFDLENBQXNCLElBQUksT0FBTyxLQUFLLENBQUMsQ0FBQyxDQUFDO1NBQ3BFO1FBQ0QsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDakMsQ0FBQztJQTJCTSxnQkFBZ0IsQ0FBQyxhQUFrQztRQUN0RCxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUN4QyxJQUFJLGFBQWEsQ0FBQyxJQUFJLEVBQUU7WUFDcEIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLEVBQUU7Z0JBQ3BCLE1BQU0sSUFBSSxLQUFLLENBQUMsOENBQThDLENBQUMsQ0FBQzthQUNuRTtZQUNELE1BQU0sY0FBYyxHQUFHLGFBQWEsQ0FBQyxjQUFjLElBQUksYUFBYSxDQUFDLGNBQWMsQ0FBQyxNQUFNLEdBQUcsQ0FBQztnQkFDMUYsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxjQUFjO2dCQUM5QixDQUFDLENBQUMsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLGdCQUFnQixhQUFhLENBQUMsSUFBSSxFQUFFLEVBQUU7d0JBQzdELFdBQVcsRUFBRSxzQkFBc0IsYUFBYSxDQUFDLElBQUksU0FBUzt3QkFDOUQsR0FBRyxFQUFFLGFBQWEsQ0FBQyxHQUFHO3FCQUN6QixDQUFDLENBQUMsQ0FBQztZQUNaLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLEdBQUcsSUFBSSxHQUFHLENBQUMsV0FBVyxDQUFDO2dCQUM3RCxjQUFjLEVBQUUsY0FBYztnQkFDOUIsV0FBVyxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUM7YUFDaEQsQ0FBQyxDQUFDO1NBQ047UUFDRCxPQUFPLElBQUksQ0FBQztJQUNoQixDQUFDO0lBQ0Q7O09BRUc7SUFDSyxvQkFBb0IsQ0FBQyxjQUFxQztRQUM5RCxNQUFNLE9BQU8sR0FBaUQsRUFBRSxDQUFDO1FBQ2pFLEtBQUssTUFBTSxNQUFNLElBQUksY0FBYyxFQUFFO1lBQ2pDLE1BQU0sY0FBYyxHQUFHLE1BQU0sQ0FBQyxHQUFHO2dCQUM3QixDQUFDLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQztnQkFDbEYsQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUNoQixPQUFPLENBQUMsSUFBSSxDQUFDO2dCQUNULFVBQVUsRUFBRSxNQUFNLENBQUMsSUFBSTtnQkFDdkIsY0FBYyxFQUFFLE1BQU0sQ0FBQyxRQUFRLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsRUFBRSxJQUFJLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztnQkFDNUcsYUFBYSxFQUFFLE1BQU0sQ0FBQyxPQUFPO2dCQUM3QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUk7Z0JBQ2pCLDJCQUEyQixFQUFFLGNBQWM7YUFDOUMsQ0FBQyxDQUFDO1NBQ047UUFDRCxPQUFPLE9BQU8sQ0FBQztJQUNuQixDQUFDO0NBQ0o7QUEzRUQsa0NBMkVDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZWMyIGZyb20gXCIuLi8uLi9hd3MtZWMyXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJ1xuaW1wb3J0IHsgQ29uc3RydWN0LCBJUmVzb3VyY2UsIExhenksIFJlc291cmNlIH0gZnJvbSBcIi4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBJSW5zdGFuY2VFbmdpbmUgfSBmcm9tICcuL2luc3RhbmNlLWVuZ2luZSc7XG5pbXBvcnQgeyBDZm5PcHRpb25Hcm91cCB9IGZyb20gJy4vcmRzLmdlbmVyYXRlZCc7XG4vKipcbiAqIEFuIG9wdGlvbiBncm91cFxuICovXG5leHBvcnQgaW50ZXJmYWNlIElPcHRpb25Hcm91cCBleHRlbmRzIElSZXNvdXJjZSB7XG4gICAgLyoqXG4gICAgICogVGhlIG5hbWUgb2YgdGhlIG9wdGlvbiBncm91cC5cbiAgICAgKlxuICAgICAqIEBhdHRyaWJ1dGVcbiAgICAgKi9cbiAgICByZWFkb25seSBvcHRpb25Hcm91cE5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBBZGRzIGEgY29uZmlndXJhdGlvbiB0byB0aGlzIE9wdGlvbkdyb3VwLlxuICAgICAqIFRoaXMgbWV0aG9kIGlzIGEgbm8tb3AgZm9yIGFuIGltcG9ydGVkIE9wdGlvbkdyb3VwLlxuICAgICAqXG4gICAgICogQHJldHVybnMgdHJ1ZSBpZiB0aGUgT3B0aW9uQ29uZmlndXJhdGlvbiB3YXMgc3VjY2Vzc2Z1bGx5IGFkZGVkLlxuICAgICAqL1xuICAgIGFkZENvbmZpZ3VyYXRpb24oY29uZmlndXJhdGlvbjogT3B0aW9uQ29uZmlndXJhdGlvbik6IGJvb2xlYW47XG59XG4vKipcbiAqIENvbmZpZ3VyYXRpb24gcHJvcGVydGllcyBmb3IgYW4gb3B0aW9uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE9wdGlvbkNvbmZpZ3VyYXRpb24ge1xuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSBvcHRpb24uXG4gICAgICovXG4gICAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBzZXR0aW5ncyBmb3IgdGhlIG9wdGlvbi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gbm8gc2V0dGluZ3NcbiAgICAgKi9cbiAgICByZWFkb25seSBzZXR0aW5ncz86IHtcbiAgICAgICAgW25hbWU6IHN0cmluZ106IHN0cmluZztcbiAgICB9O1xuICAgIC8qKlxuICAgICAqIFRoZSB2ZXJzaW9uIGZvciB0aGUgb3B0aW9uLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBubyB2ZXJzaW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgdmVyc2lvbj86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgcG9ydCBudW1iZXIgdGhhdCB0aGlzIG9wdGlvbiB1c2VzLiBJZiBgcG9ydGAgaXMgc3BlY2lmaWVkIHRoZW4gYHZwY2BcbiAgICAgKiBtdXN0IGFsc28gYmUgc3BlY2lmaWVkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBubyBwb3J0XG4gICAgICovXG4gICAgcmVhZG9ubHkgcG9ydD86IG51bWJlcjtcbiAgICAvKipcbiAgICAgKiBUaGUgVlBDIHdoZXJlIGEgc2VjdXJpdHkgZ3JvdXAgc2hvdWxkIGJlIGNyZWF0ZWQgZm9yIHRoaXMgb3B0aW9uLiBJZiBgdnBjYFxuICAgICAqIGlzIHNwZWNpZmllZCB0aGVuIGBwb3J0YCBtdXN0IGFsc28gYmUgc3BlY2lmaWVkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBubyBWUENcbiAgICAgKi9cbiAgICByZWFkb25seSB2cGM/OiBlYzIuSVZwYztcbiAgICAvKipcbiAgICAgKiBPcHRpb25hbCBsaXN0IG9mIHNlY3VyaXR5IGdyb3VwcyB0byB1c2UgZm9yIHRoaXMgb3B0aW9uLCBpZiBgdnBjYCBpcyBzcGVjaWZpZWQuXG4gICAgICogSWYgbm8gZ3JvdXBzIGFyZSBwcm92aWRlZCwgYSBkZWZhdWx0IG9uZSB3aWxsIGJlIGNyZWF0ZWQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIGEgZGVmYXVsdCBncm91cCB3aWxsIGJlIGNyZWF0ZWQgaWYgYHBvcnRgIG9yIGB2cGNgIGFyZSBzcGVjaWZpZWQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cHM/OiBlYzIuSVNlY3VyaXR5R3JvdXBbXTtcbn1cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGFuIE9wdGlvbkdyb3VwLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE9wdGlvbkdyb3VwUHJvcHMge1xuICAgIC8qKlxuICAgICAqIFRoZSBkYXRhYmFzZSBlbmdpbmUgdGhhdCB0aGlzIG9wdGlvbiBncm91cCBpcyBhc3NvY2lhdGVkIHdpdGguXG4gICAgICovXG4gICAgcmVhZG9ubHkgZW5naW5lOiBJSW5zdGFuY2VFbmdpbmU7XG4gICAgLyoqXG4gICAgICogQSBkZXNjcmlwdGlvbiBvZiB0aGUgb3B0aW9uIGdyb3VwLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgYSBDREsgZ2VuZXJhdGVkIGRlc2NyaXB0aW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGNvbmZpZ3VyYXRpb25zIGZvciB0aGlzIG9wdGlvbiBncm91cC5cbiAgICAgKi9cbiAgICByZWFkb25seSBjb25maWd1cmF0aW9uczogT3B0aW9uQ29uZmlndXJhdGlvbltdO1xufVxuLyoqXG4gKiBBbiBvcHRpb24gZ3JvdXBcbiAqL1xuZXhwb3J0IGNsYXNzIE9wdGlvbkdyb3VwIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJT3B0aW9uR3JvdXAge1xuICAgIC8qKlxuICAgICAqIEltcG9ydCBhbiBleGlzdGluZyBvcHRpb24gZ3JvdXAuXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmcm9tT3B0aW9uR3JvdXBOYW1lKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIG9wdGlvbkdyb3VwTmFtZTogc3RyaW5nKTogSU9wdGlvbkdyb3VwIHtcbiAgICAgICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzb3VyY2Uge1xuICAgICAgICAgICAgcHVibGljIHJlYWRvbmx5IG9wdGlvbkdyb3VwTmFtZSA9IG9wdGlvbkdyb3VwTmFtZTtcbiAgICAgICAgICAgIHB1YmxpYyBhZGRDb25maWd1cmF0aW9uKF86IE9wdGlvbkNvbmZpZ3VyYXRpb24pIHsgcmV0dXJuIGZhbHNlOyB9XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogVGhlIG5hbWUgb2YgdGhlIG9wdGlvbiBncm91cC5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgb3B0aW9uR3JvdXBOYW1lOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGNvbm5lY3Rpb25zIG9iamVjdCBmb3IgdGhlIG9wdGlvbnMuXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IG9wdGlvbkNvbm5lY3Rpb25zOiB7XG4gICAgICAgIFtrZXk6IHN0cmluZ106IGVjMi5Db25uZWN0aW9ucztcbiAgICB9ID0ge307XG4gICAgcHJpdmF0ZSByZWFkb25seSBjb25maWd1cmF0aW9uczogT3B0aW9uQ29uZmlndXJhdGlvbltdID0gW107XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IE9wdGlvbkdyb3VwUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgY29uc3QgbWFqb3JFbmdpbmVWZXJzaW9uID0gcHJvcHMuZW5naW5lLmVuZ2luZVZlcnNpb24/Lm1ham9yVmVyc2lvbjtcbiAgICAgICAgaWYgKCFtYWpvckVuZ2luZVZlcnNpb24pIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihcIk9wdGlvbkdyb3VwIGNhbm5vdCBiZSB1c2VkIHdpdGggYW4gZW5naW5lIHRoYXQgZG9lc24ndCBzcGVjaWZ5IGEgdmVyc2lvblwiKTtcbiAgICAgICAgfVxuICAgICAgICBwcm9wcy5jb25maWd1cmF0aW9ucy5mb3JFYWNoKGNvbmZpZyA9PiB0aGlzLmFkZENvbmZpZ3VyYXRpb24oY29uZmlnKSk7XG4gICAgICAgIGNvbnN0IG9wdGlvbkdyb3VwID0gbmV3IENmbk9wdGlvbkdyb3VwKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgICAgICAgIGVuZ2luZU5hbWU6IHByb3BzLmVuZ2luZS5lbmdpbmVUeXBlLFxuICAgICAgICAgICAgbWFqb3JFbmdpbmVWZXJzaW9uLFxuICAgICAgICAgICAgb3B0aW9uR3JvdXBEZXNjcmlwdGlvbjogcHJvcHMuZGVzY3JpcHRpb24gfHwgYE9wdGlvbiBncm91cCBmb3IgJHtwcm9wcy5lbmdpbmUuZW5naW5lVHlwZX0gJHttYWpvckVuZ2luZVZlcnNpb259YCxcbiAgICAgICAgICAgIG9wdGlvbkNvbmZpZ3VyYXRpb25zOiBMYXp5LmFueVZhbHVlKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5yZW5kZXJDb25maWd1cmF0aW9ucyh0aGlzLmNvbmZpZ3VyYXRpb25zKSB9KSxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMub3B0aW9uR3JvdXBOYW1lID0gb3B0aW9uR3JvdXAucmVmO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkQ29uZmlndXJhdGlvbihjb25maWd1cmF0aW9uOiBPcHRpb25Db25maWd1cmF0aW9uKSB7XG4gICAgICAgIHRoaXMuY29uZmlndXJhdGlvbnMucHVzaChjb25maWd1cmF0aW9uKTtcbiAgICAgICAgaWYgKGNvbmZpZ3VyYXRpb24ucG9ydCkge1xuICAgICAgICAgICAgaWYgKCFjb25maWd1cmF0aW9uLnZwYykge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignYHBvcnRgIGFuZCBgdnBjYCBtdXN0IGJlIHNwZWNpZmllZCB0b2dldGhlci4nKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGNvbnN0IHNlY3VyaXR5R3JvdXBzID0gY29uZmlndXJhdGlvbi5zZWN1cml0eUdyb3VwcyAmJiBjb25maWd1cmF0aW9uLnNlY3VyaXR5R3JvdXBzLmxlbmd0aCA+IDBcbiAgICAgICAgICAgICAgICA/IGNvbmZpZ3VyYXRpb24uc2VjdXJpdHlHcm91cHNcbiAgICAgICAgICAgICAgICA6IFtuZXcgZWMyLlNlY3VyaXR5R3JvdXAodGhpcywgYFNlY3VyaXR5R3JvdXAke2NvbmZpZ3VyYXRpb24ubmFtZX1gLCB7XG4gICAgICAgICAgICAgICAgICAgICAgICBkZXNjcmlwdGlvbjogYFNlY3VyaXR5IGdyb3VwIGZvciAke2NvbmZpZ3VyYXRpb24ubmFtZX0gb3B0aW9uYCxcbiAgICAgICAgICAgICAgICAgICAgICAgIHZwYzogY29uZmlndXJhdGlvbi52cGMsXG4gICAgICAgICAgICAgICAgICAgIH0pXTtcbiAgICAgICAgICAgIHRoaXMub3B0aW9uQ29ubmVjdGlvbnNbY29uZmlndXJhdGlvbi5uYW1lXSA9IG5ldyBlYzIuQ29ubmVjdGlvbnMoe1xuICAgICAgICAgICAgICAgIHNlY3VyaXR5R3JvdXBzOiBzZWN1cml0eUdyb3VwcyxcbiAgICAgICAgICAgICAgICBkZWZhdWx0UG9ydDogZWMyLlBvcnQudGNwKGNvbmZpZ3VyYXRpb24ucG9ydCksXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gdHJ1ZTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmVuZGVycyB0aGUgb3B0aW9uIGNvbmZpZ3VyYXRpb25zIHNwZWNpZmljYXRpb25zLlxuICAgICAqL1xuICAgIHByaXZhdGUgcmVuZGVyQ29uZmlndXJhdGlvbnMoY29uZmlndXJhdGlvbnM6IE9wdGlvbkNvbmZpZ3VyYXRpb25bXSk6IENmbk9wdGlvbkdyb3VwLk9wdGlvbkNvbmZpZ3VyYXRpb25Qcm9wZXJ0eVtdIHtcbiAgICAgICAgY29uc3QgY29uZmlnczogQ2ZuT3B0aW9uR3JvdXAuT3B0aW9uQ29uZmlndXJhdGlvblByb3BlcnR5W10gPSBbXTtcbiAgICAgICAgZm9yIChjb25zdCBjb25maWcgb2YgY29uZmlndXJhdGlvbnMpIHtcbiAgICAgICAgICAgIGNvbnN0IHNlY3VyaXR5R3JvdXBzID0gY29uZmlnLnZwY1xuICAgICAgICAgICAgICAgID8gdGhpcy5vcHRpb25Db25uZWN0aW9uc1tjb25maWcubmFtZV0uc2VjdXJpdHlHcm91cHMubWFwKHNnID0+IHNnLnNlY3VyaXR5R3JvdXBJZClcbiAgICAgICAgICAgICAgICA6IHVuZGVmaW5lZDtcbiAgICAgICAgICAgIGNvbmZpZ3MucHVzaCh7XG4gICAgICAgICAgICAgICAgb3B0aW9uTmFtZTogY29uZmlnLm5hbWUsXG4gICAgICAgICAgICAgICAgb3B0aW9uU2V0dGluZ3M6IGNvbmZpZy5zZXR0aW5ncyAmJiBPYmplY3QuZW50cmllcyhjb25maWcuc2V0dGluZ3MpLm1hcCgoW25hbWUsIHZhbHVlXSkgPT4gKHsgbmFtZSwgdmFsdWUgfSkpLFxuICAgICAgICAgICAgICAgIG9wdGlvblZlcnNpb246IGNvbmZpZy52ZXJzaW9uLFxuICAgICAgICAgICAgICAgIHBvcnQ6IGNvbmZpZy5wb3J0LFxuICAgICAgICAgICAgICAgIHZwY1NlY3VyaXR5R3JvdXBNZW1iZXJzaGlwczogc2VjdXJpdHlHcm91cHMsXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gY29uZmlncztcbiAgICB9XG59XG4iXX0=