"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstanceEngine = exports.SqlServerEngineVersion = exports.OracleEngineVersion = exports.OracleLegacyEngineVersion = exports.PostgresEngineVersion = exports.MysqlEngineVersion = exports.MariaDbEngineVersion = void 0;
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const option_group_1 = require("./option-group");
class InstanceEngineBase {
    constructor(props) {
        var _a;
        this.engineType = props.engineType;
        this.features = props.features;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.engineVersion = props.version;
        this.parameterGroupFamily = (_a = props.parameterGroupFamily) !== null && _a !== void 0 ? _a : (this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined);
    }
    bindToInstance(_scope, options) {
        if (options.timezone && !this.supportsTimezone) {
            throw new Error(`timezone property can not be configured for ${this.engineType}`);
        }
        return {
            features: this.features,
            optionGroup: options.optionGroup,
        };
    }
    /** Defines whether this Instance Engine can support timezone properties. */
    get supportsTimezone() { return false; }
}
/**
 * The versions for the MariaDB instance engines
 * (those returned by {@link DatabaseInstanceEngine.mariaDb}).
 */
class MariaDbEngineVersion {
    constructor(mariaDbFullVersion, mariaDbMajorVersion) {
        this.mariaDbFullVersion = mariaDbFullVersion;
        this.mariaDbMajorVersion = mariaDbMajorVersion;
    }
    /**
     * Create a new MariaDbEngineVersion with an arbitrary version.
     *
     * @param mariaDbFullVersion the full version string,
     *   for example "10.5.28"
     * @param mariaDbMajorVersion the major version of the engine,
     *   for example "10.5"
     */
    static of(mariaDbFullVersion, mariaDbMajorVersion) {
        return new MariaDbEngineVersion(mariaDbFullVersion, mariaDbMajorVersion);
    }
}
exports.MariaDbEngineVersion = MariaDbEngineVersion;
/** Version "10.0" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_0 = MariaDbEngineVersion.of('10.0', '10.0');
/** Version "10.0.17". */
MariaDbEngineVersion.VER_10_0_17 = MariaDbEngineVersion.of('10.0.17', '10.0');
/** Version "10.0.24". */
MariaDbEngineVersion.VER_10_0_24 = MariaDbEngineVersion.of('10.0.24', '10.0');
/** Version "10.0.28". */
MariaDbEngineVersion.VER_10_0_28 = MariaDbEngineVersion.of('10.0.28', '10.0');
/** Version "10.0.31". */
MariaDbEngineVersion.VER_10_0_31 = MariaDbEngineVersion.of('10.0.31', '10.0');
/** Version "10.0.32". */
MariaDbEngineVersion.VER_10_0_32 = MariaDbEngineVersion.of('10.0.32', '10.0');
/** Version "10.0.34". */
MariaDbEngineVersion.VER_10_0_34 = MariaDbEngineVersion.of('10.0.34', '10.0');
/** Version "10.0.35". */
MariaDbEngineVersion.VER_10_0_35 = MariaDbEngineVersion.of('10.0.35', '10.0');
/** Version "10.1" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_1 = MariaDbEngineVersion.of('10.1', '10.1');
/** Version "10.1.14". */
MariaDbEngineVersion.VER_10_1_14 = MariaDbEngineVersion.of('10.1.14', '10.1');
/** Version "10.1.19". */
MariaDbEngineVersion.VER_10_1_19 = MariaDbEngineVersion.of('10.1.19', '10.1');
/** Version "10.1.23". */
MariaDbEngineVersion.VER_10_1_23 = MariaDbEngineVersion.of('10.1.23', '10.1');
/** Version "10.1.26". */
MariaDbEngineVersion.VER_10_1_26 = MariaDbEngineVersion.of('10.1.26', '10.1');
/** Version "10.1.31". */
MariaDbEngineVersion.VER_10_1_31 = MariaDbEngineVersion.of('10.1.31', '10.1');
/** Version "10.1.34". */
MariaDbEngineVersion.VER_10_1_34 = MariaDbEngineVersion.of('10.1.34', '10.1');
/** Version "10.2" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_2 = MariaDbEngineVersion.of('10.2', '10.2');
/** Version "10.2.11". */
MariaDbEngineVersion.VER_10_2_11 = MariaDbEngineVersion.of('10.2.11', '10.2');
/** Version "10.2.12". */
MariaDbEngineVersion.VER_10_2_12 = MariaDbEngineVersion.of('10.2.12', '10.2');
/** Version "10.2.15". */
MariaDbEngineVersion.VER_10_2_15 = MariaDbEngineVersion.of('10.2.15', '10.2');
/** Version "10.2.21". */
MariaDbEngineVersion.VER_10_2_21 = MariaDbEngineVersion.of('10.2.21', '10.2');
/** Version "10.3" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_3 = MariaDbEngineVersion.of('10.3', '10.3');
/** Version "10.3.8". */
MariaDbEngineVersion.VER_10_3_8 = MariaDbEngineVersion.of('10.3.8', '10.3');
/** Version "10.3.13". */
MariaDbEngineVersion.VER_10_3_13 = MariaDbEngineVersion.of('10.3.13', '10.3');
/** Version "10.3.20". */
MariaDbEngineVersion.VER_10_3_20 = MariaDbEngineVersion.of('10.3.20', '10.3');
/** Version "10.3.23". */
MariaDbEngineVersion.VER_10_3_23 = MariaDbEngineVersion.of('10.3.23', '10.3');
/** Version "10.4" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_4 = MariaDbEngineVersion.of('10.4', '10.4');
/** Version "10.4.8". */
MariaDbEngineVersion.VER_10_4_8 = MariaDbEngineVersion.of('10.4.8', '10.4');
/** Version "10.4.13". */
MariaDbEngineVersion.VER_10_4_13 = MariaDbEngineVersion.of('10.4.13', '10.4');
class MariaDbInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'mariadb',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MARIADB_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.mariaDbFullVersion,
                    majorVersion: version.mariaDbMajorVersion,
                }
                : undefined,
        });
    }
    bindToInstance(scope, options) {
        if (options.domain) {
            throw new Error(`domain property cannot be configured for ${this.engineType}`);
        }
        return super.bindToInstance(scope, options);
    }
}
/**
 * The versions for the MySQL instance engines
 * (those returned by {@link DatabaseInstanceEngine.mysql}).
 */
class MysqlEngineVersion {
    constructor(mysqlFullVersion, mysqlMajorVersion) {
        this.mysqlFullVersion = mysqlFullVersion;
        this.mysqlMajorVersion = mysqlMajorVersion;
    }
    /**
     * Create a new MysqlEngineVersion with an arbitrary version.
     *
     * @param mysqlFullVersion the full version string,
     *   for example "8.1.43"
     * @param mysqlMajorVersion the major version of the engine,
     *   for example "8.1"
     */
    static of(mysqlFullVersion, mysqlMajorVersion) {
        return new MysqlEngineVersion(mysqlFullVersion, mysqlMajorVersion);
    }
}
exports.MysqlEngineVersion = MysqlEngineVersion;
/** Version "5.5" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_5_5 = MysqlEngineVersion.of('5.5', '5.5');
/** Version "5.5.46". */
MysqlEngineVersion.VER_5_5_46 = MysqlEngineVersion.of('5.5.46', '5.5');
/** Version "5.5.53". */
MysqlEngineVersion.VER_5_5_53 = MysqlEngineVersion.of('5.5.53', '5.5');
/** Version "5.5.57". */
MysqlEngineVersion.VER_5_5_57 = MysqlEngineVersion.of('5.5.57', '5.5');
/** Version "5.5.59". */
MysqlEngineVersion.VER_5_5_59 = MysqlEngineVersion.of('5.5.59', '5.5');
/** Version "5.5.61". */
MysqlEngineVersion.VER_5_5_61 = MysqlEngineVersion.of('5.5.61', '5.5');
/** Version "5.6" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_5_6 = MysqlEngineVersion.of('5.6', '5.6');
/** Version "5.6.34". */
MysqlEngineVersion.VER_5_6_34 = MysqlEngineVersion.of('5.6.34', '5.6');
/** Version "5.6.35". */
MysqlEngineVersion.VER_5_6_35 = MysqlEngineVersion.of('5.6.35', '5.6');
/** Version "5.6.37". */
MysqlEngineVersion.VER_5_6_37 = MysqlEngineVersion.of('5.6.37', '5.6');
/** Version "5.6.39". */
MysqlEngineVersion.VER_5_6_39 = MysqlEngineVersion.of('5.6.39', '5.6');
/** Version "5.6.40". */
MysqlEngineVersion.VER_5_6_40 = MysqlEngineVersion.of('5.6.40', '5.6');
/** Version "5.6.41". */
MysqlEngineVersion.VER_5_6_41 = MysqlEngineVersion.of('5.6.41', '5.6');
/** Version "5.6.43". */
MysqlEngineVersion.VER_5_6_43 = MysqlEngineVersion.of('5.6.43', '5.6');
/** Version "5.6.44". */
MysqlEngineVersion.VER_5_6_44 = MysqlEngineVersion.of('5.6.44', '5.6');
/** Version "5.6.46". */
MysqlEngineVersion.VER_5_6_46 = MysqlEngineVersion.of('5.6.46', '5.6');
/** Version "5.6.48". */
MysqlEngineVersion.VER_5_6_48 = MysqlEngineVersion.of('5.6.48', '5.6');
/** Version "5.7" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_5_7 = MysqlEngineVersion.of('5.7', '5.7');
/** Version "5.7.16". */
MysqlEngineVersion.VER_5_7_16 = MysqlEngineVersion.of('5.7.16', '5.7');
/** Version "5.7.17". */
MysqlEngineVersion.VER_5_7_17 = MysqlEngineVersion.of('5.7.17', '5.7');
/** Version "5.7.19". */
MysqlEngineVersion.VER_5_7_19 = MysqlEngineVersion.of('5.7.19', '5.7');
/** Version "5.7.21". */
MysqlEngineVersion.VER_5_7_21 = MysqlEngineVersion.of('5.7.21', '5.7');
/** Version "5.7.22". */
MysqlEngineVersion.VER_5_7_22 = MysqlEngineVersion.of('5.7.22', '5.7');
/** Version "5.7.23". */
MysqlEngineVersion.VER_5_7_23 = MysqlEngineVersion.of('5.7.23', '5.7');
/** Version "5.7.24". */
MysqlEngineVersion.VER_5_7_24 = MysqlEngineVersion.of('5.7.24', '5.7');
/** Version "5.7.25". */
MysqlEngineVersion.VER_5_7_25 = MysqlEngineVersion.of('5.7.25', '5.7');
/** Version "5.7.26". */
MysqlEngineVersion.VER_5_7_26 = MysqlEngineVersion.of('5.7.26', '5.7');
/** Version "5.7.28". */
MysqlEngineVersion.VER_5_7_28 = MysqlEngineVersion.of('5.7.28', '5.7');
/** Version "5.7.30". */
MysqlEngineVersion.VER_5_7_30 = MysqlEngineVersion.of('5.7.30', '5.7');
/** Version "8.0" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_8_0 = MysqlEngineVersion.of('8.0', '8.0');
/** Version "8.0.11". */
MysqlEngineVersion.VER_8_0_11 = MysqlEngineVersion.of('8.0.11', '8.0');
/** Version "8.0.13". */
MysqlEngineVersion.VER_8_0_13 = MysqlEngineVersion.of('8.0.13', '8.0');
/** Version "8.0.15". */
MysqlEngineVersion.VER_8_0_15 = MysqlEngineVersion.of('8.0.15', '8.0');
/** Version "8.0.16". */
MysqlEngineVersion.VER_8_0_16 = MysqlEngineVersion.of('8.0.16', '8.0');
/** Version "8.0.17". */
MysqlEngineVersion.VER_8_0_17 = MysqlEngineVersion.of('8.0.17', '8.0');
/** Version "8.0.19". */
MysqlEngineVersion.VER_8_0_19 = MysqlEngineVersion.of('8.0.19', '8.0');
class MySqlInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'mysql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.mysqlFullVersion,
                    majorVersion: version.mysqlMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * The versions for the PostgreSQL instance engines
 * (those returned by {@link DatabaseInstanceEngine.postgres}).
 */
class PostgresEngineVersion {
    constructor(postgresFullVersion, postgresMajorVersion, postgresFeatures) {
        this.postgresFullVersion = postgresFullVersion;
        this.postgresMajorVersion = postgresMajorVersion;
        this._features = {
            s3Import: (postgresFeatures === null || postgresFeatures === void 0 ? void 0 : postgresFeatures.s3Import) ? 's3Import' : undefined,
        };
    }
    /**
     * Create a new PostgresEngineVersion with an arbitrary version.
     *
     * @param postgresFullVersion the full version string,
     *   for example "13.11"
     * @param postgresMajorVersion the major version of the engine,
     *   for example "13"
     */
    static of(postgresFullVersion, postgresMajorVersion, postgresFeatures) {
        return new PostgresEngineVersion(postgresFullVersion, postgresMajorVersion, postgresFeatures);
    }
}
exports.PostgresEngineVersion = PostgresEngineVersion;
/** Version "9.5" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_9_5 = PostgresEngineVersion.of('9.5', '9.5');
/** Version "9.5.2". */
PostgresEngineVersion.VER_9_5_2 = PostgresEngineVersion.of('9.5.2', '9.5');
/** Version "9.5.4". */
PostgresEngineVersion.VER_9_5_4 = PostgresEngineVersion.of('9.5.4', '9.5');
/** Version "9.5.6". */
PostgresEngineVersion.VER_9_5_6 = PostgresEngineVersion.of('9.5.6', '9.5');
/** Version "9.5.7". */
PostgresEngineVersion.VER_9_5_7 = PostgresEngineVersion.of('9.5.7', '9.5');
/** Version "9.5.9". */
PostgresEngineVersion.VER_9_5_9 = PostgresEngineVersion.of('9.5.9', '9.5');
/** Version "9.5.10". */
PostgresEngineVersion.VER_9_5_10 = PostgresEngineVersion.of('9.5.10', '9.5');
/** Version "9.5.12". */
PostgresEngineVersion.VER_9_5_12 = PostgresEngineVersion.of('9.5.12', '9.5');
/** Version "9.5.13". */
PostgresEngineVersion.VER_9_5_13 = PostgresEngineVersion.of('9.5.13', '9.5');
/** Version "9.5.14". */
PostgresEngineVersion.VER_9_5_14 = PostgresEngineVersion.of('9.5.14', '9.5');
/** Version "9.5.15". */
PostgresEngineVersion.VER_9_5_15 = PostgresEngineVersion.of('9.5.15', '9.5');
/** Version "9.5.16". */
PostgresEngineVersion.VER_9_5_16 = PostgresEngineVersion.of('9.5.16', '9.5');
/** Version "9.5.18". */
PostgresEngineVersion.VER_9_5_18 = PostgresEngineVersion.of('9.5.18', '9.5');
/** Version "9.5.19". */
PostgresEngineVersion.VER_9_5_19 = PostgresEngineVersion.of('9.5.19', '9.5');
/** Version "9.5.20". */
PostgresEngineVersion.VER_9_5_20 = PostgresEngineVersion.of('9.5.20', '9.5');
/** Version "9.5.21". */
PostgresEngineVersion.VER_9_5_21 = PostgresEngineVersion.of('9.5.21', '9.5');
/** Version "9.5.22". */
PostgresEngineVersion.VER_9_5_22 = PostgresEngineVersion.of('9.5.22', '9.5');
/** Version "9.6" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_9_6 = PostgresEngineVersion.of('9.6', '9.6');
/** Version "9.6.1". */
PostgresEngineVersion.VER_9_6_1 = PostgresEngineVersion.of('9.6.1', '9.6');
/** Version "9.6.2". */
PostgresEngineVersion.VER_9_6_2 = PostgresEngineVersion.of('9.6.2', '9.6');
/** Version "9.6.3". */
PostgresEngineVersion.VER_9_6_3 = PostgresEngineVersion.of('9.6.3', '9.6');
/** Version "9.6.5". */
PostgresEngineVersion.VER_9_6_5 = PostgresEngineVersion.of('9.6.5', '9.6');
/** Version "9.6.6". */
PostgresEngineVersion.VER_9_6_6 = PostgresEngineVersion.of('9.6.6', '9.6');
/** Version "9.6.8". */
PostgresEngineVersion.VER_9_6_8 = PostgresEngineVersion.of('9.6.8', '9.6');
/** Version "9.6.9". */
PostgresEngineVersion.VER_9_6_9 = PostgresEngineVersion.of('9.6.9', '9.6');
/** Version "9.6.10". */
PostgresEngineVersion.VER_9_6_10 = PostgresEngineVersion.of('9.6.10', '9.6');
/** Version "9.6.11". */
PostgresEngineVersion.VER_9_6_11 = PostgresEngineVersion.of('9.6.11', '9.6');
/** Version "9.6.12". */
PostgresEngineVersion.VER_9_6_12 = PostgresEngineVersion.of('9.6.12', '9.6');
/** Version "9.6.14". */
PostgresEngineVersion.VER_9_6_14 = PostgresEngineVersion.of('9.6.14', '9.6');
/** Version "9.6.15". */
PostgresEngineVersion.VER_9_6_15 = PostgresEngineVersion.of('9.6.15', '9.6');
/** Version "9.6.16". */
PostgresEngineVersion.VER_9_6_16 = PostgresEngineVersion.of('9.6.16', '9.6');
/** Version "9.6.17". */
PostgresEngineVersion.VER_9_6_17 = PostgresEngineVersion.of('9.6.17', '9.6');
/** Version "9.6.18". */
PostgresEngineVersion.VER_9_6_18 = PostgresEngineVersion.of('9.6.18', '9.6');
/** Version "10" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_10 = PostgresEngineVersion.of('10', '10');
/** Version "10.1". */
PostgresEngineVersion.VER_10_1 = PostgresEngineVersion.of('10.1', '10');
/** Version "10.3". */
PostgresEngineVersion.VER_10_3 = PostgresEngineVersion.of('10.3', '10');
/** Version "10.4". */
PostgresEngineVersion.VER_10_4 = PostgresEngineVersion.of('10.4', '10');
/** Version "10.5". */
PostgresEngineVersion.VER_10_5 = PostgresEngineVersion.of('10.5', '10');
/** Version "10.6". */
PostgresEngineVersion.VER_10_6 = PostgresEngineVersion.of('10.6', '10');
/** Version "10.7". */
PostgresEngineVersion.VER_10_7 = PostgresEngineVersion.of('10.7', '10', { s3Import: true });
/** Version "10.9". */
PostgresEngineVersion.VER_10_9 = PostgresEngineVersion.of('10.9', '10', { s3Import: true });
/** Version "10.10". */
PostgresEngineVersion.VER_10_10 = PostgresEngineVersion.of('10.10', '10', { s3Import: true });
/** Version "10.11". */
PostgresEngineVersion.VER_10_11 = PostgresEngineVersion.of('10.11', '10', { s3Import: true });
/** Version "10.12". */
PostgresEngineVersion.VER_10_12 = PostgresEngineVersion.of('10.12', '10', { s3Import: true });
/** Version "10.13". */
PostgresEngineVersion.VER_10_13 = PostgresEngineVersion.of('10.13', '10', { s3Import: true });
/** Version "11" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_11 = PostgresEngineVersion.of('11', '11', { s3Import: true });
/** Version "11.1". */
PostgresEngineVersion.VER_11_1 = PostgresEngineVersion.of('11.1', '11', { s3Import: true });
/** Version "11.2". */
PostgresEngineVersion.VER_11_2 = PostgresEngineVersion.of('11.2', '11', { s3Import: true });
/** Version "11.4". */
PostgresEngineVersion.VER_11_4 = PostgresEngineVersion.of('11.4', '11', { s3Import: true });
/** Version "11.5". */
PostgresEngineVersion.VER_11_5 = PostgresEngineVersion.of('11.5', '11', { s3Import: true });
/** Version "11.6". */
PostgresEngineVersion.VER_11_6 = PostgresEngineVersion.of('11.6', '11', { s3Import: true });
/** Version "11.7". */
PostgresEngineVersion.VER_11_7 = PostgresEngineVersion.of('11.7', '11', { s3Import: true });
/** Version "11.8". */
PostgresEngineVersion.VER_11_8 = PostgresEngineVersion.of('11.8', '11', { s3Import: true });
/** Version "12" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_12 = PostgresEngineVersion.of('12', '12', { s3Import: true });
/** Version "12.2". */
PostgresEngineVersion.VER_12_2 = PostgresEngineVersion.of('12.2', '12', { s3Import: true });
/** Version "12.3". */
PostgresEngineVersion.VER_12_3 = PostgresEngineVersion.of('12.3', '12', { s3Import: true });
/**
 * The instance engine for PostgreSQL.
 */
class PostgresInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'postgres',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.postgresFullVersion,
                    majorVersion: version.postgresMajorVersion,
                }
                : undefined,
            features: version ? version === null || version === void 0 ? void 0 : version._features : { s3Import: 's3Import' },
        });
    }
}
/**
 * The versions for the legacy Oracle instance engines
 * (those returned by {@link DatabaseInstanceEngine.oracleSe}
 * and {@link DatabaseInstanceEngine.oracleSe1}).
 * Note: RDS will stop allowing creating new databases with this version in August 2020.
 *
 * @deprecated instances can no longer be created with these engine versions. See https://forums.aws.amazon.com/ann.jspa?annID=7341
 */
class OracleLegacyEngineVersion {
    constructor(oracleLegacyFullVersion, oracleLegacyMajorVersion) {
        this.oracleLegacyFullVersion = oracleLegacyFullVersion;
        this.oracleLegacyMajorVersion = oracleLegacyMajorVersion;
    }
    static of(oracleLegacyFullVersion, oracleLegacyMajorVersion) {
        return new OracleLegacyEngineVersion(oracleLegacyFullVersion, oracleLegacyMajorVersion);
    }
}
exports.OracleLegacyEngineVersion = OracleLegacyEngineVersion;
/** Version "11.2" (only a major version, without a specific minor version). */
OracleLegacyEngineVersion.VER_11_2 = OracleLegacyEngineVersion.of('11.2', '11.2');
/** Version "11.2.0.2.v2". */
OracleLegacyEngineVersion.VER_11_2_0_2_V2 = OracleLegacyEngineVersion.of('11.2.0.2.v2', '11.2');
/** Version "11.2.0.4.v1". */
OracleLegacyEngineVersion.VER_11_2_0_4_V1 = OracleLegacyEngineVersion.of('11.2.0.4.v1', '11.2');
/** Version "11.2.0.4.v3". */
OracleLegacyEngineVersion.VER_11_2_0_4_V3 = OracleLegacyEngineVersion.of('11.2.0.4.v3', '11.2');
/** Version "11.2.0.4.v4". */
OracleLegacyEngineVersion.VER_11_2_0_4_V4 = OracleLegacyEngineVersion.of('11.2.0.4.v4', '11.2');
/** Version "11.2.0.4.v5". */
OracleLegacyEngineVersion.VER_11_2_0_4_V5 = OracleLegacyEngineVersion.of('11.2.0.4.v5', '11.2');
/** Version "11.2.0.4.v6". */
OracleLegacyEngineVersion.VER_11_2_0_4_V6 = OracleLegacyEngineVersion.of('11.2.0.4.v6', '11.2');
/** Version "11.2.0.4.v7". */
OracleLegacyEngineVersion.VER_11_2_0_4_V7 = OracleLegacyEngineVersion.of('11.2.0.4.v7', '11.2');
/** Version "11.2.0.4.v8". */
OracleLegacyEngineVersion.VER_11_2_0_4_V8 = OracleLegacyEngineVersion.of('11.2.0.4.v8', '11.2');
/** Version "11.2.0.4.v9". */
OracleLegacyEngineVersion.VER_11_2_0_4_V9 = OracleLegacyEngineVersion.of('11.2.0.4.v9', '11.2');
/** Version "11.2.0.4.v10". */
OracleLegacyEngineVersion.VER_11_2_0_4_V10 = OracleLegacyEngineVersion.of('11.2.0.4.v10', '11.2');
/** Version "11.2.0.4.v11". */
OracleLegacyEngineVersion.VER_11_2_0_4_V11 = OracleLegacyEngineVersion.of('11.2.0.4.v11', '11.2');
/** Version "11.2.0.4.v12". */
OracleLegacyEngineVersion.VER_11_2_0_4_V12 = OracleLegacyEngineVersion.of('11.2.0.4.v12', '11.2');
/** Version "11.2.0.4.v13". */
OracleLegacyEngineVersion.VER_11_2_0_4_V13 = OracleLegacyEngineVersion.of('11.2.0.4.v13', '11.2');
/** Version "11.2.0.4.v14". */
OracleLegacyEngineVersion.VER_11_2_0_4_V14 = OracleLegacyEngineVersion.of('11.2.0.4.v14', '11.2');
/** Version "11.2.0.4.v15". */
OracleLegacyEngineVersion.VER_11_2_0_4_V15 = OracleLegacyEngineVersion.of('11.2.0.4.v15', '11.2');
/** Version "11.2.0.4.v16". */
OracleLegacyEngineVersion.VER_11_2_0_4_V16 = OracleLegacyEngineVersion.of('11.2.0.4.v16', '11.2');
/** Version "11.2.0.4.v17". */
OracleLegacyEngineVersion.VER_11_2_0_4_V17 = OracleLegacyEngineVersion.of('11.2.0.4.v17', '11.2');
/** Version "11.2.0.4.v18". */
OracleLegacyEngineVersion.VER_11_2_0_4_V18 = OracleLegacyEngineVersion.of('11.2.0.4.v18', '11.2');
/** Version "11.2.0.4.v19". */
OracleLegacyEngineVersion.VER_11_2_0_4_V19 = OracleLegacyEngineVersion.of('11.2.0.4.v19', '11.2');
/** Version "11.2.0.4.v20". */
OracleLegacyEngineVersion.VER_11_2_0_4_V20 = OracleLegacyEngineVersion.of('11.2.0.4.v20', '11.2');
/** Version "11.2.0.4.v21". */
OracleLegacyEngineVersion.VER_11_2_0_4_V21 = OracleLegacyEngineVersion.of('11.2.0.4.v21', '11.2');
/** Version "11.2.0.4.v22". */
OracleLegacyEngineVersion.VER_11_2_0_4_V22 = OracleLegacyEngineVersion.of('11.2.0.4.v22', '11.2');
/** Version "11.2.0.4.v23". */
OracleLegacyEngineVersion.VER_11_2_0_4_V23 = OracleLegacyEngineVersion.of('11.2.0.4.v23', '11.2');
/** Version "11.2.0.4.v24". */
OracleLegacyEngineVersion.VER_11_2_0_4_V24 = OracleLegacyEngineVersion.of('11.2.0.4.v24', '11.2');
/**
 * The versions for the Oracle instance engines
 * (those returned by {@link DatabaseInstanceEngine.oracleSe2} and
 * {@link DatabaseInstanceEngine.oracleEe}).
 */
class OracleEngineVersion {
    constructor(oracleFullVersion, oracleMajorVersion) {
        this.oracleFullVersion = oracleFullVersion;
        this.oracleMajorVersion = oracleMajorVersion;
    }
    /**
     * Creates a new OracleEngineVersion with an arbitrary version.
     *
     * @param oracleFullVersion the full version string,
     *   for example "19.0.0.0.ru-2019-10.rur-2019-10.r1"
     * @param oracleMajorVersion the major version of the engine,
     *   for example "19"
     */
    static of(oracleFullVersion, oracleMajorVersion) {
        return new OracleEngineVersion(oracleFullVersion, oracleMajorVersion);
    }
}
exports.OracleEngineVersion = OracleEngineVersion;
/** Version "12.1" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_12_1 = OracleEngineVersion.of('12.1', '12.1');
/** Version "12.1.0.2.v1". */
OracleEngineVersion.VER_12_1_0_2_V1 = OracleEngineVersion.of('12.1.0.2.v1', '12.1');
/** Version "12.1.0.2.v2". */
OracleEngineVersion.VER_12_1_0_2_V2 = OracleEngineVersion.of('12.1.0.2.v2', '12.1');
/** Version "12.1.0.2.v3". */
OracleEngineVersion.VER_12_1_0_2_V3 = OracleEngineVersion.of('12.1.0.2.v3', '12.1');
/** Version "12.1.0.2.v4". */
OracleEngineVersion.VER_12_1_0_2_V4 = OracleEngineVersion.of('12.1.0.2.v4', '12.1');
/** Version "12.1.0.2.v5". */
OracleEngineVersion.VER_12_1_0_2_V5 = OracleEngineVersion.of('12.1.0.2.v5', '12.1');
/** Version "12.1.0.2.v6". */
OracleEngineVersion.VER_12_1_0_2_V6 = OracleEngineVersion.of('12.1.0.2.v6', '12.1');
/** Version "12.1.0.2.v7". */
OracleEngineVersion.VER_12_1_0_2_V7 = OracleEngineVersion.of('12.1.0.2.v7', '12.1');
/** Version "12.1.0.2.v8". */
OracleEngineVersion.VER_12_1_0_2_V8 = OracleEngineVersion.of('12.1.0.2.v8', '12.1');
/** Version "12.1.0.2.v9". */
OracleEngineVersion.VER_12_1_0_2_V9 = OracleEngineVersion.of('12.1.0.2.v9', '12.1');
/** Version "12.1.0.2.v10". */
OracleEngineVersion.VER_12_1_0_2_V10 = OracleEngineVersion.of('12.1.0.2.v10', '12.1');
/** Version "12.1.0.2.v11". */
OracleEngineVersion.VER_12_1_0_2_V11 = OracleEngineVersion.of('12.1.0.2.v11', '12.1');
/** Version "12.1.0.2.v12". */
OracleEngineVersion.VER_12_1_0_2_V12 = OracleEngineVersion.of('12.1.0.2.v12', '12.1');
/** Version "12.1.0.2.v13". */
OracleEngineVersion.VER_12_1_0_2_V13 = OracleEngineVersion.of('12.1.0.2.v13', '12.1');
/** Version "12.1.0.2.v14". */
OracleEngineVersion.VER_12_1_0_2_V14 = OracleEngineVersion.of('12.1.0.2.v14', '12.1');
/** Version "12.1.0.2.v15". */
OracleEngineVersion.VER_12_1_0_2_V15 = OracleEngineVersion.of('12.1.0.2.v15', '12.1');
/** Version "12.1.0.2.v16". */
OracleEngineVersion.VER_12_1_0_2_V16 = OracleEngineVersion.of('12.1.0.2.v16', '12.1');
/** Version "12.1.0.2.v17". */
OracleEngineVersion.VER_12_1_0_2_V17 = OracleEngineVersion.of('12.1.0.2.v17', '12.1');
/** Version "12.1.0.2.v18". */
OracleEngineVersion.VER_12_1_0_2_V18 = OracleEngineVersion.of('12.1.0.2.v18', '12.1');
/** Version "12.1.0.2.v19". */
OracleEngineVersion.VER_12_1_0_2_V19 = OracleEngineVersion.of('12.1.0.2.v19', '12.1');
/** Version "12.1.0.2.v20". */
OracleEngineVersion.VER_12_1_0_2_V20 = OracleEngineVersion.of('12.1.0.2.v20', '12.1');
/** Version "12.2" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_12_2 = OracleEngineVersion.of('12.2', '12.2');
/** Version "12.2.0.1.ru-2018-10.rur-2018-10.r1". */
OracleEngineVersion.VER_12_2_0_1_2018_10_R1 = OracleEngineVersion.of('12.2.0.1.ru-2018-10.rur-2018-10.r1', '12.2');
/** Version "12.2.0.1.ru-2019-01.rur-2019-01.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_01_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-01.rur-2019-01.r1', '12.2');
/** Version "12.2.0.1.ru-2019-04.rur-2019-04.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_04_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-04.rur-2019-04.r1', '12.2');
/** Version "12.2.0.1.ru-2019-07.rur-2019-07.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_07_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-07.rur-2019-07.r1', '12.2');
/** Version "12.2.0.1.ru-2019-10.rur-2019-10.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_10_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-10.rur-2019-10.r1', '12.2');
/** Version "12.2.0.1.ru-2020-01.rur-2020-01.r1". */
OracleEngineVersion.VER_12_2_0_1_2020_01_R1 = OracleEngineVersion.of('12.2.0.1.ru-2020-01.rur-2020-01.r1', '12.2');
/** Version "12.2.0.1.ru-2020-04.rur-2020-04.r1". */
OracleEngineVersion.VER_12_2_0_1_2020_04_R1 = OracleEngineVersion.of('12.2.0.1.ru-2020-04.rur-2020-04.r1', '12.2');
/** Version "18" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_18 = OracleEngineVersion.of('18', '18');
/** Version "18.0.0.0.ru-2019-07.rur-2019-07.r1". */
OracleEngineVersion.VER_18_0_0_0_2019_07_R1 = OracleEngineVersion.of('18.0.0.0.ru-2019-07.rur-2019-07.r1', '18');
/** Version "18.0.0.0.ru-2019-10.rur-2019-10.r1". */
OracleEngineVersion.VER_18_0_0_0_2019_10_R1 = OracleEngineVersion.of('18.0.0.0.ru-2019-10.rur-2019-10.r1', '18');
/** Version "18.0.0.0.ru-2020-01.rur-2020-01.r1". */
OracleEngineVersion.VER_18_0_0_0_2020_01_R1 = OracleEngineVersion.of('18.0.0.0.ru-2020-01.rur-2020-01.r1', '18');
/** Version "18.0.0.0.ru-2020-04.rur-2020-04.r1". */
OracleEngineVersion.VER_18_0_0_0_2020_04_R1 = OracleEngineVersion.of('18.0.0.0.ru-2020-04.rur-2020-04.r1', '18');
/** Version "19" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_19 = OracleEngineVersion.of('19', '19');
/** Version "19.0.0.0.ru-2019-07.rur-2019-07.r1". */
OracleEngineVersion.VER_19_0_0_0_2019_07_R1 = OracleEngineVersion.of('19.0.0.0.ru-2019-07.rur-2019-07.r1', '19');
/** Version "19.0.0.0.ru-2019-10.rur-2019-10.r1". */
OracleEngineVersion.VER_19_0_0_0_2019_10_R1 = OracleEngineVersion.of('19.0.0.0.ru-2019-10.rur-2019-10.r1', '19');
/** Version "19.0.0.0.ru-2020-01.rur-2020-01.r1". */
OracleEngineVersion.VER_19_0_0_0_2020_01_R1 = OracleEngineVersion.of('19.0.0.0.ru-2020-01.rur-2020-01.r1', '19');
/** Version "19.0.0.0.ru-2020-04.rur-2020-04.r1". */
OracleEngineVersion.VER_19_0_0_0_2020_04_R1 = OracleEngineVersion.of('19.0.0.0.ru-2020-04.rur-2020-04.r1', '19');
class OracleInstanceEngineBase extends InstanceEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER,
            parameterGroupFamily: props.version ? `${props.engineType}-${props.version.majorVersion}` : undefined,
            features: {
                s3Import: 'S3_INTEGRATION',
                s3Export: 'S3_INTEGRATION',
            },
        });
    }
    bindToInstance(scope, options) {
        const config = super.bindToInstance(scope, options);
        let optionGroup = options.optionGroup;
        if (options.s3ImportRole || options.s3ExportRole) {
            if (!optionGroup) {
                optionGroup = new option_group_1.OptionGroup(scope, 'InstanceOptionGroup', {
                    engine: this,
                    configurations: [],
                });
            }
            // https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html
            optionGroup.addConfiguration({
                name: 'S3_INTEGRATION',
                version: '1.0',
            });
        }
        return {
            ...config,
            optionGroup,
        };
    }
}
/** @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341 */
class OracleSeInstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se',
            version: version
                ? {
                    fullVersion: version.oracleLegacyFullVersion,
                    majorVersion: version.oracleLegacyMajorVersion,
                }
                : {
                    majorVersion: '11.2',
                },
        });
    }
}
/** @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341 */
class OracleSe1InstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se1',
            version: version
                ? {
                    fullVersion: version.oracleLegacyFullVersion,
                    majorVersion: version.oracleLegacyMajorVersion,
                }
                : {
                    majorVersion: '11.2',
                },
        });
    }
}
class OracleSe2InstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se2',
            version: version
                ? {
                    fullVersion: version.oracleFullVersion,
                    majorVersion: version.oracleMajorVersion,
                }
                : undefined,
        });
    }
}
class OracleEeInstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-ee',
            version: version
                ? {
                    fullVersion: version.oracleFullVersion,
                    majorVersion: version.oracleMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * The versions for the SQL Server instance engines
 * (those returned by {@link DatabaseInstanceEngine.sqlServerSe},
 * {@link DatabaseInstanceEngine.sqlServerEx}, {@link DatabaseInstanceEngine.sqlServerWeb}
 * and {@link DatabaseInstanceEngine.sqlServerEe}).
 */
class SqlServerEngineVersion {
    constructor(sqlServerFullVersion, sqlServerMajorVersion) {
        this.sqlServerFullVersion = sqlServerFullVersion;
        this.sqlServerMajorVersion = sqlServerMajorVersion;
    }
    /**
     * Create a new SqlServerEngineVersion with an arbitrary version.
     *
     * @param sqlServerFullVersion the full version string,
     *   for example "15.00.3049.1.v1"
     * @param sqlServerMajorVersion the major version of the engine,
     *   for example "15.00"
     */
    static of(sqlServerFullVersion, sqlServerMajorVersion) {
        return new SqlServerEngineVersion(sqlServerFullVersion, sqlServerMajorVersion);
    }
}
exports.SqlServerEngineVersion = SqlServerEngineVersion;
/** Version "11.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_11 = SqlServerEngineVersion.of('11.00', '11.00');
/** Version "11.00.5058.0.v1". */
SqlServerEngineVersion.VER_11_00_5058_0_V1 = SqlServerEngineVersion.of('11.00.5058.0.v1', '11.00');
/** Version "11.00.6020.0.v1". */
SqlServerEngineVersion.VER_11_00_6020_0_V1 = SqlServerEngineVersion.of('11.00.6020.0.v1', '11.00');
/** Version "11.00.6594.0.v1". */
SqlServerEngineVersion.VER_11_00_6594_0_V1 = SqlServerEngineVersion.of('11.00.6594.0.v1', '11.00');
/** Version "11.00.7462.6.v1". */
SqlServerEngineVersion.VER_11_00_7462_6_V1 = SqlServerEngineVersion.of('11.00.7462.6.v1', '11.00');
/** Version "11.00.7493.4.v1". */
SqlServerEngineVersion.VER_11_00_7493_4_V1 = SqlServerEngineVersion.of('11.00.7493.4.v1', '11.00');
/** Version "12.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_12 = SqlServerEngineVersion.of('12.00', '12.00');
/** Version "12.00.5000.0.v1". */
SqlServerEngineVersion.VER_12_00_5000_0_V1 = SqlServerEngineVersion.of('12.00.5000.0.v1', '12.00');
/** Version "12.00.5546.0.v1". */
SqlServerEngineVersion.VER_12_00_5546_0_V1 = SqlServerEngineVersion.of('12.00.5546.0.v1', '12.00');
/** Version "12.00.5571.0.v1". */
SqlServerEngineVersion.VER_12_00_5571_0_V1 = SqlServerEngineVersion.of('12.00.5571.0.v1', '12.00');
/** Version "12.00.6293.0.v1". */
SqlServerEngineVersion.VER_12_00_6293_0_V1 = SqlServerEngineVersion.of('12.00.6293.0.v1', '12.00');
/** Version "12.00.6329.1.v1". */
SqlServerEngineVersion.VER_12_00_6329_1_V1 = SqlServerEngineVersion.of('12.00.6329.1.v1', '12.00');
/** Version "13.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_13 = SqlServerEngineVersion.of('13.00', '13.00');
/** Version "13.00.2164.0.v1". */
SqlServerEngineVersion.VER_13_00_2164_0_V1 = SqlServerEngineVersion.of('13.00.2164.0.v1', '13.00');
/** Version "13.00.4422.0.v1". */
SqlServerEngineVersion.VER_13_00_4422_0_V1 = SqlServerEngineVersion.of('13.00.4422.0.v1', '13.00');
/** Version "13.00.4451.0.v1". */
SqlServerEngineVersion.VER_13_00_4451_0_V1 = SqlServerEngineVersion.of('13.00.4451.0.v1', '13.00');
/** Version "13.00.4466.4.v1". */
SqlServerEngineVersion.VER_13_00_4466_4_V1 = SqlServerEngineVersion.of('13.00.4466.4.v1', '13.00');
/** Version "13.00.4522.0.v1". */
SqlServerEngineVersion.VER_13_00_4522_0_V1 = SqlServerEngineVersion.of('13.00.4522.0.v1', '13.00');
/** Version "13.00.5216.0.v1". */
SqlServerEngineVersion.VER_13_00_5216_0_V1 = SqlServerEngineVersion.of('13.00.5216.0.v1', '13.00');
/** Version "13.00.5292.0.v1". */
SqlServerEngineVersion.VER_13_00_5292_0_V1 = SqlServerEngineVersion.of('13.00.5292.0.v1', '13.00');
/** Version "13.00.5366.0.v1". */
SqlServerEngineVersion.VER_13_00_5366_0_V1 = SqlServerEngineVersion.of('13.00.5366.0.v1', '13.00');
/** Version "13.00.5426.0.v1". */
SqlServerEngineVersion.VER_13_00_5426_0_V1 = SqlServerEngineVersion.of('13.00.5426.0.v1', '13.00');
/** Version "13.00.5598.27.v1". */
SqlServerEngineVersion.VER_13_00_5598_27_V1 = SqlServerEngineVersion.of('13.00.5598.27.v1', '13.00');
/** Version "14.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_14 = SqlServerEngineVersion.of('14.00', '14.00');
/** Version "14.00.1000.169.v1". */
SqlServerEngineVersion.VER_14_00_1000_169_V1 = SqlServerEngineVersion.of('14.00.1000.169.v1', '14.00');
/** Version "14.00.3015.40.v1". */
SqlServerEngineVersion.VER_14_00_3015_40_V1 = SqlServerEngineVersion.of('14.00.3015.40.v1', '14.00');
/** Version "14.00.3035.2.v1". */
SqlServerEngineVersion.VER_14_00_3035_2_V1 = SqlServerEngineVersion.of('14.00.3035.2.v1', '14.00');
/** Version "14.00.3049.1.v1". */
SqlServerEngineVersion.VER_14_00_3049_1_V1 = SqlServerEngineVersion.of('14.00.3049.1.v1', '14.00');
/** Version "14.00.3192.2.v1". */
SqlServerEngineVersion.VER_14_00_3192_2_V1 = SqlServerEngineVersion.of('14.00.3192.2.v1', '14.00');
/** Version "15.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_15 = SqlServerEngineVersion.of('15.00', '15.00');
/** Version "15.00.4043.16.v1". */
SqlServerEngineVersion.VER_15_00_4043_16_V1 = SqlServerEngineVersion.of('15.00.4043.16.v1', '15.00');
class SqlServerInstanceEngineBase extends InstanceEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER,
            version: props.version
                ? {
                    fullVersion: props.version.sqlServerFullVersion,
                    majorVersion: props.version.sqlServerMajorVersion,
                }
                : undefined,
            parameterGroupFamily: props.version
                // for some reason, even though SQL Server major versions usually end in '.00',
                // the ParameterGroup family has to end in '.0'
                ? `${props.engineType}-${props.version.sqlServerMajorVersion.endsWith('.00')
                    ? props.version.sqlServerMajorVersion.slice(0, -1)
                    : props.version.sqlServerMajorVersion}`
                : undefined,
            features: {
                s3Import: 'S3_INTEGRATION',
                s3Export: 'S3_INTEGRATION',
            },
        });
    }
    bindToInstance(scope, options) {
        var _a;
        const config = super.bindToInstance(scope, options);
        let optionGroup = options.optionGroup;
        const s3Role = (_a = options.s3ImportRole) !== null && _a !== void 0 ? _a : options.s3ExportRole;
        if (s3Role) {
            if (options.s3ImportRole && options.s3ExportRole && options.s3ImportRole !== options.s3ExportRole) {
                throw new Error('S3 import and export roles must be the same for SQL Server engines');
            }
            if (!optionGroup) {
                optionGroup = new option_group_1.OptionGroup(scope, 'InstanceOptionGroup', {
                    engine: this,
                    configurations: [],
                });
            }
            // https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.SQLServer.Options.BackupRestore.html
            optionGroup.addConfiguration({
                name: 'SQLSERVER_BACKUP_RESTORE',
                settings: { IAM_ROLE_ARN: s3Role.roleArn },
            });
        }
        return {
            ...config,
            optionGroup,
        };
    }
    get supportsTimezone() { return true; }
}
class SqlServerSeInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-se',
            version,
        });
    }
}
class SqlServerExInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-ex',
            version,
        });
    }
}
class SqlServerWebInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-web',
            version,
        });
    }
}
class SqlServerEeInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-ee',
            version,
        });
    }
}
/**
 * A database instance engine. Provides mapping to DatabaseEngine used for
 * secret rotation.
 */
class DatabaseInstanceEngine {
    /** Creates a new MariaDB instance engine. */
    static mariaDb(props) {
        return new MariaDbInstanceEngine(props.version);
    }
    /** Creates a new MySQL instance engine. */
    static mysql(props) {
        return new MySqlInstanceEngine(props.version);
    }
    /** Creates a new PostgreSQL instance engine. */
    static postgres(props) {
        return new PostgresInstanceEngine(props.version);
    }
    /**
     * Creates a new Oracle Standard Edition instance engine.
     * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
     */
    static oracleSe(props) {
        return new OracleSeInstanceEngine(props.version);
    }
    /**
     * Creates a new Oracle Standard Edition 1 instance engine.
     * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
     */
    static oracleSe1(props) {
        return new OracleSe1InstanceEngine(props.version);
    }
    /** Creates a new Oracle Standard Edition 1 instance engine. */
    static oracleSe2(props) {
        return new OracleSe2InstanceEngine(props.version);
    }
    /** Creates a new Oracle Enterprise Edition instance engine. */
    static oracleEe(props) {
        return new OracleEeInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Standard Edition instance engine. */
    static sqlServerSe(props) {
        return new SqlServerSeInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Express Edition instance engine. */
    static sqlServerEx(props) {
        return new SqlServerExInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Web Edition instance engine. */
    static sqlServerWeb(props) {
        return new SqlServerWebInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Enterprise Edition instance engine. */
    static sqlServerEe(props) {
        return new SqlServerEeInstanceEngine(props.version);
    }
}
exports.DatabaseInstanceEngine = DatabaseInstanceEngine;
/**
 * The unversioned 'mariadb' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link mariaDb()} method
 */
DatabaseInstanceEngine.MARIADB = new MariaDbInstanceEngine();
/**
 * The unversioned 'mysql' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link mysql()} method
 */
DatabaseInstanceEngine.MYSQL = new MySqlInstanceEngine();
/**
 * The unversioned 'oracle-ee' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link oracleEe()} method
 */
DatabaseInstanceEngine.ORACLE_EE = new OracleEeInstanceEngine();
/**
 * The unversioned 'oracle-se2' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link oracleSe2()} method
 */
DatabaseInstanceEngine.ORACLE_SE2 = new OracleSe2InstanceEngine();
/**
 * The unversioned 'oracle-se1' instance engine.
 *
 * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
 */
DatabaseInstanceEngine.ORACLE_SE1 = new OracleSe1InstanceEngine();
/**
 * The unversioned 'oracle-se' instance engine.
 *
 * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
 */
DatabaseInstanceEngine.ORACLE_SE = new OracleSeInstanceEngine();
/**
 * The unversioned 'postgres' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link postgres()} method
 */
DatabaseInstanceEngine.POSTGRES = new PostgresInstanceEngine();
/**
 * The unversioned 'sqlserver-ee' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerEe()} method
 */
DatabaseInstanceEngine.SQL_SERVER_EE = new SqlServerEeInstanceEngine();
/**
 * The unversioned 'sqlserver-se' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerSe()} method
 */
DatabaseInstanceEngine.SQL_SERVER_SE = new SqlServerSeInstanceEngine();
/**
 * The unversioned 'sqlserver-ex' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerEx()} method
 */
DatabaseInstanceEngine.SQL_SERVER_EX = new SqlServerExInstanceEngine();
/**
 * The unversioned 'sqlserver-web' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerWeb()} method
 */
DatabaseInstanceEngine.SQL_SERVER_WEB = new SqlServerWebInstanceEngine();
//# sourceMappingURL=data:application/json;base64,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