"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterFromSnapshot = exports.DatabaseCluster = exports.DatabaseClusterBase = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const util_1 = require("./private/util");
const props_1 = require("./props");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * A new or imported clustered database.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Add a new db proxy to this cluster.
     */
    addProxy(id, options) {
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromCluster(this),
            ...options,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
exports.DatabaseClusterBase = DatabaseClusterBase;
/**
 * Abstract base for ``DatabaseCluster`` and ``DatabaseClusterFromSnapshot``
 */
class DatabaseClusterNew extends DatabaseClusterBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        this.instanceIdentifiers = [];
        this.instanceEndpoints = [];
        const { subnetIds } = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            core_1.Annotations.of(this).addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        this.subnetGroup = (_a = props.subnetGroup) !== null && _a !== void 0 ? _a : new subnet_group_1.SubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} database`,
            vpc: props.instanceProps.vpc,
            vpcSubnets: props.instanceProps.vpcSubnets,
            removalPolicy: props.removalPolicy === core_1.RemovalPolicy.RETAIN ? props.removalPolicy : undefined,
        });
        this.securityGroups = (_b = props.instanceProps.securityGroups) !== null && _b !== void 0 ? _b : [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'RDS security group',
                vpc: props.instanceProps.vpc,
            }),
        ];
        let { s3ImportRole, s3ExportRole } = util_1.setupS3ImportExport(this, props);
        // bind the engine to the Cluster
        const clusterEngineBindConfig = props.engine.bindToCluster(this, {
            s3ImportRole,
            s3ExportRole,
            parameterGroup: props.parameterGroup,
        });
        const clusterAssociatedRoles = [];
        if (s3ImportRole) {
            clusterAssociatedRoles.push({ roleArn: s3ImportRole.roleArn, featureName: (_c = clusterEngineBindConfig.features) === null || _c === void 0 ? void 0 : _c.s3Import });
        }
        if (s3ExportRole) {
            clusterAssociatedRoles.push({ roleArn: s3ExportRole.roleArn, featureName: (_d = clusterEngineBindConfig.features) === null || _d === void 0 ? void 0 : _d.s3Export });
        }
        const clusterParameterGroup = (_e = props.parameterGroup) !== null && _e !== void 0 ? _e : clusterEngineBindConfig.parameterGroup;
        const clusterParameterGroupConfig = clusterParameterGroup === null || clusterParameterGroup === void 0 ? void 0 : clusterParameterGroup.bindToCluster({});
        this.newCfnProps = {
            // Basic
            engine: props.engine.engineType,
            engineVersion: (_f = props.engine.engineVersion) === null || _f === void 0 ? void 0 : _f.fullVersion,
            dbClusterIdentifier: props.clusterIdentifier,
            dbSubnetGroupName: this.subnetGroup.subnetGroupName,
            vpcSecurityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
            port: (_g = props.port) !== null && _g !== void 0 ? _g : clusterEngineBindConfig.port,
            dbClusterParameterGroupName: clusterParameterGroupConfig === null || clusterParameterGroupConfig === void 0 ? void 0 : clusterParameterGroupConfig.parameterGroupName,
            associatedRoles: clusterAssociatedRoles.length > 0 ? clusterAssociatedRoles : undefined,
            deletionProtection: util_1.defaultDeletionProtection(props.deletionProtection, props.removalPolicy),
            // Admin
            backupRetentionPeriod: (_j = (_h = props.backup) === null || _h === void 0 ? void 0 : _h.retention) === null || _j === void 0 ? void 0 : _j.toDays(),
            preferredBackupWindow: (_k = props.backup) === null || _k === void 0 ? void 0 : _k.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            databaseName: props.defaultDatabaseName,
            enableCloudwatchLogsExports: props.cloudwatchLogsExports,
        };
    }
}
/**
 * Represents an imported database cluster.
 */
class ImportedDatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.clusterIdentifier = attrs.clusterIdentifier;
        const defaultPort = attrs.port ? ec2.Port.tcp(attrs.port) : undefined;
        this.connections = new ec2.Connections({
            securityGroups: attrs.securityGroups,
            defaultPort,
        });
        this._clusterEndpoint = (attrs.clusterEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port) : undefined;
        this._clusterReadEndpoint = (attrs.readerEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port) : undefined;
        this._instanceIdentifiers = attrs.instanceIdentifiers;
        this._instanceEndpoints = (attrs.instanceEndpointAddresses && attrs.port)
            ? attrs.instanceEndpointAddresses.map(addr => new endpoint_1.Endpoint(addr, attrs.port))
            : undefined;
    }
    get clusterEndpoint() {
        if (!this._clusterEndpoint) {
            throw new Error('Cannot access `clusterEndpoint` of an imported cluster without an endpoint address and port');
        }
        return this._clusterEndpoint;
    }
    get clusterReadEndpoint() {
        if (!this._clusterReadEndpoint) {
            throw new Error('Cannot access `clusterReadEndpoint` of an imported cluster without a readerEndpointAddress and port');
        }
        return this._clusterReadEndpoint;
    }
    get instanceIdentifiers() {
        if (!this._instanceIdentifiers) {
            throw new Error('Cannot access `instanceIdentifiers` of an imported cluster without provided instanceIdentifiers');
        }
        return this._instanceIdentifiers;
    }
    get instanceEndpoints() {
        if (!this._instanceEndpoints) {
            throw new Error('Cannot access `instanceEndpoints` of an imported cluster without instanceEndpointAddresses and port');
        }
        return this._instanceEndpoints;
    }
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::RDS::DBCluster
 */
class DatabaseCluster extends DatabaseClusterNew {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        this.vpc = props.instanceProps.vpc;
        this.vpcSubnets = props.instanceProps.vpcSubnets;
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        let secret;
        if (!props.masterUser.password) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.username,
                encryptionKey: props.masterUser.encryptionKey,
            });
        }
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            ...this.newCfnProps,
            // Admin
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUser.username,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : (props.masterUser.password
                    ? props.masterUser.password.toString()
                    : undefined),
            // Encryption
            kmsKeyId: (_a = props.storageEncryptionKey) === null || _a === void 0 ? void 0 : _a.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        util_1.applyRemovalPolicy(cluster, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
        setLogRetention(this, props);
        createInstances(this, props, this.subnetGroup);
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        return new ImportedDatabaseCluster(scope, id, attrs);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
}
exports.DatabaseCluster = DatabaseCluster;
/**
 * A database cluster restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseClusterFromSnapshot extends DatabaseClusterNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            ...this.newCfnProps,
            snapshotIdentifier: props.snapshotIdentifier,
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        util_1.applyRemovalPolicy(cluster, props.removalPolicy);
        setLogRetention(this, props);
        createInstances(this, props, this.subnetGroup);
    }
}
exports.DatabaseClusterFromSnapshot = DatabaseClusterFromSnapshot;
/**
 * Sets up CloudWatch log retention if configured.
 * A function rather than protected member to prevent exposing ``DatabaseClusterBaseProps``.
 */
function setLogRetention(cluster, props) {
    if (props.cloudwatchLogsExports) {
        const unsupportedLogTypes = props.cloudwatchLogsExports.filter(logType => !props.engine.supportedLogTypes.includes(logType));
        if (unsupportedLogTypes.length > 0) {
            throw new Error(`Unsupported logs for the current engine type: ${unsupportedLogTypes.join(',')}`);
        }
        if (props.cloudwatchLogsRetention) {
            for (const log of props.cloudwatchLogsExports) {
                new logs.LogRetention(cluster, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/cluster/${cluster.clusterIdentifier}/${log}`,
                    retention: props.cloudwatchLogsRetention,
                    role: props.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * Creates the instances for the cluster.
 * A function rather than a protected method on ``DatabaseClusterNew`` to avoid exposing
 * ``DatabaseClusterNew`` and ``DatabaseClusterBaseProps`` in the API.
 */
function createInstances(cluster, props, subnetGroup) {
    var _a, _b, _c, _d;
    const instanceCount = props.instances != null ? props.instances : 2;
    if (instanceCount < 1) {
        throw new Error('At least one instance is required');
    }
    const instanceIdentifiers = [];
    const instanceEndpoints = [];
    const portAttribute = cluster.clusterEndpoint.port;
    const instanceProps = props.instanceProps;
    // Get the actual subnet objects so we can depend on internet connectivity.
    const internetConnected = instanceProps.vpc.selectSubnets(instanceProps.vpcSubnets).internetConnectivityEstablished;
    let monitoringRole;
    if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
        monitoringRole = props.monitoringRole || new aws_iam_1.Role(cluster, 'MonitoringRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole'),
            ],
        });
    }
    const enablePerformanceInsights = instanceProps.enablePerformanceInsights
        || instanceProps.performanceInsightRetention !== undefined || instanceProps.performanceInsightEncryptionKey !== undefined;
    if (enablePerformanceInsights && instanceProps.enablePerformanceInsights === false) {
        throw new Error('`enablePerformanceInsights` disabled, but `performanceInsightRetention` or `performanceInsightEncryptionKey` was set');
    }
    const instanceType = (_a = instanceProps.instanceType) !== null && _a !== void 0 ? _a : ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.MEDIUM);
    const instanceParameterGroupConfig = (_b = instanceProps.parameterGroup) === null || _b === void 0 ? void 0 : _b.bindToInstance({});
    for (let i = 0; i < instanceCount; i++) {
        const instanceIndex = i + 1;
        const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}` :
            props.clusterIdentifier != null ? `${props.clusterIdentifier}instance${instanceIndex}` :
                undefined;
        const publiclyAccessible = instanceProps.vpcSubnets && instanceProps.vpcSubnets.subnetType === ec2.SubnetType.PUBLIC;
        const instance = new rds_generated_1.CfnDBInstance(cluster, `Instance${instanceIndex}`, {
            // Link to cluster
            engine: props.engine.engineType,
            engineVersion: (_c = props.engine.engineVersion) === null || _c === void 0 ? void 0 : _c.fullVersion,
            dbClusterIdentifier: cluster.clusterIdentifier,
            dbInstanceIdentifier: instanceIdentifier,
            // Instance properties
            dbInstanceClass: databaseInstanceType(instanceType),
            publiclyAccessible,
            enablePerformanceInsights: enablePerformanceInsights || instanceProps.enablePerformanceInsights,
            performanceInsightsKmsKeyId: (_d = instanceProps.performanceInsightEncryptionKey) === null || _d === void 0 ? void 0 : _d.keyArn,
            performanceInsightsRetentionPeriod: enablePerformanceInsights
                ? (instanceProps.performanceInsightRetention || props_1.PerformanceInsightRetention.DEFAULT)
                : undefined,
            // This is already set on the Cluster. Unclear to me whether it should be repeated or not. Better yes.
            dbSubnetGroupName: subnetGroup.subnetGroupName,
            dbParameterGroupName: instanceParameterGroupConfig === null || instanceParameterGroupConfig === void 0 ? void 0 : instanceParameterGroupConfig.parameterGroupName,
            monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
            monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
        });
        // If removalPolicy isn't explicitly set,
        // it's Snapshot for Cluster.
        // Because of that, in this case,
        // we can safely use the CFN default of Delete for DbInstances with dbClusterIdentifier set.
        if (props.removalPolicy) {
            util_1.applyRemovalPolicy(instance, props.removalPolicy);
        }
        // We must have a dependency on the NAT gateway provider here to create
        // things in the right order.
        instance.node.addDependency(internetConnected);
        instanceIdentifiers.push(instance.ref);
        instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute));
    }
    return { instanceEndpoints, instanceIdentifiers };
}
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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