"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterEngine = exports.AuroraPostgresEngineVersion = exports.AuroraMysqlEngineVersion = exports.AuroraEngineVersion = void 0;
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const parameter_group_1 = require("./parameter-group");
class ClusterEngineBase {
    constructor(props) {
        this.engineType = props.engineType;
        this.features = props.features;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.defaultPort = props.defaultPort;
        this.engineVersion = props.engineVersion;
        this.parameterGroupFamily = this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined;
    }
    bindToCluster(scope, options) {
        var _a;
        const parameterGroup = (_a = options.parameterGroup) !== null && _a !== void 0 ? _a : this.defaultParameterGroup(scope);
        return {
            parameterGroup,
            port: this.defaultPort,
            features: this.features,
        };
    }
}
class MySqlClusterEngineBase extends ClusterEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            engineVersion: props.engineVersion ? props.engineVersion : { majorVersion: props.defaultMajorVersion },
        });
        this.supportedLogTypes = ['error', 'general', 'slowquery', 'audit'];
    }
    bindToCluster(scope, options) {
        var _a;
        const config = super.bindToCluster(scope, options);
        const parameterGroup = (_a = options.parameterGroup) !== null && _a !== void 0 ? _a : (options.s3ImportRole || options.s3ExportRole
            ? new parameter_group_1.ParameterGroup(scope, 'ClusterParameterGroup', {
                engine: this,
            })
            : config.parameterGroup);
        if (options.s3ImportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_load_from_s3_role', options.s3ImportRole.roleArn);
        }
        if (options.s3ExportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_select_into_s3_role', options.s3ExportRole.roleArn);
        }
        return {
            ...config,
            parameterGroup,
        };
    }
}
/**
 * The versions for the Aurora cluster engine
 * (those returned by {@link DatabaseClusterEngine.aurora}).
 */
class AuroraEngineVersion {
    constructor(auroraFullVersion, auroraMajorVersion = '5.6') {
        this.auroraFullVersion = auroraFullVersion;
        this.auroraMajorVersion = auroraMajorVersion;
    }
    /**
     * Create a new AuroraEngineVersion with an arbitrary version.
     *
     * @param auroraFullVersion the full version string,
     *   for example "5.6.mysql_aurora.1.78.3.6"
     * @param auroraMajorVersion the major version of the engine,
     *   defaults to "5.6"
     */
    static of(auroraFullVersion, auroraMajorVersion) {
        return new AuroraEngineVersion(auroraFullVersion, auroraMajorVersion);
    }
    static builtIn_5_6(minorVersion, addStandardPrefix = true) {
        return new AuroraEngineVersion(`5.6.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraEngineVersion = AuroraEngineVersion;
/** Version "5.6.10a". */
AuroraEngineVersion.VER_10A = AuroraEngineVersion.builtIn_5_6('10a', false);
/** Version "5.6.mysql_aurora.1.17.9". */
AuroraEngineVersion.VER_1_17_9 = AuroraEngineVersion.builtIn_5_6('1.17.9');
/** Version "5.6.mysql_aurora.1.19.0". */
AuroraEngineVersion.VER_1_19_0 = AuroraEngineVersion.builtIn_5_6('1.19.0');
/** Version "5.6.mysql_aurora.1.19.1". */
AuroraEngineVersion.VER_1_19_1 = AuroraEngineVersion.builtIn_5_6('1.19.1');
/** Version "5.6.mysql_aurora.1.19.2". */
AuroraEngineVersion.VER_1_19_2 = AuroraEngineVersion.builtIn_5_6('1.19.2');
/** Version "5.6.mysql_aurora.1.19.5". */
AuroraEngineVersion.VER_1_19_5 = AuroraEngineVersion.builtIn_5_6('1.19.5');
/** Version "5.6.mysql_aurora.1.19.6". */
AuroraEngineVersion.VER_1_19_6 = AuroraEngineVersion.builtIn_5_6('1.19.6');
/** Version "5.6.mysql_aurora.1.20.0". */
AuroraEngineVersion.VER_1_20_0 = AuroraEngineVersion.builtIn_5_6('1.20.0');
/** Version "5.6.mysql_aurora.1.20.1". */
AuroraEngineVersion.VER_1_20_1 = AuroraEngineVersion.builtIn_5_6('1.20.1');
/** Version "5.6.mysql_aurora.1.21.0". */
AuroraEngineVersion.VER_1_21_0 = AuroraEngineVersion.builtIn_5_6('1.21.0');
/** Version "5.6.mysql_aurora.1.22.0". */
AuroraEngineVersion.VER_1_22_0 = AuroraEngineVersion.builtIn_5_6('1.22.0');
/** Version "5.6.mysql_aurora.1.22.1". */
AuroraEngineVersion.VER_1_22_1 = AuroraEngineVersion.builtIn_5_6('1.22.1');
/** Version "5.6.mysql_aurora.1.22.1.3". */
AuroraEngineVersion.VER_1_22_1_3 = AuroraEngineVersion.builtIn_5_6('1.22.1.3');
/** Version "5.6.mysql_aurora.1.22.2". */
AuroraEngineVersion.VER_1_22_2 = AuroraEngineVersion.builtIn_5_6('1.22.2');
class AuroraClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora',
            engineVersion: version
                ? {
                    fullVersion: version.auroraFullVersion,
                    majorVersion: version.auroraMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.6',
        });
    }
    defaultParameterGroup(_scope) {
        // the default.aurora5.6 ParameterGroup is actually the default,
        // so just return undefined in this case
        return undefined;
    }
}
/**
 * The versions for the Aurora MySQL cluster engine
 * (those returned by {@link DatabaseClusterEngine.auroraMysql}).
 */
class AuroraMysqlEngineVersion {
    constructor(auroraMysqlFullVersion, auroraMysqlMajorVersion = '5.7') {
        this.auroraMysqlFullVersion = auroraMysqlFullVersion;
        this.auroraMysqlMajorVersion = auroraMysqlMajorVersion;
    }
    /**
     * Create a new AuroraMysqlEngineVersion with an arbitrary version.
     *
     * @param auroraMysqlFullVersion the full version string,
     *   for example "5.7.mysql_aurora.2.78.3.6"
     * @param auroraMysqlMajorVersion the major version of the engine,
     *   defaults to "5.7"
     */
    static of(auroraMysqlFullVersion, auroraMysqlMajorVersion) {
        return new AuroraMysqlEngineVersion(auroraMysqlFullVersion, auroraMysqlMajorVersion);
    }
    static builtIn_5_7(minorVersion, addStandardPrefix = true) {
        return new AuroraMysqlEngineVersion(`5.7.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraMysqlEngineVersion = AuroraMysqlEngineVersion;
/** Version "5.7.12". */
AuroraMysqlEngineVersion.VER_5_7_12 = AuroraMysqlEngineVersion.builtIn_5_7('12', false);
/** Version "5.7.mysql_aurora.2.03.2". */
AuroraMysqlEngineVersion.VER_2_03_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.2');
/** Version "5.7.mysql_aurora.2.03.3". */
AuroraMysqlEngineVersion.VER_2_03_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.3');
/** Version "5.7.mysql_aurora.2.03.4". */
AuroraMysqlEngineVersion.VER_2_03_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.4');
/** Version "5.7.mysql_aurora.2.04.0". */
AuroraMysqlEngineVersion.VER_2_04_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.0');
/** Version "5.7.mysql_aurora.2.04.1". */
AuroraMysqlEngineVersion.VER_2_04_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.1');
/** Version "5.7.mysql_aurora.2.04.2". */
AuroraMysqlEngineVersion.VER_2_04_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.2');
/** Version "5.7.mysql_aurora.2.04.3". */
AuroraMysqlEngineVersion.VER_2_04_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.3');
/** Version "5.7.mysql_aurora.2.04.4". */
AuroraMysqlEngineVersion.VER_2_04_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.4');
/** Version "5.7.mysql_aurora.2.04.5". */
AuroraMysqlEngineVersion.VER_2_04_5 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.5');
/** Version "5.7.mysql_aurora.2.04.6". */
AuroraMysqlEngineVersion.VER_2_04_6 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.6');
/** Version "5.7.mysql_aurora.2.04.7". */
AuroraMysqlEngineVersion.VER_2_04_7 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.7');
/** Version "5.7.mysql_aurora.2.04.8". */
AuroraMysqlEngineVersion.VER_2_04_8 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.8');
/** Version "5.7.mysql_aurora.2.05.0". */
AuroraMysqlEngineVersion.VER_2_05_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.05.0');
/** Version "5.7.mysql_aurora.2.06.0". */
AuroraMysqlEngineVersion.VER_2_06_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.06.0');
/** Version "5.7.mysql_aurora.2.07.0". */
AuroraMysqlEngineVersion.VER_2_07_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.0');
/** Version "5.7.mysql_aurora.2.07.1". */
AuroraMysqlEngineVersion.VER_2_07_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.1');
/** Version "5.7.mysql_aurora.2.07.2". */
AuroraMysqlEngineVersion.VER_2_07_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.2');
/** Version "5.7.mysql_aurora.2.08.0". */
AuroraMysqlEngineVersion.VER_2_08_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.0');
/** Version "5.7.mysql_aurora.2.08.1". */
AuroraMysqlEngineVersion.VER_2_08_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.1');
class AuroraMysqlClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-mysql',
            engineVersion: version
                ? {
                    fullVersion: version.auroraMysqlFullVersion,
                    majorVersion: version.auroraMysqlMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.7',
        });
    }
    defaultParameterGroup(scope) {
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraMySqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * The versions for the Aurora PostgreSQL cluster engine
 * (those returned by {@link DatabaseClusterEngine.auroraPostgres}).
 */
class AuroraPostgresEngineVersion {
    constructor(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        this.auroraPostgresFullVersion = auroraPostgresFullVersion;
        this.auroraPostgresMajorVersion = auroraPostgresMajorVersion;
        this._features = {
            s3Import: (auroraPostgresFeatures === null || auroraPostgresFeatures === void 0 ? void 0 : auroraPostgresFeatures.s3Import) ? 's3Import' : undefined,
            s3Export: (auroraPostgresFeatures === null || auroraPostgresFeatures === void 0 ? void 0 : auroraPostgresFeatures.s3Export) ? 's3Export' : undefined,
        };
    }
    /**
     * Create a new AuroraPostgresEngineVersion with an arbitrary version.
     *
     * @param auroraPostgresFullVersion the full version string,
     *   for example "9.6.25.1"
     * @param auroraPostgresMajorVersion the major version of the engine,
     *   for example "9.6"
     */
    static of(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        return new AuroraPostgresEngineVersion(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures);
    }
}
exports.AuroraPostgresEngineVersion = AuroraPostgresEngineVersion;
/** Version "9.6.8". */
AuroraPostgresEngineVersion.VER_9_6_8 = AuroraPostgresEngineVersion.of('9.6.8', '9.6');
/** Version "9.6.9". */
AuroraPostgresEngineVersion.VER_9_6_9 = AuroraPostgresEngineVersion.of('9.6.9', '9.6');
/** Version "9.6.11". */
AuroraPostgresEngineVersion.VER_9_6_11 = AuroraPostgresEngineVersion.of('9.6.11', '9.6');
/** Version "9.6.12". */
AuroraPostgresEngineVersion.VER_9_6_12 = AuroraPostgresEngineVersion.of('9.6.12', '9.6');
/** Version "9.6.16". */
AuroraPostgresEngineVersion.VER_9_6_16 = AuroraPostgresEngineVersion.of('9.6.16', '9.6');
/** Version "9.6.17". */
AuroraPostgresEngineVersion.VER_9_6_17 = AuroraPostgresEngineVersion.of('9.6.17', '9.6');
/** Version "10.4". */
AuroraPostgresEngineVersion.VER_10_4 = AuroraPostgresEngineVersion.of('10.4', '10');
/** Version "10.5". */
AuroraPostgresEngineVersion.VER_10_5 = AuroraPostgresEngineVersion.of('10.5', '10');
/** Version "10.6". */
AuroraPostgresEngineVersion.VER_10_6 = AuroraPostgresEngineVersion.of('10.6', '10');
/** Version "10.7". */
AuroraPostgresEngineVersion.VER_10_7 = AuroraPostgresEngineVersion.of('10.7', '10', { s3Import: true });
/** Version "10.11". */
AuroraPostgresEngineVersion.VER_10_11 = AuroraPostgresEngineVersion.of('10.11', '10', { s3Import: true, s3Export: true });
/** Version "10.12". */
AuroraPostgresEngineVersion.VER_10_12 = AuroraPostgresEngineVersion.of('10.12', '10', { s3Import: true, s3Export: true });
/** Version "11.4". */
AuroraPostgresEngineVersion.VER_11_4 = AuroraPostgresEngineVersion.of('11.4', '11', { s3Import: true });
/** Version "11.6". */
AuroraPostgresEngineVersion.VER_11_6 = AuroraPostgresEngineVersion.of('11.6', '11', { s3Import: true, s3Export: true });
/** Version "11.7". */
AuroraPostgresEngineVersion.VER_11_7 = AuroraPostgresEngineVersion.of('11.7', '11', { s3Import: true, s3Export: true });
class AuroraPostgresClusterEngine extends ClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-postgresql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            defaultPort: 5432,
            engineVersion: version
                ? {
                    fullVersion: version.auroraPostgresFullVersion,
                    majorVersion: version.auroraPostgresMajorVersion,
                }
                : undefined,
            features: version
                ? {
                    s3Import: version._features.s3Import ? AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME : undefined,
                    s3Export: version._features.s3Export ? AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME : undefined,
                }
                : {
                    s3Import: AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME,
                    s3Export: AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME,
                },
        });
        this.supportedLogTypes = ['postgresql'];
    }
    bindToCluster(scope, options) {
        var _a, _b, _c;
        const config = super.bindToCluster(scope, options);
        // skip validation for unversioned as it might be supported/unsupported. we cannot reliably tell at compile-time
        if ((_a = this.engineVersion) === null || _a === void 0 ? void 0 : _a.fullVersion) {
            if (options.s3ImportRole && !((_b = config.features) === null || _b === void 0 ? void 0 : _b.s3Import)) {
                throw new Error(`s3Import is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Import feature.`);
            }
            if (options.s3ExportRole && !((_c = config.features) === null || _c === void 0 ? void 0 : _c.s3Export)) {
                throw new Error(`s3Export is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Export feature.`);
            }
        }
        return config;
    }
    defaultParameterGroup(scope) {
        if (!this.parameterGroupFamily) {
            throw new Error('Could not create a new ParameterGroup for an unversioned aurora-postgresql cluster engine. ' +
                'Please either use a versioned engine, or pass an explicit ParameterGroup when creating the cluster');
        }
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraPostgreSqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * feature name for the S3 data import feature
 */
AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME = 's3Import';
/**
 * feature name for the S3 data export feature
 */
AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME = 's3Export';
/**
 * A database cluster engine. Provides mapping to the serverless application
 * used for secret rotation.
 */
class DatabaseClusterEngine {
    /** Creates a new plain Aurora database cluster engine. */
    static aurora(props) {
        return new AuroraClusterEngine(props.version);
    }
    /** Creates a new Aurora MySQL database cluster engine. */
    static auroraMysql(props) {
        return new AuroraMysqlClusterEngine(props.version);
    }
    /** Creates a new Aurora PostgreSQL database cluster engine. */
    static auroraPostgres(props) {
        return new AuroraPostgresClusterEngine(props.version);
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
/**
 * The unversioned 'aurora' cluster engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link aurora()} method
 */
DatabaseClusterEngine.AURORA = new AuroraClusterEngine();
/**
 * The unversioned 'aurora-msql' cluster engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link auroraMysql()} method
 */
DatabaseClusterEngine.AURORA_MYSQL = new AuroraMysqlClusterEngine();
/**
 * The unversioned 'aurora-postgresql' cluster engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link auroraPostgres()} method
 */
DatabaseClusterEngine.AURORA_POSTGRESQL = new AuroraPostgresClusterEngine();
//# sourceMappingURL=data:application/json;base64,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