"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alias = void 0;
const appscaling = require("../../aws-applicationautoscaling"); // Automatically re-written from '@aws-cdk/aws-applicationautoscaling'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const function_base_1 = require("./function-base");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const scalable_function_attribute_1 = require("./private/scalable-function-attribute");
/**
 * A new alias to a particular version of a Lambda function.
 */
class Alias extends function_base_1.QualifiedFunctionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.aliasName,
        });
        this.canCreatePermissions = true;
        this.lambda = props.version.lambda;
        this.aliasName = this.physicalName;
        this.version = props.version;
        const alias = new lambda_generated_1.CfnAlias(this, 'Resource', {
            name: this.aliasName,
            description: props.description,
            functionName: this.version.lambda.functionName,
            functionVersion: props.version.version,
            routingConfig: this.determineRoutingConfig(props),
            provisionedConcurrencyConfig: this.determineProvisionedConcurrency(props),
        });
        // Use a Service Linked Role
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        this.scalingRole = iam.Role.fromRoleArn(this, 'ScalingRole', this.stack.formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/lambda.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_LambdaConcurrency',
        }));
        this.functionArn = this.getResourceArnAttribute(alias.ref, {
            service: 'lambda',
            resource: 'function',
            resourceName: `${this.lambda.functionName}:${this.physicalName}`,
            sep: ':',
        });
        this.qualifier = lambda_version_1.extractQualifierFromArn(alias.ref);
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        // ARN parsing splits on `:`, so we can only get the function's name from the ARN as resourceName...
        // And we're parsing it out (instead of using the underlying function directly) in order to have use of it incur
        // an implicit dependency on the resource.
        this.functionName = `${this.stack.parseArn(this.functionArn, ':').resourceName}:${this.aliasName}`;
    }
    static fromAliasAttributes(scope, id, attrs) {
        class Imported extends function_base_1.QualifiedFunctionBase {
            constructor() {
                super(...arguments);
                this.aliasName = attrs.aliasName;
                this.version = attrs.aliasVersion;
                this.lambda = attrs.aliasVersion.lambda;
                this.functionArn = `${attrs.aliasVersion.lambda.functionArn}:${attrs.aliasName}`;
                this.functionName = `${attrs.aliasVersion.lambda.functionName}:${attrs.aliasName}`;
                this.grantPrincipal = attrs.aliasVersion.grantPrincipal;
                this.role = attrs.aliasVersion.role;
                this.canCreatePermissions = this._isStackAccount();
                this.qualifier = attrs.aliasName;
            }
        }
        return new Imported(scope, id);
    }
    get grantPrincipal() {
        return this.version.grantPrincipal;
    }
    get role() {
        return this.version.role;
    }
    metric(metricName, props = {}) {
        // Metrics on Aliases need the "bare" function name, and the alias' ARN, this differes from the base behavior.
        return super.metric(metricName, {
            dimensions: {
                FunctionName: this.lambda.functionName,
                // construct the name from the underlying lambda so that alarms on an alias
                // don't cause a circular dependency with CodeDeploy
                // see: https://github.com/aws/aws-cdk/issues/2231
                Resource: `${this.lambda.functionName}:${this.aliasName}`,
            },
            ...props,
        });
    }
    /**
     * Configure provisioned concurrency autoscaling on a function alias. Returns a scalable attribute that can call
     * `scaleOnUtilization()` and `scaleOnSchedule()`.
     *
     * @param options Autoscaling options
     */
    addAutoScaling(options) {
        var _a;
        if (this.scalableAlias) {
            throw new Error('AutoScaling already enabled for this alias');
        }
        return this.scalableAlias = new scalable_function_attribute_1.ScalableFunctionAttribute(this, 'AliasScaling', {
            minCapacity: (_a = options.minCapacity) !== null && _a !== void 0 ? _a : 1,
            maxCapacity: options.maxCapacity,
            resourceId: `function:${this.functionName}`,
            dimension: 'lambda:function:ProvisionedConcurrency',
            serviceNamespace: appscaling.ServiceNamespace.LAMBDA,
            role: this.scalingRole,
        });
    }
    /**
     * Calculate the routingConfig parameter from the input props
     */
    determineRoutingConfig(props) {
        if (!props.additionalVersions || props.additionalVersions.length === 0) {
            return undefined;
        }
        this.validateAdditionalWeights(props.additionalVersions);
        return {
            additionalVersionWeights: props.additionalVersions.map(vw => {
                return {
                    functionVersion: vw.version.version,
                    functionWeight: vw.weight,
                };
            }),
        };
    }
    /**
     * Validate that the additional version weights make sense
     *
     * We validate that they are positive and add up to something <= 1.
     */
    validateAdditionalWeights(weights) {
        const total = weights.map(w => {
            if (w.weight < 0 || w.weight > 1) {
                throw new Error(`Additional version weight must be between 0 and 1, got: ${w.weight}`);
            }
            return w.weight;
        }).reduce((a, x) => a + x);
        if (total > 1) {
            throw new Error(`Sum of additional version weights must not exceed 1, got: ${total}`);
        }
    }
    /**
     * Validate that the provisionedConcurrentExecutions makes sense
     *
     * Member must have value greater than or equal to 1
     */
    determineProvisionedConcurrency(props) {
        if (!props.provisionedConcurrentExecutions) {
            return undefined;
        }
        if (props.provisionedConcurrentExecutions <= 0) {
            throw new Error('provisionedConcurrentExecutions must have value greater than or equal to 1');
        }
        return { provisionedConcurrentExecutions: props.provisionedConcurrentExecutions };
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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