"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Role = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const grant_1 = require("./grant");
const iam_generated_1 = require("./iam.generated");
const policy_1 = require("./policy");
const policy_document_1 = require("./policy-document");
const policy_statement_1 = require("./policy-statement");
const principals_1 = require("./principals");
const immutable_role_1 = require("./private/immutable-role");
const util_1 = require("./util");
/**
 * IAM Role
 *
 * Defines an IAM role. The role is created with an assume policy document associated with
 * the specified AWS service principal defined in `serviceAssumeRole`.
 */
class Role extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.roleName,
        });
        this.grantPrincipal = this;
        this.principalAccount = this.env.account;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.managedPolicies = [];
        this.attachedPolicies = new util_1.AttachedPolicies();
        const externalIds = props.externalIds || [];
        if (props.externalId) {
            externalIds.push(props.externalId);
        }
        this.assumeRolePolicy = createAssumeRolePolicy(props.assumedBy, externalIds);
        this.managedPolicies.push(...props.managedPolicies || []);
        this.inlinePolicies = props.inlinePolicies || {};
        this.permissionsBoundary = props.permissionsBoundary;
        const maxSessionDuration = props.maxSessionDuration && props.maxSessionDuration.toSeconds();
        validateMaxSessionDuration(maxSessionDuration);
        const description = (props.description && ((_a = props.description) === null || _a === void 0 ? void 0 : _a.length) > 0) ? props.description : undefined;
        if (description && description.length > 1000) {
            throw new Error('Role description must be no longer than 1000 characters.');
        }
        const role = new iam_generated_1.CfnRole(this, 'Resource', {
            assumeRolePolicyDocument: this.assumeRolePolicy,
            managedPolicyArns: core_1.Lazy.listValue({ produce: () => this.managedPolicies.map(p => p.managedPolicyArn) }, { omitEmpty: true }),
            policies: _flatten(this.inlinePolicies),
            path: props.path,
            permissionsBoundary: this.permissionsBoundary ? this.permissionsBoundary.managedPolicyArn : undefined,
            roleName: this.physicalName,
            maxSessionDuration,
            description,
        });
        this.roleId = role.attrRoleId;
        this.roleArn = this.getResourceArnAttribute(role.attrArn, {
            region: '',
            service: 'iam',
            resource: 'role',
            resourceName: this.physicalName,
        });
        this.roleName = this.getResourceNameAttribute(role.ref);
        this.policyFragment = new principals_1.ArnPrincipal(this.roleArn).policyFragment;
        function _flatten(policies) {
            if (policies == null || Object.keys(policies).length === 0) {
                return undefined;
            }
            const result = new Array();
            for (const policyName of Object.keys(policies)) {
                const policyDocument = policies[policyName];
                result.push({ policyName, policyDocument });
            }
            return result;
        }
    }
    /**
     * Import an external role by ARN.
     *
     * If the imported Role ARN is a Token (such as a
     * `CfnParameter.valueAsString` or a `Fn.importValue()`) *and* the referenced
     * role has a `path` (like `arn:...:role/AdminRoles/Alice`), the
     * `roleName` property will not resolve to the correct value. Instead it
     * will resolve to the first path component. We unfortunately cannot express
     * the correct calculation of the full path name as a CloudFormation
     * expression. In this scenario the Role ARN should be supplied without the
     * `path` in order to resolve the correct role resource.
     *
     * @param scope construct scope
     * @param id construct id
     * @param roleArn the ARN of the role to import
     * @param options allow customizing the behavior of the returned role
     */
    static fromRoleArn(scope, id, roleArn, options = {}) {
        const scopeStack = core_1.Stack.of(scope);
        const parsedArn = scopeStack.parseArn(roleArn);
        const resourceName = parsedArn.resourceName;
        const roleAccount = parsedArn.account;
        // service roles have an ARN like 'arn:aws:iam::<account>:role/service-role/<roleName>'
        // or 'arn:aws:iam::<account>:role/service-role/servicename.amazonaws.com/service-role/<roleName>'
        // we want to support these as well, so we just use the element after the last slash as role name
        const roleName = resourceName.split('/').pop();
        class Import extends core_1.Resource {
            constructor(_scope, _id) {
                super(_scope, _id, {
                    account: roleAccount,
                });
                this.grantPrincipal = this;
                this.principalAccount = roleAccount;
                this.assumeRoleAction = 'sts:AssumeRole';
                this.policyFragment = new principals_1.ArnPrincipal(roleArn).policyFragment;
                this.roleArn = roleArn;
                this.roleName = roleName;
                this.attachedPolicies = new util_1.AttachedPolicies();
            }
            addToPolicy(statement) {
                return this.addToPrincipalPolicy(statement).statementAdded;
            }
            addToPrincipalPolicy(statement) {
                if (!this.defaultPolicy) {
                    this.defaultPolicy = new policy_1.Policy(this, 'Policy');
                    this.attachInlinePolicy(this.defaultPolicy);
                }
                this.defaultPolicy.addStatements(statement);
                return { statementAdded: true, policyDependable: this.defaultPolicy };
            }
            attachInlinePolicy(policy) {
                const thisAndPolicyAccountComparison = core_1.Token.compareStrings(this.env.account, policy.env.account);
                const equalOrAnyUnresolved = thisAndPolicyAccountComparison === core_1.TokenComparison.SAME ||
                    thisAndPolicyAccountComparison === core_1.TokenComparison.BOTH_UNRESOLVED ||
                    thisAndPolicyAccountComparison === core_1.TokenComparison.ONE_UNRESOLVED;
                if (equalOrAnyUnresolved) {
                    this.attachedPolicies.attach(policy);
                    policy.attachToRole(this);
                }
            }
            addManagedPolicy(_policy) {
                // FIXME: Add warning that we're ignoring this
            }
            /**
             * Grant permissions to the given principal to pass this role.
             */
            grantPassRole(identity) {
                return this.grant(identity, 'iam:PassRole');
            }
            /**
             * Grant the actions defined in actions to the identity Principal on this resource.
             */
            grant(grantee, ...actions) {
                return grant_1.Grant.addToPrincipal({
                    grantee,
                    actions,
                    resourceArns: [this.roleArn],
                    scope: this,
                });
            }
        }
        const importedRole = new Import(scope, id);
        const roleArnAndScopeStackAccountComparison = core_1.Token.compareStrings(importedRole.env.account, scopeStack.account);
        const equalOrAnyUnresolved = roleArnAndScopeStackAccountComparison === core_1.TokenComparison.SAME ||
            roleArnAndScopeStackAccountComparison === core_1.TokenComparison.BOTH_UNRESOLVED ||
            roleArnAndScopeStackAccountComparison === core_1.TokenComparison.ONE_UNRESOLVED;
        // we only return an immutable Role if both accounts were explicitly provided, and different
        return options.mutable !== false && equalOrAnyUnresolved
            ? importedRole
            : new immutable_role_1.ImmutableRole(scope, `ImmutableRole${id}`, importedRole);
    }
    /**
     * Adds a permission to the role's default policy document.
     * If there is no default policy attached to this role, it will be created.
     * @param statement The permission statement to add to the policy document
     */
    addToPrincipalPolicy(statement) {
        if (!this.defaultPolicy) {
            this.defaultPolicy = new policy_1.Policy(this, 'DefaultPolicy');
            this.attachInlinePolicy(this.defaultPolicy);
        }
        this.defaultPolicy.addStatements(statement);
        return { statementAdded: true, policyDependable: this.defaultPolicy };
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    /**
     * Attaches a managed policy to this role.
     * @param policy The the managed policy to attach.
     */
    addManagedPolicy(policy) {
        if (this.managedPolicies.find(mp => mp === policy)) {
            return;
        }
        this.managedPolicies.push(policy);
    }
    /**
     * Attaches a policy to this role.
     * @param policy The policy to attach
     */
    attachInlinePolicy(policy) {
        this.attachedPolicies.attach(policy);
        policy.attachToRole(this);
    }
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(grantee, ...actions) {
        return grant_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.roleArn],
            scope: this,
        });
    }
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(identity) {
        return this.grant(identity, 'iam:PassRole');
    }
    /**
     * Return a copy of this Role object whose Policies will not be updated
     *
     * Use the object returned by this method if you want this Role to be used by
     * a construct without it automatically updating the Role's Policies.
     *
     * If you do, you are responsible for adding the correct statements to the
     * Role's policies yourself.
     */
    withoutPolicyUpdates() {
        if (!this.immutableRole) {
            this.immutableRole = new immutable_role_1.ImmutableRole(this.node.scope, `ImmutableRole${this.node.id}`, this);
        }
        return this.immutableRole;
    }
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.assumeRolePolicy) === null || _a === void 0 ? void 0 : _a.validateForResourcePolicy()) || []);
        for (const policy of Object.values(this.inlinePolicies)) {
            errors.push(...policy.validateForIdentityPolicy());
        }
        return errors;
    }
}
exports.Role = Role;
function createAssumeRolePolicy(principal, externalIds) {
    const statement = new AwsStarStatement();
    statement.addPrincipals(principal);
    statement.addActions(principal.assumeRoleAction);
    if (externalIds.length) {
        statement.addCondition('StringEquals', { 'sts:ExternalId': externalIds.length === 1 ? externalIds[0] : externalIds });
    }
    const doc = new policy_document_1.PolicyDocument();
    doc.addStatements(statement);
    return doc;
}
function validateMaxSessionDuration(duration) {
    if (duration === undefined) {
        return;
    }
    if (duration < 3600 || duration > 43200) {
        throw new Error(`maxSessionDuration is set to ${duration}, but must be >= 3600sec (1hr) and <= 43200sec (12hrs)`);
    }
}
/**
 * A PolicyStatement that normalizes its Principal field differently
 *
 * Normally, "anyone" is normalized to "Principal: *", but this statement
 * normalizes to "Principal: { AWS: * }".
 */
class AwsStarStatement extends policy_statement_1.PolicyStatement {
    toStatementJson() {
        const stat = super.toStatementJson();
        if (stat.Principal === '*') {
            stat.Principal = { AWS: '*' };
        }
        return stat;
    }
}
//# sourceMappingURL=data:application/json;base64,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