"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AcceleratorSecurityGroup = void 0;
const aws_ec2_1 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const custom_resources_1 = require("../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
/**
 * The security group used by a Global Accelerator to send traffic to resources in a VPC.
 */
class AcceleratorSecurityGroup {
    /**
     * Lookup the Global Accelerator security group at CloudFormation deployment time.
     *
     * As of this writing, Global Accelerators (AGA) create a single security group per VPC. AGA security groups are shared
     * by all AGAs in an account. Additionally, there is no CloudFormation mechanism to reference the AGA security groups.
     *
     * This makes creating security group rules which allow traffic from an AGA complicated in CDK. This lookup will identify
     * the AGA security group for a given VPC at CloudFormation deployment time, and lets you create rules for traffic from AGA
     * to other resources created by CDK.
     */
    static fromVpc(scope, id, vpc, endpointGroup) {
        // The security group name is always 'GlobalAccelerator'
        const globalAcceleratorSGName = 'GlobalAccelerator';
        // How to reference the security group name in the response from EC2
        const ec2ResponseSGIdField = 'SecurityGroups.0.GroupId';
        // The AWS Custom Resource that make a call to EC2 to get the security group ID, for the given VPC
        const lookupAcceleratorSGCustomResource = new custom_resources_1.AwsCustomResource(scope, id + 'CustomResource', {
            onCreate: {
                service: 'EC2',
                action: 'describeSecurityGroups',
                parameters: {
                    Filters: [
                        {
                            Name: 'group-name',
                            Values: [
                                globalAcceleratorSGName,
                            ],
                        },
                        {
                            Name: 'vpc-id',
                            Values: [
                                vpc.vpcId,
                            ],
                        },
                    ],
                },
                // We get back a list of responses, but the list should be of length 0 or 1
                // Getting no response means no resources have been linked to the AGA
                physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse(ec2ResponseSGIdField),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        // Look up the security group ID
        const sg = aws_ec2_1.SecurityGroup.fromSecurityGroupId(scope, id, lookupAcceleratorSGCustomResource.getResponseField(ec2ResponseSGIdField));
        // We add a dependency on the endpoint group, guaranteeing that CloudFormation won't
        // try and look up the SG before AGA creates it. The SG is created when a VPC resource
        // is associated with an AGA
        lookupAcceleratorSGCustomResource.node.addDependency(endpointGroup);
        return sg;
    }
    constructor() { }
}
exports.AcceleratorSecurityGroup = AcceleratorSecurityGroup;
//# sourceMappingURL=data:application/json;base64,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