"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsTask = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const ecs = require("../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const util_1 = require("./util");
/**
 * Start a task on an ECS cluster
 */
class EcsTask {
    constructor(props) {
        this.props = props;
        if (props.securityGroup !== undefined && props.securityGroups !== undefined) {
            throw new Error('Only one of SecurityGroup or SecurityGroups can be populated.');
        }
        this.cluster = props.cluster;
        this.taskDefinition = props.taskDefinition;
        this.taskCount = props.taskCount !== undefined ? props.taskCount : 1;
        this.platformVersion = props.platformVersion;
        if (props.role) {
            const role = props.role;
            this.createEventRolePolicyStatements().forEach(role.addToPolicy.bind(role));
            this.role = role;
        }
        else {
            this.role = util_1.singletonEventRole(this.taskDefinition, this.createEventRolePolicyStatements());
        }
        // Security groups are only configurable with the "awsvpc" network mode.
        if (this.taskDefinition.networkMode !== ecs.NetworkMode.AWS_VPC) {
            if (props.securityGroup !== undefined || props.securityGroups !== undefined) {
                cdk.Annotations.of(this.taskDefinition).addWarning('security groups are ignored when network mode is not awsvpc');
            }
            return;
        }
        if (props.securityGroups) {
            this.securityGroups = props.securityGroups;
            return;
        }
        let securityGroup = props.securityGroup || this.taskDefinition.node.tryFindChild('SecurityGroup');
        securityGroup = securityGroup || new ec2.SecurityGroup(this.taskDefinition, 'SecurityGroup', { vpc: this.props.cluster.vpc });
        this.securityGroup = securityGroup; // Maintain backwards-compatibility for customers that read the generated security group.
        this.securityGroups = [securityGroup];
    }
    /**
     * Allows using tasks as target of EventBridge events
     */
    bind(_rule, _id) {
        const arn = this.cluster.clusterArn;
        const role = this.role;
        const containerOverrides = this.props.containerOverrides && this.props.containerOverrides
            .map(({ containerName, ...overrides }) => ({ name: containerName, ...overrides }));
        const input = { containerOverrides };
        const taskCount = this.taskCount;
        const taskDefinitionArn = this.taskDefinition.taskDefinitionArn;
        const subnetSelection = this.props.subnetSelection || { subnetType: ec2.SubnetType.PRIVATE };
        const assignPublicIp = subnetSelection.subnetType === ec2.SubnetType.PUBLIC ? 'ENABLED' : 'DISABLED';
        const baseEcsParameters = { taskCount, taskDefinitionArn };
        const ecsParameters = this.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC
            ? {
                ...baseEcsParameters,
                launchType: this.taskDefinition.isEc2Compatible ? 'EC2' : 'FARGATE',
                platformVersion: this.platformVersion,
                networkConfiguration: {
                    awsVpcConfiguration: {
                        subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                        assignPublicIp,
                        securityGroups: this.securityGroups && this.securityGroups.map(sg => sg.securityGroupId),
                    },
                },
            }
            : baseEcsParameters;
        return {
            id: '',
            arn,
            role,
            ecsParameters,
            input: events.RuleTargetInput.fromObject(input),
            targetResource: this.taskDefinition,
        };
    }
    createEventRolePolicyStatements() {
        const policyStatements = [new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.taskDefinition.taskDefinitionArn],
                conditions: {
                    ArnEquals: { 'ecs:cluster': this.cluster.clusterArn },
                },
            })];
        // If it so happens that a Task Execution Role was created for the TaskDefinition,
        // then the EventBridge Role must have permissions to pass it (otherwise it doesn't).
        if (this.taskDefinition.executionRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.executionRole.roleArn],
            }));
        }
        // For Fargate task we need permission to pass the task role.
        if (this.taskDefinition.isFargateCompatible) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.taskRole.roleArn],
            }));
        }
        return policyStatements;
    }
}
exports.EcsTask = EcsTask;
//# sourceMappingURL=data:application/json;base64,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