"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LegacyCluster = void 0;
const autoscaling = require("../../aws-autoscaling"); // Automatically re-written from '@aws-cdk/aws-autoscaling'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cluster_1 = require("./cluster");
const cluster_resource_1 = require("./cluster-resource");
const eks_generated_1 = require("./eks.generated");
const managed_nodegroup_1 = require("./managed-nodegroup");
const user_data_1 = require("./user-data");
// defaults are based on https://eksctl.io
const DEFAULT_CAPACITY_COUNT = 2;
const DEFAULT_CAPACITY_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
/**
 * A Cluster represents a managed Kubernetes Service (EKS)
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 *
 * @resource AWS::EKS::Cluster
 */
class LegacyCluster extends core_1.Resource {
    /**
     * Initiates an EKS Cluster with the supplied arguments
     *
     * @param scope a Construct, most likely a cdk.Stack created
     * @param name the name of the Construct to create
     * @param props properties in the IClusterProps interface
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        const stack = core_1.Stack.of(this);
        this.vpc = props.vpc || new ec2.Vpc(this, 'DefaultVpc');
        this.version = props.version;
        this.tagSubnets();
        // this is the role used by EKS when interacting with AWS resources
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('eks.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy'),
            ],
        });
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'ControlPlaneSecurityGroup', {
            vpc: this.vpc,
            description: 'EKS Control Plane Security Group',
        });
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(443),
        });
        // Get subnetIds for all selected subnets
        const placements = props.vpcSubnets || [{ subnetType: ec2.SubnetType.PUBLIC }, { subnetType: ec2.SubnetType.PRIVATE }];
        const subnetIds = [...new Set(Array().concat(...placements.map(s => this.vpc.selectSubnets(s).subnetIds)))];
        const clusterProps = {
            name: this.physicalName,
            roleArn: this.role.roleArn,
            version: props.version.version,
            resourcesVpcConfig: {
                securityGroupIds: [securityGroup.securityGroupId],
                subnetIds,
            },
            ...(props.secretsEncryptionKey ? {
                encryptionConfig: [{
                        provider: {
                            keyArn: props.secretsEncryptionKey.keyArn,
                        },
                        resources: ['secrets'],
                    }],
            } : {}),
        };
        const resource = new eks_generated_1.CfnCluster(this, 'Resource', clusterProps);
        this.clusterName = this.getResourceNameAttribute(resource.ref);
        this.clusterArn = this.getResourceArnAttribute(resource.attrArn, cluster_resource_1.clusterArnComponents(this.physicalName));
        this.clusterEndpoint = resource.attrEndpoint;
        this.clusterCertificateAuthorityData = resource.attrCertificateAuthorityData;
        this.clusterSecurityGroupId = resource.attrClusterSecurityGroupId;
        this.clusterEncryptionConfigKeyArn = resource.attrEncryptionConfigKeyArn;
        const updateConfigCommandPrefix = `aws eks update-kubeconfig --name ${this.clusterName}`;
        const getTokenCommandPrefix = `aws eks get-token --cluster-name ${this.clusterName}`;
        const commonCommandOptions = [`--region ${stack.region}`];
        if (props.outputClusterName) {
            new core_1.CfnOutput(this, 'ClusterName', { value: this.clusterName });
        }
        // allocate default capacity if non-zero (or default).
        const minCapacity = props.defaultCapacity === undefined ? DEFAULT_CAPACITY_COUNT : props.defaultCapacity;
        if (minCapacity > 0) {
            const instanceType = props.defaultCapacityInstance || DEFAULT_CAPACITY_TYPE;
            this.defaultCapacity = props.defaultCapacityType === cluster_1.DefaultCapacityType.EC2 ?
                this.addCapacity('DefaultCapacity', { instanceType, minCapacity }) : undefined;
            this.defaultNodegroup = props.defaultCapacityType !== cluster_1.DefaultCapacityType.EC2 ?
                this.addNodegroup('DefaultCapacity', { instanceType, minSize: minCapacity }) : undefined;
        }
        const outputConfigCommand = props.outputConfigCommand === undefined ? true : props.outputConfigCommand;
        if (outputConfigCommand) {
            const postfix = commonCommandOptions.join(' ');
            new core_1.CfnOutput(this, 'ConfigCommand', { value: `${updateConfigCommandPrefix} ${postfix}` });
            new core_1.CfnOutput(this, 'GetTokenCommand', { value: `${getTokenCommandPrefix} ${postfix}` });
        }
    }
    /**
     * Import an existing cluster
     *
     * @param scope the construct scope, in most cases 'this'
     * @param id the id or name to import as
     * @param attrs the cluster properties to use for importing information
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Add nodes to this EKS cluster
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     */
    addCapacity(id, options) {
        if (options.machineImageType === cluster_1.MachineImageType.BOTTLEROCKET && options.bootstrapOptions !== undefined) {
            throw new Error('bootstrapOptions is not supported for Bottlerocket');
        }
        const asg = new autoscaling.AutoScalingGroup(this, id, {
            ...options,
            vpc: this.vpc,
            machineImage: options.machineImageType === cluster_1.MachineImageType.BOTTLEROCKET ?
                new BottleRocketImage() :
                new cluster_1.EksOptimizedImage({
                    nodeType: nodeTypeForInstanceType(options.instanceType),
                    kubernetesVersion: this.version.version,
                }),
            updateType: options.updateType || autoscaling.UpdateType.ROLLING_UPDATE,
            instanceType: options.instanceType,
        });
        this.addAutoScalingGroup(asg, {
            mapRole: options.mapRole,
            bootstrapOptions: options.bootstrapOptions,
            bootstrapEnabled: options.bootstrapEnabled,
            machineImageType: options.machineImageType,
        });
        return asg;
    }
    /**
     * Add managed nodegroup to this Amazon EKS cluster
     *
     * This method will create a new managed nodegroup and add into the capacity.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     * @param id The ID of the nodegroup
     * @param options options for creating a new nodegroup
     */
    addNodegroup(id, options) {
        return new managed_nodegroup_1.Nodegroup(this, `Nodegroup${id}`, {
            cluster: this,
            ...options,
        });
    }
    /**
     * Add compute capacity to this EKS cluster in the form of an AutoScalingGroup
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * Prefer to use `addCapacity` if possible.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     * @param autoScalingGroup [disable-awslint:ref-via-interface]
     * @param options options for adding auto scaling groups, like customizing the bootstrap script
     */
    addAutoScalingGroup(autoScalingGroup, options) {
        // self rules
        autoScalingGroup.connections.allowInternally(ec2.Port.allTraffic());
        // Cluster to:nodes rules
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcp(443));
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcpRange(1025, 65535));
        // Allow HTTPS from Nodes to Cluster
        autoScalingGroup.connections.allowTo(this, ec2.Port.tcp(443));
        // Allow all node outbound traffic
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allTcp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allUdp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allIcmp());
        const bootstrapEnabled = options.bootstrapEnabled !== undefined ? options.bootstrapEnabled : true;
        if (options.bootstrapOptions && !bootstrapEnabled) {
            throw new Error('Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false');
        }
        if (bootstrapEnabled) {
            const userData = options.machineImageType === cluster_1.MachineImageType.BOTTLEROCKET ?
                user_data_1.renderBottlerocketUserData(this) :
                user_data_1.renderAmazonLinuxUserData(this.clusterName, autoScalingGroup, options.bootstrapOptions);
            autoScalingGroup.addUserData(...userData);
        }
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        // EKS Required Tags
        core_1.Tags.of(autoScalingGroup).add(`kubernetes.io/cluster/${this.clusterName}`, 'owned', {
            applyToLaunchedInstances: true,
        });
        if (options.mapRole) {
            throw new Error('Cannot map instance IAM role to RBAC if kubectl is disabled for the cluster');
        }
        // since we are not mapping the instance role to RBAC, synthesize an
        // output so it can be pasted into `aws-auth-cm.yaml`
        new core_1.CfnOutput(autoScalingGroup, 'InstanceRoleARN', {
            value: autoScalingGroup.role.roleArn,
        });
    }
    addManifest(_id, ..._manifest) {
        throw new Error('legacy cluster does not support adding kubernetes manifests');
    }
    addChart(_id, _options) {
        throw new Error('legacy cluster does not support adding helm charts');
    }
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    tagSubnets() {
        const tagAllSubnets = (type, subnets, tag) => {
            for (const subnet of subnets) {
                // if this is not a concrete subnet, attach a construct warning
                if (!ec2.Subnet.isVpcSubnet(subnet)) {
                    // message (if token): "could not auto-tag public/private subnet with tag..."
                    // message (if not token): "count not auto-tag public/private subnet xxxxx with tag..."
                    const subnetID = core_1.Token.isUnresolved(subnet.subnetId) ? '' : ` ${subnet.subnetId}`;
                    core_1.Annotations.of(this).addWarning(`Could not auto-tag ${type} subnet${subnetID} with "${tag}=1", please remember to do this manually`);
                    continue;
                }
                core_1.Tags.of(subnet).add(tag, '1');
            }
        };
        // https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
        tagAllSubnets('private', this.vpc.privateSubnets, 'kubernetes.io/role/internal-elb');
        tagAllSubnets('public', this.vpc.publicSubnets, 'kubernetes.io/role/elb');
    }
}
exports.LegacyCluster = LegacyCluster;
/**
 * Import a cluster to use in another stack
 */
class ImportedCluster extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.props = props;
        this.connections = new ec2.Connections();
        this.clusterName = props.clusterName;
        this.clusterArn = this.stack.formatArn(cluster_resource_1.clusterArnComponents(props.clusterName));
        let i = 1;
        for (const sgid of (_a = props.securityGroupIds) !== null && _a !== void 0 ? _a : []) {
            this.connections.addSecurityGroup(ec2.SecurityGroup.fromSecurityGroupId(this, `SecurityGroup${i}`, sgid));
            i++;
        }
    }
    addManifest(_id, ..._manifest) {
        throw new Error('legacy cluster does not support adding kubernetes manifests');
    }
    addChart(_id, _options) {
        throw new Error('legacy cluster does not support adding helm charts');
    }
    get vpc() {
        if (!this.props.vpc) {
            throw new Error('"vpc" is not defined for this imported cluster');
        }
        return this.props.vpc;
    }
    get clusterSecurityGroupId() {
        if (!this.props.clusterSecurityGroupId) {
            throw new Error('"clusterSecurityGroupId" is not defined for this imported cluster');
        }
        return this.props.clusterSecurityGroupId;
    }
    get clusterEndpoint() {
        if (!this.props.clusterEndpoint) {
            throw new Error('"clusterEndpoint" is not defined for this imported cluster');
        }
        return this.props.clusterEndpoint;
    }
    get clusterCertificateAuthorityData() {
        if (!this.props.clusterCertificateAuthorityData) {
            throw new Error('"clusterCertificateAuthorityData" is not defined for this imported cluster');
        }
        return this.props.clusterCertificateAuthorityData;
    }
    get clusterEncryptionConfigKeyArn() {
        if (!this.props.clusterEncryptionConfigKeyArn) {
            throw new Error('"clusterEncryptionConfigKeyArn" is not defined for this imported cluster');
        }
        return this.props.clusterEncryptionConfigKeyArn;
    }
}
/**
 * Construct an Bottlerocket image from the latest AMI published in SSM
 */
class BottleRocketImage {
    /**
     * Constructs a new instance of the BottleRocketImage class.
     */
    constructor() {
        // only 1.15 is currently available
        this.kubernetesVersion = '1.15';
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/bottlerocket/aws-k8s-${this.kubernetesVersion}/x86_64/latest/image_id`;
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.custom(''),
        };
    }
}
const GPU_INSTANCETYPES = ['p2', 'p3', 'g4'];
const INFERENTIA_INSTANCETYPES = ['inf1'];
function nodeTypeForInstanceType(instanceType) {
    return GPU_INSTANCETYPES.includes(instanceType.toString().substring(0, 2)) ? cluster_1.NodeType.GPU :
        INFERENTIA_INSTANCETYPES.includes(instanceType.toString().substring(0, 4)) ? cluster_1.NodeType.INFERENTIA :
            cluster_1.NodeType.STANDARD;
}
//# sourceMappingURL=data:application/json;base64,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