"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Compatibility = exports.Scope = exports.PidMode = exports.IpcMode = exports.NetworkMode = exports.TaskDefinition = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const container_definition_1 = require("../container-definition");
const ecs_generated_1 = require("../ecs.generated");
const firelens_log_router_1 = require("../firelens-log-router");
const aws_log_driver_1 = require("../log-drivers/aws-log-driver");
class TaskDefinitionBase extends core_1.Resource {
    /**
     * Return true if the task definition can be run on an EC2 cluster
     */
    get isEc2Compatible() {
        return isEc2Compatible(this.compatibility);
    }
    /**
     * Return true if the task definition can be run on a Fargate cluster
     */
    get isFargateCompatible() {
        return isFargateCompatible(this.compatibility);
    }
}
/**
 * The base class for all task definitions.
 */
class TaskDefinition extends TaskDefinitionBase {
    /**
     * Constructs a new instance of the TaskDefinition class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The container definitions.
         */
        this.containers = new Array();
        /**
         * All volumes
         */
        this.volumes = [];
        /**
         * Placement constraints for task instances
         */
        this.placementConstraints = new Array();
        this.family = props.family || this.node.uniqueId;
        this.compatibility = props.compatibility;
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        this.networkMode = props.networkMode !== undefined ? props.networkMode :
            this.isFargateCompatible ? NetworkMode.AWS_VPC : NetworkMode.BRIDGE;
        if (this.isFargateCompatible && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`Fargate tasks can only have AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.proxyConfiguration && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`ProxyConfiguration can only be used with AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.placementConstraints && props.placementConstraints.length > 0 && this.isFargateCompatible) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        if (this.isFargateCompatible && (!props.cpu || !props.memoryMiB)) {
            throw new Error(`Fargate-compatible tasks require both CPU (${props.cpu}) and memory (${props.memoryMiB}) specifications`);
        }
        this._executionRole = props.executionRole;
        this.taskRole = props.taskRole || new iam.Role(this, 'TaskRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        const taskDef = new ecs_generated_1.CfnTaskDefinition(this, 'Resource', {
            containerDefinitions: core_1.Lazy.anyValue({ produce: () => this.renderContainers() }, { omitEmptyArray: true }),
            volumes: core_1.Lazy.anyValue({ produce: () => this.renderVolumes() }, { omitEmptyArray: true }),
            executionRoleArn: core_1.Lazy.stringValue({ produce: () => this.executionRole && this.executionRole.roleArn }),
            family: this.family,
            taskRoleArn: this.taskRole.roleArn,
            requiresCompatibilities: [
                ...(isEc2Compatible(props.compatibility) ? ['EC2'] : []),
                ...(isFargateCompatible(props.compatibility) ? ['FARGATE'] : []),
            ],
            networkMode: this.renderNetworkMode(this.networkMode),
            placementConstraints: core_1.Lazy.anyValue({
                produce: () => !isFargateCompatible(this.compatibility) ? this.placementConstraints : undefined,
            }, { omitEmptyArray: true }),
            proxyConfiguration: props.proxyConfiguration ? props.proxyConfiguration.bind(this.stack, this) : undefined,
            cpu: props.cpu,
            memory: props.memoryMiB,
            ipcMode: props.ipcMode,
            pidMode: props.pidMode,
        });
        if (props.placementConstraints) {
            props.placementConstraints.forEach(pc => this.addPlacementConstraint(pc));
        }
        this.taskDefinitionArn = taskDef.ref;
    }
    /**
     * Imports a task definition from the specified task definition ARN.
     *
     * The task will have a compatibility of EC2+Fargate.
     */
    static fromTaskDefinitionArn(scope, id, taskDefinitionArn) {
        class Import extends TaskDefinitionBase {
            constructor() {
                super(...arguments);
                this.taskDefinitionArn = taskDefinitionArn;
                this.compatibility = Compatibility.EC2_AND_FARGATE;
                this.executionRole = undefined;
            }
        }
        return new Import(scope, id);
    }
    get executionRole() {
        return this._executionRole;
    }
    renderVolumes() {
        return this.volumes.map(renderVolume);
        function renderVolume(spec) {
            return {
                host: spec.host,
                name: spec.name,
                dockerVolumeConfiguration: spec.dockerVolumeConfiguration && {
                    autoprovision: spec.dockerVolumeConfiguration.autoprovision,
                    driver: spec.dockerVolumeConfiguration.driver,
                    driverOpts: spec.dockerVolumeConfiguration.driverOpts,
                    labels: spec.dockerVolumeConfiguration.labels,
                    scope: spec.dockerVolumeConfiguration.scope,
                },
                efsVolumeConfiguration: spec.efsVolumeConfiguration && {
                    fileSystemId: spec.efsVolumeConfiguration.fileSystemId,
                    authorizationConfig: spec.efsVolumeConfiguration.authorizationConfig,
                    rootDirectory: spec.efsVolumeConfiguration.rootDirectory,
                    transitEncryption: spec.efsVolumeConfiguration.transitEncryption,
                    transitEncryptionPort: spec.efsVolumeConfiguration.transitEncryptionPort,
                },
            };
        }
    }
    /**
     * Validate the existence of the input target and set default values.
     *
     * @internal
     */
    _validateTarget(options) {
        const targetContainer = this.findContainer(options.containerName);
        if (targetContainer === undefined) {
            throw new Error(`No container named '${options.containerName}'. Did you call "addContainer()"?`);
        }
        const targetProtocol = options.protocol || container_definition_1.Protocol.TCP;
        const targetContainerPort = options.containerPort || targetContainer.containerPort;
        const portMapping = targetContainer.findPortMapping(targetContainerPort, targetProtocol);
        if (portMapping === undefined) {
            // eslint-disable-next-line max-len
            throw new Error(`Container '${targetContainer}' has no mapping for port ${options.containerPort} and protocol ${targetProtocol}. Did you call "container.addPortMappings()"?`);
        }
        return {
            containerName: options.containerName,
            portMapping,
        };
    }
    /**
     * Returns the port range to be opened that match the provided container name and container port.
     *
     * @internal
     */
    _portRangeFromPortMapping(portMapping) {
        if (portMapping.hostPort !== undefined && portMapping.hostPort !== 0) {
            return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.hostPort) : ec2.Port.tcp(portMapping.hostPort);
        }
        if (this.networkMode === NetworkMode.BRIDGE || this.networkMode === NetworkMode.NAT) {
            return EPHEMERAL_PORT_RANGE;
        }
        return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.containerPort) : ec2.Port.tcp(portMapping.containerPort);
    }
    /**
     * Adds a policy statement to the task IAM role.
     */
    addToTaskRolePolicy(statement) {
        this.taskRole.addToPolicy(statement);
    }
    /**
     * Adds a policy statement to the task execution IAM role.
     */
    addToExecutionRolePolicy(statement) {
        this.obtainExecutionRole().addToPolicy(statement);
    }
    /**
     * Adds a new container to the task definition.
     */
    addContainer(id, props) {
        return new container_definition_1.ContainerDefinition(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Adds a firelens log router to the task definition.
     */
    addFirelensLogRouter(id, props) {
        // only one firelens log router is allowed in each task.
        if (this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
            throw new Error('Firelens log router is already added in this task.');
        }
        return new firelens_log_router_1.FirelensLogRouter(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Links a container to this task definition.
     * @internal
     */
    _linkContainer(container) {
        this.containers.push(container);
        if (this.defaultContainer === undefined && container.essential) {
            this.defaultContainer = container;
        }
    }
    /**
     * Adds a volume to the task definition.
     */
    addVolume(volume) {
        this.volumes.push(volume);
    }
    /**
     * Adds the specified placement constraint to the task definition.
     */
    addPlacementConstraint(constraint) {
        if (isFargateCompatible(this.compatibility)) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        this.placementConstraints.push(...constraint.toJson());
    }
    /**
     * Adds the specified extention to the task definition.
     *
     * Extension can be used to apply a packaged modification to
     * a task definition.
     */
    addExtension(extension) {
        extension.extend(this);
    }
    /**
     * Creates the task execution IAM role if it doesn't already exist.
     */
    obtainExecutionRole() {
        if (!this._executionRole) {
            this._executionRole = new iam.Role(this, 'ExecutionRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
        }
        return this._executionRole;
    }
    /**
     * Validates the task definition.
     */
    validate() {
        const ret = super.validate();
        if (isEc2Compatible(this.compatibility)) {
            // EC2 mode validations
            // Container sizes
            for (const container of this.containers) {
                if (!container.memoryLimitSpecified) {
                    ret.push(`ECS Container ${container.containerName} must have at least one of 'memoryLimitMiB' or 'memoryReservationMiB' specified`);
                }
            }
        }
        return ret;
    }
    /**
     * Returns the container that match the provided containerName.
     */
    findContainer(containerName) {
        return this.containers.find(c => c.containerName === containerName);
    }
    renderNetworkMode(networkMode) {
        return (networkMode === NetworkMode.NAT) ? undefined : networkMode;
    }
    renderContainers() {
        // add firelens log router container if any application container is using firelens log driver,
        // also check if already created log router container
        for (const container of this.containers) {
            if (container.logDriverConfig && container.logDriverConfig.logDriver === 'awsfirelens'
                && !this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
                this.addFirelensLogRouter('log-router', {
                    image: firelens_log_router_1.obtainDefaultFluentBitECRImage(this, container.logDriverConfig),
                    firelensConfig: {
                        type: firelens_log_router_1.FirelensLogRouterType.FLUENTBIT,
                    },
                    logging: new aws_log_driver_1.AwsLogDriver({ streamPrefix: 'firelens' }),
                    memoryReservationMiB: 50,
                });
                break;
            }
        }
        return this.containers.map(x => x.renderContainerDefinition());
    }
}
exports.TaskDefinition = TaskDefinition;
/**
 * The port range to open up for dynamic port mapping
 */
const EPHEMERAL_PORT_RANGE = ec2.Port.tcpRange(32768, 65535);
/**
 * The networking mode to use for the containers in the task.
 */
var NetworkMode;
(function (NetworkMode) {
    /**
     * The task's containers do not have external connectivity and port mappings can't be specified in the container definition.
     */
    NetworkMode["NONE"] = "none";
    /**
     * The task utilizes Docker's built-in virtual network which runs inside each container instance.
     */
    NetworkMode["BRIDGE"] = "bridge";
    /**
     * The task is allocated an elastic network interface.
     */
    NetworkMode["AWS_VPC"] = "awsvpc";
    /**
     * The task bypasses Docker's built-in virtual network and maps container ports directly to the EC2 instance's network interface directly.
     *
     * In this mode, you can't run multiple instantiations of the same task on a
     * single container instance when port mappings are used.
     */
    NetworkMode["HOST"] = "host";
    /**
     * The task utilizes NAT network mode required by Windows containers.
     *
     * This is the only supported network mode for Windows containers. For more information, see
     * [Task Definition Parameters](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#network_mode).
     */
    NetworkMode["NAT"] = "nat";
})(NetworkMode = exports.NetworkMode || (exports.NetworkMode = {}));
/**
 * The IPC resource namespace to use for the containers in the task.
 */
var IpcMode;
(function (IpcMode) {
    /**
     * If none is specified, then IPC resources within the containers of a task are private and not
     * shared with other containers in a task or on the container instance
     */
    IpcMode["NONE"] = "none";
    /**
     * If host is specified, then all containers within the tasks that specified the host IPC mode on
     * the same container instance share the same IPC resources with the host Amazon EC2 instance.
     */
    IpcMode["HOST"] = "host";
    /**
     * If task is specified, all containers within the specified task share the same IPC resources.
     */
    IpcMode["TASK"] = "task";
})(IpcMode = exports.IpcMode || (exports.IpcMode = {}));
/**
 * The process namespace to use for the containers in the task.
 */
var PidMode;
(function (PidMode) {
    /**
     * If host is specified, then all containers within the tasks that specified the host PID mode
     * on the same container instance share the same process namespace with the host Amazon EC2 instance.
     */
    PidMode["HOST"] = "host";
    /**
     * If task is specified, all containers within the specified task share the same process namespace.
     */
    PidMode["TASK"] = "task";
})(PidMode = exports.PidMode || (exports.PidMode = {}));
/**
 * The scope for the Docker volume that determines its lifecycle.
 * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
 * Docker volumes that are scoped as shared persist after the task stops.
 */
var Scope;
(function (Scope) {
    /**
     * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
     */
    Scope["TASK"] = "task";
    /**
     * Docker volumes that are scoped as shared persist after the task stops.
     */
    Scope["SHARED"] = "shared";
})(Scope = exports.Scope || (exports.Scope = {}));
/**
 * The task launch type compatibility requirement.
 */
var Compatibility;
(function (Compatibility) {
    /**
     * The task should specify the EC2 launch type.
     */
    Compatibility[Compatibility["EC2"] = 0] = "EC2";
    /**
     * The task should specify the Fargate launch type.
     */
    Compatibility[Compatibility["FARGATE"] = 1] = "FARGATE";
    /**
     * The task can specify either the EC2 or Fargate launch types.
     */
    Compatibility[Compatibility["EC2_AND_FARGATE"] = 2] = "EC2_AND_FARGATE";
})(Compatibility = exports.Compatibility || (exports.Compatibility = {}));
/**
 * Return true if the given task definition can be run on an EC2 cluster
 */
function isEc2Compatible(compatibility) {
    return [Compatibility.EC2, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
/**
 * Return true if the given task definition can be run on a Fargate cluster
 */
function isFargateCompatible(compatibility) {
    return [Compatibility.FARGATE, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
//# sourceMappingURL=data:application/json;base64,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