"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkMultipleTargetGroupsServiceBase = void 0;
const aws_ecs_1 = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const aws_elasticloadbalancingv2_1 = require("../../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const aws_route53_1 = require("../../../aws-route53"); // Automatically re-written from '@aws-cdk/aws-route53'
const aws_route53_targets_1 = require("../../../aws-route53-targets"); // Automatically re-written from '@aws-cdk/aws-route53-targets'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * The base class for NetworkMultipleTargetGroupsEc2Service and NetworkMultipleTargetGroupsFargateService classes.
 */
class NetworkMultipleTargetGroupsServiceBase extends core_1.Construct {
    /**
     * Constructs a new instance of the NetworkMultipleTargetGroupsServiceBase class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.listeners = new Array();
        this.targetGroups = new Array();
        this.loadBalancers = new Array();
        this.validateInput(props);
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        this.desiredCount = props.desiredCount || 1;
        if (props.taskImageOptions) {
            this.logDriver = this.createLogDriver(props.taskImageOptions.enableLogging, props.taskImageOptions.logDriver);
        }
        if (props.loadBalancers) {
            for (const lbProps of props.loadBalancers) {
                const lb = this.createLoadBalancer(lbProps.name, lbProps.publicLoadBalancer);
                this.loadBalancers.push(lb);
                for (const listenerProps of lbProps.listeners) {
                    const listener = this.createListener(listenerProps.name, lb, listenerProps.port || 80);
                    this.listeners.push(listener);
                }
                this.createDomainName(lb, lbProps.domainName, lbProps.domainZone);
                new core_1.CfnOutput(this, `LoadBalancerDNS${lb.node.id}`, { value: lb.loadBalancerDnsName });
            }
            // set up default load balancer and listener.
            this.loadBalancer = this.loadBalancers[0];
            this.listener = this.listeners[0];
        }
        else {
            this.loadBalancer = this.createLoadBalancer('LB');
            this.listener = this.createListener('PublicListener', this.loadBalancer, 80);
            this.createDomainName(this.loadBalancer);
            new core_1.CfnOutput(this, 'LoadBalancerDNS', { value: this.loadBalancer.loadBalancerDnsName });
        }
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs.
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
    findListener(name) {
        if (!name) {
            return this.listener;
        }
        for (const listener of this.listeners) {
            if (listener.node.id === name) {
                return listener;
            }
        }
        throw new Error(`Listener ${name} is not defined. Did you define listener with name ${name}?`);
    }
    registerECSTargets(service, container, targets) {
        for (const targetProps of targets) {
            const targetGroup = this.findListener(targetProps.listener).addTargets(`ECSTargetGroup${container.containerName}${targetProps.containerPort}`, {
                port: 80,
                targets: [
                    service.loadBalancerTarget({
                        containerName: container.containerName,
                        containerPort: targetProps.containerPort,
                    }),
                ],
            });
            this.targetGroups.push(targetGroup);
        }
        if (this.targetGroups.length === 0) {
            throw new Error('At least one target group should be specified.');
        }
        return this.targetGroups[0];
    }
    addPortMappingForTargets(container, targets) {
        for (const target of targets) {
            if (!container.findPortMapping(target.containerPort, aws_ecs_1.Protocol.TCP)) {
                container.addPortMappings({
                    containerPort: target.containerPort,
                });
            }
        }
    }
    /**
     * Create log driver if logging is enabled.
     */
    createLogDriver(enableLoggingProp, logDriverProp) {
        const enableLogging = enableLoggingProp !== undefined ? enableLoggingProp : true;
        const logDriver = logDriverProp !== undefined
            ? logDriverProp : enableLogging
            ? this.createAWSLogDriver(this.node.id) : undefined;
        return logDriver;
    }
    validateInput(props) {
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        if (props.loadBalancers) {
            if (props.loadBalancers.length === 0) {
                throw new Error('At least one load balancer must be specified');
            }
            for (const lbProps of props.loadBalancers) {
                if (lbProps.listeners.length === 0) {
                    throw new Error('At least one listener must be specified');
                }
            }
        }
    }
    createLoadBalancer(name, publicLoadBalancer) {
        const internetFacing = publicLoadBalancer !== undefined ? publicLoadBalancer : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        return new aws_elasticloadbalancingv2_1.NetworkLoadBalancer(this, name, lbProps);
    }
    createListener(name, lb, port) {
        return lb.addListener(name, {
            port,
        });
    }
    createDomainName(loadBalancer, name, zone) {
        if (typeof name !== 'undefined') {
            if (typeof zone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            new aws_route53_1.ARecord(this, `DNS${loadBalancer.node.id}`, {
                zone,
                recordName: name,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
            });
        }
    }
}
exports.NetworkMultipleTargetGroupsServiceBase = NetworkMultipleTargetGroupsServiceBase;
//# sourceMappingURL=data:application/json;base64,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