"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = exports.RepositoryBase = void 0;
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cr = require("../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
const ecr_generated_1 = require("./ecr.generated");
const lifecycle_1 = require("./lifecycle");
/**
 * Base class for ECR repository. Reused between imported repositories and owned repositories.
 */
class RepositoryBase extends core_1.Resource {
    /**
     * The URI of this repository (represents the latest image):
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY
     *
     */
    get repositoryUri() {
        return this.repositoryUriForTag();
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[:TAG]
     *
     * @param tag Optional image tag
     */
    repositoryUriForTag(tag) {
        const tagSuffix = tag ? `:${tag}` : '';
        const parts = this.stack.parseArn(this.repositoryArn);
        return `${parts.account}.dkr.ecr.${parts.region}.${this.stack.urlSuffix}/${this.repositoryName}${tagSuffix}`;
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                requestParameters: {
                    repositoryName: [this.repositoryName],
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image is pushed to this
     * repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailImagePushed(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutImage'],
                requestParameters: {
                    imageTag: options.imageTag ? [options.imageTag] : undefined,
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image scan is completed
     *
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onImageScanCompleted(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['ECR Image Scan'],
            detail: {
                'repository-name': [this.repositoryName],
                'scan-status': ['COMPLETE'],
                'image-tags': options.imageTags ? options.imageTags : undefined,
            },
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for repository events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.ecr'],
            resources: [this.repositoryArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Grant the given principal identity permissions to perform the actions on this repository
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.repositoryArn],
            resourceSelfArns: [],
            resource: this,
        });
    }
    /**
     * Grant the given identity permissions to use the images in this repository
     */
    grantPull(grantee) {
        const ret = this.grant(grantee, 'ecr:BatchCheckLayerAvailability', 'ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage');
        iam.Grant.addToPrincipal({
            grantee,
            actions: ['ecr:GetAuthorizationToken'],
            resourceArns: ['*'],
            scope: this,
        });
        return ret;
    }
    /**
     * Grant the given identity permissions to pull and push images to this repository.
     */
    grantPullPush(grantee) {
        this.grantPull(grantee);
        return this.grant(grantee, 'ecr:PutImage', 'ecr:InitiateLayerUpload', 'ecr:UploadLayerPart', 'ecr:CompleteLayerUpload');
    }
}
exports.RepositoryBase = RepositoryBase;
/**
 * Define an ECR repository
 */
class Repository extends RepositoryBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.repositoryName,
        });
        this.lifecycleRules = new Array();
        const resource = new ecr_generated_1.CfnRepository(this, 'Resource', {
            repositoryName: this.physicalName,
            // It says "Text", but they actually mean "Object".
            repositoryPolicyText: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            lifecyclePolicy: core_1.Lazy.anyValue({ produce: () => this.renderLifecyclePolicy() }),
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.registryId = props.lifecycleRegistryId;
        if (props.lifecycleRules) {
            props.lifecycleRules.forEach(this.addLifecycleRule.bind(this));
        }
        this.repositoryName = this.getResourceNameAttribute(resource.ref);
        this.repositoryArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'ecr',
            resource: 'repository',
            resourceName: this.physicalName,
        });
        // image scanOnPush
        if (props.imageScanOnPush) {
            new cr.AwsCustomResource(this, 'ImageScanOnPush', {
                resourceType: 'Custom::ECRImageScanOnPush',
                onUpdate: {
                    service: 'ECR',
                    action: 'putImageScanningConfiguration',
                    parameters: {
                        repositoryName: this.repositoryName,
                        imageScanningConfiguration: {
                            scanOnPush: props.imageScanOnPush,
                        },
                    },
                    physicalResourceId: cr.PhysicalResourceId.of(this.repositoryArn),
                },
                onDelete: {
                    service: 'ECR',
                    action: 'putImageScanningConfiguration',
                    parameters: {
                        repositoryName: this.repositoryName,
                        imageScanningConfiguration: {
                            scanOnPush: false,
                        },
                    },
                    physicalResourceId: cr.PhysicalResourceId.of(this.repositoryArn),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({ resources: [this.repositoryArn] }),
            });
        }
    }
    /**
     * Import a repository
     */
    static fromRepositoryAttributes(scope, id, attrs) {
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = attrs.repositoryName;
                this.repositoryArn = attrs.repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    static fromRepositoryArn(scope, id, repositoryArn) {
        // if repositoryArn is a token, the repository name is also required. this is because
        // repository names can include "/" (e.g. foo/bar/myrepo) and it is impossible to
        // parse the name from an ARN using CloudFormation's split/select.
        if (core_1.Token.isUnresolved(repositoryArn)) {
            throw new Error('"repositoryArn" is a late-bound value, and therefore "repositoryName" is required. Use `fromRepositoryAttributes` instead');
        }
        const repositoryName = repositoryArn.split('/').slice(1).join('/');
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    static fromRepositoryName(scope, id, repositoryName) {
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = Repository.arnForLocalRepository(repositoryName, scope);
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    /**
     * Returns an ECR ARN for a repository that resides in the same account/region
     * as the current stack.
     */
    static arnForLocalRepository(repositoryName, scope, account) {
        return core_1.Stack.of(scope).formatArn({
            account,
            service: 'ecr',
            resource: 'repository',
            resourceName: repositoryName,
        });
    }
    addToResourcePolicy(statement) {
        if (this.policyDocument === undefined) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
        return { statementAdded: false, policyDependable: this.policyDocument };
    }
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policyDocument) === null || _a === void 0 ? void 0 : _a.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Add a life cycle rule to the repository
     *
     * Life cycle rules automatically expire images from the repository that match
     * certain conditions.
     */
    addLifecycleRule(rule) {
        // Validate rule here so users get errors at the expected location
        if (rule.tagStatus === undefined) {
            rule = { ...rule, tagStatus: rule.tagPrefixList === undefined ? lifecycle_1.TagStatus.ANY : lifecycle_1.TagStatus.TAGGED };
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.TAGGED && (rule.tagPrefixList === undefined || rule.tagPrefixList.length === 0)) {
            throw new Error('TagStatus.Tagged requires the specification of a tagPrefixList');
        }
        if (rule.tagStatus !== lifecycle_1.TagStatus.TAGGED && rule.tagPrefixList !== undefined) {
            throw new Error('tagPrefixList can only be specified when tagStatus is set to Tagged');
        }
        if ((rule.maxImageAge !== undefined) === (rule.maxImageCount !== undefined)) {
            throw new Error(`Life cycle rule must contain exactly one of 'maxImageAge' and 'maxImageCount', got: ${JSON.stringify(rule)}`);
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.ANY && this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY).length > 0) {
            throw new Error('Life cycle can only have one TagStatus.Any rule');
        }
        this.lifecycleRules.push({ ...rule });
    }
    /**
     * Render the life cycle policy object
     */
    renderLifecyclePolicy() {
        const stack = core_1.Stack.of(this);
        let lifecyclePolicyText;
        if (this.lifecycleRules.length === 0 && !this.registryId) {
            return undefined;
        }
        if (this.lifecycleRules.length > 0) {
            lifecyclePolicyText = JSON.stringify(stack.resolve({
                rules: this.orderedLifecycleRules().map(renderLifecycleRule),
            }));
        }
        return {
            lifecyclePolicyText,
            registryId: this.registryId,
        };
    }
    /**
     * Return life cycle rules with automatic ordering applied.
     *
     * Also applies validation of the 'any' rule.
     */
    orderedLifecycleRules() {
        if (this.lifecycleRules.length === 0) {
            return [];
        }
        const prioritizedRules = this.lifecycleRules.filter(r => r.rulePriority !== undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const autoPrioritizedRules = this.lifecycleRules.filter(r => r.rulePriority === undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const anyRules = this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
        if (anyRules.length > 0 && anyRules[0].rulePriority !== undefined && autoPrioritizedRules.length > 0) {
            // Supporting this is too complex for very little value. We just prohibit it.
            throw new Error("Cannot combine prioritized TagStatus.Any rule with unprioritized rules. Remove rulePriority from the 'Any' rule.");
        }
        const prios = prioritizedRules.map(r => r.rulePriority);
        let autoPrio = (prios.length > 0 ? Math.max(...prios) : 0) + 1;
        const ret = new Array();
        for (const rule of prioritizedRules.concat(autoPrioritizedRules).concat(anyRules)) {
            ret.push({
                ...rule,
                rulePriority: rule.rulePriority !== undefined ? rule.rulePriority : autoPrio++,
            });
        }
        // Do validation on the final array--might still be wrong because the user supplied all prios, but incorrectly.
        validateAnyRuleLast(ret);
        return ret;
    }
}
exports.Repository = Repository;
function validateAnyRuleLast(rules) {
    const anyRules = rules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
    if (anyRules.length === 1) {
        const maxPrio = Math.max(...rules.map(r => r.rulePriority));
        if (anyRules[0].rulePriority !== maxPrio) {
            throw new Error(`TagStatus.Any rule must have highest priority, has ${anyRules[0].rulePriority} which is smaller than ${maxPrio}`);
        }
    }
}
/**
 * Render the lifecycle rule to JSON
 */
function renderLifecycleRule(rule) {
    return {
        rulePriority: rule.rulePriority,
        description: rule.description,
        selection: {
            tagStatus: rule.tagStatus || lifecycle_1.TagStatus.ANY,
            tagPrefixList: rule.tagPrefixList,
            countType: rule.maxImageAge !== undefined ? "sinceImagePushed" /* SINCE_IMAGE_PUSHED */ : "imageCountMoreThan" /* IMAGE_COUNT_MORE_THAN */,
            countNumber: rule.maxImageAge !== undefined ? rule.maxImageAge.toDays() : rule.maxImageCount,
            countUnit: rule.maxImageAge !== undefined ? 'days' : undefined,
        },
        action: {
            type: 'expire',
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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