"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Volume = exports.EbsDeviceVolumeType = exports.synthesizeBlockDeviceMappings = exports.BlockDeviceVolume = void 0;
const crypto = require("crypto");
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const aws_kms_1 = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const ec2_generated_1 = require("./ec2.generated");
/**
 * Describes a block device mapping for an EC2 instance or Auto Scaling group.
 */
class BlockDeviceVolume {
    /**
     * @param ebsDevice EBS device info
     * @param virtualName Virtual device name
     */
    constructor(ebsDevice, virtualName) {
        this.ebsDevice = ebsDevice;
        this.virtualName = virtualName;
    }
    /**
     * Creates a new Elastic Block Storage device
     *
     * @param volumeSize The volume size, in Gibibytes (GiB)
     * @param options additional device options
     */
    static ebs(volumeSize, options = {}) {
        return new this({ ...options, volumeSize });
    }
    /**
     * Creates a new Elastic Block Storage device from an existing snapshot
     *
     * @param snapshotId The snapshot ID of the volume to use
     * @param options additional device options
     */
    static ebsFromSnapshot(snapshotId, options = {}) {
        return new this({ ...options, snapshotId });
    }
    /**
     * Creates a virtual, ephemeral device.
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index. Must be equal or greater than 0
     */
    static ephemeral(volumeIndex) {
        if (volumeIndex < 0) {
            throw new Error(`volumeIndex must be a number starting from 0, got "${volumeIndex}"`);
        }
        return new this(undefined, `ephemeral${volumeIndex}`);
    }
}
exports.BlockDeviceVolume = BlockDeviceVolume;
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== EbsDeviceVolumeType.IO1) {
                core_1.Annotations.of(construct).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName,
            ebs,
            virtualName,
            noDevice: mappingEnabled === false ? {} : undefined,
        };
    });
}
exports.synthesizeBlockDeviceMappings = synthesizeBlockDeviceMappings;
/**
 * Supported EBS volume types for blockDevices
 */
var EbsDeviceVolumeType;
(function (EbsDeviceVolumeType) {
    /**
     * Magnetic
     */
    EbsDeviceVolumeType["STANDARD"] = "standard";
    /**
     *  Provisioned IOPS SSD
     */
    EbsDeviceVolumeType["IO1"] = "io1";
    /**
     * General Purpose SSD
     */
    EbsDeviceVolumeType["GP2"] = "gp2";
    /**
     * Throughput Optimized HDD
     */
    EbsDeviceVolumeType["ST1"] = "st1";
    /**
     * Cold HDD
     */
    EbsDeviceVolumeType["SC1"] = "sc1";
    /**
     * General purpose SSD volume that balances price and performance for a wide variety of workloads.
     */
    EbsDeviceVolumeType["GENERAL_PURPOSE_SSD"] = "gp2";
    /**
     * Highest-performance SSD volume for mission-critical low-latency or high-throughput workloads.
     */
    EbsDeviceVolumeType["PROVISIONED_IOPS_SSD"] = "io1";
    /**
     * Low-cost HDD volume designed for frequently accessed, throughput-intensive workloads.
     */
    EbsDeviceVolumeType["THROUGHPUT_OPTIMIZED_HDD"] = "st1";
    /**
     * Lowest cost HDD volume designed for less frequently accessed workloads.
     */
    EbsDeviceVolumeType["COLD_HDD"] = "sc1";
    /**
     * Magnetic volumes are backed by magnetic drives and are suited for workloads where data is accessed infrequently, and scenarios where low-cost
     * storage for small volume sizes is important.
     */
    EbsDeviceVolumeType["MAGNETIC"] = "standard";
})(EbsDeviceVolumeType = exports.EbsDeviceVolumeType || (exports.EbsDeviceVolumeType = {}));
/**
 * Common behavior of Volumes. Users should not use this class directly, and instead use ``Volume``.
 */
class VolumeBase extends core_1.Resource {
    grantAttachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:AttachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        if (this.encryptionKey) {
            // When attaching a volume, the EC2 Service will need to grant to itself permission
            // to be able to decrypt the encryption key. We restrict the CreateGrant for principle
            // of least privilege, in accordance with best practices.
            // See: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-permissions
            const kmsGrant = this.encryptionKey.grant(grantee, 'kms:CreateGrant');
            kmsGrant.principalStatement.addConditions({
                Bool: { 'kms:GrantIsForAWSResource': true },
                StringEquals: {
                    'kms:ViaService': `ec2.${core_1.Stack.of(this).region}.amazonaws.com`,
                    'kms:GrantConstraintType': 'EncryptionContextSubset',
                },
            });
        }
        return result;
    }
    grantAttachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantAttach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantAttachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    grantDetachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:DetachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        // Note: No encryption key permissions are required to detach an encrypted volume.
        return result;
    }
    grantDetachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantDetach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantDetachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    collectGrantResourceArns(instances) {
        const stack = core_1.Stack.of(this);
        const resourceArns = [
            `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:volume/${this.volumeId}`,
        ];
        const instanceArnPrefix = `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:instance`;
        if (instances) {
            instances.forEach(instance => resourceArns.push(`${instanceArnPrefix}/${instance === null || instance === void 0 ? void 0 : instance.instanceId}`));
        }
        else {
            resourceArns.push(`${instanceArnPrefix}/*`);
        }
        return resourceArns;
    }
    calculateResourceTagValue(constructs) {
        const md5 = crypto.createHash('md5');
        constructs.forEach(construct => md5.update(construct.node.uniqueId));
        return md5.digest('hex');
    }
}
/**
 * Creates a new EBS Volume in AWS EC2.
 */
class Volume extends VolumeBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.volumeName,
        });
        this.validateProps(props);
        const resource = new ec2_generated_1.CfnVolume(this, 'Resource', {
            availabilityZone: props.availabilityZone,
            autoEnableIo: props.autoEnableIo,
            encrypted: props.encrypted,
            kmsKeyId: (_a = props.encryptionKey) === null || _a === void 0 ? void 0 : _a.keyArn,
            iops: props.iops,
            multiAttachEnabled: (_b = props.enableMultiAttach) !== null && _b !== void 0 ? _b : false,
            size: (_c = props.size) === null || _c === void 0 ? void 0 : _c.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }),
            snapshotId: props.snapshotId,
            volumeType: (_d = props.volumeType) !== null && _d !== void 0 ? _d : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD,
        });
        this.volumeId = resource.ref;
        this.availabilityZone = props.availabilityZone;
        this.encryptionKey = props.encryptionKey;
        if (this.encryptionKey) {
            // Per: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-requirements
            const principal = new aws_kms_1.ViaServicePrincipal(`ec2.${core_1.Stack.of(this).region}.amazonaws.com`, new aws_iam_1.AccountRootPrincipal()).withConditions({
                StringEquals: {
                    'kms:CallerAccount': core_1.Stack.of(this).account,
                },
            });
            const grant = this.encryptionKey.grant(principal, 
            // Describe & Generate are required to be able to create the CMK-encrypted Volume.
            'kms:DescribeKey', 'kms:GenerateDataKeyWithoutPlainText');
            if (props.snapshotId) {
                // ReEncrypt is required for when re-encrypting from an encrypted snapshot.
                (_e = grant.principalStatement) === null || _e === void 0 ? void 0 : _e.addActions('kms:ReEncrypt*');
            }
        }
    }
    /**
     * Import an existing EBS Volume into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Volume in the construct tree.
     * @param attrs the attributes of the imported Volume
     */
    static fromVolumeAttributes(scope, id, attrs) {
        class Import extends VolumeBase {
            constructor() {
                super(...arguments);
                this.volumeId = attrs.volumeId;
                this.availabilityZone = attrs.availabilityZone;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        // Check that the provided volumeId looks like it could be valid.
        if (!core_1.Token.isUnresolved(attrs.volumeId) && !/^vol-[0-9a-fA-F]+$/.test(attrs.volumeId)) {
            throw new Error('`volumeId` does not match expected pattern. Expected `vol-<hexadecmial value>` (ex: `vol-05abe246af`) or a Token');
        }
        return new Import(scope, id);
    }
    validateProps(props) {
        var _a;
        if (!(props.size || props.snapshotId)) {
            throw new Error('Must provide at least one of `size` or `snapshotId`');
        }
        if (props.snapshotId && !core_1.Token.isUnresolved(props.snapshotId) && !/^snap-[0-9a-fA-F]+$/.test(props.snapshotId)) {
            throw new Error('`snapshotId` does match expected pattern. Expected `snap-<hexadecmial value>` (ex: `snap-05abe246af`) or Token');
        }
        if (props.encryptionKey && !props.encrypted) {
            throw new Error('`encrypted` must be true when providing an `encryptionKey`.');
        }
        if (props.iops) {
            if (props.volumeType !== EbsDeviceVolumeType.PROVISIONED_IOPS_SSD) {
                throw new Error('`iops` may only be specified if the `volumeType` is `PROVISIONED_IOPS_SSD`/`IO1`');
            }
            if (props.iops < 100 || props.iops > 64000) {
                throw new Error('`iops` must be in the range 100 to 64,000, inclusive.');
            }
            if (props.size && (props.iops > 50 * props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }))) {
                throw new Error('`iops` has a maximum ratio of 50 IOPS/GiB.');
            }
        }
        if (props.enableMultiAttach && props.volumeType !== EbsDeviceVolumeType.PROVISIONED_IOPS_SSD) {
            throw new Error('multi-attach is supported exclusively on `PROVISIONED_IOPS_SSD` volumes.');
        }
        if (props.size) {
            const size = props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL });
            // Enforce maximum volume size:
            // https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html#ebs-volume-characteristics
            const sizeRanges = {};
            sizeRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD] = { Min: 1, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = { Min: 4, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.THROUGHPUT_OPTIMIZED_HDD] = { Min: 500, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.COLD_HDD] = { Min: 500, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.MAGNETIC] = { Min: 1, Max: 1000 };
            const volumeType = (_a = props.volumeType) !== null && _a !== void 0 ? _a : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            const { Min, Max } = sizeRanges[volumeType];
            if (size < Min || size > Max) {
                throw new Error(`\`${volumeType}\` volumes must be between ${Min} GiB and ${Max} GiB in size.`);
            }
        }
    }
}
exports.Volume = Volume;
//# sourceMappingURL=data:application/json;base64,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