"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubnetFilter = void 0;
const network_util_1 = require("./network-util");
/**
 * Contains logic which chooses a set of subnets from a larger list, in conjunction
 * with SubnetSelection, to determine where to place AWS resources such as VPC
 * endpoints, EC2 instances, etc.
 */
class SubnetFilter {
    /**
     * Chooses subnets which are in one of the given availability zones.
    */
    static availabilityZones(availabilityZones) {
        return new AvailabilityZoneSubnetFilter(availabilityZones);
    }
    /**
     * Chooses subnets such that there is at most one per availability zone.
    */
    static onePerAz() {
        return new OnePerAZSubnetFilter();
    }
    /**
     * Chooses subnets which contain any of the specified IP addresses.
    */
    static containsIpAddresses(ipv4addrs) {
        return new ContainsIpAddressesSubnetFilter(ipv4addrs);
    }
    /**
     * Executes the subnet filtering logic, returning a filtered set of subnets.
     */
    selectSubnets(_subnets) {
        throw new Error('Cannot select subnets with an abstract SubnetFilter. `selectSubnets` needs to be implmemented.');
    }
}
exports.SubnetFilter = SubnetFilter;
/**
 * Chooses subnets which are in one of the given availability zones.
 */
class AvailabilityZoneSubnetFilter extends SubnetFilter {
    constructor(availabilityZones) {
        super();
        this.availabilityZones = availabilityZones;
    }
    /**
     * Executes the subnet filtering logic.
     */
    selectSubnets(subnets) {
        return subnets.filter(s => this.availabilityZones.includes(s.availabilityZone));
    }
}
/**
 * Chooses subnets such that there is at most one per availability zone.
 */
class OnePerAZSubnetFilter extends SubnetFilter {
    constructor() {
        super();
    }
    /**
     * Executes the subnet filtering logic.
     */
    selectSubnets(subnets) {
        return this.retainOnePerAz(subnets);
    }
    retainOnePerAz(subnets) {
        const azsSeen = new Set();
        return subnets.filter(subnet => {
            if (azsSeen.has(subnet.availabilityZone)) {
                return false;
            }
            azsSeen.add(subnet.availabilityZone);
            return true;
        });
    }
}
/**
 * Chooses subnets which contain any of the specified IP addresses.
 */
class ContainsIpAddressesSubnetFilter extends SubnetFilter {
    constructor(ipAddresses) {
        super();
        this.ipAddresses = ipAddresses;
    }
    /**
     * Executes the subnet filtering logic.
     */
    selectSubnets(subnets) {
        return this.retainByIp(subnets, this.ipAddresses);
    }
    retainByIp(subnets, ips) {
        const cidrBlockObjs = ips.map(ip => {
            const ipNum = network_util_1.NetworkUtils.ipToNum(ip);
            return new network_util_1.CidrBlock(ipNum, 32);
        });
        return subnets.filter(s => {
            const subnetCidrBlock = new network_util_1.CidrBlock(s.ipv4CidrBlock);
            return cidrBlockObjs.some(cidr => subnetCidrBlock.containsCidr(cidr));
        });
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3VibmV0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic3VibmV0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLGlEQUF5RDtBQUV6RDs7OztHQUlHO0FBQ0gsTUFBc0IsWUFBWTtJQUM5Qjs7TUFFRTtJQUNLLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxpQkFBMkI7UUFDdkQsT0FBTyxJQUFJLDRCQUE0QixDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDL0QsQ0FBQztJQUNEOztNQUVFO0lBQ0ssTUFBTSxDQUFDLFFBQVE7UUFDbEIsT0FBTyxJQUFJLG9CQUFvQixFQUFFLENBQUM7SUFDdEMsQ0FBQztJQUNEOztNQUVFO0lBQ0ssTUFBTSxDQUFDLG1CQUFtQixDQUFDLFNBQW1CO1FBQ2pELE9BQU8sSUFBSSwrQkFBK0IsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUMxRCxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxhQUFhLENBQUMsUUFBbUI7UUFDcEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxnR0FBZ0csQ0FBQyxDQUFDO0lBQ3RILENBQUM7Q0FDSjtBQXpCRCxvQ0F5QkM7QUFDRDs7R0FFRztBQUNILE1BQU0sNEJBQTZCLFNBQVEsWUFBWTtJQUVuRCxZQUFZLGlCQUEyQjtRQUNuQyxLQUFLLEVBQUUsQ0FBQztRQUNSLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxpQkFBaUIsQ0FBQztJQUMvQyxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxhQUFhLENBQUMsT0FBa0I7UUFDbkMsT0FBTyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDO0lBQ3BGLENBQUM7Q0FDSjtBQUNEOztHQUVHO0FBQ0gsTUFBTSxvQkFBcUIsU0FBUSxZQUFZO0lBQzNDO1FBQ0ksS0FBSyxFQUFFLENBQUM7SUFDWixDQUFDO0lBQ0Q7O09BRUc7SUFDSSxhQUFhLENBQUMsT0FBa0I7UUFDbkMsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFDTyxjQUFjLENBQUMsT0FBa0I7UUFDckMsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLEVBQVUsQ0FBQztRQUNsQyxPQUFPLE9BQU8sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDM0IsSUFBSSxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFO2dCQUN0QyxPQUFPLEtBQUssQ0FBQzthQUNoQjtZQUNELE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLGdCQUFnQixDQUFDLENBQUM7WUFDckMsT0FBTyxJQUFJLENBQUM7UUFDaEIsQ0FBQyxDQUFDLENBQUM7SUFDUCxDQUFDO0NBQ0o7QUFDRDs7R0FFRztBQUNILE1BQU0sK0JBQWdDLFNBQVEsWUFBWTtJQUV0RCxZQUFZLFdBQXFCO1FBQzdCLEtBQUssRUFBRSxDQUFDO1FBQ1IsSUFBSSxDQUFDLFdBQVcsR0FBRyxXQUFXLENBQUM7SUFDbkMsQ0FBQztJQUNEOztPQUVHO0lBQ0ksYUFBYSxDQUFDLE9BQWtCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFDTyxVQUFVLENBQUMsT0FBa0IsRUFBRSxHQUFhO1FBQ2hELE1BQU0sYUFBYSxHQUFHLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUU7WUFDL0IsTUFBTSxLQUFLLEdBQUcsMkJBQVksQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDdkMsT0FBTyxJQUFJLHdCQUFTLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3BDLENBQUMsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ3RCLE1BQU0sZUFBZSxHQUFHLElBQUksd0JBQVMsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLENBQUM7WUFDdkQsT0FBTyxhQUFhLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBQzFFLENBQUMsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztDQUNKIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ2lkckJsb2NrLCBOZXR3b3JrVXRpbHMgfSBmcm9tICcuL25ldHdvcmstdXRpbCc7XG5pbXBvcnQgeyBJU3VibmV0IH0gZnJvbSAnLi92cGMnO1xuLyoqXG4gKiBDb250YWlucyBsb2dpYyB3aGljaCBjaG9vc2VzIGEgc2V0IG9mIHN1Ym5ldHMgZnJvbSBhIGxhcmdlciBsaXN0LCBpbiBjb25qdW5jdGlvblxuICogd2l0aCBTdWJuZXRTZWxlY3Rpb24sIHRvIGRldGVybWluZSB3aGVyZSB0byBwbGFjZSBBV1MgcmVzb3VyY2VzIHN1Y2ggYXMgVlBDXG4gKiBlbmRwb2ludHMsIEVDMiBpbnN0YW5jZXMsIGV0Yy5cbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIFN1Ym5ldEZpbHRlciB7XG4gICAgLyoqXG4gICAgICogQ2hvb3NlcyBzdWJuZXRzIHdoaWNoIGFyZSBpbiBvbmUgb2YgdGhlIGdpdmVuIGF2YWlsYWJpbGl0eSB6b25lcy5cbiAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgYXZhaWxhYmlsaXR5Wm9uZXMoYXZhaWxhYmlsaXR5Wm9uZXM6IHN0cmluZ1tdKTogU3VibmV0RmlsdGVyIHtcbiAgICAgICAgcmV0dXJuIG5ldyBBdmFpbGFiaWxpdHlab25lU3VibmV0RmlsdGVyKGF2YWlsYWJpbGl0eVpvbmVzKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ2hvb3NlcyBzdWJuZXRzIHN1Y2ggdGhhdCB0aGVyZSBpcyBhdCBtb3N0IG9uZSBwZXIgYXZhaWxhYmlsaXR5IHpvbmUuXG4gICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIG9uZVBlckF6KCk6IFN1Ym5ldEZpbHRlciB7XG4gICAgICAgIHJldHVybiBuZXcgT25lUGVyQVpTdWJuZXRGaWx0ZXIoKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ2hvb3NlcyBzdWJuZXRzIHdoaWNoIGNvbnRhaW4gYW55IG9mIHRoZSBzcGVjaWZpZWQgSVAgYWRkcmVzc2VzLlxuICAgICovXG4gICAgcHVibGljIHN0YXRpYyBjb250YWluc0lwQWRkcmVzc2VzKGlwdjRhZGRyczogc3RyaW5nW10pOiBTdWJuZXRGaWx0ZXIge1xuICAgICAgICByZXR1cm4gbmV3IENvbnRhaW5zSXBBZGRyZXNzZXNTdWJuZXRGaWx0ZXIoaXB2NGFkZHJzKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogRXhlY3V0ZXMgdGhlIHN1Ym5ldCBmaWx0ZXJpbmcgbG9naWMsIHJldHVybmluZyBhIGZpbHRlcmVkIHNldCBvZiBzdWJuZXRzLlxuICAgICAqL1xuICAgIHB1YmxpYyBzZWxlY3RTdWJuZXRzKF9zdWJuZXRzOiBJU3VibmV0W10pOiBJU3VibmV0W10ge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBzZWxlY3Qgc3VibmV0cyB3aXRoIGFuIGFic3RyYWN0IFN1Ym5ldEZpbHRlci4gYHNlbGVjdFN1Ym5ldHNgIG5lZWRzIHRvIGJlIGltcGxtZW1lbnRlZC4nKTtcbiAgICB9XG59XG4vKipcbiAqIENob29zZXMgc3VibmV0cyB3aGljaCBhcmUgaW4gb25lIG9mIHRoZSBnaXZlbiBhdmFpbGFiaWxpdHkgem9uZXMuXG4gKi9cbmNsYXNzIEF2YWlsYWJpbGl0eVpvbmVTdWJuZXRGaWx0ZXIgZXh0ZW5kcyBTdWJuZXRGaWx0ZXIge1xuICAgIHByaXZhdGUgcmVhZG9ubHkgYXZhaWxhYmlsaXR5Wm9uZXM6IHN0cmluZ1tdO1xuICAgIGNvbnN0cnVjdG9yKGF2YWlsYWJpbGl0eVpvbmVzOiBzdHJpbmdbXSkge1xuICAgICAgICBzdXBlcigpO1xuICAgICAgICB0aGlzLmF2YWlsYWJpbGl0eVpvbmVzID0gYXZhaWxhYmlsaXR5Wm9uZXM7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEV4ZWN1dGVzIHRoZSBzdWJuZXQgZmlsdGVyaW5nIGxvZ2ljLlxuICAgICAqL1xuICAgIHB1YmxpYyBzZWxlY3RTdWJuZXRzKHN1Ym5ldHM6IElTdWJuZXRbXSk6IElTdWJuZXRbXSB7XG4gICAgICAgIHJldHVybiBzdWJuZXRzLmZpbHRlcihzID0+IHRoaXMuYXZhaWxhYmlsaXR5Wm9uZXMuaW5jbHVkZXMocy5hdmFpbGFiaWxpdHlab25lKSk7XG4gICAgfVxufVxuLyoqXG4gKiBDaG9vc2VzIHN1Ym5ldHMgc3VjaCB0aGF0IHRoZXJlIGlzIGF0IG1vc3Qgb25lIHBlciBhdmFpbGFiaWxpdHkgem9uZS5cbiAqL1xuY2xhc3MgT25lUGVyQVpTdWJuZXRGaWx0ZXIgZXh0ZW5kcyBTdWJuZXRGaWx0ZXIge1xuICAgIGNvbnN0cnVjdG9yKCkge1xuICAgICAgICBzdXBlcigpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBFeGVjdXRlcyB0aGUgc3VibmV0IGZpbHRlcmluZyBsb2dpYy5cbiAgICAgKi9cbiAgICBwdWJsaWMgc2VsZWN0U3VibmV0cyhzdWJuZXRzOiBJU3VibmV0W10pOiBJU3VibmV0W10ge1xuICAgICAgICByZXR1cm4gdGhpcy5yZXRhaW5PbmVQZXJBeihzdWJuZXRzKTtcbiAgICB9XG4gICAgcHJpdmF0ZSByZXRhaW5PbmVQZXJBeihzdWJuZXRzOiBJU3VibmV0W10pOiBJU3VibmV0W10ge1xuICAgICAgICBjb25zdCBhenNTZWVuID0gbmV3IFNldDxzdHJpbmc+KCk7XG4gICAgICAgIHJldHVybiBzdWJuZXRzLmZpbHRlcihzdWJuZXQgPT4ge1xuICAgICAgICAgICAgaWYgKGF6c1NlZW4uaGFzKHN1Ym5ldC5hdmFpbGFiaWxpdHlab25lKSkge1xuICAgICAgICAgICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGF6c1NlZW4uYWRkKHN1Ym5ldC5hdmFpbGFiaWxpdHlab25lKTtcbiAgICAgICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgICB9KTtcbiAgICB9XG59XG4vKipcbiAqIENob29zZXMgc3VibmV0cyB3aGljaCBjb250YWluIGFueSBvZiB0aGUgc3BlY2lmaWVkIElQIGFkZHJlc3Nlcy5cbiAqL1xuY2xhc3MgQ29udGFpbnNJcEFkZHJlc3Nlc1N1Ym5ldEZpbHRlciBleHRlbmRzIFN1Ym5ldEZpbHRlciB7XG4gICAgcHJpdmF0ZSByZWFkb25seSBpcEFkZHJlc3Nlczogc3RyaW5nW107XG4gICAgY29uc3RydWN0b3IoaXBBZGRyZXNzZXM6IHN0cmluZ1tdKSB7XG4gICAgICAgIHN1cGVyKCk7XG4gICAgICAgIHRoaXMuaXBBZGRyZXNzZXMgPSBpcEFkZHJlc3NlcztcbiAgICB9XG4gICAgLyoqXG4gICAgICogRXhlY3V0ZXMgdGhlIHN1Ym5ldCBmaWx0ZXJpbmcgbG9naWMuXG4gICAgICovXG4gICAgcHVibGljIHNlbGVjdFN1Ym5ldHMoc3VibmV0czogSVN1Ym5ldFtdKTogSVN1Ym5ldFtdIHtcbiAgICAgICAgcmV0dXJuIHRoaXMucmV0YWluQnlJcChzdWJuZXRzLCB0aGlzLmlwQWRkcmVzc2VzKTtcbiAgICB9XG4gICAgcHJpdmF0ZSByZXRhaW5CeUlwKHN1Ym5ldHM6IElTdWJuZXRbXSwgaXBzOiBzdHJpbmdbXSk6IElTdWJuZXRbXSB7XG4gICAgICAgIGNvbnN0IGNpZHJCbG9ja09ianMgPSBpcHMubWFwKGlwID0+IHtcbiAgICAgICAgICAgIGNvbnN0IGlwTnVtID0gTmV0d29ya1V0aWxzLmlwVG9OdW0oaXApO1xuICAgICAgICAgICAgcmV0dXJuIG5ldyBDaWRyQmxvY2soaXBOdW0sIDMyKTtcbiAgICAgICAgfSk7XG4gICAgICAgIHJldHVybiBzdWJuZXRzLmZpbHRlcihzID0+IHtcbiAgICAgICAgICAgIGNvbnN0IHN1Ym5ldENpZHJCbG9jayA9IG5ldyBDaWRyQmxvY2socy5pcHY0Q2lkckJsb2NrKTtcbiAgICAgICAgICAgIHJldHVybiBjaWRyQmxvY2tPYmpzLnNvbWUoY2lkciA9PiBzdWJuZXRDaWRyQmxvY2suY29udGFpbnNDaWRyKGNpZHIpKTtcbiAgICAgICAgfSk7XG4gICAgfVxufVxuIl19