"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NatInstanceImage = exports.NatInstanceProvider = exports.NatProvider = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const connections_1 = require("./connections");
const instance_1 = require("./instance");
const machine_image_1 = require("./machine-image");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const vpc_1 = require("./vpc");
/**
 * NAT providers
 *
 * Determines what type of NAT provider to create, either NAT gateways or NAT
 * instance.
 *
 * @experimental
 */
class NatProvider {
    /**
     * Use NAT Gateways to provide NAT services for your VPC
     *
     * NAT gateways are managed by AWS.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html
     */
    static gateway() {
        return new NatGatewayProvider();
    }
    /**
     * Use NAT instances to provide NAT services for your VPC
     *
     * NAT instances are managed by you, but in return allow more configuration.
     *
     * Be aware that instances created using this provider will not be
     * automatically replaced if they are stopped for any reason. You should implement
     * your own NatProvider based on AutoScaling groups if you need that.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/VPC_NAT_Instance.html
     */
    static instance(props) {
        return new NatInstanceProvider(props);
    }
}
exports.NatProvider = NatProvider;
/**
 * Provider for NAT Gateways
 */
class NatGatewayProvider extends NatProvider {
    constructor() {
        super(...arguments);
        this.gateways = new PrefSet();
    }
    configureNat(options) {
        // Create the NAT gateways
        for (const sub of options.natSubnets) {
            const gateway = sub.addNatGateway();
            this.gateways.add(sub.availabilityZone, gateway.ref);
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            this.configureSubnet(sub);
        }
    }
    configureSubnet(subnet) {
        const az = subnet.availabilityZone;
        const gatewayId = this.gateways.pick(az);
        subnet.addRoute('DefaultRoute', {
            routerType: vpc_1.RouterType.NAT_GATEWAY,
            routerId: gatewayId,
            enablesInternetConnectivity: true,
        });
    }
    get configuredGateways() {
        return this.gateways.values().map(x => ({ az: x[0], gatewayId: x[1] }));
    }
}
/**
 * NAT provider which uses NAT Instances
 */
class NatInstanceProvider extends NatProvider {
    constructor(props) {
        super();
        this.props = props;
        this.gateways = new PrefSet();
    }
    configureNat(options) {
        var _a, _b;
        // Create the NAT instances. They can share a security group and a Role.
        const machineImage = this.props.machineImage || new NatInstanceImage();
        this._securityGroup = (_a = this.props.securityGroup) !== null && _a !== void 0 ? _a : new security_group_1.SecurityGroup(options.vpc, 'NatSecurityGroup', {
            vpc: options.vpc,
            description: 'Security Group for NAT instances',
        });
        this._connections = new connections_1.Connections({ securityGroups: [this._securityGroup] });
        if ((_b = this.props.allowAllTraffic) !== null && _b !== void 0 ? _b : true) {
            this.connections.allowFromAnyIpv4(port_1.Port.allTraffic());
        }
        // FIXME: Ideally, NAT instances don't have a role at all, but
        // 'Instance' does not allow that right now.
        const role = new iam.Role(options.vpc, 'NatRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        for (const sub of options.natSubnets) {
            const natInstance = new instance_1.Instance(sub, 'NatInstance', {
                instanceType: this.props.instanceType,
                machineImage,
                sourceDestCheck: false,
                vpc: options.vpc,
                vpcSubnets: { subnets: [sub] },
                securityGroup: this._securityGroup,
                role,
                keyName: this.props.keyName,
            });
            // NAT instance routes all traffic, both ways
            this.gateways.add(sub.availabilityZone, natInstance);
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            this.configureSubnet(sub);
        }
    }
    /**
     * The Security Group associated with the NAT instances
     */
    get securityGroup() {
        if (!this._securityGroup) {
            throw new Error('Pass the NatInstanceProvider to a Vpc before accessing \'securityGroup\'');
        }
        return this._securityGroup;
    }
    /**
     * Manage the Security Groups associated with the NAT instances
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Pass the NatInstanceProvider to a Vpc before accessing \'connections\'');
        }
        return this._connections;
    }
    get configuredGateways() {
        return this.gateways.values().map(x => ({ az: x[0], gatewayId: x[1].instanceId }));
    }
    configureSubnet(subnet) {
        const az = subnet.availabilityZone;
        const gatewayId = this.gateways.pick(az).instanceId;
        subnet.addRoute('DefaultRoute', {
            routerType: vpc_1.RouterType.INSTANCE,
            routerId: gatewayId,
            enablesInternetConnectivity: true,
        });
    }
}
exports.NatInstanceProvider = NatInstanceProvider;
/**
 * Preferential set
 *
 * Picks the value with the given key if available, otherwise distributes
 * evenly among the available options.
 */
class PrefSet {
    constructor() {
        this.map = {};
        this.vals = new Array();
        this.next = 0;
    }
    add(pref, value) {
        this.map[pref] = value;
        this.vals.push([pref, value]);
    }
    pick(pref) {
        if (this.vals.length === 0) {
            throw new Error('Cannot pick, set is empty');
        }
        if (pref in this.map) {
            return this.map[pref];
        }
        return this.vals[this.next++ % this.vals.length][1];
    }
    values() {
        return this.vals;
    }
}
/**
 * Machine image representing the latest NAT instance image
 *
 * @experimental
 */
class NatInstanceImage extends machine_image_1.LookupMachineImage {
    constructor() {
        super({
            name: 'amzn-ami-vpc-nat-*',
            owners: ['amazon'],
        });
    }
}
exports.NatInstanceImage = NatInstanceImage;
//# sourceMappingURL=data:application/json;base64,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