"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitSource = exports.InitService = exports.InitPackage = exports.InitUser = exports.InitGroup = exports.InitFile = exports.InitCommand = exports.InitCommandWaitDuration = exports.InitElement = exports.InitServiceRestartHandle = void 0;
const fs = require("fs");
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * An object that represents reasons to restart an InitService
 *
 * Pass an instance of this object to the `InitFile`, `InitCommand`,
 * `InitSource` and `InitPackage` objects, and finally to an `InitService`
 * itself to cause the actions (files, commands, sources, and packages)
 * to trigger a restart of the service.
 *
 * For example, the following will run a custom command to install Nginx,
 * and trigger the nginx service to be restarted after the command has run.
 *
 * ```ts
 * const handle = new ec2.InitServiceRestartHandle();
 * ec2.CloudFormationInit.fromElements(
 *   ec2.InitCommand.shellCommand('/usr/bin/custom-nginx-install.sh', { serviceRestartHandles: [handle] }),
 *   ec2.InitService.enable('nginx', { serviceRestartHandle: handle }),
 * );
 * ```
 */
class InitServiceRestartHandle {
    constructor() {
        this.commands = new Array();
        this.files = new Array();
        this.sources = new Array();
        this.packages = {};
    }
    /**
     * Add a command key to the restart set
     * @internal
     */
    _addCommand(key) {
        return this.commands.push(key);
    }
    /**
     * Add a file key to the restart set
     * @internal
     */
    _addFile(key) {
        return this.files.push(key);
    }
    /**
     * Add a source key to the restart set
     * @internal
     */
    _addSource(key) {
        return this.sources.push(key);
    }
    /**
     * Add a package key to the restart set
     * @internal
     */
    _addPackage(packageType, key) {
        if (!this.packages[packageType]) {
            this.packages[packageType] = [];
        }
        this.packages[packageType].push(key);
    }
    /**
     * Render the restart handles for use in an InitService declaration
     * @internal
     */
    _renderRestartHandles() {
        const nonEmpty = (x) => x.length > 0 ? x : undefined;
        return {
            commands: nonEmpty(this.commands),
            files: nonEmpty(this.files),
            packages: Object.keys(this.packages).length > 0 ? this.packages : undefined,
            sources: nonEmpty(this.sources),
        };
    }
}
exports.InitServiceRestartHandle = InitServiceRestartHandle;
/**
 * Base class for all CloudFormation Init elements
 */
class InitElement {
}
exports.InitElement = InitElement;
/**
 * Represents a duration to wait after a command has finished, in case of a reboot (Windows only).
 */
class InitCommandWaitDuration {
    /** Wait for a specified duration after a command. */
    static of(duration) {
        return new class extends InitCommandWaitDuration {
            /** @internal */
            _render() { return duration.toSeconds(); }
        }();
    }
    /** Do not wait for this command. */
    static none() {
        return InitCommandWaitDuration.of(core_1.Duration.seconds(0));
    }
    /** cfn-init will exit and resume only after a reboot. */
    static forever() {
        return new class extends InitCommandWaitDuration {
            /** @internal */
            _render() { return 'forever'; }
        }();
    }
}
exports.InitCommandWaitDuration = InitCommandWaitDuration;
/**
 * Command to execute on the instance
 */
class InitCommand extends InitElement {
    constructor(command, options) {
        super();
        this.command = command;
        this.options = options;
        this.elementType = cfn_init_internal_1.InitElementType.COMMAND.toString();
    }
    /**
     * Run a shell command
     *
     * Remember that some characters like `&`, `|`, `;`, `>` etc. have special meaning in a shell and
     * need to be preceded by a `\` if you want to treat them as part of a filename.
     */
    static shellCommand(shellCommand, options = {}) {
        return new InitCommand([shellCommand], options);
    }
    /**
     * Run a command from an argv array
     *
     * You do not need to escape space characters or enclose command parameters in quotes.
     */
    static argvCommand(argv, options = {}) {
        if (argv.length === 0) {
            throw new Error('Cannot define argvCommand with an empty arguments');
        }
        return new InitCommand(argv, options);
    }
    /** @internal */
    _bind(options) {
        var _a, _b;
        const commandKey = this.options.key || `${options.index}`.padStart(3, '0'); // 001, 005, etc.
        if (options.platform !== cfn_init_internal_1.InitPlatform.WINDOWS && this.options.waitAfterCompletion !== undefined) {
            throw new Error(`Command '${this.command}': 'waitAfterCompletion' is only valid for Windows systems.`);
        }
        for (const handle of (_a = this.options.serviceRestartHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addCommand(commandKey);
        }
        return {
            config: {
                [commandKey]: {
                    command: this.command,
                    env: this.options.env,
                    cwd: this.options.cwd,
                    test: this.options.testCmd,
                    ignoreErrors: this.options.ignoreErrors,
                    waitAfterCompletion: (_b = this.options.waitAfterCompletion) === null || _b === void 0 ? void 0 : _b._render(),
                },
            },
        };
    }
}
exports.InitCommand = InitCommand;
/**
 * Create files on the EC2 instance.
 */
class InitFile extends InitElement {
    constructor(fileName, options) {
        super();
        this.fileName = fileName;
        this.options = options;
        this.elementType = cfn_init_internal_1.InitElementType.FILE.toString();
    }
    /**
     * Use a literal string as the file content
     */
    static fromString(fileName, content, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        content,
                        encoding: this.options.base64Encoded ? 'base64' : 'plain',
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Write a symlink with the given symlink target
     */
    static symlink(fileName, target, options = {}) {
        const { mode, ...otherOptions } = options;
        if (mode && mode.slice(0, 3) !== '120') {
            throw new Error('File mode for symlinks must begin with 120XXX');
        }
        return InitFile.fromString(fileName, target, { mode: (mode || '120644'), ...otherOptions });
    }
    /**
     * Use a JSON-compatible object as the file content, write it to a JSON file.
     *
     * May contain tokens.
     */
    static fromObject(fileName, obj, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        content: obj,
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Read a file from disk and use its contents
     *
     * The file will be embedded in the template, so care should be taken to not
     * exceed the template size.
     *
     * If options.base64encoded is set to true, this will base64-encode the file's contents.
     */
    static fromFileInline(targetFileName, sourceFileName, options = {}) {
        const encoding = options.base64Encoded ? 'base64' : 'utf8';
        const fileContents = fs.readFileSync(sourceFileName).toString(encoding);
        return InitFile.fromString(targetFileName, fileContents, options);
    }
    /**
     * Download from a URL at instance startup time
     */
    static fromUrl(fileName, url, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: url,
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Download a file from an S3 bucket at instance startup time
     */
    static fromS3Object(fileName, bucket, key, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                bucket.grantRead(bindOptions.instanceRole, key);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: bucket.urlForObject(key),
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, bucket.bucketName),
                };
            }
        }(fileName, options);
    }
    /**
     * Create an asset from the given file
     *
     * This is appropriate for files that are too large to embed into the template.
     */
    static fromAsset(targetFileName, path, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                const asset = new s3_assets.Asset(bindOptions.scope, `${targetFileName}Asset`, {
                    path,
                    ...options,
                });
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: asset.httpUrl,
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetFileName, options);
    }
    /**
     * Use a file from an asset at instance startup time
     */
    static fromExistingAsset(targetFileName, asset, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: asset.httpUrl,
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetFileName, options);
    }
    /** @internal */
    _bind(bindOptions) {
        var _a;
        for (const handle of (_a = this.options.serviceRestartHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addFile(this.fileName);
        }
        return this._doBind(bindOptions);
    }
    /**
     * Render the standard config block, given content vars
     * @internal
     */
    _standardConfig(fileOptions, platform, contentVars) {
        if (platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            if (fileOptions.group || fileOptions.owner || fileOptions.mode) {
                throw new Error('Owner, group, and mode options not supported for Windows.');
            }
            return {
                [this.fileName]: { ...contentVars },
            };
        }
        return {
            [this.fileName]: {
                ...contentVars,
                mode: fileOptions.mode || '000644',
                owner: fileOptions.owner || 'root',
                group: fileOptions.group || 'root',
            },
        };
    }
}
exports.InitFile = InitFile;
/**
 * Create Linux/UNIX groups and assign group IDs.
 *
 * Not supported for Windows systems.
 */
class InitGroup extends InitElement {
    constructor(groupName, groupId) {
        super();
        this.groupName = groupName;
        this.groupId = groupId;
        this.elementType = cfn_init_internal_1.InitElementType.GROUP.toString();
    }
    /**
     * Create a group from its name, and optionally, group id
     */
    static fromName(groupName, groupId) {
        return new InitGroup(groupName, groupId);
    }
    /** @internal */
    _bind(options) {
        if (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            throw new Error('Init groups are not supported on Windows');
        }
        return {
            config: {
                [this.groupName]: this.groupId !== undefined ? { gid: this.groupId } : {},
            },
        };
    }
}
exports.InitGroup = InitGroup;
/**
 * Create Linux/UNIX users and to assign user IDs.
 *
 * Users are created as non-interactive system users with a shell of
 * /sbin/nologin. This is by design and cannot be modified.
 *
 * Not supported for Windows systems.
 */
class InitUser extends InitElement {
    constructor(userName, userOptions) {
        super();
        this.userName = userName;
        this.userOptions = userOptions;
        this.elementType = cfn_init_internal_1.InitElementType.USER.toString();
    }
    /**
     * Create a user from user name.
     */
    static fromName(userName, options = {}) {
        return new InitUser(userName, options);
    }
    /** @internal */
    _bind(options) {
        if (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            throw new Error('Init users are not supported on Windows');
        }
        return {
            config: {
                [this.userName]: {
                    uid: this.userOptions.userId,
                    groups: this.userOptions.groups,
                    homeDir: this.userOptions.homeDir,
                },
            },
        };
    }
}
exports.InitUser = InitUser;
/**
 * A package to be installed during cfn-init time
 */
class InitPackage extends InitElement {
    constructor(type, versions, packageName, serviceHandles) {
        super();
        this.type = type;
        this.versions = versions;
        this.packageName = packageName;
        this.serviceHandles = serviceHandles;
        this.elementType = cfn_init_internal_1.InitElementType.PACKAGE.toString();
    }
    /**
     * Install an RPM from an HTTP URL or a location on disk
     */
    static rpm(location, options = {}) {
        return new InitPackage('rpm', [location], options.key, options.serviceRestartHandles);
    }
    /**
     * Install a package using Yum
     */
    static yum(packageName, options = {}) {
        var _a;
        return new InitPackage('yum', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install a package from RubyGems
     */
    static rubyGem(gemName, options = {}) {
        var _a;
        return new InitPackage('rubygems', (_a = options.version) !== null && _a !== void 0 ? _a : [], gemName, options.serviceRestartHandles);
    }
    /**
     * Install a package from PyPI
     */
    static python(packageName, options = {}) {
        var _a;
        return new InitPackage('python', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install a package using APT
     */
    static apt(packageName, options = {}) {
        var _a;
        return new InitPackage('apt', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install an MSI package from an HTTP URL or a location on disk
     */
    static msi(location, options = {}) {
        // The MSI package version must be a string, not an array.
        return new class extends InitPackage {
            renderPackageVersions() { return location; }
        }('msi', [location], options.key, options.serviceRestartHandles);
    }
    /** @internal */
    _bind(options) {
        var _a;
        if ((this.type === 'msi') !== (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS)) {
            if (this.type === 'msi') {
                throw new Error('MSI installers are only supported on Windows systems.');
            }
            else {
                throw new Error('Windows only supports the MSI package type');
            }
        }
        if (!this.packageName && !['rpm', 'msi'].includes(this.type)) {
            throw new Error('Package name must be specified for all package types besides RPM and MSI.');
        }
        const packageName = this.packageName || `${options.index}`.padStart(3, '0');
        for (const handle of (_a = this.serviceHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addPackage(this.type, packageName);
        }
        return {
            config: {
                [this.type]: {
                    [packageName]: this.renderPackageVersions(),
                },
            },
        };
    }
    renderPackageVersions() {
        return this.versions;
    }
}
exports.InitPackage = InitPackage;
/**
 * A services that be enabled, disabled or restarted when the instance is launched.
 */
class InitService extends InitElement {
    constructor(serviceName, serviceOptions) {
        super();
        this.serviceName = serviceName;
        this.serviceOptions = serviceOptions;
        this.elementType = cfn_init_internal_1.InitElementType.SERVICE.toString();
    }
    /**
     * Enable and start the given service, optionally restarting it
     */
    static enable(serviceName, options = {}) {
        var _a;
        const { enabled, ensureRunning, ...otherOptions } = options;
        return new InitService(serviceName, {
            enabled: enabled !== null && enabled !== void 0 ? enabled : true,
            ensureRunning: (_a = ensureRunning !== null && ensureRunning !== void 0 ? ensureRunning : enabled) !== null && _a !== void 0 ? _a : true,
            ...otherOptions,
        });
    }
    /**
     * Disable and stop the given service
     */
    static disable(serviceName) {
        return new InitService(serviceName, { enabled: false, ensureRunning: false });
    }
    /** @internal */
    _bind(options) {
        var _a;
        const serviceManager = options.platform === cfn_init_internal_1.InitPlatform.LINUX ? 'sysvinit' : 'windows';
        return {
            config: {
                [serviceManager]: {
                    [this.serviceName]: {
                        enabled: this.serviceOptions.enabled,
                        ensureRunning: this.serviceOptions.ensureRunning,
                        ...(_a = this.serviceOptions.serviceRestartHandle) === null || _a === void 0 ? void 0 : _a._renderRestartHandles(),
                    },
                },
            },
        };
    }
}
exports.InitService = InitService;
/**
 * Extract an archive into a directory
 */
class InitSource extends InitElement {
    constructor(targetDirectory, serviceHandles) {
        super();
        this.targetDirectory = targetDirectory;
        this.serviceHandles = serviceHandles;
        this.elementType = cfn_init_internal_1.InitElementType.SOURCE.toString();
    }
    /**
     * Retrieve a URL and extract it into the given directory
     */
    static fromUrl(targetDirectory, url, options = {}) {
        return new class extends InitSource {
            _doBind() {
                return {
                    config: { [this.targetDirectory]: url },
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Extract a GitHub branch into a given directory
     */
    static fromGitHub(targetDirectory, owner, repo, refSpec, options = {}) {
        return InitSource.fromUrl(targetDirectory, `https://github.com/${owner}/${repo}/tarball/${refSpec !== null && refSpec !== void 0 ? refSpec : 'master'}`, options);
    }
    /**
     * Extract an archive stored in an S3 bucket into the given directory
     */
    static fromS3Object(targetDirectory, bucket, key, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                bucket.grantRead(bindOptions.instanceRole, key);
                return {
                    config: { [this.targetDirectory]: bucket.urlForObject(key) },
                    authentication: standardS3Auth(bindOptions.instanceRole, bucket.bucketName),
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Create an InitSource from an asset created from the given path.
     */
    static fromAsset(targetDirectory, path, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                const asset = new s3_assets.Asset(bindOptions.scope, `${targetDirectory}Asset`, {
                    path,
                    ...bindOptions,
                });
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: { [this.targetDirectory]: asset.httpUrl },
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Extract a directory from an existing directory asset.
     */
    static fromExistingAsset(targetDirectory, asset, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: { [this.targetDirectory]: asset.httpUrl },
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /** @internal */
    _bind(options) {
        var _a;
        for (const handle of (_a = this.serviceHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addSource(this.targetDirectory);
        }
        // Delegate actual bind to subclasses
        return this._doBind(options);
    }
}
exports.InitSource = InitSource;
/**
 * Render a standard S3 auth block for use in AWS::CloudFormation::Authentication
 *
 * This block is the same every time (modulo bucket name), so it has the same
 * key every time so the blocks are merged into one in the final render.
 */
function standardS3Auth(role, bucketName) {
    return {
        S3AccessCreds: {
            type: 'S3',
            roleName: role.roleName,
            buckets: [bucketName],
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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