"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildEnvironmentVariableType = exports.WindowsBuildImage = exports.WindowsImageType = exports.LinuxBuildImage = exports.ImagePullPrincipalType = exports.ComputeType = exports.Project = void 0;
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const aws_ecr_assets_1 = require("../../aws-ecr-assets"); // Automatically re-written from '@aws-cdk/aws-ecr-assets'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const build_spec_1 = require("./build-spec");
const cache_1 = require("./cache");
const codebuild_generated_1 = require("./codebuild.generated");
const codepipeline_artifacts_1 = require("./codepipeline-artifacts");
const no_artifacts_1 = require("./no-artifacts");
const no_source_1 = require("./no-source");
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const report_group_utils_1 = require("./report-group-utils");
const source_types_1 = require("./source-types");
/**
 * Represents a reference to a CodeBuild Project.
 *
 * If you're managing the Project alongside the rest of your CDK resources,
 * use the {@link Project} class.
 *
 * If you want to reference an already existing Project
 * (or one defined in a different CDK Stack),
 * use the {@link import} method.
 */
class ProjectBase extends core_1.Resource {
    /**
     * Access the Connections object.
     * Will fail if this Project does not have a VPC set.
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Only VPC-associated Projects have security groups to manage. Supply the "vpc" parameter when creating the Project');
        }
        return this._connections;
    }
    /**
     * Add a permission only if there's a policy attached.
     * @param statement The permissions statement to add
     */
    addToRolePolicy(statement) {
        if (this.role) {
            this.role.addToPolicy(statement);
        }
    }
    /**
     * Defines a CloudWatch event rule triggered when something happens with this project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codebuild'],
            detail: {
                'project-name': [this.projectName],
            },
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule triggered when the build project state
     * changes. You can filter specific build status events using an event
     * pattern filter on the `build-status` detail field:
     *
     *    const rule = project.onStateChange('OnBuildStarted', { target });
     *    rule.addEventPattern({
     *      detail: {
     *        'build-status': [
     *          "IN_PROGRESS",
     *          "SUCCEEDED",
     *          "FAILED",
     *          "STOPPED"
     *        ]
     *      }
     *    });
     *
     * You can also use the methods `onBuildFailed` and `onBuildSucceeded` to define rules for
     * these specific state changes.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build State Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule that triggers upon phase change of this
     * build project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onPhaseChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build Phase Change'],
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build starts.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildStarted(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['IN_PROGRESS'],
            },
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build fails.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildFailed(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['FAILED'],
            },
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build completes successfully.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildSucceeded(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['SUCCEEDED'],
            },
        });
        return rule;
    }
    /**
     * @returns a CloudWatch metric associated with this build project.
     * @param metricName The name of the metric
     * @param props Customization properties
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/CodeBuild',
            metricName,
            dimensions: { ProjectName: this.projectName },
            ...props,
        }).attachTo(this);
    }
    /**
     * Measures the number of builds triggered.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricBuilds(props) {
        return this.metric('Builds', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * Measures the duration of all builds over time.
     *
     * Units: Seconds
     *
     * Valid CloudWatch statistics: Average (recommended), Maximum, Minimum
     *
     * @default average over 5 minutes
     */
    metricDuration(props) {
        return this.metric('Duration', {
            statistic: 'avg',
            ...props,
        });
    }
    /**
     * Measures the number of successful builds.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricSucceededBuilds(props) {
        return this.metric('SucceededBuilds', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * Measures the number of builds that failed because of client error or
     * because of a timeout.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricFailedBuilds(props) {
        return this.metric('FailedBuilds', {
            statistic: 'sum',
            ...props,
        });
    }
}
/**
 * A representation of a CodeBuild Project.
 */
class Project extends ProjectBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.projectName,
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        this.buildImage = (props.environment && props.environment.buildImage) || LinuxBuildImage.STANDARD_1_0;
        // let source "bind" to the project. this usually involves granting permissions
        // for the code build role to interact with the source.
        this.source = props.source || new no_source_1.NoSource();
        const sourceConfig = this.source.bind(this, this);
        if (props.badge && !this.source.badgeSupported) {
            throw new Error(`Badge is not supported for source type ${this.source.type}`);
        }
        const artifacts = props.artifacts
            ? props.artifacts
            : (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE
                ? new codepipeline_artifacts_1.CodePipelineArtifacts()
                : new no_artifacts_1.NoArtifacts());
        const artifactsConfig = artifacts.bind(this, this);
        const cache = props.cache || cache_1.Cache.none();
        // give the caching strategy the option to grant permissions to any required resources
        cache._bind(this);
        // Inject download commands for asset if requested
        const environmentVariables = props.environmentVariables || {};
        const buildSpec = props.buildSpec;
        if (this.source.type === source_types_1.NO_SOURCE_TYPE && (buildSpec === undefined || !buildSpec.isImmediate)) {
            throw new Error("If the Project's source is NoSource, you need to provide a concrete buildSpec");
        }
        this._secondarySources = [];
        this._secondarySourceVersions = [];
        this._fileSystemLocations = [];
        for (const secondarySource of props.secondarySources || []) {
            this.addSecondarySource(secondarySource);
        }
        this._secondaryArtifacts = [];
        for (const secondaryArtifact of props.secondaryArtifacts || []) {
            this.addSecondaryArtifact(secondaryArtifact);
        }
        this.validateCodePipelineSettings(artifacts);
        for (const fileSystemLocation of props.fileSystemLocations || []) {
            this.addFileSystemLocation(fileSystemLocation);
        }
        const resource = new codebuild_generated_1.CfnProject(this, 'Resource', {
            description: props.description,
            source: {
                ...sourceConfig.sourceProperty,
                buildSpec: buildSpec && buildSpec.toBuildSpec(),
            },
            artifacts: artifactsConfig.artifactsProperty,
            serviceRole: this.role.roleArn,
            environment: this.renderEnvironment(props.environment, environmentVariables),
            fileSystemLocations: core_1.Lazy.anyValue({ produce: () => this.renderFileSystemLocations() }),
            // lazy, because we have a setter for it in setEncryptionKey
            encryptionKey: core_1.Lazy.stringValue({ produce: () => this._encryptionKey && this._encryptionKey.keyArn }),
            badgeEnabled: props.badge,
            cache: cache._toCloudFormation(),
            name: this.physicalName,
            timeoutInMinutes: props.timeout && props.timeout.toMinutes(),
            secondarySources: core_1.Lazy.anyValue({ produce: () => this.renderSecondarySources() }),
            secondarySourceVersions: core_1.Lazy.anyValue({ produce: () => this.renderSecondarySourceVersions() }),
            secondaryArtifacts: core_1.Lazy.anyValue({ produce: () => this.renderSecondaryArtifacts() }),
            triggers: sourceConfig.buildTriggers,
            sourceVersion: sourceConfig.sourceVersion,
            vpcConfig: this.configureVpc(props),
        });
        this.addVpcRequiredPermissions(props, resource);
        this.projectArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'codebuild',
            resource: 'project',
            resourceName: this.physicalName,
        });
        this.projectName = this.getResourceNameAttribute(resource.ref);
        this.addToRolePolicy(this.createLoggingPermission());
        // add permissions to create and use test report groups
        // with names starting with the project's name,
        // unless the customer explicitly opts out of it
        if (props.grantReportGroupPermissions !== false) {
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    'codebuild:CreateReportGroup',
                    'codebuild:CreateReport',
                    'codebuild:UpdateReport',
                    'codebuild:BatchPutTestCases',
                ],
                resources: [report_group_utils_1.renderReportGroupArn(this, `${this.projectName}-*`)],
            }));
        }
        if (props.encryptionKey) {
            this.encryptionKey = props.encryptionKey;
        }
        // bind
        const bindFunction = this.buildImage.bind;
        if (bindFunction) {
            bindFunction.call(this.buildImage, this, this, {});
        }
    }
    static fromProjectArn(scope, id, projectArn) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.projectArn = projectArn;
                this.projectName = core_1.Stack.of(scope).parseArn(projectArn).resourceName;
                this.role = undefined;
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a Project defined either outside the CDK,
     * or in a different CDK Stack
     * (and exported using the {@link export} method).
     *
     * @note if you're importing a CodeBuild Project for use
     *   in a CodePipeline, make sure the existing Project
     *   has permissions to access the S3 Bucket of that Pipeline -
     *   otherwise, builds in that Pipeline will always fail.
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param projectName the name of the project to import
     * @returns a reference to the existing Project
     */
    static fromProjectName(scope, id, projectName) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.role = undefined;
                this.projectArn = core_1.Stack.of(this).formatArn({
                    service: 'codebuild',
                    resource: 'project',
                    resourceName: projectName,
                });
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
                this.projectName = projectName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Convert the environment variables map of string to {@link BuildEnvironmentVariable},
     * which is the customer-facing type, to a list of {@link CfnProject.EnvironmentVariableProperty},
     * which is the representation of environment variables in CloudFormation.
     *
     * @param environmentVariables the map of string to environment variables
     * @returns an array of {@link CfnProject.EnvironmentVariableProperty} instances
     */
    static serializeEnvVariables(environmentVariables) {
        return Object.keys(environmentVariables).map(name => ({
            name,
            type: environmentVariables[name].type || BuildEnvironmentVariableType.PLAINTEXT,
            value: environmentVariables[name].value,
        }));
    }
    /**
     * Adds a secondary source to the Project.
     *
     * @param secondarySource the source to add as a secondary source
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondarySource(secondarySource) {
        if (!secondarySource.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary sources');
        }
        const secondarySourceConfig = secondarySource.bind(this, this);
        this._secondarySources.push(secondarySourceConfig.sourceProperty);
        if (secondarySourceConfig.sourceVersion) {
            this._secondarySourceVersions.push({
                sourceIdentifier: secondarySource.identifier,
                sourceVersion: secondarySourceConfig.sourceVersion,
            });
        }
    }
    /**
     * Adds a fileSystemLocation to the Project.
     *
     * @param fileSystemLocation the fileSystemLocation to add
     */
    addFileSystemLocation(fileSystemLocation) {
        const fileSystemConfig = fileSystemLocation.bind(this, this);
        this._fileSystemLocations.push(fileSystemConfig.location);
    }
    /**
     * Adds a secondary artifact to the Project.
     *
     * @param secondaryArtifact the artifact to add as a secondary artifact
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondaryArtifact(secondaryArtifact) {
        if (!secondaryArtifact.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary artifacts');
        }
        this._secondaryArtifacts.push(secondaryArtifact.bind(this, this).artifactsProperty);
    }
    /**
     * A callback invoked when the given project is added to a CodePipeline.
     *
     * @param _scope the construct the binding is taking place in
     * @param options additional options for the binding
     */
    bindToCodePipeline(_scope, options) {
        // work around a bug in CodeBuild: it ignores the KMS key set on the pipeline,
        // and always uses its own, project-level key
        if (options.artifactBucket.encryptionKey && !this._encryptionKey) {
            // we cannot safely do this assignment if the key is of type kms.Key,
            // and belongs to a stack in a different account or region than the project
            // (that would cause an illegal reference, as KMS keys don't have physical names)
            const keyStack = core_1.Stack.of(options.artifactBucket.encryptionKey);
            const projectStack = core_1.Stack.of(this);
            if (!(options.artifactBucket.encryptionKey instanceof kms.Key &&
                (keyStack.account !== projectStack.account || keyStack.region !== projectStack.region))) {
                this.encryptionKey = options.artifactBucket.encryptionKey;
            }
        }
    }
    /**
     * @override
     */
    validate() {
        const ret = new Array();
        if (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) {
            if (this._secondarySources.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary sources. ' +
                    "Use the CodeBuild Pipeline Actions' `extraInputs` property instead");
            }
            if (this._secondaryArtifacts.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary artifacts. ' +
                    "Use the CodeBuild Pipeline Actions' `outputs` property instead");
            }
        }
        return ret;
    }
    set encryptionKey(encryptionKey) {
        this._encryptionKey = encryptionKey;
        encryptionKey.grantEncryptDecrypt(this);
    }
    createLoggingPermission() {
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: `/aws/codebuild/${this.projectName}`,
        });
        const logGroupStarArn = `${logGroupArn}:*`;
        return new iam.PolicyStatement({
            resources: [logGroupArn, logGroupStarArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        });
    }
    renderEnvironment(env = {}, projectVars = {}) {
        var _a;
        const vars = {};
        const containerVars = env.environmentVariables || {};
        // first apply environment variables from the container definition
        for (const name of Object.keys(containerVars)) {
            vars[name] = containerVars[name];
        }
        // now apply project-level vars
        for (const name of Object.keys(projectVars)) {
            vars[name] = projectVars[name];
        }
        const hasEnvironmentVars = Object.keys(vars).length > 0;
        const errors = this.buildImage.validate(env);
        if (errors.length > 0) {
            throw new Error('Invalid CodeBuild environment: ' + errors.join('\n'));
        }
        const imagePullPrincipalType = this.buildImage.imagePullPrincipalType === ImagePullPrincipalType.CODEBUILD
            ? undefined
            : ImagePullPrincipalType.SERVICE_ROLE;
        if (this.buildImage.repository) {
            if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
                this.buildImage.repository.grantPull(this);
            }
            else {
                const statement = new iam.PolicyStatement({
                    principals: [new iam.ServicePrincipal('codebuild.amazonaws.com')],
                    actions: ['ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage', 'ecr:BatchCheckLayerAvailability'],
                });
                statement.sid = 'CodeBuild';
                this.buildImage.repository.addToResourcePolicy(statement);
            }
        }
        if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
            (_a = this.buildImage.secretsManagerCredentials) === null || _a === void 0 ? void 0 : _a.grantRead(this);
        }
        return {
            type: this.buildImage.type,
            image: this.buildImage.imageId,
            imagePullCredentialsType: imagePullPrincipalType,
            registryCredential: this.buildImage.secretsManagerCredentials
                ? {
                    credentialProvider: 'SECRETS_MANAGER',
                    credential: this.buildImage.secretsManagerCredentials.secretArn,
                }
                : undefined,
            privilegedMode: env.privileged || false,
            computeType: env.computeType || this.buildImage.defaultComputeType,
            environmentVariables: hasEnvironmentVars ? Project.serializeEnvVariables(vars) : undefined,
        };
    }
    renderFileSystemLocations() {
        return this._fileSystemLocations.length === 0
            ? undefined
            : this._fileSystemLocations;
    }
    renderSecondarySources() {
        return this._secondarySources.length === 0
            ? undefined
            : this._secondarySources;
    }
    renderSecondarySourceVersions() {
        return this._secondarySourceVersions.length === 0
            ? undefined
            : this._secondarySourceVersions;
    }
    renderSecondaryArtifacts() {
        return this._secondaryArtifacts.length === 0
            ? undefined
            : this._secondaryArtifacts;
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * codebuild creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroups || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if ((props.securityGroups && props.securityGroups.length > 0) && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic generated security group for CodeBuild ' + this.node.uniqueId,
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        return {
            vpcId: props.vpc.vpcId,
            subnets: props.vpc.selectSubnets(props.subnetSelection).subnetIds,
            securityGroupIds: this.connections.securityGroups.map(s => s.securityGroupId),
        };
    }
    addVpcRequiredPermissions(props, project) {
        if (!props.vpc || !this.role) {
            return;
        }
        this.role.addToPolicy(new iam.PolicyStatement({
            resources: [`arn:aws:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:network-interface/*`],
            actions: ['ec2:CreateNetworkInterfacePermission'],
            conditions: {
                StringEquals: {
                    'ec2:Subnet': props.vpc
                        .selectSubnets(props.subnetSelection).subnetIds
                        .map(si => `arn:aws:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:subnet/${si}`),
                    'ec2:AuthorizedService': 'codebuild.amazonaws.com',
                },
            },
        }));
        const policy = new iam.Policy(this, 'PolicyDocument', {
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        'ec2:CreateNetworkInterface',
                        'ec2:DescribeNetworkInterfaces',
                        'ec2:DeleteNetworkInterface',
                        'ec2:DescribeSubnets',
                        'ec2:DescribeSecurityGroups',
                        'ec2:DescribeDhcpOptions',
                        'ec2:DescribeVpcs',
                    ],
                }),
            ],
        });
        this.role.attachInlinePolicy(policy);
        // add an explicit dependency between the EC2 Policy and this Project -
        // otherwise, creating the Project fails, as it requires these permissions
        // to be already attached to the Project's Role
        project.node.addDependency(policy);
    }
    validateCodePipelineSettings(artifacts) {
        const sourceType = this.source.type;
        const artifactsType = artifacts.type;
        if ((sourceType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE ||
            artifactsType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) &&
            (sourceType !== artifactsType)) {
            throw new Error('Both source and artifacts must be set to CodePipeline');
        }
    }
}
exports.Project = Project;
/**
 * Build machine compute type.
 */
var ComputeType;
(function (ComputeType) {
    ComputeType["SMALL"] = "BUILD_GENERAL1_SMALL";
    ComputeType["MEDIUM"] = "BUILD_GENERAL1_MEDIUM";
    ComputeType["LARGE"] = "BUILD_GENERAL1_LARGE";
    ComputeType["X2_LARGE"] = "BUILD_GENERAL1_2XLARGE";
})(ComputeType = exports.ComputeType || (exports.ComputeType = {}));
/**
 * The type of principal CodeBuild will use to pull your build Docker image.
 */
var ImagePullPrincipalType;
(function (ImagePullPrincipalType) {
    /**
     * CODEBUILD specifies that CodeBuild uses its own identity when pulling the image.
     * This means the resource policy of the ECR repository that hosts the image will be modified to trust
     * CodeBuild's service principal.
     * This is the required principal type when using CodeBuild's pre-defined images.
     */
    ImagePullPrincipalType["CODEBUILD"] = "CODEBUILD";
    /**
     * SERVICE_ROLE specifies that AWS CodeBuild uses the project's role when pulling the image.
     * The role will be granted pull permissions on the ECR repository hosting the image.
     */
    ImagePullPrincipalType["SERVICE_ROLE"] = "SERVICE_ROLE";
})(ImagePullPrincipalType = exports.ImagePullPrincipalType || (exports.ImagePullPrincipalType = {}));
class ArmBuildImage {
    constructor(imageId) {
        this.type = 'ARM_CONTAINER';
        this.defaultComputeType = ComputeType.LARGE;
        this.imagePullPrincipalType = ImagePullPrincipalType.CODEBUILD;
        this.imageId = imageId;
    }
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType &&
            buildEnvironment.computeType !== ComputeType.LARGE) {
            ret.push(`ARM images only support ComputeType '${ComputeType.LARGE}' - ` +
                `'${buildEnvironment.computeType}' was given`);
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
/**
 * A CodeBuild image running Linux.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - LinuxBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }])
 * - LinuxBuildImage.fromEcrRepository(repo[, tag])
 * - LinuxBuildImage.fromAsset(parent, id, props)
 *
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class LinuxBuildImage {
    constructor(props) {
        this.type = 'LINUX_CONTAINER';
        this.defaultComputeType = ComputeType.SMALL;
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Linux build image from a Docker Hub image.
     */
    static fromDockerRegistry(name, options = {}) {
        return new LinuxBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
        });
    }
    /**
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tag Image tag (default "latest")
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new LinuxBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository,
        });
    }
    /**
     * Uses an Docker image asset as a Linux build image.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new LinuxBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository: asset.repository,
        });
    }
    /**
     * Uses a Docker image provided by CodeBuild.
     *
     * @returns A Docker image provided by CodeBuild.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     *
     * @param id The image identifier
     * @example 'aws/codebuild/standard:4.0'
     */
    static fromCodeBuildImageId(id) {
        return LinuxBuildImage.codeBuildImage(id);
    }
    static codeBuildImage(name) {
        return new LinuxBuildImage({
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
        });
    }
    validate(_) {
        return [];
    }
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
exports.LinuxBuildImage = LinuxBuildImage;
LinuxBuildImage.STANDARD_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:1.0');
LinuxBuildImage.STANDARD_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:2.0');
LinuxBuildImage.STANDARD_3_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:3.0');
/** The `aws/codebuild/standard:4.0` build image. */
LinuxBuildImage.STANDARD_4_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:4.0');
LinuxBuildImage.AMAZON_LINUX_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:1.0');
LinuxBuildImage.AMAZON_LINUX_2_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:2.0');
/** The Amazon Linux 2 x86_64 standard image, version `3.0`. */
LinuxBuildImage.AMAZON_LINUX_2_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:3.0');
LinuxBuildImage.AMAZON_LINUX_2_ARM = new ArmBuildImage('aws/codebuild/amazonlinux2-aarch64-standard:1.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_BASE = LinuxBuildImage.codeBuildImage('aws/codebuild/ubuntu-base:14.04');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_24_4_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:24.4.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_26_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:26.1.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOCKER_17_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:17.09.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOCKER_18_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:18.09.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_10 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.10');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.11');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_8 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-8');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_9 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-9');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-11');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_14_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.14.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.1.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_8_11_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:8.11.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_6_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:6.3.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_5_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:5.6');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_7_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.7.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_6_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.6.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_5_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.5.2');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_4_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.4.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_3_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.3.6');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_2_7_12 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:2.7.12');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.3');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.3.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_2_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.2.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.1');
/**
 * Environment type for Windows Docker images
 */
var WindowsImageType;
(function (WindowsImageType) {
    /**
     * The standard environment type, WINDOWS_CONTAINER
     */
    WindowsImageType["STANDARD"] = "WINDOWS_CONTAINER";
    /**
     * The WINDOWS_SERVER_2019_CONTAINER environment type
     */
    WindowsImageType["SERVER_2019"] = "WINDOWS_SERVER_2019_CONTAINER";
})(WindowsImageType = exports.WindowsImageType || (exports.WindowsImageType = {}));
/**
 * A CodeBuild image running Windows.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - WindowsBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }, imageType])
 * - WindowsBuildImage.fromEcrRepository(repo[, tag, imageType])
 * - WindowsBuildImage.fromAsset(parent, id, props, [, imageType])
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class WindowsBuildImage {
    constructor(props) {
        var _a;
        this.defaultComputeType = ComputeType.MEDIUM;
        this.type = ((_a = props.imageType) !== null && _a !== void 0 ? _a : WindowsImageType.STANDARD).toString();
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Windows build image from a Docker Hub image.
     */
    static fromDockerRegistry(name, options = {}, imageType = WindowsImageType.STANDARD) {
        return new WindowsBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
        });
    }
    /**
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tag Image tag (default "latest")
     */
    static fromEcrRepository(repository, tag = 'latest', imageType = WindowsImageType.STANDARD) {
        return new WindowsBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository,
        });
    }
    /**
     * Uses an Docker image asset as a Windows build image.
     */
    static fromAsset(scope, id, props, imageType = WindowsImageType.STANDARD) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new WindowsBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository: asset.repository,
        });
    }
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType === ComputeType.SMALL) {
            ret.push('Windows images do not support the Small ComputeType');
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return build_spec_1.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    // Would love to do downloading here and executing in the next step,
                    // but I don't know how to propagate the value of $TEMPDIR.
                    //
                    // Punting for someone who knows PowerShell well enough.
                    commands: [],
                },
                build: {
                    commands: [
                        'Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName',
                        `aws s3 cp s3://$env:${run_script_linux_build_spec_1.S3_BUCKET_ENV}/$env:${run_script_linux_build_spec_1.S3_KEY_ENV} $TEMPDIR\\scripts.zip`,
                        'New-Item -ItemType Directory -Path $TEMPDIR\\scriptdir',
                        'Expand-Archive -Path $TEMPDIR/scripts.zip -DestinationPath $TEMPDIR\\scriptdir',
                        '$env:SCRIPT_DIR = "$TEMPDIR\\scriptdir"',
                        `& $TEMPDIR\\scriptdir\\${entrypoint}`,
                    ],
                },
            },
        });
    }
}
exports.WindowsBuildImage = WindowsBuildImage;
/**
 * Corresponds to the standard CodeBuild image `aws/codebuild/windows-base:1.0`.
 *
 * @deprecated `WindowsBuildImage.WINDOWS_BASE_2_0` should be used instead.
 */
WindowsBuildImage.WIN_SERVER_CORE_2016_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * The standard CodeBuild image `aws/codebuild/windows-base:2.0`, which is
 * based off Windows Server Core 2016.
 */
WindowsBuildImage.WINDOWS_BASE_2_0 = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * The standard CodeBuild image `aws/codebuild/windows-base:2019-1.0`, which is
 * based off Windows Server Core 2019.
 */
WindowsBuildImage.WIN_SERVER_CORE_2019_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2019-1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
    imageType: WindowsImageType.SERVER_2019,
});
var BuildEnvironmentVariableType;
(function (BuildEnvironmentVariableType) {
    /**
     * An environment variable in plaintext format.
     */
    BuildEnvironmentVariableType["PLAINTEXT"] = "PLAINTEXT";
    /**
     * An environment variable stored in Systems Manager Parameter Store.
     */
    BuildEnvironmentVariableType["PARAMETER_STORE"] = "PARAMETER_STORE";
    /**
     * An environment variable stored in AWS Secrets Manager.
     */
    BuildEnvironmentVariableType["SECRETS_MANAGER"] = "SECRETS_MANAGER";
})(BuildEnvironmentVariableType = exports.BuildEnvironmentVariableType || (exports.BuildEnvironmentVariableType = {}));
//# sourceMappingURL=data:application/json;base64,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