"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginBase = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 *
 * @experimental
 */
class OriginBase {
    constructor(domainName, props = {}) {
        var _a;
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, (_a = props.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        this.domainName = domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     */
    bind(_scope, options) {
        var _a;
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return {
            originProperty: {
                domainName: this.domainName,
                id: options.originId,
                originPath: this.originPath,
                connectionAttempts: this.connectionAttempts,
                connectionTimeout: (_a = this.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                originCustomHeaders: this.renderCustomHeaders(),
                s3OriginConfig,
                customOriginConfig,
            },
        };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.substr(0, path.length - 1);
        }
        return path;
    }
}
exports.OriginBase = OriginBase;
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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