"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SlackChannelConfiguration = exports.LoggingLevel = void 0;
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const chatbot_generated_1 = require("./chatbot.generated");
/**
 * Logging levels include ERROR, INFO, or NONE.
 */
var LoggingLevel;
(function (LoggingLevel) {
    /**
     * ERROR
     */
    LoggingLevel["ERROR"] = "ERROR";
    /**
     * INFO
     */
    LoggingLevel["INFO"] = "INFO";
    /**
     * NONE
     */
    LoggingLevel["NONE"] = "NONE";
})(LoggingLevel = exports.LoggingLevel || (exports.LoggingLevel = {}));
/**
 * Either a new or imported Slack channel configuration
 */
class SlackChannelConfigurationBase extends cdk.Resource {
    /**
     * Adds extra permission to iam-role of Slack channel configuration
     * @param statement
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Return the given named metric for this SlackChannelConfiguration
     */
    metric(metricName, props) {
        // AWS Chatbot publishes metrics to us-east-1 regardless of stack region
        // https://docs.aws.amazon.com/chatbot/latest/adminguide/monitoring-cloudwatch.html
        return new cloudwatch.Metric({
            namespace: 'AWS/Chatbot',
            region: 'us-east-1',
            dimensions: {
                ConfigurationName: this.slackChannelConfigurationName,
            },
            metricName,
            ...props,
        });
    }
}
/**
 * A new Slack channel configuration
 */
class SlackChannelConfiguration extends SlackChannelConfigurationBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.slackChannelConfigurationName,
        });
        this.role = props.role || new iam.Role(this, 'ConfigurationRole', {
            assumedBy: new iam.ServicePrincipal('chatbot.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const configuration = new chatbot_generated_1.CfnSlackChannelConfiguration(this, 'Resource', {
            configurationName: props.slackChannelConfigurationName,
            iamRoleArn: this.role.roleArn,
            slackWorkspaceId: props.slackWorkspaceId,
            slackChannelId: props.slackChannelId,
            snsTopicArns: (_a = props.notificationTopics) === null || _a === void 0 ? void 0 : _a.map(topic => topic.topicArn),
            loggingLevel: (_b = props.loggingLevel) === null || _b === void 0 ? void 0 : _b.toString(),
        });
        // Log retention
        // AWS Chatbot publishes logs to us-east-1 regardless of stack region https://docs.aws.amazon.com/chatbot/latest/adminguide/cloudwatch-logs.html
        if (props.logRetention) {
            new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/chatbot/${props.slackChannelConfigurationName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logGroupRegion: 'us-east-1',
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
        }
        this.slackChannelConfigurationArn = configuration.ref;
        this.slackChannelConfigurationName = props.slackChannelConfigurationName;
    }
    /**
     * Import an existing Slack channel configuration provided an ARN
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param slackChannelConfigurationArn configuration ARN (i.e. arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack)
     *
     * @returns a reference to the existing Slack channel configuration
     */
    static fromSlackChannelConfigurationArn(scope, id, slackChannelConfigurationArn) {
        const re = /^slack-channel\//;
        const resourceName = cdk.Stack.of(scope).parseArn(slackChannelConfigurationArn).resourceName;
        if (!re.test(resourceName)) {
            throw new Error('The ARN of a Slack integration must be in the form: arn:aws:chatbot:{region}:{account}:chat-configuration/slack-channel/{slackChannelName}');
        }
        class Import extends SlackChannelConfigurationBase {
            constructor(s, i) {
                super(s, i);
                /**
                 * @attribute
                 */
                this.slackChannelConfigurationArn = slackChannelConfigurationArn;
                this.role = undefined;
                /**
                 * Returns a name of Slack channel configuration
                 *
                 * NOTE:
                 * For example: arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack
                 * The ArnComponents API will return `slack-channel/my-slack`
                 * It need to handle that to gets a correct name.`my-slack`
                 */
                this.slackChannelConfigurationName = resourceName.substring('slack-channel/'.length);
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for All SlackChannelConfigurations
     */
    static metricAll(metricName, props) {
        // AWS Chatbot publishes metrics to us-east-1 regardless of stack region
        // https://docs.aws.amazon.com/chatbot/latest/adminguide/monitoring-cloudwatch.html
        return new cloudwatch.Metric({
            namespace: 'AWS/Chatbot',
            region: 'us-east-1',
            metricName,
            ...props,
        });
    }
}
exports.SlackChannelConfiguration = SlackChannelConfiguration;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2xhY2stY2hhbm5lbC1jb25maWd1cmF0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic2xhY2stY2hhbm5lbC1jb25maWd1cmF0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLG1EQUFtRCxDQUFDLDBEQUEwRDtBQUM5RyxxQ0FBcUMsQ0FBQyxtREFBbUQ7QUFDekYsdUNBQXVDLENBQUMsb0RBQW9EO0FBRTVGLGtDQUFrQyxDQUFDLGdEQUFnRDtBQUNuRiwyREFBbUU7QUFtRW5FOztHQUVHO0FBQ0gsSUFBWSxZQWFYO0FBYkQsV0FBWSxZQUFZO0lBQ3BCOztPQUVHO0lBQ0gsK0JBQWUsQ0FBQTtJQUNmOztPQUVHO0lBQ0gsNkJBQWEsQ0FBQTtJQUNiOztPQUVHO0lBQ0gsNkJBQWEsQ0FBQTtBQUNqQixDQUFDLEVBYlcsWUFBWSxHQUFaLG9CQUFZLEtBQVosb0JBQVksUUFhdkI7QUFnQ0Q7O0dBRUc7QUFDSCxNQUFlLDZCQUE4QixTQUFRLEdBQUcsQ0FBQyxRQUFRO0lBSzdEOzs7T0FHRztJQUNJLGVBQWUsQ0FBQyxTQUE4QjtRQUNqRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRTtZQUNaLE9BQU87U0FDVjtRQUNELElBQUksQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUNEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFVBQWtCLEVBQUUsS0FBZ0M7UUFDOUQsd0VBQXdFO1FBQ3hFLG1GQUFtRjtRQUNuRixPQUFPLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUN6QixTQUFTLEVBQUUsYUFBYTtZQUN4QixNQUFNLEVBQUUsV0FBVztZQUNuQixVQUFVLEVBQUU7Z0JBQ1IsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLDZCQUE2QjthQUN4RDtZQUNELFVBQVU7WUFDVixHQUFHLEtBQUs7U0FDWCxDQUFDLENBQUM7SUFDUCxDQUFDO0NBQ0o7QUFDRDs7R0FFRztBQUNILE1BQWEseUJBQTBCLFNBQVEsNkJBQTZCO0lBdUR4RSxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQXFDOztRQUMvRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNiLFlBQVksRUFBRSxLQUFLLENBQUMsNkJBQTZCO1NBQ3BELENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksSUFBSSxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO1lBQzlELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyx1QkFBdUIsQ0FBQztTQUMvRCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUM7UUFDaEMsTUFBTSxhQUFhLEdBQUcsSUFBSSxnREFBNEIsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3JFLGlCQUFpQixFQUFFLEtBQUssQ0FBQyw2QkFBNkI7WUFDdEQsVUFBVSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTztZQUM3QixnQkFBZ0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCO1lBQ3hDLGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYztZQUNwQyxZQUFZLFFBQUUsS0FBSyxDQUFDLGtCQUFrQiwwQ0FBRSxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDO1lBQ3BFLFlBQVksUUFBRSxLQUFLLENBQUMsWUFBWSwwQ0FBRSxRQUFRLEVBQUU7U0FDL0MsQ0FBQyxDQUFDO1FBQ0gsZ0JBQWdCO1FBQ2hCLGdKQUFnSjtRQUNoSixJQUFJLEtBQUssQ0FBQyxZQUFZLEVBQUU7WUFDcEIsSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7Z0JBQ3hDLFlBQVksRUFBRSxnQkFBZ0IsS0FBSyxDQUFDLDZCQUE2QixFQUFFO2dCQUNuRSxTQUFTLEVBQUUsS0FBSyxDQUFDLFlBQVk7Z0JBQzdCLElBQUksRUFBRSxLQUFLLENBQUMsZ0JBQWdCO2dCQUM1QixjQUFjLEVBQUUsV0FBVztnQkFDM0Isd0JBQXdCLEVBQUUsS0FBSyxDQUFDLHdCQUF3QjthQUMzRCxDQUFDLENBQUM7U0FDTjtRQUNELElBQUksQ0FBQyw0QkFBNEIsR0FBRyxhQUFhLENBQUMsR0FBRyxDQUFDO1FBQ3RELElBQUksQ0FBQyw2QkFBNkIsR0FBRyxLQUFLLENBQUMsNkJBQTZCLENBQUM7SUFDN0UsQ0FBQztJQW5GRDs7Ozs7OztPQU9HO0lBQ0ksTUFBTSxDQUFDLGdDQUFnQyxDQUFDLEtBQW9CLEVBQUUsRUFBVSxFQUFFLDRCQUFvQztRQUNqSCxNQUFNLEVBQUUsR0FBRyxrQkFBa0IsQ0FBQztRQUM5QixNQUFNLFlBQVksR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsNEJBQTRCLENBQUMsQ0FBQyxZQUFzQixDQUFDO1FBQ3ZHLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFO1lBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMsNElBQTRJLENBQUMsQ0FBQztTQUNqSztRQUNELE1BQU0sTUFBTyxTQUFRLDZCQUE2QjtZQWdCOUMsWUFBWSxDQUFnQixFQUFFLENBQVM7Z0JBQ25DLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7Z0JBaEJoQjs7bUJBRUc7Z0JBQ00saUNBQTRCLEdBQUcsNEJBQTRCLENBQUM7Z0JBQzVELFNBQUksR0FBZSxTQUFTLENBQUM7Z0JBRXRDOzs7Ozs7O21CQU9HO2dCQUNNLGtDQUE2QixHQUFHLFlBQVksQ0FBQyxTQUFTLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLENBQUM7Z0JBR3JGLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztZQUN2RSxDQUFDO1NBQ0o7UUFDRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsU0FBUyxDQUFDLFVBQWtCLEVBQUUsS0FBZ0M7UUFDeEUsd0VBQXdFO1FBQ3hFLG1GQUFtRjtRQUNuRixPQUFPLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUN6QixTQUFTLEVBQUUsYUFBYTtZQUN4QixNQUFNLEVBQUUsV0FBVztZQUNuQixVQUFVO1lBQ1YsR0FBRyxLQUFLO1NBQ1gsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztDQW1DSjtBQXJGRCw4REFxRkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjbG91ZHdhdGNoIGZyb20gXCIuLi8uLi9hd3MtY2xvdWR3YXRjaFwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3Vkd2F0Y2gnXG5pbXBvcnQgKiBhcyBpYW0gZnJvbSBcIi4uLy4uL2F3cy1pYW1cIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nXG5pbXBvcnQgKiBhcyBsb2dzIGZyb20gXCIuLi8uLi9hd3MtbG9nc1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWxvZ3MnXG5pbXBvcnQgKiBhcyBzbnMgZnJvbSBcIi4uLy4uL2F3cy1zbnNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zbnMnXG5pbXBvcnQgKiBhcyBjZGsgZnJvbSBcIi4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBDZm5TbGFja0NoYW5uZWxDb25maWd1cmF0aW9uIH0gZnJvbSAnLi9jaGF0Ym90LmdlbmVyYXRlZCc7XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgbmV3IFNsYWNrIGNoYW5uZWwgY29uZmlndXJhdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb25Qcm9wcyB7XG4gICAgLyoqXG4gICAgICogVGhlIG5hbWUgb2YgU2xhY2sgY2hhbm5lbCBjb25maWd1cmF0aW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2xhY2tDaGFubmVsQ29uZmlndXJhdGlvbk5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgcGVybWlzc2lvbiByb2xlIG9mIFNsYWNrIGNoYW5uZWwgY29uZmlndXJhdGlvblxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBBIHJvbGUgd2lsbCBiZSBjcmVhdGVkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG4gICAgLyoqXG4gICAgICogVGhlIElEIG9mIHRoZSBTbGFjayB3b3Jrc3BhY2UgYXV0aG9yaXplZCB3aXRoIEFXUyBDaGF0Ym90LlxuICAgICAqXG4gICAgICogVG8gZ2V0IHRoZSB3b3Jrc3BhY2UgSUQsIHlvdSBtdXN0IHBlcmZvcm0gdGhlIGluaXRpYWwgYXV0aG9yaXphdGlvbiBmbG93IHdpdGggU2xhY2sgaW4gdGhlIEFXUyBDaGF0Ym90IGNvbnNvbGUuXG4gICAgICogVGhlbiB5b3UgY2FuIGNvcHkgYW5kIHBhc3RlIHRoZSB3b3Jrc3BhY2UgSUQgZnJvbSB0aGUgY29uc29sZS5cbiAgICAgKiBGb3IgbW9yZSBkZXRhaWxzLCBzZWUgc3RlcHMgMS00IGluIFNldHRpbmcgVXAgQVdTIENoYXRib3Qgd2l0aCBTbGFjayBpbiB0aGUgQVdTIENoYXRib3QgVXNlciBHdWlkZS5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jaGF0Ym90L2xhdGVzdC9hZG1pbmd1aWRlL3NldHRpbmctdXAuaHRtbCNTZXR1cF9pbnRyb1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHNsYWNrV29ya3NwYWNlSWQ6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgSUQgb2YgdGhlIFNsYWNrIGNoYW5uZWwuXG4gICAgICpcbiAgICAgKiBUbyBnZXQgdGhlIElELCBvcGVuIFNsYWNrLCByaWdodCBjbGljayBvbiB0aGUgY2hhbm5lbCBuYW1lIGluIHRoZSBsZWZ0IHBhbmUsIHRoZW4gY2hvb3NlIENvcHkgTGluay5cbiAgICAgKiBUaGUgY2hhbm5lbCBJRCBpcyB0aGUgOS1jaGFyYWN0ZXIgc3RyaW5nIGF0IHRoZSBlbmQgb2YgdGhlIFVSTC4gRm9yIGV4YW1wbGUsIEFCQ0JCTFpaWi5cbiAgICAgKi9cbiAgICByZWFkb25seSBzbGFja0NoYW5uZWxJZDogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBTTlMgdG9waWNzIHRoYXQgZGVsaXZlciBub3RpZmljYXRpb25zIHRvIEFXUyBDaGF0Ym90LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm9uZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IG5vdGlmaWNhdGlvblRvcGljcz86IHNucy5JVG9waWNbXTtcbiAgICAvKipcbiAgICAgKiBTcGVjaWZpZXMgdGhlIGxvZ2dpbmcgbGV2ZWwgZm9yIHRoaXMgY29uZmlndXJhdGlvbi5cbiAgICAgKiBUaGlzIHByb3BlcnR5IGFmZmVjdHMgdGhlIGxvZyBlbnRyaWVzIHB1c2hlZCB0byBBbWF6b24gQ2xvdWRXYXRjaCBMb2dzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTG9nZ2luZ0xldmVsLk5PTkVcbiAgICAgKi9cbiAgICByZWFkb25seSBsb2dnaW5nTGV2ZWw/OiBMb2dnaW5nTGV2ZWw7XG4gICAgLyoqXG4gICAgICogVGhlIG51bWJlciBvZiBkYXlzIGxvZyBldmVudHMgYXJlIGtlcHQgaW4gQ2xvdWRXYXRjaCBMb2dzLiBXaGVuIHVwZGF0aW5nXG4gICAgICogdGhpcyBwcm9wZXJ0eSwgdW5zZXR0aW5nIGl0IGRvZXNuJ3QgcmVtb3ZlIHRoZSBsb2cgcmV0ZW50aW9uIHBvbGljeS4gVG9cbiAgICAgKiByZW1vdmUgdGhlIHJldGVudGlvbiBwb2xpY3ksIHNldCB0aGUgdmFsdWUgdG8gYElORklOSVRFYC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGxvZ3MuUmV0ZW50aW9uRGF5cy5JTkZJTklURVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGxvZ1JldGVudGlvbj86IGxvZ3MuUmV0ZW50aW9uRGF5cztcbiAgICAvKipcbiAgICAgKiBUaGUgSUFNIHJvbGUgZm9yIHRoZSBMYW1iZGEgZnVuY3Rpb24gYXNzb2NpYXRlZCB3aXRoIHRoZSBjdXN0b20gcmVzb3VyY2VcbiAgICAgKiB0aGF0IHNldHMgdGhlIHJldGVudGlvbiBwb2xpY3kuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEEgbmV3IHJvbGUgaXMgY3JlYXRlZC5cbiAgICAgKi9cbiAgICByZWFkb25seSBsb2dSZXRlbnRpb25Sb2xlPzogaWFtLklSb2xlO1xuICAgIC8qKlxuICAgICAqIFdoZW4gbG9nIHJldGVudGlvbiBpcyBzcGVjaWZpZWQsIGEgY3VzdG9tIHJlc291cmNlIGF0dGVtcHRzIHRvIGNyZWF0ZSB0aGUgQ2xvdWRXYXRjaCBsb2cgZ3JvdXAuXG4gICAgICogVGhlc2Ugb3B0aW9ucyBjb250cm9sIHRoZSByZXRyeSBwb2xpY3kgd2hlbiBpbnRlcmFjdGluZyB3aXRoIENsb3VkV2F0Y2ggQVBJcy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gRGVmYXVsdCBBV1MgU0RLIHJldHJ5IG9wdGlvbnMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgbG9nUmV0ZW50aW9uUmV0cnlPcHRpb25zPzogbG9ncy5Mb2dSZXRlbnRpb25SZXRyeU9wdGlvbnM7XG59XG4vKipcbiAqIExvZ2dpbmcgbGV2ZWxzIGluY2x1ZGUgRVJST1IsIElORk8sIG9yIE5PTkUuXG4gKi9cbmV4cG9ydCBlbnVtIExvZ2dpbmdMZXZlbCB7XG4gICAgLyoqXG4gICAgICogRVJST1JcbiAgICAgKi9cbiAgICBFUlJPUiA9ICdFUlJPUicsXG4gICAgLyoqXG4gICAgICogSU5GT1xuICAgICAqL1xuICAgIElORk8gPSAnSU5GTycsXG4gICAgLyoqXG4gICAgICogTk9ORVxuICAgICAqL1xuICAgIE5PTkUgPSAnTk9ORSdcbn1cbi8qKlxuICogUmVwcmVzZW50cyBhIFNsYWNrIGNoYW5uZWwgY29uZmlndXJhdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIElTbGFja0NoYW5uZWxDb25maWd1cmF0aW9uIGV4dGVuZHMgY2RrLklSZXNvdXJjZSwgaWFtLklHcmFudGFibGUge1xuICAgIC8qKlxuICAgICAqIFRoZSBBUk4gb2YgdGhlIFNsYWNrIGNoYW5uZWwgY29uZmlndXJhdGlvblxuICAgICAqIEluIHRoZSBmb3JtIG9mIGFybjphd3M6Y2hhdGJvdDp7cmVnaW9ufTp7YWNjb3VudH06Y2hhdC1jb25maWd1cmF0aW9uL3NsYWNrLWNoYW5uZWwve3NsYWNrQ2hhbm5lbE5hbWV9XG4gICAgICogQGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb25Bcm46IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiBTbGFjayBjaGFubmVsIGNvbmZpZ3VyYXRpb25cbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2xhY2tDaGFubmVsQ29uZmlndXJhdGlvbk5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgcGVybWlzc2lvbiByb2xlIG9mIFNsYWNrIGNoYW5uZWwgY29uZmlndXJhdGlvblxuICAgICAqIEBhdHRyaWJ1dGVcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gQSByb2xlIHdpbGwgYmUgY3JlYXRlZC5cbiAgICAgKi9cbiAgICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuICAgIC8qKlxuICAgICAqIEFkZHMgYSBzdGF0ZW1lbnQgdG8gdGhlIElBTSByb2xlLlxuICAgICAqL1xuICAgIGFkZFRvUm9sZVBvbGljeShzdGF0ZW1lbnQ6IGlhbS5Qb2xpY3lTdGF0ZW1lbnQpOiB2b2lkO1xuICAgIC8qKlxuICAgICAqIFJldHVybiB0aGUgZ2l2ZW4gbmFtZWQgbWV0cmljIGZvciB0aGlzIFNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb25cbiAgICAgKi9cbiAgICBtZXRyaWMobWV0cmljTmFtZTogc3RyaW5nLCBwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljO1xufVxuLyoqXG4gKiBFaXRoZXIgYSBuZXcgb3IgaW1wb3J0ZWQgU2xhY2sgY2hhbm5lbCBjb25maWd1cmF0aW9uXG4gKi9cbmFic3RyYWN0IGNsYXNzIFNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb25CYXNlIGV4dGVuZHMgY2RrLlJlc291cmNlIGltcGxlbWVudHMgSVNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb24ge1xuICAgIGFic3RyYWN0IHJlYWRvbmx5IHNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb25Bcm46IHN0cmluZztcbiAgICBhYnN0cmFjdCByZWFkb25seSBzbGFja0NoYW5uZWxDb25maWd1cmF0aW9uTmFtZTogc3RyaW5nO1xuICAgIGFic3RyYWN0IHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBpYW0uSVByaW5jaXBhbDtcbiAgICBhYnN0cmFjdCByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuICAgIC8qKlxuICAgICAqIEFkZHMgZXh0cmEgcGVybWlzc2lvbiB0byBpYW0tcm9sZSBvZiBTbGFjayBjaGFubmVsIGNvbmZpZ3VyYXRpb25cbiAgICAgKiBAcGFyYW0gc3RhdGVtZW50XG4gICAgICovXG4gICAgcHVibGljIGFkZFRvUm9sZVBvbGljeShzdGF0ZW1lbnQ6IGlhbS5Qb2xpY3lTdGF0ZW1lbnQpOiB2b2lkIHtcbiAgICAgICAgaWYgKCF0aGlzLnJvbGUpIHtcbiAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLnJvbGUuYWRkVG9QcmluY2lwYWxQb2xpY3koc3RhdGVtZW50KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJuIHRoZSBnaXZlbiBuYW1lZCBtZXRyaWMgZm9yIHRoaXMgU2xhY2tDaGFubmVsQ29uZmlndXJhdGlvblxuICAgICAqL1xuICAgIHB1YmxpYyBtZXRyaWMobWV0cmljTmFtZTogc3RyaW5nLCBwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICAgICAgLy8gQVdTIENoYXRib3QgcHVibGlzaGVzIG1ldHJpY3MgdG8gdXMtZWFzdC0xIHJlZ2FyZGxlc3Mgb2Ygc3RhY2sgcmVnaW9uXG4gICAgICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jaGF0Ym90L2xhdGVzdC9hZG1pbmd1aWRlL21vbml0b3JpbmctY2xvdWR3YXRjaC5odG1sXG4gICAgICAgIHJldHVybiBuZXcgY2xvdWR3YXRjaC5NZXRyaWMoe1xuICAgICAgICAgICAgbmFtZXNwYWNlOiAnQVdTL0NoYXRib3QnLFxuICAgICAgICAgICAgcmVnaW9uOiAndXMtZWFzdC0xJyxcbiAgICAgICAgICAgIGRpbWVuc2lvbnM6IHtcbiAgICAgICAgICAgICAgICBDb25maWd1cmF0aW9uTmFtZTogdGhpcy5zbGFja0NoYW5uZWxDb25maWd1cmF0aW9uTmFtZSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBtZXRyaWNOYW1lLFxuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgIH0pO1xuICAgIH1cbn1cbi8qKlxuICogQSBuZXcgU2xhY2sgY2hhbm5lbCBjb25maWd1cmF0aW9uXG4gKi9cbmV4cG9ydCBjbGFzcyBTbGFja0NoYW5uZWxDb25maWd1cmF0aW9uIGV4dGVuZHMgU2xhY2tDaGFubmVsQ29uZmlndXJhdGlvbkJhc2Uge1xuICAgIC8qKlxuICAgICAqIEltcG9ydCBhbiBleGlzdGluZyBTbGFjayBjaGFubmVsIGNvbmZpZ3VyYXRpb24gcHJvdmlkZWQgYW4gQVJOXG4gICAgICogQHBhcmFtIHNjb3BlIFRoZSBwYXJlbnQgY3JlYXRpbmcgY29uc3RydWN0XG4gICAgICogQHBhcmFtIGlkIFRoZSBjb25zdHJ1Y3QncyBuYW1lXG4gICAgICogQHBhcmFtIHNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb25Bcm4gY29uZmlndXJhdGlvbiBBUk4gKGkuZS4gYXJuOmF3czpjaGF0Ym90OjoxMjM0NTY3ODkwOmNoYXQtY29uZmlndXJhdGlvbi9zbGFjay1jaGFubmVsL215LXNsYWNrKVxuICAgICAqXG4gICAgICogQHJldHVybnMgYSByZWZlcmVuY2UgdG8gdGhlIGV4aXN0aW5nIFNsYWNrIGNoYW5uZWwgY29uZmlndXJhdGlvblxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbVNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb25Bcm4oc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb25Bcm46IHN0cmluZyk6IElTbGFja0NoYW5uZWxDb25maWd1cmF0aW9uIHtcbiAgICAgICAgY29uc3QgcmUgPSAvXnNsYWNrLWNoYW5uZWxcXC8vO1xuICAgICAgICBjb25zdCByZXNvdXJjZU5hbWUgPSBjZGsuU3RhY2sub2Yoc2NvcGUpLnBhcnNlQXJuKHNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb25Bcm4pLnJlc291cmNlTmFtZSBhcyBzdHJpbmc7XG4gICAgICAgIGlmICghcmUudGVzdChyZXNvdXJjZU5hbWUpKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1RoZSBBUk4gb2YgYSBTbGFjayBpbnRlZ3JhdGlvbiBtdXN0IGJlIGluIHRoZSBmb3JtOiBhcm46YXdzOmNoYXRib3Q6e3JlZ2lvbn06e2FjY291bnR9OmNoYXQtY29uZmlndXJhdGlvbi9zbGFjay1jaGFubmVsL3tzbGFja0NoYW5uZWxOYW1lfScpO1xuICAgICAgICB9XG4gICAgICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb25CYXNlIHtcbiAgICAgICAgICAgIC8qKlxuICAgICAgICAgICAgICogQGF0dHJpYnV0ZVxuICAgICAgICAgICAgICovXG4gICAgICAgICAgICByZWFkb25seSBzbGFja0NoYW5uZWxDb25maWd1cmF0aW9uQXJuID0gc2xhY2tDaGFubmVsQ29uZmlndXJhdGlvbkFybjtcbiAgICAgICAgICAgIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGUgPSB1bmRlZmluZWQ7XG4gICAgICAgICAgICByZWFkb25seSBncmFudFByaW5jaXBhbDogaWFtLklQcmluY2lwYWw7XG4gICAgICAgICAgICAvKipcbiAgICAgICAgICAgICAqIFJldHVybnMgYSBuYW1lIG9mIFNsYWNrIGNoYW5uZWwgY29uZmlndXJhdGlvblxuICAgICAgICAgICAgICpcbiAgICAgICAgICAgICAqIE5PVEU6XG4gICAgICAgICAgICAgKiBGb3IgZXhhbXBsZTogYXJuOmF3czpjaGF0Ym90OjoxMjM0NTY3ODkwOmNoYXQtY29uZmlndXJhdGlvbi9zbGFjay1jaGFubmVsL215LXNsYWNrXG4gICAgICAgICAgICAgKiBUaGUgQXJuQ29tcG9uZW50cyBBUEkgd2lsbCByZXR1cm4gYHNsYWNrLWNoYW5uZWwvbXktc2xhY2tgXG4gICAgICAgICAgICAgKiBJdCBuZWVkIHRvIGhhbmRsZSB0aGF0IHRvIGdldHMgYSBjb3JyZWN0IG5hbWUuYG15LXNsYWNrYFxuICAgICAgICAgICAgICovXG4gICAgICAgICAgICByZWFkb25seSBzbGFja0NoYW5uZWxDb25maWd1cmF0aW9uTmFtZSA9IHJlc291cmNlTmFtZS5zdWJzdHJpbmcoJ3NsYWNrLWNoYW5uZWwvJy5sZW5ndGgpO1xuICAgICAgICAgICAgY29uc3RydWN0b3IoczogY2RrLkNvbnN0cnVjdCwgaTogc3RyaW5nKSB7XG4gICAgICAgICAgICAgICAgc3VwZXIocywgaSk7XG4gICAgICAgICAgICAgICAgdGhpcy5ncmFudFByaW5jaXBhbCA9IG5ldyBpYW0uVW5rbm93blByaW5jaXBhbCh7IHJlc291cmNlOiB0aGlzIH0pO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJldHVybiB0aGUgZ2l2ZW4gbmFtZWQgbWV0cmljIGZvciBBbGwgU2xhY2tDaGFubmVsQ29uZmlndXJhdGlvbnNcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIG1ldHJpY0FsbChtZXRyaWNOYW1lOiBzdHJpbmcsIHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgICAgICAvLyBBV1MgQ2hhdGJvdCBwdWJsaXNoZXMgbWV0cmljcyB0byB1cy1lYXN0LTEgcmVnYXJkbGVzcyBvZiBzdGFjayByZWdpb25cbiAgICAgICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NoYXRib3QvbGF0ZXN0L2FkbWluZ3VpZGUvbW9uaXRvcmluZy1jbG91ZHdhdGNoLmh0bWxcbiAgICAgICAgcmV0dXJuIG5ldyBjbG91ZHdhdGNoLk1ldHJpYyh7XG4gICAgICAgICAgICBuYW1lc3BhY2U6ICdBV1MvQ2hhdGJvdCcsXG4gICAgICAgICAgICByZWdpb246ICd1cy1lYXN0LTEnLFxuICAgICAgICAgICAgbWV0cmljTmFtZSxcbiAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgcmVhZG9ubHkgc2xhY2tDaGFubmVsQ29uZmlndXJhdGlvbkFybjogc3RyaW5nO1xuICAgIHJlYWRvbmx5IHNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb25OYW1lOiBzdHJpbmc7XG4gICAgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcbiAgICByZWFkb25seSBncmFudFByaW5jaXBhbDogaWFtLklQcmluY2lwYWw7XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBTbGFja0NoYW5uZWxDb25maWd1cmF0aW9uUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICAgICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLnNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb25OYW1lLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5yb2xlID0gcHJvcHMucm9sZSB8fCBuZXcgaWFtLlJvbGUodGhpcywgJ0NvbmZpZ3VyYXRpb25Sb2xlJywge1xuICAgICAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2NoYXRib3QuYW1hem9uYXdzLmNvbScpLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5ncmFudFByaW5jaXBhbCA9IHRoaXMucm9sZTtcbiAgICAgICAgY29uc3QgY29uZmlndXJhdGlvbiA9IG5ldyBDZm5TbGFja0NoYW5uZWxDb25maWd1cmF0aW9uKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgICAgICAgIGNvbmZpZ3VyYXRpb25OYW1lOiBwcm9wcy5zbGFja0NoYW5uZWxDb25maWd1cmF0aW9uTmFtZSxcbiAgICAgICAgICAgIGlhbVJvbGVBcm46IHRoaXMucm9sZS5yb2xlQXJuLFxuICAgICAgICAgICAgc2xhY2tXb3Jrc3BhY2VJZDogcHJvcHMuc2xhY2tXb3Jrc3BhY2VJZCxcbiAgICAgICAgICAgIHNsYWNrQ2hhbm5lbElkOiBwcm9wcy5zbGFja0NoYW5uZWxJZCxcbiAgICAgICAgICAgIHNuc1RvcGljQXJuczogcHJvcHMubm90aWZpY2F0aW9uVG9waWNzPy5tYXAodG9waWMgPT4gdG9waWMudG9waWNBcm4pLFxuICAgICAgICAgICAgbG9nZ2luZ0xldmVsOiBwcm9wcy5sb2dnaW5nTGV2ZWw/LnRvU3RyaW5nKCksXG4gICAgICAgIH0pO1xuICAgICAgICAvLyBMb2cgcmV0ZW50aW9uXG4gICAgICAgIC8vIEFXUyBDaGF0Ym90IHB1Ymxpc2hlcyBsb2dzIHRvIHVzLWVhc3QtMSByZWdhcmRsZXNzIG9mIHN0YWNrIHJlZ2lvbiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2hhdGJvdC9sYXRlc3QvYWRtaW5ndWlkZS9jbG91ZHdhdGNoLWxvZ3MuaHRtbFxuICAgICAgICBpZiAocHJvcHMubG9nUmV0ZW50aW9uKSB7XG4gICAgICAgICAgICBuZXcgbG9ncy5Mb2dSZXRlbnRpb24odGhpcywgJ0xvZ1JldGVudGlvbicsIHtcbiAgICAgICAgICAgICAgICBsb2dHcm91cE5hbWU6IGAvYXdzL2NoYXRib3QvJHtwcm9wcy5zbGFja0NoYW5uZWxDb25maWd1cmF0aW9uTmFtZX1gLFxuICAgICAgICAgICAgICAgIHJldGVudGlvbjogcHJvcHMubG9nUmV0ZW50aW9uLFxuICAgICAgICAgICAgICAgIHJvbGU6IHByb3BzLmxvZ1JldGVudGlvblJvbGUsXG4gICAgICAgICAgICAgICAgbG9nR3JvdXBSZWdpb246ICd1cy1lYXN0LTEnLFxuICAgICAgICAgICAgICAgIGxvZ1JldGVudGlvblJldHJ5T3B0aW9uczogcHJvcHMubG9nUmV0ZW50aW9uUmV0cnlPcHRpb25zLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5zbGFja0NoYW5uZWxDb25maWd1cmF0aW9uQXJuID0gY29uZmlndXJhdGlvbi5yZWY7XG4gICAgICAgIHRoaXMuc2xhY2tDaGFubmVsQ29uZmlndXJhdGlvbk5hbWUgPSBwcm9wcy5zbGFja0NoYW5uZWxDb25maWd1cmF0aW9uTmFtZTtcbiAgICB9XG59XG4iXX0=