import * as ec2 from "../../aws-ec2";
import * as iam from "../../aws-iam";
import { Construct, IResource, Resource } from "../../core";
/**
 * Property to specify if the compute environment
 * uses On-Demand or SpotFleet compute resources.
 */
export declare enum ComputeResourceType {
    /**
     * Resources will be EC2 On-Demand resources.
     */
    ON_DEMAND = "EC2",
    /**
     * Resources will be EC2 SpotFleet resources.
     */
    SPOT = "SPOT"
}
/**
 * Properties for how to prepare compute resources
 * that are provisioned for a compute environment.
 */
export declare enum AllocationStrategy {
    /**
     * Batch will use the best fitting instance type will be used
     * when assigning a batch job in this compute environment.
     */
    BEST_FIT = "BEST_FIT",
    /**
     * Batch will select additional instance types that are large enough to
     * meet the requirements of the jobs in the queue, with a preference for
     * instance types with a lower cost per unit vCPU.
     */
    BEST_FIT_PROGRESSIVE = "BEST_FIT_PROGRESSIVE",
    /**
     * This is only available for Spot Instance compute resources and will select
     * additional instance types that are large enough to meet the requirements of
     * the jobs in the queue, with a preference for instance types that are less
     * likely to be interrupted.
     */
    SPOT_CAPACITY_OPTIMIZED = "SPOT_CAPACITY_OPTIMIZED"
}
/**
 * Launch template property specification
 */
export interface LaunchTemplateSpecification {
    /**
     * The Launch template name
     */
    readonly launchTemplateName: string;
    /**
     * The launch template version to be used (optional).
     *
     * @default - the default version of the launch template
     */
    readonly version?: string;
}
/**
 * Properties for defining the structure of the batch compute cluster.
 */
export interface ComputeResources {
    /**
     * The allocation strategy to use for the compute resource in case not enough instances of the best
     * fitting instance type can be allocated. This could be due to availability of the instance type in
     * the region or Amazon EC2 service limits. If this is not specified, the default for the EC2
     * ComputeResourceType is BEST_FIT, which will use only the best fitting instance type, waiting for
     * additional capacity if it's not available. This allocation strategy keeps costs lower but can limit
     * scaling. If you are using Spot Fleets with BEST_FIT then the Spot Fleet IAM Role must be specified.
     * BEST_FIT_PROGRESSIVE will select an additional instance type that is large enough to meet the
     * requirements of the jobs in the queue, with a preference for an instance type with a lower cost.
     * The default value for the SPOT instance type is SPOT_CAPACITY_OPTIMIZED, which is only available for
     * for this type of compute resources and will select an additional instance type that is large enough
     * to meet the requirements of the jobs in the queue, with a preference for an instance type that is
     * less likely to be interrupted.
     *
     * @default AllocationStrategy.BEST_FIT
     */
    readonly allocationStrategy?: AllocationStrategy;
    /**
     * The Amazon ECS instance profile applied to Amazon EC2 instances in a compute environment. You can specify
     * the short name or full Amazon Resource Name (ARN) of an instance profile. For example, ecsInstanceRole or
     * arn:aws:iam::<aws_account_id>:instance-profile/ecsInstanceRole . For more information, see Amazon ECS
     * Instance Role in the AWS Batch User Guide.
     *
     * @default - a new role will be created.
     * @link https://docs.aws.amazon.com/batch/latest/userguide/instance_IAM_role.html
     */
    readonly instanceRole?: string;
    /**
     * An optional launch template to associate with your compute resources.
     * For more information, see README file.
     *
     * @default - no custom launch template will be used
     * @link https://docs.aws.amazon.com/batch/latest/userguide/launch-templates.html
     */
    readonly launchTemplate?: LaunchTemplateSpecification;
    /**
     * The types of EC2 instances that may be launched in the compute environment. You can specify instance
     * families to launch any instance type within those families (for example, c4 or p3), or you can specify
     * specific sizes within a family (such as c4.8xlarge). You can also choose optimal to pick instance types
     * (from the C, M, and R instance families) on the fly that match the demand of your job queues.
     *
     * @default optimal
     */
    readonly instanceTypes?: ec2.InstanceType[];
    /**
     * The EC2 security group(s) associated with instances launched in the compute environment.
     *
     * @default - AWS default security group.
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The VPC network that all compute resources will be connected to.
     */
    readonly vpc: ec2.IVpc;
    /**
     * The VPC subnets into which the compute resources are launched.
     *
     * @default - private subnets of the supplied VPC.
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * The type of compute environment: ON_DEMAND or SPOT.
     *
     * @default ON_DEMAND
     */
    readonly type?: ComputeResourceType;
    /**
     * This property will be ignored if you set the environment type to ON_DEMAND.
     *
     * The maximum percentage that a Spot Instance price can be when compared with the On-Demand price for
     * that instance type before instances are launched. For example, if your maximum percentage is 20%,
     * then the Spot price must be below 20% of the current On-Demand price for that EC2 instance. You always
     * pay the lowest (market) price and never more than your maximum percentage. If you leave this field empty,
     * the default value is 100% of the On-Demand price.
     *
     * @default 100
     */
    readonly bidPercentage?: number;
    /**
     * The desired number of EC2 vCPUS in the compute environment.
     *
     * @default - no desired vcpu value will be used.
     */
    readonly desiredvCpus?: number;
    /**
     * The maximum number of EC2 vCPUs that an environment can reach. Each vCPU is equivalent to
     * 1,024 CPU shares. You must specify at least one vCPU.
     *
     * @default 256
     */
    readonly maxvCpus?: number;
    /**
     * The minimum number of EC2 vCPUs that an environment should maintain (even if the compute environment state is DISABLED).
     * Each vCPU is equivalent to 1,024 CPU shares. By keeping this set to 0 you will not have instance time wasted when
     * there is no work to be run. If you set this above zero you will maintain that number of vCPUs at all times.
     *
     * @default 0
     */
    readonly minvCpus?: number;
    /**
     * The EC2 key pair that is used for instances launched in the compute environment.
     * If no key is defined, then SSH access is not allowed to provisioned compute resources.
     *
     * @default - no SSH access will be possible.
     */
    readonly ec2KeyPair?: string;
    /**
     * The Amazon Machine Image (AMI) ID used for instances launched in the compute environment.
     *
     * @default - no image will be used.
     */
    readonly image?: ec2.IMachineImage;
    /**
     * This property will be ignored if you set the environment type to ON_DEMAND.
     *
     * The Amazon Resource Name (ARN) of the Amazon EC2 Spot Fleet IAM role applied to a SPOT compute environment.
     * For more information, see Amazon EC2 Spot Fleet Role in the AWS Batch User Guide.
     *
     * @link https://docs.aws.amazon.com/batch/latest/userguide/spot_fleet_IAM_role.html
     * @default - no fleet role will be used.
     */
    readonly spotFleetRole?: iam.IRole;
    /**
     * Key-value pair tags to be applied to resources that are launched in the compute environment.
     * For AWS Batch, these take the form of "String1": "String2", where String1 is the tag key and
     * String2 is the tag value—for example, { "Name": "AWS Batch Instance - C4OnDemand" }.
     *
     * @default - no tags will be assigned on compute resources.
     */
    readonly computeResourcesTags?: {
        [key: string]: string;
    };
}
/**
 * Properties for creating a new Compute Environment
 */
export interface ComputeEnvironmentProps {
    /**
     * A name for the compute environment.
     *
     * Up to 128 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed.
     *
     * @default - CloudFormation-generated name
     */
    readonly computeEnvironmentName?: string;
    /**
     * The details of the required compute resources for the managed compute environment.
     *
     * If specified, and this is an unmanaged compute environment, will throw an error.
     *
     * By default, AWS Batch managed compute environments use a recent, approved version of the
     * Amazon ECS-optimized AMI for compute resources.
     *
     * @default - CloudFormation defaults
     * @link https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html
     */
    readonly computeResources?: ComputeResources;
    /**
     * The state of the compute environment. If the state is set to true, then the compute
     * environment accepts jobs from a queue and can scale out automatically based on queues.
     *
     * @default true
     */
    readonly enabled?: boolean;
    /**
     * The IAM role used by Batch to make calls to other AWS services on your behalf for managing
     * the resources that you use with the service. By default, this role is created for you using
     * the AWS managed service policy for Batch.
     *
     * @link https://docs.aws.amazon.com/batch/latest/userguide/service_IAM_role.html
     *
     * @default - Role using the 'service-role/AWSBatchServiceRole' policy.
     */
    readonly serviceRole?: iam.IRole;
    /**
     * Determines if AWS should manage the allocation of compute resources for processing jobs.
     * If set to false, then you are in charge of providing the compute resource details.
     *
     * @default true
     */
    readonly managed?: boolean;
}
/**
 * Properties of a compute environment.
 */
export interface IComputeEnvironment extends IResource {
    /**
     * The ARN of this compute environment.
     *
     * @attribute
     */
    readonly computeEnvironmentArn: string;
    /**
     * The name of this compute environment.
     *
     * @attribute
     */
    readonly computeEnvironmentName: string;
}
/**
 * Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 */
export declare class ComputeEnvironment extends Resource implements IComputeEnvironment {
    /**
     * Fetches an existing batch compute environment by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param computeEnvironmentArn
     */
    static fromComputeEnvironmentArn(scope: Construct, id: string, computeEnvironmentArn: string): IComputeEnvironment;
    /**
     * The ARN of this compute environment.
     *
     * @attribute
     */
    readonly computeEnvironmentArn: string;
    /**
     * The name of this compute environment.
     *
     * @attribute
     */
    readonly computeEnvironmentName: string;
    constructor(scope: Construct, id: string, props?: ComputeEnvironmentProps);
    private isManaged;
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    private validateProps;
    private buildInstanceTypes;
    private buildSecurityGroupIds;
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    private getSpotFleetRole;
}
