!  Nutation module utility, modified versions of SOFA subroutines.
!
!************************************************************************
      SUBROUTINE XYS2KA (DATE1, DATE2, CENT, DCENT, FA2K, FAD2K, &
     &                   X, Y, S )
!
!  - - - - - - - - -
!   X Y S 2 0 0 0 A
!  - - - - - - - - -
!
!  Implementation of Capitaine et al. (2002) series for X, Y and s+XY/2,
!  consistent with the IAU 2000A precession-nutation model.
!
!  Annexe to IERS Conventions 2000, Chapter 5
!
!  Given:
!     DATE1,DATE2   d      TT date (JD = DATE1+DATE2)
!
!  Returned:
!     X(2)          d      X-coordinate of CIP unit vector in GCRS
!                           and its time derivative.
!     Y(2)          d      Y-coordinate of CIP unit vector in GCRS
!                           and its time derivative.
!     S(2)          d      the quantity S, positioning the CEO
!                           and its time derivative.
!
!  Calls SOFA routine iau_ANPM.
!
!  This revision:  2002 December 17
!
!*************************************************************************
!   Modified version of XYS2000A for use in program Calc for VLBI analysis
!   and/or correlation. Name changed to XYS22KA. Input/outputs modified.
!   Time in centuries and funadmental luni-solar and planetary arguments
!   passed in. Time derivatives added. D. Gordon,
!   NASA/GSFC, Aug. 2004.
!
!-----------------------------------------------------------------------
!
      IMPLICIT NONE
!
!     REAL*8           iau_ANPM
      REAL*8           DATE1, DATE2, CENT, DCENT, X(2), Y(2), S(2)
!  2Pi
      REAL*8           D2PI
      PARAMETER ( D2PI = 6.283185307179586476925287D0 )
!  Arcseconds to radians
      REAL*8           DAS2R
      PARAMETER ( DAS2R = 4.848136811095359935899141D-6 )
!  Reference epoch (J2000), JD
      REAL*8           DJ0
      PARAMETER ( DJ0 = 2451545D0 )
!  Days per Julian century
      REAL*8           DJC
      PARAMETER ( DJC = 36525D0 )
!  Time since J2000, in Julian centuries
      REAL*8           T
!  Miscellaneous
      INTEGER I, J
      REAL*8           A, S0, S1, S2, S3, S4, S5
!
!  Fundamental arguments
      REAL*8           FA(14), FAD(14), FA2K(14), FAD2K(14)
      REAL*8           dA, dS0, dS1, dS2, dS3, dS4, dS5, dTdt
!
!  ------------------------------
!  The series for X: declarations
!  ------------------------------
!
!  Number of terms in the series
      INTEGER NXP, NX0, NX1, NX2, NX3, NX4
      PARAMETER ( NXP=6, NX0=1306, NX1=253, NX2=36, NX3=4, NX4=1 )
!
!  Polynomial coefficients
      REAL*8           XP ( NXP )
!
!  Coefficients of l,l',F,D,Om,LMe,LVe,LE,LMa,LJu,LSa,LU,LN,pA
      INTEGER KX0 ( 14, NX0 ), &
     &        KX1 ( 14, NX1 ), &
     &        KX2 ( 14, NX2 ), &
     &        KX3 ( 14, NX3 ), &
     &        KX4 ( 14, NX4 )
!
!  Sine and cosine coefficients
      REAL*8           SX0 ( 2, NX0 ), &
     &                 SX1 ( 2, NX1 ), &
     &                 SX2 ( 2, NX2 ), &
     &                 SX3 ( 2, NX3 ), &
     &                 SX4 ( 2, NX4 )
!
!  ------------------------------
!  The series for Y: declarations
!  ------------------------------
!
!  Number of terms in the series
      INTEGER NYP, NY0, NY1, NY2, NY3, NY4
      PARAMETER ( NYP=6, NY0= 962, NY1=277, NY2=30, NY3=5, NY4=1 )
!
!  Polynomial coefficients
      REAL*8           YP ( NYP )
!
!  Coefficients of l,l',F,D,Om,LMe,LVe,LE,LMa,LJu,LSa,LU,LN,pA
      INTEGER KY0 ( 14, NY0 ), &
     &        KY1 ( 14, NY1 ), &
     &        KY2 ( 14, NY2 ), &
     &        KY3 ( 14, NY3 ), &
     &        KY4 ( 14, NY4 )
!
!  Sine and cosine coefficients
      REAL*8           SY0 ( 2, NY0 ), &
     &                 SY1 ( 2, NY1 ), &
     &                 SY2 ( 2, NY2 ), &
     &                 SY3 ( 2, NY3 ), &
     &                 SY4 ( 2, NY4 )
!
!  -----------------------------------
!  The series for s+XY/2: declarations
!  -----------------------------------
!
!  Number of terms in the series
      INTEGER NSP, NS0, NS1, NS2, NS3, NS4
      PARAMETER ( NSP=6, NS0=  33, NS1=  3, NS2=25, NS3=4, NS4=1 )
!
!  Polynomial coefficients
      REAL*8           SP ( NSP )
!
!  Coefficients of l,l',F,D,Om
      INTEGER KS0 ( 14, NS0 ), &
     &        KS1 ( 14, NS1 ), &
     &        KS2 ( 14, NS2 ), &
     &        KS3 ( 14, NS3 ), &
     &        KS4 ( 14, NS4 )
!
!  Sine and cosine coefficients
      REAL*8           SS0 ( 2, NS0 ), &
     &                 SS1 ( 2, NS1 ), &
     &                 SS2 ( 2, NS2 ), &
     &                 SS3 ( 2, NS3 ), &
     &                 SS4 ( 2, NS4 )
!
!  ----------------------------------
!  The series for X: numerical values
!  ----------------------------------
!
!  Polynomial coefficients
      DATA XP /     -16616.99D-6, &
     &          2004191742.88D-6, &
     &             -427219.05D-6, &
     &             -198620.54D-6, &
     &                 -46.05D-6, &
     &                   5.98D-6 /
!
!  Argument coefficients for t^0 terms
      DATA ( ( KX0(I,J), I=1,14), J =    1,   10 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   11,   20 ) / &
     &  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   21,   30 ) / &
     &  0,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   31,   40 ) / &
     &  1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0, &
     &  1,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   41,   50 ) / &
     &  0,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   51,   60 ) / &
     &  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0, -1, &
     &  0,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   61,   70 ) / &
     &  1, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   71,   80 ) / &
     &  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0, -2, &
     &  1,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -1,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   81,   90 ) / &
     &  1,  0,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &  2,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =   91,  100 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  2, &
     &  1,  0, -4,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  101,  110 ) / &
     &  0,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -1,  2,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  111,  120 ) / &
     &  0,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  121,  130 ) / &
     &  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  131,  140 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  2, -8,  3,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  6, -8,  3,  0,  0,  0,  2, &
     &  0,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2, &
     &  1, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  2, -5,  0,  0,  0, &
     &  3,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  141,  150 ) / &
     &  2, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  2, -2,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0, -2,  0,  2,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  151,  160 ) / &
     &  0,  3,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &  1, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  1,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  161,  170 ) / &
     &  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  2, &
     &  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0,  0, -2,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0,  0,  0, -2,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  171,  180 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  2, &
     &  1,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,-18, 16,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  181,  190 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1, &
     &  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,-10,  3,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  191,  200 ) / &
     &  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, -2, &
     &  1,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  2, &
     &  4,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -1,  0,  0,  0, &
     &  1,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  201,  210 ) / &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  0,  0, -5,  6,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  1, &
     &  2,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -2, &
     &  0,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  211,  220 ) / &
     &  2,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0, -2, &
     &  0,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -4,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -2,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &  0,  0,  2, -3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  4,  0, -2,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J =  221,  230 ) / &
     &  0,  0,  4,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  2, &
     &  1,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0,  0,  2, &
     &  2,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-11,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  231,  240 ) / &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  2, &
     &  0,  0,  1, -1,  1,  0, -5,  7,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1, &
     &  1,  0,  0, -2,  0,  0, 19,-21,  3,  0,  0,  0,  0,  0, &
     &  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J =  241,  250 ) / &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  2,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0, -2, &
     &  0,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  2, &
     &  3,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  251,  260 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  2, &
     &  1,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  2, &
     &  2,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0, -1, &
     &  1,  0,  0, -1,  0,  0, -3,  4,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  261,  270 ) / &
     &  0,  2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  2, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  271,  280 ) / &
     &  4,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0,  3, -8,  3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0, -2, &
     &  2,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  1,  0,  8,-13,  0,  0,  0,  0,  0,  0, &
     &  2,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  281,  290 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, -1, &
     &  1, -1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0, -1,  0,-18, 16,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  291,  300 ) / &
     &  2,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -5,  8, -3,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0,  0, -1,  0, -1,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0, -8, 13,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0, -6,  8,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  301,  310 ) / &
     &  1, -1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0, -2,  5,  0,  0,  0, &
     &  1,  0, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  311,  320 ) / &
     &  0,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  2, -5,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  2, &
     &  2,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  6,-16,  4,  5,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  321,  330 ) / &
     &  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  1,  0,-18, 16,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  331,  340 ) / &
     &  2,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -7,  0,  0,  0,  0,  0, -2, &
     &  2,  0, -4,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  1,  0,  0,  0, &
     &  3,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  341,  350 ) / &
     &  2,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -1,  0,  0,  0, &
     &  0,  2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  2, &
     &  1, -1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, -1 /
      DATA ( ( KX0(I,J), I=1,14), J =  351,  360 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  1, -2,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  2, &
     &  1,  1, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0, -1 /
      DATA ( ( KX0(I,J), I=1,14), J =  361,  370 ) / &
     &  0,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0,  0, &
     &  3, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0,  0, -1,  0,  0, -1,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  371,  380 ) / &
     &  0,  2,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1, &
     &  2,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  381,  390 ) / &
     &  1,  0,  0, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1, &
     &  0,  0,  1, -1,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0, &
     &  3,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  0,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J =  391,  400 ) / &
     &  0,  0,  0,  0,  0,  0,  8,-10,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0, -1, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &  1, -2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0, -2, &
     &  0,  0,  1, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  401,  410 ) / &
     &  3,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  2, &
     &  2,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0, &
     &  2, -1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  411,  420 ) / &
     &  2,  0, -4, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0, -2, &
     &  1,  1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  4,  0, -3,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J =  421,  430 ) / &
     &  1,  0,  2, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  2, -5,  0,  0,  0, &
     &  2,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  1, -1,  0,  0, -5,  7,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  431,  440 ) / &
     &  2,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0, -1, &
     &  1, -1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  441,  450 ) / &
     &  2, -2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  2, &
     &  2,  0,  0, -2,  0,  0,  0, -6,  8,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  451,  460 ) / &
     &  1, -1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  2,  0, &
     &  2,  0,  2,  0,  2,  0,  0,  2,  0, -3,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0, -2, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -4, 10,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0, -5,  6,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  461,  470 ) / &
     &  2, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  4,  0, -1,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -1, -1,  0,  0,  0,  3, -7,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -4,  5,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  471,  480 ) / &
     &  0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  2,  0,  0,  0, &
     &  1, -1, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -8,  3,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  9,-11,  0,  0,  0,  0,  0, -2, &
     &  1,  0,  0, -2,  0,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  1,  0,  0, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  4,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  481,  490 ) / &
     &  0,  0,  2, -2,  1,  0,  0, -9, 13,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0,  0,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -4, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  491,  500 ) / &
     &  0,  0,  0,  0,  1,  0,  0, -2,  4,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  2, -3,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0, &
     &  0,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  501,  510 ) / &
     &  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -8,  3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  1, &
     &  1,  1, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  511,  520 ) / &
     &  2,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -1,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  6,-10,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  1, &
     &  1,  1,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  521,  530 ) / &
     &  2,  0,  2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  2, &
     &  1,  1, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &  2, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -4,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  531,  540 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0, -4,  4,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0, -3,  5,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  3,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -2,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -1,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  541,  550 ) / &
     &  3,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  7, -8,  3,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0, -2, &
     &  1, -2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0,  0, -1,  0, -2,  5,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  551,  560 ) / &
     &  3,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0,  0, -1,  0, -1,  1,  0,  0,  0, &
     &  1,  0,  0, -1,  0,  0,  0, -3,  4,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  7,-10,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -8,  3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0, -1, &
     &  3,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  4,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  561,  570 ) / &
     &  1,  0,  0, -1, -1,  0, -3,  4,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  5,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -9,  4,  0,  0,  0,  0, -2, &
     &  0,  0,  1, -1,  1,  0,  8,-14,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  3, -6,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  571,  580 ) / &
     &  1,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  8,-15,  0,  0,  0,  0,  0, &
     &  0,  2, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -5,  4,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  581,  590 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  1,  0, -6,  8,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0, -8, 12,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  591,  600 ) / &
     &  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0, -3,  0,  0,  0,  0, &
     &  2,  0, -2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -1,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  601,  610 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0, -3,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  1, &
     &  0,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0, -1,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0, -1,  0,  0,  0,  0, &
     &  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  611,  620 ) / &
     &  2,  1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -4,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  0,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -9, 15,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8, -8,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  7,-11,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J =  621,  630 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0, &
     &  1,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  5,  0,  0,  2, &
     &  0,  1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  631,  640 ) / &
     &  3,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  641,  650 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -4,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0,  2, -5,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  2, -5,  0,  0,  2, &
     &  1,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0, -1 /
      DATA ( ( KX0(I,J), I=1,14), J =  651,  660 ) / &
     &  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0, 11,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  6,-15,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -8,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0,  1,  0,  0,  0,  2, &
     &  1,  0,  0, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  3, -7,  4,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  661,  670 ) / &
     &  0,  1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0, -8, 11,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -2,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  2, &
     &  2,  0,  0, -2,  0,  0,  0, -2,  0,  4, -3,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  0, -1 /
      DATA ( ( KX0(I,J), I=1,14), J =  671,  680 ) / &
     &  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  6,-11,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  5,  0, -2,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -4,  0,  0,  0,  0, &
     &  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0, -2,  0,  3, -1,  0,  0,  0, &
     &  2, -1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0, -2,  2,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  681,  690 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &  3,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0, -6,  8,  0,  0,  0,  0,  0,  0, &
     &  1,  2, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  1,  0,  0,  0, &
     &  3,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  691,  700 ) / &
     &  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -5,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  1,  0, -3,  7, -4,  0,  0,  0,  0,  0, &
     &  0,  0,  4, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  3, -5,  0,  2,  0,  0,  0,  0, &
     &  0,  1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0,  0, -1,  0,  0,  2,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  701,  710 ) / &
     &  0,  0,  2, -2,  0,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  0,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -2,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  2,  6,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  2, -4,  0, -3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  9, -9,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,  2, &
     &  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  711,  720 ) / &
     &  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  2, -4,  0,  0,  0,  0,  0, &
     &  1, -2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  721,  730 ) / &
     &  3, -1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  3,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  3,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  7,-13,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  3,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J =  731,  740 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  1, &
     &  1, -1,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -2,  0,  0,  5,  0,  0,  0, &
     &  3,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  1, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0, -3,  3,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  741,  750 ) / &
     &  1,  0,  0, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0, -5,  6,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &  0,  0,  1, -1, -1,  0,  0,  0, -2,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  9,-17,  0,  0,  0,  0,  0, &
     &  3, -1, -2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  1 /
      DATA ( ( KX0(I,J), I=1,14), J =  751,  760 ) / &
     &  0,  0,  0,  0,  0,  0,  5,-10,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  6,-11,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0, -2, &
     &  1,  2,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  761,  770 ) / &
     &  2,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -4,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -5,  6,  0,  0,  0,  0,  0, &
     &  0,  0,  3,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -1, -1, -1,  0,  0, -1,  0,  3,  0,  0,  0,  0, &
     &  1,  2, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  771,  780 ) / &
     &  0,  0,  4, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  7,-13,  0,  0,  0,  0,  0, &
     &  5,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  9,-12,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J =  781,  790 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  4, -8,  1,  5,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  3, -5,  0,  2,  0,  0,  0,  0, &
     &  1,  1, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0, -1,  2,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  791,  800 ) / &
     &  0,  2, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -1, -1, -1,  0,  0,  3, -7,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &  2,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  3,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  3,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  801,  810 ) / &
     &  1, -1, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  6, -5,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  5,  0, -3,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  811,  820 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0, -2, &
     &  0,  0,  1, -1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  1, -2, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0, -1,  0, -2,  5,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  3, -5,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  821,  830 ) / &
     &  1,  1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -8, 15,  0,  0,  0,  0,  0, &
     &  0,  2,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  1,  0,  0, -6,  8,  0,  0,  0,  0,  0, &
     &  3, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0, -1,  1,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -1, -1, -2,  0,  0, -1,  0,  2,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  831,  840 ) / &
     &  1,  2,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  0,  0,  0, -9, 13,  0,  0,  0,  0,  0, &
     &  3,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  0,  0, 20,-21,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  2,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -8, 11,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  2,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  2,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-12,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  841,  850 ) / &
     &  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -6,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -2,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  2,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  4,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  851,  860 ) / &
     &  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -8,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -6,  0,  0,  0,  0, -2, &
     &  0,  0,  2, -2,  1, -1,  0,  2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  861,  870 ) / &
     &  1, -2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0, &
     &  3,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -9, 17,  0,  0,  0,  0,  0, &
     &  1,  1,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  871,  880 ) / &
     &  3,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -2,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8, -8,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  8,-10,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  1 /
      DATA ( ( KX0(I,J), I=1,14), J =  881,  890 ) / &
     &  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  1, -4,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  4,  0,  0, -2,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -2,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -6,  0,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  891,  900 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  3, -7,  4,  0,  0,  0,  0,  0, &
     &  1, -1,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -2,  0,  0,  0, &
     &  0,  0,  2,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0,  0, -2,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  901,  910 ) / &
     &  0,  0,  0,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &  1,  1,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2, -1,  0, -5,  6,  0,  0,  0,  0,  0,  0, &
     &  1, -2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &  2,  0, -4, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  911,  920 ) / &
     &  2,  0, -2, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -1, -1, -1,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &  4, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  2,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  2, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,  0, -2,  0,  2,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  921,  930 ) / &
     &  1,  0,  0, -2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  1, -1,  0,-18, 17,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-12,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  8,-16,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  7, -8,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J =  931,  940 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  4, -8,  1,  5,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -7,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  2, &
     &  0,  1,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  941,  950 ) / &
     &  2,  1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -2,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0, -1,  0, -1,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  1,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0, -3,  3,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  951,  960 ) / &
     &  1, -2,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1,  0,  0,  0, &
     &  1,  0, -2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  961,  970 ) / &
     &  0,  0,  2,  3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -2,  0,-10,  3,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  0,  0,  0, -2,  2,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  0,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  971,  980 ) / &
     &  4, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -2,  0,  4, -5,  0,  0,  0, &
     &  2,  0, -1, -1, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  3,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,  0, -4,  8, -3,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J =  981,  990 ) / &
     &  1,  0,  0,  0,  0,  0, -1,  1,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  0,  0, 17,-16,  0, -2,  0,  0,  0,  0, &
     &  1,  0, -1, -1, -1,  0, 20,-20,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -2,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  0,  3, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  1, -2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  9, -9,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  9,-11,  0,  0,  0,  0,  0, -1 /
      DATA ( ( KX0(I,J), I=1,14), J =  991, 1000 ) / &
     &  0,  0,  0,  0,  0,  0,  6,-10,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  5,-10,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  4,  0, -4,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0, -4,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -5,  0,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J = 1001, 1010 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  5,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -5,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, -2, -2, &
     &  0,  2,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1011, 1020 ) / &
     &  1,  0, -1,  1, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1, -1,  0,  0, -2,  2,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  5, -8,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  1, -1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1021, 1030 ) / &
     &  0,  0,  1, -1,  2,  0, -5,  7,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1, -1,  0,  0, -1,  0, -1,  0,  0,  0,  0, &
     &  2,  0, -2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1031, 1040 ) / &
     &  1,  0,  0, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &  2,  0,  0, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -4, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1041, 1050 ) / &
     &  3,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0,  3, -6,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0, -4,  5,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1051, 1060 ) / &
     &  0,  0,  1, -1,  0,  0,  0, -2,  2,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0,  0, -1,  0,  0,  1,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &  0,  1, -2,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &  3,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0, -2,  0,  2,  2,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1061, 1070 ) / &
     &  2,  0,  0, -2,  0,  0,  0, -4,  4,  0,  0,  0,  0,  0, &
     &  2, -2,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  0,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1071, 1080 ) / &
     &  0,  0,  2, -2,  1,  0,  0, -3,  0,  3,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0, -5,  5,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  1, -3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0, -1,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -4,  6,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2,  0,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8, -9,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  7,-10,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  1 /
      DATA ( ( KX0(I,J), I=1,14), J = 1081, 1090 ) / &
     &  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -8,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  7, -8,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  6,-10,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J = 1091, 1100 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  3, -8,  3,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -2,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0, &
     &  2,  1,  0, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0, -3,  4,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1101, 1110 ) / &
     &  3, -1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  0,  0, -4,  4,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1111, 1120 ) / &
     &  3,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -4,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -2,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  1,  0, -2,  0,  0,  0,  0, &
     &  3,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  3, -4,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1121, 1130 ) / &
     &  0,  1,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0, -8, 12,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0, -1,  0,-10,  3,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  1,  0,-10,  3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  0,  0,  1,  0,  0,  0,  0, &
     &  0,  1,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1131, 1140 ) / &
     &  1,  0,  2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -6,  8,  0,  0,  0,  0,  0, &
     &  2, -1, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  1,  0,  0, -5,  6,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2, -1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  2,  0, -1, -1,  1,  0,  0,  3, -7,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  1,  0,  0, -1, -1,  0,  0, -3,  4,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1141, 1150 ) / &
     &  2,  2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  4,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -2,  0,  3, -1,  0,  0,  0, &
     &  2, -1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  2, -3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0, -2,  3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0, -2,  2,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1151, 1160 ) / &
     &  5,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0, -2, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -2, -2,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  2,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0, -8, 11,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1161, 1170 ) / &
     &  2,  0,  2, -2,  2,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  2,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  2,  0, -1,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  0, -1,  0,  2,  0,  0,  0,  0,  0,  0, &
     &  4,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -6,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -4,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0,  0, -1,  0,  0, -2,  0,  0,  0, &
     &  1, -1, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1171, 1180 ) / &
     &  2,  0, -1, -1,  0,  0,  0, -1,  0,  3,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, &
     &  4,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  1, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3, -1,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1181, 1190 ) / &
     &  2,  0,  1, -3,  1,  0, -6,  7,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  2, -5,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0, -2,  0,  5, -5,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0, -2,  0,  1,  5,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0, -2,  0,  0,  2,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0, -4,  4,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0, -2,  0,  0,  5, -9,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -5, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  3, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1191, 1200 ) / &
     &  1,  1, -2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  2,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &  1,  0, -1,  0, -1,  0, -3,  5,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1, -1,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0, &
     &  1,  0, -1, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -2,  0,  0,  1,  0, -1,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1201, 1210 ) / &
     &  0,  0,  2,  2,  2,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -2,  0,  1,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -4,  4,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -7,  9,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0,-10, 15,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0,  1, -4,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  1, -3,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -1,  2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -4,  6,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1211, 1220 ) / &
     &  0,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &  0,  0,  0,  2,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  9,-13,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  8,-11,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  8,-14,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  7,-11,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  6, -4,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1221, 1230 ) / &
     &  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  5, -6, -4,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -8,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -3,  0,  2,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J = 1231, 1240 ) / &
     &  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  1, -4,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  9,-17,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  7,-12,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  6, -4,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  6, -8,  1,  5,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J = 1241, 1250 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  6,-10,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  5,  0, -4,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -8,  3,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  5,-13,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  5,-16,  4,  5,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0, -1 /
      DATA ( ( KX0(I,J), I=1,14), J = 1251, 1260 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0, -5,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -7,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -9,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  2,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -3,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -8,  1,  5,  0,  0, -2 /
      DATA ( ( KX0(I,J), I=1,14), J = 1261, 1270 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  1, -5,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -3,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  5,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -6,  3,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2 /
      DATA ( ( KX0(I,J), I=1,14), J = 1271, 1280 ) / &
     &  1,  1,  0, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0, -2,  0,  0,  5,  0,  0,  0, &
     &  3,  0, -2, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -6,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1281, 1290 ) / &
     &  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  7,-13,  0,  0,  0,  0,  0, &
     &  2,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1, -1,  0, -5,  7,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0, -3,  5,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1291, 1300 ) / &
     &  0,  0,  1,  1,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0, -3,  4,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0, -1,  2,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0, -1,  0,  0, -1,  0,  0,  0, &
     &  2,  1,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX0(I,J), I=1,14), J = 1301,  NX0 ) / &
     &  1,  0,  0, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &  1,  0, -2,  4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4, -4,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
!
!  Argument coefficients for t^1 terms
      DATA ( ( KX1(I,J), I=1,14), J =    1,   10 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   11,   20 ) / &
     &  1,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   21,   30 ) / &
     &  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0, &
     &  0,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   31,   40 ) / &
     &  0,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   41,   50 ) / &
     &  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0, -1, &
     &  0,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -1,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   51,   60 ) / &
     &  1, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0, -2, &
     &  1,  0, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   61,   70 ) / &
     &  1,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   71,   80 ) / &
     &  2,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  2, &
     &  1,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -4,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -1,  2,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   81,   90 ) / &
     &  0,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -1, &
     &  1,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =   91,  100 ) / &
     &  0,  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  2, -5,  0,  0,  0, &
     &  1, -1,  0, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -8,  3,  0,  0,  0, -2, &
     &  1, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  101,  110 ) / &
     &  0,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  6, -8,  3,  0,  0,  0,  2, &
     &  0,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  111,  120 ) / &
     &  1, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -2, &
     &  2, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &  0,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2 /
      DATA ( ( KX1(I,J), I=1,14), J =  121,  130 ) / &
     &  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0,  0, -2,  0,  0,  0,  0,  0, &
     &  0,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  3,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  131,  140 ) / &
     &  1,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1, &
     &  1, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  2, &
     &  1,  0, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  2, &
     &  1,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  141,  150 ) / &
     &  1,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -1,  0,  0,  0, &
     &  1,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  1 /
      DATA ( ( KX1(I,J), I=1,14), J =  151,  160 ) / &
     &  2,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, -2, &
     &  0,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  2, &
     &  1,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  161,  170 ) / &
     &  1,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -4,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1, &
     &  0,  0,  1, -1,  1,  0, -5,  7,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  171,  180 ) / &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  2,  0,  0,  0, &
     &  0,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0, -2, &
     &  1,  0, -2,  0, -2,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  4,  0, -2,  0,  0,  0,  2, &
     &  0,  0,  4,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  181,  190 ) / &
     &  0,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0,  0,  2, &
     &  1,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-11,  0,  0,  0,  0,  0, -2, &
     &  3,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  191,  200 ) / &
     &  1,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0, -1, &
     &  1,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  2, &
     &  3,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  2 /
      DATA ( ( KX1(I,J), I=1,14), J =  201,  210 ) / &
     &  2,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  2, &
     &  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  2, &
     &  0,  0,  2, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0,  3, -8,  3,  0,  0,  0,  0, &
     &  0,  0,  2, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  211,  220 ) / &
     &  0,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  8,-13,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  2, &
     &  3,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  221,  230 ) / &
     &  1,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, -1, &
     &  2,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  2, &
     &  0,  0,  1, -1,  1,  0,  0, -5,  8, -3,  0,  0,  0,  0, &
     &  1,  0,  0,  0, -1,  0,-18, 16,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0, -2, &
     &  1, -1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  231,  240 ) / &
     &  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0, -2, &
     &  1,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  2, &
     &  1, -1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0, -8, 13,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0, -2 /
      DATA ( ( KX1(I,J), I=1,14), J =  241,  250 ) / &
     &  4,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0, -2,  5,  0,  0,  0, &
     &  2,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  2, -5,  0,  0,  0, &
     &  0,  2,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0, &
     &  2,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  1,  0,-18, 16,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX1(I,J), I=1,14), J =  251,  NX1 ) / &
     &  1,  2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
!
!  Argument coefficients for t^2 terms
      DATA ( ( KX2(I,J), I=1,14), J =    1,   10 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX2(I,J), I=1,14), J =   11,   20 ) / &
     &  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX2(I,J), I=1,14), J =   21,   30 ) / &
     &  0,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KX2(I,J), I=1,14), J =   31,  NX2 ) / &
     &  1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
!
!  Argument coefficients for t^3 terms
      DATA ( ( KX3(I,J), I=1,14), J =    1,  NX3 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
!
!  Argument coefficients for t^4 terms
      DATA ( ( KX4(I,J), I=1,14), J =    1,  NX4 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
!
!  Sine and cosine coefficients for t^0 terms
      DATA ( ( SX0(I,J), I=1,2), J =    1,   10 ) / &
     &         -6844318.44D-6,       +1328.67D-6, &
     &          -523908.04D-6,        -544.76D-6, &
     &           -90552.22D-6,        +111.23D-6, &
     &           +82168.76D-6,         -27.64D-6, &
     &           +58707.02D-6,        +470.05D-6, &
     &           +28288.28D-6,         -34.69D-6, &
     &           -20557.78D-6,         -20.84D-6, &
     &           -15406.85D-6,         +15.12D-6, &
     &           -11991.74D-6,         +32.46D-6, &
     &            -8584.95D-6,          +4.42D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   11,   20 ) / &
     &            -6245.02D-6,          -6.68D-6, &
     &            +5095.50D-6,          +7.19D-6, &
     &            -4910.93D-6,          +0.76D-6, &
     &            +2521.07D-6,          -5.97D-6, &
     &            +2511.85D-6,          +1.07D-6, &
     &            +2372.58D-6,          +5.93D-6, &
     &            +2307.58D-6,          -7.52D-6, &
     &            -2053.16D-6,          +5.13D-6, &
     &            +1898.27D-6,          -0.72D-6, &
     &            -1825.49D-6,          +1.23D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   21,   30 ) / &
     &            -1534.09D-6,          +6.29D-6, &
     &            -1292.02D-6,          +0.00D-6, &
     &            -1234.96D-6,          +5.21D-6, &
     &            +1163.22D-6,          -2.94D-6, &
     &            +1137.48D-6,          -0.04D-6, &
     &            +1029.70D-6,          -2.63D-6, &
     &             -866.48D-6,          +0.52D-6, &
     &             -813.13D-6,          +0.40D-6, &
     &             +664.57D-6,          -0.40D-6, &
     &             -628.24D-6,          -0.64D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   31,   40 ) / &
     &             -603.52D-6,          +0.44D-6, &
     &             -556.26D-6,          +3.16D-6, &
     &             -512.37D-6,          -1.47D-6, &
     &             +506.65D-6,          +2.54D-6, &
     &             +438.51D-6,          -0.56D-6, &
     &             +405.91D-6,          +0.99D-6, &
     &             -122.67D-6,        +203.78D-6, &
     &             -305.78D-6,          +1.75D-6, &
     &             +300.99D-6,          -0.44D-6, &
     &             -292.37D-6,          -0.32D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   41,   50 ) / &
     &             +284.09D-6,          +0.32D-6, &
     &             -264.02D-6,          +0.99D-6, &
     &             +261.54D-6,          -0.95D-6, &
     &             +256.30D-6,          -0.28D-6, &
     &             -250.54D-6,          +0.08D-6, &
     &             +230.72D-6,          +0.08D-6, &
     &             +229.78D-6,          -0.60D-6, &
     &             -212.82D-6,          +0.84D-6, &
     &             +196.64D-6,          -0.84D-6, &
     &             +188.95D-6,          -0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   51,   60 ) / &
     &             +187.95D-6,          -0.24D-6, &
     &             -160.15D-6,         -14.04D-6, &
     &             -172.95D-6,          -0.40D-6, &
     &             -168.26D-6,          +0.20D-6, &
     &             +161.79D-6,          +0.24D-6, &
     &             +161.34D-6,          +0.20D-6, &
     &              +57.44D-6,         +95.82D-6, &
     &             +142.16D-6,          +0.20D-6, &
     &             -134.81D-6,          +0.20D-6, &
     &             +132.81D-6,          -0.52D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   61,   70 ) / &
     &             -130.31D-6,          +0.04D-6, &
     &             +121.98D-6,          -0.08D-6, &
     &             -115.40D-6,          +0.60D-6, &
     &             -114.49D-6,          +0.32D-6, &
     &             +112.14D-6,          +0.28D-6, &
     &             +105.29D-6,          +0.44D-6, &
     &              +98.69D-6,          -0.28D-6, &
     &              +91.31D-6,          -0.40D-6, &
     &              +86.74D-6,          -0.08D-6, &
     &              -18.38D-6,         +63.80D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   71,   80 ) / &
     &              +82.14D-6,          +0.00D-6, &
     &              +79.03D-6,          -0.24D-6, &
     &               +0.00D-6,         -79.08D-6, &
     &              -78.56D-6,          +0.00D-6, &
     &              +47.73D-6,         +23.79D-6, &
     &              +66.03D-6,          -0.20D-6, &
     &              +62.65D-6,          -0.24D-6, &
     &              +60.50D-6,          +0.36D-6, &
     &              +59.07D-6,          +0.00D-6, &
     &              +57.28D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   81,   90 ) / &
     &              -55.66D-6,          +0.16D-6, &
     &              -54.81D-6,          -0.08D-6, &
     &              -53.22D-6,          -0.20D-6, &
     &              -52.95D-6,          +0.32D-6, &
     &              -52.27D-6,          +0.00D-6, &
     &              +51.32D-6,          +0.00D-6, &
     &              -51.00D-6,          -0.12D-6, &
     &              +51.02D-6,          +0.00D-6, &
     &              -48.65D-6,          -1.15D-6, &
     &              +48.29D-6,          +0.20D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =   91,  100 ) / &
     &              -46.38D-6,          +0.00D-6, &
     &              -45.59D-6,          -0.12D-6, &
     &              -43.76D-6,          +0.36D-6, &
     &              -40.58D-6,          -1.00D-6, &
     &               +0.00D-6,         -41.53D-6, &
     &              +40.54D-6,          -0.04D-6, &
     &              +40.33D-6,          -0.04D-6, &
     &              -38.57D-6,          +0.08D-6, &
     &              +37.75D-6,          +0.04D-6, &
     &              +37.15D-6,          -0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  101,  110 ) / &
     &              +36.68D-6,          -0.04D-6, &
     &              -18.30D-6,         -17.30D-6, &
     &              -17.86D-6,         +17.10D-6, &
     &              -34.81D-6,          +0.04D-6, &
     &              -33.22D-6,          +0.08D-6, &
     &              +32.43D-6,          -0.04D-6, &
     &              -30.47D-6,          +0.04D-6, &
     &              -29.53D-6,          +0.04D-6, &
     &              +28.50D-6,          -0.08D-6, &
     &              +28.35D-6,          -0.16D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  111,  120 ) / &
     &              -28.00D-6,          +0.00D-6, &
     &              -27.61D-6,          +0.20D-6, &
     &              -26.77D-6,          +0.08D-6, &
     &              +26.54D-6,          -0.12D-6, &
     &              +26.54D-6,          +0.04D-6, &
     &              -26.17D-6,          +0.00D-6, &
     &              -25.42D-6,          -0.08D-6, &
     &              -16.91D-6,          +8.43D-6, &
     &               +0.32D-6,         +24.42D-6, &
     &              -19.53D-6,          +5.09D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  121,  130 ) / &
     &              -23.79D-6,          +0.00D-6, &
     &              +23.66D-6,          +0.00D-6, &
     &              -23.47D-6,          +0.16D-6, &
     &              +23.39D-6,          -0.12D-6, &
     &              -23.49D-6,          +0.00D-6, &
     &              -23.28D-6,          -0.08D-6, &
     &              -22.99D-6,          +0.04D-6, &
     &              -22.67D-6,          -0.08D-6, &
     &               +9.35D-6,         +13.29D-6, &
     &              +22.47D-6,          -0.04D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  131,  140 ) / &
     &               +4.89D-6,         -16.55D-6, &
     &               +4.89D-6,         -16.51D-6, &
     &              +21.28D-6,          -0.08D-6, &
     &              +20.57D-6,          +0.64D-6, &
     &              +21.01D-6,          +0.00D-6, &
     &               +1.23D-6,         -19.13D-6, &
     &              -19.97D-6,          +0.12D-6, &
     &              +19.65D-6,          -0.08D-6, &
     &              +19.58D-6,          -0.12D-6, &
     &              +19.61D-6,          -0.08D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  141,  150 ) / &
     &              -19.41D-6,          +0.08D-6, &
     &              -19.49D-6,          +0.00D-6, &
     &              -18.64D-6,          +0.00D-6, &
     &              +18.58D-6,          +0.04D-6, &
     &              -18.42D-6,          +0.00D-6, &
     &              +18.22D-6,          +0.00D-6, &
     &               -0.72D-6,         -17.34D-6, &
     &              -18.02D-6,          -0.04D-6, &
     &              +17.74D-6,          +0.08D-6, &
     &              +17.46D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  151,  160 ) / &
     &              -17.42D-6,          +0.00D-6, &
     &               -6.60D-6,         +10.70D-6, &
     &              +16.43D-6,          +0.52D-6, &
     &              -16.75D-6,          +0.04D-6, &
     &              +16.55D-6,          -0.08D-6, &
     &              +16.39D-6,          -0.08D-6, &
     &              +13.88D-6,          -2.47D-6, &
     &              +15.69D-6,          +0.00D-6, &
     &              -15.52D-6,          +0.00D-6, &
     &               +3.34D-6,         +11.86D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  161,  170 ) / &
     &              +14.72D-6,          -0.32D-6, &
     &              +14.92D-6,          -0.04D-6, &
     &               -3.26D-6,         +11.62D-6, &
     &              -14.64D-6,          +0.00D-6, &
     &               +0.00D-6,         +14.47D-6, &
     &              -14.37D-6,          +0.00D-6, &
     &              +14.32D-6,          -0.04D-6, &
     &              -14.10D-6,          +0.04D-6, &
     &              +10.86D-6,          +3.18D-6, &
     &              -10.58D-6,          -3.10D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  171,  180 ) / &
     &               -3.62D-6,          +9.86D-6, &
     &              -13.48D-6,          +0.00D-6, &
     &              +13.41D-6,          -0.04D-6, &
     &              +13.32D-6,          -0.08D-6, &
     &              -13.33D-6,          -0.04D-6, &
     &              -13.29D-6,          +0.00D-6, &
     &               -0.20D-6,         +13.05D-6, &
     &               +0.00D-6,         +13.13D-6, &
     &               -8.99D-6,          +4.02D-6, &
     &              -12.93D-6,          +0.04D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  181,  190 ) / &
     &               +2.03D-6,         +10.82D-6, &
     &              -12.78D-6,          +0.04D-6, &
     &              +12.24D-6,          +0.04D-6, &
     &               +8.71D-6,          +3.54D-6, &
     &              +11.98D-6,          -0.04D-6, &
     &              -11.38D-6,          +0.04D-6, &
     &              -11.30D-6,          +0.00D-6, &
     &              +11.14D-6,          -0.04D-6, &
     &              +10.98D-6,          +0.00D-6, &
     &              -10.98D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  191,  200 ) / &
     &               +0.44D-6,         -10.38D-6, &
     &              +10.46D-6,          +0.08D-6, &
     &              -10.42D-6,          +0.00D-6, &
     &              -10.30D-6,          +0.08D-6, &
     &               +6.92D-6,          +3.34D-6, &
     &              +10.07D-6,          +0.04D-6, &
     &              +10.02D-6,          +0.00D-6, &
     &               -9.75D-6,          +0.04D-6, &
     &               +9.75D-6,          +0.00D-6, &
     &               +9.67D-6,          -0.04D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  201,  210 ) / &
     &               -1.99D-6,          +7.72D-6, &
     &               +0.40D-6,          +9.27D-6, &
     &               -3.42D-6,          +6.09D-6, &
     &               +0.56D-6,          -8.67D-6, &
     &               -9.19D-6,          +0.00D-6, &
     &               +9.11D-6,          +0.00D-6, &
     &               +9.07D-6,          +0.00D-6, &
     &               +1.63D-6,          +6.96D-6, &
     &               -8.47D-6,          +0.00D-6, &
     &               -8.28D-6,          +0.04D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  211,  220 ) / &
     &               +8.27D-6,          +0.04D-6, &
     &               -8.04D-6,          +0.00D-6, &
     &               +7.91D-6,          +0.00D-6, &
     &               -7.84D-6,          -0.04D-6, &
     &               -7.64D-6,          +0.08D-6, &
     &               +5.21D-6,          -2.51D-6, &
     &               -5.77D-6,          +1.87D-6, &
     &               +5.01D-6,          -2.51D-6, &
     &               -7.48D-6,          +0.00D-6, &
     &               -7.32D-6,          -0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  221,  230 ) / &
     &               +7.40D-6,          -0.04D-6, &
     &               +7.44D-6,          +0.00D-6, &
     &               +6.32D-6,          -1.11D-6, &
     &               -6.13D-6,          -1.19D-6, &
     &               +0.20D-6,          -6.88D-6, &
     &               +6.92D-6,          +0.04D-6, &
     &               +6.48D-6,          -0.48D-6, &
     &               -6.94D-6,          +0.00D-6, &
     &               +2.47D-6,          -4.46D-6, &
     &               -2.23D-6,          -4.65D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  231,  240 ) / &
     &               -1.07D-6,          -5.69D-6, &
     &               +4.97D-6,          -1.71D-6, &
     &               +5.57D-6,          +1.07D-6, &
     &               -6.48D-6,          +0.08D-6, &
     &               +2.03D-6,          +4.53D-6, &
     &               +4.10D-6,          -2.39D-6, &
     &               +0.00D-6,          -6.44D-6, &
     &               -6.40D-6,          +0.00D-6, &
     &               +6.32D-6,          +0.00D-6, &
     &               +2.67D-6,          -3.62D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  241,  250 ) / &
     &               -1.91D-6,          -4.38D-6, &
     &               -2.43D-6,          -3.82D-6, &
     &               +6.20D-6,          +0.00D-6, &
     &               -3.38D-6,          -2.78D-6, &
     &               -6.12D-6,          +0.04D-6, &
     &               -6.09D-6,          -0.04D-6, &
     &               -6.01D-6,          -0.04D-6, &
     &               +3.18D-6,          -2.82D-6, &
     &               -5.05D-6,          +0.84D-6, &
     &               +5.85D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  251,  260 ) / &
     &               +5.69D-6,          -0.12D-6, &
     &               +5.73D-6,          -0.04D-6, &
     &               +5.61D-6,          +0.00D-6, &
     &               +5.49D-6,          +0.00D-6, &
     &               -5.33D-6,          +0.04D-6, &
     &               -5.29D-6,          +0.00D-6, &
     &               +5.25D-6,          +0.00D-6, &
     &               +0.99D-6,          +4.22D-6, &
     &               -0.99D-6,          +4.22D-6, &
     &               +0.00D-6,          +5.21D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  261,  270 ) / &
     &               +5.13D-6,          +0.04D-6, &
     &               -4.90D-6,          +0.00D-6, &
     &               -3.10D-6,          +1.79D-6, &
     &               -4.81D-6,          +0.04D-6, &
     &               -4.75D-6,          +0.00D-6, &
     &               +4.70D-6,          -0.04D-6, &
     &               -4.69D-6,          +0.00D-6, &
     &               -4.65D-6,          +0.00D-6, &
     &               +4.65D-6,          +0.00D-6, &
     &               -4.57D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  271,  280 ) / &
     &               +4.49D-6,          -0.04D-6, &
     &               -4.53D-6,          +0.00D-6, &
     &               +0.00D-6,          -4.53D-6, &
     &               +0.00D-6,          -4.53D-6, &
     &               -4.53D-6,          +0.00D-6, &
     &               +4.50D-6,          +0.00D-6, &
     &               -4.49D-6,          +0.00D-6, &
     &               +1.83D-6,          +2.63D-6, &
     &               +4.38D-6,          +0.00D-6, &
     &               +0.88D-6,          -3.46D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  281,  290 ) / &
     &               -2.70D-6,          +1.55D-6, &
     &               -4.22D-6,          +0.00D-6, &
     &               -4.10D-6,          -0.12D-6, &
     &               +3.54D-6,          -0.64D-6, &
     &               -3.50D-6,          +0.68D-6, &
     &               +4.18D-6,          +0.00D-6, &
     &               +4.14D-6,          +0.00D-6, &
     &               +4.10D-6,          +0.00D-6, &
     &               -4.06D-6,          +0.00D-6, &
     &               +2.70D-6,          -1.35D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  291,  300 ) / &
     &               -4.04D-6,          +0.00D-6, &
     &               -3.98D-6,          -0.04D-6, &
     &               -3.98D-6,          +0.04D-6, &
     &               +4.02D-6,          +0.00D-6, &
     &               +3.94D-6,          +0.00D-6, &
     &               +0.84D-6,          -3.10D-6, &
     &               +3.30D-6,          +0.60D-6, &
     &               -1.59D-6,          +2.27D-6, &
     &               -3.66D-6,          -0.20D-6, &
     &               -3.10D-6,          -0.72D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  301,  310 ) / &
     &               -3.82D-6,          +0.00D-6, &
     &               -3.62D-6,          -0.16D-6, &
     &               -3.74D-6,          +0.00D-6, &
     &               +3.74D-6,          +0.00D-6, &
     &               -3.74D-6,          +0.00D-6, &
     &               -3.71D-6,          +0.00D-6, &
     &               +3.02D-6,          +0.68D-6, &
     &               +3.70D-6,          +0.00D-6, &
     &               +3.30D-6,          +0.40D-6, &
     &               -3.66D-6,          +0.04D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  311,  320 ) / &
     &               +3.66D-6,          +0.04D-6, &
     &               -3.62D-6,          +0.00D-6, &
     &               -3.61D-6,          +0.00D-6, &
     &               -2.90D-6,          +0.68D-6, &
     &               +0.80D-6,          -2.78D-6, &
     &               +3.54D-6,          +0.00D-6, &
     &               -3.54D-6,          +0.00D-6, &
     &               -3.50D-6,          +0.00D-6, &
     &               +3.45D-6,          +0.00D-6, &
     &               +0.00D-6,          -3.42D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  321,  330 ) / &
     &               +3.38D-6,          +0.00D-6, &
     &               +2.27D-6,          -1.11D-6, &
     &               -3.34D-6,          +0.00D-6, &
     &               +3.34D-6,          +0.00D-6, &
     &               -3.30D-6,          +0.01D-6, &
     &               +3.31D-6,          +0.00D-6, &
     &               +3.30D-6,          +0.00D-6, &
     &               -3.30D-6,          +0.00D-6, &
     &               -1.39D-6,          -1.91D-6, &
     &               +3.30D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  331,  340 ) / &
     &               +3.26D-6,          +0.00D-6, &
     &               +3.26D-6,          +0.00D-6, &
     &               +3.22D-6,          -0.04D-6, &
     &               -3.26D-6,          +0.00D-6, &
     &               +2.51D-6,          -0.64D-6, &
     &               +3.14D-6,          +0.00D-6, &
     &               -2.63D-6,          -0.48D-6, &
     &               +3.10D-6,          +0.00D-6, &
     &               -3.06D-6,          +0.00D-6, &
     &               +2.94D-6,          -0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  341,  350 ) / &
     &               +3.06D-6,          +0.00D-6, &
     &               +0.00D-6,          +2.98D-6, &
     &               +2.98D-6,          +0.00D-6, &
     &               +2.07D-6,          +0.91D-6, &
     &               -2.98D-6,          +0.00D-6, &
     &               +2.94D-6,          +0.00D-6, &
     &               -2.94D-6,          +0.00D-6, &
     &               -2.94D-6,          +0.00D-6, &
     &               -2.90D-6,          +0.00D-6, &
     &               -0.56D-6,          -2.35D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  351,  360 ) / &
     &               -1.47D-6,          +1.39D-6, &
     &               +2.80D-6,          +0.00D-6, &
     &               -2.74D-6,          +0.00D-6, &
     &               -0.12D-6,          +2.63D-6, &
     &               +2.15D-6,          -0.60D-6, &
     &               -2.70D-6,          +0.00D-6, &
     &               +1.79D-6,          -0.88D-6, &
     &               -0.48D-6,          +2.19D-6, &
     &               +0.44D-6,          +2.23D-6, &
     &               +0.52D-6,          +2.07D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  361,  370 ) / &
     &               -2.59D-6,          +0.00D-6, &
     &               +2.55D-6,          +0.00D-6, &
     &               -1.11D-6,          +1.43D-6, &
     &               -2.51D-6,          +0.00D-6, &
     &               -2.51D-6,          +0.00D-6, &
     &               +2.51D-6,          +0.00D-6, &
     &               +0.00D-6,          -2.50D-6, &
     &               +2.47D-6,          +0.00D-6, &
     &               +2.11D-6,          -0.36D-6, &
     &               +1.67D-6,          +0.80D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  371,  380 ) / &
     &               +2.46D-6,          +0.00D-6, &
     &               -2.43D-6,          +0.00D-6, &
     &               -2.39D-6,          +0.00D-6, &
     &               -1.83D-6,          +0.56D-6, &
     &               -0.44D-6,          -1.95D-6, &
     &               +0.24D-6,          +2.15D-6, &
     &               +2.39D-6,          +0.00D-6, &
     &               +2.35D-6,          +0.00D-6, &
     &               +2.27D-6,          +0.00D-6, &
     &               -2.22D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  381,  390 ) / &
     &               -1.03D-6,          -1.15D-6, &
     &               +1.87D-6,          +0.32D-6, &
     &               -0.32D-6,          -1.87D-6, &
     &               +2.15D-6,          +0.00D-6, &
     &               -0.80D-6,          +1.35D-6, &
     &               +2.11D-6,          +0.00D-6, &
     &               -2.11D-6,          +0.00D-6, &
     &               -0.56D-6,          -1.55D-6, &
     &               +2.11D-6,          +0.00D-6, &
     &               -0.84D-6,          -1.27D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  391,  400 ) / &
     &               -1.99D-6,          +0.12D-6, &
     &               -0.24D-6,          +1.87D-6, &
     &               -0.24D-6,          -1.87D-6, &
     &               -2.03D-6,          +0.00D-6, &
     &               +2.03D-6,          +0.00D-6, &
     &               +2.03D-6,          +0.00D-6, &
     &               +2.03D-6,          +0.00D-6, &
     &               -0.40D-6,          +1.59D-6, &
     &               +1.99D-6,          +0.00D-6, &
     &               +1.95D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  401,  410 ) / &
     &               +1.95D-6,          +0.00D-6, &
     &               +1.91D-6,          +0.00D-6, &
     &               +1.19D-6,          -0.72D-6, &
     &               +1.87D-6,          +0.00D-6, &
     &               +1.87D-6,          +0.00D-6, &
     &               -1.27D-6,          +0.60D-6, &
     &               +0.72D-6,          -1.15D-6, &
     &               -0.99D-6,          +0.88D-6, &
     &               +1.87D-6,          +0.00D-6, &
     &               -1.87D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  411,  420 ) / &
     &               -1.83D-6,          +0.00D-6, &
     &               -1.79D-6,          +0.00D-6, &
     &               -1.79D-6,          +0.00D-6, &
     &               +1.79D-6,          +0.00D-6, &
     &               +0.00D-6,          -1.79D-6, &
     &               -1.79D-6,          +0.00D-6, &
     &               -1.75D-6,          +0.00D-6, &
     &               -1.75D-6,          +0.00D-6, &
     &               +1.75D-6,          +0.00D-6, &
     &               -1.47D-6,          -0.28D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  421,  430 ) / &
     &               -1.71D-6,          +0.00D-6, &
     &               +1.71D-6,          +0.00D-6, &
     &               +0.32D-6,          +1.39D-6, &
     &               +0.28D-6,          -1.43D-6, &
     &               -0.52D-6,          -1.19D-6, &
     &               +1.67D-6,          +0.00D-6, &
     &               -1.67D-6,          +0.00D-6, &
     &               +0.76D-6,          -0.91D-6, &
     &               -0.32D-6,          +1.35D-6, &
     &               -1.39D-6,          -0.28D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  431,  440 ) / &
     &               +1.63D-6,          +0.00D-6, &
     &               -1.59D-6,          +0.00D-6, &
     &               +1.03D-6,          -0.56D-6, &
     &               +1.59D-6,          +0.00D-6, &
     &               +1.55D-6,          +0.00D-6, &
     &               -0.28D-6,          -1.27D-6, &
     &               -0.64D-6,          +0.91D-6, &
     &               -0.32D-6,          -1.23D-6, &
     &               -1.55D-6,          +0.00D-6, &
     &               -1.51D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  441,  450 ) / &
     &               +1.51D-6,          +0.00D-6, &
     &               -1.51D-6,          +0.00D-6, &
     &               +1.51D-6,          +0.00D-6, &
     &               +1.47D-6,          +0.00D-6, &
     &               +1.47D-6,          +0.00D-6, &
     &               +0.95D-6,          -0.52D-6, &
     &               +1.23D-6,          -0.24D-6, &
     &               +0.60D-6,          +0.88D-6, &
     &               -1.47D-6,          +0.00D-6, &
     &               -1.43D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  451,  460 ) / &
     &               +1.43D-6,          +0.00D-6, &
     &               +1.43D-6,          +0.00D-6, &
     &               -0.68D-6,          -0.76D-6, &
     &               +0.95D-6,          -0.48D-6, &
     &               -0.95D-6,          -0.48D-6, &
     &               -1.19D-6,          -0.24D-6, &
     &               +0.36D-6,          -1.07D-6, &
     &               +0.95D-6,          +0.48D-6, &
     &               +1.43D-6,          +0.00D-6, &
     &               +1.39D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  461,  470 ) / &
     &               +1.39D-6,          +0.00D-6, &
     &               -1.39D-6,          +0.00D-6, &
     &               -1.39D-6,          +0.00D-6, &
     &               +0.00D-6,          +1.39D-6, &
     &               -0.12D-6,          -1.27D-6, &
     &               +0.56D-6,          +0.84D-6, &
     &               -0.44D-6,          -0.95D-6, &
     &               +0.32D-6,          -1.07D-6, &
     &               +1.03D-6,          -0.36D-6, &
     &               -0.28D-6,          +1.11D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  471,  480 ) / &
     &               +0.44D-6,          +0.95D-6, &
     &               -1.35D-6,          +0.00D-6, &
     &               +0.88D-6,          +0.48D-6, &
     &               -1.35D-6,          +0.00D-6, &
     &               +1.35D-6,          +0.00D-6, &
     &               +1.35D-6,          +0.00D-6, &
     &               -1.31D-6,          +0.00D-6, &
     &               +1.31D-6,          +0.00D-6, &
     &               -1.19D-6,          -0.12D-6, &
     &               +1.27D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  481,  490 ) / &
     &               +0.40D-6,          -0.88D-6, &
     &               +1.27D-6,          +0.00D-6, &
     &               +1.27D-6,          +0.00D-6, &
     &               -0.16D-6,          -1.11D-6, &
     &               -0.84D-6,          +0.44D-6, &
     &               +0.84D-6,          -0.44D-6, &
     &               +0.84D-6,          -0.44D-6, &
     &               -1.27D-6,          +0.00D-6, &
     &               -1.27D-6,          +0.00D-6, &
     &               +1.27D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  491,  500 ) / &
     &               -0.44D-6,          -0.84D-6, &
     &               +0.00D-6,          -1.27D-6, &
     &               -1.27D-6,          +0.00D-6, &
     &               -1.23D-6,          +0.00D-6, &
     &               -1.23D-6,          +0.00D-6, &
     &               +1.23D-6,          +0.00D-6, &
     &               +0.00D-6,          +1.23D-6, &
     &               -0.12D-6,          +1.11D-6, &
     &               +1.22D-6,          +0.00D-6, &
     &               +1.19D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  501,  510 ) / &
     &               -0.24D-6,          +0.95D-6, &
     &               -0.76D-6,          -0.44D-6, &
     &               +0.91D-6,          +0.28D-6, &
     &               +1.19D-6,          +0.00D-6, &
     &               +1.19D-6,          +0.00D-6, &
     &               +0.00D-6,          +1.19D-6, &
     &               +1.15D-6,          +0.00D-6, &
     &               +0.00D-6,          +1.15D-6, &
     &               -1.15D-6,          +0.00D-6, &
     &               +1.15D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  511,  520 ) / &
     &               -1.15D-6,          +0.00D-6, &
     &               +1.15D-6,          +0.00D-6, &
     &               +1.15D-6,          +0.00D-6, &
     &               -0.95D-6,          +0.20D-6, &
     &               +0.24D-6,          +0.91D-6, &
     &               -1.15D-6,          +0.00D-6, &
     &               -1.12D-6,          +0.00D-6, &
     &               -1.11D-6,          +0.00D-6, &
     &               -1.11D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.95D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  521,  530 ) / &
     &               -1.11D-6,          +0.00D-6, &
     &               +1.11D-6,          +0.00D-6, &
     &               +0.20D-6,          -0.91D-6, &
     &               -0.72D-6,          -0.40D-6, &
     &               -1.11D-6,          +0.00D-6, &
     &               -1.11D-6,          +0.00D-6, &
     &               +1.07D-6,          +0.00D-6, &
     &               -1.07D-6,          +0.00D-6, &
     &               +0.76D-6,          -0.32D-6, &
     &               +0.00D-6,          -1.07D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  531,  540 ) / &
     &               +1.07D-6,          +0.00D-6, &
     &               +1.07D-6,          +0.00D-6, &
     &               -1.07D-6,          +0.00D-6, &
     &               +1.07D-6,          +0.00D-6, &
     &               -0.84D-6,          -0.24D-6, &
     &               +0.00D-6,          -1.03D-6, &
     &               +1.03D-6,          +0.00D-6, &
     &               -1.03D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.80D-6, &
     &               +0.20D-6,          +0.84D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  541,  550 ) / &
     &               -1.03D-6,          +0.00D-6, &
     &               -1.03D-6,          +0.00D-6, &
     &               -0.99D-6,          +0.00D-6, &
     &               +0.24D-6,          +0.76D-6, &
     &               -0.99D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.84D-6, &
     &               -0.99D-6,          +0.00D-6, &
     &               -0.64D-6,          +0.36D-6, &
     &               +0.99D-6,          +0.00D-6, &
     &               +0.36D-6,          -0.64D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  551,  560 ) / &
     &               -0.95D-6,          +0.00D-6, &
     &               -0.95D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.95D-6, &
     &               +0.64D-6,          +0.32D-6, &
     &               +0.00D-6,          -0.95D-6, &
     &               +0.84D-6,          +0.12D-6, &
     &               +0.20D-6,          +0.76D-6, &
     &               -0.95D-6,          +0.00D-6, &
     &               +0.95D-6,          +0.00D-6, &
     &               -0.95D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  561,  570 ) / &
     &               +0.00D-6,          +0.92D-6, &
     &               +0.91D-6,          +0.00D-6, &
     &               +0.91D-6,          +0.00D-6, &
     &               +0.40D-6,          +0.52D-6, &
     &               -0.91D-6,          +0.00D-6, &
     &               -0.56D-6,          +0.36D-6, &
     &               +0.44D-6,          -0.48D-6, &
     &               -0.91D-6,          +0.00D-6, &
     &               -0.91D-6,          +0.00D-6, &
     &               -0.36D-6,          -0.56D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  571,  580 ) / &
     &               +0.91D-6,          +0.00D-6, &
     &               -0.88D-6,          +0.00D-6, &
     &               -0.88D-6,          +0.00D-6, &
     &               +0.60D-6,          -0.28D-6, &
     &               +0.88D-6,          +0.00D-6, &
     &               +0.36D-6,          -0.52D-6, &
     &               -0.52D-6,          +0.36D-6, &
     &               +0.52D-6,          +0.36D-6, &
     &               +0.00D-6,          +0.88D-6, &
     &               +0.56D-6,          +0.32D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  581,  590 ) / &
     &               +0.64D-6,          -0.24D-6, &
     &               +0.88D-6,          +0.00D-6, &
     &               +0.88D-6,          +0.00D-6, &
     &               +0.88D-6,          +0.00D-6, &
     &               +0.84D-6,          +0.00D-6, &
     &               -0.68D-6,          -0.16D-6, &
     &               +0.84D-6,          +0.00D-6, &
     &               +0.56D-6,          +0.28D-6, &
     &               -0.16D-6,          +0.68D-6, &
     &               +0.64D-6,          -0.20D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  591,  600 ) / &
     &               +0.16D-6,          +0.68D-6, &
     &               +0.72D-6,          -0.12D-6, &
     &               -0.83D-6,          +0.00D-6, &
     &               -0.80D-6,          +0.00D-6, &
     &               +0.80D-6,          +0.00D-6, &
     &               -0.80D-6,          +0.00D-6, &
     &               +0.28D-6,          +0.52D-6, &
     &               +0.68D-6,          -0.12D-6, &
     &               +0.00D-6,          -0.80D-6, &
     &               -0.32D-6,          +0.48D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  601,  610 ) / &
     &               +0.36D-6,          -0.44D-6, &
     &               -0.36D-6,          -0.44D-6, &
     &               -0.80D-6,          +0.00D-6, &
     &               +0.79D-6,          +0.00D-6, &
     &               +0.74D-6,          -0.04D-6, &
     &               -0.76D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.76D-6, &
     &               +0.16D-6,          +0.60D-6, &
     &               -0.76D-6,          +0.00D-6, &
     &               -0.76D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  611,  620 ) / &
     &               +0.76D-6,          +0.00D-6, &
     &               -0.76D-6,          +0.00D-6, &
     &               +0.76D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.64D-6, &
     &               +0.76D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.76D-6, &
     &               +0.76D-6,          +0.00D-6, &
     &               +0.64D-6,          -0.12D-6, &
     &               +0.16D-6,          -0.60D-6, &
     &               +0.76D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  621,  630 ) / &
     &               +0.00D-6,          -0.76D-6, &
     &               +0.28D-6,          -0.48D-6, &
     &               +0.32D-6,          +0.44D-6, &
     &               -0.76D-6,          +0.00D-6, &
     &               +0.72D-6,          +0.00D-6, &
     &               +0.72D-6,          +0.00D-6, &
     &               +0.48D-6,          -0.24D-6, &
     &               -0.72D-6,          +0.00D-6, &
     &               +0.72D-6,          +0.00D-6, &
     &               -0.72D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  631,  640 ) / &
     &               -0.72D-6,          +0.00D-6, &
     &               -0.71D-6,          +0.00D-6, &
     &               -0.68D-6,          +0.00D-6, &
     &               -0.68D-6,          +0.00D-6, &
     &               +0.68D-6,          +0.00D-6, &
     &               +0.68D-6,          +0.00D-6, &
     &               +0.68D-6,          +0.00D-6, &
     &               -0.68D-6,          +0.00D-6, &
     &               +0.56D-6,          -0.12D-6, &
     &               -0.68D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  641,  650 ) / &
     &               -0.68D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.48D-6, &
     &               -0.44D-6,          -0.24D-6, &
     &               -0.68D-6,          +0.00D-6, &
     &               +0.64D-6,          +0.00D-6, &
     &               +0.64D-6,          +0.00D-6, &
     &               -0.64D-6,          +0.00D-6, &
     &               +0.64D-6,          +0.00D-6, &
     &               -0.64D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.52D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  651,  660 ) / &
     &               -0.12D-6,          -0.52D-6, &
     &               -0.16D-6,          -0.48D-6, &
     &               -0.20D-6,          -0.44D-6, &
     &               -0.44D-6,          +0.20D-6, &
     &               -0.44D-6,          +0.20D-6, &
     &               +0.24D-6,          -0.40D-6, &
     &               -0.20D-6,          -0.44D-6, &
     &               -0.64D-6,          +0.00D-6, &
     &               +0.40D-6,          -0.24D-6, &
     &               -0.64D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  661,  670 ) / &
     &               +0.64D-6,          +0.00D-6, &
     &               -0.63D-6,          +0.00D-6, &
     &               -0.60D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.60D-6, &
     &               -0.60D-6,          +0.00D-6, &
     &               -0.60D-6,          +0.00D-6, &
     &               +0.60D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.60D-6, &
     &               +0.24D-6,          -0.36D-6, &
     &               +0.12D-6,          +0.48D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  671,  680 ) / &
     &               +0.48D-6,          -0.12D-6, &
     &               +0.12D-6,          +0.48D-6, &
     &               +0.24D-6,          -0.36D-6, &
     &               +0.36D-6,          +0.24D-6, &
     &               +0.12D-6,          +0.48D-6, &
     &               +0.44D-6,          +0.16D-6, &
     &               -0.60D-6,          +0.00D-6, &
     &               -0.60D-6,          +0.00D-6, &
     &               +0.60D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.60D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  681,  690 ) / &
     &               +0.59D-6,          +0.00D-6, &
     &               -0.56D-6,          +0.00D-6, &
     &               -0.44D-6,          -0.12D-6, &
     &               +0.56D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.56D-6, &
     &               -0.56D-6,          +0.00D-6, &
     &               -0.56D-6,          +0.00D-6, &
     &               +0.56D-6,          +0.00D-6, &
     &               -0.56D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.40D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  691,  700 ) / &
     &               +0.44D-6,          -0.12D-6, &
     &               +0.20D-6,          -0.36D-6, &
     &               -0.36D-6,          -0.20D-6, &
     &               -0.56D-6,          +0.00D-6, &
     &               +0.55D-6,          +0.00D-6, &
     &               +0.52D-6,          +0.00D-6, &
     &               -0.52D-6,          +0.00D-6, &
     &               +0.52D-6,          +0.00D-6, &
     &               +0.52D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.36D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  701,  710 ) / &
     &               -0.52D-6,          +0.00D-6, &
     &               -0.52D-6,          +0.00D-6, &
     &               -0.52D-6,          +0.00D-6, &
     &               -0.52D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.52D-6, &
     &               +0.52D-6,          +0.00D-6, &
     &               -0.52D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.40D-6, &
     &               +0.52D-6,          +0.00D-6, &
     &               -0.52D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  711,  720 ) / &
     &               +0.00D-6,          -0.52D-6, &
     &               +0.52D-6,          +0.00D-6, &
     &               +0.52D-6,          +0.00D-6, &
     &               -0.51D-6,          +0.00D-6, &
     &               -0.51D-6,          +0.00D-6, &
     &               +0.48D-6,          +0.00D-6, &
     &               +0.48D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.32D-6, &
     &               -0.48D-6,          +0.00D-6, &
     &               -0.48D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  721,  730 ) / &
     &               +0.48D-6,          +0.00D-6, &
     &               +0.48D-6,          +0.00D-6, &
     &               -0.48D-6,          +0.00D-6, &
     &               -0.12D-6,          -0.36D-6, &
     &               -0.32D-6,          +0.16D-6, &
     &               +0.32D-6,          -0.16D-6, &
     &               -0.12D-6,          -0.36D-6, &
     &               +0.16D-6,          +0.32D-6, &
     &               +0.20D-6,          -0.28D-6, &
     &               -0.20D-6,          -0.28D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  731,  740 ) / &
     &               -0.36D-6,          +0.12D-6, &
     &               -0.48D-6,          +0.00D-6, &
     &               +0.32D-6,          -0.16D-6, &
     &               +0.48D-6,          +0.00D-6, &
     &               -0.48D-6,          +0.00D-6, &
     &               -0.48D-6,          +0.00D-6, &
     &               -0.48D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.48D-6, &
     &               +0.48D-6,          +0.00D-6, &
     &               -0.48D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  741,  750 ) / &
     &               -0.48D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.48D-6, &
     &               +0.44D-6,          +0.00D-6, &
     &               -0.32D-6,          -0.12D-6, &
     &               -0.44D-6,          +0.00D-6, &
     &               +0.20D-6,          -0.24D-6, &
     &               +0.44D-6,          +0.00D-6, &
     &               -0.44D-6,          +0.00D-6, &
     &               +0.44D-6,          +0.00D-6, &
     &               +0.20D-6,          -0.24D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  751,  760 ) / &
     &               +0.12D-6,          +0.32D-6, &
     &               -0.20D-6,          +0.24D-6, &
     &               +0.32D-6,          -0.12D-6, &
     &               +0.00D-6,          +0.44D-6, &
     &               +0.00D-6,          +0.44D-6, &
     &               +0.44D-6,          +0.00D-6, &
     &               -0.44D-6,          +0.00D-6, &
     &               -0.44D-6,          +0.00D-6, &
     &               -0.44D-6,          +0.00D-6, &
     &               +0.44D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  761,  770 ) / &
     &               +0.44D-6,          +0.00D-6, &
     &               +0.40D-6,          +0.00D-6, &
     &               -0.40D-6,          +0.00D-6, &
     &               -0.40D-6,          +0.00D-6, &
     &               -0.40D-6,          +0.00D-6, &
     &               +0.40D-6,          +0.00D-6, &
     &               +0.24D-6,          +0.16D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.12D-6,          +0.28D-6, &
     &               +0.40D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  771,  780 ) / &
     &               -0.40D-6,          +0.00D-6, &
     &               +0.40D-6,          +0.00D-6, &
     &               +0.40D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               -0.40D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.20D-6,          -0.20D-6, &
     &               -0.40D-6,          +0.00D-6, &
     &               -0.40D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  781,  790 ) / &
     &               -0.12D-6,          -0.28D-6, &
     &               +0.40D-6,          +0.00D-6, &
     &               +0.40D-6,          +0.00D-6, &
     &               +0.40D-6,          +0.00D-6, &
     &               +0.40D-6,          +0.00D-6, &
     &               +0.40D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.40D-6, &
     &               -0.20D-6,          -0.16D-6, &
     &               +0.36D-6,          +0.00D-6, &
     &               +0.36D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  791,  800 ) / &
     &               +0.24D-6,          -0.12D-6, &
     &               +0.20D-6,          -0.16D-6, &
     &               +0.00D-6,          +0.36D-6, &
     &               +0.00D-6,          +0.36D-6, &
     &               -0.36D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.24D-6, &
     &               -0.36D-6,          +0.00D-6, &
     &               -0.36D-6,          +0.00D-6, &
     &               -0.36D-6,          +0.00D-6, &
     &               -0.36D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  801,  810 ) / &
     &               +0.36D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.36D-6, &
     &               +0.00D-6,          +0.36D-6, &
     &               +0.00D-6,          +0.36D-6, &
     &               -0.36D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.36D-6, &
     &               +0.12D-6,          -0.24D-6, &
     &               -0.24D-6,          +0.12D-6, &
     &               -0.36D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.36D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  811,  820 ) / &
     &               +0.36D-6,          +0.00D-6, &
     &               +0.24D-6,          -0.12D-6, &
     &               +0.00D-6,          -0.36D-6, &
     &               -0.36D-6,          +0.00D-6, &
     &               +0.36D-6,          +0.00D-6, &
     &               +0.36D-6,          +0.00D-6, &
     &               -0.36D-6,          +0.00D-6, &
     &               +0.36D-6,          +0.00D-6, &
     &               -0.13D-6,          +0.22D-6, &
     &               -0.32D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  821,  830 ) / &
     &               -0.32D-6,          +0.00D-6, &
     &               +0.32D-6,          +0.00D-6, &
     &               -0.20D-6,          -0.12D-6, &
     &               +0.32D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.20D-6, &
     &               -0.32D-6,          +0.00D-6, &
     &               +0.32D-6,          +0.00D-6, &
     &               -0.32D-6,          +0.00D-6, &
     &               -0.32D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.32D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  831,  840 ) / &
     &               +0.32D-6,          +0.00D-6, &
     &               +0.32D-6,          +0.00D-6, &
     &               +0.12D-6,          -0.20D-6, &
     &               -0.32D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.32D-6, &
     &               +0.32D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.32D-6, &
     &               +0.00D-6,          -0.32D-6, &
     &               +0.00D-6,          -0.32D-6, &
     &               +0.32D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  841,  850 ) / &
     &               -0.32D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.32D-6, &
     &               +0.32D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.32D-6, &
     &               +0.00D-6,          -0.32D-6, &
     &               +0.32D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.16D-6, &
     &               -0.16D-6,          +0.16D-6, &
     &               +0.00D-6,          +0.32D-6, &
     &               +0.20D-6,          +0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  851,  860 ) / &
     &               +0.20D-6,          +0.12D-6, &
     &               -0.20D-6,          +0.12D-6, &
     &               +0.12D-6,          +0.20D-6, &
     &               +0.12D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.32D-6, &
     &               -0.32D-6,          +0.00D-6, &
     &               +0.32D-6,          +0.00D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               -0.28D-6,          +0.00D-6, &
     &               +0.28D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  861,  870 ) / &
     &               +0.28D-6,          +0.00D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.12D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               -0.28D-6,          +0.00D-6, &
     &               -0.12D-6,          -0.16D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               -0.28D-6,          +0.00D-6, &
     &               -0.28D-6,          +0.00D-6, &
     &               +0.28D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  871,  880 ) / &
     &               +0.28D-6,          +0.00D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               -0.28D-6,          +0.00D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.28D-6, &
     &               +0.00D-6,          +0.28D-6, &
     &               +0.00D-6,          -0.28D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  881,  890 ) / &
     &               -0.28D-6,          +0.00D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               -0.12D-6,          -0.16D-6, &
     &               +0.00D-6,          +0.28D-6, &
     &               +0.00D-6,          -0.28D-6, &
     &               +0.12D-6,          -0.16D-6, &
     &               -0.28D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.28D-6, &
     &               +0.00D-6,          +0.28D-6, &
     &               +0.00D-6,          -0.28D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  891,  900 ) / &
     &               +0.28D-6,          +0.00D-6, &
     &               -0.28D-6,          +0.00D-6, &
     &               -0.28D-6,          +0.00D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               +0.12D-6,          -0.16D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               -0.28D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.28D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  901,  910 ) / &
     &               +0.00D-6,          -0.28D-6, &
     &               +0.28D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.24D-6, &
     &               +0.24D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               +0.24D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               +0.24D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  911,  920 ) / &
     &               +0.24D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.24D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               +0.24D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               +0.24D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  921,  930 ) / &
     &               +0.24D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.24D-6, &
     &               +0.24D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               +0.24D-6,          +0.00D-6, &
     &               +0.24D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.24D-6, &
     &               +0.12D-6,          -0.12D-6, &
     &               +0.00D-6,          -0.24D-6, &
     &               -0.24D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  931,  940 ) / &
     &               +0.00D-6,          +0.24D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.24D-6, &
     &               +0.24D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               +0.24D-6,          +0.00D-6, &
     &               +0.24D-6,          +0.00D-6, &
     &               +0.24D-6,          +0.00D-6, &
     &               +0.24D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  941,  950 ) / &
     &               +0.24D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               -0.24D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.24D-6, &
     &               +0.00D-6,          -0.24D-6, &
     &               +0.24D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  951,  960 ) / &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  961,  970 ) / &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.20D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  971,  980 ) / &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  981,  990 ) / &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6 /
      DATA ( ( SX0(I,J), I=1,2), J =  991, 1000 ) / &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1001, 1010 ) / &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1011, 1020 ) / &
     &               +0.00D-6,          -0.20D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1021, 1030 ) / &
     &               -0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.16D-6, &
     &               +0.16D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1031, 1040 ) / &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1041, 1050 ) / &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               -0.04D-6,          +0.12D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.16D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.16D-6, &
     &               +0.00D-6,          -0.16D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1051, 1060 ) / &
     &               +0.00D-6,          +0.16D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.16D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.16D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1061, 1070 ) / &
     &               +0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.16D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6 /
!
      DATA ( ( SX0(I,J), I=1,2), J = 1071, 1080 ) / &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.16D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.16D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.16D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.16D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1081, 1090 ) / &
     &               +0.00D-6,          -0.16D-6, &
     &               +0.00D-6,          -0.16D-6, &
     &               +0.00D-6,          +0.16D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.16D-6, &
     &               +0.00D-6,          -0.16D-6, &
     &               +0.00D-6,          +0.16D-6, &
     &               +0.00D-6,          +0.16D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1091, 1100 ) / &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.16D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.16D-6, &
     &               +0.00D-6,          -0.16D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.16D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1101, 1110 ) / &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.16D-6, &
     &               +0.00D-6,          +0.16D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1111, 1120 ) / &
     &               +0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.16D-6, &
     &               -0.16D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.16D-6, &
     &               +0.16D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1121, 1130 ) / &
     &               +0.16D-6,          +0.00D-6, &
     &               -0.15D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1131, 1140 ) / &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1141, 1150 ) / &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1151, 1160 ) / &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1161, 1170 ) / &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1171, 1180 ) / &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.08D-6,          +0.04D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1181, 1190 ) / &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1191, 1200 ) / &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1201, 1210 ) / &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1211, 1220 ) / &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1221, 1230 ) / &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1231, 1240 ) / &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1241, 1250 ) / &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1251, 1260 ) / &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.00D-6,          -0.12D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1261, 1270 ) / &
     &               +0.00D-6,          +0.12D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1271, 1280 ) / &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1281, 1290 ) / &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1291, 1300 ) / &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6 /
      DATA ( ( SX0(I,J), I=1,2), J = 1301,  NX0 ) / &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.11D-6,          +0.00D-6 /
!
!  Sine and cosine coefficients for t^1 terms
      DATA ( ( SX1(I,J), I=1,2), J =    1,   10 ) / &
     &            -3328.48D-6,     +205833.15D-6, &
     &             +197.53D-6,      +12814.01D-6, &
     &              +41.19D-6,       +2187.91D-6, &
     &              -35.85D-6,       -2004.36D-6, &
     &              +59.15D-6,        +501.82D-6, &
     &               -5.82D-6,        +448.76D-6, &
     &             -179.56D-6,        +164.33D-6, &
     &               +5.67D-6,        +288.49D-6, &
     &              +23.85D-6,        -214.50D-6, &
     &               +2.87D-6,        -154.91D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   11,   20 ) / &
     &               +2.14D-6,        -119.21D-6, &
     &               +1.17D-6,         -74.33D-6, &
     &               +1.47D-6,         +70.31D-6, &
     &               -0.42D-6,         +58.94D-6, &
     &               -0.95D-6,         +57.12D-6, &
     &               -1.08D-6,         -54.19D-6, &
     &               +0.92D-6,         +36.78D-6, &
     &               +0.68D-6,         -31.01D-6, &
     &               +0.74D-6,         +29.60D-6, &
     &               -0.60D-6,         -27.59D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   21,   30 ) / &
     &              -11.11D-6,         -15.07D-6, &
     &               -0.40D-6,         -24.05D-6, &
     &               -0.81D-6,         +19.06D-6, &
     &               +3.18D-6,         +15.32D-6, &
     &               -0.08D-6,         -17.90D-6, &
     &               -0.16D-6,         +15.55D-6, &
     &               -0.77D-6,         +14.40D-6, &
     &               -0.25D-6,         +11.67D-6, &
     &               +6.18D-6,          +3.58D-6, &
     &               -1.00D-6,          -7.27D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   31,   40 ) / &
     &               -0.99D-6,          +6.87D-6, &
     &               -0.27D-6,          +7.49D-6, &
     &               -0.30D-6,          +7.31D-6, &
     &               +0.20D-6,          +7.30D-6, &
     &               +0.33D-6,          +6.80D-6, &
     &               +0.27D-6,          -6.81D-6, &
     &               +0.35D-6,          +6.08D-6, &
     &               +0.35D-6,          +6.09D-6, &
     &               -0.14D-6,          -6.19D-6, &
     &               +0.14D-6,          +6.02D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   41,   50 ) / &
     &               +2.69D-6,          -2.76D-6, &
     &               -0.08D-6,          -4.93D-6, &
     &               +2.85D-6,          -1.77D-6, &
     &               -0.07D-6,          -4.27D-6, &
     &               -3.71D-6,          +0.38D-6, &
     &               +3.75D-6,          +0.04D-6, &
     &               -0.82D-6,          -2.73D-6, &
     &               -0.06D-6,          +2.93D-6, &
     &               -0.04D-6,          +2.83D-6, &
     &               +0.08D-6,          +2.75D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   51,   60 ) / &
     &               +0.07D-6,          +2.75D-6, &
     &               -0.07D-6,          +2.70D-6, &
     &               -0.07D-6,          +2.52D-6, &
     &               -0.05D-6,          -2.53D-6, &
     &               -0.04D-6,          +2.40D-6, &
     &               -0.06D-6,          -2.37D-6, &
     &               +0.69D-6,          -1.45D-6, &
     &               -0.04D-6,          +2.00D-6, &
     &               +1.99D-6,          +0.02D-6, &
     &               -0.94D-6,          +1.07D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   61,   70 ) / &
     &               +0.04D-6,          +1.91D-6, &
     &               -0.58D-6,          -1.36D-6, &
     &               -0.51D-6,          -1.25D-6, &
     &               -0.04D-6,          -1.59D-6, &
     &               +0.39D-6,          -1.23D-6, &
     &               +0.03D-6,          -1.57D-6, &
     &               -0.03D-6,          +1.50D-6, &
     &               +0.04D-6,          +1.48D-6, &
     &               -0.04D-6,          +1.45D-6, &
     &               +0.02D-6,          -1.36D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   71,   80 ) / &
     &               +0.03D-6,          -1.32D-6, &
     &               -0.03D-6,          -1.24D-6, &
     &               -0.02D-6,          -1.18D-6, &
     &               -0.03D-6,          +1.16D-6, &
     &               +0.02D-6,          +1.13D-6, &
     &               +0.04D-6,          -1.11D-6, &
     &               +0.02D-6,          +1.11D-6, &
     &               +0.03D-6,          -1.10D-6, &
     &               +0.03D-6,          +1.04D-6, &
     &               -0.51D-6,          +0.56D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   81,   90 ) / &
     &               +0.02D-6,          -0.98D-6, &
     &               -0.02D-6,          -0.94D-6, &
     &               -0.02D-6,          -0.89D-6, &
     &               -0.02D-6,          -0.88D-6, &
     &               +0.31D-6,          +0.60D-6, &
     &               +0.02D-6,          -0.87D-6, &
     &               -0.02D-6,          -0.87D-6, &
     &               -0.01D-6,          +0.83D-6, &
     &               -0.02D-6,          +0.77D-6, &
     &               +0.42D-6,          -0.36D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =   91,  100 ) / &
     &               -0.01D-6,          -0.73D-6, &
     &               +0.01D-6,          +0.71D-6, &
     &               +0.01D-6,          +0.68D-6, &
     &               +0.02D-6,          +0.66D-6, &
     &               -0.01D-6,          -0.62D-6, &
     &               -0.01D-6,          +0.62D-6, &
     &               -0.58D-6,          -0.03D-6, &
     &               -0.01D-6,          +0.58D-6, &
     &               +0.44D-6,          +0.14D-6, &
     &               +0.02D-6,          +0.56D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  101,  110 ) / &
     &               +0.01D-6,          -0.57D-6, &
     &               -0.13D-6,          -0.45D-6, &
     &               +0.01D-6,          +0.56D-6, &
     &               +0.01D-6,          -0.55D-6, &
     &               +0.01D-6,          +0.55D-6, &
     &               -0.52D-6,          +0.03D-6, &
     &               -0.01D-6,          +0.54D-6, &
     &               -0.01D-6,          -0.51D-6, &
     &               -0.41D-6,          -0.11D-6, &
     &               -0.01D-6,          +0.50D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  111,  120 ) / &
     &               +0.01D-6,          +0.48D-6, &
     &               +0.45D-6,          -0.04D-6, &
     &               +0.01D-6,          -0.48D-6, &
     &               +0.01D-6,          +0.46D-6, &
     &               -0.23D-6,          +0.24D-6, &
     &               +0.01D-6,          +0.46D-6, &
     &               +0.35D-6,          -0.11D-6, &
     &               +0.01D-6,          +0.45D-6, &
     &               +0.01D-6,          -0.45D-6, &
     &               +0.00D-6,          -0.45D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  121,  130 ) / &
     &               -0.01D-6,          +0.44D-6, &
     &               +0.35D-6,          +0.09D-6, &
     &               +0.01D-6,          +0.42D-6, &
     &               -0.01D-6,          -0.41D-6, &
     &               +0.09D-6,          -0.33D-6, &
     &               +0.00D-6,          +0.41D-6, &
     &               +0.01D-6,          +0.40D-6, &
     &               -0.01D-6,          -0.39D-6, &
     &               -0.39D-6,          -0.01D-6, &
     &               +0.01D-6,          -0.39D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  131,  140 ) / &
     &               -0.01D-6,          +0.38D-6, &
     &               +0.32D-6,          -0.07D-6, &
     &               -0.01D-6,          +0.36D-6, &
     &               -0.01D-6,          -0.36D-6, &
     &               +0.01D-6,          -0.34D-6, &
     &               +0.01D-6,          -0.34D-6, &
     &               +0.01D-6,          +0.33D-6, &
     &               -0.01D-6,          -0.32D-6, &
     &               +0.01D-6,          +0.32D-6, &
     &               -0.01D-6,          -0.32D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  141,  150 ) / &
     &               -0.01D-6,          -0.31D-6, &
     &               -0.31D-6,          +0.00D-6, &
     &               -0.07D-6,          -0.24D-6, &
     &               +0.10D-6,          -0.21D-6, &
     &               -0.01D-6,          -0.30D-6, &
     &               -0.01D-6,          +0.29D-6, &
     &               -0.01D-6,          -0.29D-6, &
     &               +0.00D-6,          +0.29D-6, &
     &               +0.23D-6,          +0.06D-6, &
     &               +0.26D-6,          +0.02D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  151,  160 ) / &
     &               +0.00D-6,          -0.27D-6, &
     &               +0.25D-6,          +0.02D-6, &
     &               +0.09D-6,          -0.18D-6, &
     &               +0.01D-6,          +0.25D-6, &
     &               +0.14D-6,          -0.11D-6, &
     &               +0.00D-6,          -0.25D-6, &
     &               +0.01D-6,          +0.24D-6, &
     &               -0.01D-6,          -0.24D-6, &
     &               +0.00D-6,          +0.23D-6, &
     &               +0.01D-6,          +0.23D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  161,  170 ) / &
     &               -0.01D-6,          -0.23D-6, &
     &               +0.00D-6,          -0.23D-6, &
     &               +0.00D-6,          -0.22D-6, &
     &               +0.00D-6,          +0.21D-6, &
     &               +0.01D-6,          +0.21D-6, &
     &               -0.17D-6,          +0.03D-6, &
     &               -0.17D-6,          +0.03D-6, &
     &               +0.00D-6,          -0.19D-6, &
     &               +0.14D-6,          -0.06D-6, &
     &               +0.03D-6,          -0.17D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  171,  180 ) / &
     &               -0.13D-6,          +0.06D-6, &
     &               +0.00D-6,          +0.19D-6, &
     &               +0.00D-6,          +0.19D-6, &
     &               -0.06D-6,          -0.13D-6, &
     &               +0.00D-6,          +0.18D-6, &
     &               -0.09D-6,          -0.09D-6, &
     &               +0.10D-6,          -0.09D-6, &
     &               +0.06D-6,          +0.12D-6, &
     &               +0.00D-6,          +0.18D-6, &
     &               +0.00D-6,          -0.18D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  181,  190 ) / &
     &               +0.00D-6,          +0.17D-6, &
     &               -0.03D-6,          +0.15D-6, &
     &               -0.01D-6,          -0.16D-6, &
     &               +0.00D-6,          +0.17D-6, &
     &               +0.00D-6,          -0.17D-6, &
     &               +0.11D-6,          +0.06D-6, &
     &               +0.00D-6,          -0.17D-6, &
     &               -0.08D-6,          +0.09D-6, &
     &               -0.17D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.16D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  191,  200 ) / &
     &               +0.01D-6,          +0.15D-6, &
     &               -0.13D-6,          -0.03D-6, &
     &               +0.00D-6,          +0.15D-6, &
     &               +0.00D-6,          +0.15D-6, &
     &               -0.13D-6,          +0.03D-6, &
     &               +0.10D-6,          -0.06D-6, &
     &               -0.07D-6,          +0.08D-6, &
     &               -0.09D-6,          -0.06D-6, &
     &               +0.00D-6,          +0.15D-6, &
     &               -0.07D-6,          -0.08D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  201,  210 ) / &
     &               +0.00D-6,          -0.14D-6, &
     &               +0.02D-6,          +0.12D-6, &
     &               +0.07D-6,          +0.08D-6, &
     &               -0.03D-6,          -0.11D-6, &
     &               -0.01D-6,          -0.14D-6, &
     &               +0.00D-6,          -0.14D-6, &
     &               +0.02D-6,          -0.12D-6, &
     &               +0.00D-6,          -0.14D-6, &
     &               +0.00D-6,          +0.14D-6, &
     &               +0.00D-6,          +0.14D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  211,  220 ) / &
     &               +0.00D-6,          +0.13D-6, &
     &               +0.08D-6,          -0.06D-6, &
     &               +0.00D-6,          +0.13D-6, &
     &               +0.00D-6,          +0.13D-6, &
     &               +0.01D-6,          +0.13D-6, &
     &               +0.00D-6,          +0.13D-6, &
     &               +0.00D-6,          +0.13D-6, &
     &               -0.02D-6,          -0.11D-6, &
     &               +0.08D-6,          -0.04D-6, &
     &               +0.00D-6,          +0.13D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  221,  230 ) / &
     &               +0.00D-6,          +0.13D-6, &
     &               +0.01D-6,          -0.12D-6, &
     &               +0.00D-6,          +0.12D-6, &
     &               -0.02D-6,          -0.11D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.04D-6,          +0.08D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.00D-6,          +0.12D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  231,  240 ) / &
     &               +0.00D-6,          -0.12D-6, &
     &               +0.00D-6,          -0.11D-6, &
     &               +0.03D-6,          -0.09D-6, &
     &               +0.00D-6,          +0.11D-6, &
     &               -0.11D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.11D-6, &
     &               +0.00D-6,          -0.11D-6, &
     &               +0.07D-6,          +0.05D-6, &
     &               +0.11D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.11D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  241,  250 ) / &
     &               +0.00D-6,          -0.11D-6, &
     &               +0.02D-6,          -0.09D-6, &
     &               +0.00D-6,          +0.11D-6, &
     &               +0.02D-6,          +0.09D-6, &
     &               +0.00D-6,          -0.11D-6, &
     &               +0.00D-6,          +0.11D-6, &
     &               -0.08D-6,          -0.02D-6, &
     &               +0.00D-6,          -0.10D-6, &
     &               +0.00D-6,          -0.10D-6, &
     &               -0.03D-6,          -0.07D-6 /
      DATA ( ( SX1(I,J), I=1,2), J =  251,  NX1 ) / &
     &               +0.00D-6,          +0.10D-6, &
     &               +0.00D-6,          +0.10D-6, &
     &               +0.00D-6,          -0.10D-6 /
!
!  Sine and cosine coefficients for t^2 terms
      DATA ( ( SX2(I,J), I=1,2), J =    1,   10 ) / &
     &            +2038.00D-6,         +82.26D-6, &
     &             +155.75D-6,          -2.70D-6, &
     &              +26.92D-6,          -0.45D-6, &
     &              -24.43D-6,          +0.46D-6, &
     &              -17.36D-6,          -0.50D-6, &
     &               -8.41D-6,          +0.01D-6, &
     &               +6.08D-6,          -1.36D-6, &
     &               +4.59D-6,          +0.17D-6, &
     &               +3.57D-6,          -0.06D-6, &
     &               +2.54D-6,          +0.60D-6 /
      DATA ( ( SX2(I,J), I=1,2), J =   11,   20 ) / &
     &               +1.86D-6,          +0.00D-6, &
     &               -1.52D-6,          -0.07D-6, &
     &               +1.46D-6,          +0.04D-6, &
     &               -0.75D-6,          -0.02D-6, &
     &               -0.75D-6,          +0.00D-6, &
     &               -0.71D-6,          -0.01D-6, &
     &               -0.69D-6,          +0.02D-6, &
     &               +0.61D-6,          +0.02D-6, &
     &               +0.54D-6,          -0.04D-6, &
     &               -0.56D-6,          +0.00D-6 /
      DATA ( ( SX2(I,J), I=1,2), J =   21,   30 ) / &
     &               +0.46D-6,          -0.02D-6, &
     &               +0.38D-6,          -0.01D-6, &
     &               +0.37D-6,          -0.02D-6, &
     &               -0.34D-6,          +0.01D-6, &
     &               -0.35D-6,          +0.00D-6, &
     &               -0.31D-6,          +0.00D-6, &
     &               +0.19D-6,          -0.09D-6, &
     &               +0.26D-6,          +0.00D-6, &
     &               +0.24D-6,          -0.01D-6, &
     &               -0.20D-6,          +0.00D-6 /
      DATA ( ( SX2(I,J), I=1,2), J =   31,  NX2 ) / &
     &               +0.18D-6,          -0.01D-6, &
     &               +0.17D-6,          +0.00D-6, &
     &               +0.15D-6,          +0.01D-6, &
     &               -0.15D-6,          +0.00D-6, &
     &               -0.13D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6 /
!
!  Sine and cosine coefficients for t^3 terms
      DATA ( ( SX3(I,J), I=1,2), J =    1,  NX3 ) / &
     &               +1.76D-6,         -20.39D-6, &
     &               +0.00D-6,          -1.27D-6, &
     &               +0.00D-6,          -0.22D-6, &
     &               +0.00D-6,          +0.20D-6 /
!
!  Sine and cosine coefficients for t^4 terms
      DATA ( ( SX4(I,J), I=1,2), J =    1,  NX4 ) / &
     &               -0.10D-6,          -0.02D-6 /
!
!  ----------------------------------
!  The series for Y: numerical values
!  ----------------------------------
!
!  Polynomial coefficients
      DATA YP /      -6950.78D-6, &
     &              -25381.99D-6, &
     &           -22407250.99D-6, &
     &                1842.28D-6, &
     &                1113.06D-6, &
     &                   0.99D-6 /
!
!  Argument coefficients for t^0 terms
      DATA ( ( KY0(I,J), I=1,14), J =    1,   10 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   11,   20 ) / &
     &  1,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   21,   30 ) / &
     &  1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0, &
     &  0,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   31,   40 ) / &
     &  1,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   41,   50 ) / &
     &  2,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0, -1, &
     &  0,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -1,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   51,   60 ) / &
     &  1,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0, -2, &
     &  1,  0, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   61,   70 ) / &
     &  1, -1,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   71,   80 ) / &
     &  2,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  2, &
     &  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -4,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -1,  2,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   81,   90 ) / &
     &  0,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -1, &
     &  3,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =   91,  100 ) / &
     &  0,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  2, -5,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -8,  3,  0,  0,  0, -2, &
     &  1, -1,  0, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  101,  110 ) / &
     &  1,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  6, -8,  3,  0,  0,  0,  2, &
     &  0,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -2, &
     &  2, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  111,  120 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2, &
     &  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &  1, -1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, -2, &
     &  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0,  0, -2,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  121,  130 ) / &
     &  1,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  3,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1, &
     &  1,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  131,  140 ) / &
     &  1, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  2, &
     &  1,  0, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  2, &
     &  1,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -1,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  141,  150 ) / &
     &  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  0,  0,  0,  0, &
     &  1,  0, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  1, &
     &  2,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  151,  160 ) / &
     &  1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -4,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  161,  170 ) / &
     &  1,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -5,  7,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  2,  0,  0,  0, &
     &  0,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  171,  180 ) / &
     &  1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -2,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  4,  0, -2,  0,  0,  0,  2, &
     &  0,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  2, &
     &  0,  0,  4,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  181,  190 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  8,-11,  0,  0,  0,  0,  0, -2, &
     &  1,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0, -1, &
     &  2,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  191,  200 ) / &
     &  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0, -2, &
     &  1,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  2, &
     &  3,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  2, &
     &  1,  0,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  2, &
     &  0,  0,  2, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  201,  210 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  2, &
     &  1,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0,  3, -8,  3,  0,  0,  0,  0, &
     &  0,  0,  2, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  8,-13,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  1, -1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  211,  220 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  2, &
     &  2,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, -1, &
     &  1,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  221,  230 ) / &
     &  1,  0,  0,  0, -1,  0,-18, 16,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -5,  8, -3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0, -2, &
     &  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0, -8, 13,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0, -2, &
     &  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0, -2,  5,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  231,  240 ) / &
     &  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  2, -5,  0,  0,  0, &
     &  2,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0, &
     &  0,  2,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  241,  250 ) / &
     &  1,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  1,  0,-18, 16,  0,  0,  0,  0,  0,  0, &
     &  2, -1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  251,  260 ) / &
     &  0,  1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  1,  0,  0,  0, &
     &  3,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  1, -2,  0,  0,  0,  0,  0, &
     &  2, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  261,  270 ) / &
     &  0,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, -1, &
     &  2, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  2, &
     &  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  0,  0,  0,  2, &
     &  1,  1, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  271,  280 ) / &
     &  0,  0,  0,  0,  0,  0,  3, -7,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0, -1, &
     &  1, -1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -4,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  281,  290 ) / &
     &  2,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1, &
     &  0,  0,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  291,  300 ) / &
     &  1,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0,  0, -2, &
     &  2,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1, &
     &  1,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  301,  310 ) / &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0, -1, &
     &  0,  0,  2, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  1,  0, -1,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -1 /
      DATA ( ( KY0(I,J), I=1,14), J =  311,  320 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  6,-16,  4,  5,  0,  0, -2, &
     &  2,  0,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0, -1, &
     &  2,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  321,  330 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  8,-10,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0, -1, &
     &  0,  1,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  331,  340 ) / &
     &  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  2, &
     &  1,  0, -2,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  341,  350 ) / &
     &  1, -1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0, -1, &
     &  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  351,  360 ) / &
     &  2, -1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -4,  5,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -4, 10,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  2,  0, &
     &  2,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  4,  0, -3,  0,  0,  0,  2, &
     &  0,  0,  0,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  361,  370 ) / &
     &  2,  0, -4, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0, -1, &
     &  2,  0, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -9, 13,  0,  0,  0,  0,  0, &
     &  0,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  371,  380 ) / &
     &  0,  0,  0,  0,  1,  0,  0, -2,  4,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  2, -3,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  1, &
     &  2,  0,  2,  0,  2,  0,  0,  2,  0, -3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  1 /
      DATA ( ( KY0(I,J), I=1,14), J =  381,  390 ) / &
     &  1,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  2,  0,  0,  0, &
     &  1, -1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  391,  400 ) / &
     &  0,  1,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  9,-11,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  4,  0, -1,  0,  0,  0,  2, &
     &  0,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -8,  3,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0, -2, &
     &  1,  1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  401,  410 ) / &
     &  0,  0,  0,  0,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0, -4,  4,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -1,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  411,  420 ) / &
     &  1,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0, -3,  5,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1, -1,  0, -3,  4,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  8,-14,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  3, -6,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  6,-10,  0,  0,  0,  0,  0, -2 /
      DATA ( ( KY0(I,J), I=1,14), J =  421,  430 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  1, &
     &  3,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -4, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -1,  0,  0,  2, &
     &  0,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  8,-15,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  431,  440 ) / &
     &  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  1, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -4,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  2, &
     &  1,  1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  441,  450 ) / &
     &  2,  0,  0, -2,  1,  0, -6,  8,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  3,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  7, -8,  3,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  1, &
     &  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0, -2 /
      DATA ( ( KY0(I,J), I=1,14), J =  451,  460 ) / &
     &  1, -2,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0, -1,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0, -1,  0,  0,  0,  0, &
     &  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  1, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  461,  470 ) / &
     &  1, -1,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -9, 15,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  7,-10,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  3, -5,  4,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  3, -9,  4,  0,  0,  0,  0, -2, &
     &  1,  0,  4,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -4,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  471,  480 ) / &
     &  2,  0, -2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  481,  490 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0, -2, &
     &  5,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0, -3,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -1,  0,  0,  2, &
     &  1, -1,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  491,  500 ) / &
     &  1, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  7,-11,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  4, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  1 /
      DATA ( ( KY0(I,J), I=1,14), J =  501,  510 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  2, -5,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  5,  0,  0,  2, &
     &  0,  0,  0,  0,  1,  0, -3,  7, -4,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  3, -7,  4,  0,  0,  0,  0,  0, &
     &  0,  1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  511,  520 ) / &
     &  1,  0, -2, -3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0, -6,  8,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  1,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  521,  530 ) / &
     &  0,  0,  0,  0,  1,  0,  3, -5,  0,  2,  0,  0,  0,  0, &
     &  0,  1,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  2, -4,  0, -3,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  9, -9,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0, 11,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  6,-15,  0,  0,  0,  0, -2 /
      DATA ( ( KY0(I,J), I=1,14), J =  531,  540 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0,  2, -5,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0,  1,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -4,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -8,  0,  0,  0,  0, -2, &
     &  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0, -2,  2,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  541,  550 ) / &
     &  2, -1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &  0,  0,  2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1, -1,  0,  0,  0, -2,  0,  0,  0,  0,  0, &
     &  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  2, -4,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  551,  560 ) / &
     &  1,  1,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  5,  0, -2,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0, -1, &
     &  2,  0,  0, -2, -1,  0,  0, -2,  0,  0,  5,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  561,  570 ) / &
     &  2,  0, -1, -1, -1,  0,  0, -1,  0,  3,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -2,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0, -8, 11,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  3,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  571,  580 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -5,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  1, &
     &  4,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  581,  590 ) / &
     &  1,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  3, -5,  0,  0,  0,  0,  0,  0, &
     &  1, -2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  1,  0,  0, -6,  8,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -5,  6,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  591,  600 ) / &
     &  2,  0, -1, -1, -1,  0,  0,  3, -7,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &  0,  1, -2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3, -1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  7,-13,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  3,  0,  0,  0,  2, &
     &  3, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  601,  610 ) / &
     &  3, -1, -2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  6,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  9,-12,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  6,-11,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0, -2, &
     &  0,  0,  2, -2,  1, -1,  0,  2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  2,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  611,  620 ) / &
     &  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  5,-10,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  5,  0, -3,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  2, &
     &  0,  0,  3,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -4,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  621,  630 ) / &
     &  0,  0,  2, -2,  2,  0, -5,  6,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0, -1,  2,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  631,  640 ) / &
     &  4,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -8, 15,  0,  0,  0,  0,  0, &
     &  1,  0,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -2,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -2,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  4,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  641,  650 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  2,  2,  0,  0,  0,  0,  2, &
     &  2, -2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  6,-10,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  6, -5,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  2, &
     &  5,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  651,  660 ) / &
     &  0,  0,  2, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  2,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -8, 11,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  2,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8, -8,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  8,-10,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  1 /
      DATA ( ( KY0(I,J), I=1,14), J =  661,  670 ) / &
     &  0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  2, -6,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -2,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2, &
     &  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  671,  680 ) / &
     &  1, -2, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0, -1,  0,  2,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  681,  690 ) / &
     &  0,  0,  2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0, -1,  0, -2,  5,  0,  0,  0, &
     &  2,  0, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2, -1,  0, -5,  6,  0,  0,  0,  0,  0,  0, &
     &  2, -2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1, -1,  0,  0, -1,  0, -1,  0,  0,  0,  0, &
     &  1, -2,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  691,  700 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1,  0,  0,  0, &
     &  2,  0,  0, -3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -6,  8,  0,  0,  0,  0,  0, &
     &  2,  0, -1, -1,  1,  0,  0,  3, -7,  0,  0,  0,  0,  0, &
     &  2,  1,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0, &
     &  2, -1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0, -1,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  701,  710 ) / &
     &  0,  0,  2,  0,  2,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &  0,  1,  0, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -9, 17,  0,  0,  0,  0,  0, &
     &  1,  1,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -2,  0,-10,  3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -7,  9,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  1, -4,  0,  0,  0,  0,  0, -2 /
      DATA ( ( KY0(I,J), I=1,14), J =  711,  720 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  6,-10,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -6,  0,  0,  0,  0, -2, &
     &  4,  0,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -2,  0,  4, -5,  0,  0,  0, &
     &  2,  0,  0, -2, -2,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -1, -1, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  721,  730 ) / &
     &  1,  0,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  1,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  1, -1,  0,-18, 17,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1, -1, -1,  0, 20,-20,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -2,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  731,  740 ) / &
     &  0,  0,  1, -1,  1,  0,  1, -2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  9,-11,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  8,-16,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  4,  0,  0, -2,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -8,  1,  5,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -2,  0,  0,  2 /
      DATA ( ( KY0(I,J), I=1,14), J =  741,  750 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -6,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  2, &
     &  1,  0,  0, -1, -1,  0,  0, -2,  2,  0,  0,  0,  0,  0, &
     &  2, -1,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  751,  760 ) / &
     &  2,  0, -1, -1, -1,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &  1,  0,  0, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  5, -8,  0,  0,  0,  0,  0,  0, &
     &  1, -2, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  761,  770 ) / &
     &  0,  0,  1, -1,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &  1, -2,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -4, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0,  0, -2,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  771,  780 ) / &
     &  0,  0,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  1,  0, -2,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  781,  790 ) / &
     &  0,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  1,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  791,  800 ) / &
     &  1,  0, -2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &  2,  0,  0, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  1,  0,  0, -5,  6,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  801,  810 ) / &
     &  1,  0,  0, -2, -1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  2,  0, -4,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &  2, -1,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1, -1,  0,  0, -3,  4,  0,  0,  0,  0,  0, &
     &  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -2,  0,  3, -1,  0,  0,  0, &
     &  1, -1, -4,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  811,  820 ) / &
     &  3,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -6,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -1, -1, -2,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &  4,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  821,  830 ) / &
     &  3,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0, -2, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  2, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -2,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  831,  840 ) / &
     &  1,  1, -2, -3, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2, -6, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  3,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  1, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  0, -1,  0, -3,  5,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -2,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  1, -1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  841,  850 ) / &
     &  0,  3, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -3,  0,  3,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0, -4,  4,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0, -5,  5,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  1, -3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0,  1, -4,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  1, -3,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  851,  860 ) / &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0, -1,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -4,  6,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-12,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  7,-10,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  7,-11,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  6, -4,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0, -1 /
      DATA ( ( KY0(I,J), I=1,14), J =  861,  870 ) / &
     &  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  3, -8,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  1, -4,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  9,-17,  0,  0,  0,  0, -2 /
      DATA ( ( KY0(I,J), I=1,14), J =  871,  880 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  7, -8,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  5,-10,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  4,  0, -4,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0, -1 /
      DATA ( ( KY0(I,J), I=1,14), J =  881,  890 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  4, -8,  1,  5,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -8,  3,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -2,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -5,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -7,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  1, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0, -1 /
      DATA ( ( KY0(I,J), I=1,14), J =  891,  900 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  5,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -5,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  2, &
     &  1,  0,  4,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  4,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  7,-13,  0,  0,  0,  0,  0, &
     &  2,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  901,  910 ) / &
     &  1, -1,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0, -8, 11,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -6,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  1,  2,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0, -2, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1,  1,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  911,  920 ) / &
     &  2, -1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -4, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -2,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -6, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -2,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  921,  930 ) / &
     &  3,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2, -3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0, -5,  7,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0,  0, -1,  0, -1,  0,  0,  0,  0, &
     &  2,  1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  3, -4,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  931,  940 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &  0,  1,  2, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  2,  0, -8, 12,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  941,  950 ) / &
     &  0,  1, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0, -1,  0,-10,  3,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  1,  0,-10,  3,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  4, -4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  0,  0,  1,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  951,  960 ) / &
     &  3,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -1, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  2,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  2, -3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0, -2,  3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0, -2,  2,  0,  0,  0,  0,  0,  0, &
     &  2, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY0(I,J), I=1,14), J =  961,  NY0 ) / &
     &  2,  0, -2, -2, -2,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  1,  2,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
!
!  Argument coefficients for t^1 terms
      DATA ( ( KY1(I,J), I=1,14), J =    1,   10 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   11,   20 ) / &
     &  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   21,   30 ) / &
     &  0,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   31,   40 ) / &
     &  1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0, &
     &  1,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   41,   50 ) / &
     &  0,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   51,   60 ) / &
     &  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0, -1, &
     &  0,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   61,   70 ) / &
     &  1, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   71,   80 ) / &
     &  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0, -2, &
     &  1,  0, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -1,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   81,   90 ) / &
     &  1,  0,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &  2,  0, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =   91,  100 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  2, &
     &  1,  0, -4,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  101,  110 ) / &
     &  0,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -1,  2,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  111,  120 ) / &
     &  0,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  121,  130 ) / &
     &  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -1, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  131,  140 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  2, -8,  3,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  6, -8,  3,  0,  0,  0,  2, &
     &  0,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2, &
     &  1, -1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  2, -5,  0,  0,  0, &
     &  3,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  141,  150 ) / &
     &  2, -1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  2, -2,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0, -2,  0,  2,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  151,  160 ) / &
     &  0,  3,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &  1, -1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &  1,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  161,  170 ) / &
     &  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  2, &
     &  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0,  0, -2,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  0,  0,  0,  0, -2,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  171,  180 ) / &
     &  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0, &
     &  1,  0,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,-18, 16,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  181,  190 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1, &
     &  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,-10,  3,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2, -1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  191,  200 ) / &
     &  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, -2, &
     &  1,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  2, &
     &  4,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -1,  0,  0,  0, &
     &  1,  0, -2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  4, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  201,  210 ) / &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  0,  0, -5,  6,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  1, &
     &  2,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -2, &
     &  0,  2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  211,  220 ) / &
     &  2,  0, -2, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0, -2, &
     &  0,  1,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -4,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -2,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &  0,  0,  2, -3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  4,  0, -2,  0,  0,  0,  2 /
      DATA ( ( KY1(I,J), I=1,14), J =  221,  230 ) / &
     &  0,  0,  4,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  2, &
     &  1,  1, -2, -4, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0,  0,  2, &
     &  2,  0,  0, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-11,  0,  0,  0,  0,  0, -2, &
     &  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0, -2 /
      DATA ( ( KY1(I,J), I=1,14), J =  231,  240 ) / &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  2, &
     &  0,  0,  1, -1,  1,  0, -5,  7,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1, &
     &  1,  0,  2,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  0,  0, 19,-21,  3,  0,  0,  0,  0,  0, &
     &  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  2,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  241,  250 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0, -2, &
     &  0,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  2, &
     &  3,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  251,  260 ) / &
     &  0,  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  2, &
     &  1,  1,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  2, &
     &  2,  1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -1,  0,  0, -3,  4,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0, -1 /
      DATA ( ( KY1(I,J), I=1,14), J =  261,  270 ) / &
     &  0,  2, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -4, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  3,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0, -2, &
     &  0,  0,  2, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0, -2, &
     &  2,  1,  0, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY1(I,J), I=1,14), J =  271,  NY1 ) / &
     &  0,  0,  1, -1,  1,  0,  0,  3, -8,  3,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  2, &
     &  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0,  0,  0, -2, &
     &  4,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1, -1,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0, -2 /
!
!  Argument coefficients for t^2 terms
      DATA ( ( KY2(I,J), I=1,14), J =    1,   10 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY2(I,J), I=1,14), J =   11,   20 ) / &
     &  1,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KY2(I,J), I=1,14), J =   21,  NY2 ) / &
     &  1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  2,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0, &
     &  0,  1,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
!
!  Argument coefficients for t^3 terms
      DATA ( ( KY3(I,J), I=1,14), J =    1,  NY3 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
!
!  Argument coefficients for t^4 terms
      DATA ( ( KY4(I,J), I=1,14), J =    1,  NY4 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
!
!  Sine and cosine coefficients for t^0 terms
      DATA ( ( SY0(I,J), I=1,2), J =    1,   10 ) / &
     &            +1538.18D-6,    +9205236.26D-6, &
     &             -458.66D-6,     +573033.42D-6, &
     &             +137.41D-6,      +97846.69D-6, &
     &              -29.05D-6,      -89618.24D-6, &
     &              -17.40D-6,      +22438.42D-6, &
     &              +31.80D-6,      +20069.50D-6, &
     &              +36.70D-6,      +12902.66D-6, &
     &              -13.20D-6,       -9592.72D-6, &
     &             -192.40D-6,       +7387.02D-6, &
     &               +3.92D-6,       -6918.22D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   11,   20 ) / &
     &               +0.40D-6,       -5331.13D-6, &
     &               -0.90D-6,       -3323.89D-6, &
     &               +7.50D-6,       +3143.98D-6, &
     &               +7.80D-6,       +2636.13D-6, &
     &               -6.60D-6,       +2554.51D-6, &
     &               -2.00D-6,       -2423.59D-6, &
     &               +6.80D-6,       +1645.01D-6, &
     &               +0.00D-6,       -1387.00D-6, &
     &               +5.90D-6,       +1323.81D-6, &
     &               -0.30D-6,       -1233.89D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   21,   30 ) / &
     &               +0.30D-6,       -1075.60D-6, &
     &               -4.48D-6,        +852.85D-6, &
     &               +0.10D-6,        -800.34D-6, &
     &              +35.80D-6,        -674.99D-6, &
     &               -1.40D-6,        +695.54D-6, &
     &               -0.50D-6,        +684.99D-6, &
     &               -2.62D-6,        +643.75D-6, &
     &               -1.50D-6,        +522.11D-6, &
     &             +273.50D-6,        +164.70D-6, &
     &               +1.40D-6,        +335.24D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   31,   40 ) / &
     &               +1.90D-6,        +326.60D-6, &
     &               +0.40D-6,        +327.11D-6, &
     &               -0.50D-6,        -325.03D-6, &
     &               -0.40D-6,        +307.03D-6, &
     &               +0.50D-6,        +304.17D-6, &
     &               -0.10D-6,        -304.46D-6, &
     &               -0.40D-6,        -276.81D-6, &
     &               +0.90D-6,        +272.05D-6, &
     &               +0.30D-6,        +272.22D-6, &
     &               +1.20D-6,        +269.45D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   41,   50 ) / &
     &               +0.10D-6,        -220.67D-6, &
     &             +128.60D-6,         -77.10D-6, &
     &               +0.10D-6,        -190.79D-6, &
     &             +167.90D-6,          +0.00D-6, &
     &               -8.20D-6,        -123.48D-6, &
     &               +0.10D-6,        +131.04D-6, &
     &               +0.40D-6,        +126.64D-6, &
     &               +2.90D-6,        -122.28D-6, &
     &               +0.70D-6,        +123.20D-6, &
     &               +0.40D-6,        +123.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   51,   60 ) / &
     &               -0.30D-6,        +120.70D-6, &
     &               -0.50D-6,        +112.90D-6, &
     &               -0.20D-6,        -112.94D-6, &
     &               +0.20D-6,        +107.31D-6, &
     &               -0.30D-6,        -106.20D-6, &
     &              +31.90D-6,         -64.10D-6, &
     &               +0.00D-6,         +89.50D-6, &
     &              +89.10D-6,          +0.00D-6, &
     &               +0.00D-6,         +85.32D-6, &
     &               -0.20D-6,         -71.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   61,   70 ) / &
     &               +0.00D-6,         -70.01D-6, &
     &              +13.90D-6,         -55.30D-6, &
     &               +0.00D-6,         +67.25D-6, &
     &               +0.40D-6,         +66.29D-6, &
     &               -0.40D-6,         +64.70D-6, &
     &               +1.30D-6,         -60.90D-6, &
     &               -0.20D-6,         -60.92D-6, &
     &               +0.20D-6,         -59.20D-6, &
     &               +1.10D-6,         -55.55D-6, &
     &               +0.00D-6,         -55.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   71,   80 ) / &
     &               -0.10D-6,         -52.69D-6, &
     &               -0.20D-6,         +51.80D-6, &
     &               +1.00D-6,         -49.51D-6, &
     &               +0.00D-6,         +50.50D-6, &
     &               +2.50D-6,         +47.70D-6, &
     &               +0.10D-6,         +49.59D-6, &
     &               +0.10D-6,         -49.00D-6, &
     &              -23.20D-6,         +24.60D-6, &
     &               +0.40D-6,         +46.50D-6, &
     &               -0.10D-6,         -44.04D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   81,   90 ) / &
     &               -0.10D-6,         -42.19D-6, &
     &              +13.30D-6,         +26.90D-6, &
     &               -0.10D-6,         -39.90D-6, &
     &               -0.10D-6,         -39.50D-6, &
     &               +0.00D-6,         -39.11D-6, &
     &               -0.10D-6,         -38.92D-6, &
     &               +0.10D-6,         +36.95D-6, &
     &               -0.10D-6,         +34.59D-6, &
     &               +0.20D-6,         -32.55D-6, &
     &               -0.10D-6,         +31.61D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =   91,  100 ) / &
     &               +0.00D-6,         +30.40D-6, &
     &               +0.20D-6,         +29.40D-6, &
     &               +0.00D-6,         -27.91D-6, &
     &               +0.10D-6,         +27.50D-6, &
     &              -25.70D-6,          -1.70D-6, &
     &              +19.90D-6,          +5.90D-6, &
     &               +0.00D-6,         +25.80D-6, &
     &               +0.20D-6,         +25.20D-6, &
     &               +0.00D-6,         -25.31D-6, &
     &               +0.20D-6,         +25.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  101,  110 ) / &
     &               -0.10D-6,         +24.40D-6, &
     &               +0.10D-6,         -24.40D-6, &
     &              -23.30D-6,          +0.90D-6, &
     &               -0.10D-6,         +24.00D-6, &
     &              -18.00D-6,          -5.30D-6, &
     &               -0.10D-6,         -22.80D-6, &
     &               -0.10D-6,         +22.50D-6, &
     &               +0.10D-6,         +21.60D-6, &
     &               +0.00D-6,         -21.30D-6, &
     &               +0.10D-6,         +20.70D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  111,  120 ) / &
     &               +0.70D-6,         -20.10D-6, &
     &               +0.00D-6,         +20.51D-6, &
     &              +15.90D-6,          -4.50D-6, &
     &               +0.20D-6,         -19.94D-6, &
     &               +0.00D-6,         +20.11D-6, &
     &              +15.60D-6,          +4.40D-6, &
     &               +0.00D-6,         -20.00D-6, &
     &               +0.00D-6,         +19.80D-6, &
     &               +0.00D-6,         +18.91D-6, &
     &               +4.30D-6,         -14.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  121,  130 ) / &
     &               -0.10D-6,         -18.50D-6, &
     &               -0.10D-6,         +18.40D-6, &
     &               +0.00D-6,         +18.10D-6, &
     &               +1.00D-6,         +16.81D-6, &
     &               -0.10D-6,         -17.60D-6, &
     &              -17.60D-6,          +0.00D-6, &
     &               -1.30D-6,         -16.26D-6, &
     &               +0.00D-6,         -17.41D-6, &
     &              +14.50D-6,          -2.70D-6, &
     &               +0.00D-6,         +17.08D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  131,  140 ) / &
     &               +0.00D-6,         +16.21D-6, &
     &               +0.00D-6,         -16.00D-6, &
     &               +0.00D-6,         -15.31D-6, &
     &               +0.00D-6,         -15.10D-6, &
     &               +0.00D-6,         +14.70D-6, &
     &               +0.00D-6,         +14.40D-6, &
     &               -0.10D-6,         -14.30D-6, &
     &               +0.00D-6,         -14.40D-6, &
     &               +0.00D-6,         -13.81D-6, &
     &               +4.50D-6,          -9.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  141,  150 ) / &
     &              -13.80D-6,          +0.00D-6, &
     &               +0.00D-6,         -13.38D-6, &
     &               -0.10D-6,         +13.10D-6, &
     &              +10.30D-6,          +2.70D-6, &
     &               +0.00D-6,         +12.80D-6, &
     &               +0.00D-6,         -12.80D-6, &
     &              +11.70D-6,          +0.80D-6, &
     &               +0.00D-6,         -12.00D-6, &
     &              +11.30D-6,          +0.50D-6, &
     &               +0.00D-6,         +11.40D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  151,  160 ) / &
     &               +0.00D-6,         -11.20D-6, &
     &               +0.10D-6,         +10.90D-6, &
     &               +0.10D-6,         -10.77D-6, &
     &               +0.00D-6,         -10.80D-6, &
     &               -0.20D-6,         +10.47D-6, &
     &               +0.00D-6,         +10.50D-6, &
     &               +0.00D-6,         -10.40D-6, &
     &               +0.00D-6,         +10.40D-6, &
     &               +0.00D-6,         -10.20D-6, &
     &               +0.00D-6,         -10.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  161,  170 ) / &
     &               +0.00D-6,          +9.60D-6, &
     &               +0.10D-6,          +9.40D-6, &
     &               -7.60D-6,          +1.70D-6, &
     &               -7.70D-6,          +1.40D-6, &
     &               +1.40D-6,          -7.50D-6, &
     &               +6.10D-6,          -2.70D-6, &
     &               +0.00D-6,          -8.70D-6, &
     &               -5.90D-6,          +2.60D-6, &
     &               +0.00D-6,          +8.40D-6, &
     &               -0.20D-6,          -8.11D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  171,  180 ) / &
     &               -2.60D-6,          -5.70D-6, &
     &               +0.00D-6,          +8.30D-6, &
     &               +2.70D-6,          +5.50D-6, &
     &               +4.20D-6,          -4.00D-6, &
     &               -0.10D-6,          +8.00D-6, &
     &               +0.00D-6,          +8.09D-6, &
     &               -1.30D-6,          +6.70D-6, &
     &               +0.00D-6,          -7.90D-6, &
     &               +0.00D-6,          +7.80D-6, &
     &               -7.50D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  181,  190 ) / &
     &               -0.50D-6,          -7.20D-6, &
     &               +4.90D-6,          +2.70D-6, &
     &               +0.00D-6,          +7.50D-6, &
     &               +0.00D-6,          -7.50D-6, &
     &               +0.00D-6,          -7.49D-6, &
     &               +0.00D-6,          -7.20D-6, &
     &               +0.10D-6,          +6.90D-6, &
     &               -5.60D-6,          +1.40D-6, &
     &               -5.70D-6,          -1.30D-6, &
     &               +0.00D-6,          +6.90D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  191,  200 ) / &
     &               +4.20D-6,          -2.70D-6, &
     &               +0.00D-6,          +6.90D-6, &
     &               -3.10D-6,          +3.70D-6, &
     &               -3.90D-6,          -2.90D-6, &
     &               +0.00D-6,          +6.60D-6, &
     &               -3.10D-6,          -3.50D-6, &
     &               +1.10D-6,          -5.39D-6, &
     &               +0.00D-6,          -6.40D-6, &
     &               +0.90D-6,          +5.50D-6, &
     &               +0.00D-6,          -6.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  201,  210 ) / &
     &               -0.10D-6,          -6.20D-6, &
     &               +0.00D-6,          -6.10D-6, &
     &               +0.00D-6,          +6.10D-6, &
     &               +0.00D-6,          +6.10D-6, &
     &               +3.50D-6,          -2.50D-6, &
     &               +0.00D-6,          +6.00D-6, &
     &               +0.00D-6,          +5.90D-6, &
     &               -0.90D-6,          -4.80D-6, &
     &               +0.00D-6,          +5.70D-6, &
     &               +0.10D-6,          +5.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  211,  220 ) / &
     &               +0.00D-6,          +5.70D-6, &
     &               +0.00D-6,          +5.70D-6, &
     &               +0.00D-6,          +5.60D-6, &
     &               +0.00D-6,          +5.60D-6, &
     &               +0.20D-6,          -5.40D-6, &
     &               -0.90D-6,          -4.70D-6, &
     &               -0.40D-6,          -5.10D-6, &
     &               +0.00D-6,          +5.50D-6, &
     &               +0.00D-6,          -5.40D-6, &
     &               +0.00D-6,          -5.40D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  221,  230 ) / &
     &               +1.80D-6,          +3.60D-6, &
     &               +0.00D-6,          +5.30D-6, &
     &               +0.00D-6,          -5.30D-6, &
     &               +0.00D-6,          -5.20D-6, &
     &               +0.00D-6,          -5.19D-6, &
     &               +3.00D-6,          +2.10D-6, &
     &               +0.00D-6,          -5.10D-6, &
     &               +0.00D-6,          +5.07D-6, &
     &               +0.90D-6,          -4.10D-6, &
     &               -5.00D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  231,  240 ) / &
     &               +0.00D-6,          -5.00D-6, &
     &               +0.00D-6,          +5.00D-6, &
     &               +0.00D-6,          -5.00D-6, &
     &               +0.00D-6,          -4.90D-6, &
     &               +4.90D-6,          +0.00D-6, &
     &               +0.00D-6,          -4.90D-6, &
     &               +0.90D-6,          +3.90D-6, &
     &               +0.00D-6,          +4.80D-6, &
     &               -3.70D-6,          -1.10D-6, &
     &               +0.00D-6,          -4.72D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  241,  250 ) / &
     &               +0.00D-6,          +4.71D-6, &
     &               +0.00D-6,          -4.50D-6, &
     &               -1.50D-6,          -3.00D-6, &
     &               +0.00D-6,          -4.50D-6, &
     &               +0.30D-6,          -4.11D-6, &
     &               +0.00D-6,          +4.40D-6, &
     &               +0.00D-6,          -4.40D-6, &
     &               +0.00D-6,          +4.39D-6, &
     &               +0.00D-6,          -4.30D-6, &
     &               +0.00D-6,          +4.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  251,  260 ) / &
     &               +0.00D-6,          -4.30D-6, &
     &               +0.20D-6,          +4.03D-6, &
     &               +0.20D-6,          +4.00D-6, &
     &               -0.60D-6,          +3.50D-6, &
     &               +0.00D-6,          +4.10D-6, &
     &               +0.00D-6,          +4.00D-6, &
     &               +0.00D-6,          -4.00D-6, &
     &               +0.00D-6,          -3.91D-6, &
     &               +1.90D-6,          +2.00D-6, &
     &               +0.00D-6,          +3.90D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  261,  270 ) / &
     &               +0.00D-6,          +3.90D-6, &
     &               +0.00D-6,          -3.90D-6, &
     &               +3.10D-6,          -0.80D-6, &
     &               +0.00D-6,          +3.90D-6, &
     &               +0.00D-6,          +3.90D-6, &
     &               +0.00D-6,          +3.80D-6, &
     &               -0.20D-6,          +3.51D-6, &
     &               +0.00D-6,          -3.60D-6, &
     &               -2.10D-6,          +1.50D-6, &
     &               +0.00D-6,          -3.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  271,  280 ) / &
     &               +0.70D-6,          +2.80D-6, &
     &               -2.80D-6,          +0.70D-6, &
     &               +0.00D-6,          -3.50D-6, &
     &               -2.90D-6,          -0.60D-6, &
     &               +0.00D-6,          -3.40D-6, &
     &               +0.00D-6,          +3.40D-6, &
     &               +0.00D-6,          +3.36D-6, &
     &               +0.50D-6,          +2.80D-6, &
     &               +2.60D-6,          -0.70D-6, &
     &               +1.00D-6,          -2.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  281,  290 ) / &
     &               +0.00D-6,          -3.30D-6, &
     &               +0.00D-6,          +3.30D-6, &
     &               +0.00D-6,          +3.23D-6, &
     &               +0.00D-6,          +3.20D-6, &
     &               +0.00D-6,          -3.20D-6, &
     &               +0.00D-6,          -3.20D-6, &
     &               +0.00D-6,          +3.20D-6, &
     &               +2.90D-6,          -0.30D-6, &
     &               +0.08D-6,          +3.05D-6, &
     &               -0.70D-6,          -2.40D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  291,  300 ) / &
     &               +0.00D-6,          -3.08D-6, &
     &               +0.00D-6,          +3.00D-6, &
     &               -1.60D-6,          +1.40D-6, &
     &               -2.90D-6,          -0.10D-6, &
     &               +0.00D-6,          -2.90D-6, &
     &               -2.50D-6,          +0.40D-6, &
     &               +0.40D-6,          -2.50D-6, &
     &               +0.00D-6,          -2.90D-6, &
     &               +0.00D-6,          +2.89D-6, &
     &               +0.00D-6,          -2.80D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  301,  310 ) / &
     &               -2.50D-6,          +0.30D-6, &
     &               -2.50D-6,          -0.30D-6, &
     &               +0.00D-6,          -2.70D-6, &
     &               +2.70D-6,          +0.00D-6, &
     &               +0.00D-6,          -2.60D-6, &
     &               +0.00D-6,          -2.60D-6, &
     &               +0.00D-6,          +2.60D-6, &
     &               +2.10D-6,          +0.50D-6, &
     &               +0.00D-6,          +2.50D-6, &
     &               +0.80D-6,          +1.70D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  311,  320 ) / &
     &               +1.90D-6,          -0.60D-6, &
     &               +0.00D-6,          -2.50D-6, &
     &               +0.00D-6,          -2.40D-6, &
     &               +0.00D-6,          +2.40D-6, &
     &               +0.00D-6,          -2.40D-6, &
     &               +0.00D-6,          +2.40D-6, &
     &               -1.90D-6,          +0.50D-6, &
     &               -0.10D-6,          -2.30D-6, &
     &               +0.00D-6,          +2.30D-6, &
     &               +0.00D-6,          -2.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  321,  330 ) / &
     &               -1.40D-6,          +0.90D-6, &
     &               -0.10D-6,          -2.20D-6, &
     &               -0.20D-6,          -2.00D-6, &
     &               +0.00D-6,          +2.20D-6, &
     &               +0.00D-6,          -2.20D-6, &
     &               +0.00D-6,          +2.20D-6, &
     &               +0.00D-6,          +2.20D-6, &
     &               -1.80D-6,          -0.40D-6, &
     &               +0.00D-6,          +2.20D-6, &
     &               +0.00D-6,          +2.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  331,  340 ) / &
     &               -1.70D-6,          +0.40D-6, &
     &               -0.80D-6,          -1.30D-6, &
     &               -1.30D-6,          -0.80D-6, &
     &               +0.00D-6,          +2.10D-6, &
     &               +0.00D-6,          +2.10D-6, &
     &               +0.00D-6,          -2.10D-6, &
     &               +0.00D-6,          -2.10D-6, &
     &               +0.00D-6,          +2.10D-6, &
     &               +0.00D-6,          -2.00D-6, &
     &               +0.00D-6,          +2.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  341,  350 ) / &
     &               +0.00D-6,          +2.00D-6, &
     &               +0.00D-6,          +2.00D-6, &
     &               +0.00D-6,          -2.00D-6, &
     &               +2.00D-6,          +0.00D-6, &
     &               +1.10D-6,          -0.90D-6, &
     &               +1.60D-6,          -0.40D-6, &
     &               +0.00D-6,          -1.91D-6, &
     &               +0.00D-6,          -1.90D-6, &
     &               +0.00D-6,          +1.90D-6, &
     &               +0.00D-6,          -1.90D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  351,  360 ) / &
     &               +0.00D-6,          +1.90D-6, &
     &               +1.50D-6,          +0.40D-6, &
     &               -1.50D-6,          -0.40D-6, &
     &               -1.40D-6,          -0.50D-6, &
     &               -1.00D-6,          +0.90D-6, &
     &               +0.00D-6,          -1.90D-6, &
     &               -0.30D-6,          +1.60D-6, &
     &               +0.00D-6,          +1.90D-6, &
     &               +0.00D-6,          +1.90D-6, &
     &               +0.00D-6,          -1.80D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  361,  370 ) / &
     &               +0.00D-6,          -1.80D-6, &
     &               -1.10D-6,          +0.70D-6, &
     &               +0.20D-6,          -1.60D-6, &
     &               +0.00D-6,          +1.80D-6, &
     &               +0.00D-6,          -1.71D-6, &
     &               -1.20D-6,          -0.50D-6, &
     &               +1.50D-6,          +0.20D-6, &
     &               -0.60D-6,          -1.10D-6, &
     &               +0.60D-6,          +1.10D-6, &
     &               -0.60D-6,          -1.10D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  371,  380 ) / &
     &               -1.10D-6,          +0.60D-6, &
     &               -1.70D-6,          +0.00D-6, &
     &               +0.00D-6,          +1.60D-6, &
     &               +0.00D-6,          -1.60D-6, &
     &               +0.00D-6,          -1.60D-6, &
     &               +1.20D-6,          -0.40D-6, &
     &               -0.50D-6,          -1.10D-6, &
     &               +0.60D-6,          +1.00D-6, &
     &               -1.30D-6,          -0.30D-6, &
     &               +0.30D-6,          -1.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  381,  390 ) / &
     &               +0.00D-6,          +1.60D-6, &
     &               +0.00D-6,          -1.60D-6, &
     &               +0.00D-6,          -1.60D-6, &
     &               +1.10D-6,          -0.50D-6, &
     &               +0.00D-6,          -1.50D-6, &
     &               +0.00D-6,          -1.50D-6, &
     &               +0.00D-6,          +1.50D-6, &
     &               +0.00D-6,          -1.50D-6, &
     &               +0.00D-6,          -1.50D-6, &
     &               +1.50D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  391,  400 ) / &
     &               +0.00D-6,          -1.50D-6, &
     &               +1.30D-6,          -0.20D-6, &
     &               +0.00D-6,          -1.50D-6, &
     &               -1.20D-6,          -0.30D-6, &
     &               -1.40D-6,          +0.10D-6, &
     &               -0.50D-6,          +1.00D-6, &
     &               -0.50D-6,          +1.00D-6, &
     &               +0.20D-6,          -1.30D-6, &
     &               +0.00D-6,          +1.50D-6, &
     &               +0.00D-6,          +1.50D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  401,  410 ) / &
     &               +0.00D-6,          +1.50D-6, &
     &               +0.00D-6,          +1.49D-6, &
     &               +0.00D-6,          -1.41D-6, &
     &               +0.00D-6,          +1.41D-6, &
     &               +0.00D-6,          -1.40D-6, &
     &               +0.00D-6,          -1.40D-6, &
     &               +0.00D-6,          +1.40D-6, &
     &               +0.00D-6,          -1.40D-6, &
     &               +1.10D-6,          -0.30D-6, &
     &               +0.00D-6,          -1.40D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  411,  420 ) / &
     &               +0.00D-6,          +1.40D-6, &
     &               +1.40D-6,          +0.00D-6, &
     &               -0.30D-6,          +1.10D-6, &
     &               +0.20D-6,          +1.20D-6, &
     &               -1.30D-6,          +0.00D-6, &
     &               +0.00D-6,          -1.30D-6, &
     &               +0.00D-6,          +1.30D-6, &
     &               -0.70D-6,          -0.60D-6, &
     &               -0.80D-6,          +0.50D-6, &
     &               -0.20D-6,          -1.10D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  421,  430 ) / &
!
     &               +1.10D-6,          +0.20D-6, &
     &               +0.00D-6,          -1.30D-6, &
     &               +0.00D-6,          -1.30D-6, &
     &               +0.00D-6,          -1.30D-6, &
     &               +0.00D-6,          -1.30D-6, &
     &               +0.00D-6,          -1.29D-6, &
     &               +0.00D-6,          +1.20D-6, &
     &               +0.00D-6,          -1.20D-6, &
     &               -0.40D-6,          -0.80D-6, &
     &               +0.00D-6,          +1.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  431,  440 ) / &
     &               +1.20D-6,          +0.00D-6, &
     &               -0.70D-6,          -0.50D-6, &
     &               -1.00D-6,          +0.20D-6, &
     &               -1.00D-6,          +0.20D-6, &
     &               +0.20D-6,          -1.00D-6, &
     &               +0.40D-6,          +0.80D-6, &
     &               -0.40D-6,          +0.80D-6, &
     &               +0.00D-6,          -1.20D-6, &
     &               +0.00D-6,          +1.15D-6, &
     &               +0.00D-6,          +1.10D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  441,  450 ) / &
     &               -0.20D-6,          +0.90D-6, &
     &               -1.10D-6,          +0.00D-6, &
     &               +0.00D-6,          -1.10D-6, &
     &               -1.10D-6,          +0.00D-6, &
     &               +0.00D-6,          +1.10D-6, &
     &               +0.00D-6,          +1.10D-6, &
     &               +0.00D-6,          +1.10D-6, &
     &               +0.60D-6,          -0.50D-6, &
     &               -0.90D-6,          -0.20D-6, &
     &               -0.40D-6,          -0.70D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  451,  460 ) / &
     &               -0.50D-6,          +0.60D-6, &
     &               +0.00D-6,          +1.10D-6, &
     &               +0.00D-6,          -1.10D-6, &
     &               +0.00D-6,          +1.00D-6, &
     &               +1.00D-6,          +0.00D-6, &
     &               +0.80D-6,          -0.20D-6, &
     &               +0.00D-6,          +1.00D-6, &
     &               +0.00D-6,          +1.00D-6, &
     &               +0.00D-6,          -1.00D-6, &
     &               -1.00D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  461,  470 ) / &
     &               +0.00D-6,          +1.00D-6, &
     &               +1.00D-6,          +0.00D-6, &
     &               +1.00D-6,          +0.00D-6, &
     &               -0.80D-6,          -0.20D-6, &
     &               +0.40D-6,          +0.60D-6, &
     &               -0.40D-6,          -0.60D-6, &
     &               +0.00D-6,          -1.00D-6, &
     &               +0.00D-6,          +1.00D-6, &
     &               +0.00D-6,          +1.00D-6, &
     &               +0.00D-6,          +1.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  471,  480 ) / &
     &               +0.00D-6,          +1.00D-6, &
     &               +0.00D-6,          -1.00D-6, &
     &               +0.00D-6,          +0.91D-6, &
     &               +0.10D-6,          +0.80D-6, &
     &               +0.00D-6,          +0.90D-6, &
     &               +0.00D-6,          +0.90D-6, &
     &               +0.00D-6,          -0.90D-6, &
     &               +0.00D-6,          -0.90D-6, &
     &               -0.70D-6,          -0.20D-6, &
     &               +0.70D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  481,  490 ) / &
     &               -0.30D-6,          +0.60D-6, &
     &               +0.00D-6,          +0.90D-6, &
     &               +0.00D-6,          +0.90D-6, &
     &               +0.00D-6,          -0.90D-6, &
     &               -0.50D-6,          -0.40D-6, &
     &               -0.90D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.90D-6, &
     &               +0.00D-6,          +0.90D-6, &
     &               +0.00D-6,          +0.90D-6, &
     &               +0.00D-6,          -0.90D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  491,  500 ) / &
     &               +0.00D-6,          -0.90D-6, &
     &               +0.00D-6,          -0.80D-6, &
     &               +0.00D-6,          +0.80D-6, &
     &               +0.00D-6,          -0.80D-6, &
     &               +0.10D-6,          +0.70D-6, &
     &               -0.70D-6,          +0.10D-6, &
     &               -0.60D-6,          +0.20D-6, &
     &               +0.20D-6,          +0.60D-6, &
     &               +0.00D-6,          +0.80D-6, &
     &               -0.50D-6,          +0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  501,  510 ) / &
     &               -0.50D-6,          -0.30D-6, &
     &               -0.50D-6,          -0.30D-6, &
     &               +0.00D-6,          -0.80D-6, &
     &               -0.30D-6,          +0.50D-6, &
     &               -0.80D-6,          +0.00D-6, &
     &               -0.30D-6,          -0.50D-6, &
     &               -0.30D-6,          +0.50D-6, &
     &               -0.30D-6,          -0.50D-6, &
     &               +0.00D-6,          +0.80D-6, &
     &               +0.00D-6,          -0.80D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  511,  520 ) / &
     &               +0.00D-6,          -0.80D-6, &
     &               +0.00D-6,          -0.80D-6, &
     &               +0.00D-6,          +0.80D-6, &
     &               +0.00D-6,          +0.80D-6, &
     &               +0.00D-6,          -0.80D-6, &
     &               +0.00D-6,          +0.76D-6, &
     &               +0.00D-6,          +0.70D-6, &
     &               +0.10D-6,          -0.60D-6, &
     &               +0.00D-6,          +0.70D-6, &
     &               +0.70D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  521,  530 ) / &
     &               +0.00D-6,          -0.70D-6, &
     &               +0.00D-6,          -0.70D-6, &
     &               +0.00D-6,          +0.70D-6, &
     &               +0.00D-6,          -0.70D-6, &
     &               -0.70D-6,          +0.00D-6, &
     &               -0.50D-6,          +0.20D-6, &
     &               -0.20D-6,          -0.50D-6, &
     &               +0.50D-6,          -0.20D-6, &
     &               +0.20D-6,          +0.50D-6, &
     &               -0.20D-6,          -0.50D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  531,  540 ) / &
     &               +0.50D-6,          -0.20D-6, &
     &               -0.50D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.70D-6, &
     &               +0.00D-6,          -0.70D-6, &
     &               +0.70D-6,          +0.00D-6, &
     &               -0.60D-6,          -0.10D-6, &
     &               +0.60D-6,          -0.10D-6, &
     &               +0.40D-6,          +0.30D-6, &
     &               +0.00D-6,          +0.70D-6, &
     &               +0.70D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  541,  550 ) / &
     &               +0.00D-6,          +0.70D-6, &
     &               +0.00D-6,          +0.70D-6, &
     &               +0.00D-6,          +0.70D-6, &
     &               +0.00D-6,          -0.60D-6, &
     &               +0.00D-6,          +0.60D-6, &
     &               +0.10D-6,          -0.50D-6, &
     &               +0.00D-6,          +0.60D-6, &
     &               +0.40D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.60D-6, &
     &               +0.00D-6,          -0.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  551,  560 ) / &
     &               +0.00D-6,          +0.60D-6, &
     &               +0.00D-6,          -0.60D-6, &
     &               +0.00D-6,          -0.60D-6, &
     &               +0.50D-6,          +0.10D-6, &
     &               -0.50D-6,          -0.10D-6, &
     &               -0.10D-6,          -0.50D-6, &
     &               +0.10D-6,          +0.50D-6, &
     &               +0.50D-6,          -0.10D-6, &
     &               -0.10D-6,          +0.50D-6, &
     &               +0.00D-6,          -0.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  561,  570 ) / &
     &               -0.40D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.60D-6, &
     &               +0.60D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.60D-6, &
     &               +0.20D-6,          +0.40D-6, &
     &               -0.40D-6,          +0.20D-6, &
     &               +0.30D-6,          +0.30D-6, &
     &               +0.40D-6,          -0.20D-6, &
     &               -0.40D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  571,  580 ) / &
     &               +0.00D-6,          +0.60D-6, &
     &               +0.40D-6,          +0.20D-6, &
     &               -0.20D-6,          -0.40D-6, &
     &               +0.00D-6,          +0.60D-6, &
     &               +0.00D-6,          -0.60D-6, &
     &               +0.00D-6,          +0.60D-6, &
     &               +0.00D-6,          +0.60D-6, &
     &               +0.00D-6,          -0.60D-6, &
     &               +0.00D-6,          -0.60D-6, &
     &               +0.00D-6,          -0.60D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  581,  590 ) / &
     &               +0.00D-6,          -0.50D-6, &
     &               +0.00D-6,          +0.50D-6, &
     &               +0.10D-6,          +0.40D-6, &
     &               +0.00D-6,          -0.50D-6, &
     &               +0.10D-6,          +0.40D-6, &
     &               +0.00D-6,          +0.50D-6, &
     &               +0.00D-6,          +0.50D-6, &
     &               +0.00D-6,          -0.50D-6, &
     &               +0.30D-6,          -0.20D-6, &
     &               -0.20D-6,          +0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  591,  600 ) / &
     &               +0.20D-6,          +0.30D-6, &
     &               +0.40D-6,          -0.10D-6, &
     &               +0.40D-6,          +0.10D-6, &
     &               +0.00D-6,          -0.50D-6, &
     &               +0.00D-6,          -0.50D-6, &
     &               +0.30D-6,          +0.20D-6, &
     &               -0.30D-6,          +0.20D-6, &
     &               +0.20D-6,          +0.30D-6, &
     &               -0.30D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.50D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  601,  610 ) / &
     &               +0.00D-6,          +0.50D-6, &
     &               +0.00D-6,          +0.50D-6, &
     &               -0.50D-6,          +0.00D-6, &
     &               +0.50D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.50D-6, &
     &               -0.50D-6,          +0.00D-6, &
     &               -0.50D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.50D-6, &
     &               +0.40D-6,          +0.10D-6, &
     &               -0.40D-6,          -0.10D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  611,  620 ) / &
     &               +0.40D-6,          -0.10D-6, &
     &               -0.40D-6,          +0.10D-6, &
     &               +0.10D-6,          +0.40D-6, &
     &               +0.10D-6,          +0.40D-6, &
     &               -0.50D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.50D-6, &
     &               +0.00D-6,          -0.50D-6, &
     &               +0.00D-6,          +0.50D-6, &
     &               +0.00D-6,          +0.50D-6, &
     &               +0.00D-6,          -0.50D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  621,  630 ) / &
     &               +0.50D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.50D-6, &
     &               +0.00D-6,          +0.50D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               -0.20D-6,          +0.20D-6, &
     &               -0.10D-6,          +0.30D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.40D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.00D-6,          -0.40D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  631,  640 ) / &
     &               +0.00D-6,          +0.40D-6, &
     &               +0.00D-6,          +0.40D-6, &
     &               -0.10D-6,          +0.30D-6, &
     &               +0.00D-6,          +0.40D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.00D-6,          +0.40D-6, &
     &               -0.20D-6,          -0.20D-6, &
     &               +0.20D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  641,  650 ) / &
     &               +0.20D-6,          +0.20D-6, &
     &               -0.10D-6,          +0.30D-6, &
     &               -0.30D-6,          +0.10D-6, &
     &               +0.10D-6,          +0.30D-6, &
     &               -0.10D-6,          +0.30D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.00D-6,          +0.40D-6, &
     &               +0.00D-6,          +0.40D-6, &
     &               +0.00D-6,          +0.40D-6, &
     &               +0.00D-6,          +0.40D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  651,  660 ) / &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.40D-6,          +0.00D-6, &
     &               -0.40D-6,          +0.00D-6, &
     &               -0.40D-6,          +0.00D-6, &
     &               -0.40D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.40D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               -0.40D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  661,  670 ) / &
     &               +0.40D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.00D-6,          +0.40D-6, &
     &               +0.00D-6,          +0.40D-6, &
     &               +0.40D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               -0.40D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.40D-6, &
     &               +0.40D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.40D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  671,  680 ) / &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               -0.10D-6,          +0.30D-6, &
     &               +0.00D-6,          +0.40D-6, &
     &               +0.00D-6,          +0.40D-6, &
     &               +0.00D-6,          +0.40D-6, &
     &               +0.00D-6,          -0.40D-6, &
     &               +0.40D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  681,  690 ) / &
     &               +0.00D-6,          +0.40D-6, &
     &               +0.21D-6,          +0.10D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               -0.30D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.20D-6,          +0.10D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          +0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  691,  700 ) / &
     &               +0.30D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               -0.20D-6,          +0.10D-6, &
     &               -0.10D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.10D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          -0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  701,  710 ) / &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               -0.20D-6,          +0.10D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               -0.10D-6,          -0.20D-6, &
     &               -0.10D-6,          +0.20D-6, &
     &               +0.20D-6,          -0.10D-6, &
     &               -0.10D-6,          -0.20D-6, &
     &               +0.20D-6,          +0.10D-6, &
     &               +0.20D-6,          -0.10D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  711,  720 ) / &
     &               -0.20D-6,          -0.10D-6, &
     &               -0.10D-6,          -0.20D-6, &
     &               +0.20D-6,          -0.10D-6, &
     &               +0.20D-6,          +0.10D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.30D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  721,  730 ) / &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.30D-6,          +0.00D-6, &
     &               +0.30D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          -0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  731,  740 ) / &
     &               +0.30D-6,          +0.00D-6, &
     &               +0.30D-6,          +0.00D-6, &
     &               -0.30D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               -0.30D-6,          +0.00D-6, &
     &               +0.30D-6,          +0.00D-6, &
     &               -0.30D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.30D-6,          +0.00D-6, &
     &               -0.30D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  741,  750 ) / &
     &               +0.30D-6,          +0.00D-6, &
     &               +0.30D-6,          +0.00D-6, &
     &               +0.30D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.30D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          -0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  751,  760 ) / &
     &               +0.30D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.30D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          -0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  761,  770 ) / &
     &               +0.30D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          +0.30D-6, &
     &               -0.30D-6,          +0.00D-6, &
     &               -0.30D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.30D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  771,  780 ) / &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               -0.20D-6,          -0.10D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          -0.30D-6, &
     &               +0.00D-6,          -0.21D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  781,  790 ) / &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  791,  800 ) / &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  801,  810 ) / &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.10D-6,          -0.10D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  811,  820 ) / &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  821,  830 ) / &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  831,  840 ) / &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  841,  850 ) / &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.20D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  851,  860 ) / &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  861,  870 ) / &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  871,  880 ) / &
     &               -0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  881,  890 ) / &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.20D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  891,  900 ) / &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.10D-6,          -0.10D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  901,  910 ) / &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               -0.20D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  911,  920 ) / &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  921,  930 ) / &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               -0.20D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  931,  940 ) / &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          +0.20D-6, &
     &               +0.00D-6,          -0.19D-6, &
     &               +0.00D-6,          +0.17D-6, &
     &               +0.00D-6,          +0.11D-6, &
     &               +0.00D-6,          -0.10D-6, &
     &               +0.10D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  941,  950 ) / &
     &               +0.00D-6,          -0.10D-6, &
     &               +0.00D-6,          +0.10D-6, &
     &               +0.00D-6,          -0.10D-6, &
     &               +0.00D-6,          -0.10D-6, &
     &               +0.00D-6,          -0.10D-6, &
     &               +0.00D-6,          -0.10D-6, &
     &               +0.00D-6,          -0.10D-6, &
     &               +0.00D-6,          +0.10D-6, &
     &               +0.00D-6,          -0.10D-6, &
     &               -0.10D-6,          +0.00D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  951,  960 ) / &
     &               +0.00D-6,          -0.10D-6, &
     &               +0.00D-6,          -0.10D-6, &
     &               +0.00D-6,          +0.10D-6, &
     &               +0.00D-6,          -0.10D-6, &
     &               +0.00D-6,          -0.10D-6, &
     &               +0.10D-6,          +0.00D-6, &
     &               +0.10D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.10D-6, &
     &               +0.00D-6,          -0.10D-6, &
     &               +0.00D-6,          -0.10D-6 /
      DATA ( ( SY0(I,J), I=1,2), J =  961,  NY0 ) / &
     &               +0.00D-6,          +0.10D-6, &
     &               +0.00D-6,          +0.10D-6 /
!
!  Sine and cosine coefficients for t^1 terms
      DATA ( ( SY1(I,J), I=1,2), J =    1,   10 ) / &
     &          +153041.82D-6,        +878.89D-6, &
     &           +11714.49D-6,        -289.32D-6, &
     &            +2024.68D-6,         -50.99D-6, &
     &            -1837.33D-6,         +47.75D-6, &
     &            -1312.21D-6,         -28.91D-6, &
     &             -632.54D-6,          +0.78D-6, &
     &             +459.68D-6,         -67.23D-6, &
     &             +344.50D-6,          +1.46D-6, &
     &             +268.14D-6,          -7.03D-6, &
     &             +192.06D-6,         +29.80D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   11,   20 ) / &
     &             +139.64D-6,          +0.15D-6, &
     &             -113.94D-6,          -1.06D-6, &
     &             +109.81D-6,          +3.18D-6, &
     &              -56.37D-6,          +0.13D-6, &
     &              -56.17D-6,          -0.02D-6, &
     &              -53.05D-6,          -1.23D-6, &
     &              -51.60D-6,          +0.17D-6, &
     &              +45.91D-6,          -0.11D-6, &
     &              -42.45D-6,          +0.02D-6, &
     &              +40.82D-6,          -1.03D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   21,   30 ) / &
     &              +34.30D-6,          -1.24D-6, &
     &              +28.89D-6,          +0.00D-6, &
     &              +27.61D-6,          -1.22D-6, &
     &              -25.43D-6,          +1.00D-6, &
     &              -26.01D-6,          +0.07D-6, &
     &              -23.02D-6,          +0.06D-6, &
     &              +19.37D-6,          -0.01D-6, &
     &              +14.05D-6,          -4.19D-6, &
     &              +18.18D-6,          -0.01D-6, &
     &              -14.86D-6,          -0.09D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   31,   40 ) / &
     &              +13.49D-6,          -0.01D-6, &
     &              +12.44D-6,          -0.27D-6, &
     &              +11.46D-6,          +0.03D-6, &
     &              -11.33D-6,          -0.06D-6, &
     &               -9.81D-6,          +0.01D-6, &
     &               -9.08D-6,          -0.02D-6, &
     &               +2.74D-6,          -4.56D-6, &
     &               +6.84D-6,          -0.04D-6, &
     &               -6.73D-6,          +0.01D-6, &
     &               +6.54D-6,          +0.01D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   41,   50 ) / &
     &               -6.35D-6,          -0.01D-6, &
     &               +5.90D-6,          -0.02D-6, &
     &               -5.85D-6,          +0.02D-6, &
     &               -5.73D-6,          +0.01D-6, &
     &               +5.60D-6,          +0.00D-6, &
     &               -5.16D-6,          +0.00D-6, &
     &               -5.14D-6,          +0.01D-6, &
     &               +4.76D-6,          -0.02D-6, &
     &               -4.40D-6,          +0.02D-6, &
     &               -4.22D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   51,   60 ) / &
     &               -4.20D-6,          +0.01D-6, &
     &               +3.58D-6,          +0.31D-6, &
     &               +3.87D-6,          +0.01D-6, &
     &               +3.76D-6,          +0.00D-6, &
     &               -3.62D-6,          -0.01D-6, &
     &               -3.61D-6,          +0.00D-6, &
     &               -1.28D-6,          -2.14D-6, &
     &               -3.18D-6,          +0.00D-6, &
     &               +3.01D-6,          +0.00D-6, &
     &               -2.97D-6,          +0.01D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   61,   70 ) / &
     &               +2.91D-6,          +0.00D-6, &
     &               -2.73D-6,          +0.00D-6, &
     &               +2.58D-6,          -0.01D-6, &
     &               +2.56D-6,          -0.01D-6, &
     &               -2.51D-6,          -0.01D-6, &
     &               -2.35D-6,          -0.01D-6, &
     &               -2.21D-6,          +0.01D-6, &
     &               -2.04D-6,          +0.01D-6, &
     &               -1.94D-6,          +0.00D-6, &
     &               +0.41D-6,          -1.43D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   71,   80 ) / &
     &               -1.84D-6,          +0.00D-6, &
     &               -1.77D-6,          +0.01D-6, &
     &               +0.00D-6,          +1.77D-6, &
     &               +1.76D-6,          +0.00D-6, &
     &               -1.07D-6,          -0.53D-6, &
     &               -1.48D-6,          +0.00D-6, &
     &               -1.40D-6,          +0.01D-6, &
     &               -1.35D-6,          -0.01D-6, &
     &               -1.32D-6,          +0.00D-6, &
     &               -1.28D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   81,   90 ) / &
     &               +1.24D-6,          +0.00D-6, &
     &               +1.23D-6,          +0.00D-6, &
     &               +1.19D-6,          +0.00D-6, &
     &               +1.18D-6,          -0.01D-6, &
     &               +1.17D-6,          +0.00D-6, &
     &               -1.15D-6,          +0.00D-6, &
     &               +1.14D-6,          +0.00D-6, &
     &               -1.14D-6,          +0.00D-6, &
     &               +1.09D-6,          +0.03D-6, &
     &               -1.08D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =   91,  100 ) / &
     &               +1.04D-6,          +0.00D-6, &
     &               +1.02D-6,          +0.00D-6, &
     &               +0.98D-6,          -0.01D-6, &
     &               +0.91D-6,          +0.02D-6, &
     &               +0.00D-6,          +0.93D-6, &
     &               -0.91D-6,          +0.00D-6, &
     &               -0.90D-6,          +0.00D-6, &
     &               +0.86D-6,          +0.00D-6, &
     &               -0.84D-6,          +0.00D-6, &
     &               -0.83D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  101,  110 ) / &
     &               -0.82D-6,          +0.00D-6, &
     &               +0.41D-6,          +0.39D-6, &
     &               +0.40D-6,          -0.38D-6, &
     &               +0.78D-6,          +0.00D-6, &
     &               +0.74D-6,          +0.00D-6, &
     &               -0.73D-6,          +0.00D-6, &
     &               +0.68D-6,          +0.00D-6, &
     &               +0.66D-6,          +0.00D-6, &
     &               -0.64D-6,          +0.00D-6, &
     &               -0.63D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  111,  120 ) / &
     &               +0.63D-6,          +0.00D-6, &
     &               +0.62D-6,          +0.00D-6, &
     &               +0.60D-6,          +0.00D-6, &
     &               -0.59D-6,          +0.00D-6, &
     &               -0.59D-6,          +0.00D-6, &
     &               +0.59D-6,          +0.00D-6, &
     &               +0.57D-6,          +0.00D-6, &
     &               +0.38D-6,          -0.19D-6, &
     &               -0.01D-6,          -0.55D-6, &
     &               +0.44D-6,          -0.11D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  121,  130 ) / &
     &               +0.53D-6,          +0.00D-6, &
     &               -0.53D-6,          +0.00D-6, &
     &               +0.52D-6,          +0.00D-6, &
     &               -0.52D-6,          +0.00D-6, &
     &               +0.53D-6,          +0.00D-6, &
     &               +0.52D-6,          +0.00D-6, &
     &               +0.51D-6,          +0.00D-6, &
     &               +0.51D-6,          +0.00D-6, &
     &               -0.21D-6,          -0.30D-6, &
     &               -0.50D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  131,  140 ) / &
     &               -0.11D-6,          +0.37D-6, &
     &               -0.11D-6,          +0.37D-6, &
     &               -0.48D-6,          +0.00D-6, &
     &               -0.46D-6,          -0.01D-6, &
     &               -0.47D-6,          +0.00D-6, &
     &               -0.03D-6,          +0.43D-6, &
     &               +0.45D-6,          +0.00D-6, &
     &               -0.44D-6,          +0.00D-6, &
     &               -0.44D-6,          +0.00D-6, &
     &               -0.44D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  141,  150 ) / &
     &               +0.43D-6,          +0.00D-6, &
     &               +0.44D-6,          +0.00D-6, &
     &               +0.42D-6,          +0.00D-6, &
     &               -0.42D-6,          +0.00D-6, &
     &               +0.41D-6,          +0.00D-6, &
     &               -0.41D-6,          +0.00D-6, &
     &               +0.02D-6,          +0.39D-6, &
     &               +0.40D-6,          +0.00D-6, &
     &               -0.40D-6,          +0.00D-6, &
     &               -0.39D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  151,  160 ) / &
     &               +0.39D-6,          +0.00D-6, &
     &               +0.15D-6,          -0.24D-6, &
     &               -0.37D-6,          -0.01D-6, &
     &               +0.37D-6,          +0.00D-6, &
     &               -0.37D-6,          +0.00D-6, &
     &               -0.37D-6,          +0.00D-6, &
     &               -0.31D-6,          +0.06D-6, &
     &               -0.35D-6,          +0.00D-6, &
     &               +0.35D-6,          +0.00D-6, &
     &               -0.07D-6,          -0.27D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  161,  170 ) / &
     &               -0.33D-6,          +0.01D-6, &
     &               -0.33D-6,          +0.00D-6, &
     &               +0.07D-6,          -0.26D-6, &
     &               +0.33D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.32D-6, &
     &               +0.32D-6,          +0.00D-6, &
     &               -0.32D-6,          +0.00D-6, &
     &               +0.32D-6,          +0.00D-6, &
     &               -0.24D-6,          -0.07D-6, &
     &               +0.24D-6,          +0.07D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  171,  180 ) / &
     &               +0.30D-6,          +0.00D-6, &
     &               +0.08D-6,          -0.22D-6, &
     &               -0.30D-6,          +0.00D-6, &
     &               -0.30D-6,          +0.00D-6, &
     &               +0.30D-6,          +0.00D-6, &
     &               +0.30D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.29D-6, &
     &               +0.00D-6,          -0.29D-6, &
     &               +0.20D-6,          -0.09D-6, &
     &               +0.29D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  181,  190 ) / &
     &               -0.05D-6,          -0.24D-6, &
     &               +0.29D-6,          +0.00D-6, &
     &               -0.27D-6,          +0.00D-6, &
     &               -0.19D-6,          -0.08D-6, &
     &               -0.27D-6,          +0.00D-6, &
     &               +0.25D-6,          +0.00D-6, &
     &               +0.25D-6,          +0.00D-6, &
     &               -0.25D-6,          +0.00D-6, &
     &               +0.25D-6,          +0.00D-6, &
     &               -0.25D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  191,  200 ) / &
     &               -0.01D-6,          +0.23D-6, &
     &               -0.23D-6,          +0.00D-6, &
     &               +0.23D-6,          +0.00D-6, &
     &               +0.23D-6,          +0.00D-6, &
     &               -0.15D-6,          -0.07D-6, &
     &               -0.23D-6,          +0.00D-6, &
     &               -0.22D-6,          +0.00D-6, &
     &               +0.22D-6,          +0.00D-6, &
     &               -0.22D-6,          +0.00D-6, &
     &               -0.22D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  201,  210 ) / &
     &               +0.04D-6,          -0.17D-6, &
     &               -0.01D-6,          -0.21D-6, &
     &               +0.08D-6,          -0.14D-6, &
     &               -0.01D-6,          +0.19D-6, &
     &               +0.21D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               -0.04D-6,          -0.16D-6, &
     &               +0.19D-6,          +0.00D-6, &
     &               +0.19D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  211,  220 ) / &
     &               -0.19D-6,          +0.00D-6, &
     &               +0.18D-6,          +0.00D-6, &
     &               -0.18D-6,          +0.00D-6, &
     &               +0.18D-6,          +0.00D-6, &
     &               +0.17D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.06D-6, &
     &               +0.13D-6,          -0.04D-6, &
     &               -0.11D-6,          +0.06D-6, &
     &               +0.17D-6,          +0.00D-6, &
     &               +0.16D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  221,  230 ) / &
     &               -0.17D-6,          +0.00D-6, &
     &               -0.17D-6,          +0.00D-6, &
     &               -0.14D-6,          +0.02D-6, &
     &               +0.14D-6,          +0.03D-6, &
     &               +0.00D-6,          +0.15D-6, &
     &               -0.15D-6,          +0.00D-6, &
     &               -0.14D-6,          +0.01D-6, &
     &               +0.16D-6,          +0.00D-6, &
     &               -0.06D-6,          +0.10D-6, &
     &               +0.05D-6,          +0.10D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  231,  240 ) / &
     &               +0.02D-6,          +0.13D-6, &
     &               -0.11D-6,          +0.04D-6, &
     &               -0.12D-6,          -0.02D-6, &
     &               -0.05D-6,          -0.10D-6, &
     &               +0.14D-6,          +0.00D-6, &
     &               -0.09D-6,          +0.05D-6, &
     &               +0.00D-6,          +0.14D-6, &
     &               +0.14D-6,          +0.00D-6, &
     &               -0.14D-6,          +0.00D-6, &
     &               +0.04D-6,          +0.10D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  241,  250 ) / &
     &               -0.06D-6,          +0.08D-6, &
     &               +0.05D-6,          +0.09D-6, &
     &               -0.14D-6,          +0.00D-6, &
     &               +0.08D-6,          +0.06D-6, &
     &               +0.14D-6,          +0.00D-6, &
     &               +0.14D-6,          +0.00D-6, &
     &               +0.13D-6,          +0.00D-6, &
     &               -0.07D-6,          +0.06D-6, &
     &               +0.11D-6,          -0.02D-6, &
     &               -0.13D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  251,  260 ) / &
     &               -0.13D-6,          +0.00D-6, &
     &               -0.13D-6,          +0.00D-6, &
     &               -0.13D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               +0.12D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               +0.00D-6,          -0.12D-6, &
     &               -0.02D-6,          -0.09D-6, &
     &               +0.02D-6,          -0.09D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  261,  270 ) / &
     &               -0.11D-6,          +0.00D-6, &
     &               +0.11D-6,          +0.00D-6, &
     &               +0.07D-6,          -0.04D-6, &
     &               +0.11D-6,          +0.00D-6, &
     &               +0.11D-6,          +0.00D-6, &
     &               -0.11D-6,          +0.00D-6, &
     &               +0.10D-6,          +0.00D-6, &
     &               -0.10D-6,          +0.00D-6, &
     &               +0.10D-6,          +0.00D-6, &
     &               +0.10D-6,          +0.00D-6 /
      DATA ( ( SY1(I,J), I=1,2), J =  271,  NY1 ) / &
     &               +0.10D-6,          +0.00D-6, &
     &               +0.00D-6,          +0.10D-6, &
     &               +0.00D-6,          +0.10D-6, &
     &               -0.10D-6,          +0.00D-6, &
     &               +0.10D-6,          +0.00D-6, &
     &               -0.10D-6,          +0.00D-6, &
     &               +0.10D-6,          +0.00D-6 /
!
!  Sine and cosine coefficients for t^2 terms
      DATA ( ( SY2(I,J), I=1,2), J =    1,   10 ) / &
     &             +121.15D-6,       -2301.27D-6, &
     &               -0.98D-6,        -143.27D-6, &
     &               -0.27D-6,         -24.46D-6, &
     &               +0.24D-6,         +22.41D-6, &
     &               -1.19D-6,          -5.61D-6, &
     &               +3.57D-6,          -1.83D-6, &
     &               +0.24D-6,          -5.02D-6, &
     &               -0.04D-6,          -3.23D-6, &
     &               -0.48D-6,          +2.40D-6, &
     &               -0.10D-6,          +1.73D-6 /
      DATA ( ( SY2(I,J), I=1,2), J =   11,   20 ) / &
     &               -0.01D-6,          +1.33D-6, &
     &               -0.04D-6,          +0.83D-6, &
     &               -0.05D-6,          -0.79D-6, &
     &               +0.03D-6,          -0.66D-6, &
     &               +0.00D-6,          -0.64D-6, &
     &               +0.04D-6,          +0.61D-6, &
     &               -0.01D-6,          -0.41D-6, &
     &               -0.01D-6,          +0.35D-6, &
     &               -0.01D-6,          -0.33D-6, &
     &               +0.01D-6,          +0.31D-6 /
      DATA ( ( SY2(I,J), I=1,2), J =   21,  NY2 ) / &
     &               +0.01D-6,          +0.27D-6, &
     &               -0.07D-6,          -0.17D-6, &
     &               +0.07D-6,          +0.17D-6, &
     &               +0.02D-6,          -0.21D-6, &
     &               +0.01D-6,          +0.20D-6, &
     &               +0.01D-6,          -0.17D-6, &
     &               +0.01D-6,          -0.16D-6, &
     &               +0.00D-6,          -0.13D-6, &
     &               -0.07D-6,          -0.04D-6, &
     &               +0.02D-6,          +0.08D-6 /
!
!  Sine and cosine coefficients for t^3 terms
      DATA ( ( SY3(I,J), I=1,2), J =    1,  NY3 ) / &
     &              -15.23D-6,          -1.62D-6, &
     &               -1.16D-6,          -0.01D-6, &
     &               -0.20D-6,          +0.00D-6, &
     &               +0.18D-6,          +0.00D-6, &
     &               +0.13D-6,          +0.00D-6 /
!
!  Sine and cosine coefficients for t^4 terms
      DATA ( ( SY4(I,J), I=1,2), J =    1,  NY4 ) / &
     &               -0.01D-6,          +0.11D-6 /
!
!  ---------------------------------------
!  The series for s+XY/2: numerical values
!  ---------------------------------------
!
!  Polynomial coefficients
      DATA SP /    94D-6, &
     &           3808.35D-6, &
     &           -119.94D-6, &
     &         -72574.09D-6, &
     &             27.70D-6, &
     &             15.61D-6 /
!
!  Argument coefficients for t^0
      DATA ( ( KS0(I,J), I=1,14), J =    1,   10 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KS0(I,J), I=1,14), J =   11,   20 ) / &
     &  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4, -4,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KS0(I,J), I=1,14), J =   21,   30 ) / &
     &  0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -1, &
     &  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  4, -2,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KS0(I,J), I=1,14), J =   31,  NS0 ) / &
     &  0,  0,  2, -2,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
!
!  Argument coefficients for t^1
      DATA ( ( KS1(I,J), I=1,14), J =    1,  NS1 ) / &
     &  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
!
!  Argument coefficients for t^2
      DATA ( ( KS2(I,J), I=1,14), J =    1,   10 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  1, -2,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KS2(I,J), I=1,14), J =   11,   20 ) / &
     &  1,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0, -2, -2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
      DATA ( ( KS2(I,J), I=1,14), J =   21,  NS2 ) / &
     &  0,  0,  2,  2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  1,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
!
!  Argument coefficients for t^3
      DATA ( ( KS3(I,J), I=1,14), J =    1,  NS3 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0, &
     &  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
!
!  Argument coefficients for t^4
      DATA ( ( KS4(I,J), I=1,14), J =    1,  NS4 ) / &
     &  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0 /
!
!  Sine and cosine coefficients for t^0
      DATA ( ( SS0(I,J), I=1,2), J =    1,   10 ) / &
     &            -2640.73D-6,          +0.39D-6, &
     &              -63.53D-6,          +0.02D-6, &
     &              -11.75D-6,          -0.01D-6, &
     &              -11.21D-6,          -0.01D-6, &
     &               +4.57D-6,          +0.00D-6, &
     &               -2.02D-6,          +0.00D-6, &
     &               -1.98D-6,          +0.00D-6, &
     &               +1.72D-6,          +0.00D-6, &
     &               +1.41D-6,          +0.01D-6, &
     &               +1.26D-6,          +0.01D-6 /
      DATA ( ( SS0(I,J), I=1,2), J =   11,   20 ) / &
     &               +0.63D-6,          +0.00D-6, &
     &               +0.63D-6,          +0.00D-6, &
     &               -0.46D-6,          +0.00D-6, &
     &               -0.45D-6,          +0.00D-6, &
     &               -0.36D-6,          +0.00D-6, &
     &               +0.24D-6,          +0.12D-6, &
     &               -0.32D-6,          +0.00D-6, &
     &               -0.28D-6,          +0.00D-6, &
     &               -0.27D-6,          +0.00D-6, &
     &               -0.26D-6,          +0.00D-6 /
      DATA ( ( SS0(I,J), I=1,2), J =   21,   30 ) / &
     &               +0.21D-6,          +0.00D-6, &
     &               -0.19D-6,          +0.00D-6, &
     &               -0.18D-6,          +0.00D-6, &
     &               +0.10D-6,          -0.05D-6, &
     &               -0.15D-6,          +0.00D-6, &
     &               +0.14D-6,          +0.00D-6, &
     &               +0.14D-6,          +0.00D-6, &
     &               -0.14D-6,          +0.00D-6, &
     &               -0.14D-6,          +0.00D-6, &
     &               -0.13D-6,          +0.00D-6 /
      DATA ( ( SS0(I,J), I=1,2), J =   31,  NS0 ) / &
     &               +0.11D-6,          +0.00D-6, &
     &               -0.11D-6,          +0.00D-6, &
     &               -0.11D-6,          +0.00D-6 /
!
!  Sine and cosine coefficients for t^1
      DATA ( ( SS1(I,J), I=1,2), J =    1,  NS1 ) / &
     &               -0.07D-6,          +3.57D-6, &
     &               +1.71D-6,          -0.03D-6, &
     &               +0.00D-6,          +0.48D-6 /
!
!  Sine and cosine coefficients for t^2
      DATA ( ( SS2(I,J), I=1,2), J =    1,   10 ) / &
     &             +743.53D-6,          -0.17D-6, &
     &              +56.91D-6,          +0.06D-6, &
     &               +9.84D-6,          -0.01D-6, &
     &               -8.85D-6,          +0.01D-6, &
     &               -6.38D-6,          -0.05D-6, &
     &               -3.07D-6,          +0.00D-6, &
     &               +2.23D-6,          +0.00D-6, &
     &               +1.67D-6,          +0.00D-6, &
     &               +1.30D-6,          +0.00D-6, &
     &               +0.93D-6,          +0.00D-6 /
      DATA ( ( SS2(I,J), I=1,2), J =   11,   20 ) / &
     &               +0.68D-6,          +0.00D-6, &
     &               -0.55D-6,          +0.00D-6, &
     &               +0.53D-6,          +0.00D-6, &
     &               -0.27D-6,          +0.00D-6, &
     &               -0.27D-6,          +0.00D-6, &
     &               -0.26D-6,          +0.00D-6, &
     &               -0.25D-6,          +0.00D-6, &
     &               +0.22D-6,          +0.00D-6, &
     &               -0.21D-6,          +0.00D-6, &
     &               +0.20D-6,          +0.00D-6 /
      DATA ( ( SS2(I,J), I=1,2), J =   21,  NS2 ) / &
     &               +0.17D-6,          +0.00D-6, &
     &               +0.13D-6,          +0.00D-6, &
     &               -0.13D-6,          +0.00D-6, &
     &               -0.12D-6,          +0.00D-6, &
     &               -0.11D-6,          +0.00D-6 /
!
!  Sine and cosine coefficients for t^3
      DATA ( ( SS3(I,J), I=1,2), J =    1,  NS3 ) / &
     &               +0.30D-6,         -23.51D-6, &
     &               -0.03D-6,          -1.39D-6, &
     &               -0.01D-6,          -0.24D-6, &
     &               +0.00D-6,          +0.22D-6 /
!
!  Sine and cosine coefficients for t^4
      DATA ( ( SS4(I,J), I=1,2), J =    1,  NS4 ) / &
     &               -0.26D-6,          -0.01D-6 /
!
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
!
!  Interval between fundamental epoch J2000.0 and current date (JC).
!     T = ( ( DATE1-DJ0 ) + DATE2 ) / DJC
      T = CENT
!  Derivative of T w.r.t. time (centuries/sec)
      dTdt = DCENT
!
!  Fundamental Arguments
!
!  Mean Anomaly of the Moon.
!     FA(1) = iau_ANPM ( ( 485868.249036D0 +
!    :                   ( 715923.2178D0 +
!    :                   (     31.8792D0 +
!    :                   (      0.051635D0 +
!    :                   (     -0.00024470D0 )
!    :                   * T ) * T ) * T ) * T ) * DAS2R
!    :                   + MOD ( 1325D0*T, 1D0 ) * D2PI )
!
!  Mean Anomaly of the Sun.
!     FA(2) = iau_ANPM ( ( 1287104.793048D0 +
!    :                   ( 1292581.0481D0 +
!    :                   (      -0.5532D0 +
!    :                   (      +0.000136D0 +
!    :                   (      -0.00001149D0 )
!    :                   * T ) * T ) * T ) * T ) * DAS2R
!    :                   + MOD ( 99D0*T, 1D0 ) * D2PI )
!
!  Mean Longitude of the Moon minus Mean Longitude of the Ascending
!  Node of the Moon.
!     FA(3) = iau_ANPM ( (  335779.526232D0 +
!    :                   (  295262.8478D0 +
!    :                   (     -12.7512D0 +
!    :                   (      -0.001037D0 +
!    :                   (       0.00000417D0 )
!    :                   * T ) * T ) * T ) * T ) * DAS2R
!    :                   + MOD ( 1342D0*T, 1D0 ) * D2PI )
!
!  Mean Elongation of the Moon from the Sun.
!     FA(4) = iau_ANPM ( ( 1072260.703692D0 +
!    :                   ( 1105601.2090D0 +
!    :                   (      -6.3706D0 +
!    :                   (       0.006593D0 +
!    :                   (      -0.00003169D0 )
!    :                   * T ) * T ) * T ) * T ) * DAS2R
!    :                   + MOD ( 1236D0*T, 1D0 ) * D2PI )
!
!  Mean Longitude of the Ascending Node of the Moon.
!     FA(5) = iau_ANPM ( (  450160.398036D0 +
!    :                   ( -482890.5431D0 +
!    :                   (       7.4722D0 +
!    :                   (       0.007702D0 +
!    :                   (      -0.00005939D0 )
!    :                   * T ) * T ) * T ) * T ) * DAS2R
!    :                   + MOD ( -5D0*T, 1D0 ) * D2PI )
!
!*******************************************************************
!  FA(1) - FA(5) simplified, with time derivatives
!  Mean Anomaly of the Moon.
!     FA(1) = (485868.249036D0 + 715923.2178D0*T +
!    :        31.8792D0*T**2   + 0.051635D0*T**3 +
!    :        -0.00024470D0*T**4) * DAS2R
!    :        + DMOD (1325D0*T, 1D0) * D2PI
!     FA(1) = DMOD(FA(1),D2PI)
!     FAD(1) = (                  715923.2178D0   +
!    :       2.D0*31.8792D0*T  + 3.D0*0.051635D0*T**2 +
!    :        -4.D0*0.00024470D0*T**3) * dTdt * DAS2R
!    :        + 1325D0 * dTdt * D2PI
!     FAD(1) = FAD(1)/(36525.D0*86400.D0)
!
!  Mean Anomaly of the Sun.
!     FA(2) =  (1287104.793048D0 + 1292581.0481D0*T +
!    :         -0.5532D0*T**2    + 0.000136D0*T**3 +
!    :         -0.00001149D0*T**4) * DAS2R
!    :         + DMOD (99D0*T, 1D0) * D2PI
!     FA(2) = DMOD(FA(2),D2PI)
!     FAD(2) =  (                   1292581.0481D0 +
!    :         -2.D0*0.5532D0*T + 3.D0*0.000136D0*T**2 +
!    :         -4.D0*0.00001149D0*T**3) * dTdt * DAS2R
!    :         + 99D0 * dTdt * D2PI
!     FAD(2) = FAD(2)/(36525.D0*86400.D0)
!
!  Mean Longitude of the Moon minus Mean Longitude of the Ascending
!  Node of the Moon.
!     FA(3) = (335779.526232D0 + 295262.8478D0*T +
!    :        -12.7512D0*T**2  + -0.001037D0*T**3 +
!    :        0.00000417D0*T**4) * DAS2R
!    :        + DMOD (1342D0*T, 1D0) * D2PI
!     FA(3) = DMOD(FA(3),D2PI)
!     FAD(3) = (                  295262.8478D0 +
!    :        -2.D0*12.7512D0*T  + -3.D0*0.001037D0*T**2 +
!    :        4.D0*0.00000417D0*T**3) * dTdt * DAS2R
!    :        + 1342D0 * dTdt * D2PI
!     FAD(3) = FAD(3)/(36525.D0*86400.D0)
!
!  Mean Elongation of the Moon from the Sun.
!     FA(4) = (1072260.703692D0 + 1105601.2090D0*T +
!    :        -6.3706D0*T**2    + 0.006593D0*T**3 +
!    :        -0.00003169D0*T**4) * DAS2R
!    :        + DMOD (1236D0*T, 1D0) * D2PI
!     FA(4) = DMOD(FA(4),D2PI)
!     FAD(4) = (                   1105601.2090D0 +
!    :        -2.D0*6.3706D0*T + 3.D0*0.006593D0*T**2 +
!    :        -4.D0*0.00003169D0*T**3) * dTdt * DAS2R
!    :        + 1236D0 * dTdt * D2PI
!     FAD(4) = FAD(4)/(36525.D0*86400.D0)
!
!  Mean Longitude of the Ascending Node of the Moon.
!     FA(5) = (450160.398036D0 + -482890.5431D0*T +
!    :        7.4722D0*T**2    + 0.007702D0*T**3 +
!    :        -0.00005939D0*T**4) * DAS2R
!    :        + DMOD (-5D0*T, 1D0) * D2PI
!     FA(5) = DMOD(FA(5),D2PI)
!     FAD(5) = (                  -482890.5431D0 +
!    :        2.D0*7.4722D0*T + 3.D0*0.007702D0*T**2 +
!    :        -4.D0*0.00005939D0*T**3) * dTdt * DAS2R
!    :        + -5D0 * dTdt * D2PI
!     FAD(5) = FAD(5)/(36525.D0*86400.D0)
!
!*******************************************************************
!
!     FA( 6) = iau_ANPM ( 4.402608842D0 + 2608.7903141574D0 * T )
!     FA( 7) = iau_ANPM ( 3.176146697D0 + 1021.3285546211D0 * T )
!     FA( 8) = iau_ANPM ( 1.753470314D0 +  628.3075849991D0 * T )
!     FA( 9) = iau_ANPM ( 6.203480913D0 +  334.0612426700D0 * T )
!     FA(10) = iau_ANPM ( 0.599546497D0 +   52.9690962641D0 * T )
!     FA(11) = iau_ANPM ( 0.874016757D0 +   21.3299104960D0 * T )
!     FA(12) = iau_ANPM ( 5.481293872D0 +    7.4781598567D0 * T )
!     FA(13) = iau_ANPM ( 5.311886287D0 +    3.8133035638D0 * T )
!     FA(14) =          ( 0.024381750D0 +    0.00000538691D0 * T ) * T
!
!     FAD( 6) = 2608.7903141574D0/(36525.D0*86400.D0)
!     FAD( 7) = 1021.3285546211D0/(36525.D0*86400.D0)
!     FAD( 8) = 628.3075849991D0/(36525.D0*86400.D0)
!     FAD( 9) = 334.0612426700D0/(36525.D0*86400.D0)
!     FAD(10) = 52.9690962641D0/(36525.D0*86400.D0)
!     FAD(11) = 21.3299104960D0/(36525.D0*86400.D0)
!     FAD(12) = 7.4781598567D0/(36525.D0*86400.D0)
!     FAD(13) = 3.8133035638D0/(36525.D0*86400.D0)
!     FAD(14) = (0.024381750D0 + 2.D0*0.00000538691D0*T)/
!    *            (36525.D0*86400.D0)
!
!     FAD( 6) = 2608.7903141574D0 * dTdt
!     FAD( 7) = 1021.3285546211D0 * dTdt
!     FAD( 8) = 628.3075849991D0 * dTdt
!     FAD( 9) = 334.0612426700D0 * dTdt
!     FAD(10) = 52.9690962641D0 * dTdt
!     FAD(11) = 21.3299104960D0 * dTdt
!     FAD(12) = 7.4781598567D0 * dTdt
!     FAD(13) = 3.8133035638D0 * dTdt
!     FAD(14) = (0.024381750D0 + 2.D0*0.00000538691D0*T) * dTdt
!
!
!  Evaluate X.
      S0 = XP(1)
      S1 = XP(2)
      S2 = XP(3)
      S3 = XP(4)
      S4 = XP(5)
      S5 = XP(6)
       dS0 = 0.D0
       dS1 = 0.D0
       dS2 = 0.D0
       dS3 = 0.D0
       dS4 = 0.D0
       dS5 = 0.D0
!
      DO I = NX0,1,-1
         A = 0.D0
          dA = 0.D0
         DO J=1,14
            A = A + DBLE(KX0(J,I))*FA2K(J)
            dA = dA + DBLE(KX0(J,I))*FAD2K(J)
         END DO
         S0 = S0 + ( SX0(1,I)*SIN(A) + SX0(2,I)*COS(A) )
         dS0 =  dS0 + (SX0(1,I)*COS(A)*dA - SX0(2,I)*SIN(A)*dA)
      END DO
      DO I = NX1,1,-1
         A = 0D0
          dA = 0D0
         DO J=1,14
            A = A + DBLE(KX1(J,I))*FA2K(J)
            dA = dA + DBLE(KX1(J,I))*FAD2K(J)
         END DO
         S1 = S1 + ( SX1(1,I)*SIN(A) + SX1(2,I)*COS(A) )
         dS1 = dS1 + (SX1(1,I)*COS(A)*dA - SX1(2,I)*SIN(A)*dA)
      END DO
      DO I = NX2,1,-1
         A = 0D0
          dA = 0D0
         DO J=1,14
            A = A + DBLE(KX2(J,I))*FA2K(J)
            dA = dA + DBLE(KX2(J,I))*FAD2K(J)
         END DO
         S2 = S2 + ( SX2(1,I)*SIN(A) + SX2(2,I)*COS(A) )
         dS2 = dS2 + (SX2(1,I)*COS(A)*dA - SX2(2,I)*SIN(A)*dA)
      END DO
      DO I = NX3,1,-1
         A = 0D0
          dA = 0D0
         DO J=1,14
            A = A + DBLE(KX3(J,I))*FA2K(J)
            dA = dA + DBLE(KX3(J,I))*FAD2K(J)
         END DO
         S3 = S3 + ( SX3(1,I)*SIN(A) + SX3(2,I)*COS(A) )
         dS3 = dS3 + (SX3(1,I)*COS(A)*dA - SX3(2,I)*SIN(A)*dA)
      END DO
      DO I = NX4,1,-1
         A = 0D0
          dA = 0D0
         DO J=1,14
            A = A + DBLE(KX4(J,I))*FA2K(J)
            dA = dA + DBLE(KX4(J,I))*FAD2K(J)
         END DO
         S4 = S4 + ( SX4(1,I)*SIN(A) + SX4(2,I)*COS(A) )
         dS4 = dS4 + (SX4(1,I)*COS(A)*dA - SX4(2,I)*SIN(A)*dA)
      END DO
      X(1) = ( S0 + &
     &    ( S1 + &
     &    ( S2 + &
     &    ( S3 + &
     &    ( S4 + &
     &      S5 * T ) * T ) * T ) * T ) * T ) * DAS2R
!   Time derivative:
!     X(2) = (dS0 + (S1 + dS1*T) + (2.D0*S2*T + dS2*T**2) +
!    *        (3.D0*S3*T**2 + dS3*T**3 ) + (4.D0*S4*T**3 + dS4*T**4 ) +
!    *        (5.D0*S5*T**4 + dS5*T**5) ) * DAS2R/(36525.D0*86400.D0)
      X(2) = ( dS0 + (S1*dTdt + dS1*T) + (2.D0*S2*T*dTdt + dS2*T**2) + &
     &        (3.D0*S3*T**2*dTdt + dS3*T**3 ) + &
     &        (4.D0*S4*T**3*dTdt + dS4*T**4 ) + &
     &        (5.D0*S5*T**4*dTdt + dS5*T**5) ) * DAS2R
!
!  Evaluate Y.
      S0 = YP(1)
      S1 = YP(2)
      S2 = YP(3)
      S3 = YP(4)
      S4 = YP(5)
      S5 = YP(6)
       dS0 = 0.D0
       dS1 = 0.D0
       dS2 = 0.D0
       dS3 = 0.D0
       dS4 = 0.D0
       dS5 = 0.D0
!
      DO I = NY0,1,-1
         A = 0D0
          dA = 0D0
         DO J=1,14
            A = A + DBLE(KY0(J,I))*FA2K(J)
            dA = dA + DBLE(KY0(J,I))*FAD2K(J)
         END DO
         S0 = S0 + ( SY0(1,I)*SIN(A) + SY0(2,I)*COS(A) )
         dS0 = dS0 + (SY0(1,I)*COS(A)*dA - SY0(2,I)*SIN(A)*dA)
      END DO
      DO I = NY1,1,-1
         A = 0D0
          dA = 0D0
         DO J=1,14
            A = A + DBLE(KY1(J,I))*FA2K(J)
            dA = dA + DBLE(KY1(J,I))*FAD2K(J)
         END DO
         S1 = S1 + ( SY1(1,I)*SIN(A) + SY1(2,I)*COS(A) )
         dS1 = dS1 + (SY1(1,I)*COS(A)*dA - SY1(2,I)*SIN(A)*dA)
      END DO
      DO I = NY2,1,-1
         A = 0D0
          dA = 0D0
         DO J=1,14
            A = A + DBLE(KY2(J,I))*FA2K(J)
            dA = dA + DBLE(KY2(J,I))*FAD2K(J)
         END DO
         S2 = S2 + ( SY2(1,I)*SIN(A) + SY2(2,I)*COS(A) )
         dS2 = dS2 + (SY2(1,I)*COS(A)*dA - SY2(2,I)*SIN(A)*dA)
      END DO
      DO I = NY3,1,-1
         A = 0D0
          dA = 0D0
         DO J=1,14
            A = A + DBLE(KY3(J,I))*FA2K(J)
            dA = dA + DBLE(KY3(J,I))*FAD2K(J)
         END DO
         S3 = S3 + ( SY3(1,I)*SIN(A) + SY3(2,I)*COS(A) )
         dS3 = dS3 + (SY3(1,I)*COS(A)*dA - SY3(2,I)*SIN(A)*dA)
      END DO
      DO I = NY4,1,-1
         A = 0D0
          dA = 0D0
         DO J=1,14
            A = A + DBLE(KY4(J,I))*FA2K(J)
            dA = dA + DBLE(KY4(J,I))*FAD2K(J)
         END DO
         S4 = S4 + ( SY4(1,I)*SIN(A) + SY4(2,I)*COS(A) )
         dS4 = dS4 + ( SY4(1,I)*COS(A)*dA - SY4(2,I)*SIN(A)*dA)
      END DO
      Y(1) = ( S0 + &
     &    ( S1 + &
     &    ( S2 + &
     &    ( S3 + &
     &    ( S4 + &
     &      S5 * T ) * T ) * T ) * T ) * T ) * DAS2R
!  Time derivative:
!     Y(2) = (dS0 + (S1 + dS1*T) + (2.D0*S2*T + dS2*T**2) +
!    *        (3.D0*S3*T**2 + dS3*T**3 ) + (4.D0*S4*T**3 + dS4*T**4 ) +
!    *        (5.D0*S5*T**4 + dS5*T**5) ) * DAS2R/(36525.D0*86400.D0)
      Y(2) = ( dS0 + (S1*dTdt + dS1*T) + (2.D0*S2*T*dTdt + dS2*T**2) + &
     &        (3.D0*S3*T**2*dTdt + dS3*T**3 ) + &
     &        (4.D0*S4*T**3*dTdt + dS4*T**4 ) + &
     &        (5.D0*S5*T**4*dTdt + dS5*T**5) ) * DAS2R
!
!  Evaluate S.
      S0 = SP(1)
      S1 = SP(2)
      S2 = SP(3)
      S3 = SP(4)
      S4 = SP(5)
      S5 = SP(6)
       dS0 = 0.D0
       dS1 = 0.D0
       dS2 = 0.D0
       dS3 = 0.D0
       dS4 = 0.D0
       dS5 = 0.D0
!
      DO I = NS0,1,-1
         A = 0D0
          dA = 0D0
         DO J=1,14
            A = A + DBLE(KS0(J,I))*FA2K(J)
            dA = dA + DBLE(KS0(J,I))*FAD2K(J)
         END DO
         S0 = S0 + ( SS0(1,I)*SIN(A) + SS0(2,I)*COS(A) )
         dS0 = dS0 + (SS0(1,I)*COS(A)*dA - SS0(2,I)*SIN(A)*dA)
      END DO
      DO I = NS1,1,-1
         A = 0D0
          dA = 0D0
         DO J=1,14
            A = A + DBLE(KS1(J,I))*FA2K(J)
            dA = dA + DBLE(KS1(J,I))*FAD2K(J)
         END DO
         S1 = S1 + ( SS1(1,I)*SIN(A) + SS1(2,I)*COS(A) )
         dS1 = dS1 + (SS1(1,I)*COS(A)*dA - SS1(2,I)*SIN(A)*dA)
      END DO
      DO I = NS2,1,-1
         A = 0D0
          dA = 0D0
         DO J=1,14
            A = A + DBLE(KS2(J,I))*FA2K(J)
            dA = dA + DBLE(KS2(J,I))*FAD2K(J)
         END DO
         S2 = S2 + ( SS2(1,I)*SIN(A) + SS2(2,I)*COS(A) )
         dS2 = dS2 + (SS2(1,I)*COS(A)*dA - SS2(2,I)*SIN(A)*dA)
      END DO
      DO I = NS3,1,-1
         A = 0D0
          dA = 0D0
         DO J=1,14
            A = A + DBLE(KS3(J,I))*FA2K(J)
            dA = dA + DBLE(KS3(J,I))*FAD2K(J)
         END DO
         S3 = S3 + ( SS3(1,I)*SIN(A) + SS3(2,I)*COS(A) )
         dS3 = dS3 + (SS3(1,I)*COS(A)*dA - SS3(2,I)*SIN(A)*dA)
      END DO
      DO I = NS4,1,-1
         A = 0D0
          dA = 0D0
         DO J=1,14
            A = A + DBLE(KS4(J,I))*FA2K(J)
            dA = dA + DBLE(KS4(J,I))*FAD2K(J)
         END DO
         S4 = S4 + ( SS4(1,I)*SIN(A) + SS4(2,I)*COS(A) )
         dS4 = dS4 + (SS4(1,I)*COS(A)*dA - SS4(2,I)*SIN(A)*dA)
      END DO
      S(1) = ( S0 + &
     &    ( S1 + &
     &    ( S2 + &
     &    ( S3 + &
     &    ( S4 + &
     &      S5 * T ) * T ) * T ) * T ) * T ) * DAS2R - X(1)*Y(1)/2D0
!  Time derivative:
!     S(2) = ( (dS0 + (S1 + dS1*T) + (2.D0*S2*T + dS2*T**2) +
!    *        (3.D0*S3*T**2 + dS3*T**3 ) + (4.D0*S4*T**3 + dS4*T**4 ) +
!    *        (5.D0*S5*T**4 + dS5*T**5) ) * DAS2R
!    *        - (X(2)*Y(1)/2.D0 + X(1)*Y(2)/2.D0) ) /(36525.D0*86400.D0)
!     S(2) = ( (dS0 + (S1 + dS1*T) + (2.D0*S2*T + dS2*T**2) +
!    *        (3.D0*S3*T**2 + dS3*T**3 ) + (4.D0*S4*T**3 + dS4*T**4 ) +
!    *        (5.D0*S5*T**4 + dS5*T**5) ) * DAS2R/(36525.D0*86400.D0)
!    *        - (X(2)*Y(1)/2.D0 + X(1)*Y(2)/2.D0) )
      S(2) = ( (dS0 + (S1*dTdt + dS1*T) + (2.D0*S2*T*dTdt + dS2*T**2) + &
     &        (3.D0*S3*T**2*dTdt + dS3*T**3 ) + &
     &        (4.D0*S4*T**3*dTdt + dS4*T**4 ) + &
     &        (5.D0*S5*T**4*dTdt + dS5*T**5) ) * DAS2R &
     &      - (X(2)*Y(1)/2.D0 + X(1)*Y(2)/2.D0) )
!
      END
!************************************************************************
!  Copyright (C) 2001
!  Standards Of Fundamental Astronomy Review Board
!  of the International Astronomical Union.
!
!  This routine is part of the International Astronomical Union's
!  SOFA (Standards of Fundamental Astronomy) software collection.
!  =====================
!  SOFA Software License
!  =====================
!
!  NOTICE TO USER:
!
!  BY USING THIS SOFTWARE YOU ACCEPT THE FOLLOWING TERMS AND CONDITIONS
!  WHICH APPLY TO ITS USE.
!
!  1. The Software is owned by the IAU SOFA Review Board ("the Board").
!
!  2. The Software is made available free of charge for use by:
!
!     a) private individuals for non-profit research; and
!
!     b) non-profit educational, academic and research institutions.
!
!  3. Commercial use of the Software is specifically excluded from the
!     terms and conditions of this license.  Commercial use of the
!     Software is subject to the prior written agreement of the Board on
!     terms to be agreed.
!
!  4. The provision of any version of the Software under the terms and
!     conditions specified herein does not imply that future versions
!     will also be made available under the same terms and conditions.
!
!  5. The user may modify the Software for his/her own purposes.  The
!     user may distribute the modified software provided that the Board
!     is informed and that a copy of the modified software is made
!     available to the Board on request.  All modifications made by the
!     user shall be clearly identified to show how the modified software
!     differs from the original Software, and the name(s) of the
!     affected routine(s) shall be changed.  The original SOFA Software
!     License text must be present.
!
!  6. In any published work produced by the user and which includes
!     results achieved by using the Software, the user shall acknowledge
!     that the Software was used in producing the information contained
!     in such publication.
!
!  7. The user may incorporate or embed the Software into other software
!     products which he/she may then give away free of charge but not
!     sell provided the user makes due acknowledgement of the use which
!     he/she has made of the Software in creating such software
!     products.  Any redistribution of the Software in this way shall be
!     made under the same terms and conditions under which the user
!     received it from the SOFA Center.
!
!  8. The user shall not cause the Software to be brought into
!     disrepute, either by misuse, or use for inappropriate tasks, or by
!     inappropriate modification.
!
!  9. The Software is provided to the user "as is" and the Board makes
!     no warranty as to its use or performance.   The Board does not and
!     cannot warrant the performance or results which the user may
!     obtain by using the Software.  The Board makes no warranties,
!     express or implied, as to non-infringement of third party rights,
!     merchantability, or fitness for any particular purpose.  In no
!     event will the Board be liable to the user for any consequential,
!     incidental, or special damages, including any lost profits or lost
!     savings, even if a Board representative has been advised of such
!     damages, or for any claim by any third party.
!
!  Correspondence concerning SOFA software should be addressed as
!  follows:
!
!     Internet email: sofa@rl.ac.uk
!     Postal address: IAU SOFA Center
!                     Rutherford Appleton Laboratory
!                     Chilton, Didcot, Oxon OX11 0QX
!                     United Kingdom
!
!************************************************************************
!
      SUBROUTINE NU2KA (CENT,DCENT,DATE1,DATE2,FA2K,FAD2K,DPSI,DEPS )
!
!  - - - - - - - -
!   N U 2 0 0 0 A
!  - - - - - - - -
!
!  Nutation, IAU 2000A model (MHB_2000 without FCN).
!
!  Annexe to IERS Conventions 2000, Chapter 5
!
!  Given:
!     DATE1,DATE2    d   TT date (JD = DATE1+DATE2)
!
!  Returned:
!     DPSI,DEPS      d   nutation (luni-solar + planetary, radians)
!
!  This revision:  2002 November 25
!
!*************************************************************************
!   Modified version of NU2000A for use in program Calc for VLBI analysis
!   and/or corrleation. Name changed to NU2KA. Input/outputs  modified.
!   Time derivatives of DPSI and DEPS added. D. Gordon, NASA/GSFC,
!   Aug. 2004.
!
!-----------------------------------------------------------------------
!
      IMPLICIT NONE
!
      REAL*8 CENT, DCENT, DATE1, DATE2, DPSI(2), DEPS(2)
!
! Fundamental arguments from subroutine NUTFA
      REAL*8 FA2K(14), FAD2K(14)
!     COMMON / NFA2K / FA2K, FAD2K
!
!  Arcseconds to radians
      DOUBLE PRECISION DAS2R
      PARAMETER ( DAS2R = 4.848136811095359935899141D-6 )
!
!  Milliarcseconds to radians
      DOUBLE PRECISION DMAS2R
      PARAMETER ( DMAS2R = DAS2R / 1.D3 )
!
!  Arc seconds in a full circle
      DOUBLE PRECISION TURNAS
      PARAMETER ( TURNAS = 1296000.D0 )
!
!  2Pi
      DOUBLE PRECISION D2PI
      PARAMETER ( D2PI = 6.283185307179586476925287D0 )
!
!  Units of 0.1 microarcsecond to radians
      DOUBLE PRECISION U2R
      PARAMETER ( U2R = DAS2R/1.D7 )
!
!  Reference epoch (J2000), JD
      DOUBLE PRECISION DJ0
      PARAMETER ( DJ0 = 2451545.D0 )
!
!  Days per Julian century
      DOUBLE PRECISION DJC
      PARAMETER ( DJC = 36525.D0 )
!
!  Miscellaneous
      DOUBLE PRECISION T, EL, ELP, F, D, OM, ARG, DP, DE, SARG, CARG, &
     &                 DPSILS, DEPSLS, DELP, &
     &                 AL, ALSU, AF, AD, AOM, ALME, ALVE, ALEA, ALMA, &
     &                 ALJU, ALSA, ALUR, ALNE, APA, DPSIPL, DEPSPL, &
     &                 DALNE
      REAL*8 dARG, dSARG, dCARG, dDP, dDE, dDPSILS, dDEPSLS, dAL, &
     &       dALSU, dAF, dAD, dAOM, dAPA, dDPSIPL, dDEPSPL
      INTEGER I, J
!
!  -------------------------
!  Luni-Solar nutation model
!  -------------------------
!
!  Number of terms in the luni-solar nutation model
      INTEGER NLS
      PARAMETER ( NLS = 678 )
!
!  Coefficients for fundamental arguments
      INTEGER NALS(5,NLS)
!
!  Longitude and obliquity coefficients
      DOUBLE PRECISION CLS(6,NLS)
!
!  ---------------
!  Planetary terms
!  ---------------
!
!  Number of terms in the planetary nutation model
      INTEGER NPL
      PARAMETER ( NPL = 687 )
!
!  Coefficients for fundamental arguments
      INTEGER NAPL(14,NPL)
!
!  Longitude and obliquity coefficients
      INTEGER ICPL(4,NPL)
!
!  ----------------------------------------
!  Tables of argument and term coefficients
!  ----------------------------------------
!
!  Luni-Solar argument multipliers:
!               L     L'    F     D     Om
!
      DATA ( ( NALS(I,J), I=1,5 ), J=  1, 10 ) / &
     &          0,    0,    0,    0,    1, &
     &          0,    0,    2,   -2,    2, &
     &          0,    0,    2,    0,    2, &
     &          0,    0,    0,    0,    2, &
     &          0,    1,    0,    0,    0, &
     &          0,    1,    2,   -2,    2, &
     &          1,    0,    0,    0,    0, &
     &          0,    0,    2,    0,    1, &
     &          1,    0,    2,    0,    2, &
     &          0,   -1,    2,   -2,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J= 11, 20 ) / &
     &          0,    0,    2,   -2,    1, &
     &         -1,    0,    2,    0,    2, &
     &         -1,    0,    0,    2,    0, &
     &          1,    0,    0,    0,    1, &
     &         -1,    0,    0,    0,    1, &
     &         -1,    0,    2,    2,    2, &
     &          1,    0,    2,    0,    1, &
     &         -2,    0,    2,    0,    1, &
     &          0,    0,    0,    2,    0, &
     &          0,    0,    2,    2,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J= 21, 30 ) / &
     &          0,   -2,    2,   -2,    2, &
     &         -2,    0,    0,    2,    0, &
     &          2,    0,    2,    0,    2, &
     &          1,    0,    2,   -2,    2, &
     &         -1,    0,    2,    0,    1, &
     &          2,    0,    0,    0,    0, &
     &          0,    0,    2,    0,    0, &
     &          0,    1,    0,    0,    1, &
     &         -1,    0,    0,    2,    1, &
     &          0,    2,    2,   -2,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J= 31, 40 ) / &
     &          0,    0,   -2,    2,    0, &
     &          1,    0,    0,   -2,    1, &
     &          0,   -1,    0,    0,    1, &
     &         -1,    0,    2,    2,    1, &
     &          0,    2,    0,    0,    0, &
     &          1,    0,    2,    2,    2, &
     &         -2,    0,    2,    0,    0, &
     &          0,    1,    2,    0,    2, &
     &          0,    0,    2,    2,    1, &
     &          0,   -1,    2,    0,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J= 41, 50 ) / &
     &          0,    0,    0,    2,    1, &
     &          1,    0,    2,   -2,    1, &
     &          2,    0,    2,   -2,    2, &
     &         -2,    0,    0,    2,    1, &
     &          2,    0,    2,    0,    1, &
     &          0,   -1,    2,   -2,    1, &
     &          0,    0,    0,   -2,    1, &
     &         -1,   -1,    0,    2,    0, &
     &          2,    0,    0,   -2,    1, &
     &          1,    0,    0,    2,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J= 51, 60 ) / &
     &          0,    1,    2,   -2,    1, &
     &          1,   -1,    0,    0,    0, &
     &         -2,    0,    2,    0,    2, &
     &          3,    0,    2,    0,    2, &
     &          0,   -1,    0,    2,    0, &
     &          1,   -1,    2,    0,    2, &
     &          0,    0,    0,    1,    0, &
     &         -1,   -1,    2,    2,    2, &
     &         -1,    0,    2,    0,    0, &
     &          0,   -1,    2,    2,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J= 61, 70 ) / &
     &         -2,    0,    0,    0,    1, &
     &          1,    1,    2,    0,    2, &
     &          2,    0,    0,    0,    1, &
     &         -1,    1,    0,    1,    0, &
     &          1,    1,    0,    0,    0, &
     &          1,    0,    2,    0,    0, &
     &         -1,    0,    2,   -2,    1, &
     &          1,    0,    0,    0,    2, &
     &         -1,    0,    0,    1,    0, &
     &          0,    0,    2,    1,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J= 71, 80 ) / &
     &         -1,    0,    2,    4,    2, &
     &         -1,    1,    0,    1,    1, &
     &          0,   -2,    2,   -2,    1, &
     &          1,    0,    2,    2,    1, &
     &         -2,    0,    2,    2,    2, &
     &         -1,    0,    0,    0,    2, &
     &          1,    1,    2,   -2,    2, &
     &         -2,    0,    2,    4,    2, &
     &         -1,    0,    4,    0,    2, &
     &          2,    0,    2,   -2,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J= 81, 90 ) / &
     &          2,    0,    2,    2,    2, &
     &          1,    0,    0,    2,    1, &
     &          3,    0,    0,    0,    0, &
     &          3,    0,    2,   -2,    2, &
     &          0,    0,    4,   -2,    2, &
     &          0,    1,    2,    0,    1, &
     &          0,    0,   -2,    2,    1, &
     &          0,    0,    2,   -2,    3, &
     &         -1,    0,    0,    4,    0, &
     &          2,    0,   -2,    0,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J= 91,100 ) / &
     &         -2,    0,    0,    4,    0, &
     &         -1,   -1,    0,    2,    1, &
     &         -1,    0,    0,    1,    1, &
     &          0,    1,    0,    0,    2, &
     &          0,    0,   -2,    0,    1, &
     &          0,   -1,    2,    0,    1, &
     &          0,    0,    2,   -1,    2, &
     &          0,    0,    2,    4,    2, &
     &         -2,   -1,    0,    2,    0, &
     &          1,    1,    0,   -2,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=101,110 ) / &
     &         -1,    1,    0,    2,    0, &
     &         -1,    1,    0,    1,    2, &
     &          1,   -1,    0,    0,    1, &
     &          1,   -1,    2,    2,    2, &
     &         -1,    1,    2,    2,    2, &
     &          3,    0,    2,    0,    1, &
     &          0,    1,   -2,    2,    0, &
     &         -1,    0,    0,   -2,    1, &
     &          0,    1,    2,    2,    2, &
     &         -1,   -1,    2,    2,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=111,120 ) / &
     &          0,   -1,    0,    0,    2, &
     &          1,    0,    2,   -4,    1, &
     &         -1,    0,   -2,    2,    0, &
     &          0,   -1,    2,    2,    1, &
     &          2,   -1,    2,    0,    2, &
     &          0,    0,    0,    2,    2, &
     &          1,   -1,    2,    0,    1, &
     &         -1,    1,    2,    0,    2, &
     &          0,    1,    0,    2,    0, &
     &          0,   -1,   -2,    2,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=121,130 ) / &
     &          0,    3,    2,   -2,    2, &
     &          0,    0,    0,    1,    1, &
     &         -1,    0,    2,    2,    0, &
     &          2,    1,    2,    0,    2, &
     &          1,    1,    0,    0,    1, &
     &          1,    1,    2,    0,    1, &
     &          2,    0,    0,    2,    0, &
     &          1,    0,   -2,    2,    0, &
     &         -1,    0,    0,    2,    2, &
     &          0,    1,    0,    1,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=131,140 ) / &
     &          0,    1,    0,   -2,    1, &
     &         -1,    0,    2,   -2,    2, &
     &          0,    0,    0,   -1,    1, &
     &         -1,    1,    0,    0,    1, &
     &          1,    0,    2,   -1,    2, &
     &          1,   -1,    0,    2,    0, &
     &          0,    0,    0,    4,    0, &
     &          1,    0,    2,    1,    2, &
     &          0,    0,    2,    1,    1, &
     &          1,    0,    0,   -2,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=141,150 ) / &
     &         -1,    0,    2,    4,    1, &
     &          1,    0,   -2,    0,    1, &
     &          1,    1,    2,   -2,    1, &
     &          0,    0,    2,    2,    0, &
     &         -1,    0,    2,   -1,    1, &
     &         -2,    0,    2,    2,    1, &
     &          4,    0,    2,    0,    2, &
     &          2,   -1,    0,    0,    0, &
     &          2,    1,    2,   -2,    2, &
     &          0,    1,    2,    1,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=151,160 ) / &
     &          1,    0,    4,   -2,    2, &
     &         -1,   -1,    0,    0,    1, &
     &          0,    1,    0,    2,    1, &
     &         -2,    0,    2,    4,    1, &
     &          2,    0,    2,    0,    0, &
     &          1,    0,    0,    1,    0, &
     &         -1,    0,    0,    4,    1, &
     &         -1,    0,    4,    0,    1, &
     &          2,    0,    2,    2,    1, &
     &          0,    0,    2,   -3,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=161,170 ) / &
     &         -1,   -2,    0,    2,    0, &
     &          2,    1,    0,    0,    0, &
     &          0,    0,    4,    0,    2, &
     &          0,    0,    0,    0,    3, &
     &          0,    3,    0,    0,    0, &
     &          0,    0,    2,   -4,    1, &
     &          0,   -1,    0,    2,    1, &
     &          0,    0,    0,    4,    1, &
     &         -1,   -1,    2,    4,    2, &
     &          1,    0,    2,    4,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=171,180 ) / &
     &         -2,    2,    0,    2,    0, &
     &         -2,   -1,    2,    0,    1, &
     &         -2,    0,    0,    2,    2, &
     &         -1,   -1,    2,    0,    2, &
     &          0,    0,    4,   -2,    1, &
     &          3,    0,    2,   -2,    1, &
     &         -2,   -1,    0,    2,    1, &
     &          1,    0,    0,   -1,    1, &
     &          0,   -2,    0,    2,    0, &
     &         -2,    0,    0,    4,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=181,190 ) / &
     &         -3,    0,    0,    0,    1, &
     &          1,    1,    2,    2,    2, &
     &          0,    0,    2,    4,    1, &
     &          3,    0,    2,    2,    2, &
     &         -1,    1,    2,   -2,    1, &
     &          2,    0,    0,   -4,    1, &
     &          0,    0,    0,   -2,    2, &
     &          2,    0,    2,   -4,    1, &
     &         -1,    1,    0,    2,    1, &
     &          0,    0,    2,   -1,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=191,200 ) / &
     &          0,   -2,    2,    2,    2, &
     &          2,    0,    0,    2,    1, &
     &          4,    0,    2,   -2,    2, &
     &          2,    0,    0,   -2,    2, &
     &          0,    2,    0,    0,    1, &
     &          1,    0,    0,   -4,    1, &
     &          0,    2,    2,   -2,    1, &
     &         -3,    0,    0,    4,    0, &
     &         -1,    1,    2,    0,    1, &
     &         -1,   -1,    0,    4,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=201,210 ) / &
     &         -1,   -2,    2,    2,    2, &
     &         -2,   -1,    2,    4,    2, &
     &          1,   -1,    2,    2,    1, &
     &         -2,    1,    0,    2,    0, &
     &         -2,    1,    2,    0,    1, &
     &          2,    1,    0,   -2,    1, &
     &         -3,    0,    2,    0,    1, &
     &         -2,    0,    2,   -2,    1, &
     &         -1,    1,    0,    2,    2, &
     &          0,   -1,    2,   -1,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=211,220 ) / &
     &         -1,    0,    4,   -2,    2, &
     &          0,   -2,    2,    0,    2, &
     &         -1,    0,    2,    1,    2, &
     &          2,    0,    0,    0,    2, &
     &          0,    0,    2,    0,    3, &
     &         -2,    0,    4,    0,    2, &
     &         -1,    0,   -2,    0,    1, &
     &         -1,    1,    2,    2,    1, &
     &          3,    0,    0,    0,    1, &
     &         -1,    0,    2,    3,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=221,230 ) / &
     &          2,   -1,    2,    0,    1, &
     &          0,    1,    2,    2,    1, &
     &          0,   -1,    2,    4,    2, &
     &          2,   -1,    2,    2,    2, &
     &          0,    2,   -2,    2,    0, &
     &         -1,   -1,    2,   -1,    1, &
     &          0,   -2,    0,    0,    1, &
     &          1,    0,    2,   -4,    2, &
     &          1,   -1,    0,   -2,    1, &
     &         -1,   -1,    2,    0,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=231,240 ) / &
     &          1,   -1,    2,   -2,    2, &
     &         -2,   -1,    0,    4,    0, &
     &         -1,    0,    0,    3,    0, &
     &         -2,   -1,    2,    2,    2, &
     &          0,    2,    2,    0,    2, &
     &          1,    1,    0,    2,    0, &
     &          2,    0,    2,   -1,    2, &
     &          1,    0,    2,    1,    1, &
     &          4,    0,    0,    0,    0, &
     &          2,    1,    2,    0,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=241,250 ) / &
     &          3,   -1,    2,    0,    2, &
     &         -2,    2,    0,    2,    1, &
     &          1,    0,    2,   -3,    1, &
     &          1,    1,    2,   -4,    1, &
     &         -1,   -1,    2,   -2,    1, &
     &          0,   -1,    0,   -1,    1, &
     &          0,   -1,    0,   -2,    1, &
     &         -2,    0,    0,    0,    2, &
     &         -2,    0,   -2,    2,    0, &
     &         -1,    0,   -2,    4,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=251,260 ) / &
     &          1,   -2,    0,    0,    0, &
     &          0,    1,    0,    1,    1, &
     &         -1,    2,    0,    2,    0, &
     &          1,   -1,    2,   -2,    1, &
     &          1,    2,    2,   -2,    2, &
     &          2,   -1,    2,   -2,    2, &
     &          1,    0,    2,   -1,    1, &
     &          2,    1,    2,   -2,    1, &
     &         -2,    0,    0,   -2,    1, &
     &          1,   -2,    2,    0,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=261,270 ) / &
     &          0,    1,    2,    1,    1, &
     &          1,    0,    4,   -2,    1, &
     &         -2,    0,    4,    2,    2, &
     &          1,    1,    2,    1,    2, &
     &          1,    0,    0,    4,    0, &
     &          1,    0,    2,    2,    0, &
     &          2,    0,    2,    1,    2, &
     &          3,    1,    2,    0,    2, &
     &          4,    0,    2,    0,    1, &
     &         -2,   -1,    2,    0,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=271,280 ) / &
     &          0,    1,   -2,    2,    1, &
     &          1,    0,   -2,    1,    0, &
     &          0,   -1,   -2,    2,    1, &
     &          2,   -1,    0,   -2,    1, &
     &         -1,    0,    2,   -1,    2, &
     &          1,    0,    2,   -3,    2, &
     &          0,    1,    2,   -2,    3, &
     &          0,    0,    2,   -3,    1, &
     &         -1,    0,   -2,    2,    1, &
     &          0,    0,    2,   -4,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=281,290 ) / &
     &         -2,    1,    0,    0,    1, &
     &         -1,    0,    0,   -1,    1, &
     &          2,    0,    2,   -4,    2, &
     &          0,    0,    4,   -4,    4, &
     &          0,    0,    4,   -4,    2, &
     &         -1,   -2,    0,    2,    1, &
     &         -2,    0,    0,    3,    0, &
     &          1,    0,   -2,    2,    1, &
     &         -3,    0,    2,    2,    2, &
     &         -3,    0,    2,    2,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=291,300 ) / &
     &         -2,    0,    2,    2,    0, &
     &          2,   -1,    0,    0,    1, &
     &         -2,    1,    2,    2,    2, &
     &          1,    1,    0,    1,    0, &
     &          0,    1,    4,   -2,    2, &
     &         -1,    1,    0,   -2,    1, &
     &          0,    0,    0,   -4,    1, &
     &          1,   -1,    0,    2,    1, &
     &          1,    1,    0,    2,    1, &
     &         -1,    2,    2,    2,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=301,310 ) / &
     &          3,    1,    2,   -2,    2, &
     &          0,   -1,    0,    4,    0, &
     &          2,   -1,    0,    2,    0, &
     &          0,    0,    4,    0,    1, &
     &          2,    0,    4,   -2,    2, &
     &         -1,   -1,    2,    4,    1, &
     &          1,    0,    0,    4,    1, &
     &          1,   -2,    2,    2,    2, &
     &          0,    0,    2,    3,    2, &
     &         -1,    1,    2,    4,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=311,320 ) / &
     &          3,    0,    0,    2,    0, &
     &         -1,    0,    4,    2,    2, &
     &          1,    1,    2,    2,    1, &
     &         -2,    0,    2,    6,    2, &
     &          2,    1,    2,    2,    2, &
     &         -1,    0,    2,    6,    2, &
     &          1,    0,    2,    4,    1, &
     &          2,    0,    2,    4,    2, &
     &          1,    1,   -2,    1,    0, &
     &         -3,    1,    2,    1,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=321,330 ) / &
     &          2,    0,   -2,    0,    2, &
     &         -1,    0,    0,    1,    2, &
     &         -4,    0,    2,    2,    1, &
     &         -1,   -1,    0,    1,    0, &
     &          0,    0,   -2,    2,    2, &
     &          1,    0,    0,   -1,    2, &
     &          0,   -1,    2,   -2,    3, &
     &         -2,    1,    2,    0,    0, &
     &          0,    0,    2,   -2,    4, &
     &         -2,   -2,    0,    2,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=331,340 ) / &
     &         -2,    0,   -2,    4,    0, &
     &          0,   -2,   -2,    2,    0, &
     &          1,    2,    0,   -2,    1, &
     &          3,    0,    0,   -4,    1, &
     &         -1,    1,    2,   -2,    2, &
     &          1,   -1,    2,   -4,    1, &
     &          1,    1,    0,   -2,    2, &
     &         -3,    0,    2,    0,    0, &
     &         -3,    0,    2,    0,    2, &
     &         -2,    0,    0,    1,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=341,350 ) / &
     &          0,    0,   -2,    1,    0, &
     &         -3,    0,    0,    2,    1, &
     &         -1,   -1,   -2,    2,    0, &
     &          0,    1,    2,   -4,    1, &
     &          2,    1,    0,   -4,    1, &
     &          0,    2,    0,   -2,    1, &
     &          1,    0,    0,   -3,    1, &
     &         -2,    0,    2,   -2,    2, &
     &         -2,   -1,    0,    0,    1, &
     &         -4,    0,    0,    2,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=351,360 ) / &
     &          1,    1,    0,   -4,    1, &
     &         -1,    0,    2,   -4,    1, &
     &          0,    0,    4,   -4,    1, &
     &          0,    3,    2,   -2,    2, &
     &         -3,   -1,    0,    4,    0, &
     &         -3,    0,    0,    4,    1, &
     &          1,   -1,   -2,    2,    0, &
     &         -1,   -1,    0,    2,    2, &
     &          1,   -2,    0,    0,    1, &
     &          1,   -1,    0,    0,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=361,370 ) / &
     &          0,    0,    0,    1,    2, &
     &         -1,   -1,    2,    0,    0, &
     &          1,   -2,    2,   -2,    2, &
     &          0,   -1,    2,   -1,    1, &
     &         -1,    0,    2,    0,    3, &
     &          1,    1,    0,    0,    2, &
     &         -1,    1,    2,    0,    0, &
     &          1,    2,    0,    0,    0, &
     &         -1,    2,    2,    0,    2, &
     &         -1,    0,    4,   -2,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=371,380 ) / &
     &          3,    0,    2,   -4,    2, &
     &          1,    2,    2,   -2,    1, &
     &          1,    0,    4,   -4,    2, &
     &         -2,   -1,    0,    4,    1, &
     &          0,   -1,    0,    2,    2, &
     &         -2,    1,    0,    4,    0, &
     &         -2,   -1,    2,    2,    1, &
     &          2,    0,   -2,    2,    0, &
     &          1,    0,    0,    1,    1, &
     &          0,    1,    0,    2,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=381,390 ) / &
     &          1,   -1,    2,   -1,    2, &
     &         -2,    0,    4,    0,    1, &
     &          2,    1,    0,    0,    1, &
     &          0,    1,    2,    0,    0, &
     &          0,   -1,    4,   -2,    2, &
     &          0,    0,    4,   -2,    4, &
     &          0,    2,    2,    0,    1, &
     &         -3,    0,    0,    6,    0, &
     &         -1,   -1,    0,    4,    1, &
     &          1,   -2,    0,    2,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=391,400 ) / &
     &         -1,    0,    0,    4,    2, &
     &         -1,   -2,    2,    2,    1, &
     &         -1,    0,    0,   -2,    2, &
     &          1,    0,   -2,   -2,    1, &
     &          0,    0,   -2,   -2,    1, &
     &         -2,    0,   -2,    0,    1, &
     &          0,    0,    0,    3,    1, &
     &          0,    0,    0,    3,    0, &
     &         -1,    1,    0,    4,    0, &
     &         -1,   -1,    2,    2,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=401,410 ) / &
     &         -2,    0,    2,    3,    2, &
     &          1,    0,    0,    2,    2, &
     &          0,   -1,    2,    1,    2, &
     &          3,   -1,    0,    0,    0, &
     &          2,    0,    0,    1,    0, &
     &          1,   -1,    2,    0,    0, &
     &          0,    0,    2,    1,    0, &
     &          1,    0,    2,    0,    3, &
     &          3,    1,    0,    0,    0, &
     &          3,   -1,    2,   -2,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=411,420 ) / &
     &          2,    0,    2,   -1,    1, &
     &          1,    1,    2,    0,    0, &
     &          0,    0,    4,   -1,    2, &
     &          1,    2,    2,    0,    2, &
     &         -2,    0,    0,    6,    0, &
     &          0,   -1,    0,    4,    1, &
     &         -2,   -1,    2,    4,    1, &
     &          0,   -2,    2,    2,    1, &
     &          0,   -1,    2,    2,    0, &
     &         -1,    0,    2,    3,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=421,430 ) / &
     &         -2,    1,    2,    4,    2, &
     &          2,    0,    0,    2,    2, &
     &          2,   -2,    2,    0,    2, &
     &         -1,    1,    2,    3,    2, &
     &          3,    0,    2,   -1,    2, &
     &          4,    0,    2,   -2,    1, &
     &         -1,    0,    0,    6,    0, &
     &         -1,   -2,    2,    4,    2, &
     &         -3,    0,    2,    6,    2, &
     &         -1,    0,    2,    4,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=431,440 ) / &
     &          3,    0,    0,    2,    1, &
     &          3,   -1,    2,    0,    1, &
     &          3,    0,    2,    0,    0, &
     &          1,    0,    4,    0,    2, &
     &          5,    0,    2,   -2,    2, &
     &          0,   -1,    2,    4,    1, &
     &          2,   -1,    2,    2,    1, &
     &          0,    1,    2,    4,    2, &
     &          1,   -1,    2,    4,    2, &
     &          3,   -1,    2,    2,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=441,450 ) / &
     &          3,    0,    2,    2,    1, &
     &          5,    0,    2,    0,    2, &
     &          0,    0,    2,    6,    2, &
     &          4,    0,    2,    2,    2, &
     &          0,   -1,    1,   -1,    1, &
     &         -1,    0,    1,    0,    3, &
     &          0,   -2,    2,   -2,    3, &
     &          1,    0,   -1,    0,    1, &
     &          2,   -2,    0,   -2,    1, &
     &         -1,    0,    1,    0,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=451,460 ) / &
     &         -1,    0,    1,    0,    1, &
     &         -1,   -1,    2,   -1,    2, &
     &         -2,    2,    0,    2,    2, &
     &         -1,    0,    1,    0,    0, &
     &         -4,    1,    2,    2,    2, &
     &         -3,    0,    2,    1,    1, &
     &         -2,   -1,    2,    0,    2, &
     &          1,    0,   -2,    1,    1, &
     &          2,   -1,   -2,    0,    1, &
     &         -4,    0,    2,    2,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=461,470 ) / &
     &         -3,    1,    0,    3,    0, &
     &         -1,    0,   -1,    2,    0, &
     &          0,   -2,    0,    0,    2, &
     &          0,   -2,    0,    0,    2, &
     &         -3,    0,    0,    3,    0, &
     &         -2,   -1,    0,    2,    2, &
     &         -1,    0,   -2,    3,    0, &
     &         -4,    0,    0,    4,    0, &
     &          2,    1,   -2,    0,    1, &
     &          2,   -1,    0,   -2,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=471,480 ) / &
     &          0,    0,    1,   -1,    0, &
     &         -1,    2,    0,    1,    0, &
     &         -2,    1,    2,    0,    2, &
     &          1,    1,    0,   -1,    1, &
     &          1,    0,    1,   -2,    1, &
     &          0,    2,    0,    0,    2, &
     &          1,   -1,    2,   -3,    1, &
     &         -1,    1,    2,   -1,    1, &
     &         -2,    0,    4,   -2,    2, &
     &         -2,    0,    4,   -2,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=481,490 ) / &
     &         -2,   -2,    0,    2,    1, &
     &         -2,    0,   -2,    4,    0, &
     &          1,    2,    2,   -4,    1, &
     &          1,    1,    2,   -4,    2, &
     &         -1,    2,    2,   -2,    1, &
     &          2,    0,    0,   -3,    1, &
     &         -1,    2,    0,    0,    1, &
     &          0,    0,    0,   -2,    0, &
     &         -1,   -1,    2,   -2,    2, &
     &         -1,    1,    0,    0,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=491,500 ) / &
     &          0,    0,    0,   -1,    2, &
     &         -2,    1,    0,    1,    0, &
     &          1,   -2,    0,   -2,    1, &
     &          1,    0,   -2,    0,    2, &
     &         -3,    1,    0,    2,    0, &
     &         -1,    1,   -2,    2,    0, &
     &         -1,   -1,    0,    0,    2, &
     &         -3,    0,    0,    2,    0, &
     &         -3,   -1,    0,    2,    0, &
     &          2,    0,    2,   -6,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=501,510 ) / &
     &          0,    1,    2,   -4,    2, &
     &          2,    0,    0,   -4,    2, &
     &         -2,    1,    2,   -2,    1, &
     &          0,   -1,    2,   -4,    1, &
     &          0,    1,    0,   -2,    2, &
     &         -1,    0,    0,   -2,    0, &
     &          2,    0,   -2,   -2,    1, &
     &         -4,    0,    2,    0,    1, &
     &         -1,   -1,    0,   -1,    1, &
     &          0,    0,   -2,    0,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=511,520 ) / &
     &         -3,    0,    0,    1,    0, &
     &         -1,    0,   -2,    1,    0, &
     &         -2,    0,   -2,    2,    1, &
     &          0,    0,   -4,    2,    0, &
     &         -2,   -1,   -2,    2,    0, &
     &          1,    0,    2,   -6,    1, &
     &         -1,    0,    2,   -4,    2, &
     &          1,    0,    0,   -4,    2, &
     &          2,    1,    2,   -4,    2, &
     &          2,    1,    2,   -4,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=521,530 ) / &
     &          0,    1,    4,   -4,    4, &
     &          0,    1,    4,   -4,    2, &
     &         -1,   -1,   -2,    4,    0, &
     &         -1,   -3,    0,    2,    0, &
     &         -1,    0,   -2,    4,    1, &
     &         -2,   -1,    0,    3,    0, &
     &          0,    0,   -2,    3,    0, &
     &         -2,    0,    0,    3,    1, &
     &          0,   -1,    0,    1,    0, &
     &         -3,    0,    2,    2,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=531,540 ) / &
     &          1,    1,   -2,    2,    0, &
     &         -1,    1,    0,    2,    2, &
     &          1,   -2,    2,   -2,    1, &
     &          0,    0,    1,    0,    2, &
     &          0,    0,    1,    0,    1, &
     &          0,    0,    1,    0,    0, &
     &         -1,    2,    0,    2,    1, &
     &          0,    0,    2,    0,    2, &
     &         -2,    0,    2,    0,    2, &
     &          2,    0,    0,   -1,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=541,550 ) / &
     &          3,    0,    0,   -2,    1, &
     &          1,    0,    2,   -2,    3, &
     &          1,    2,    0,    0,    1, &
     &          2,    0,    2,   -3,    2, &
     &         -1,    1,    4,   -2,    2, &
     &         -2,   -2,    0,    4,    0, &
     &          0,   -3,    0,    2,    0, &
     &          0,    0,   -2,    4,    0, &
     &         -1,   -1,    0,    3,    0, &
     &         -2,    0,    0,    4,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=551,560 ) / &
     &         -1,    0,    0,    3,    1, &
     &          2,   -2,    0,    0,    0, &
     &          1,   -1,    0,    1,    0, &
     &         -1,    0,    0,    2,    0, &
     &          0,   -2,    2,    0,    1, &
     &         -1,    0,    1,    2,    1, &
     &         -1,    1,    0,    3,    0, &
     &         -1,   -1,    2,    1,    2, &
     &          0,   -1,    2,    0,    0, &
     &         -2,    1,    2,    2,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=561,570 ) / &
     &          2,   -2,    2,   -2,    2, &
     &          1,    1,    0,    1,    1, &
     &          1,    0,    1,    0,    1, &
     &          1,    0,    1,    0,    0, &
     &          0,    2,    0,    2,    0, &
     &          2,   -1,    2,   -2,    1, &
     &          0,   -1,    4,   -2,    1, &
     &          0,    0,    4,   -2,    3, &
     &          0,    1,    4,   -2,    1, &
     &          4,    0,    2,   -4,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=571,580 ) / &
     &          2,    2,    2,   -2,    2, &
     &          2,    0,    4,   -4,    2, &
     &         -1,   -2,    0,    4,    0, &
     &         -1,   -3,    2,    2,    2, &
     &         -3,    0,    2,    4,    2, &
     &         -3,    0,    2,   -2,    1, &
     &         -1,   -1,    0,   -2,    1, &
     &         -3,    0,    0,    0,    2, &
     &         -3,    0,   -2,    2,    0, &
     &          0,    1,    0,   -4,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=581,590 ) / &
     &         -2,    1,    0,   -2,    1, &
     &         -4,    0,    0,    0,    1, &
     &         -1,    0,    0,   -4,    1, &
     &         -3,    0,    0,   -2,    1, &
     &          0,    0,    0,    3,    2, &
     &         -1,    1,    0,    4,    1, &
     &          1,   -2,    2,    0,    1, &
     &          0,    1,    0,    3,    0, &
     &         -1,    0,    2,    2,    3, &
     &          0,    0,    2,    2,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=591,600 ) / &
     &         -2,    0,    2,    2,    2, &
     &         -1,    1,    2,    2,    0, &
     &          3,    0,    0,    0,    2, &
     &          2,    1,    0,    1,    0, &
     &          2,   -1,    2,   -1,    2, &
     &          0,    0,    2,    0,    1, &
     &          0,    0,    3,    0,    3, &
     &          0,    0,    3,    0,    2, &
     &         -1,    2,    2,    2,    1, &
     &         -1,    0,    4,    0,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=601,610 ) / &
     &          1,    2,    2,    0,    1, &
     &          3,    1,    2,   -2,    1, &
     &          1,    1,    4,   -2,    2, &
     &         -2,   -1,    0,    6,    0, &
     &          0,   -2,    0,    4,    0, &
     &         -2,    0,    0,    6,    1, &
     &         -2,   -2,    2,    4,    2, &
     &          0,   -3,    2,    2,    2, &
     &          0,    0,    0,    4,    2, &
     &         -1,   -1,    2,    3,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=611,620 ) / &
     &         -2,    0,    2,    4,    0, &
     &          2,   -1,    0,    2,    1, &
     &          1,    0,    0,    3,    0, &
     &          0,    1,    0,    4,    1, &
     &          0,    1,    0,    4,    0, &
     &          1,   -1,    2,    1,    2, &
     &          0,    0,    2,    2,    3, &
     &          1,    0,    2,    2,    2, &
     &         -1,    0,    2,    2,    2, &
     &         -2,    0,    4,    2,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=621,630 ) / &
     &          2,    1,    0,    2,    1, &
     &          2,    1,    0,    2,    0, &
     &          2,   -1,    2,    0,    0, &
     &          1,    0,    2,    1,    0, &
     &          0,    1,    2,    2,    0, &
     &          2,    0,    2,    0,    3, &
     &          3,    0,    2,    0,    2, &
     &          1,    0,    2,    0,    2, &
     &          1,    0,    3,    0,    3, &
     &          1,    1,    2,    1,    1 /
      DATA ( ( NALS(I,J), I=1,5 ), J=631,640 ) / &
     &          0,    2,    2,    2,    2, &
     &          2,    1,    2,    0,    0, &
     &          2,    0,    4,   -2,    1, &
     &          4,    1,    2,   -2,    2, &
     &         -1,   -1,    0,    6,    0, &
     &         -3,   -1,    2,    6,    2, &
     &         -1,    0,    0,    6,    1, &
     &         -3,    0,    2,    6,    1, &
     &          1,   -1,    0,    4,    1, &
     &          1,   -1,    0,    4,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=641,650 ) / &
     &         -2,    0,    2,    5,    2, &
     &          1,   -2,    2,    2,    1, &
     &          3,   -1,    0,    2,    0, &
     &          1,   -1,    2,    2,    0, &
     &          0,    0,    2,    3,    1, &
     &         -1,    1,    2,    4,    1, &
     &          0,    1,    2,    3,    2, &
     &         -1,    0,    4,    2,    1, &
     &          2,    0,    2,    1,    1, &
     &          5,    0,    0,    0,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=651,660 ) / &
     &          2,    1,    2,    1,    2, &
     &          1,    0,    4,    0,    1, &
     &          3,    1,    2,    0,    1, &
     &          3,    0,    4,   -2,    2, &
     &         -2,   -1,    2,    6,    2, &
     &          0,    0,    0,    6,    0, &
     &          0,   -2,    2,    4,    2, &
     &         -2,    0,    2,    6,    1, &
     &          2,    0,    0,    4,    1, &
     &          2,    0,    0,    4,    0 /
      DATA ( ( NALS(I,J), I=1,5 ), J=661,670 ) / &
     &          2,   -2,    2,    2,    2, &
     &          0,    0,    2,    4,    0, &
     &          1,    0,    2,    3,    2, &
     &          4,    0,    0,    2,    0, &
     &          2,    0,    2,    2,    0, &
     &          0,    0,    4,    2,    2, &
     &          4,   -1,    2,    0,    2, &
     &          3,    0,    2,    1,    2, &
     &          2,    1,    2,    2,    1, &
     &          4,    1,    2,    0,    2 /
      DATA ( ( NALS(I,J), I=1,5 ), J=671,678 ) / &
     &         -1,   -1,    2,    6,    2, &
     &         -1,    0,    2,    6,    1, &
     &          1,   -1,    2,    4,    1, &
     &          1,    1,    2,    4,    2, &
     &          3,    1,    2,    2,    2, &
     &          5,    0,    2,    0,    1, &
     &          2,   -1,    2,    4,    2, &
     &          2,    0,    2,    4,    1 /
!
!  Luni-Solar nutation coefficients, unit 1e-7 arcsec:
!  longitude (sin, t*sin, cos), obliquity (cos, t*cos, sin)
!
      DATA ( ( CLS(I,J), I=1,6 ), J=  1, 10 ) / &
     & -172064161D0, -174666D0,  33386D0, 92052331D0,  9086D0, 15377D0, &
     &  -13170906D0,   -1675D0, -13696D0,  5730336D0, -3015D0, -4587D0, &
     &   -2276413D0,    -234D0,   2796D0,   978459D0,  -485D0,  1374D0, &
     &    2074554D0,     207D0,   -698D0,  -897492D0,   470D0,  -291D0, &
     &    1475877D0,   -3633D0,  11817D0,    73871D0,  -184D0, -1924D0, &
     &    -516821D0,    1226D0,   -524D0,   224386D0,  -677D0,  -174D0, &
     &     711159D0,      73D0,   -872D0,    -6750D0,     0D0,   358D0, &
     &    -387298D0,    -367D0,    380D0,   200728D0,    18D0,   318D0, &
     &    -301461D0,     -36D0,    816D0,   129025D0,   -63D0,   367D0, &
     &     215829D0,    -494D0,    111D0,   -95929D0,   299D0,   132D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J= 11, 20 ) / &
     &     128227D0,     137D0,    181D0,   -68982D0,    -9D0,    39D0, &
     &     123457D0,      11D0,     19D0,   -53311D0,    32D0,    -4D0, &
     &     156994D0,      10D0,   -168D0,    -1235D0,     0D0,    82D0, &
     &      63110D0,      63D0,     27D0,   -33228D0,     0D0,    -9D0, &
     &     -57976D0,     -63D0,   -189D0,    31429D0,     0D0,   -75D0, &
     &     -59641D0,     -11D0,    149D0,    25543D0,   -11D0,    66D0, &
     &     -51613D0,     -42D0,    129D0,    26366D0,     0D0,    78D0, &
     &      45893D0,      50D0,     31D0,   -24236D0,   -10D0,    20D0, &
     &      63384D0,      11D0,   -150D0,    -1220D0,     0D0,    29D0, &
     &     -38571D0,      -1D0,    158D0,    16452D0,   -11D0,    68D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J= 21, 30 ) / &
     &      32481D0,       0D0,      0D0,   -13870D0,     0D0,     0D0, &
     &     -47722D0,       0D0,    -18D0,      477D0,     0D0,   -25D0, &
     &     -31046D0,      -1D0,    131D0,    13238D0,   -11D0,    59D0, &
     &      28593D0,       0D0,     -1D0,   -12338D0,    10D0,    -3D0, &
     &      20441D0,      21D0,     10D0,   -10758D0,     0D0,    -3D0, &
     &      29243D0,       0D0,    -74D0,     -609D0,     0D0,    13D0, &
     &      25887D0,       0D0,    -66D0,     -550D0,     0D0,    11D0, &
     &     -14053D0,     -25D0,     79D0,     8551D0,    -2D0,   -45D0, &
     &      15164D0,      10D0,     11D0,    -8001D0,     0D0,    -1D0, &
     &     -15794D0,      72D0,    -16D0,     6850D0,   -42D0,    -5D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J= 31, 40 ) / &
     &      21783D0,       0D0,     13D0,     -167D0,     0D0,    13D0, &
     &     -12873D0,     -10D0,    -37D0,     6953D0,     0D0,   -14D0, &
     &     -12654D0,      11D0,     63D0,     6415D0,     0D0,    26D0, &
     &     -10204D0,       0D0,     25D0,     5222D0,     0D0,    15D0, &
     &      16707D0,     -85D0,    -10D0,      168D0,    -1D0,    10D0, &
     &      -7691D0,       0D0,     44D0,     3268D0,     0D0,    19D0, &
     &     -11024D0,       0D0,    -14D0,      104D0,     0D0,     2D0, &
     &       7566D0,     -21D0,    -11D0,    -3250D0,     0D0,    -5D0, &
     &      -6637D0,     -11D0,     25D0,     3353D0,     0D0,    14D0, &
     &      -7141D0,      21D0,      8D0,     3070D0,     0D0,     4D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J= 41, 50 ) / &
     &      -6302D0,     -11D0,      2D0,     3272D0,     0D0,     4D0, &
     &       5800D0,      10D0,      2D0,    -3045D0,     0D0,    -1D0, &
     &       6443D0,       0D0,     -7D0,    -2768D0,     0D0,    -4D0, &
     &      -5774D0,     -11D0,    -15D0,     3041D0,     0D0,    -5D0, &
     &      -5350D0,       0D0,     21D0,     2695D0,     0D0,    12D0, &
     &      -4752D0,     -11D0,     -3D0,     2719D0,     0D0,    -3D0, &
     &      -4940D0,     -11D0,    -21D0,     2720D0,     0D0,    -9D0, &
     &       7350D0,       0D0,     -8D0,      -51D0,     0D0,     4D0, &
     &       4065D0,       0D0,      6D0,    -2206D0,     0D0,     1D0, &
     &       6579D0,       0D0,    -24D0,     -199D0,     0D0,     2D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J= 51, 60 ) / &
     &       3579D0,       0D0,      5D0,    -1900D0,     0D0,     1D0, &
     &       4725D0,       0D0,     -6D0,      -41D0,     0D0,     3D0, &
     &      -3075D0,       0D0,     -2D0,     1313D0,     0D0,    -1D0, &
     &      -2904D0,       0D0,     15D0,     1233D0,     0D0,     7D0, &
     &       4348D0,       0D0,    -10D0,      -81D0,     0D0,     2D0, &
     &      -2878D0,       0D0,      8D0,     1232D0,     0D0,     4D0, &
     &      -4230D0,       0D0,      5D0,      -20D0,     0D0,    -2D0, &
     &      -2819D0,       0D0,      7D0,     1207D0,     0D0,     3D0, &
     &      -4056D0,       0D0,      5D0,       40D0,     0D0,    -2D0, &
     &      -2647D0,       0D0,     11D0,     1129D0,     0D0,     5D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J= 61, 70 ) / &
     &      -2294D0,       0D0,    -10D0,     1266D0,     0D0,    -4D0, &
     &       2481D0,       0D0,     -7D0,    -1062D0,     0D0,    -3D0, &
     &       2179D0,       0D0,     -2D0,    -1129D0,     0D0,    -2D0, &
     &       3276D0,       0D0,      1D0,       -9D0,     0D0,     0D0, &
     &      -3389D0,       0D0,      5D0,       35D0,     0D0,    -2D0, &
     &       3339D0,       0D0,    -13D0,     -107D0,     0D0,     1D0, &
     &      -1987D0,       0D0,     -6D0,     1073D0,     0D0,    -2D0, &
     &      -1981D0,       0D0,      0D0,      854D0,     0D0,     0D0, &
     &       4026D0,       0D0,   -353D0,     -553D0,     0D0,  -139D0, &
     &       1660D0,       0D0,     -5D0,     -710D0,     0D0,    -2D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J= 71, 80 ) / &
     &      -1521D0,       0D0,      9D0,      647D0,     0D0,     4D0, &
     &       1314D0,       0D0,      0D0,     -700D0,     0D0,     0D0, &
     &      -1283D0,       0D0,      0D0,      672D0,     0D0,     0D0, &
     &      -1331D0,       0D0,      8D0,      663D0,     0D0,     4D0, &
     &       1383D0,       0D0,     -2D0,     -594D0,     0D0,    -2D0, &
     &       1405D0,       0D0,      4D0,     -610D0,     0D0,     2D0, &
     &       1290D0,       0D0,      0D0,     -556D0,     0D0,     0D0, &
     &      -1214D0,       0D0,      5D0,      518D0,     0D0,     2D0, &
     &       1146D0,       0D0,     -3D0,     -490D0,     0D0,    -1D0, &
     &       1019D0,       0D0,     -1D0,     -527D0,     0D0,    -1D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J= 81, 90 ) / &
     &      -1100D0,       0D0,      9D0,      465D0,     0D0,     4D0, &
     &       -970D0,       0D0,      2D0,      496D0,     0D0,     1D0, &
     &       1575D0,       0D0,     -6D0,      -50D0,     0D0,     0D0, &
     &        934D0,       0D0,     -3D0,     -399D0,     0D0,    -1D0, &
     &        922D0,       0D0,     -1D0,     -395D0,     0D0,    -1D0, &
     &        815D0,       0D0,     -1D0,     -422D0,     0D0,    -1D0, &
     &        834D0,       0D0,      2D0,     -440D0,     0D0,     1D0, &
     &       1248D0,       0D0,      0D0,     -170D0,     0D0,     1D0, &
     &       1338D0,       0D0,     -5D0,      -39D0,     0D0,     0D0, &
     &        716D0,       0D0,     -2D0,     -389D0,     0D0,    -1D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J= 91,100 ) / &
     &       1282D0,       0D0,     -3D0,      -23D0,     0D0,     1D0, &
     &        742D0,       0D0,      1D0,     -391D0,     0D0,     0D0, &
     &       1020D0,       0D0,    -25D0,     -495D0,     0D0,   -10D0, &
     &        715D0,       0D0,     -4D0,     -326D0,     0D0,     2D0, &
     &       -666D0,       0D0,     -3D0,      369D0,     0D0,    -1D0, &
     &       -667D0,       0D0,      1D0,      346D0,     0D0,     1D0, &
     &       -704D0,       0D0,      0D0,      304D0,     0D0,     0D0, &
     &       -694D0,       0D0,      5D0,      294D0,     0D0,     2D0, &
     &      -1014D0,       0D0,     -1D0,        4D0,     0D0,    -1D0, &
     &       -585D0,       0D0,     -2D0,      316D0,     0D0,    -1D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=101,110 ) / &
     &       -949D0,       0D0,      1D0,        8D0,     0D0,    -1D0, &
     &       -595D0,       0D0,      0D0,      258D0,     0D0,     0D0, &
     &        528D0,       0D0,      0D0,     -279D0,     0D0,     0D0, &
     &       -590D0,       0D0,      4D0,      252D0,     0D0,     2D0, &
     &        570D0,       0D0,     -2D0,     -244D0,     0D0,    -1D0, &
     &       -502D0,       0D0,      3D0,      250D0,     0D0,     2D0, &
     &       -875D0,       0D0,      1D0,       29D0,     0D0,     0D0, &
     &       -492D0,       0D0,     -3D0,      275D0,     0D0,    -1D0, &
     &        535D0,       0D0,     -2D0,     -228D0,     0D0,    -1D0, &
     &       -467D0,       0D0,      1D0,      240D0,     0D0,     1D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=111,120 ) / &
     &        591D0,       0D0,      0D0,     -253D0,     0D0,     0D0, &
     &       -453D0,       0D0,     -1D0,      244D0,     0D0,    -1D0, &
     &        766D0,       0D0,      1D0,        9D0,     0D0,     0D0, &
     &       -446D0,       0D0,      2D0,      225D0,     0D0,     1D0, &
     &       -488D0,       0D0,      2D0,      207D0,     0D0,     1D0, &
     &       -468D0,       0D0,      0D0,      201D0,     0D0,     0D0, &
     &       -421D0,       0D0,      1D0,      216D0,     0D0,     1D0, &
     &        463D0,       0D0,      0D0,     -200D0,     0D0,     0D0, &
     &       -673D0,       0D0,      2D0,       14D0,     0D0,     0D0, &
     &        658D0,       0D0,      0D0,       -2D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=121,130 ) / &
     &       -438D0,       0D0,      0D0,      188D0,     0D0,     0D0, &
     &       -390D0,       0D0,      0D0,      205D0,     0D0,     0D0, &
     &        639D0,     -11D0,     -2D0,      -19D0,     0D0,     0D0, &
     &        412D0,       0D0,     -2D0,     -176D0,     0D0,    -1D0, &
     &       -361D0,       0D0,      0D0,      189D0,     0D0,     0D0, &
     &        360D0,       0D0,     -1D0,     -185D0,     0D0,    -1D0, &
     &        588D0,       0D0,     -3D0,      -24D0,     0D0,     0D0, &
     &       -578D0,       0D0,      1D0,        5D0,     0D0,     0D0, &
     &       -396D0,       0D0,      0D0,      171D0,     0D0,     0D0, &
     &        565D0,       0D0,     -1D0,       -6D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=131,140 ) / &
     &       -335D0,       0D0,     -1D0,      184D0,     0D0,    -1D0, &
     &        357D0,       0D0,      1D0,     -154D0,     0D0,     0D0, &
     &        321D0,       0D0,      1D0,     -174D0,     0D0,     0D0, &
     &       -301D0,       0D0,     -1D0,      162D0,     0D0,     0D0, &
     &       -334D0,       0D0,      0D0,      144D0,     0D0,     0D0, &
     &        493D0,       0D0,     -2D0,      -15D0,     0D0,     0D0, &
     &        494D0,       0D0,     -2D0,      -19D0,     0D0,     0D0, &
     &        337D0,       0D0,     -1D0,     -143D0,     0D0,    -1D0, &
     &        280D0,       0D0,     -1D0,     -144D0,     0D0,     0D0, &
     &        309D0,       0D0,      1D0,     -134D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=141,150 ) / &
     &       -263D0,       0D0,      2D0,      131D0,     0D0,     1D0, &
     &        253D0,       0D0,      1D0,     -138D0,     0D0,     0D0, &
     &        245D0,       0D0,      0D0,     -128D0,     0D0,     0D0, &
     &        416D0,       0D0,     -2D0,      -17D0,     0D0,     0D0, &
     &       -229D0,       0D0,      0D0,      128D0,     0D0,     0D0, &
     &        231D0,       0D0,      0D0,     -120D0,     0D0,     0D0, &
     &       -259D0,       0D0,      2D0,      109D0,     0D0,     1D0, &
     &        375D0,       0D0,     -1D0,       -8D0,     0D0,     0D0, &
     &        252D0,       0D0,      0D0,     -108D0,     0D0,     0D0, &
     &       -245D0,       0D0,      1D0,      104D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=151,160 ) / &
     &        243D0,       0D0,     -1D0,     -104D0,     0D0,     0D0, &
     &        208D0,       0D0,      1D0,     -112D0,     0D0,     0D0, &
     &        199D0,       0D0,      0D0,     -102D0,     0D0,     0D0, &
     &       -208D0,       0D0,      1D0,      105D0,     0D0,     0D0, &
     &        335D0,       0D0,     -2D0,      -14D0,     0D0,     0D0, &
     &       -325D0,       0D0,      1D0,        7D0,     0D0,     0D0, &
     &       -187D0,       0D0,      0D0,       96D0,     0D0,     0D0, &
     &        197D0,       0D0,     -1D0,     -100D0,     0D0,     0D0, &
     &       -192D0,       0D0,      2D0,       94D0,     0D0,     1D0, &
     &       -188D0,       0D0,      0D0,       83D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=161,170 ) / &
     &        276D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &       -286D0,       0D0,      1D0,        6D0,     0D0,     0D0, &
     &        186D0,       0D0,     -1D0,      -79D0,     0D0,     0D0, &
     &       -219D0,       0D0,      0D0,       43D0,     0D0,     0D0, &
     &        276D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &       -153D0,       0D0,     -1D0,       84D0,     0D0,     0D0, &
     &       -156D0,       0D0,      0D0,       81D0,     0D0,     0D0, &
     &       -154D0,       0D0,      1D0,       78D0,     0D0,     0D0, &
     &       -174D0,       0D0,      1D0,       75D0,     0D0,     0D0, &
     &       -163D0,       0D0,      2D0,       69D0,     0D0,     1D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=171,180 ) / &
     &       -228D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &         91D0,       0D0,     -4D0,      -54D0,     0D0,    -2D0, &
     &        175D0,       0D0,      0D0,      -75D0,     0D0,     0D0, &
     &       -159D0,       0D0,      0D0,       69D0,     0D0,     0D0, &
     &        141D0,       0D0,      0D0,      -72D0,     0D0,     0D0, &
     &        147D0,       0D0,      0D0,      -75D0,     0D0,     0D0, &
     &       -132D0,       0D0,      0D0,       69D0,     0D0,     0D0, &
     &        159D0,       0D0,    -28D0,      -54D0,     0D0,    11D0, &
     &        213D0,       0D0,      0D0,       -4D0,     0D0,     0D0, &
     &        123D0,       0D0,      0D0,      -64D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=181,190 ) / &
     &       -118D0,       0D0,     -1D0,       66D0,     0D0,     0D0, &
     &        144D0,       0D0,     -1D0,      -61D0,     0D0,     0D0, &
     &       -121D0,       0D0,      1D0,       60D0,     0D0,     0D0, &
     &       -134D0,       0D0,      1D0,       56D0,     0D0,     1D0, &
     &       -105D0,       0D0,      0D0,       57D0,     0D0,     0D0, &
     &       -102D0,       0D0,      0D0,       56D0,     0D0,     0D0, &
     &        120D0,       0D0,      0D0,      -52D0,     0D0,     0D0, &
     &        101D0,       0D0,      0D0,      -54D0,     0D0,     0D0, &
     &       -113D0,       0D0,      0D0,       59D0,     0D0,     0D0, &
     &       -106D0,       0D0,      0D0,       61D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=191,200 ) / &
     &       -129D0,       0D0,      1D0,       55D0,     0D0,     0D0, &
     &       -114D0,       0D0,      0D0,       57D0,     0D0,     0D0, &
     &        113D0,       0D0,     -1D0,      -49D0,     0D0,     0D0, &
     &       -102D0,       0D0,      0D0,       44D0,     0D0,     0D0, &
     &        -94D0,       0D0,      0D0,       51D0,     0D0,     0D0, &
     &       -100D0,       0D0,     -1D0,       56D0,     0D0,     0D0, &
     &         87D0,       0D0,      0D0,      -47D0,     0D0,     0D0, &
     &        161D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &         96D0,       0D0,      0D0,      -50D0,     0D0,     0D0, &
     &        151D0,       0D0,     -1D0,       -5D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=201,210 ) / &
     &       -104D0,       0D0,      0D0,       44D0,     0D0,     0D0, &
     &       -110D0,       0D0,      0D0,       48D0,     0D0,     0D0, &
     &       -100D0,       0D0,      1D0,       50D0,     0D0,     0D0, &
     &         92D0,       0D0,     -5D0,       12D0,     0D0,    -2D0, &
     &         82D0,       0D0,      0D0,      -45D0,     0D0,     0D0, &
     &         82D0,       0D0,      0D0,      -45D0,     0D0,     0D0, &
     &        -78D0,       0D0,      0D0,       41D0,     0D0,     0D0, &
     &        -77D0,       0D0,      0D0,       43D0,     0D0,     0D0, &
     &          2D0,       0D0,      0D0,       54D0,     0D0,     0D0, &
     &         94D0,       0D0,      0D0,      -40D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=211,220 ) / &
     &        -93D0,       0D0,      0D0,       40D0,     0D0,     0D0, &
     &        -83D0,       0D0,     10D0,       40D0,     0D0,    -2D0, &
     &         83D0,       0D0,      0D0,      -36D0,     0D0,     0D0, &
     &        -91D0,       0D0,      0D0,       39D0,     0D0,     0D0, &
     &        128D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &        -79D0,       0D0,      0D0,       34D0,     0D0,     0D0, &
     &        -83D0,       0D0,      0D0,       47D0,     0D0,     0D0, &
     &         84D0,       0D0,      0D0,      -44D0,     0D0,     0D0, &
     &         83D0,       0D0,      0D0,      -43D0,     0D0,     0D0, &
     &         91D0,       0D0,      0D0,      -39D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=221,230 ) / &
     &        -77D0,       0D0,      0D0,       39D0,     0D0,     0D0, &
     &         84D0,       0D0,      0D0,      -43D0,     0D0,     0D0, &
     &        -92D0,       0D0,      1D0,       39D0,     0D0,     0D0, &
     &        -92D0,       0D0,      1D0,       39D0,     0D0,     0D0, &
     &        -94D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         68D0,       0D0,      0D0,      -36D0,     0D0,     0D0, &
     &        -61D0,       0D0,      0D0,       32D0,     0D0,     0D0, &
     &         71D0,       0D0,      0D0,      -31D0,     0D0,     0D0, &
     &         62D0,       0D0,      0D0,      -34D0,     0D0,     0D0, &
     &        -63D0,       0D0,      0D0,       33D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=231,240 ) / &
     &        -73D0,       0D0,      0D0,       32D0,     0D0,     0D0, &
     &        115D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &       -103D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         63D0,       0D0,      0D0,      -28D0,     0D0,     0D0, &
     &         74D0,       0D0,      0D0,      -32D0,     0D0,     0D0, &
     &       -103D0,       0D0,     -3D0,        3D0,     0D0,    -1D0, &
     &        -69D0,       0D0,      0D0,       30D0,     0D0,     0D0, &
     &         57D0,       0D0,      0D0,      -29D0,     0D0,     0D0, &
     &         94D0,       0D0,      0D0,       -4D0,     0D0,     0D0, &
     &         64D0,       0D0,      0D0,      -33D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=241,250 ) / &
     &        -63D0,       0D0,      0D0,       26D0,     0D0,     0D0, &
     &        -38D0,       0D0,      0D0,       20D0,     0D0,     0D0, &
     &        -43D0,       0D0,      0D0,       24D0,     0D0,     0D0, &
     &        -45D0,       0D0,      0D0,       23D0,     0D0,     0D0, &
     &         47D0,       0D0,      0D0,      -24D0,     0D0,     0D0, &
     &        -48D0,       0D0,      0D0,       25D0,     0D0,     0D0, &
     &         45D0,       0D0,      0D0,      -26D0,     0D0,     0D0, &
     &         56D0,       0D0,      0D0,      -25D0,     0D0,     0D0, &
     &         88D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &        -75D0,       0D0,      0D0,        0D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=251,260 ) / &
     &         85D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         49D0,       0D0,      0D0,      -26D0,     0D0,     0D0, &
     &        -74D0,       0D0,     -3D0,       -1D0,     0D0,    -1D0, &
     &        -39D0,       0D0,      0D0,       21D0,     0D0,     0D0, &
     &         45D0,       0D0,      0D0,      -20D0,     0D0,     0D0, &
     &         51D0,       0D0,      0D0,      -22D0,     0D0,     0D0, &
     &        -40D0,       0D0,      0D0,       21D0,     0D0,     0D0, &
     &         41D0,       0D0,      0D0,      -21D0,     0D0,     0D0, &
     &        -42D0,       0D0,      0D0,       24D0,     0D0,     0D0, &
     &        -51D0,       0D0,      0D0,       22D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=261,270 ) / &
     &        -42D0,       0D0,      0D0,       22D0,     0D0,     0D0, &
     &         39D0,       0D0,      0D0,      -21D0,     0D0,     0D0, &
     &         46D0,       0D0,      0D0,      -18D0,     0D0,     0D0, &
     &        -53D0,       0D0,      0D0,       22D0,     0D0,     0D0, &
     &         82D0,       0D0,      0D0,       -4D0,     0D0,     0D0, &
     &         81D0,       0D0,     -1D0,       -4D0,     0D0,     0D0, &
     &         47D0,       0D0,      0D0,      -19D0,     0D0,     0D0, &
     &         53D0,       0D0,      0D0,      -23D0,     0D0,     0D0, &
     &        -45D0,       0D0,      0D0,       22D0,     0D0,     0D0, &
     &        -44D0,       0D0,      0D0,       -2D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=271,280 ) / &
     &        -33D0,       0D0,      0D0,       16D0,     0D0,     0D0, &
     &        -61D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &         28D0,       0D0,      0D0,      -15D0,     0D0,     0D0, &
     &        -38D0,       0D0,      0D0,       19D0,     0D0,     0D0, &
     &        -33D0,       0D0,      0D0,       21D0,     0D0,     0D0, &
     &        -60D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         48D0,       0D0,      0D0,      -10D0,     0D0,     0D0, &
     &         27D0,       0D0,      0D0,      -14D0,     0D0,     0D0, &
     &         38D0,       0D0,      0D0,      -20D0,     0D0,     0D0, &
     &         31D0,       0D0,      0D0,      -13D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=281,290 ) / &
     &        -29D0,       0D0,      0D0,       15D0,     0D0,     0D0, &
     &         28D0,       0D0,      0D0,      -15D0,     0D0,     0D0, &
     &        -32D0,       0D0,      0D0,       15D0,     0D0,     0D0, &
     &         45D0,       0D0,      0D0,       -8D0,     0D0,     0D0, &
     &        -44D0,       0D0,      0D0,       19D0,     0D0,     0D0, &
     &         28D0,       0D0,      0D0,      -15D0,     0D0,     0D0, &
     &        -51D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -36D0,       0D0,      0D0,       20D0,     0D0,     0D0, &
     &         44D0,       0D0,      0D0,      -19D0,     0D0,     0D0, &
     &         26D0,       0D0,      0D0,      -14D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=291,300 ) / &
     &        -60D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         35D0,       0D0,      0D0,      -18D0,     0D0,     0D0, &
     &        -27D0,       0D0,      0D0,       11D0,     0D0,     0D0, &
     &         47D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &         36D0,       0D0,      0D0,      -15D0,     0D0,     0D0, &
     &        -36D0,       0D0,      0D0,       20D0,     0D0,     0D0, &
     &        -35D0,       0D0,      0D0,       19D0,     0D0,     0D0, &
     &        -37D0,       0D0,      0D0,       19D0,     0D0,     0D0, &
     &         32D0,       0D0,      0D0,      -16D0,     0D0,     0D0, &
     &         35D0,       0D0,      0D0,      -14D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=301,310 ) / &
     &         32D0,       0D0,      0D0,      -13D0,     0D0,     0D0, &
     &         65D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &         47D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &         32D0,       0D0,      0D0,      -16D0,     0D0,     0D0, &
     &         37D0,       0D0,      0D0,      -16D0,     0D0,     0D0, &
     &        -30D0,       0D0,      0D0,       15D0,     0D0,     0D0, &
     &        -32D0,       0D0,      0D0,       16D0,     0D0,     0D0, &
     &        -31D0,       0D0,      0D0,       13D0,     0D0,     0D0, &
     &         37D0,       0D0,      0D0,      -16D0,     0D0,     0D0, &
     &         31D0,       0D0,      0D0,      -13D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=311,320 ) / &
     &         49D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &         32D0,       0D0,      0D0,      -13D0,     0D0,     0D0, &
     &         23D0,       0D0,      0D0,      -12D0,     0D0,     0D0, &
     &        -43D0,       0D0,      0D0,       18D0,     0D0,     0D0, &
     &         26D0,       0D0,      0D0,      -11D0,     0D0,     0D0, &
     &        -32D0,       0D0,      0D0,       14D0,     0D0,     0D0, &
     &        -29D0,       0D0,      0D0,       14D0,     0D0,     0D0, &
     &        -27D0,       0D0,      0D0,       12D0,     0D0,     0D0, &
     &         30D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -11D0,       0D0,      0D0,        5D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=321,330 ) / &
     &        -21D0,       0D0,      0D0,       10D0,     0D0,     0D0, &
     &        -34D0,       0D0,      0D0,       15D0,     0D0,     0D0, &
     &        -10D0,       0D0,      0D0,        6D0,     0D0,     0D0, &
     &        -36D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -9D0,       0D0,      0D0,        4D0,     0D0,     0D0, &
     &        -12D0,       0D0,      0D0,        5D0,     0D0,     0D0, &
     &        -21D0,       0D0,      0D0,        5D0,     0D0,     0D0, &
     &        -29D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &        -15D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &        -20D0,       0D0,      0D0,        0D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=331,340 ) / &
     &         28D0,       0D0,      0D0,        0D0,     0D0,    -2D0, &
     &         17D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -22D0,       0D0,      0D0,       12D0,     0D0,     0D0, &
     &        -14D0,       0D0,      0D0,        7D0,     0D0,     0D0, &
     &         24D0,       0D0,      0D0,      -11D0,     0D0,     0D0, &
     &         11D0,       0D0,      0D0,       -6D0,     0D0,     0D0, &
     &         14D0,       0D0,      0D0,       -6D0,     0D0,     0D0, &
     &         24D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         18D0,       0D0,      0D0,       -8D0,     0D0,     0D0, &
     &        -38D0,       0D0,      0D0,        0D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=341,350 ) / &
     &        -31D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -16D0,       0D0,      0D0,        8D0,     0D0,     0D0, &
     &         29D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -18D0,       0D0,      0D0,       10D0,     0D0,     0D0, &
     &        -10D0,       0D0,      0D0,        5D0,     0D0,     0D0, &
     &        -17D0,       0D0,      0D0,       10D0,     0D0,     0D0, &
     &          9D0,       0D0,      0D0,       -4D0,     0D0,     0D0, &
     &         16D0,       0D0,      0D0,       -6D0,     0D0,     0D0, &
     &         22D0,       0D0,      0D0,      -12D0,     0D0,     0D0, &
     &         20D0,       0D0,      0D0,        0D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=351,360 ) / &
     &        -13D0,       0D0,      0D0,        6D0,     0D0,     0D0, &
     &        -17D0,       0D0,      0D0,        9D0,     0D0,     0D0, &
     &        -14D0,       0D0,      0D0,        8D0,     0D0,     0D0, &
     &          0D0,       0D0,      0D0,       -7D0,     0D0,     0D0, &
     &         14D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         19D0,       0D0,      0D0,      -10D0,     0D0,     0D0, &
     &        -34D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -20D0,       0D0,      0D0,        8D0,     0D0,     0D0, &
     &          9D0,       0D0,      0D0,       -5D0,     0D0,     0D0, &
     &        -18D0,       0D0,      0D0,        7D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=361,370 ) / &
     &         13D0,       0D0,      0D0,       -6D0,     0D0,     0D0, &
     &         17D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -12D0,       0D0,      0D0,        5D0,     0D0,     0D0, &
     &         15D0,       0D0,      0D0,       -8D0,     0D0,     0D0, &
     &        -11D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &         13D0,       0D0,      0D0,       -5D0,     0D0,     0D0, &
     &        -18D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -35D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          9D0,       0D0,      0D0,       -4D0,     0D0,     0D0, &
     &        -19D0,       0D0,      0D0,       10D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=371,380 ) / &
     &        -26D0,       0D0,      0D0,       11D0,     0D0,     0D0, &
     &          8D0,       0D0,      0D0,       -4D0,     0D0,     0D0, &
     &        -10D0,       0D0,      0D0,        4D0,     0D0,     0D0, &
     &         10D0,       0D0,      0D0,       -6D0,     0D0,     0D0, &
     &        -21D0,       0D0,      0D0,        9D0,     0D0,     0D0, &
     &        -15D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          9D0,       0D0,      0D0,       -5D0,     0D0,     0D0, &
     &        -29D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -19D0,       0D0,      0D0,       10D0,     0D0,     0D0, &
     &         12D0,       0D0,      0D0,       -5D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=381,390 ) / &
     &         22D0,       0D0,      0D0,       -9D0,     0D0,     0D0, &
     &        -10D0,       0D0,      0D0,        5D0,     0D0,     0D0, &
     &        -20D0,       0D0,      0D0,       11D0,     0D0,     0D0, &
     &        -20D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -17D0,       0D0,      0D0,        7D0,     0D0,     0D0, &
     &         15D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &          8D0,       0D0,      0D0,       -4D0,     0D0,     0D0, &
     &         14D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -12D0,       0D0,      0D0,        6D0,     0D0,     0D0, &
     &         25D0,       0D0,      0D0,        0D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=391,400 ) / &
     &        -13D0,       0D0,      0D0,        6D0,     0D0,     0D0, &
     &        -14D0,       0D0,      0D0,        8D0,     0D0,     0D0, &
     &         13D0,       0D0,      0D0,       -5D0,     0D0,     0D0, &
     &        -17D0,       0D0,      0D0,        9D0,     0D0,     0D0, &
     &        -12D0,       0D0,      0D0,        6D0,     0D0,     0D0, &
     &        -10D0,       0D0,      0D0,        5D0,     0D0,     0D0, &
     &         10D0,       0D0,      0D0,       -6D0,     0D0,     0D0, &
     &        -15D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -22D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         28D0,       0D0,      0D0,       -1D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=401,410 ) / &
     &         15D0,       0D0,      0D0,       -7D0,     0D0,     0D0, &
     &         23D0,       0D0,      0D0,      -10D0,     0D0,     0D0, &
     &         12D0,       0D0,      0D0,       -5D0,     0D0,     0D0, &
     &         29D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &        -25D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &         22D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -18D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         15D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &        -23D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         12D0,       0D0,      0D0,       -5D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=411,420 ) / &
     &         -8D0,       0D0,      0D0,        4D0,     0D0,     0D0, &
     &        -19D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -10D0,       0D0,      0D0,        4D0,     0D0,     0D0, &
     &         21D0,       0D0,      0D0,       -9D0,     0D0,     0D0, &
     &         23D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &        -16D0,       0D0,      0D0,        8D0,     0D0,     0D0, &
     &        -19D0,       0D0,      0D0,        9D0,     0D0,     0D0, &
     &        -22D0,       0D0,      0D0,       10D0,     0D0,     0D0, &
     &         27D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &         16D0,       0D0,      0D0,       -8D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=421,430 ) / &
     &         19D0,       0D0,      0D0,       -8D0,     0D0,     0D0, &
     &          9D0,       0D0,      0D0,       -4D0,     0D0,     0D0, &
     &         -9D0,       0D0,      0D0,        4D0,     0D0,     0D0, &
     &         -9D0,       0D0,      0D0,        4D0,     0D0,     0D0, &
     &         -8D0,       0D0,      0D0,        4D0,     0D0,     0D0, &
     &         18D0,       0D0,      0D0,       -9D0,     0D0,     0D0, &
     &         16D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &        -10D0,       0D0,      0D0,        4D0,     0D0,     0D0, &
     &        -23D0,       0D0,      0D0,        9D0,     0D0,     0D0, &
     &         16D0,       0D0,      0D0,       -1D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=431,440 ) / &
     &        -12D0,       0D0,      0D0,        6D0,     0D0,     0D0, &
     &         -8D0,       0D0,      0D0,        4D0,     0D0,     0D0, &
     &         30D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &         24D0,       0D0,      0D0,      -10D0,     0D0,     0D0, &
     &         10D0,       0D0,      0D0,       -4D0,     0D0,     0D0, &
     &        -16D0,       0D0,      0D0,        7D0,     0D0,     0D0, &
     &        -16D0,       0D0,      0D0,        7D0,     0D0,     0D0, &
     &         17D0,       0D0,      0D0,       -7D0,     0D0,     0D0, &
     &        -24D0,       0D0,      0D0,       10D0,     0D0,     0D0, &
     &        -12D0,       0D0,      0D0,        5D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=441,450 ) / &
     &        -24D0,       0D0,      0D0,       11D0,     0D0,     0D0, &
     &        -23D0,       0D0,      0D0,        9D0,     0D0,     0D0, &
     &        -13D0,       0D0,      0D0,        5D0,     0D0,     0D0, &
     &        -15D0,       0D0,      0D0,        7D0,     0D0,     0D0, &
     &          0D0,       0D0,  -1988D0,        0D0,     0D0, -1679D0, &
     &          0D0,       0D0,    -63D0,        0D0,     0D0,   -27D0, &
     &         -4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          0D0,       0D0,      5D0,        0D0,     0D0,     4D0, &
     &          5D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &          0D0,       0D0,    364D0,        0D0,     0D0,   176D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=451,460 ) / &
     &          0D0,       0D0,  -1044D0,        0D0,     0D0,  -891D0, &
     &         -3D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          0D0,       0D0,    330D0,        0D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &         -5D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,        0D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=461,470 ) / &
     &          3D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          0D0,       0D0,      5D0,        0D0,     0D0,     0D0, &
     &          0D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          6D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &         -7D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &        -12D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -1D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=471,480 ) / &
     &         -5D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -7D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &          7D0,       0D0,      0D0,       -4D0,     0D0,     0D0, &
     &          0D0,       0D0,    -12D0,        0D0,     0D0,   -10D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         -7D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=481,490 ) / &
     &         -3D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &          0D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &          7D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &         -5D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -5D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -2D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=491,500 ) / &
     &         -8D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &          9D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          6D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &         -5D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -7D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        2D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=501,510 ) / &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &         -5D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          9D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          9D0,       0D0,      0D0,       -3D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=511,520 ) / &
     &         -4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          8D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &          6D0,       0D0,      0D0,       -3D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=521,530 ) / &
     &          3D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &         -7D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          9D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -5D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &        -13D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -7D0,       0D0,      0D0,        0D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=531,540 ) / &
     &         10D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &         10D0,       0D0,     13D0,        6D0,     0D0,    -5D0, &
     &          0D0,       0D0,     30D0,        0D0,     0D0,    14D0, &
     &          0D0,       0D0,   -162D0,        0D0,     0D0,  -138D0, &
     &          0D0,       0D0,     75D0,        0D0,     0D0,     0D0, &
     &         -7D0,       0D0,      0D0,        4D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -2D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=541,550 ) / &
     &          5D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         -5D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          6D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          9D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -7D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        1D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=551,560 ) / &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          7D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -6D0,       0D0,     -3D0,        3D0,     0D0,     1D0, &
     &          0D0,       0D0,     -3D0,        0D0,     0D0,    -2D0, &
     &         11D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &         11D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        2D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=561,570 ) / &
     &         -1D0,       0D0,      3D0,        3D0,     0D0,    -1D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          0D0,       0D0,    -13D0,        0D0,     0D0,   -11D0, &
     &          3D0,       0D0,      6D0,        0D0,     0D0,     0D0, &
     &         -7D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &         -7D0,       0D0,      0D0,        3D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=571,580 ) / &
     &          8D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         11D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          8D0,       0D0,      0D0,       -4D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &         11D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -6D0,       0D0,      0D0,        3D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=581,590 ) / &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         -8D0,       0D0,      0D0,        4D0,     0D0,     0D0, &
     &         -7D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &          6D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &         -6D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &          6D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          6D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -2D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=591,600 ) / &
     &         -5D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          6D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          0D0,       0D0,    -26D0,        0D0,     0D0,   -11D0, &
     &          0D0,       0D0,    -10D0,        0D0,     0D0,    -5D0, &
     &          5D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &        -13D0,       0D0,      0D0,        0D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=601,610 ) / &
     &          3D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          7D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         -6D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         -5D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         -7D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -2D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=611,620 ) / &
     &         13D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &        -11D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          6D0,       0D0,      0D0,       -3D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=621,630 ) / &
     &          3D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &        -12D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &          0D0,       0D0,     -5D0,        0D0,     0D0,    -2D0, &
     &         -7D0,       0D0,      0D0,        4D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=631,640 ) / &
     &          6D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &         -5D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         12D0,       0D0,      0D0,        0D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=641,650 ) / &
     &          3D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          6D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &         -6D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          6D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &          6D0,       0D0,      0D0,        0D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=651,660 ) / &
     &         -6D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          7D0,       0D0,      0D0,       -4D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &         -5D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         -6D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &         -6D0,       0D0,      0D0,        3D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         10D0,       0D0,      0D0,        0D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=661,670 ) / &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          7D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          7D0,       0D0,      0D0,       -3D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &         11D0,       0D0,      0D0,        0D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &         -6D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          5D0,       0D0,      0D0,       -2D0,     0D0,     0D0 /
      DATA ( ( CLS(I,J), I=1,6 ), J=671,678 ) / &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         -4D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        2D0,     0D0,     0D0, &
     &          4D0,       0D0,      0D0,       -2D0,     0D0,     0D0, &
     &          3D0,       0D0,      0D0,       -1D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        1D0,     0D0,     0D0, &
     &         -3D0,       0D0,      0D0,        2D0,     0D0,     0D0 /
!
!  Planetary argument multipliers:
!              L   L'  F   D   Om  Me  Ve  E  Ma  Ju  Sa  Ur  Ne  pre
!
      DATA ( ( NAPL(I,J), I=1,14 ), J=  1, 10 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -8, 16, -4, -5,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  2,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -4,  8, -1, -5,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0,  0,  3, -8,  3,  0,  0,  0,  0, &
     &        -1,  0,  0,  0,  0,  0, 10, -3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  6, -3,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J= 11, 20 ) / &
     &         0,  0,  1, -1,  1,  0,  0, -5,  8, -3,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -4,  8, -3,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  4, -8,  1,  5,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -5,  6,  4,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0,  2, -5,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  1 /
      DATA ( ( NAPL(I,J), I=1,14 ), J= 21, 30 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  2, &
     &         2,  0, -1, -1,  0,  0,  0,  3, -7,  0,  0,  0,  0,  0, &
     &         1,  0,  0, -2,  0,  0, 19,-21,  3,  0,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  2, -4,  0, -3,  0,  0,  0,  0, &
     &         1,  0,  0, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0, -4, 10,  0,  0,  0, &
     &        -2,  0,  0,  2,  1,  0,  0,  2,  0,  0, -5,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  3, -7,  4,  0,  0,  0,  0,  0, &
     &         0,  0, -1,  1,  0,  0,  0,  1,  0,  1, -1,  0,  0,  0, &
     &        -2,  0,  0,  2,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J= 31, 40 ) / &
     &        -1,  0,  0,  0,  0,  0, 18,-16,  0,  0,  0,  0,  0,  0, &
     &        -2,  0,  1,  1,  2,  0,  0,  1,  0, -2,  0,  0,  0,  0, &
     &        -1,  0,  1, -1,  1,  0, 18,-17,  0,  0,  0,  0,  0,  0, &
     &        -1,  0,  0,  1,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -8, 13,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  2, -2,  2,  0, -8, 11,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -8, 13,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  8,-14,  0,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J= 41, 50 ) / &
     &         0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  1, &
     &        -2,  0,  0,  2,  1,  0,  0,  2,  0, -4,  5,  0,  0,  0, &
     &        -2,  0,  0,  2,  2,  0,  3, -3,  0,  0,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  0,  0,  0,  2,  0, -3,  1,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  3, -5,  0,  2,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  0,  0,  0,  2,  0, -4,  3,  0,  0,  0, &
     &         0,  0, -1,  1,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0, -1,  2,  0,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  2,  0,  0, -2,  2,  0,  0,  0,  0,  0, &
     &        -1,  0,  1,  0,  1,  0,  3, -5,  0,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J= 51, 60 ) / &
     &        -1,  0,  0,  1,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  0,  0,  0,  2,  0, -2, -2,  0,  0,  0, &
     &        -2,  0,  2,  0,  2,  0,  0, -5,  9,  0,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0, -1,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  2,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  2, &
     &        -1,  0,  0,  1,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0, &
     &         0,  0, -1,  1,  0,  0,  0,  1,  0,  0,  2,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J= 61, 70 ) / &
     &         0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  2,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0, -9, 17,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  2,  0, -3,  5,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  2,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0, &
     &         1,  0,  0, -2,  0,  0, 17,-16,  0, -2,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0,  1, -3,  0,  0,  0, &
     &        -2,  0,  0,  2,  1,  0,  0,  5, -6,  0,  0,  0,  0,  0, &
     &         0,  0, -2,  2,  0,  0,  0,  9,-13,  0,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  1,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J= 71, 80 ) / &
     &         0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0, &
     &         0,  0, -1,  1,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0, &
     &         0,  0, -2,  2,  0,  0,  5, -6,  0,  0,  0,  0,  0,  0, &
     &         0,  0, -1,  1,  1,  0,  5, -7,  0,  0,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  0,  0,  6, -8,  0,  0,  0,  0,  0,  0, &
     &         2,  0,  1, -3,  1,  0, -6,  7,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  2,  0,  0,  0,  0,  1,  0,  0,  0,  0, &
     &         0,  0, -1,  1,  1,  0,  0,  1,  0,  1,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  2,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  1 /
      DATA ( ( NAPL(I,J), I=1,14 ), J= 81, 90 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -8, 15,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -8, 15,  0,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0,  0, -9, 15,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0, &
     &         1,  0, -1, -1,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0, &
     &         2,  0,  0, -2,  0,  0,  2, -5,  0,  0,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  0,  0,  0,  2,  0, -5,  5,  0,  0,  0, &
     &         2,  0,  0, -2,  1,  0,  0, -6,  8,  0,  0,  0,  0,  0, &
     &         2,  0,  0, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J= 91,100 ) / &
     &        -2,  0,  1,  1,  0,  0,  0,  1,  0, -3,  0,  0,  0,  0, &
     &        -2,  0,  1,  1,  1,  0,  0,  1,  0, -3,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  0,  0,  0,  2,  0, -1, -5,  0,  0,  0, &
     &        -1,  0,  0,  1,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &        -1,  0,  1,  1,  1,  0,-20, 20,  0,  0,  0,  0,  0,  0, &
     &         1,  0,  0, -2,  0,  0, 20,-21,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0,  8,-15,  0,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0,  0,-10, 15,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=101,110 ) / &
     &         0,  0, -1,  1,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  4,  0,  0,  0, &
     &         2,  0,  0, -2,  1,  0, -6,  8,  0,  0,  0,  0,  0,  0, &
     &         0,  0, -2,  2,  1,  0,  5, -6,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -1,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  1,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=111,120 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2, &
     &         0,  0,  2, -2,  1,  0,  0, -9, 13,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0,  7,-13,  0,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  9,-17,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -9, 17,  0,  0,  0,  0,  2, &
     &         1,  0,  0, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0, &
     &         1,  0,  0, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  2,  0,  0, -1,  2,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=121,130 ) / &
     &         0,  0, -1,  1,  1,  0,  0,  0,  2,  0,  0,  0,  0,  0, &
     &         0,  0, -2,  2,  0,  1,  0, -2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  3, -5,  0,  2,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  1,  0,  0,  2,  0, -3,  1,  0,  0,  0, &
     &        -2,  0,  0,  2,  1,  0,  3, -3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  8,-13,  0,  0,  0,  0,  0,  0, &
     &         0,  0, -1,  1,  0,  0,  8,-12,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0, -8, 11,  0,  0,  0,  0,  0,  0, &
     &        -1,  0,  0,  1,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0, &
     &        -1,  0,  0,  0,  1,  0, 18,-16,  0,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=131,140 ) / &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  1,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  3, -7,  4,  0,  0,  0,  0,  0, &
     &        -2,  0,  1,  1,  1,  0,  0, -3,  7,  0,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  2,  0,  0, -1,  0, -2,  5,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0,  0,  0, -2,  5,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &         1,  0,  0,  0,  1,  0,-10,  3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0, &
     &        -1,  0,  0,  0,  1,  0, 10, -3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=141,150 ) / &
     &         0,  0,  0,  0,  1,  0,  0,  0,  0,  2, -5,  0,  0,  0, &
     &         0,  0, -1,  1,  0,  0,  0,  1,  0,  2, -5,  0,  0,  0, &
     &         2,  0, -1, -1,  1,  0,  0,  3, -7,  0,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  0,  0,  0,  2,  0,  0, -5,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0, -3,  7, -4,  0,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &         1,  0,  0,  0,  1,  0,-18, 16,  0,  0,  0,  0,  0,  0, &
     &        -2,  0,  1,  1,  1,  0,  0,  1,  0, -2,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  2,  0, -8, 12,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0, -8, 13,  0,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=151,160 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0,  0,  0, -2,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -1,  2,  0,  0,  0,  0,  1, &
     &        -1,  0,  0,  1,  1,  0,  3, -4,  0,  0,  0,  0,  0,  0, &
     &        -1,  0,  0,  1,  1,  0,  0,  3, -4,  0,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -2,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  2,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=161,170 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  2, &
     &         0,  0,  1, -1,  0,  0,  3, -6,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0, -3,  5,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  2,  0, -3,  4,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0, -2,  4,  0,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0, &
     &         0,  0, -1,  1,  0,  0,  5, -7,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  5, -8,  0,  0,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  1,  0,  6, -8,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0, -8, 15,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=171,180 ) / &
     &        -2,  0,  0,  2,  1,  0,  0,  2,  0, -3,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  1,  0,  0,  6, -8,  0,  0,  0,  0,  0, &
     &         1,  0,  0, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=181,190 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  2, &
     &         0,  0,  1, -1,  2,  0,  0, -1,  0,  0, -1,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1,  0,  0,  0, &
     &         0,  0, -1,  1,  0,  0,  0,  1,  0,  0, -1,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -7, 13,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  7,-13,  0,  0,  0,  0,  0, &
     &         2,  0,  0, -2,  1,  0,  0, -5,  6,  0,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0,  0, -8, 11,  0,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1, -1,  0,  2,  0,  0,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=191,200 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  3,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  2, &
     &        -2,  0,  0,  2,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &         2,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  2,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  2,  0,  0,  0, -2,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=201,210 ) / &
     &         0,  0,  0,  0,  1,  0,  0,  1, -2,  0,  0,  0,  0,  0, &
     &         0,  0, -1,  1,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0, &
     &         0,  0, -1,  1,  0,  0,  0,  1,  0,  0, -2,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  2,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0,  3, -6,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=211,220 ) / &
     &         0,  0,  2, -2,  2,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0,  0,  1, -4,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=221,230 ) / &
     &         0,  0,  2, -2,  2,  0, -5,  6,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0, -5,  7,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  2,  0,  0, -1,  0, -1,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0,  0,  0, -1,  0,  0,  0,  0, &
     &         0,  0, -1,  1,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0,  0, -2,  0,  1,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=231,240 ) / &
     &         0,  0,  0,  0,  0,  0,  0, -6, 11,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  6,-11,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0, -1,  0,  4,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0,  0,  0, &
     &         2,  0,  0, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  0,  0,  0,  2,  0,  0, -2,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0,  0, -7,  9,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=241,250 ) / &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  2, &
     &         0,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  2, &
     &         0,  0,  0,  0,  1,  0,  3, -5,  0,  0,  0,  0,  0,  0, &
     &         0,  0, -1,  1,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0,  2, -4,  0,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0,  0, -4,  4,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=251,260 ) / &
     &         0,  0,  1, -1,  2,  0, -5,  7,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0,  0, -4,  6,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  2, &
     &         0,  0, -1,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  2, -3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -5,  9,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -5,  9,  0,  0,  0,  0,  1 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=261,270 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0,  0, &
     &         0,  0, -1,  1,  0,  0,  0,  1,  0, -2,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &        -2,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &         0,  0, -2,  2,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -2,  3,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -2,  3,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=271,280 ) / &
     &         0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0,  0, -1,  0,  3,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  4, -8,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -4,  8,  0,  0,  0,  0,  2, &
     &         0,  0, -2,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=281,290 ) / &
     &         0,  0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -5, 10,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  1,  0, -1,  2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=291,300 ) / &
     &         0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0,  1, -3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -1,  2,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0, -1,  2,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -7, 11,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -7, 11,  0,  0,  0,  0,  0,  1, &
     &         0,  0, -2,  2,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0, -4,  4,  0,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=301,310 ) / &
     &         0,  0, -1,  1,  0,  0,  4, -5,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0, -4,  6,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0, -4,  5,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=311,320 ) / &
     &        -2,  0,  0,  2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, &
     &         0,  0, -1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -1,  0,  5,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -1,  3,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -7, 12,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -1,  1,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -1,  1,  0,  0,  0,  0,  0,  1 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=321,330 ) / &
     &         0,  0,  1, -1,  1,  0, -1,  0,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0,  1, -2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -1,  0,  4,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0, -1,  1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=331,340 ) / &
     &         0,  0,  2, -2,  1,  0,  0, -3,  0,  3,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -3,  7,  0,  0,  0,  0,  2, &
     &        -2,  0,  0,  2,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -1,  0,  3,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -1,  0,  3,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  0,  1 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=341,350 ) / &
     &         0,  0,  1, -1,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -6,  9,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -6,  9,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0,  1,  0, -2,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  3, -4,  0,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=351,360 ) / &
     &         0,  0,  0,  0,  0,  0,  0, -1,  0,  2,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -5,  9,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -3,  4,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -3,  4,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=361,370 ) / &
     &         0,  0,  0,  0,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -3,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0,  1, -5,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -1,  0,  1,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  5,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -2,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=371,380 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -1,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -8, 14,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0,  2, -5,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  5, -8,  3,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  5, -8,  3,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  3, -8,  3,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=381,390 ) / &
     &         0,  0,  0,  0,  0,  0,  0, -3,  8, -3,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  5,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -8, 12,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -8, 12,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0,  1, -2,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,  0,  0,  2, &
     &         0,  0,  2, -2,  1,  0, -5,  5,  0,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=391,400 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -1,  4,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -5,  7,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -5,  7,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=401,410 ) / &
     &         0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0, -1,  0,  3,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -2,  6,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  1,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -6,  9,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -2,  2,  0,  0,  0,  0,  0,  1 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=411,420 ) / &
     &         0,  0,  1, -1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  0,  3,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -5,  7,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -1,  3,  0,  0,  0,  0,  0,  1 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=421,430 ) / &
     &         0,  0,  1, -1,  1,  0, -1,  2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -1,  3,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -7, 10,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -7, 10,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -4,  8,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -4,  5,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -4,  5,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=431,440 ) / &
     &         0,  0,  0,  0,  0,  0,  0, -2,  0,  5,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -9, 13,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -1,  5,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -2,  0,  4,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0, -4,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -2,  7,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=441,450 ) / &
     &         0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -6,  8,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -6,  8,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0, -3,  9,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=451,460 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -5, 10,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -3,  3,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -3,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=461,470 ) / &
     &         0,  0,  0,  0,  0,  0,  0, -5, 13,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -2,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -2,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -1,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -6, 15,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -8, 15,  0,  0,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=471,480 ) / &
     &         0,  0,  0,  0,  0,  0, -3,  9, -4,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0,  2, -5,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -2,  8, -1, -5,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  6, -8,  3,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  1, -1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=481,490 ) / &
     &         0,  0,  0,  0,  0,  0,  0, -6, 16, -4, -5,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -2,  8, -3,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -2,  8, -3,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  6, -8,  1,  5,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  5,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  3, -5,  4,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -8, 11,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -8, 11,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0, -8, 11,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, 11,  0,  0,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=491,500 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  3, -3,  0,  2,  0,  0,  0,  2, &
     &         0,  0,  2, -2,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &         0,  0,  1, -1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -3,  7,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -5,  6,  0,  0,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=501,510 ) / &
     &         0,  0,  0,  0,  0,  0, -5,  6,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0,  2,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -1,  6,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=511,520 ) / &
     &         0,  0,  0,  0,  0,  0, -1,  4,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0, -1,  4,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -7,  9,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -7,  9,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -4,  4,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=521,530 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -3,  0,  5,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -9, 12,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  3,  0, -4,  0,  0,  0,  0, &
     &         0,  0,  2, -2,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  7, -8,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  3,  0, -3,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=531,540 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  3,  0, -3,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -2,  6,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -6,  7,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=541,550 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -2,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -1,  0,  0,  2, &
     &         0,  0,  2, -2,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0, -8, 16,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  3,  0,  2, -5,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  7, -8,  3,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -5, 16, -4, -5,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=551,560 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0, -1,  8, -3,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -8, 10,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -8, 10,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0, -8, 10,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  2,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  3,  0,  1,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -3,  8,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -5,  5,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=561,570 ) / &
     &         0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  6, -5,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  7, -8,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=571,580 ) / &
     &         0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -9, 11,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -9, 11,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  4,  0, -4,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  4,  0, -3,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -6,  6,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  4,  0, -2,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=581,590 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  6, -4,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  4,  0, -1,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  4,  0,  0, -2,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  5, -2,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  8, -9,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=591,600 ) / &
     &         0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0, -7,  7,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  5,  0, -4,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=601,610 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  5,  0, -3,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  5,  0, -2,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -8,  8,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  8, -8,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0, -9,  9,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0, -9,  9,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0, -9,  9,  0,  0,  0,  0,  0,  1 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=611,620 ) / &
     &         0,  0,  0,  0,  0,  0,  9, -9,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  6, -4,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2, &
     &         0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  1, &
     &         0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=621,630 ) / &
     &         0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, &
     &         1,  0,  0, -2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &         1,  0,  0, -2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &         1,  0,  0, -2,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &         1,  0,  0, -2,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &        -1,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0, &
     &        -1,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &        -1,  0,  0,  2,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &         1,  0,  0, -2,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &        -2,  0,  0,  2,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=631,640 ) / &
     &        -1,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0, &
     &        -1,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &        -1,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &        -1,  0,  0,  2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &         1,  0, -1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &        -1,  0,  0,  2,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0, &
     &        -2,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0, &
     &         1,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &        -1,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0, &
     &         1,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=641,650 ) / &
     &        -1,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &        -1,  0,  0,  2,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &         0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &        -1,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &        -1,  0,  0,  2,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0, &
     &         1,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &         1,  0,  2, -2,  2,  0, -3,  3,  0,  0,  0,  0,  0,  0, &
     &         1,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0, &
     &         1,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &         1,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=651,660 ) / &
     &         0,  0,  0, -2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0, -2,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &         0,  0,  2,  0,  2,  0, -2,  2,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  2,  0,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &         0,  0,  2,  0,  2,  0, -1,  1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  2,  0,  2,  0, -2,  3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &         0,  0,  1,  1,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &         1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &        -1,  0,  2,  0,  2,  0, 10, -3,  0,  0,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=661,670 ) / &
     &         0,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &         1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  2,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0, &
     &         0,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &        -1,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &         2,  0,  2, -2,  2,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &         1,  0,  2,  0,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0, &
     &         0,  0,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &        -1,  0,  2,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &        -2,  0,  2,  2,  2,  0,  0,  2,  0, -2,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=671,680 ) / &
     &         0,  0,  2,  0,  2,  0,  2, -3,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  2,  0,  2,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  2,  0,  2,  0,  0,  1,  0, -1,  0,  0,  0,  0, &
     &         0,  0,  2,  0,  2,  0,  2, -2,  0,  0,  0,  0,  0,  0, &
     &        -1,  0,  2,  2,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0, &
     &         1,  0,  2,  0,  2,  0, -1,  1,  0,  0,  0,  0,  0,  0, &
     &        -1,  0,  2,  2,  2,  0,  0,  2,  0, -3,  0,  0,  0,  0, &
     &         2,  0,  2,  0,  2,  0,  0,  2,  0, -3,  0,  0,  0,  0, &
     &         1,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0, &
     &         1,  0,  2,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0 /
      DATA ( ( NAPL(I,J), I=1,14 ), J=681,687 ) / &
     &         1,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &         2,  0,  2,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0, &
     &        -1,  0,  2,  2,  2,  0,  0,  2,  0, -2,  0,  0,  0,  0, &
     &        -1,  0,  2,  2,  2,  0,  3, -3,  0,  0,  0,  0,  0,  0, &
     &         1,  0,  2,  0,  2,  0,  1, -1,  0,  0,  0,  0,  0,  0, &
     &         0,  0,  2,  2,  2,  0,  0,  2,  0, -2,  0,  0,  0,  0 /
!
!  Planetary nutation coefficients, unit 1e-7 arcsec:
!  longitude (sin, cos), obliquity (sin, cos)
!
      DATA ( ( ICPL(I,J), I=1,4 ), J=  1, 10 ) / &
     &       1440,          0,          0,          0, &
     &         56,       -117,        -42,        -40, &
     &        125,        -43,          0,        -54, &
     &          0,          5,          0,          0, &
     &          3,         -7,         -3,          0, &
     &          3,          0,          0,         -2, &
     &       -114,          0,          0,         61, &
     &       -219,         89,          0,          0, &
     &         -3,          0,          0,          0, &
     &       -462,       1604,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J= 11, 20 ) / &
     &         99,          0,          0,        -53, &
     &         -3,          0,          0,          2, &
     &          0,          6,          2,          0, &
     &          3,          0,          0,          0, &
     &        -12,          0,          0,          0, &
     &         14,       -218,        117,          8, &
     &         31,       -481,       -257,        -17, &
     &       -491,        128,          0,          0, &
     &      -3084,       5123,       2735,       1647, &
     &      -1444,       2409,      -1286,       -771 /
      DATA ( ( ICPL(I,J), I=1,4 ), J= 21, 30 ) / &
     &         11,        -24,        -11,         -9, &
     &         26,         -9,          0,          0, &
     &        103,        -60,          0,          0, &
     &          0,        -13,         -7,          0, &
     &        -26,        -29,        -16,         14, &
     &          9,        -27,        -14,         -5, &
     &         12,          0,          0,         -6, &
     &         -7,          0,          0,          0, &
     &          0,         24,          0,          0, &
     &        284,          0,          0,       -151 /
      DATA ( ( ICPL(I,J), I=1,4 ), J= 31, 40 ) / &
     &        226,        101,          0,          0, &
     &          0,         -8,         -2,          0, &
     &          0,         -6,         -3,          0, &
     &          5,          0,          0,         -3, &
     &        -41,        175,         76,         17, &
     &          0,         15,          6,          0, &
     &        425,        212,       -133,        269, &
     &       1200,        598,        319,       -641, &
     &        235,        334,          0,          0, &
     &         11,        -12,         -7,         -6 /
      DATA ( ( ICPL(I,J), I=1,4 ), J= 41, 50 ) / &
     &          5,         -6,          3,          3, &
     &         -5,          0,          0,          3, &
     &          6,          0,          0,         -3, &
     &         15,          0,          0,          0, &
     &         13,          0,          0,         -7, &
     &         -6,         -9,          0,          0, &
     &        266,        -78,          0,          0, &
     &       -460,       -435,       -232,        246, &
     &          0,         15,          7,          0, &
     &         -3,          0,          0,          2 /
      DATA ( ( ICPL(I,J), I=1,4 ), J= 51, 60 ) / &
     &          0,        131,          0,          0, &
     &          4,          0,          0,          0, &
     &          0,          3,          0,          0, &
     &          0,          4,          2,          0, &
     &          0,          3,          0,          0, &
     &        -17,        -19,        -10,          9, &
     &         -9,        -11,          6,         -5, &
     &         -6,          0,          0,          3, &
     &        -16,          8,          0,          0, &
     &          0,          3,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J= 61, 70 ) / &
     &         11,         24,         11,         -5, &
     &         -3,         -4,         -2,          1, &
     &          3,          0,          0,         -1, &
     &          0,         -8,         -4,          0, &
     &          0,          3,          0,          0, &
     &          0,          5,          0,          0, &
     &          0,          3,          2,          0, &
     &         -6,          4,          2,          3, &
     &         -3,         -5,          0,          0, &
     &         -5,          0,          0,          2 /
      DATA ( ( ICPL(I,J), I=1,4 ), J= 71, 80 ) / &
     &          4,         24,         13,         -2, &
     &        -42,         20,          0,          0, &
     &        -10,        233,          0,          0, &
     &         -3,          0,          0,          1, &
     &         78,        -18,          0,          0, &
     &          0,          3,          1,          0, &
     &          0,         -3,         -1,          0, &
     &          0,         -4,         -2,          1, &
     &          0,         -8,         -4,         -1, &
     &          0,         -5,          3,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J= 81, 90 ) / &
     &         -7,          0,          0,          3, &
     &        -14,          8,          3,          6, &
     &          0,          8,         -4,          0, &
     &          0,         19,         10,          0, &
     &         45,        -22,          0,          0, &
     &         -3,          0,          0,          0, &
     &          0,         -3,          0,          0, &
     &          0,          3,          0,          0, &
     &          3,          5,          3,         -2, &
     &         89,        -16,         -9,        -48 /
      DATA ( ( ICPL(I,J), I=1,4 ), J= 91,100 ) / &
     &          0,          3,          0,          0, &
     &         -3,          7,          4,          2, &
     &       -349,        -62,          0,          0, &
     &        -15,         22,          0,          0, &
     &         -3,          0,          0,          0, &
     &        -53,          0,          0,          0, &
     &          5,          0,          0,         -3, &
     &          0,         -8,          0,          0, &
     &         15,         -7,         -4,         -8, &
     &         -3,          0,          0,          1 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=101,110 ) / &
     &        -21,        -78,          0,          0, &
     &         20,        -70,        -37,        -11, &
     &          0,          6,          3,          0, &
     &          5,          3,          2,         -2, &
     &        -17,         -4,         -2,          9, &
     &          0,          6,          3,          0, &
     &         32,         15,         -8,         17, &
     &        174,         84,         45,        -93, &
     &         11,         56,          0,          0, &
     &        -66,        -12,         -6,         35 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=111,120 ) / &
     &         47,          8,          4,        -25, &
     &          0,          8,          4,          0, &
     &         10,        -22,        -12,         -5, &
     &         -3,          0,          0,          2, &
     &        -24,         12,          0,          0, &
     &          5,         -6,          0,          0, &
     &          3,          0,          0,         -2, &
     &          4,          3,          1,         -2, &
     &          0,         29,         15,          0, &
     &         -5,         -4,         -2,          2 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=121,130 ) / &
     &          8,         -3,         -1,         -5, &
     &          0,         -3,          0,          0, &
     &         10,          0,          0,          0, &
     &          3,          0,          0,         -2, &
     &         -5,          0,          0,          3, &
     &         46,         66,         35,        -25, &
     &        -14,          7,          0,          0, &
     &          0,          3,          2,          0, &
     &         -5,          0,          0,          0, &
     &        -68,        -34,        -18,         36 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=131,140 ) / &
     &          0,         14,          7,          0, &
     &         10,         -6,         -3,         -5, &
     &         -5,         -4,         -2,          3, &
     &         -3,          5,          2,          1, &
     &         76,         17,          9,        -41, &
     &         84,        298,        159,        -45, &
     &          3,          0,          0,         -1, &
     &         -3,          0,          0,          2, &
     &         -3,          0,          0,          1, &
     &        -82,        292,        156,         44 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=141,150 ) / &
     &        -73,         17,          9,         39, &
     &         -9,        -16,          0,          0, &
     &          3,          0,         -1,         -2, &
     &         -3,          0,          0,          0, &
     &         -9,         -5,         -3,          5, &
     &       -439,          0,          0,          0, &
     &         57,        -28,        -15,        -30, &
     &          0,         -6,         -3,          0, &
     &         -4,          0,          0,          2, &
     &        -40,         57,         30,         21 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=151,160 ) / &
     &         23,          7,          3,        -13, &
     &        273,         80,         43,       -146, &
     &       -449,        430,          0,          0, &
     &         -8,        -47,        -25,          4, &
     &          6,         47,         25,         -3, &
     &          0,         23,         13,          0, &
     &         -3,          0,          0,          2, &
     &          3,         -4,         -2,         -2, &
     &        -48,       -110,        -59,         26, &
     &         51,        114,         61,        -27 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=161,170 ) / &
     &       -133,          0,          0,         57, &
     &          0,          4,          0,          0, &
     &        -21,         -6,         -3,         11, &
     &          0,         -3,         -1,          0, &
     &        -11,        -21,        -11,          6, &
     &        -18,       -436,       -233,          9, &
     &         35,         -7,          0,          0, &
     &          0,          5,          3,          0, &
     &         11,         -3,         -1,         -6, &
     &         -5,         -3,         -1,          3 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=171,180 ) / &
     &        -53,         -9,         -5,         28, &
     &          0,          3,          2,          1, &
     &          4,          0,          0,         -2, &
     &          0,         -4,          0,          0, &
     &        -50,        194,        103,         27, &
     &        -13,         52,         28,          7, &
     &        -91,        248,          0,          0, &
     &          6,         49,         26,         -3, &
     &         -6,        -47,        -25,          3, &
     &          0,          5,          3,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=181,190 ) / &
     &         52,         23,         10,        -23, &
     &         -3,          0,          0,          1, &
     &          0,          5,          3,          0, &
     &         -4,          0,          0,          0, &
     &         -4,          8,          3,          2, &
     &         10,          0,          0,          0, &
     &          3,          0,          0,         -2, &
     &          0,          8,          4,          0, &
     &          0,          8,          4,          1, &
     &         -4,          0,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=191,200 ) / &
     &         -4,          0,          0,          0, &
     &         -8,          4,          2,          4, &
     &          8,         -4,         -2,         -4, &
     &          0,         15,          7,          0, &
     &       -138,          0,          0,          0, &
     &          0,         -7,         -3,          0, &
     &          0,         -7,         -3,          0, &
     &         54,          0,          0,        -29, &
     &          0,         10,          4,          0, &
     &         -7,          0,          0,          3 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=201,210 ) / &
     &        -37,         35,         19,         20, &
     &          0,          4,          0,          0, &
     &         -4,          9,          0,          0, &
     &          8,          0,          0,         -4, &
     &         -9,        -14,         -8,          5, &
     &         -3,         -9,         -5,          3, &
     &       -145,         47,          0,          0, &
     &        -10,         40,         21,          5, &
     &         11,        -49,        -26,         -7, &
     &      -2150,          0,          0,        932 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=211,220 ) / &
     &        -12,          0,          0,          5, &
     &         85,          0,          0,        -37, &
     &          4,          0,          0,         -2, &
     &          3,          0,          0,         -2, &
     &        -86,        153,          0,          0, &
     &         -6,          9,          5,          3, &
     &          9,        -13,         -7,         -5, &
     &         -8,         12,          6,          4, &
     &        -51,          0,          0,         22, &
     &        -11,       -268,       -116,          5 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=221,230 ) / &
     &          0,         12,          5,          0, &
     &          0,          7,          3,          0, &
     &         31,          6,          3,        -17, &
     &        140,         27,         14,        -75, &
     &         57,         11,          6,        -30, &
     &        -14,        -39,          0,          0, &
     &          0,         -6,         -2,          0, &
     &          4,         15,          8,         -2, &
     &          0,          4,          0,          0, &
     &         -3,          0,          0,          1 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=231,240 ) / &
     &          0,         11,          5,          0, &
     &          9,          6,          0,          0, &
     &         -4,         10,          4,          2, &
     &          5,          3,          0,          0, &
     &         16,          0,          0,         -9, &
     &         -3,          0,          0,          0, &
     &          0,          3,          2,         -1, &
     &          7,          0,          0,         -3, &
     &        -25,         22,          0,          0, &
     &         42,        223,        119,        -22 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=241,250 ) / &
     &        -27,       -143,        -77,         14, &
     &          9,         49,         26,         -5, &
     &      -1166,          0,          0,        505, &
     &         -5,          0,          0,          2, &
     &         -6,          0,          0,          3, &
     &         -8,          0,          1,          4, &
     &          0,         -4,          0,          0, &
     &        117,          0,          0,        -63, &
     &         -4,          8,          4,          2, &
     &          3,          0,          0,         -2 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=251,260 ) / &
     &         -5,          0,          0,          2, &
     &          0,         31,          0,          0, &
     &         -5,          0,          1,          3, &
     &          4,          0,          0,         -2, &
     &         -4,          0,          0,          2, &
     &        -24,        -13,         -6,         10, &
     &          3,          0,          0,          0, &
     &          0,        -32,        -17,          0, &
     &          8,         12,          5,         -3, &
     &          3,          0,          0,         -1 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=261,270 ) / &
     &          7,         13,          0,          0, &
     &         -3,         16,          0,          0, &
     &         50,          0,          0,        -27, &
     &          0,         -5,         -3,          0, &
     &         13,          0,          0,          0, &
     &          0,          5,          3,          1, &
     &         24,          5,          2,        -11, &
     &          5,        -11,         -5,         -2, &
     &         30,         -3,         -2,        -16, &
     &         18,          0,          0,         -9 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=271,280 ) / &
     &          8,        614,          0,          0, &
     &          3,         -3,         -1,         -2, &
     &          6,         17,          9,         -3, &
     &         -3,         -9,         -5,          2, &
     &          0,          6,          3,         -1, &
     &       -127,         21,          9,         55, &
     &          3,          5,          0,          0, &
     &         -6,        -10,         -4,          3, &
     &          5,          0,          0,          0, &
     &         16,          9,          4,         -7 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=281,290 ) / &
     &          3,          0,          0,         -2, &
     &          0,         22,          0,          0, &
     &          0,         19,         10,          0, &
     &          7,          0,          0,         -4, &
     &          0,         -5,         -2,          0, &
     &          0,          3,          1,          0, &
     &         -9,          3,          1,          4, &
     &         17,          0,          0,         -7, &
     &          0,         -3,         -2,         -1, &
     &        -20,         34,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=291,300 ) / &
     &        -10,          0,          1,          5, &
     &         -4,          0,          0,          2, &
     &         22,        -87,          0,          0, &
     &         -4,          0,          0,          2, &
     &         -3,         -6,         -2,          1, &
     &        -16,         -3,         -1,          7, &
     &          0,         -3,         -2,          0, &
     &          4,          0,          0,          0, &
     &        -68,         39,          0,          0, &
     &         27,          0,          0,        -14 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=301,310 ) / &
     &          0,         -4,          0,          0, &
     &        -25,          0,          0,          0, &
     &        -12,         -3,         -2,          6, &
     &          3,          0,          0,         -1, &
     &          3,         66,         29,         -1, &
     &        490,          0,          0,       -213, &
     &        -22,         93,         49,         12, &
     &         -7,         28,         15,          4, &
     &         -3,         13,          7,          2, &
     &        -46,         14,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=311,320 ) / &
     &         -5,          0,          0,          0, &
     &          2,          1,          0,          0, &
     &          0,         -3,          0,          0, &
     &        -28,          0,          0,         15, &
     &          5,          0,          0,         -2, &
     &          0,          3,          0,          0, &
     &        -11,          0,          0,          5, &
     &          0,          3,          1,          0, &
     &         -3,          0,          0,          1, &
     &         25,        106,         57,        -13 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=321,330 ) / &
     &          5,         21,         11,         -3, &
     &       1485,          0,          0,          0, &
     &         -7,        -32,        -17,          4, &
     &          0,          5,          3,          0, &
     &         -6,         -3,         -2,          3, &
     &         30,         -6,         -2,        -13, &
     &         -4,          4,          0,          0, &
     &        -19,          0,          0,         10, &
     &          0,          4,          2,         -1, &
     &          0,          3,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=331,340 ) / &
     &          4,          0,          0,         -2, &
     &          0,         -3,         -1,          0, &
     &         -3,          0,          0,          0, &
     &          5,          3,          1,         -2, &
     &          0,         11,          0,          0, &
     &        118,          0,          0,        -52, &
     &          0,         -5,         -3,          0, &
     &        -28,         36,          0,          0, &
     &          5,         -5,          0,          0, &
     &         14,        -59,        -31,         -8 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=341,350 ) / &
     &          0,          9,          5,          1, &
     &       -458,          0,          0,        198, &
     &          0,        -45,        -20,          0, &
     &          9,          0,          0,         -5, &
     &          0,         -3,          0,          0, &
     &          0,         -4,         -2,         -1, &
     &         11,          0,          0,         -6, &
     &          6,          0,          0,         -2, &
     &        -16,         23,          0,          0, &
     &          0,         -4,         -2,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=351,360 ) / &
     &         -5,          0,          0,          2, &
     &       -166,        269,          0,          0, &
     &         15,          0,          0,         -8, &
     &         10,          0,          0,         -4, &
     &        -78,         45,          0,          0, &
     &          0,         -5,         -2,          0, &
     &          7,          0,          0,         -4, &
     &         -5,        328,          0,          0, &
     &          3,          0,          0,         -2, &
     &          5,          0,          0,         -2 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=361,370 ) / &
     &          0,          3,          1,          0, &
     &         -3,          0,          0,          0, &
     &         -3,          0,          0,          0, &
     &          0,         -4,         -2,          0, &
     &      -1223,        -26,          0,          0, &
     &          0,          7,          3,          0, &
     &          3,          0,          0,          0, &
     &          0,          3,          2,          0, &
     &         -6,         20,          0,          0, &
     &       -368,          0,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=371,380 ) / &
     &        -75,          0,          0,          0, &
     &         11,          0,          0,         -6, &
     &          3,          0,          0,         -2, &
     &         -3,          0,          0,          1, &
     &        -13,        -30,          0,          0, &
     &         21,          3,          0,          0, &
     &         -3,          0,          0,          1, &
     &         -4,          0,          0,          2, &
     &          8,        -27,          0,          0, &
     &        -19,        -11,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=381,390 ) / &
     &         -4,          0,          0,          2, &
     &          0,          5,          2,          0, &
     &         -6,          0,          0,          2, &
     &         -8,          0,          0,          0, &
     &         -1,          0,          0,          0, &
     &        -14,          0,          0,          6, &
     &          6,          0,          0,          0, &
     &        -74,          0,          0,         32, &
     &          0,         -3,         -1,          0, &
     &          4,          0,          0,         -2 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=391,400 ) / &
     &          8,         11,          0,          0, &
     &          0,          3,          2,          0, &
     &       -262,          0,          0,        114, &
     &          0,         -4,          0,          0, &
     &         -7,          0,          0,          4, &
     &          0,        -27,        -12,          0, &
     &        -19,         -8,         -4,          8, &
     &        202,          0,          0,        -87, &
     &         -8,         35,         19,          5, &
     &          0,          4,          2,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=401,410 ) / &
     &         16,         -5,          0,          0, &
     &          5,          0,          0,         -3, &
     &          0,         -3,          0,          0, &
     &          1,          0,          0,          0, &
     &        -35,        -48,        -21,         15, &
     &         -3,         -5,         -2,          1, &
     &          6,          0,          0,         -3, &
     &          3,          0,          0,         -1, &
     &          0,         -5,          0,          0, &
     &         12,         55,         29,         -6 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=411,420 ) / &
     &          0,          5,          3,          0, &
     &       -598,          0,          0,          0, &
     &         -3,        -13,         -7,          1, &
     &         -5,         -7,         -3,          2, &
     &          3,          0,          0,         -1, &
     &          5,         -7,          0,          0, &
     &          4,          0,          0,         -2, &
     &         16,         -6,          0,          0, &
     &          8,         -3,          0,          0, &
     &          8,        -31,        -16,         -4 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=421,430 ) / &
     &          0,          3,          1,          0, &
     &        113,          0,          0,        -49, &
     &          0,        -24,        -10,          0, &
     &          4,          0,          0,         -2, &
     &         27,          0,          0,          0, &
     &         -3,          0,          0,          1, &
     &          0,         -4,         -2,          0, &
     &          5,          0,          0,         -2, &
     &          0,         -3,          0,          0, &
     &        -13,          0,          0,          6 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=431,440 ) / &
     &          5,          0,          0,         -2, &
     &        -18,        -10,         -4,          8, &
     &         -4,        -28,          0,          0, &
     &         -5,          6,          3,          2, &
     &         -3,          0,          0,          1, &
     &         -5,         -9,         -4,          2, &
     &         17,          0,          0,         -7, &
     &         11,          4,          0,          0, &
     &          0,         -6,         -2,          0, &
     &         83,         15,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=441,450 ) / &
     &         -4,          0,          0,          2, &
     &          0,       -114,        -49,          0, &
     &        117,          0,          0,        -51, &
     &         -5,         19,         10,          2, &
     &         -3,          0,          0,          0, &
     &         -3,          0,          0,          2, &
     &          0,         -3,         -1,          0, &
     &          3,          0,          0,          0, &
     &          0,         -6,         -2,          0, &
     &        393,          3,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=451,460 ) / &
     &         -4,         21,         11,          2, &
     &         -6,          0,         -1,          3, &
     &         -3,          8,          4,          1, &
     &          8,          0,          0,          0, &
     &         18,        -29,        -13,         -8, &
     &          8,         34,         18,         -4, &
     &         89,          0,          0,          0, &
     &          3,         12,          6,         -1, &
     &         54,        -15,         -7,        -24, &
     &          0,          3,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=461,470 ) / &
     &          3,          0,          0,         -1, &
     &          0,         35,          0,          0, &
     &       -154,        -30,        -13,         67, &
     &         15,          0,          0,          0, &
     &          0,          4,          2,          0, &
     &          0,          9,          0,          0, &
     &         80,        -71,        -31,        -35, &
     &          0,        -20,         -9,          0, &
     &         11,          5,          2,         -5, &
     &         61,        -96,        -42,        -27 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=471,480 ) / &
     &         14,          9,          4,         -6, &
     &        -11,         -6,         -3,          5, &
     &          0,         -3,         -1,          0, &
     &        123,       -415,       -180,        -53, &
     &          0,          0,          0,        -35, &
     &         -5,          0,          0,          0, &
     &          7,        -32,        -17,         -4, &
     &          0,         -9,         -5,          0, &
     &          0,         -4,          2,          0, &
     &        -89,          0,          0,         38 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=481,490 ) / &
     &          0,        -86,        -19,         -6, &
     &          0,          0,        -19,          6, &
     &       -123,       -416,       -180,         53, &
     &          0,         -3,         -1,          0, &
     &         12,         -6,         -3,         -5, &
     &        -13,          9,          4,          6, &
     &          0,        -15,         -7,          0, &
     &          3,          0,          0,         -1, &
     &        -62,        -97,        -42,         27, &
     &        -11,          5,          2,          5 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=491,500 ) / &
     &          0,        -19,         -8,          0, &
     &         -3,          0,          0,          1, &
     &          0,          4,          2,          0, &
     &          0,          3,          0,          0, &
     &          0,          4,          2,          0, &
     &        -85,        -70,        -31,         37, &
     &        163,        -12,         -5,        -72, &
     &        -63,        -16,         -7,         28, &
     &        -21,        -32,        -14,          9, &
     &          0,         -3,         -1,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=501,510 ) / &
     &          3,          0,          0,         -2, &
     &          0,          8,          0,          0, &
     &          3,         10,          4,         -1, &
     &          3,          0,          0,         -1, &
     &          0,         -7,         -3,          0, &
     &          0,         -4,         -2,          0, &
     &          6,         19,          0,          0, &
     &          5,       -173,        -75,         -2, &
     &          0,         -7,         -3,          0, &
     &          7,        -12,         -5,         -3 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=511,520 ) / &
     &         -3,          0,          0,          2, &
     &          3,         -4,         -2,         -1, &
     &         74,          0,          0,        -32, &
     &         -3,         12,          6,          2, &
     &         26,        -14,         -6,        -11, &
     &         19,          0,          0,         -8, &
     &          6,         24,         13,         -3, &
     &         83,          0,          0,          0, &
     &          0,        -10,         -5,          0, &
     &         11,         -3,         -1,         -5 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=521,530 ) / &
     &          3,          0,          1,         -1, &
     &          3,          0,          0,         -1, &
     &         -4,          0,          0,          0, &
     &          5,        -23,        -12,         -3, &
     &       -339,          0,          0,        147, &
     &          0,        -10,         -5,          0, &
     &          5,          0,          0,          0, &
     &          3,          0,          0,         -1, &
     &          0,         -4,         -2,          0, &
     &         18,         -3,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=531,540 ) / &
!
     &          9,        -11,         -5,         -4, &
     &         -8,          0,          0,          4, &
     &          3,          0,          0,         -1, &
     &          0,          9,          0,          0, &
     &          6,         -9,         -4,         -2, &
     &         -4,        -12,          0,          0, &
     &         67,        -91,        -39,        -29, &
     &         30,        -18,         -8,        -13, &
     &          0,          0,          0,          0, &
     &          0,       -114,        -50,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=541,550 ) / &
     &          0,          0,          0,         23, &
     &        517,         16,          7,       -224, &
     &          0,         -7,         -3,          0, &
     &        143,         -3,         -1,        -62, &
     &         29,          0,          0,        -13, &
     &         -4,          0,          0,          2, &
     &         -6,          0,          0,          3, &
     &          5,         12,          5,         -2, &
     &        -25,          0,          0,         11, &
     &         -3,          0,          0,          1 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=551,560 ) / &
     &          0,          4,          2,          0, &
     &        -22,         12,          5,         10, &
     &         50,          0,          0,        -22, &
     &          0,          7,          4,          0, &
     &          0,          3,          1,          0, &
     &         -4,          4,          2,          2, &
     &         -5,        -11,         -5,          2, &
     &          0,          4,          2,          0, &
     &          4,         17,          9,         -2, &
     &         59,          0,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=561,570 ) / &
     &          0,         -4,         -2,          0, &
     &         -8,          0,          0,          4, &
     &         -3,          0,          0,          0, &
     &          4,        -15,         -8,         -2, &
     &        370,         -8,          0,       -160, &
     &          0,          0,         -3,          0, &
     &          0,          3,          1,          0, &
     &         -6,          3,          1,          3, &
     &          0,          6,          0,          0, &
     &        -10,          0,          0,          4 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=571,580 ) / &
     &          0,          9,          4,          0, &
     &          4,         17,          7,         -2, &
     &         34,          0,          0,        -15, &
     &          0,          5,          3,          0, &
     &         -5,          0,          0,          2, &
     &        -37,         -7,         -3,         16, &
     &          3,         13,          7,         -2, &
     &         40,          0,          0,          0, &
     &          0,         -3,         -2,          0, &
     &       -184,         -3,         -1,         80 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=581,590 ) / &
     &         -3,          0,          0,          1, &
     &         -3,          0,          0,          0, &
     &          0,        -10,         -6,         -1, &
     &         31,         -6,          0,        -13, &
     &         -3,        -32,        -14,          1, &
     &         -7,          0,          0,          3, &
     &          0,         -8,         -4,          0, &
     &          3,         -4,          0,          0, &
     &          0,          4,          0,          0, &
     &          0,          3,          1,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=591,600 ) / &
     &         19,        -23,        -10,          2, &
     &          0,          0,          0,        -10, &
     &          0,          3,          2,          0, &
     &          0,          9,          5,         -1, &
     &         28,          0,          0,          0, &
     &          0,         -7,         -4,          0, &
     &          8,         -4,          0,         -4, &
     &          0,          0,         -2,          0, &
     &          0,          3,          0,          0, &
     &         -3,          0,          0,          1 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=601,610 ) / &
     &         -9,          0,          1,          4, &
     &          3,         12,          5,         -1, &
     &         17,         -3,         -1,          0, &
     &          0,          7,          4,          0, &
     &         19,          0,          0,          0, &
     &          0,         -5,         -3,          0, &
     &         14,         -3,          0,         -1, &
     &          0,          0,         -1,          0, &
     &          0,          0,          0,         -5, &
     &          0,          5,          3,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=611,620 ) / &
     &         13,          0,          0,          0, &
     &          0,         -3,         -2,          0, &
     &          2,          9,          4,          3, &
     &          0,          0,          0,         -4, &
     &          8,          0,          0,          0, &
     &          0,          4,          2,          0, &
     &          6,          0,          0,         -3, &
     &          6,          0,          0,          0, &
     &          0,          3,          1,          0, &
     &          5,          0,          0,         -2 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=621,630 ) / &
     &          3,          0,          0,         -1, &
     &         -3,          0,          0,          0, &
     &          6,          0,          0,          0, &
     &          7,          0,          0,          0, &
     &         -4,          0,          0,          0, &
     &          4,          0,          0,          0, &
     &          6,          0,          0,          0, &
     &          0,         -4,          0,          0, &
     &          0,         -4,          0,          0, &
     &          5,          0,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=631,640 ) / &
     &         -3,          0,          0,          0, &
     &          4,          0,          0,          0, &
     &         -5,          0,          0,          0, &
     &          4,          0,          0,          0, &
     &          0,          3,          0,          0, &
     &         13,          0,          0,          0, &
     &         21,         11,          0,          0, &
     &          0,         -5,          0,          0, &
     &          0,         -5,         -2,          0, &
     &          0,          5,          3,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=641,650 ) / &
     &          0,         -5,          0,          0, &
     &         -3,          0,          0,          2, &
     &         20,         10,          0,          0, &
     &        -34,          0,          0,          0, &
     &        -19,          0,          0,          0, &
     &          3,          0,          0,         -2, &
     &         -3,          0,          0,          1, &
     &         -6,          0,          0,          3, &
     &         -4,          0,          0,          0, &
     &          3,          0,          0,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=651,660 ) / &
     &          3,          0,          0,          0, &
     &          4,          0,          0,          0, &
     &          3,          0,          0,         -1, &
     &          6,          0,          0,         -3, &
     &         -8,          0,          0,          3, &
     &          0,          3,          1,          0, &
     &         -3,          0,          0,          0, &
     &          0,         -3,         -2,          0, &
     &        126,        -63,        -27,        -55, &
     &         -5,          0,          1,          2 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=661,670 ) / &
     &         -3,         28,         15,          2, &
     &          5,          0,          1,         -2, &
     &          0,          9,          4,          1, &
     &          0,          9,          4,         -1, &
     &       -126,        -63,        -27,         55, &
     &          3,          0,          0,         -1, &
     &         21,        -11,         -6,        -11, &
     &          0,         -4,          0,          0, &
     &        -21,        -11,         -6,         11, &
     &         -3,          0,          0,          1 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=671,680 ) / &
     &          0,          3,          1,          0, &
     &          8,          0,          0,         -4, &
     &         -6,          0,          0,          3, &
     &         -3,          0,          0,          1, &
     &          3,          0,          0,         -1, &
     &         -3,          0,          0,          1, &
     &         -5,          0,          0,          2, &
     &         24,        -12,         -5,        -11, &
     &          0,          3,          1,          0, &
     &          0,          3,          1,          0 /
      DATA ( ( ICPL(I,J), I=1,4 ), J=681,687 ) / &
     &          0,          3,          2,          0, &
     &        -24,        -12,         -5,         10, &
     &          4,          0,         -1,         -2, &
     &         13,          0,          0,         -6, &
     &          7,          0,          0,         -3, &
     &          3,          0,          0,         -1, &
     &          3,          0,          0,         -1 /
!
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
!  Interval between fundamental epoch J2000.0 and given date (JC).
!     T = ( ( DATE1-DJ0 ) + DATE2 ) / DJC
      T = CENT
!
!  -------------------
!  LUNI-SOLAR NUTATION
!  -------------------
!
!  Fundamental (Delaunay) arguments from Simon et al. (1994)
!
!  Mean anomaly of the Moon.
!     EL  = MOD (         485868.249036D0 +
!    :            T*( 1717915923.2178D0 +
!    :            T*(         31.8792D0 +
!    :            T*(          0.051635D0 +
!    :            T*(        - 0.00024470D0 )))), TURNAS ) * DAS2R
!
!  Mean anomaly of the Sun.
!!!  ELP Differs slightly from FA2K(2)
      ELP = MOD (        1287104.79305D0 + &
     &            T*(  129596581.0481D0 + &
     &            T*(        - 0.5532D0 + &
     &            T*(          0.000136D0 + &
     &            T*(        - 0.00001149D0 )))), TURNAS ) * DAS2R
!!!  DELP equals FAD2K(2)
!     DELP   =  (    129596581.0481D0 +
!    :         -2.D0*0.5532D0*T + 3.D0*0.000136D0*T**2 +
!    :         -4.D0*0.00001149D0*T**3) * DCENT * DAS2R
!     DELP   = DELP/(36525.D0*86400.D0)
!        print *,' NU2KA/ELP,DELP: ', ELP,DELP
!
!  Mean argument of the latitude of the Moon.
!     F   = MOD (         335779.526232D0 +
!    :            T*( 1739527262.8478D0 +
!    :            T*(       - 12.7512D0 +
!    :            T*(       -  0.001037D0 +
!    :            T*(          0.00000417D0 )))), TURNAS ) * DAS2R
!
!  Mean elongation of the Moon from the Sun.
!     D   = MOD (        1072260.70369D0 +
!    :            T*( 1602961601.2090D0 +
!    :            T*(        - 6.3706D0 +
!    :            T*(          0.006593D0 +
!    :            T*(        - 0.00003169D0 )))), TURNAS ) * DAS2R
!
!  Mean longitude of the ascending node of the Moon.
!     OM  = MOD (         450160.398036D0 +
!    :            T*(  - 6962890.5431D0 +
!    :            T*(          7.4722D0 +
!    :            T*(          0.007702D0 +
!    :            T*(        - 0.00005939D0 )))), TURNAS ) * DAS2R
!
!
!
!  Initialize the nutation values.
!     DP = 0D0
!     DE = 0D0
!
!  Summation of luni-solar nutation series (in reverse order).
!     DO 100 I = NLS, 1, -1
!
!     Argument and functions.
!        ARG = MOD ( DBLE ( NALS(1,I) ) * EL  +
!    :               DBLE ( NALS(2,I) ) * ELP +
!    :               DBLE ( NALS(3,I) ) * F   +
!    :               DBLE ( NALS(4,I) ) * D   +
!    :               DBLE ( NALS(5,I) ) * OM, D2PI )
!        SARG = SIN(ARG)
!        CARG = COS(ARG)
!
!     Term.
!        DP = DP + ( CLS(1,I) + CLS(2,I) * T ) * SARG
!    :           +   CLS(3,I)                  * CARG
!        DE = DE + ( CLS(4,I) + CLS(5,I) * T ) * CARG
!    :           +   CLS(6,I)                  * SARG
!
!100  CONTINUE
!$$    print *,'NU200A: DP,DE #1 ', DP,DE
!  Summation of luni-solar nutation series (in reverse order).
        DP = 0D0
        DE = 0D0
        dDP = 0D0
        dDE = 0D0
      DO 101 I = NLS, 1, -1
!
!     Argument and functions.
         ARG = MOD ( DBLE ( NALS(1,I) ) * FA2K(1) + &
     &               DBLE ( NALS(2,I) ) * ELP     + &
     &               DBLE ( NALS(3,I) ) * FA2K(3) + &
     &               DBLE ( NALS(4,I) ) * FA2K(4) + &
     &               DBLE ( NALS(5,I) ) * FA2K(5), D2PI )
        dARG =     ( DBLE ( NALS(1,I) ) * FAD2K(1) + &
     &               DBLE ( NALS(2,I) ) * FAD2K(2) + &
     &               DBLE ( NALS(3,I) ) * FAD2K(3) + &
     &               DBLE ( NALS(4,I) ) * FAD2K(4) + &
     &               DBLE ( NALS(5,I) ) * FAD2K(5) )
         SARG = SIN(ARG)
         CARG = COS(ARG)
         dSARG =  COS(ARG)*dARG
         dCARG = -SIN(ARG)*dARG
!
!     Term.
         DP = DP + ( CLS(1,I) + CLS(2,I) * T ) * SARG &
     &           +   CLS(3,I)                  * CARG
         DE = DE + ( CLS(4,I) + CLS(5,I) * T ) * CARG &
     &           +   CLS(6,I)                  * SARG
!
         dDP = dDP + CLS(1,I)*dSARG + CLS(2,I)*DCENT*SARG + &
     &         CLS(2,I)*T*dSARG + CLS(3,I)*dCARG
         dDE = dDE + CLS(4,I)*dCARG + CLS(5,I)*DCENT*CARG + &
     &         CLS(5,I)*T*dCARG + CLS(6,I)*dSARG
!
 101  CONTINUE
!
!  Convert from 0.1 microarcsec units to radians.
      DPSILS = DP * U2R
      DEPSLS = DE * U2R
      dDPSILS = dDP * U2R
      dDEPSLS = dDE * U2R
!
!
!  PLANETARY NUTATION
!    [Less accurate computations than FA2K(1-5)]
!
!  Mean anomaly of the Moon.
      AL   = MOD ( 2.35555598D0 + 8328.6914269554D0 * T, D2PI )
      dAL  = 8328.6914269554D0 * DCENT
!
!  Mean anomaly of the Sun.
      ALSU = MOD ( 6.24006013D0 + 628.301955D0 * T, D2PI )
      dALSU = 628.301955D0 * DCENT
!
!  Mean argument of the latitude of the Moon.
      AF  = MOD ( 1.627905234D0 + 8433.466158131D0 * T, D2PI )
      dAF = 8433.466158131D0 * DCENT
!
!  Mean elongation of the Moon from the Sun.
      AD  = MOD ( 5.198466741D0 + 7771.3771468121D0 * T, D2PI )
      dAD = 7771.3771468121D0 * DCENT
!
!  Mean longitude of the ascending node of the Moon.
      AOM  = MOD ( 2.18243920D0 - 33.757045D0 * T, D2PI )
      dAOM = -33.757045D0 * DCENT
!
!  General accumulated precession in longitude.
      APA  = ( 0.02438175D0 + 0.00000538691D0 * T ) * T
      dAPA  = 0.02438175D0*DCENT + 2.D0*0.00000538691D0*T*DCENT
!
!  Planetary longitudes, Mercury through Neptune (Souchay et al. 1999).
      ALME = MOD ( 4.402608842D0 + 2608.7903141574D0 * T, D2PI )
      ALVE = MOD ( 3.176146697D0 + 1021.3285546211D0 * T, D2PI )
      ALEA = MOD ( 1.753470314D0 +  628.3075849991D0 * T, D2PI )
      ALMA = MOD ( 6.203480913D0 +  334.0612426700D0 * T, D2PI )
      ALJU = MOD ( 0.599546497D0 +   52.9690962641D0 * T, D2PI )
      ALSA = MOD ( 0.874016757D0 +   21.3299104960D0 * T, D2PI )
! Uranus and Neptune longitudes differ slightly from IERS Standards (2003)
      ALUR = MOD ( 5.481293871D0 +    7.4781598567D0 * T, D2PI )
      ALNE = MOD ( 5.321159000D0 +    3.8127774000D0 * T, D2PI )
!
!     DALUR     = 7.4781598567D0 * DCENT
      DALNE     = 3.812777400D0 * DCENT
!
!     Write (6,780) ALME,ALVE,ALEA,ALMA,ALJU,ALSA,ALUR,ALNE
!780  Format(' NU2000A: ALME,ALVE,ALEA,ALMA,ALJU,ALSA,ALUR,ALNE:',
!    *        /,2(5E22.15,/))
!        print *,' NU2KA/DALNE: ', DALNE
!
!  Initialize the nutation values.
      DP = 0D0
      DE = 0D0
!
!  Summation of planetary nutation series (in reverse order).
!     DO 200 I = NPL, 1, -1
!
!     Argument and functions.
!        ARG = MOD ( DBLE ( NAPL( 1,I) ) * AL   +
!    :               DBLE ( NAPL( 2,I) ) * ALSU +
!    :               DBLE ( NAPL( 3,I) ) * AF   +
!    :               DBLE ( NAPL( 4,I) ) * AD   +
!    :               DBLE ( NAPL( 5,I) ) * AOM  +
!    :               DBLE ( NAPL( 6,I) ) * ALME +
!    :               DBLE ( NAPL( 7,I) ) * ALVE +
!    :               DBLE ( NAPL( 8,I) ) * ALEA +
!    :               DBLE ( NAPL( 9,I) ) * ALMA +
!    :               DBLE ( NAPL(10,I) ) * ALJU +
!    :               DBLE ( NAPL(11,I) ) * ALSA +
!    :               DBLE ( NAPL(12,I) ) * ALUR +
!    :               DBLE ( NAPL(13,I) ) * ALNE +
!    :               DBLE ( NAPL(14,I) ) * APA, D2PI )
!        SARG = SIN(ARG)
!        CARG = COS(ARG)
!
!     Term.
!        DP = DP + DBLE( ICPL(1,I)) * SARG + DBLE( ICPL(2,I)) * CARG
!        DE = DE + DBLE( ICPL(3,I)) * SARG + DBLE( ICPL(4,I)) * CARG
!
!200  CONTINUE
!$$    print *,'NU200A: DP,DE #3 ', DP,DE
!
!  Summation of planetary nutation series (in reverse order).
         DP = 0.D0
         DE = 0.D0
         dDP = 0.D0
         dDE = 0.D0
      DO 201 I = NPL, 1, -1
!
!     Argument and functions.
         ARG = MOD ( DBLE ( NAPL( 1,I) ) * AL   + &
     &               DBLE ( NAPL( 2,I) ) * ALSU + &
     &               DBLE ( NAPL( 3,I) ) * AF   + &
     &               DBLE ( NAPL( 4,I) ) * AD   + &
     &               DBLE ( NAPL( 5,I) ) * AOM  + &
     &               DBLE ( NAPL( 6,I) ) * FA2K(6) + &
     &               DBLE ( NAPL( 7,I) ) * FA2K(7) + &
     &               DBLE ( NAPL( 8,I) ) * FA2K(8) + &
     &               DBLE ( NAPL( 9,I) ) * FA2K(9) + &
     &               DBLE ( NAPL(10,I) ) * FA2K(10) + &
     &               DBLE ( NAPL(11,I) ) * FA2K(11) + &
     &               DBLE ( NAPL(12,I) ) * ALUR     + &
     &               DBLE ( NAPL(13,I) ) * ALNE     + &
     &               DBLE ( NAPL(14,I) ) * APA, D2PI )
!
        dARG =     ( DBLE ( NAPL( 1,I) ) * dAL   + &
     &               DBLE ( NAPL( 2,I) ) * dALSU + &
     &               DBLE ( NAPL( 3,I) ) * dAF   + &
     &               DBLE ( NAPL( 4,I) ) * dAD   + &
     &               DBLE ( NAPL( 5,I) ) * dAOM  + &
     &               DBLE ( NAPL( 6,I) ) * FAD2K(6) + &
     &               DBLE ( NAPL( 7,I) ) * FAD2K(7) + &
     &               DBLE ( NAPL( 8,I) ) * FAD2K(8) + &
     &               DBLE ( NAPL( 9,I) ) * FAD2K(9) + &
     &               DBLE ( NAPL(10,I) ) * FAD2K(10) + &
     &               DBLE ( NAPL(11,I) ) * FAD2K(11) + &
     &               DBLE ( NAPL(12,I) ) * FAD2K(12) + &
     &               DBLE ( NAPL(13,I) ) * DALNE     + &
     &               DBLE ( NAPL(14,I) ) * dAPA )
!
         SARG = SIN(ARG)
         CARG = COS(ARG)
         dSARG =  COS(ARG)*dARG
         dCARG = -SIN(ARG)*dARG
!
!     Term.
         DP = DP + DBLE( ICPL(1,I)) * SARG + DBLE( ICPL(2,I)) * CARG
         DE = DE + DBLE( ICPL(3,I)) * SARG + DBLE( ICPL(4,I)) * CARG
        dDP = dDP + DBLE(ICPL(1,I)) * dSARG + DBLE(ICPL(2,I)) * dCARG
        dDE = dDE + DBLE(ICPL(3,I)) * dSARG + DBLE(ICPL(4,I)) * dCARG
!
 201  CONTINUE
!
!  Convert from 0.1 microarcsec units to radians.
      DPSIPL = DP * U2R
      DEPSPL = DE * U2R
      dDPSIPL = dDP * U2R
      dDEPSPL = dDE * U2R
!
!
!  TOTAL
!
!  Add planetary and luni-solar components.
      DPSI(1) = DPSIPL + DPSILS
      DEPS(1) = DEPSPL + DEPSLS
      DPSI(2) = dDPSIPL + dDPSILS
      DEPS(2) = dDEPSPL + dDEPSLS
!
      Return
      END
!************************************************************************
!  Copyright (C) 2001
!  Standards Of Fundamental Astronomy Review Board
!  of the International Astronomical Union.
!
!  This routine is part of the International Astronomical Union's
!  SOFA (Standards of Fundamental Astronomy) software collection.
!  =====================
!  SOFA Software License
!  =====================
!
!  NOTICE TO USER:
!
!  BY USING THIS SOFTWARE YOU ACCEPT THE FOLLOWING TERMS AND CONDITIONS
!  WHICH APPLY TO ITS USE.
!
!  1. The Software is owned by the IAU SOFA Review Board ("the Board").
!
!  2. The Software is made available free of charge for use by:
!
!     a) private individuals for non-profit research; and
!
!     b) non-profit educational, academic and research institutions.
!
!  3. Commercial use of the Software is specifically excluded from the
!     terms and conditions of this license.  Commercial use of the
!     Software is subject to the prior written agreement of the Board on
!     terms to be agreed.
!
!  4. The provision of any version of the Software under the terms and
!     conditions specified herein does not imply that future versions
!     will also be made available under the same terms and conditions.
!
!  5. The user may modify the Software for his/her own purposes.  The
!     user may distribute the modified software provided that the Board
!     is informed and that a copy of the modified software is made
!     available to the Board on request.  All modifications made by the
!     user shall be clearly identified to show how the modified software
!     differs from the original Software, and the name(s) of the
!     affected routine(s) shall be changed.  The original SOFA Software
!     License text must be present.
!
!  6. In any published work produced by the user and which includes
!     results achieved by using the Software, the user shall acknowledge
!     that the Software was used in producing the information contained
!     in such publication.
!
!  7. The user may incorporate or embed the Software into other software
!     products which he/she may then give away free of charge but not
!     sell provided the user makes due acknowledgement of the use which
!     he/she has made of the Software in creating such software
!     products.  Any redistribution of the Software in this way shall be
!     made under the same terms and conditions under which the user
!     received it from the SOFA Center.
!
!  8. The user shall not cause the Software to be brought into
!     disrepute, either by misuse, or use for inappropriate tasks, or by
!     inappropriate modification.
!
!  9. The Software is provided to the user "as is" and the Board makes
!     no warranty as to its use or performance.   The Board does not and
!     cannot warrant the performance or results which the user may
!     obtain by using the Software.  The Board makes no warranties,
!     express or implied, as to non-infringement of third party rights,
!     merchantability, or fitness for any particular purpose.  In no
!     event will the Board be liable to the user for any consequential,
!     incidental, or special damages, including any lost profits or lost
!     savings, even if a Board representative has been advised of such
!     damages, or for any claim by any third party.
!
!  Correspondence concerning SOFA software should be addressed as
!  follows:
!
!     Internet email: sofa@rl.ac.uk
!     Postal address: IAU SOFA Center
!                     Rutherford Appleton Laboratory
!                     Chilton, Didcot, Oxon OX11 0QX
!                     United Kingdom
!
!***********************************************************************
      BLOCK DATA NUTCMB
      IMPLICIT None
!
! 7.    NUTBD
!
! 7.1   NUTBD PROGRAM SPECIFICATION
!
! 7.1.1 NUTBD IS THE NUTATION MODULE BLOCK DATA INITIALIZATION SECTION.
!       THE NUTATION SERIES IS ESTABLISHED HERE. THIS VERSION CONTAINS
!       THE 1980 IAU THEORY OF NUTATION, FROM THE WORK OF J. M. WAHR,
!       SPECIFICALLY, THE WAHR NUTATION SERIES FOR AXIS B OF GILBERT &
!       DZIEWONSKI EARTH MODEL 1066A.
!
! 7.1.3 REFERENCES - 1) 'THE EXPLANATORY SUPPLEMENT TO THE AMERICAN
!                    EPHEMERIS AND NAUTICAL ALMANAC", P. 41-45, 98
!
!                    2) LIESKE, J.H., ET AL., EXPRESSIONS FOR THE
!                    PRECESSIONAL QUANTITIES BASED ON THE IAU (1976)
!                    SYSTEM OF ASTRONOMICAL CONSTANTS,
!                    ASTRON. ASTROPHYS. 58, 1-16, 1977.
!
!                    3) SEIDELMANN, P. K., 1980 IAU THEORY OF NUTATION:
!                    THE FINAL REPORT OF THE IAU WORKING GROUP ON
!                    NUTATION, CELEST. MECH. 27, PP. 79-106 (1982).
!
!                    4) WAHR, J. M., THE FORCED NUTATIONS OF ... EARTH,
!                    GEOPHYS. J. ROY. ASTR. SOC. 64, PP. 705-727 (1981).
!
! 7.2   NUTBD PROGRAM INTERFACE
!
! 7.2.1 CALLING SEQUENCE - NONE
!
! 7.2.2 COMMON BLOCK -
!
      REAL*8 X(9,120)
      COMMON / XWAHR / X
!
!!!!  Real*8 CENTJ, DJ2000, EC(4), ARGP(2,6)
      Real*8                EC(4)
!!!   Integer*4 NOT, NOP, IDP(6)
      Integer*4 NOT
!!!!  COMMON / NUTCM / CENTJ, DJ2000, EC, ARGP, NOT, NOP, IDP
      COMMON / NUTCMW /                EC,       NOT
!           VARIABLES 'TO':
!!!!           1. CENTJ   - THE NUMBER OF COORDINATE TIME DAYS PER JULIAN
!!!!                        CENTURY. (DAYS/CENTURY) (CENTJ = 36525.D0)
!!!!           2. DJ2000  - THE JULIAN DATE OF THE EPOCH JANUARY 1.5, 2000.
!!!!                        (DAYS) (DJ2000 = 2451545.D0)
!              3. EC(4)   - THE CONSTANTS APPEARING IN TERMS 1-4 IN THE
!                           CALCULATION OF THE MEAN OBLIQUITY OF THE ECLIPTIC.
!                           (ARCSEC) (SEE REFERENCES)
!                           ( EC(1) = +8.4381448D4, EC(2) = -46.815D0,
!                             EC(3) = -5.9D-4, EC(4) = +1.813D-3 )
!              4. NOT     - THE NUMBER OF TERMS IN THE NUTATION SERIES.
!                           (NOT = 106)
!              5. X(9,120)- THE ARRAY CONTAINING THE NUTATION SERIES.
!                           (X = 1980 IAU THEORY OF NUTATION)
!!!!           6. NOP     - THE NUMBER OF NUTATION TERMS DESIGNATED FOR WHICH
!!!!                        PARTIALS ARE TO BE COMPUTED. (NOP = 6)
!                                  (Obsolete?)
!!!!           7. IDP(6)  - IDENTIFICATION NUMBERS (TERM NUMBERS) OF DESIGNATED
!!!!                        NUTATION TERMS FOR WHICH PARTIALS ARE TO BE COMPUTED
!!!!                        ( IDP(1) =  1, IDP(2) =  2, IDP(3) =  3,
!!!!                          IDP(4) =  4, IDP(5) =  5, IDP(6) =  7 )
!!!!           8. ARGP(2,6)-ARGUMENTS (COMBINATIONS OF FUNDAMENTAL ARGUMENTS)
!!!!                        AND THEIR DERIVATIVES OF DESIGNATED NUTATION TERMS
!!!!                        FOR WHICH PARTIALS ARE TO BE COMPUTED.
!!!!                        (COMPUTED IN NUTW. SET TO 0.0D0 HERE)
!!!!                               (Obsolete?)
!
! 7.2.3 PROGRAM SPECIFICATIONS -
      Real*8  X1(180), X2(180), X3(180), X4(180), X5(180), X6(180)
      EQUIVALENCE (X(1,  1),X1(1))
      EQUIVALENCE (X(1, 21),X2(1))
      EQUIVALENCE (X(1, 41),X3(1))
      EQUIVALENCE (X(1, 61),X4(1))
      EQUIVALENCE (X(1, 81),X5(1))
      EQUIVALENCE (X(1,101),X6(1))
!
!!!   DATA  CENTJ  / 36525.D0 /, &
!!!  &      DJ2000 / 2451545.D0 /, &
      DATA  EC     / 8.4381448D4, -46.8150D0, -5.9D-4, 1.813D-3 /, &
     &      NOT    / 106 /
!!!  &      NOT    / 106 /, &
!!!  &      NOP    / 6 /, &
!!!  &      IDP    / 1, 2, 3, 4, 5, 7 /, &
!!!  &      ARGP   / 12 * 0.0D0 /
!***********************************************************************
!
!               1980 IAU THEORY OF NUTATION (WAHR THEORY)
!           TABLE OF MULTIPLES OF ARGUMENTS AND COEFFICIENTS
!
!                   MULTIPLE OF            LONGITUDE        OBLIQUITY
!              L    L'   F    D  OMEGA   COEFF. OF SIN    COEFF. OF COS
      DATA X1/ 0.,  0.,  0.,  0.,  1., -171996., -174.2,  92025.,  8.9, &
     &         0.,  0.,  2., -2.,  2.,  -13187.,   -1.6,   5736., -3.1, &
     &         0.,  0.,  2.,  0.,  2.,   -2274.,   -0.2,    977., -0.5, &
     &         0.,  0.,  0.,  0.,  2.,    2062.,    0.2,   -895.,  0.5, &
     &         0.,  1.,  0.,  0.,  0.,    1426.,   -3.4,     54., -0.1, &
     &         1.,  0.,  0.,  0.,  0.,     712.,    0.1,     -7.,  0.0, &
     &         0.,  1.,  2., -2.,  2.,    -517.,    1.2,    224., -0.6, &
     &         0.,  0.,  2.,  0.,  1.,    -386.,   -0.4,    200.,  0.0, &
     &         1.,  0.,  2.,  0.,  2.,    -301.,    0.0,    129., -0.1, &
     &         0., -1.,  2., -2.,  2.,     217.,   -0.5,    -95.,  0.3, &
     &         1.,  0.,  0., -2.,  0.,    -158.,    0.0,     -1.,  0.0, &
     &         0.,  0.,  2., -2.,  1.,     129.,    0.1,    -70.,  0.0, &
     &        -1.,  0.,  2.,  0.,  2.,     123.,    0.0,    -53.,  0.0, &
     &         1.,  0.,  0.,  0.,  1.,      63.,    0.1,    -33.,  0.0, &
     &         0.,  0.,  0.,  2.,  0.,      63.,    0.0,     -2.,  0.0, &
     &        -1.,  0.,  2.,  2.,  2.,     -59.,    0.0,     26.,  0.0, &
     &        -1.,  0.,  0.,  0.,  1.,     -58.,   -0.1,     32.,  0.0, &
     &         1.,  0.,  2.,  0.,  1.,     -51.,    0.0,     27.,  0.0, &
     &         2.,  0.,  0., -2.,  0.,      48.,    0.0,      1.,  0.0, &
     &        -2.,  0.,  2.,  0.,  1.,      46.,    0.0,    -24.,  0.0/
      DATA X2/ 0.,  0.,  2.,  2.,  2.,     -38.,    0.0,     16.,  0.0, &
     &         2.,  0.,  2.,  0.,  2.,     -31.,    0.0,     13.,  0.0, &
     &         2.,  0.,  0.,  0.,  0.,      29.,    0.0,     -1.,  0.0, &
     &         1.,  0.,  2., -2.,  2.,      29.,    0.0,    -12.,  0.0, &
     &         0.,  0.,  2.,  0.,  0.,      26.,    0.0,     -1.,  0.0, &
     &         0.,  0.,  2., -2.,  0.,     -22.,    0.0,      0.,  0.0, &
     &        -1.,  0.,  2.,  0.,  1.,      21.,    0.0,    -10.,  0.0, &
     &         0.,  2.,  0.,  0.,  0.,      17.,   -0.1,      0.,  0.0, &
     &         0.,  2.,  2., -2.,  2.,     -16.,    0.1,      7.,  0.0, &
     &        -1.,  0.,  0.,  2.,  1.,      16.,    0.0,     -8.,  0.0, &
     &         0.,  1.,  0.,  0.,  1.,     -15.,    0.0,      9.,  0.0, &
     &         1.,  0.,  0., -2.,  1.,     -13.,    0.0,      7.,  0.0, &
     &         0., -1.,  0.,  0.,  1.,     -12.,    0.0,      6.,  0.0, &
     &         2.,  0., -2.,  0.,  0.,      11.,    0.0,      0.,  0.0, &
     &        -1.,  0.,  2.,  2.,  1.,     -10.,    0.0,      5.,  0.0, &
     &         1.,  0.,  2.,  2.,  2.,      -8.,    0.0,      3.,  0.0, &
     &         0., -1.,  2.,  0.,  2.,      -7.,    0.0,      3.,  0.0, &
     &         0.,  0.,  2.,  2.,  1.,      -7.,    0.0,      3.,  0.0, &
     &         1.,  1.,  0., -2.,  0.,      -7.,    0.0,      0.,  0.0, &
     &         0.,  1.,  2.,  0.,  2.,       7.,    0.0,     -3.,  0.0/
      DATA X3/-2.,  0.,  0.,  2.,  1.,      -6.,    0.0,      3.,  0.0, &
     &         0.,  0.,  0.,  2.,  1.,      -6.,    0.0,      3.,  0.0, &
     &         2.,  0.,  2., -2.,  2.,       6.,    0.0,     -3.,  0.0, &
     &         1.,  0.,  0.,  2.,  0.,       6.,    0.0,      0.,  0.0, &
     &         1.,  0.,  2., -2.,  1.,       6.,    0.0,     -3.,  0.0, &
     &         0.,  0.,  0., -2.,  1.,      -5.,    0.0,      3.,  0.0, &
     &         0., -1.,  2., -2.,  1.,      -5.,    0.0,      3.,  0.0, &
     &         2.,  0.,  2.,  0.,  1.,      -5.,    0.0,      3.,  0.0, &
     &         1., -1.,  0.,  0.,  0.,       5.,    0.0,      0.,  0.0, &
     &         1.,  0.,  0., -1.,  0.,      -4.,    0.0,      0.,  0.0, &
     &         0.,  0.,  0.,  1.,  0.,      -4.,    0.0,      0.,  0.0, &
     &         0.,  1.,  0., -2.,  0.,      -4.,    0.0,      0.,  0.0, &
     &         1.,  0., -2.,  0.,  0.,       4.,    0.0,      0.,  0.0, &
     &         2.,  0.,  0., -2.,  1.,       4.,    0.0,     -2.,  0.0, &
     &         0.,  1.,  2., -2.,  1.,       4.,    0.0,     -2.,  0.0, &
     &         1.,  1.,  0.,  0.,  0.,      -3.,    0.0,      0.,  0.0, &
     &         1., -1.,  0., -1.,  0.,      -3.,    0.0,      0.,  0.0, &
     &        -1., -1.,  2.,  2.,  2.,      -3.,    0.0,      1.,  0.0, &
     &         0., -1.,  2.,  2.,  2.,      -3.,    0.0,      1.,  0.0, &
     &         1., -1.,  2.,  0.,  2.,      -3.,    0.0,      1.,  0.0/
      DATA X4/ 3.,  0.,  2.,  0.,  2.,      -3.,    0.0,      1.,  0.0, &
     &        -2.,  0.,  2.,  0.,  2.,      -3.,    0.0,      1.,  0.0, &
     &         1.,  0.,  2.,  0.,  0.,       3.,    0.0,      0.,  0.0, &
     &        -1.,  0.,  2.,  4.,  2.,      -2.,    0.0,      1.,  0.0, &
     &         1.,  0.,  0.,  0.,  2.,      -2.,    0.0,      1.,  0.0, &
     &        -1.,  0.,  2., -2.,  1.,      -2.,    0.0,      1.,  0.0, &
     &         0., -2.,  2., -2.,  1.,      -2.,    0.0,      1.,  0.0, &
     &        -2.,  0.,  0.,  0.,  1.,      -2.,    0.0,      1.,  0.0, &
     &         2.,  0.,  0.,  0.,  1.,       2.,    0.0,     -1.,  0.0, &
     &         3.,  0.,  0.,  0.,  0.,       2.,    0.0,      0.,  0.0, &
     &         1.,  1.,  2.,  0.,  2.,       2.,    0.0,     -1.,  0.0, &
     &         0.,  0.,  2.,  1.,  2.,       2.,    0.0,     -1.,  0.0, &
     &         1.,  0.,  0.,  2.,  1.,      -1.,    0.0,      0.,  0.0, &
     &         1.,  0.,  2.,  2.,  1.,      -1.,    0.0,      1.,  0.0, &
     &         1.,  1.,  0., -2.,  1.,      -1.,    0.0,      0.,  0.0, &
     &         0.,  1.,  0.,  2.,  0.,      -1.,    0.0,      0.,  0.0, &
     &         0.,  1.,  2., -2.,  0.,      -1.,    0.0,      0.,  0.0, &
     &         0.,  1., -2.,  2.,  0.,      -1.,    0.0,      0.,  0.0, &
     &         1.,  0., -2.,  2.,  0.,      -1.,    0.0,      0.,  0.0, &
     &         1.,  0., -2., -2.,  0.,      -1.,    0.0,      0.,  0.0/
      DATA X5/ 1.,  0.,  2., -2.,  0.,      -1.,    0.0,      0.,  0.0, &
     &         1.,  0.,  0., -4.,  0.,      -1.,    0.0,      0.,  0.0, &
     &         2.,  0.,  0., -4.,  0.,      -1.,    0.0,      0.,  0.0, &
     &         0.,  0.,  2.,  4.,  2.,      -1.,    0.0,      0.,  0.0, &
     &         0.,  0.,  2., -1.,  2.,      -1.,    0.0,      0.,  0.0, &
     &        -2.,  0.,  2.,  4.,  2.,      -1.,    0.0,      1.,  0.0, &
     &         2.,  0.,  2.,  2.,  2.,      -1.,    0.0,      0.,  0.0, &
     &         0., -1.,  2.,  0.,  1.,      -1.,    0.0,      0.,  0.0, &
     &         0.,  0., -2.,  0.,  1.,      -1.,    0.0,      0.,  0.0, &
     &         0.,  0.,  4., -2.,  2.,       1.,    0.0,      0.,  0.0, &
     &         0.,  1.,  0.,  0.,  2.,       1.,    0.0,      0.,  0.0, &
     &         1.,  1.,  2., -2.,  2.,       1.,    0.0,     -1.,  0.0, &
     &         3.,  0.,  2., -2.,  2.,       1.,    0.0,      0.,  0.0, &
     &        -2.,  0.,  2.,  2.,  2.,       1.,    0.0,     -1.,  0.0, &
     &        -1.,  0.,  0.,  0.,  2.,       1.,    0.0,     -1.,  0.0, &
     &         0.,  0., -2.,  2.,  1.,       1.,    0.0,      0.,  0.0, &
     &         0.,  1.,  2.,  0.,  1.,       1.,    0.0,      0.,  0.0, &
     &        -1.,  0.,  4.,  0.,  2.,       1.,    0.0,      0.,  0.0, &
     &         2.,  1.,  0., -2.,  0.,       1.,    0.0,      0.,  0.0, &
     &         2.,  0.,  0.,  2.,  0.,       1.,    0.0,      0.,  0.0/
      DATA X6/ 2.,  0.,  2., -2.,  1.,       1.,    0.0,     -1.,  0.0, &
     &         2.,  0., -2.,  0.,  1.,       1.,    0.0,      0.,  0.0, &
     &         1., -1.,  0., -2.,  0.,       1.,    0.0,      0.,  0.0, &
     &        -1.,  0.,  0.,  1.,  1.,       1.,    0.0,      0.,  0.0, &
     &        -1., -1.,  0.,  2.,  1.,       1.,    0.0,      0.,  0.0, &
     &         0.,  1.,  0.,  1.,  0.,       1.,    0.0,      0.,  0.0, &
     &                      126 *  0./
!
!***********************************************************************
!
! 7.2.4 CONSTANTS USED - CENTJ, DJ2000, EC(4), NOT, X(9,120),
!                    NOP, IDP(6), ARGP(2,6)
!
! 7.2.5 PROGRAMMER - DALE MARKHAM   01/13/77
!                    PETER DENATALE 07/12/77
!                    BRUCE SCHUPLER 12/22/77
!                    CHOPO MA       08/04/81
!                    GEORGE KAPLAN  04/24/84
!                    David Gordon   94.04.15 Converted to Implicit None.
!                    David Gordon   95.09.27 X(9,120) table changed from Real*4
!                                   to Real*8
!                    David Gordon   98.02.03 Removed X(9,120) from COMMON
!                                   /NUTCM/ and put it into COMMON /XWAHR/,
!                                   and removed it from most subroutines.
!
! 7.3   NUTBD PROGRAM STRUCTURE - NONE
!
      END
!
!**********************************************************************
      SUBROUTINE NUTW (CENT, DPSI, DEPS)
      IMPLICIT None
!
! 1.    NUTW
!
! 1.1   NUTW PROGRAM SPECIFICATION
!
! 1.1.1 THIS SUBROUTINE EVALUATES THE NUTATION SERIES AND RETURNS THE
!       VALUES FOR NUTATION IN LONGITUDE AND NUTATION IN OBLIQUITY.
!       The fundamental arguments and derivatives are calculated
!       in NUTFA to be called prior to NUTW.
!
! 1.1.2 RESTRICTIONS - NONE
!
! 1.1.3 REFERENCES - 1) KAPLAN, G. H. (ED.), THE IAU RESOLUTIONS ... ,
!                       USNO CIRCULAR NO. 163, U. S. NAVAL OBSERV'Y (1981).
!
! 1.2   NUTW PROGRAM INTERFACE
!
! 1.2.1 CALLING SEQUENCE - CALL NUTW(CENT, fa, fad, DPSI, DEPS)
!    INPUT VARIABLES:
!      1. CENT   - The number of Julian centuries between the observation
!                  epoch and January 1.5, 2000. (centuries)
!      2. FA(5)  - Fundamental arguments. (arcseconds) (see NUTFA)
!      3. FAD(5) - The CT derivatives of the fundamental arguments. (see NUTFA)
!    OUTPUT VARIABLES:
!      1. DPSI(2) - THE NUTATION IN LONGITUDE AND ITS CT TIME DERIVATIVE
!                   (ARCSEC,ARCSEC/SEC)
!      2. DEPS(2) - THE NUTATION IN OBLIQUITY AND ITS CT TIME DERIVATIVE
!                   (ARCSEC,ARCSEC/SEC)
!
! 1.2.2 COMMON BLOCKS USED -
!
      Real*8           PI, TWOPI, HALFPI, CONVD, CONVDS, CONVHS, SECDAY
      COMMON / CMATH / PI, TWOPI, HALFPI, CONVD, CONVDS, CONVHS, SECDAY
!     VARIABLES 'FROM':
!       1. CONVDS - THE CONVERSION FACTOR FROM ARCSECONDS TO RADIANS.
!       2. SECDAY - THE NUMBER OF COORDINATE TIME SECONDS PER COORDINATE
!                   TIME DAY. (SECONDS/DAY)
!
      INCLUDE 'ccon.i'
!     VARIABLES 'FROM':
!       1. KNUTD - THE NUTATION MODULE DEBUG CONTROL FLAG
!
      REAL*8 FA(5), FAD(5)
!!!!  Real*8 CENTJ, DJ2000, EC(4), ARGP(2,6)
      Real*8 CENTJ, DJ2000, EC(4)
!!!!  Integer*4 NOT, NOP, IDP(6)
      Integer*4 NOT
!!!!  COMMON / NUTCM / CENTJ, DJ2000, EC, ARGP, NOT, NOP, IDP
      COMMON / NUTCMW /                EC,       NOT
      COMMON / NUTCM / CENTJ, DJ2000
      REAL*8 X(9,120)
      COMMON / XWAHR / X
!
!     VARIABLES 'FROM':
!        1. CENTJ - THE NUMBER OF COORDINATE TIME DAYS PER JULIAN CENTURY.
!                   (DAYS/CENTURY)
!        2. NOT   - THE NUMBER OF TERMS IN THE NUTATION SERIES.
!        3. X(9,120)-ARRAY CONTAINING THE NUTATION SERIES.
!!!!     4. NOP   - THE NUMBER OF NUTATION TERMS DESIGNATED FOR WHICH
!!!!                PARTIALS ARE TO BE COMPUTED. (NOP .LE. 6)
!!!!                    (obsolete?)
!!!!     5. IDP(6)- IDENTIFICATION NUMBERS (TERM NUMBERS) OF DESIGNATED
!!!!                NUTATION TERMS FOR WHICH PARTIALS ARE TO BE COMPUTED.
!
!     VARIABLES 'TO':
!!!!     1. ARGP(2,6) - ARGUMENTS (COMBINATIONS OF FUNDAMENTAL ARGUMENTS)
!!!!                    AND THEIR DERIVATIVES OF DESIGNATED NUTATION TERMS
!!!!                    FOR WHICH PARTIALS ARE TO BE COMPUTED.
!!!!                        (obsolete?)
!
! 1.2.3 PROGRAM SPECIFICATIONS
!
      Real*8 DPSI(2),DEPS(2), CENT, SEC360, ARG, ARGDOT
      REAL*8 EL, ELP, F, D, OM, ELC(5), ELPC(5), FC(5), &
     &       DC(5), OMC(5), CENT2, CENT3
      Integer*4 I, J, N
!
!  Code added from Calc 8!!!
!     Constants used -
!       ELC(5)   - COEFFICIENTS USED IN THE CALCULATION OF EL
!       ELPC(5)  - COEFFICIENTS USED IN THE CALCULATION OF ELP
!       FC(5)    - COEFFICIENTS USED IN THE CALCULATION OF F
!       DC(5)    - COEFFICIENTS USED IN THE CALCULATION OF D
!       OMC(5)   - COEFFICIENTS USED IN THE CALCULATION OF OM
!
!       COEFFICIENTS ARE IN THE ORDER CUBIC, QUDRATIC, LINEAR, CONSTANT,
!       AND INTEGRAL NUMBER OF TURNS.
!
      DATA ELC /0.064D0,31.310D0,715922.633D0,485866.733D0,1325.0D0/
      DATA ELPC /-0.012D0,-0.577D0,1292581.224D0,1287099.804D00,99.0D0/
      DATA FC /0.011D0,-13.257D0,295263.137D0,335778.877D0,1342.0D0/
      DATA DC /0.019D0,-6.891D0,1105601.328D0,1072261.307D0,1236.0D0/
      DATA OMC /0.008D0,7.455D0,-482890.539D0,450160.280D0,-5.0D0/
!
!   SEC360 = ARCSECONDS IN ONE TURN
      DATA SEC360 / 1296000.D0 /
!
!
! 1.2.4 DATA BASE ACCESS - NONE
!
! 1.2.5 EXTERNAL INPUT/OUTPUT - POSSIBLE DEBUG
!
! 1.2.6 SUBROUTINE INTERFACE
!       CALLER SUBROUTINES - NUTG
!       CALLED SUBROUTINES - DSIN, DCOS, DMOD
!
!
! 1.2.8 PROGRAM VARIABLES
!      1. ARG    - THE COMBINATION OF FUNDAMENTAL ARGUMENTS USED
!                  TO COMPUTE THE NUTATION (RADIANS)
!      2. ARGDOT - THE DERIVATIVE OF ARG (RADIANS/CENTURY)
!
! 1.2.9 PROGRAMMER:
!          810804  CHOPO MA
!          840000  GEORGE KAPLAN
!          930901  Norbert Zacharias: put fundamental arguments computation
!                  into subroutine NUTFA.
!          David Gordon 94.04.15 Converted to Implicit None.
!          David Gordon 95.09.27 Conversion of nutation series coefficients to
!                  double precision removed - no longer necessary.
!          Jim Ryan 2002.09 Integer*4 conversion.
!          David Gordon 2004.07.30 Adding computation of fundamental
!                  arguments consistent with Wahr nutation, taken from
!                  Calc 8 and IERS Standards (1992).
!
!  Code added to compute fundamental arguments consistent with Wahr nutation,
!   taken from Calc 8.
      CENT2 = CENT * CENT
      CENT3 = CENT * CENT2
!
!  Computation of the fundamental arguments and derivatives
!
      EL = ELC(1) * CENT3 + ELC(2) * CENT2 + ELC(3) * CENT &
     &     + ELC(4) + DMOD( ELC(5) * CENT, 1.D0 ) * SEC360
      FA (1) = DMOD( EL, SEC360 )
      FAD(1) = 3.D0 * ELC(1) * CENT2 + 2.D0 * ELC(2) * CENT &
     &         + ELC(3) + ELC(5) * SEC360
!
      ELP = ELPC(1) * CENT3 + ELPC(2) * CENT2 + ELPC(3) * CENT &
     &     + ELPC(4) + DMOD( ELPC(5) * CENT, 1.D0 ) * SEC360
      FA (2) = DMOD( ELP, SEC360 )
      FAD(2) = 3.D0 * ELPC(1) * CENT2 + 2.D0 * ELPC(2) * CENT &
     &         + ELPC(3) + ELPC(5) * SEC360
!
      F = FC(1) * CENT3 + FC(2) * CENT2 + FC(3) * CENT &
     &     + FC(4) + DMOD( FC(5) * CENT, 1.D0 ) * SEC360
      FA (3) = DMOD( F, SEC360 )
      FAD(3) = 3.D0 * FC(1) * CENT2 + 2.D0 * FC(2) * CENT &
     &         + FC (3) + FC(5) * SEC360
!
      D = DC(1) * CENT3 + DC(2) * CENT2 + DC(3) * CENT &
     &     + DC(4) + DMOD( DC(5) * CENT, 1.D0 ) * SEC360
      FA (4) = DMOD( D, SEC360 )
      FAD(4) = 3.D0 * DC(1) * CENT2 + 2.D0 * DC(2) * CENT &
     &         + DC(3) + DC(5) * SEC360
!
      OM = OMC(1) * CENT3 + OMC(2) * CENT2 + OMC(3) * CENT &
     &     + OMC(4) + DMOD( OMC(5) * CENT, 1.D0 ) * SEC360
      FA (5) = DMOD( OM, SEC360 )
      FAD(5) = 3.D0 * OMC(1) * CENT2 + 2.D0 * OMC(2) * CENT &
     &         + OMC(3) + OMC(5) * SEC360
!
!  INITIALIZE OUTPUT
      DPSI(1) = 0.0D0
      DPSI(2) = 0.0D0
      DEPS(1) = 0.0D0
      DEPS(2) = 0.0D0
!
!  SUM NUTATION SERIES TERMS, FROM SMALLEST TO LARGEST
!
!!!   N = NOP
!
      DO J=1,NOT
        I = NOT + 1 - J
!
!   FORMATION OF MULTIPLES OF ARGUMENTS
!
        ARG = X(1,I) * FA(1) &                   ! EL    arcseconds
     &      + X(2,I) * FA(2) &                   ! ELP
     &      + X(3,I) * FA(3) &                   ! F
     &      + X(4,I) * FA(4) &                   ! D
     &      + X(5,I) * FA(5)                  ! OM
        ARG = DMOD(ARG,SEC360) * CONVDS     ! radian
!
!   FORMATION OF MULTIPLES FOR DERIVATIVES
!
        ARGDOT = X(1,I) * FAD(1) &   ! ELD
     &         + X(2,I) * FAD(2) &   ! ELPD
     &         + X(3,I) * FAD(3) &   ! FD
     &         + X(4,I) * FAD(4) &   ! DD
     &         + X(5,I) * FAD(5)  ! OMD
        ARGDOT = ARGDOT * CONVDS
!
!   STORE VALUES OF ARGUMENTS AND DERIVATIVES OF SPECIFIC TERMS
!
!!!     IF (N.GT.0) THEN
!!!       IF (IDP(N).EQ.I) THEN
!!!         ARGP(1,N) = ARG
!!!         ARGP(2,N) = ARGDOT
!!!         N = N - 1
!!!       ENDIF
!!!     ENDIF
!
!   EVALUATE NUTATION AND DERIVATIVES OF NUTATION
!
        DPSI(1) = (X(6,I) + X(7,I)*CENT) * DSIN(ARG) &
     &          + DPSI(1)
        DPSI(2) = DPSI(2) + X(7,I) * DSIN(ARG) + (X(6,I) &
     &          + X(7,I) * CENT) * ARGDOT * DCOS(ARG)
        DEPS(1) = (X(8,I) + X(9,I)*CENT) * DCOS(ARG) &
     &          + DEPS(1)
        DEPS(2) = DEPS(2) + X(9,I) * DCOS(ARG) - (X(8,I) &
     &          + X(9,I) * CENT) * ARGDOT * DSIN(ARG)
!
      ENDDO  ! j=1,not
!
!   CONVERT TO PROPER UNITS
!
      DPSI(1) = DPSI(1) * 1.0D-4
      DPSI(2) = DPSI(2) * 1.0D-4 / (CENTJ * SECDAY)
      DEPS(1) = DEPS(1) * 1.0D-4
      DEPS(2) = DEPS(2) * 1.0D-4 / (CENTJ * SECDAY)
!
! SEE IF WE NEED DEBUG OUTPUT
      IF (KNUTD .NE. 0) THEN
        WRITE (6,9)
    9   FORMAT (1X,'Debug output for subroutine NUTW.')
    8   FORMAT(A,4D25.16/(7X,5D25.16))
        WRITE(6,8)' SEC360  ',SEC360
        WRITE(6,8)' ARG     ',ARG
        WRITE(6,8)' ARGDOT  ',ARGDOT
        WRITE (6,9200) CONVDS, CENTJ, SECDAY, CENT, DEPS, DPSI
9200    FORMAT (1X,'CONVDS = ',D25.16,/,1X,'CENTJ = ',D25.16,/,1X, &
     &       'SECDAY = ',D25.16,/,1X,'CENT = ',D25.16,/,1X, &
     &       'DEPS = ',2(D25.16,2X),/,1X,'DPSI = ',2(D25.16,2X), /)
      ENDIF
!
      RETURN
      END
!***************************************************************
