import { Duration } from '@aws-cdk/core';
/**
 * An ECR life cycle rule.
 *
 * @stability stable
 */
export interface LifecycleRule {
    /**
     * Controls the order in which rules are evaluated (low to high).
     *
     * All rules must have a unique priority, where lower numbers have
     * higher precedence. The first rule that matches is applied to an image.
     *
     * There can only be one rule with a tagStatus of Any, and it must have
     * the highest rulePriority.
     *
     * All rules without a specified priority will have incrementing priorities
     * automatically assigned to them, higher than any rules that DO have priorities.
     *
     * @default Automatically assigned
     * @stability stable
     */
    readonly rulePriority?: number;
    /**
     * Describes the purpose of the rule.
     *
     * @default No description
     * @stability stable
     */
    readonly description?: string;
    /**
     * Select images based on tags.
     *
     * Only one rule is allowed to select untagged images, and it must
     * have the highest rulePriority.
     *
     * @default TagStatus.Tagged if tagPrefixList is given, TagStatus.Any otherwise
     * @stability stable
     */
    readonly tagStatus?: TagStatus;
    /**
     * Select images that have ALL the given prefixes in their tag.
     *
     * Only if tagStatus == TagStatus.Tagged
     *
     * @stability stable
     */
    readonly tagPrefixList?: string[];
    /**
     * The maximum number of images to retain.
     *
     * Specify exactly one of maxImageCount and maxImageAge.
     *
     * @stability stable
     */
    readonly maxImageCount?: number;
    /**
     * The maximum age of images to retain. The value must represent a number of days.
     *
     * Specify exactly one of maxImageCount and maxImageAge.
     *
     * @stability stable
     */
    readonly maxImageAge?: Duration;
}
/**
 * Select images based on tags.
 *
 * @stability stable
 */
export declare enum TagStatus {
    /**
     * Rule applies to all images.
     *
     * @stability stable
     */
    ANY = "any",
    /**
     * Rule applies to tagged images.
     *
     * @stability stable
     */
    TAGGED = "tagged",
    /**
     * Rule applies to untagged images.
     *
     * @stability stable
     */
    UNTAGGED = "untagged"
}
