# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['not_again_ai', 'not_again_ai.system', 'not_again_ai.viz']

package_data = \
{'': ['*']}

install_requires = \
['numpy>=1.22.4,<2.0.0', 'pandas>=1.4.2,<2.0.0', 'seaborn>=0.11.2,<0.12.0']

entry_points = \
{'console_scripts': ['not-again-ai = not_again_ai.cli:entry_point']}

setup_kwargs = {
    'name': 'not-again-ai',
    'version': '0.1.0',
    'description': 'Designed to once and for all collect all the little things that come up over and over again in AI projects and put them in one place.',
    'long_description': '# not-again-ai\n\n[![GitHub Actions][github-actions-badge]](https://github.com/DaveCoDev/not-again-ai/actions)\n[![Code style: black][black-badge]](https://github.com/psf/black)\n[![Imports: isort][isort-badge]](https://pycqa.github.io/isort/)\n[![Type checked with mypy][mypy-badge]](https://github.com/python/mypy)\n\n[github-actions-badge]: https://github.com/DaveCoDev/not-again-ai/workflows/python/badge.svg\n[black-badge]: https://img.shields.io/badge/code%20style-black-000000.svg\n[isort-badge]: https://img.shields.io/badge/%20imports-isort-%231674b1?style=flat&labelColor=ef8336\n[mypy-badge]: https://img.shields.io/badge/type%20checked-mypy-blue.svg\n\nHave you ever been working on a project and groaned as you go to search again on how to nicely plot a simple distribution? Or have you been frustrated at wanting to run multiple functions in parallel, but stuck between the seemingly ten different ways to do it?\n**not-again-ai** is a Python package designed to once and for all collect all these little things that come up over and over again in AI projects and put them in one place.\n\n**Documentation** available at [DaveCoDev.github.io/not-again-ai/](https://DaveCoDev.github.io/not-again-ai/).\n\n# Installation\n\nRequires: Python 3.9, 3.10\n\n```bash\n$ pip install not_again_ai\n```\n\n# Quick tour\n\n## Visualization\n\nWe currently offer two visualization tools, a time series plot and a histogram for plotting univariate distributions.\n\n```python\n>>> import numpy as np\n>>> from not_again_ai.viz.time_series import ts_lineplot\n>>> from not_again_ai.viz.distributions import univariate_distplot\n\n# get some time series data\n>>> rs = np.random.RandomState(365)\n>>> values = rs.randn(365, 4).cumsum(axis=0)\n>>> dates = pd.date_range(\'1 1 2021\', periods=365, freq=\'D\')\n# plot the time series and save it to a file\n>>> ts_lineplot(ts_data=values, save_pathname=\'myplot.png\', ts_x=dates, ts_names=[\'A\', \'B\', \'C\', \'D\'])\n\n# get a random distribution\n>>> distrib = np.random.beta(a=0.5, b=0.5, size=1000)\n# plot the distribution and save it to a file\n>>> univariate_distplot(\n...     data=distrib, \n...     save_pathname=\'mydistribution.svg\', \n...     print_summary=False, bins=100, \n...     title=r\'Beta Distribution $\\alpha=0.5, \\beta=0.5$\'\n... )\n```\n\n<p float="center">\n  <img src="assets/ts_lineplot5.svg" width="404" />\n  <img src="assets/distributions_test4.svg" width="404" /> \n</p>\n\n## Parallel\nFor when you have functions you want to execute in parallel.\n\n```python\n# execute the passed in functions in parallel without any additional arguments\n>>> result = embarrassingly_parallel_simple([do_something, do_something2], num_processes=2)\n\n# execute the function mult in parallel with the passed in arguments\n>>> args = ((2, 2), (3, 3), (4, 4))\n>>> result2 = embarrassingly_parallel(mult, args, num_processes=multiprocessing.cpu_count())\n```\n\n## Filesystem\nWe provide helpers to deal with files and directories easily and without raising uncessary errors.\n\n```python\n>>> from not_again_ai.system.files import create_file_dir\n\n# creates the directory mydir if it does not exist\n>>> create_file_dir(\'mydir/myfile.txt\')\n```\n\n# Development Information\n\nThis package uses [Poetry](https://python-poetry.org/) to manage dependencies and\nisolated [Python virtual environments](https://docs.python.org/3/library/venv.html).\n\nTo proceed, \n[install Poetry globally onto your system](https://python-poetry.org/docs/#installation).\n\n(Optional) configure Poetry to use an in-project virtual environment.\n```bash\n$ poetry config virtualenvs.in-project true\n```\n\n## Dependencies\n\nDependencies are defined in [`pyproject.toml`](./pyproject.toml) and specific versions are locked\ninto [`poetry.lock`](./poetry.lock). This allows for exact reproducible environments across\nall machines that use the project, both during development and in production.\n\nTo install all dependencies into an isolated virtual environment:\n\n> Append `--remove-untracked` to uninstall dependencies that are no longer in use from the virtual\n> environment.\n\n```bash\n$ poetry install\n```\n\nTo [activate](https://python-poetry.org/docs/basic-usage#activating-the-virtual-environment) the\nvirtual environment that is automatically created by Poetry:\n\n```bash\n$ poetry shell\n```\n\nTo deactivate the environment:\n\n```bash\n(.venv) $ exit\n```\n\nTo upgrade all dependencies to their latest versions:\n\n```bash\n$ poetry update\n```\n\n## Packaging\n\nThis project is designed as a Python package, meaning that it can be bundled up and redistributed\nas a single compressed file.\n\nPackaging is configured by:\n\n- [`pyproject.toml`](./pyproject.toml)\n\nTo package the project as both a \n[source distribution](https://docs.python.org/3/distutils/sourcedist.html) and\na [wheel](https://wheel.readthedocs.io/en/stable/):\n\n```bash\n$ poetry build\n```\n\nThis will generate `dist/not-again-ai-0.1.0.tar.gz` and `dist/not_again_ai-0.1.0-py3-none-any.whl`.\n\nRead more about the [advantages of wheels](https://pythonwheels.com/) to understand why generating\nwheel distributions are important.\n\n## Publish Distributions to PyPI\n\nSource and wheel redistributable packages can\nbe [published to PyPI](https://python-poetry.org/docs/cli#publish) or installed\ndirectly from the filesystem using `pip`.\n\n```bash\n$ poetry publish\n```\n\n# Enforcing Code Quality\n\nAutomated code quality checks are performed using \n[Nox](https://nox.thea.codes/en/stable/) and\n[`nox-poetry`](https://nox-poetry.readthedocs.io/en/stable/). Nox will automatically create virtual\nenvironments and run commands based on [`noxfile.py`](./noxfile.py) for unit testing, PEP 8 style\nguide checking, type checking and documentation generation.\n\n> Note: `nox` is installed into the virtual environment automatically by the `poetry install`\n> command above. Run `poetry shell` to activate the virtual environment.\n\nTo run all default sessions:\n\n```bash\n(.venv) $ nox\n```\n\n## Unit Testing\n\nUnit testing is performed with [pytest](https://pytest.org/). pytest has become the de facto Python\nunit testing framework. Some key advantages over the built-in\n[unittest](https://docs.python.org/3/library/unittest.html) module are:\n\n1. Significantly less boilerplate needed for tests.\n2. PEP 8 compliant names (e.g. `pytest.raises()` instead of `self.assertRaises()`).\n3. Vibrant ecosystem of plugins.\n\npytest will automatically discover and run tests by recursively searching for folders and `.py`\nfiles prefixed with `test` for any functions prefixed by `test`.\n\nThe `tests` folder is created as a Python package (i.e. there is an `__init__.py` file within it)\nbecause this helps `pytest` uniquely namespace the test files. Without this, two test files cannot\nbe named the same, even if they are in different subdirectories.\n\nCode coverage is provided by the [pytest-cov](https://pytest-cov.readthedocs.io/en/latest/) plugin.\n\nWhen running a unit test Nox session (e.g. `nox -s test`), an HTML report is generated in\nthe `htmlcov` folder showing each source file and which lines were executed during unit testing.\nOpen `htmlcov/index.html` in a web browser to view the report. Code coverage reports help identify\nareas of the project that are currently not tested.\n\npytest and code coverage are configured in [`pyproject.toml`](./pyproject.toml).\n\nTo pass arguments to `pytest` through `nox`:\n\n```bash\n(.venv) $ nox -s test -- -k invalid_factorial\n```\n\n## Code Style Checking\n\n[PEP 8](https://peps.python.org/pep-0008/) is the universally accepted style guide for\nPython code. PEP 8 code compliance is verified using [Flake8](http://flake8.pycqa.org/). Flake8 is\nconfigured in the `[tool.flake8]` section of `pyproject.toml`. Extra Flake8 plugins are also\nincluded:\n\n- `flake8-bugbear`: Find likely bugs and design problems in your program.\n- `flake8-broken-line`: Forbid using backslashes (`\\`) for line breaks.\n- `flake8-comprehensions`: Helps write better `list`/`set`/`dict` comprehensions.\n- `pep8-naming`: Ensure functions, classes, and variables are named with correct casing.\n- `pyproject-flake8`: Allow configuration of `flake8` through `pyproject.toml`.\n\nSome code style settings are included in [`.editorconfig`](./.editorconfig) and will be\nconfigured automatically in editors such as PyCharm.\n\nTo lint code, run:\n\n```bash\n(.venv) $ nox -s lint\n```\n\n## Automated Code Formatting\n\nCode is automatically formatted using [black](https://github.com/psf/black). Imports are\nautomatically sorted and grouped using [isort](https://github.com/PyCQA/isort/).\n\nThese tools are configured by:\n\n- [`pyproject.toml`](./pyproject.toml)\n\nTo automatically format code, run:\n\n```bash\n(.venv) $ nox -s fmt\n```\n\nTo verify code has been formatted, such as in a CI job:\n\n```bash\n(.venv) $ nox -s fmt_check\n```\n\n## Type Checking\n\n[Type annotations](https://docs.python.org/3/library/typing.html) allows developers to include\noptional static typing information to Python source code. This allows static analyzers such\nas [mypy](http://mypy-lang.org/), [PyCharm](https://www.jetbrains.com/pycharm/),\nor [Pyright](https://github.com/microsoft/pyright) to check that functions are used with the\ncorrect types before runtime.\n\n\n```python\ndef factorial(n: int) -> int:\n    ...\n```\n\nmypy is configured in [`pyproject.toml`](./pyproject.toml). To type check code, run:\n\n```bash\n(.venv) $ nox -s type_check\n```\n\nSee also [awesome-python-typing](https://github.com/typeddjango/awesome-python-typing).\n\n### Distributing Type Annotations\n\n[PEP 561](https://www.python.org/dev/peps/pep-0561/) defines how a Python package should\ncommunicate the presence of inline type annotations to static type\ncheckers. [mypy\'s documentation](https://mypy.readthedocs.io/en/stable/installed_packages.html)\nprovides further examples on how to do this.\n\nMypy looks for the existence of a file named [`py.typed`](./src/not-again-ai/py.typed) in the root of the\ninstalled package to indicate that inline type annotations should be checked.\n\n## Continuous Integration\n\nContinuous integration is provided by [GitHub Actions](https://github.com/features/actions). This\nruns all tests, lints, and type checking for every commit and pull request to the repository.\n\nGitHub Actions is configured in [`.github/workflows/python.yml`](./.github/workflows/python.yml).\n\n## [Visual Studio Code](https://code.visualstudio.com/docs/languages/python)\n\nInstall the [Python extension](https://marketplace.visualstudio.com/items?itemName=ms-python.python) for VSCode.\n\nDefault settings are configured in [`.vscode/settings.json`](./.vscode/settings.json). This will enable flake8 and black formatting with consistent settings.\n\n# Documentation\n\n## Generating a User Guide\n\n[Material for MkDocs](https://squidfunk.github.io/mkdocs-material/) is a powerful static site\ngenerator that combines easy-to-write Markdown, with a number of Markdown extensions that increase\nthe power of Markdown. This makes it a great fit for user guides and other technical documentation.\n\nThe example MkDocs project included in this project is configured to allow the built documentation\nto be hosted at any URL or viewed offline from the file system.\n\nTo build the user guide, run,\n\n```bash\n(.venv) $ nox -s docs\n```\n\nand open `docs/user_guide/site/index.html` using a web browser.\n\nTo build and serve the user guide with automatic rebuilding as you change the contents,\nrun:\n\n```bash\n(.venv) $ nox -s docs_serve\n``` \n\nand open <http://127.0.0.1:8000> in a browser.\n\nEach time the `main` Git branch is updated, the \n[`.github/workflows/pages.yml`](.github/workflows/pages.yml) GitHub Action will\nautomatically build the user guide and publish it to [GitHub Pages](https://pages.github.com/).\nThis is configured in the `docs_github_pages` Nox session.\n\n## Generating API Documentation\n\nThis project uses [mkdocstrings](https://github.com/mkdocstrings/mkdocstrings) plugin for\nMkDocs, which renders\n[Google-style docstrings](https://www.sphinx-doc.org/en/master/usage/extensions/napoleon.html)\ninto an MkDocs project. Google-style docstrings provide a good mix of easy-to-read docstrings in\ncode as well as nicely-rendered output.\n\n```python\n"""Computes the factorial through a recursive algorithm.\n\nArgs:\n    n: A positive input value.\n\nRaises:\n    InvalidFactorialError: If n is less than 0.\n\nReturns:\n    Computed factorial.\n"""\n```\n\n\n# Attributions\n[python-blueprint](https://github.com/johnthagen/python-blueprint) for the Python package skeleton.\n',
    'author': 'DaveCoDev',
    'author_email': 'dave.co.dev@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/DaveCoDev/not-again-ai',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.9,<3.11',
}


setup(**setup_kwargs)
