//
// Copyright 2017 Animal Logic
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
#pragma once

#include <AL/usdmaya/Api.h>

#include <pxr/pxr.h>
#include <pxr/base/tf/token.h>
#include <pxr/base/tf/type.h>
#include <pxr/usd/usd/prim.h>
#include <pxr/usd/usd/stage.h>
#include <pxr/usd/usd/typed.h>

PXR_NAMESPACE_USING_DIRECTIVE


namespace AL {
namespace usdmaya {
namespace fileio {
namespace translators {

//----------------------------------------------------------------------------------------------------------------------
#ifndef AL_GENERATING_DOCS
class TranslatorTestType
  : public UsdTyped
{
public:
  /// Compile-time constant indicating whether or not this class corresponds
  /// to a concrete instantiable prim type in scene description.  If this is
  /// true, GetStaticPrimDefinition() will return a valid prim definition with
  /// a non-empty typeName.
  AL_USDMAYA_PUBLIC
  static const bool IsConcrete = true;

  /// Construct a TranslatorTestType on UsdPrim \p prim .
  /// Equivalent to TranslatorTestType::Get(prim.GetStage(), prim.GetPath())
  /// for a \em valid \p prim, but will not immediately throw an error for
  /// an invalid \p prim
  explicit TranslatorTestType(const UsdPrim& prim=UsdPrim())
      : UsdTyped(prim)
  {
  }

  /// Construct a TranslatorTestType on the prim held by \p schemaObj .
  /// Should be preferred over TranslatorTestType(schemaObj.GetPrim()),
  /// as it preserves SchemaBase state.
  explicit TranslatorTestType(const UsdSchemaBase& schemaObj)
      : UsdTyped(schemaObj)
  {
  }

  /// Destructor.
  AL_USDMAYA_PUBLIC
  virtual ~TranslatorTestType();

  /// Return a vector of names of all pre-declared attributes for this schema
  /// class and all its ancestor classes.  Does not include attributes that
  /// may be authored by custom/extended methods of the schemas involved.
  AL_USDMAYA_PUBLIC
  static const TfTokenVector& GetSchemaAttributeNames(bool includeInherited = true);

  /// \brief Return a TranslatorTestType holding the prim adhering to this
  /// schema at \p path on \p stage.  If no prim exists at \p path on
  /// \p stage, or if the prim at that path does not adhere to this schema,
  /// return an invalid schema object.  This is shorthand for the following:
  ///
  /// \code
  /// TranslatorTestType(stage->GetPrimAtPath(path));
  /// \endcode
  ///
  AL_USDMAYA_PUBLIC
  static TranslatorTestType Get(const UsdStagePtr &stage, const SdfPath &path);

  /// \brief Attempt to ensure a \a UsdPrim adhering to this schema at \p path
  /// is defined (according to UsdPrim::IsDefined()) on this stage.
  ///
  /// If a prim adhering to this schema at \p path is already defined on this
  /// stage, return that prim.  Otherwise author an \a SdfPrimSpec with
  /// \a specifier == \a SdfSpecifierDef and this schema's prim type name for
  /// the prim at \p path at the current EditTarget.  Author \a SdfPrimSpec s
  /// with \p specifier == \a SdfSpecifierDef and empty typeName at the
  /// current EditTarget for any nonexistent, or existing but not \a Defined
  /// ancestors.
  ///
  /// The given \a path must be an absolute prim path that does not contain
  /// any variant selections.
  ///
  /// If it is impossible to author any of the necessary PrimSpecs, (for
  /// example, in case \a path cannot map to the current UsdEditTarget's
  /// namespace) issue an error and return an invalid \a UsdPrim.
  ///
  /// Note that this method may return a defined prim whose typeName does not
  /// specify this schema class, in case a stronger typeName opinion overrides
  /// the opinion at the current EditTarget.
  ///
  AL_USDMAYA_PUBLIC
  static TranslatorTestType Define(const UsdStagePtr &stage, const SdfPath &path);

private:
  // needs to invoke _GetStaticTfType.
  friend class UsdSchemaRegistry;
  static const TfType &_GetStaticTfType();

  static bool _IsTypedSchema();

  // override SchemaBase virtuals.
  virtual const TfType &_GetTfType() const;
};
#endif

//----------------------------------------------------------------------------------------------------------------------
} // translators
} // fileio
} // usdmaya
} // AL
//----------------------------------------------------------------------------------------------------------------------
