"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const lib_1 = require("../lib");
nodeunit_shim_1.nodeunitShim({
    'security group can allows all outbound traffic by default'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: true });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: '0.0.0.0/0',
                    Description: 'Allow all outbound traffic by default',
                    IpProtocol: '-1',
                },
            ],
        }));
        test.done();
    },
    'no new outbound rule is added if we are allowing all traffic anyway'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        const sg = new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: true });
        sg.addEgressRule(lib_1.Peer.anyIpv4(), lib_1.Port.tcp(86), 'This does not show up');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: '0.0.0.0/0',
                    Description: 'Allow all outbound traffic by default',
                    IpProtocol: '-1',
                },
            ],
        }));
        test.done();
    },
    'security group disallow outbound traffic by default'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: false });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: '255.255.255.255/32',
                    Description: 'Disallow all traffic',
                    FromPort: 252,
                    IpProtocol: 'icmp',
                    ToPort: 86,
                },
            ],
        }));
        test.done();
    },
    'bogus outbound rule disappears if another rule is added'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        const sg = new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: false });
        sg.addEgressRule(lib_1.Peer.anyIpv4(), lib_1.Port.tcp(86), 'This replaces the other one');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: '0.0.0.0/0',
                    Description: 'This replaces the other one',
                    FromPort: 86,
                    IpProtocol: 'tcp',
                    ToPort: 86,
                },
            ],
        }));
        test.done();
    },
    'all outbound rule cannot be added after creation'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        const sg = new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: false });
        test.throws(() => {
            sg.addEgressRule(lib_1.Peer.anyIpv4(), lib_1.Port.allTraffic(), 'All traffic');
        }, /Cannot add/);
        test.done();
    },
    'immutable imports do not add rules'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const sg = lib_1.SecurityGroup.fromSecurityGroupId(stack, 'SG1', 'test-id', { mutable: false });
        sg.addEgressRule(lib_1.Peer.anyIpv4(), lib_1.Port.tcp(86), 'This rule was not added');
        sg.addIngressRule(lib_1.Peer.anyIpv4(), lib_1.Port.tcp(86), 'This rule was not added');
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: '0.0.0.0/0',
                    Description: 'This rule was not added',
                    FromPort: 86,
                    IpProtocol: 'tcp',
                    ToPort: 86,
                },
            ],
        })));
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupIngress: [
                {
                    CidrIp: '0.0.0.0/0',
                    Description: 'This rule was not added',
                    FromPort: 86,
                    IpProtocol: 'tcp',
                    ToPort: 86,
                },
            ],
        })));
        test.done();
    },
    'peer between all types of peers and port range types'(test) {
        // GIVEN
        const stack = new core_1.Stack(undefined, 'TestStack', { env: { account: '12345678', region: 'dummy' } });
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg = new lib_1.SecurityGroup(stack, 'SG', { vpc });
        const peers = [
            new lib_1.SecurityGroup(stack, 'PeerGroup', { vpc }),
            lib_1.Peer.anyIpv4(),
            lib_1.Peer.anyIpv6(),
            lib_1.Peer.prefixList('pl-012345'),
        ];
        const ports = [
            lib_1.Port.tcp(1234),
            lib_1.Port.tcp(core_1.Lazy.numberValue({ produce: () => 5000 })),
            lib_1.Port.allTcp(),
            lib_1.Port.tcpRange(80, 90),
            lib_1.Port.udp(2345),
            lib_1.Port.udp(core_1.Lazy.numberValue({ produce: () => 7777 })),
            lib_1.Port.allUdp(),
            lib_1.Port.udpRange(85, 95),
            lib_1.Port.icmpTypeAndCode(5, 1),
            lib_1.Port.icmpType(8),
            lib_1.Port.allIcmp(),
            lib_1.Port.icmpPing(),
            lib_1.Port.allTraffic(),
        ];
        // WHEN
        for (const peer of peers) {
            for (const port of ports) {
                sg.connections.allowTo(peer, port);
                sg.connections.allowFrom(peer, port);
            }
        }
        // THEN -- no crash
        test.done();
    },
    'if tokens are used in ports, `canInlineRule` should be false to avoid cycles'(test) {
        // GIVEN
        const p1 = core_1.Lazy.numberValue({ produce: () => 80 });
        const p2 = core_1.Lazy.numberValue({ produce: () => 5000 });
        // WHEN
        const ports = [
            lib_1.Port.tcp(p1),
            lib_1.Port.tcp(p2),
            lib_1.Port.tcpRange(p1, 90),
            lib_1.Port.tcpRange(80, p2),
            lib_1.Port.tcpRange(p1, p2),
            lib_1.Port.udp(p1),
            lib_1.Port.udpRange(p1, 95),
            lib_1.Port.udpRange(85, p2),
            lib_1.Port.udpRange(p1, p2),
            lib_1.Port.icmpTypeAndCode(p1, 1),
            lib_1.Port.icmpTypeAndCode(5, p1),
            lib_1.Port.icmpTypeAndCode(p1, p2),
            lib_1.Port.icmpType(p1),
        ];
        // THEN
        for (const range of ports) {
            test.equal(range.canInlineRule, false, range.toString());
        }
        test.done();
    },
    'Peer IP CIDR validation': {
        'passes with valid IPv4 CIDR block'(test) {
            // GIVEN
            const cidrIps = ['0.0.0.0/0', '192.168.255.255/24'];
            // THEN
            for (const cidrIp of cidrIps) {
                test.equal(lib_1.Peer.ipv4(cidrIp).uniqueId, cidrIp);
            }
            test.done();
        },
        'passes with unresolved IP CIDR token'(test) {
            // GIVEN
            core_1.Token.asString(new core_1.Intrinsic('ip'));
            // THEN: don't throw
            test.done();
        },
        'throws if invalid IPv4 CIDR block'(test) {
            // THEN
            test.throws(() => {
                lib_1.Peer.ipv4('invalid');
            }, /Invalid IPv4 CIDR/);
            test.done();
        },
        'throws if missing mask in IPv4 CIDR block'(test) {
            test.throws(() => {
                lib_1.Peer.ipv4('0.0.0.0');
            }, /CIDR mask is missing in IPv4/);
            test.done();
        },
        'passes with valid IPv6 CIDR block'(test) {
            // GIVEN
            const cidrIps = [
                '::/0',
                '2001:db8::/32',
                '2001:0db8:0000:0000:0000:8a2e:0370:7334/32',
                '2001:db8::8a2e:370:7334/32',
            ];
            // THEN
            for (const cidrIp of cidrIps) {
                test.equal(lib_1.Peer.ipv6(cidrIp).uniqueId, cidrIp);
            }
            test.done();
        },
        'throws if invalid IPv6 CIDR block'(test) {
            // THEN
            test.throws(() => {
                lib_1.Peer.ipv6('invalid');
            }, /Invalid IPv6 CIDR/);
            test.done();
        },
        'throws if missing mask in IPv6 CIDR block'(test) {
            test.throws(() => {
                lib_1.Peer.ipv6('::');
            }, /IDR mask is missing in IPv6/);
            test.done();
        },
    },
});
//# sourceMappingURL=data:application/json;base64,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