"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ssm_1 = require("@aws-cdk/aws-ssm");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const lib_1 = require("../lib");
nodeunit_shim_1.nodeunitShim({
    'instance is created correctly'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.Instance(stack, 'Instance', {
            vpc,
            machineImage: new lib_1.AmazonLinuxImage(),
            instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
            InstanceType: 't3.large',
        }));
        test.done();
    },
    'instance is created with source/dest check switched off'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.Instance(stack, 'Instance', {
            vpc,
            machineImage: new lib_1.AmazonLinuxImage(),
            instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
            sourceDestCheck: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
            InstanceType: 't3.large',
            SourceDestCheck: false,
        }));
        test.done();
    },
    'instance is grantable'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const param = new aws_ssm_1.StringParameter(stack, 'Param', { stringValue: 'Foobar' });
        const instance = new lib_1.Instance(stack, 'Instance', {
            vpc,
            machineImage: new lib_1.AmazonLinuxImage(),
            instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
        });
        // WHEN
        param.grantRead(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'ssm:DescribeParameters',
                            'ssm:GetParameters',
                            'ssm:GetParameter',
                            'ssm:GetParameterHistory',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':ssm:',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    ':',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':parameter/',
                                    {
                                        Ref: 'Param165332EC',
                                    },
                                ],
                            ],
                        },
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    blockDeviceMappings: {
        'can set blockDeviceMappings'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // WHEN
            new lib_1.Instance(stack, 'Instance', {
                vpc,
                machineImage: new lib_1.AmazonLinuxImage(),
                instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
                blockDevices: [{
                        deviceName: 'ebs',
                        mappingEnabled: true,
                        volume: lib_1.BlockDeviceVolume.ebs(15, {
                            deleteOnTermination: true,
                            encrypted: true,
                            volumeType: lib_1.EbsDeviceVolumeType.IO1,
                            iops: 5000,
                        }),
                    }, {
                        deviceName: 'ebs-snapshot',
                        mappingEnabled: false,
                        volume: lib_1.BlockDeviceVolume.ebsFromSnapshot('snapshot-id', {
                            volumeSize: 500,
                            deleteOnTermination: false,
                            volumeType: lib_1.EbsDeviceVolumeType.SC1,
                        }),
                    }, {
                        deviceName: 'ephemeral',
                        volume: lib_1.BlockDeviceVolume.ephemeral(0),
                    }],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
                BlockDeviceMappings: [
                    {
                        DeviceName: 'ebs',
                        Ebs: {
                            DeleteOnTermination: true,
                            Encrypted: true,
                            Iops: 5000,
                            VolumeSize: 15,
                            VolumeType: 'io1',
                        },
                    },
                    {
                        DeviceName: 'ebs-snapshot',
                        Ebs: {
                            DeleteOnTermination: false,
                            SnapshotId: 'snapshot-id',
                            VolumeSize: 500,
                            VolumeType: 'sc1',
                        },
                        NoDevice: {},
                    },
                    {
                        DeviceName: 'ephemeral',
                        VirtualName: 'ephemeral0',
                    },
                ],
            }));
            test.done();
        },
        'throws if ephemeral volumeIndex < 0'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // THEN
            test.throws(() => {
                new lib_1.Instance(stack, 'Instance', {
                    vpc,
                    machineImage: new lib_1.AmazonLinuxImage(),
                    instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
                    blockDevices: [{
                            deviceName: 'ephemeral',
                            volume: lib_1.BlockDeviceVolume.ephemeral(-1),
                        }],
                });
            }, /volumeIndex must be a number starting from 0/);
            test.done();
        },
        'throws if volumeType === IO1 without iops'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // THEN
            test.throws(() => {
                new lib_1.Instance(stack, 'Instance', {
                    vpc,
                    machineImage: new lib_1.AmazonLinuxImage(),
                    instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
                    blockDevices: [{
                            deviceName: 'ebs',
                            volume: lib_1.BlockDeviceVolume.ebs(15, {
                                deleteOnTermination: true,
                                encrypted: true,
                                volumeType: lib_1.EbsDeviceVolumeType.IO1,
                            }),
                        }],
                });
            }, /ops property is required with volumeType: EbsDeviceVolumeType.IO1/);
            test.done();
        },
        'warning if iops without volumeType'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            const instance = new lib_1.Instance(stack, 'Instance', {
                vpc,
                machineImage: new lib_1.AmazonLinuxImage(),
                instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
                blockDevices: [{
                        deviceName: 'ebs',
                        volume: lib_1.BlockDeviceVolume.ebs(15, {
                            deleteOnTermination: true,
                            encrypted: true,
                            iops: 5000,
                        }),
                    }],
            });
            // THEN
            test.deepEqual(instance.node.metadata[0].type, cxschema.ArtifactMetadataEntryType.WARN);
            test.deepEqual(instance.node.metadata[0].data, 'iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            test.done();
        },
        'warning if iops and volumeType !== IO1'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            const instance = new lib_1.Instance(stack, 'Instance', {
                vpc,
                machineImage: new lib_1.AmazonLinuxImage(),
                instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
                blockDevices: [{
                        deviceName: 'ebs',
                        volume: lib_1.BlockDeviceVolume.ebs(15, {
                            deleteOnTermination: true,
                            encrypted: true,
                            volumeType: lib_1.EbsDeviceVolumeType.GP2,
                            iops: 5000,
                        }),
                    }],
            });
            // THEN
            test.deepEqual(instance.node.metadata[0].type, cxschema.ArtifactMetadataEntryType.WARN);
            test.deepEqual(instance.node.metadata[0].data, 'iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            test.done();
        },
    },
    'instance can be created with Private IP Address'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.Instance(stack, 'Instance', {
            vpc,
            machineImage: new lib_1.AmazonLinuxImage(),
            instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
            privateIpAddress: '10.0.0.2',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
            InstanceType: 't3.large',
            PrivateIpAddress: '10.0.0.2',
        }));
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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